#!/bin/bash

echo "Starting HealNet Model Testing..."

# Define experiment parameters (should match training parameters)
BATCH_SIZE=4               
LR=0.0001
MAX_LR=0.0001
PATIENCE=10
EPOCHS=50
DROPOUT=0.2
FOLD=1                     # Fixed fold 1
GPU="0,1,2,3"              # GPU device numbers (multiple GPUs) 
PRETRAINED=true

# Testing configuration
USE_DEMOGRAPHICS=false
# Set CROSS_EVAL to enable cross evaluation:
# - "" or empty: normal test (same data as training)
# - "matched_to_full": test models trained on matched data with full data
# - "full_to_matched": test models trained on full data with matched data
CROSS_EVAL="matched_to_full"              # Change this as needed

# HealNet-specific parameters (should match training)
DEPTH=1                    # Number of fusion layers 
LATENT_CHANNELS=256        # Number of latent tokens (l_c)
LATENT_DIM=256             # Dimension of latent tokens (l_d)
CROSS_HEADS=4              # Number of cross-attention heads
LATENT_HEADS=4             # Number of self-attention heads
CROSS_DIM_HEAD=64          # Dimension of each cross-attention head
LATENT_DIM_HEAD=64         # Dimension of each self-attention head
SELF_PER_CROSS_ATTN=1      # Self-attention layers per cross-attention
WEIGHT_TIE_LAYERS=true     # Whether to share weights across layers
SNN=true                   # Whether to use self-normalizing networks
FOURIER_ENCODE_DATA=true   # Whether to use Fourier positional encoding
NUM_FREQ_BANDS=4           # Number of frequency bands 
MAX_FREQ=5.0               # Maximum frequency for encoding 
FINAL_CLASSIFIER_HEAD=true # Whether to add final classification head
ATTN_DROPOUT=0.2           # Dropout rate for attention layers
FF_DROPOUT=0.2             # Dropout rate for feed-forward layers

# Seeds configuration (should match training seeds)
SEEDS=(42 123 1234)

# Tasks configuration
TASKS=("mortality")

# Function to find checkpoint file dynamically
find_checkpoint() {
    local checkpoint_dir="$1"
    local checkpoint_file=$(find "$checkpoint_dir" -name "*.ckpt" -type f | head -1)
    echo "$checkpoint_file"
}

# Function to determine training data config for checkpoint path
get_training_data_config() {
    local cross_eval="$1"
    if [ "$cross_eval" = "matched_to_full" ]; then
        echo "matched"
    elif [ "$cross_eval" = "full_to_matched" ]; then
        echo "full"
    else
        # For normal testing, determine based on current settings
        if [ "$MATCHED" = "true" ]; then
            echo "matched"
        else
            echo "full"
        fi
    fi
}

# 创建根目录名称：healnet-{cross_eval}
ROOT_DIR="../experiments/healnet-${CROSS_EVAL}"

echo "📁 All results will be saved under: $ROOT_DIR"
echo "🖥️  Multi-GPU testing using: $GPU"

for TASK in "${TASKS[@]}"
do
    if [ -n "$CROSS_EVAL" ]; then
        echo "Testing HealNet model for task: $TASK with cross evaluation: $CROSS_EVAL"
    else
        echo "Testing HealNet model for task: $TASK (normal test)"
    fi
    
    # Set task-specific parameters
    if [ "$TASK" = "phenotype" ]; then
        NUM_CLASSES=25
        INPUT_DIM=49          # Note: HealNet uses 49, not 498
    elif [ "$TASK" = "mortality" ]; then
        NUM_CLASSES=1
        INPUT_DIM=49
    elif [ "$TASK" = "los" ]; then
        NUM_CLASSES=10
        INPUT_DIM=49
    fi
    
    for SEED in "${SEEDS[@]}"
    do
        echo "Testing with seed $SEED for task $TASK on fold $FOLD..."
        
        # Determine the training data config for finding checkpoint
        TRAINING_DATA_CONFIG=$(get_training_data_config "$CROSS_EVAL")
        
        # Build checkpoint directory path (still use the full path to find existing checkpoints)
        CHECKPOINT_DIR="../experiments/healnet/$TASK/lightning_logs/HEALNET-model_healnet-task_${TASK}-fold_${FOLD}-batch_size_${BATCH_SIZE}-lr_${LR}-patience_${PATIENCE}-epochs_${EPOCHS}-dropout_${DROPOUT}-seed_${SEED}-pretrained_True-data_config_${TRAINING_DATA_CONFIG}/checkpoints"
        
        # Find checkpoint file
        CHECKPOINT_FILE=$(find_checkpoint "$CHECKPOINT_DIR")
        
        if [ ! -f "$CHECKPOINT_FILE" ]; then
            echo "❌ Error: Checkpoint not found in $CHECKPOINT_DIR"
            echo "Available directories:"
            ls -la "../experiments/healnet/$TASK/lightning_logs/" | grep "seed_${SEED}" || echo "No directories found for seed $SEED"
            continue
        fi
        
        echo "✅ Found checkpoint: $CHECKPOINT_FILE"
        
        # 创建自定义输出目录：root_dir/task_name
        CUSTOM_LOG_DIR="$ROOT_DIR/$TASK"
        
        # Build base command with custom log_dir
        CMD="python ../main.py \
            --model healnet \
            --mode test \
            --task $TASK \
            --fold $FOLD \
            --batch_size $BATCH_SIZE \
            --lr $LR \
            --patience $PATIENCE \
            --epochs $EPOCHS \
            --dropout $DROPOUT \
            --seed $SEED \
            --input_dim $INPUT_DIM \
            --num_classes $NUM_CLASSES \
            --depth $DEPTH \
            --latent_channels $LATENT_CHANNELS \
            --latent_dim $LATENT_DIM \
            --cross_heads $CROSS_HEADS \
            --latent_heads $LATENT_HEADS \
            --cross_dim_head $CROSS_DIM_HEAD \
            --latent_dim_head $LATENT_DIM_HEAD \
            --self_per_cross_attn $SELF_PER_CROSS_ATTN \
            --num_freq_bands $NUM_FREQ_BANDS \
            --max_freq $MAX_FREQ \
            --attn_dropout $ATTN_DROPOUT \
            --ff_dropout $FF_DROPOUT \
            --checkpoint_path $CHECKPOINT_FILE \
            --log_dir $CUSTOM_LOG_DIR"
        
        # 处理多GPU参数：将"0,1,2,3"转换为["--gpu", "0", "1", "2", "3"]
        if [[ "$GPU" == *","* ]]; then
            # 多GPU情况
            IFS=',' read -ra GPU_ARRAY <<< "$GPU"
            CMD="$CMD --gpu"
            for gpu_id in "${GPU_ARRAY[@]}"; do
                CMD="$CMD $gpu_id"
            done
        else
            # 单GPU情况
            CMD="$CMD --gpu $GPU"
        fi
        
        # Add conditional boolean parameters
        if [ "$PRETRAINED" = "true" ]; then
            CMD="$CMD --pretrained"
        fi
        
        if [ "$USE_DEMOGRAPHICS" = "true" ]; then
            CMD="$CMD --use_demographics"
        fi
        
        if [ "$WEIGHT_TIE_LAYERS" = "true" ]; then
            CMD="$CMD --weight_tie_layers"
        fi
        
        if [ "$SNN" = "true" ]; then
            CMD="$CMD --snn"
        fi
        
        if [ "$FOURIER_ENCODE_DATA" = "true" ]; then
            CMD="$CMD --fourier_encode_data"
        fi
        
        if [ "$FINAL_CLASSIFIER_HEAD" = "true" ]; then
            CMD="$CMD --final_classifier_head"
        fi
        
        # Add cross evaluation parameter if specified
        if [ -n "$CROSS_EVAL" ]; then
            CMD="$CMD --cross_eval $CROSS_EVAL"
        else
            # For normal testing, add matched parameter if needed
            if [ "$TRAINING_DATA_CONFIG" = "matched" ]; then
                CMD="$CMD --matched"
            fi
        fi
    done
done
