#!/bin/bash

echo "Starting MMTM Model Testing..."

# Define experiment parameters (should match training parameters)
BATCH_SIZE=16
LR=0.0001
PATIENCE=10
EPOCHS=50
DROPOUT=0.2
FOLD=1                     # Fixed fold 1
GPU=0                      # GPU device number
PRETRAINED=true

# Testing configuration
USE_DEMOGRAPHICS=false
CROSS_EVAL="matched_to_full"              # Change this as needed

# MMTM-specific parameters (should match training)
DIM=256                    # Hidden dimension
EHR_ENCODER="transformer"        # Options: 'lstm', 'transformer'
CXR_ENCODER="resnet50"    # Options: 'resnet50', 'vit_b_16'

# EHR Encoder Parameters
EHR_NUM_LAYERS=1          # For LSTM encoder
EHR_BIDIRECTIONAL=true   
EHR_N_HEAD=4              # For Transformer encoder
EHR_N_LAYERS_TRANS=1      # For Transformer encoder

# MMTM Fusion Parameters
MMTM_RATIO=4              # MMTM compression ratio
LAYER_AFTER=-1            # Which layer to apply MMTM fusion (-1 for all layers)

# Data Configuration
DATA_PAIRS="paired_ehr_cxr"

# Loss and Alignment Parameters
ALIGN=0.0                 # Disable alignment loss

# Seeds configuration (should match training seeds)
SEEDS=(42 123 1234)

# Tasks configuration
TASKS=("phenotype")

# Function to find checkpoint file dynamically
find_checkpoint() {
    local checkpoint_dir="$1"
    # 查找PRAUC最高的检查点文件
    local checkpoint_file=$(find "$checkpoint_dir" -name "best_model_epoch_*_prauc_*.ckpt" -type f | sort -V | tail -1)
    if [ -z "$checkpoint_file" ]; then
        # 如果没有找到，就用第一个.ckpt文件
        checkpoint_file=$(find "$checkpoint_dir" -name "*.ckpt" -type f | head -1)
    fi
    echo "$checkpoint_file"
}

# Function to determine training data config for checkpoint path
get_training_data_config() {
    local cross_eval="$1"
    if [ "$cross_eval" = "matched_to_full" ]; then
        echo "matched"
    elif [ "$cross_eval" = "full_to_matched" ]; then
        echo "full"
    else
        if [ "$MATCHED" = "true" ]; then
            echo "matched"
        else
            echo "full"
        fi
    fi
}

# 创建根目录名称：mmtm-{cross_eval}
ROOT_DIR="../experiments/mmtm-${CROSS_EVAL}"

echo "📁 All results will be saved under: $ROOT_DIR"

for TASK in "${TASKS[@]}"
do
    if [ -n "$CROSS_EVAL" ]; then
        echo "Testing MMTM model for task: $TASK with cross evaluation: $CROSS_EVAL"
    else
        echo "Testing MMTM model for task: $TASK (normal test)"
    fi
    
    # Set task-specific parameters
    if [ "$TASK" = "phenotype" ]; then
        NUM_CLASSES=25
        INPUT_DIM=49
    elif [ "$TASK" = "mortality" ]; then
        NUM_CLASSES=1
        INPUT_DIM=49
    elif [ "$TASK" = "los" ]; then
        NUM_CLASSES=10
        INPUT_DIM=49
    fi
    
    for SEED in "${SEEDS[@]}"
    do
        echo "Testing with seed $SEED for task $TASK on fold $FOLD..."
        
        # Determine the training data config for finding checkpoint
        TRAINING_DATA_CONFIG=$(get_training_data_config "$CROSS_EVAL")
        
        # Build checkpoint directory path (修复：去掉 /checkpoints)
        CHECKPOINT_DIR="../experiments/mmtm/$TASK/lightning_logs/MMTM-model_mmtm-task_${TASK}-fold_${FOLD}-batch_size_${BATCH_SIZE}-lr_${LR}-patience_${PATIENCE}-epochs_${EPOCHS}-dropout_${DROPOUT}-seed_${SEED}-align_${ALIGN}-pretrained_True-data_config_${TRAINING_DATA_CONFIG}"
        
        # Find checkpoint file
        CHECKPOINT_FILE=$(find_checkpoint "$CHECKPOINT_DIR")
        
        if [ ! -f "$CHECKPOINT_FILE" ]; then
            echo "❌ Error: Checkpoint not found in $CHECKPOINT_DIR"
            echo "Available directories:"
            ls -la "../experiments/mmtm/$TASK/lightning_logs/" | grep "seed_${SEED}" || echo "No directories found for seed $SEED"
            continue
        fi
        
        echo "✅ Found checkpoint: $CHECKPOINT_FILE"
        
        # 创建自定义输出目录：root_dir/task_name
        CUSTOM_LOG_DIR="$ROOT_DIR/$TASK"
        
        # Build base command with custom log_dir
        CMD="python ../main.py \
            --model mmtm \
            --mode test \
            --task $TASK \
            --fold $FOLD \
            --batch_size $BATCH_SIZE \
            --lr $LR \
            --patience $PATIENCE \
            --epochs $EPOCHS \
            --dropout $DROPOUT \
            --seed $SEED \
            --dim $DIM \
            --input_dim $INPUT_DIM \
            --num_classes $NUM_CLASSES \
            --ehr_encoder $EHR_ENCODER \
            --cxr_encoder $CXR_ENCODER \
            --mmtm_ratio $MMTM_RATIO \
            --layer_after $LAYER_AFTER \
            --data_pairs $DATA_PAIRS \
            --align $ALIGN \
            --gpu $GPU \
            --checkpoint_path $CHECKPOINT_FILE \
            --log_dir $CUSTOM_LOG_DIR"
        
        # Add EHR encoder specific parameters
        if [ "$EHR_ENCODER" = "lstm" ]; then
            CMD="$CMD --ehr_num_layers $EHR_NUM_LAYERS"
            if [ "$EHR_BIDIRECTIONAL" = "true" ]; then
                CMD="$CMD --ehr_bidirectional"
            fi
        elif [ "$EHR_ENCODER" = "transformer" ]; then
            CMD="$CMD --ehr_n_head $EHR_N_HEAD --ehr_n_layers $EHR_N_LAYERS_TRANS"
        fi
        
        # Add conditional parameters
        if [ "$PRETRAINED" = "true" ]; then
            CMD="$CMD --pretrained"
        fi
        
        if [ "$USE_DEMOGRAPHICS" = "true" ]; then
            CMD="$CMD --use_demographics"
        fi
        
        # Add cross evaluation parameter if specified
        if [ -n "$CROSS_EVAL" ]; then
            CMD="$CMD --cross_eval $CROSS_EVAL"
        else
            if [ "$TRAINING_DATA_CONFIG" = "matched" ]; then
                CMD="$CMD --matched"
            fi
        fi
        
    done
done