#!/bin/bash

echo "Starting 5-seed training for DAFT model on fold 1..."

# Define experiment parameters
BATCH_SIZE=16
LR=0.0001
PATIENCE=10
EPOCHS=50
FOLD=1                     # Fixed fold 1
GPU=0                      # GPU device number
MATCHED=true
USE_DEMOGRAPHICS=false
CROSS_EVAL=""              # Set to "matched_to_full" or "full_to_matched" if needed

# DAFT-specific parameters
EHR_ENCODER="transformer"  # Options: "lstm", "transformer"
CXR_ENCODER="resnet50"     # Options: "resnet50", "vit_b_16"
DIM=256                    # Hidden dimension
DROPOUT=0.2                # Dropout rate
LAYER_AFTER=-1             # Which layer to apply DAFT fusion (-1 for all layers)
DAFT_ACTIVATION="linear"   # Options: "linear", "sigmoid", "tanh"
PRETRAINED=true            # Use pretrained CXR encoder
DATA_PAIRS="paired_ehr_cxr" # Data pairs configuration

# EHR Transformer parameters (used when EHR_ENCODER="transformer")
EHR_N_HEAD=4
EHR_N_LAYERS=1

# Seeds configuration
SEEDS=(42 123 1234)

# Tasks configuration
TASKS=("mortality")

# Function to run training for a specific task
run_task_training() {
    local TASK=$1
    echo "Starting training for task: $TASK"
    
    # Set task-specific parameters
    if [ "$TASK" = "phenotype" ]; then
        NUM_CLASSES=25
        INPUT_DIM=49
    elif [ "$TASK" = "mortality" ]; then
        NUM_CLASSES=1
        INPUT_DIM=49
    fi
    
    for SEED in "${SEEDS[@]}"
    do
        echo "Training with seed $SEED for task $TASK on fold $FOLD..."
        
        # Build base command
        CMD="python ../main.py \
            --model daft \
            --mode train \
            --task $TASK \
            --fold $FOLD \
            --batch_size $BATCH_SIZE \
            --lr $LR \
            --patience $PATIENCE \
            --epochs $EPOCHS \
            --dropout $DROPOUT \
            --seed $SEED \
            --dim $DIM \
            --input_dim $INPUT_DIM \
            --num_classes $NUM_CLASSES \
            --ehr_encoder $EHR_ENCODER \
            --cxr_encoder $CXR_ENCODER \
            --layer_after $LAYER_AFTER \
            --daft_activation $DAFT_ACTIVATION \
            --data_pairs $DATA_PAIRS \
            --gpu $GPU"
        
        # Add EHR Transformer parameters (if using transformer)
        if [ "$EHR_ENCODER" = "transformer" ]; then
            CMD="$CMD --ehr_n_head $EHR_N_HEAD --ehr_n_layers $EHR_N_LAYERS"
        fi
        
        # Add conditional parameters
        if [ "$PRETRAINED" = "true" ]; then
            CMD="$CMD --pretrained"
        fi
        
        if [ "$MATCHED" = "true" ]; then
            CMD="$CMD --matched"
        fi
        
        if [ "$USE_DEMOGRAPHICS" = "true" ]; then
            CMD="$CMD --use_demographics"
        fi
        
        if [ -n "$CROSS_EVAL" ]; then
            CMD="$CMD --cross_eval $CROSS_EVAL"
        fi
        
        echo "Running command: $CMD"
        eval $CMD
        
        if [ $? -eq 0 ]; then
            echo "Seed $SEED for task $TASK completed successfully!"
        else
            echo "Error: Seed $SEED for task $TASK failed!"
            exit 1
        fi
    done
    
    echo "All 5 seeds completed for task $TASK on fold $FOLD!"
    echo "Collecting and aggregating statistics..."
    
    # Collect statistics for multi-seed experiment
    echo "Attempting statistics collection for multi-seed experiment..."
    python ../collect_seed_statistics.py \
        --experiment_dir ../experiments \
        --model daft \
        --task $TASK \
        --fold $FOLD \
        --seeds ${SEEDS[@]} \
        --batch_size $BATCH_SIZE \
        --lr $LR \
        --patience $PATIENCE \
        --epochs $EPOCHS \
        --dropout $DROPOUT \
        --dim $DIM \
        --input_dim $INPUT_DIM \
        --num_classes $NUM_CLASSES \
        --ehr_encoder $EHR_ENCODER \
        --cxr_encoder $CXR_ENCODER \
        --ehr_n_head $EHR_N_HEAD \
        --ehr_n_layers $EHR_N_LAYERS \
        --layer_after $LAYER_AFTER \
        --daft_activation $DAFT_ACTIVATION \
        --data_pairs $DATA_PAIRS \
        --pretrained $PRETRAINED \
        --matched $MATCHED \
        --use_demographics $USE_DEMOGRAPHICS
    
    if [ $? -eq 0 ]; then
        echo "Statistics collection completed successfully for task $TASK!"
    else
        echo "Warning: Statistics collection failed for task $TASK, but training completed."
    fi
}

# Main execution
echo "DAFT Multi-seed Training Script"
echo "==============================="
echo "Configuration:"
echo "  Batch Size: $BATCH_SIZE"
echo "  Learning Rate: $LR"
echo "  Epochs: $EPOCHS"
echo "  Patience: $PATIENCE"
echo "  Fold: $FOLD"
echo "  GPU: $GPU"
echo "  Seeds: ${SEEDS[*]}"
echo "  Tasks: ${TASKS[*]}"
echo "  EHR Encoder: $EHR_ENCODER"
echo "  CXR Encoder: $CXR_ENCODER"
echo "  Dim: $DIM"
echo "  Dropout: $DROPOUT"
echo "  Layer After: $LAYER_AFTER"
echo "  DAFT Activation: $DAFT_ACTIVATION"
echo "  Data Pairs: $DATA_PAIRS"
echo "  EHR N Head: $EHR_N_HEAD"
echo "  EHR N Layers: $EHR_N_LAYERS"
echo "  Pretrained: $PRETRAINED"
echo "  Matched: $MATCHED"
echo "  Use Demographics: $USE_DEMOGRAPHICS"
if [ -n "$CROSS_EVAL" ]; then
    echo "  Cross Eval: $CROSS_EVAL"
fi
echo "==============================="

# Run training for each task
for TASK in "${TASKS[@]}"
do
    run_task_training $TASK
done

echo "All tasks completed successfully!"
echo "Training results are saved in the experiments directory."