#!/bin/bash

echo "Starting 3-seed training for HealNet model on fold 1..."

# Define experiment parameters
BATCH_SIZE=4
LR=0.0001
MAX_LR=0.0001
PATIENCE=10
EPOCHS=50
DROPOUT=0.2
FOLD=1                     # Fixed fold 1
GPU="0,1,2,3"                  # GPU device numbers (multiple GPUs) - 修改为支持多GPU
MATCHED=true
USE_DEMOGRAPHICS=false
CROSS_EVAL=""              # Set to "matched_to_full" or "full_to_matched" if needed
PRETRAINED=true

# HealNet-specific parameters
DEPTH=3                    # Number of fusion layers
LATENT_CHANNELS=256        # Number of latent tokens (l_c)
LATENT_DIM=256             # Dimension of latent tokens (l_d)
CROSS_HEADS=4              # Number of cross-attention heads
LATENT_HEADS=4             # Number of self-attention heads
CROSS_DIM_HEAD=64          # Dimension of each cross-attention head
LATENT_DIM_HEAD=64         # Dimension of each self-attention head
SELF_PER_CROSS_ATTN=1      # Self-attention layers per cross-attention
WEIGHT_TIE_LAYERS=true     # Whether to share weights across layers
SNN=true                   # Whether to use self-normalizing networks
FOURIER_ENCODE_DATA=true   # Whether to use Fourier positional encoding
NUM_FREQ_BANDS=2           # Number of frequency bands
MAX_FREQ=5.0              # Maximum frequency for encoding
FINAL_CLASSIFIER_HEAD=true # Whether to add final classification head
ATTN_DROPOUT=0.2           # Dropout rate for attention layers
FF_DROPOUT=0.2             # Dropout rate for feed-forward layers

# Seeds configuration (including 42)
SEEDS=(42 123 1234)

# Tasks configuration
TASKS=("mortality")

for TASK in "${TASKS[@]}"
do
    echo "Training HealNet model for task: $TASK"
    
    # Set task-specific parameters
    if [ "$TASK" = "phenotype" ]; then
        NUM_CLASSES=25
        INPUT_DIM=49          # Note: HealNet uses 49, not 498
    elif [ "$TASK" = "mortality" ]; then
        NUM_CLASSES=1
        INPUT_DIM=49
    elif [ "$TASK" = "los" ]; then
        NUM_CLASSES=10
        INPUT_DIM=49
    fi
    
    for SEED in "${SEEDS[@]}"
    do
        echo "Training with seed $SEED for task $TASK on fold $FOLD..."
        
        # Build base command
        CMD="python ../main.py \
            --model healnet \
            --mode train \
            --task $TASK \
            --fold $FOLD \
            --batch_size $BATCH_SIZE \
            --lr $LR \
            --patience $PATIENCE \
            --epochs $EPOCHS \
            --dropout $DROPOUT \
            --seed $SEED \
            --input_dim $INPUT_DIM \
            --num_classes $NUM_CLASSES \
            --depth $DEPTH \
            --latent_channels $LATENT_CHANNELS \
            --latent_dim $LATENT_DIM \
            --cross_heads $CROSS_HEADS \
            --latent_heads $LATENT_HEADS \
            --cross_dim_head $CROSS_DIM_HEAD \
            --latent_dim_head $LATENT_DIM_HEAD \
            --self_per_cross_attn $SELF_PER_CROSS_ATTN \
            --num_freq_bands $NUM_FREQ_BANDS \
            --max_freq $MAX_FREQ \
            --attn_dropout $ATTN_DROPOUT \
            --ff_dropout $FF_DROPOUT "
        
        # 处理多GPU参数：将"0,1"转换为["--gpu", "0", "1"]
        if [[ "$GPU" == *","* ]]; then
            # 多GPU情况
            IFS=',' read -ra GPU_ARRAY <<< "$GPU"
            CMD="$CMD --gpu"
            for gpu_id in "${GPU_ARRAY[@]}"; do
                CMD="$CMD $gpu_id"
            done
        else
            # 单GPU情况
            CMD="$CMD --gpu $GPU"
        fi
        
        # Add conditional boolean parameters
        if [ "$PRETRAINED" = "true" ]; then
            CMD="$CMD --pretrained"
        fi
        
        if [ "$MATCHED" = "true" ]; then
            CMD="$CMD --matched"
        fi
        
        if [ "$USE_DEMOGRAPHICS" = "true" ]; then
            CMD="$CMD --use_demographics"
        fi
        
        if [ "$WEIGHT_TIE_LAYERS" = "true" ]; then
            CMD="$CMD --weight_tie_layers"
        fi
        
        if [ "$SNN" = "true" ]; then
            CMD="$CMD --snn"
        fi
        
        if [ "$FOURIER_ENCODE_DATA" = "true" ]; then
            CMD="$CMD --fourier_encode_data"
        fi
        
        if [ "$FINAL_CLASSIFIER_HEAD" = "true" ]; then
            CMD="$CMD --final_classifier_head"
        fi
        
        if [ -n "$CROSS_EVAL" ]; then
            CMD="$CMD --cross_eval $CROSS_EVAL"
        fi
        
        echo "Running command: $CMD"
        eval $CMD
        
        if [ $? -eq 0 ]; then
            echo "Seed $SEED for task $TASK completed successfully!"
        else
            echo "Error: Seed $SEED for task $TASK failed!"
            exit 1
        fi
    done
    
    echo "All 3 seeds completed for task $TASK on fold $FOLD!"
    echo "Task $TASK training and evaluation completed for fold $FOLD with 3 seeds!"
    echo "=================================="
done

echo "All tasks completed successfully!"
echo "Results can be found in experiments/healnet/[task_name]/"
echo "Each seed experiment creates a separate checkpoint and log directory"
echo "Multi-GPU training used: $GPU"