#!/bin/bash

echo "Starting 5-seed training for MedFuse model on fold 1..."

# Define experiment parameters
BATCH_SIZE=16
LR=0.0001
PATIENCE=10
EPOCHS=50
DROPOUT=0.2
FOLD=1                     # Fixed fold 1
GPU=1                      # GPU device number
MATCHED=true
USE_DEMOGRAPHICS=false
CROSS_EVAL=""              # Set to "matched_to_full" or "full_to_matched" if needed

# MedFuse specific parameters
EHR_ENCODER=lstm           # Options: 'lstm', 'transformer', 'drfuse'
CXR_ENCODER=resnet50       # Options: 'resnet50', 'vit_b_16', 'medfuse_cxr'
DIM=256                    # Feature dimension
LAYERS=1                   # Number of LSTM layers
VISION_BACKBONE=resnet50   # Vision backbone
FUSION_TYPE=lstm           # Fusion type: 'early', 'late', 'uni', 'lstm'
PRETRAINED=true            # Use pretrained models
DRFUSE_ENCODER=false       # Use DrFuse encoder components

# Seeds configuration (including 42)
SEEDS=(42 123 1234)

# Tasks configuration
TASKS=("phenotype")

for TASK in "${TASKS[@]}"
do
    echo "Training MedFuse model for task: $TASK"
    
    # Set task-specific parameters
    if [ "$TASK" = "phenotype" ]; then
        NUM_CLASSES=25
        VISION_NUM_CLASSES=25
        INPUT_DIM=498
        LABELS_SET=phenotype
    elif [ "$TASK" = "mortality" ]; then
        NUM_CLASSES=1
        VISION_NUM_CLASSES=1
        INPUT_DIM=498
        LABELS_SET=mortality
    fi
    
    for SEED in "${SEEDS[@]}"
    do
        echo "Training with seed $SEED for task $TASK on fold $FOLD..."
        
        # Build base command
        CMD="python ../main.py \
            --model medfuse \
            --mode train \
            --task $TASK \
            --fold $FOLD \
            --batch_size $BATCH_SIZE \
            --lr $LR \
            --patience $PATIENCE \
            --epochs $EPOCHS \
            --dropout $DROPOUT \
            --seed $SEED \
            --ehr_encoder $EHR_ENCODER \
            --cxr_encoder $CXR_ENCODER \
            --dim $DIM \
            --layers $LAYERS \
            --input_dim $INPUT_DIM \
            --num_classes $NUM_CLASSES \
            --vision_backbone $VISION_BACKBONE \
            --vision_num_classes $VISION_NUM_CLASSES \
            --labels_set $LABELS_SET \
            --fusion_type $FUSION_TYPE \
            --gpu $GPU"
        
        # Add MedFuse specific parameters
        if [ "$PRETRAINED" = "true" ]; then
            CMD="$CMD --pretrained"
        fi
        
        if [ "$DRFUSE_ENCODER" = "true" ]; then
            CMD="$CMD --drfuse_encoder"
        fi
        
        # Add conditional parameters
        if [ "$MATCHED" = "true" ]; then
            CMD="$CMD --matched"
        fi
        
        if [ "$USE_DEMOGRAPHICS" = "true" ]; then
            CMD="$CMD --use_demographics"
        fi
        
        if [ -n "$CROSS_EVAL" ]; then
            CMD="$CMD --cross_eval $CROSS_EVAL"
        fi
        
        echo "Running command: $CMD"
        eval $CMD
        
        if [ $? -eq 0 ]; then
            echo "Seed $SEED for task $TASK completed successfully!"
        else
            echo "Error: Seed $SEED for task $TASK failed!"
            exit 1
        fi
    done
    
    echo "All 5 seeds completed for task $TASK on fold $FOLD!"
    echo "Task $TASK training and evaluation completed for fold $FOLD with 5 seeds!"
    echo "=================================="
done

echo "All tasks completed successfully!"
echo "Results can be found in experiments/medfuse/[task_name]/"
echo "Each seed experiment creates a separate checkpoint and log directory"