from LIReC.lib.pslq_utils import pslq, poly_eval
import mpmath as mp
from random import getrandbits, seed
import matplotlib.pyplot as plt
import numpy as np
from pickle import loads
seed(11324037935649324490, version=2)
REP = 100

plt.rcParams['font.family'] = 'Calibri' # won't work if you don't have calibri installed

def test_pslq(n, d, with_std=False):
    rois = []
    rois2 = []
    print(n, d)
    for i in range(REP):
        mp.mp.prec = d
        coeffs = None
        while not coeffs:
            consts = [mp.mpf(getrandbits(d)) / 2 ** d for i in range(n)]
            # the reduced tolerance here is necessary, see section 4.2 for more info
            coeffs = pslq(consts, None, mp.inf, mp.inf)
            if not coeffs:
                print(consts)
        prec = poly_eval(consts, coeffs, [d], 2)
        substance = len([x for x in coeffs if x]) + sum(x.bit_length() for x in coeffs)
        rois += [prec / substance]
        rois2 += [prec - substance]
    rois = np.array(rois)
    rois2 = np.array(rois2)
    return (rois.mean(), rois.std(), rois2.mean(), rois2.std()) if with_std else (rois.mean(), rois2.mean())

maxn = 10
mind, maxd = 60, 160
N = range(2, maxn)
D = range(mind, maxd + 1, 5)
all_rois = {}
all_rois2 = {}
for n in N:
    for d in D:
        all_rois[n, d], all_rois2[n, d] = test_pslq(n, d)

#all_rois = loads(b'\x80\x04\x95\xdd\t\x00\x00\x00\x00\x00\x00}\x94(K\x02K<\x86\x94G?\xe12\xde\xa5~\xbe\x83K\x02KA\x86\x94G?\xe1\x0c\xd4@f\x86\xa0K\x02KF\x86\x94G?\xe18r>\xa4\xc6\xeaK\x02KK\x86\x94G?\xe0\xe1F\xb7\xdf\x12\x1dK\x02KP\x86\x94G?\xe0\xf3\xf147\xcavK\x02KU\x86\x94G?\xe0\xe6#x\xc7;\xfdK\x02KZ\x86\x94G?\xe0\xd0oO\xb4>\x13K\x02K_\x86\x94G?\xe0\xb5\xc7\x1c\x10\xbf1K\x02Kd\x86\x94G?\xe0\xb8\x94,\xc6\\\x03K\x02Ki\x86\x94G?\xe0\xa7s\x0f3\xc9\xe1K\x02Kn\x86\x94G?\xe0\xa5\xfa\x19\n\xdf\x1dK\x02Ks\x86\x94G?\xe0\xa5\xabQ\xce2\xafK\x02Kx\x86\x94G?\xe0\xa9\x8b\x93!G~K\x02K}\x86\x94G?\xe0\x99\x13\x19\x9a\x1f\xb0K\x02K\x82\x86\x94G?\xe0\x91\xfe\xb6M\xe9-K\x02K\x87\x86\x94G?\xe0\x8a\xdf?A\x9a\xcfK\x02K\x8c\x86\x94G?\xe0\x8d\xd7\xa6\x03\xe4\xe4K\x02K\x91\x86\x94G?\xe0waC\xf6\xb2@K\x02K\x96\x86\x94G?\xe0|\xa8+}\xca3K\x02K\x9b\x86\x94G?\xe0_\x8aI\xd7uiK\x02K\xa0\x86\x94G?\xe0c[97\xf3\xa0K\x03K<\x86\x94G?\xe6\xfbI=;\xe5\xf8K\x03KA\x86\x94G?\xe6\xdd\x10F\xdcU\xe3K\x03KF\x86\x94G?\xe6\xabv4\x97%\x85K\x03KK\x86\x94G?\xe6\xbb\x85\xdan\xbf8K\x03KP\x86\x94G?\xe6\xc0[\x89\xe0\x85VK\x03KU\x86\x94G?\xe6\xac\x00f;A\xf4K\x03KZ\x86\x94G?\xe6\x99M\xb8\x8e=jK\x03K_\x86\x94G?\xe6\x8a_\xe7\x87F\xecK\x03Kd\x86\x94G?\xe6\x9f3\x1b\xcd:\nK\x03Ki\x86\x94G?\xe6:\xe6\x016[\xafK\x03Kn\x86\x94G?\xe6\x81K\xf5\x7fXOK\x03Ks\x86\x94G?\xe6E\xdb\x10\xca%eK\x03Kx\x86\x94G?\xe6A\xf8\xe6ta\xa5K\x03K}\x86\x94G?\xe67\xbb\xbd\xd4\xd9\xd3K\x03K\x82\x86\x94G?\xe66\xe8\x7f\xfe\xe4\xadK\x03K\x87\x86\x94G?\xe6)\x18\x1b\x12\x9f\x83K\x03K\x8c\x86\x94G?\xe6\x19\xd6(4\xc0)K\x03K\x91\x86\x94G?\xe5\xed\xdc\xe5j\r\xb6K\x03K\x96\x86\x94G?\xe6\x0e|\rC\xdd-K\x03K\x9b\x86\x94G?\xe5\xfd\x8f\xfb\xf1\x9f%K\x03K\xa0\x86\x94G?\xe5\xfd\x87\x819\x0bJK\x04K<\x86\x94G?\xeai/S\xf7\xba\x9aK\x04KA\x86\x94G?\xea@"E!\xc2\x11K\x04KF\x86\x94G?\xe9\xa8V\xc9V\xf2nK\x04KK\x86\x94G?\xe9\x9c\x97\x82\xd8\xfd\x01K\x04KP\x86\x94G?\xe9\xc9=Wrc\x98K\x04KU\x86\x94G?\xe9b\x96\xe1\x81\xc4\xaeK\x04KZ\x86\x94G?\xe9W\x10\xcc;\xe2@K\x04K_\x86\x94G?\xe9r\xc6\xbfi\x10\xe1K\x04Kd\x86\x94G?\xe9\x94y!\xa2Z\xadK\x04Ki\x86\x94G?\xe8\xea\xa6\x14\x1e$:K\x04Kn\x86\x94G?\xe9z\x15Ym\xae\xe5K\x04Ks\x86\x94G?\xe9\x03\xa5K\xea\x88hK\x04Kx\x86\x94G?\xe8\xe34.\xd1\x15{K\x04K}\x86\x94G?\xe9!\xb1\xe9\tE\xc6K\x04K\x82\x86\x94G?\xe9V\x8fAw\xad\x84K\x04K\x87\x86\x94G?\xe8\xde\x0b\x94\xc1i:K\x04K\x8c\x86\x94G?\xe8\xf0\x89\x100\xf34K\x04K\x91\x86\x94G?\xe8\xd2r;\xae\t\xcdK\x04K\x96\x86\x94G?\xe8\xc6\x93\x14~\xc0\xc3K\x04K\x9b\x86\x94G?\xe8\xb5\xfe\xe3#c\x08K\x04K\xa0\x86\x94G?\xe8\xa8\xddHV\xfa[K\x05K<\x86\x94G?\xec^{%b\xb6nK\x05KA\x86\x94G?\xeb\xd9\x14r\xb9\x03FK\x05KF\x86\x94G?\xebr\x1a\xcb\xd0\xd1\xe4K\x05KK\x86\x94G?\xea\xe2\x98\xe1\xe5\xe8\nK\x05KP\x86\x94G?\xebm"J%rfK\x05KU\x86\x94G?\xeb\x0b{\x89\xb2\xe6\xd1K\x05KZ\x86\x94G?\xea\xf6\x05\xd0\xa5\x8a\x97K\x05K_\x86\x94G?\xea\xeeZ#\xc0S3K\x05Kd\x86\x94G?\xea\xc0p\xe8"\xdd)K\x05Ki\x86\x94G?\xeb-"\xd0\xa9m\tK\x05Kn\x86\x94G?\xea\xea&\xf9 \xae\xecK\x05Ks\x86\x94G?\xea\x8f{\xe1s\xea}K\x05Kx\x86\x94G?\xeaO\xbc\x11u\xf7\xb8K\x05K}\x86\x94G?\xeaL\x96Qa\x7fVK\x05K\x82\x86\x94G?\xea\xc0w\xd6e@@K\x05K\x87\x86\x94G?\xea\\`\x84\xe29\xceK\x05K\x8c\x86\x94G?\xea\x97\x8a\xc1X\x05CK\x05K\x91\x86\x94G?\xeaw\xe6\x87\xba\xc5}K\x05K\x96\x86\x94G?\xea\xac\xe6;l\xeb\x1cK\x05K\x9b\x86\x94G?\xeav\xe8;\xcb\xc2\xcfK\x05K\xa0\x86\x94G?\xea|\xd9\xcaYy\x85K\x06K<\x86\x94G?\xed\xc0\xef\xf7L\x8c\x9dK\x06KA\x86\x94G?\xec\xac\x02\x18\x85\xa6\xd1K\x06KF\x86\x94G?\xec\xc5\x9e\x0e\xe9\x8c\xf7K\x06KK\x86\x94G?\xec\xab\x7f\x89\xb63oK\x06KP\x86\x94G?\xed\x06][\xb8\x11\xd7K\x06KU\x86\x94G?\xec@J#\\\x81\xb8K\x06KZ\x86\x94G?\xec\xac\xa0N\xea\xd8\x1cK\x06K_\x86\x94G?\xec#v\xb5<JlK\x06Kd\x86\x94G?\xec&\x1b\xfaP\xa8@K\x06Ki\x86\x94G?\xeb\xaa\xcc\x1a)W\x9bK\x06Kn\x86\x94G?\xeb\xa4+\xaeF\x8e\x9dK\x06Ks\x86\x94G?\xebeS\t#\xa74K\x06Kx\x86\x94G?\xeb\xc9+g\xb1\xaa&K\x06K}\x86\x94G?\xec\x13:\xf4\x7f\xc7\x13K\x06K\x82\x86\x94G?\xeb\xa6f\xc1\xde\xa9\xc0K\x06K\x87\x86\x94G?\xeb`*\xbf\xa8&\xffK\x06K\x8c\x86\x94G?\xebw\x0b\xcf\xa0*\x94K\x06K\x91\x86\x94G?\xebZ\xca\xcb\x85\xb08K\x06K\x96\x86\x94G?\xeb<\xa76\x90\xb2\xbaK\x06K\x9b\x86\x94G?\xeb\xa9\x0e\x91\x1aK\xdfK\x06K\xa0\x86\x94G?\xea\xf6\xe0.7(\x1cK\x07K<\x86\x94G?\xed\xadx\x9f\xd2\xce\x14K\x07KA\x86\x94G?\xec\xbfx\xcb\xf8%:K\x07KF\x86\x94G?\xed!5\x97\xcc\x83jK\x07KK\x86\x94G?\xed\x06+^ \x92\xf7K\x07KP\x86\x94G?\xec\xcey\x04\xfc\x99\x13K\x07KU\x86\x94G?\xedu\x02gK\xd0\xddK\x07KZ\x86\x94G?\xec\x9c\xff\xab\x1e&LK\x07K_\x86\x94G?\xec\xbd\x7f%\x02\x06JK\x07Kd\x86\x94G?\xec\xbb\x81\xe7\xea\xa48K\x07Ki\x86\x94G?\xed-\xda\xa1`q1K\x07Kn\x86\x94G?\xed\xd4-\x966\xfceK\x07Ks\x86\x94G?\xec\xd3\xd2RO2\x8cK\x07Kx\x86\x94G?\xec\xa29\xc8k\x0eXK\x07K}\x86\x94G?\xec^Q\x8e\xd5\xb2\xfaK\x07K\x82\x86\x94G?\xec#G\xa9\x9d\x15]K\x07K\x87\x86\x94G?\xecO^.%\xf0qK\x07K\x8c\x86\x94G?\xec\x967\xf8J\x0b\x80K\x07K\x91\x86\x94G?\xecf\xba\x96\xa48\x1fK\x07K\x96\x86\x94G?\xec\xca\x97\x818>\xe4K\x07K\x9b\x86\x94G?\xec|\xfd^\xbe\xa1\x05K\x07K\xa0\x86\x94G?\xeb\xff\xdf\xe6\xd5\xf5\xfaK\x08K<\x86\x94G?\xee\xfbv:\xd4\xa84K\x08KA\x86\x94G?\xef\xbf\x90\x97\xe4\x18\xd1K\x08KF\x86\x94G?\xed\xd9X\xf3\xa2V\xceK\x08KK\x86\x94G?\xed\xfaO\xdej\xab\xa8K\x08KP\x86\x94G?\xed4\x07\xfbx\x14fK\x08KU\x86\x94G?\xec\xec\x11\xeff3\xb8K\x08KZ\x86\x94G?\xed\xc8l\xa1\rT\x12K\x08K_\x86\x94G?\xee\x1e\xf5\xe2\xa9\x0c?K\x08Kd\x86\x94G?\xedkZRs\xcd\xadK\x08Ki\x86\x94G?\xed@5;w9*K\x08Kn\x86\x94G?\xec\xaa\x01\xfe\xf0|\xccK\x08Ks\x86\x94G?\xed53%\x022qK\x08Kx\x86\x94G?\xedU\x0b)\xb1\xd6lK\x08K}\x86\x94G?\xed<wr\x91$\x8fK\x08K\x82\x86\x94G?\xec\xa5\x16"\xe5\x0c\x1aK\x08K\x87\x86\x94G?\xec\xc6\x16\xfc\x9b\xa2\xe5K\x08K\x8c\x86\x94G?\xed\to\xf1\xbb\xb5\nK\x08K\x91\x86\x94G?\xecYV7U\xd2\nK\x08K\x96\x86\x94G?\xed\x08)~S\xf2(K\x08K\x9b\x86\x94G?\xec\xdfq\x9b\xaa\xafEK\x08K\xa0\x86\x94G?\xed\x04\x9b\x97\xc3\xc0NK\tK<\x86\x94G?\xedKF<\xd4\x11\xbbK\tKA\x86\x94G?\xeeB0\xbcp1\xdfK\tKF\x86\x94G?\xeeI\x1d&\x05\x8a\x84K\tKK\x86\x94G?\xeeD|HBp\x16K\tKP\x86\x94G?\xed\xc3I\xc5\xcd\xd0\xf3K\tKU\x86\x94G?\xee3\x92\x93\xe6\xad6K\tKZ\x86\x94G?\xec\xbd\xcb**\x9aHK\tK_\x86\x94G?\xed{=a"\x88\xe4K\tKd\x86\x94G?\xed\xbcIG\x11\xf8WK\tKi\x86\x94G?\xedS\'\xb1:\x8a6K\tKn\x86\x94G?\xed\x1c=\xceJ8\xdfK\tKs\x86\x94G?\xed]\x02\xca\xa4\xd5$K\tKx\x86\x94G?\xed\x83xY\xe0\x88\xd6K\tK}\x86\x94G?\xedAB\xda`\xd9\xaaK\tK\x82\x86\x94G?\xec\xdf3V\xfb\xa3CK\tK\x87\x86\x94G?\xed:8(\x8e\xf7\rK\tK\x8c\x86\x94G?\xed@\xf4\x994\xe9\x85K\tK\x91\x86\x94G?\xec\xc4\x18\x18V\x197K\tK\x96\x86\x94G?\xec\xd8?\xd4\xf2\xe9\xf3K\tK\x9b\x86\x94G?\xed6)\xcf\x12\xc1CK\tK\xa0\x86\x94G?\xec\xc8[%\x10\x9b\xdau.')
#all_rois2 = loads(b'\x80\x04\x95\xdd\t\x00\x00\x00\x00\x00\x00}\x94(K\x02K<\x86\x94G\xc0@\x94z\xe1G\xae\x14K\x02KA\x86\x94G\xc0B\xcb\x85\x1e\xb8Q\xecK\x02KF\x86\x94G\xc0CffffffK\x02KK\x86\x94G\xc0E\xf8Q\xeb\x85\x1e\xb8K\x02KP\x86\x94G\xc0F\xbe\xb8Q\xeb\x85\x1fK\x02KU\x86\x94G\xc0I\x08\xf5\xc2\x8f\\)K\x02KZ\x86\x94G\xc0J9\x99\x99\x99\x99\x9aK\x02K_\x86\x94G\xc0LffffffK\x02Kd\x86\x94G\xc0Me\x1e\xb8Q\xeb\x85K\x02Ki\x86\x94G\xc0O\x9e\xb8Q\xeb\x85\x1fK\x02Kn\x86\x94G\xc0PW\xae\x14z\xe1HK\x02Ks\x86\x94G\xc0QX\xf5\xc2\x8f\\)K\x02Kx\x86\x94G\xc0Q\xce\xb8Q\xeb\x85\x1fK\x02K}\x86\x94G\xc0R\xe8\xf5\xc2\x8f\\)K\x02K\x82\x86\x94G\xc0S|(\xf5\xc2\x8f\\K\x02K\x87\x86\x94G\xc0T\x91\xeb\x85\x1e\xb8RK\x02K\x8c\x86\x94G\xc0U\x10\x00\x00\x00\x00\x00K\x02K\x91\x86\x94G\xc0VB\x8f\\(\xf5\xc3K\x02K\x96\x86\x94G\xc0V\xc5\xc2\x8f\\(\xf6K\x02K\x9b\x86\x94G\xc0X\x05\x1e\xb8Q\xeb\x85K\x02K\xa0\x86\x94G\xc0X\x88\xf5\xc2\x8f\\)K\x03K<\x86\x94G\xc0.\x0f\\(\xf5\xc2\x8fK\x03KA\x86\x94G\xc01\x05\x1e\xb8Q\xeb\x85K\x03KF\x86\x94G\xc02\\(\xf5\xc2\x8f\\K\x03KK\x86\x94G\xc04\x11\xeb\x85\x1e\xb8RK\x03KP\x86\x94G\xc04\xd1\xeb\x85\x1e\xb8RK\x03KU\x86\x94G\xc06\xf8Q\xeb\x85\x1e\xb8K\x03KZ\x86\x94G\xc08\x19\x99\x99\x99\x99\x9aK\x03K_\x86\x94G\xc0:\x14z\xe1G\xae\x14K\x03Kd\x86\x94G\xc0:\xb8Q\xeb\x85\x1e\xb8K\x03Ki\x86\x94G\xc0>\x00\x00\x00\x00\x00\x00K\x03Kn\x86\x94G\xc0=\xee\x14z\xe1G\xaeK\x03Ks\x86\x94G\xc0@h\xf5\xc2\x8f\\)K\x03Kx\x86\x94G\xc0@\xe0\x00\x00\x00\x00\x00K\x03K}\x86\x94G\xc0A\xea=p\xa3\xd7\nK\x03K\x82\x86\x94G\xc0BaG\xae\x14z\xe1K\x03K\x87\x86\x94G\xc0Cu\xc2\x8f\\(\xf6K\x03K\x8c\x86\x94G\xc0D \x00\x00\x00\x00\x00K\x03K\x91\x86\x94G\xc0E\x9e\xb8Q\xeb\x85\x1fK\x03K\x96\x86\x94G\xc0E\xba\xe1G\xae\x14{K\x03K\x9b\x86\x94G\xc0F\xf1\xeb\x85\x1e\xb8RK\x03K\xa0\x86\x94G\xc0Ge\x1e\xb8Q\xeb\x85K\x04K<\x86\x94G\xc0 \xb8Q\xeb\x85\x1e\xb8K\x04KA\x86\x94G\xc0#\x0f\\(\xf5\xc2\x8fK\x04KF\x86\x94G\xc0&\xf0\xa3\xd7\n=qK\x04KK\x86\x94G\xc0(\xfa\xe1G\xae\x14{K\x04KP\x86\x94G\xc0)\n=p\xa3\xd7\nK\x04KU\x86\x94G\xc0-B\x8f\\(\xf5\xc3K\x04KZ\x86\x94G\xc0.\xa3\xd7\n=p\xa4K\x04K_\x86\x94G\xc00&fffffK\x04Kd\x86\x94G\xc008Q\xeb\x85\x1e\xb8K\x04Ki\x86\x94G\xc03\x80\x00\x00\x00\x00\x00K\x04Kn\x86\x94G\xc02+\x85\x1e\xb8Q\xecK\x04Ks\x86\x94G\xc05\x14z\xe1G\xae\x14K\x04Kx\x86\x94G\xc065\xc2\x8f\\(\xf6K\x04K}\x86\x94G\xc06W\n=p\xa3\xd7K\x04K\x82\x86\x94G\xc06\x0f\\(\xf5\xc2\x8fK\x04K\x87\x86\x94G\xc09+\x85\x1e\xb8Q\xecK\x04K\x8c\x86\x94G\xc09u\xc2\x8f\\(\xf6K\x04K\x91\x86\x94G\xc0;B\x8f\\(\xf5\xc3K\x04K\x96\x86\x94G\xc0<(\xf5\xc2\x8f\\)K\x04K\x9b\x86\x94G\xc0=\xb0\xa3\xd7\n=qK\x04K\xa0\x86\x94G\xc0>\xae\x14z\xe1G\xaeK\x05K<\x86\x94G\xc0\x16\xcc\xcc\xcc\xcc\xcc\xcdK\x05KA\x86\x94G\xc0\x1c333333K\x05KF\x86\x94G\xc0\x1f\xeb\x85\x1e\xb8Q\xecK\x05KK\x86\x94G\xc0#\x8a=p\xa3\xd7\nK\x05KP\x86\x94G\xc0"Q\xeb\x85\x1e\xb8RK\x05KU\x86\x94G\xc0%\x94z\xe1G\xae\x14K\x05KZ\x86\x94G\xc0&\x99\x99\x99\x99\x99\x9aK\x05K_\x86\x94G\xc0(\n=p\xa3\xd7\nK\x05Kd\x86\x94G\xc0*\x00\x00\x00\x00\x00\x00K\x05Ki\x86\x94G\xc0(\xbdp\xa3\xd7\n=K\x05Kn\x86\x94G\xc0+B\x8f\\(\xf5\xc3K\x05Ks\x86\x94G\xc0/\x94z\xe1G\xae\x14K\x05Kx\x86\x94G\xc00\xe1G\xae\x14z\xe1K\x05K}\x86\x94G\xc01\xf8Q\xeb\x85\x1e\xb8K\x05K\x82\x86\x94G\xc00\xab\x85\x1e\xb8Q\xecK\x05K\x87\x86\x94G\xc03\x07\xae\x14z\xe1HK\x05K\x8c\x86\x94G\xc02\xa3\xd7\n=p\xa4K\x05K\x91\x86\x94G\xc03\xf5\xc2\x8f\\(\xf6K\x05K\x96\x86\x94G\xc03c\xd7\n=p\xa4K\x05K\x9b\x86\x94G\xc05B\x8f\\(\xf5\xc3K\x05K\xa0\x86\x94G\xc05\x8c\xcc\xcc\xcc\xcc\xcdK\x06K<\x86\x94G\xc0\x0f\x99\x99\x99\x99\x99\x9aK\x06KA\x86\x94G\xc0\x16\xc2\x8f\\(\xf5\xc3K\x06KF\x86\x94G\xc0\x18Q\xeb\x85\x1e\xb8RK\x06KK\x86\x94G\xc0\x1a\n=p\xa3\xd7\nK\x06KP\x86\x94G\xc0\x19(\xf5\xc2\x8f\\)K\x06KU\x86\x94G\xc0\x1f\xa3\xd7\n=p\xa4K\x06KZ\x86\x94G\xc0\x1dQ\xeb\x85\x1e\xb8RK\x06K_\x86\x94G\xc0"333333K\x06Kd\x86\x94G\xc0"\x85\x1e\xb8Q\xeb\x85K\x06Ki\x86\x94G\xc0&B\x8f\\(\xf5\xc3K\x06Kn\x86\x94G\xc0\'W\n=p\xa3\xd7K\x06Ks\x86\x94G\xc0*u\xc2\x8f\\(\xf6K\x06Kx\x86\x94G\xc0(\x8a=p\xa3\xd7\nK\x06K}\x86\x94G\xc0\'\x99\x99\x99\x99\x99\x9aK\x06K\x82\x86\x94G\xc0*\xd7\n=p\xa3\xd7K\x06K\x87\x86\x94G\xc0.\xc2\x8f\\(\xf5\xc3K\x06K\x8c\x86\x94G\xc0.\xb8Q\xeb\x85\x1e\xb8K\x06K\x91\x86\x94G\xc00\x8c\xcc\xcc\xcc\xcc\xcdK\x06K\x96\x86\x94G\xc010\xa3\xd7\n=qK\x06K\x9b\x86\x94G\xc00\x0f\\(\xf5\xc2\x8fK\x06K\xa0\x86\x94G\xc03\xca=p\xa3\xd7\nK\x07K<\x86\x94G\xc0\x11\xeb\x85\x1e\xb8Q\xecK\x07KA\x86\x94G\xc0\x19\xf5\xc2\x8f\\(\xf6K\x07KF\x86\x94G\xc0\x17Q\xeb\x85\x1e\xb8RK\x07KK\x86\x94G\xc0\x1c\n=p\xa3\xd7\nK\x07KP\x86\x94G\xc0\x1b\x8f\\(\xf5\xc2\x8fK\x07KU\x86\x94G\xc0\x19\x00\x00\x00\x00\x00\x00K\x07KZ\x86\x94G\xc0\x1f\xcc\xcc\xcc\xcc\xcc\xcdK\x07K_\x86\x94G\xc0 \xc2\x8f\\(\xf5\xc3K\x07Kd\x86\x94G\xc0 #\xd7\n=p\xa4K\x07Ki\x86\x94G\xc0\x1d\\(\xf5\xc2\x8f\\K\x07Kn\x86\x94G\xc0\x17=p\xa3\xd7\n=K\x07Ks\x86\x94G\xc0"\x00\x00\x00\x00\x00\x00K\x07Kx\x86\x94G\xc0#z\xe1G\xae\x14{K\x07K}\x86\x94G\xc0&=p\xa3\xd7\n=K\x07K\x82\x86\x94G\xc0)\x8a=p\xa3\xd7\nK\x07K\x87\x86\x94G\xc0(#\xd7\n=p\xa4K\x07K\x8c\x86\x94G\xc0&G\xae\x14z\xe1HK\x07K\x91\x86\x94G\xc0(\xbdp\xa3\xd7\n=K\x07K\x96\x86\x94G\xc0&\xbdp\xa3\xd7\n=K\x07K\x9b\x86\x94G\xc0*W\n=p\xa3\xd7K\x07K\xa0\x86\x94G\xc0.\xa3\xd7\n=p\xa4K\x08K<\x86\x94G\xc0\x0cQ\xeb\x85\x1e\xb8RK\x08KA\x86\x94G\xc0\x01333333K\x08KF\x86\x94G\xc0\x14\x00\x00\x00\x00\x00\x00K\x08KK\x86\x94G\xc0\x14333333K\x08KP\x86\x94G\xc0\x19\xd7\n=p\xa3\xd7K\x08KU\x86\x94G\xc0\x1f\xb8Q\xeb\x85\x1e\xb8K\x08KZ\x86\x94G\xc0\x17\x14z\xe1G\xae\x14K\x08K_\x86\x94G\xc0\x13\xe1G\xae\x14z\xe1K\x08Kd\x86\x94G\xc0\x1b\xeb\x85\x1e\xb8Q\xecK\x08Ki\x86\x94G\xc0\x1eG\xae\x14z\xe1HK\x08Kn\x86\x94G\xc0#aG\xae\x14z\xe1K\x08Ks\x86\x94G\xc0 \x0f\\(\xf5\xc2\x8fK\x08Kx\x86\x94G\xc0!\n=p\xa3\xd7\nK\x08K}\x86\x94G\xc0!\xa3\xd7\n=p\xa4K\x08K\x82\x86\x94G\xc0&\xbdp\xa3\xd7\n=K\x08K\x87\x86\x94G\xc0%\xe6fffffK\x08K\x8c\x86\x94G\xc0$Q\xeb\x85\x1e\xb8RK\x08K\x91\x86\x94G\xc0*#\xd7\n=p\xa4K\x08K\x96\x86\x94G\xc0%aG\xae\x14z\xe1K\x08K\x9b\x86\x94G\xc0(\x8a=p\xa3\xd7\nK\x08K\xa0\x86\x94G\xc0&W\n=p\xa3\xd7K\tK<\x86\x94G\xc0\x19Q\xeb\x85\x1e\xb8RK\tKA\x86\x94G\xc0\x15G\xae\x14z\xe1HK\tKF\x86\x94G\xc0\x11\xe1G\xae\x14z\xe1K\tKK\x86\x94G\xc0\x13\x00\x00\x00\x00\x00\x00K\tKP\x86\x94G\xc0\x17\x1e\xb8Q\xeb\x85\x1fK\tKU\x86\x94G\xc0\x14\xeb\x85\x1e\xb8Q\xecK\tKZ\x86\x94G\xc0!B\x8f\\(\xf5\xc3K\tK_\x86\x94G\xc0\x1d\x00\x00\x00\x00\x00\x00K\tKd\x86\x94G\xc0\x18\xae\x14z\xe1G\xaeK\tKi\x86\x94G\xc0\x1e\xe1G\xae\x14z\xe1K\tKn\x86\x94G\xc0!\x00\x00\x00\x00\x00\x00K\tKs\x86\x94G\xc0 \x19\x99\x99\x99\x99\x9aK\tKx\x86\x94G\xc0\x1fffffffK\tK}\x86\x94G\xc0$\x05\x1e\xb8Q\xeb\x85K\tK\x82\x86\x94G\xc0%\x1e\xb8Q\xeb\x85\x1fK\tK\x87\x86\x94G\xc0#Q\xeb\x85\x1e\xb8RK\tK\x8c\x86\x94G\xc0#\xfa\xe1G\xae\x14{K\tK\x91\x86\x94G\xc0\'\xeb\x85\x1e\xb8Q\xecK\tK\x96\x86\x94G\xc0\'\xf0\xa3\xd7\n=qK\tK\x9b\x86\x94G\xc0%\xdc(\xf5\xc2\x8f\\K\tK\xa0\x86\x94G\xc0)8Q\xeb\x85\x1e\xb8u.')

# running the following code with all_rois2 instead showcases the difference-based RoI implied in Elimelech et al.
rois = np.array([[all_rois[n,d] for d in D] for n in N])

fig, ax = plt.subplots(figsize=(4,2))
im = ax.imshow(rois, vmin=0.5, vmax=1)
cbar = ax.figure.colorbar(im, ax=ax)
cbar.ax.set_yticks(np.arange(0.5,1.01,0.1))
cbar.ax.set_ylabel('average RoI', rotation=90, va="top")

ax.set_xticks(np.arange(len(D)), labels=[d if d % 20 == 0 else '' for d in D])
ax.set_xlabel('binary digits $\\it{d}$')
ax.set_yticks(np.arange(len(N)), labels=N)
ax.set_ylabel('numbers $\\it{n}$')

plt.setp(ax.get_xticklabels(), rotation=45, ha="right", rotation_mode="anchor")
plt.title('(a)', fontweight='bold')
fig.tight_layout()
plt.show()

seed(324037935649324490, version=2)
maxn = 20
N = range(2, maxn)
all_rois3 = []
for n in N:
    all_rois3 += [test_pslq(n, 50 + 5 * n, True)[:2]]

#all_rois3 = loads(b'\x80\x04\x95*\x03\x00\x00\x00\x00\x00\x00]\x94(\x8c\x15numpy.core.multiarray\x94\x8c\x06scalar\x94\x93\x94\x8c\x05numpy\x94\x8c\x05dtype\x94\x93\x94\x8c\x02f8\x94\x89\x88\x87\x94R\x94(K\x03\x8c\x01<\x94NNNJ\xff\xff\xff\xffJ\xff\xff\xff\xffK\x00t\x94bC\x08K\x11jD\rR\xe1?\x94\x86\x94R\x94h\x03h\tC\x087e\xf1\xd7e\xa7\xa3?\x94\x86\x94R\x94\x86\x94h\x03h\tC\x08\x08\xfa?M\xa0\xff\xe6?\x94\x86\x94R\x94h\x03h\tC\x08Dl\xfb\xbdq^\xad?\x94\x86\x94R\x94\x86\x94h\x03h\tC\x08}\x8fBB~,\xea?\x94\x86\x94R\x94h\x03h\tC\x08\xa6V\nNI2\xb7?\x94\x86\x94R\x94\x86\x94h\x03h\tC\x08\xa6\xf7\x94*S\xc8\xeb?\x94\x86\x94R\x94h\x03h\tC\x08W2\xd0\x91\xf4\xad\xbc?\x94\x86\x94R\x94\x86\x94h\x03h\tC\x08#|\xfb\xa2\x82H\xec?\x94\x86\x94R\x94h\x03h\tC\x083\xc56z\x0e\xe9\xbf?\x94\x86\x94R\x94\x86\x94h\x03h\tC\x08\xd4\xc96\xf3\x01\x0f\xee?\x94\x86\x94R\x94h\x03h\tC\x08a\x89\x07\xfaD1\xbe?\x94\x86\x94R\x94\x86\x94h\x03h\tC\x08R\x1f\x8f\xdb\x1fP\xed?\x94\x86\x94R\x94h\x03h\tC\x08\xedI#\xe3\x93\xdb\xc3?\x94\x86\x94R\x94\x86\x94h\x03h\tC\x08\x83P\xd3\xd8\x84_\xed?\x94\x86\x94R\x94h\x03h\tC\x08\xe6\xba\xc9\x87#\n\xc4?\x94\x86\x94R\x94\x86\x94h\x03h\tC\x08{W\x13|e\xe4\xec?\x94\x86\x94R\x94h\x03h\tC\x08L\xe6-\xe5;\xbd\xc4?\x94\x86\x94R\x94\x86\x94h\x03h\tC\x08\x97h\xd7\x85\x8a\x05\xee?\x94\x86\x94R\x94h\x03h\tC\x08*\xc0\rJ\xbe\x84\xc5?\x94\x86\x94R\x94\x86\x94h\x03h\tC\x08\xea\xd5\x19\x19\xac\x9e\xec?\x94\x86\x94R\x94h\x03h\tC\x08\xaf\xc3\xcd\x8f\xef\xea\xc3?\x94\x86\x94R\x94\x86\x94h\x03h\tC\x08\xe6\xc3a\xbbw\x03\xec?\x94\x86\x94R\x94h\x03h\tC\x08\x86\x86$\x9b)$\xc8?\x94\x86\x94R\x94\x86\x94h\x03h\tC\x08\x1c!\x86\x80\xfb\xc7\xec?\x94\x86\x94R\x94h\x03h\tC\x08u\xe4\xbf/\xce\x82\xc6?\x94\x86\x94R\x94\x86\x94h\x03h\tC\x08M;\xc7\xf1\xf7\x02\xed?\x94\x86\x94R\x94h\x03h\tC\x08\xdb\xba\x14\xb1\x8f\xba\xc8?\x94\x86\x94R\x94\x86\x94h\x03h\tC\x08\x05\x93\x8f\t\xe5\x86\xeb?\x94\x86\x94R\x94h\x03h\tC\x08\x94k\xb8|\xffB\xc7?\x94\x86\x94R\x94\x86\x94h\x03h\tC\x08\x9f\x16\xf1Q\x91\x80\xeb?\x94\x86\x94R\x94h\x03h\tC\x08YL\xea\xe8g\xfd\xc6?\x94\x86\x94R\x94\x86\x94h\x03h\tC\x08,\xe6\xacs\x96o\xea?\x94\x86\x94R\x94h\x03h\tC\x08\xbdz^"\xb4\t\xc7?\x94\x86\x94R\x94\x86\x94h\x03h\tC\x08\x1d\\`\xb6\xfb\xda\xeb?\x94\x86\x94R\x94h\x03h\tC\x08wvjw|\xd7\xc3?\x94\x86\x94R\x94\x86\x94e.')

fig, (ax1, ax2) = plt.subplots(2, 1, sharex=True, figsize=(5,5), height_ratios=(1,2))
ax1.tick_params(bottom=False,top=False)
ax1.spines.bottom.set_visible(False)
ax2.spines.top.set_visible(False)

kwargs = dict(marker=[(-1, -.5), (1, .5)], markersize=12,
              linestyle="none", color='k', mec='k', mew=1, clip_on=False)
ax1.plot([0, 1], [0, 0], transform=ax1.transAxes, **kwargs)
ax2.plot([0, 1], [1, 1], transform=ax2.transAxes, **kwargs)

roi1 = 2310.5
roi2 = 5905.2
ax1.plot(N, [roi1 for n in N], '--', c='k')
ax1.plot(N, [roi2 for n in N], '--', c='k')
ax1.annotate(f'RoI = {roi1}', (2, roi1*1.1), fontsize='8')
ax1.annotate(f'RoI = {roi2}', (2, roi2*1.1), fontsize='8')
ax1.set_yscale('log')
ax1.set_ylim((1000,10000))
ax1.axhspan(1000, 10000, color='green', alpha=0.1)

ax2.errorbar(N, [y[0] for y in all_rois3], yerr=[3*y[1] for y in all_rois3], capsize=3, c='lightgray')
ax2.errorbar(N, [y[0] for y in all_rois3], yerr=[2*y[1] for y in all_rois3], capsize=3, c='gray')
ax2.errorbar(N, [y[0] for y in all_rois3], yerr=[y[1] for y in all_rois3], capsize=3, c='black')
ax2.plot(N, [1.25 for n in N], '--', c='lightgreen')
ax2.plot(N, [1.5 for n in N], '--', c='green')
ax2.set_xlabel('numbers $\\it{n}$')
ax2.set_ylabel('average RoI')
ax2.set_xticks(N)
ax2.set_ylim((0.5, 1.6))
ax2.axhspan(1.25, 1.5, color='lightgreen', alpha=0.1)
ax2.axhspan(1.5, 2, color='green', alpha=0.1)
ax2.annotate('~2 sigma, minimal confidence', (2, 1.35), fontsize='8')
ax2.annotate('>3 sigma, moderate confidence', (2, 1.525), fontsize='8')
ax1.set_title('(b)', fontweight='bold')

fig.tight_layout()
plt.show()
