# The Ramanujan Library
Welcome to the supplementary code for the paper **The Ramanujan Library - Automated Discovery on the Hypergraph of Integer Relations**. This README describes how to use the code to retrieve information about the constants and relations shown in figure 4, and shown in more detail in appendix F.
This repository also contains code for independently running the code used to generate the experimental data shown in figures 3 and 6.

# Installation
This code requires [Python3.8.10](https://www.python.org/downloads/release/python-3810/) or later. Once Python is installed, extract the code in your preferred location, start a command line with administrator permissions, navigate to the code's directory, and run `python setup.py install`. This ensures that all dependencies are installed, and your Python can run the code.
After installing the dependencies, if you have an IDE that supports Python, you may use it, otherwise an interactive Python environment (preferably started inside the code directory) will also work.

## Optional Installation
If you are interested in re-running the experiment code, additional dependencies are required. These dependencies are found in the `req_extra.txt` file, and installed using the `pip install -r req_extra.txt` command.

Another optional step is to set up a real [PostgreSQL](https://www.postgresql.org/) database for the code to connect to. To preserve the the anonymity of the authors, no connection to such database is made in this version of the code, and instead the code fetches data from a cached copy of the database. If you are interested in setting up your own PostgreSQL database and connecting to it, you may input its credentials inside `LIREC.db.access.DBConnection.__init__`. Once you do, run the `create.py` script on your end to initialize the database with the appropriate tables, and named constants. This also allows you to use the `db.session` to make [SQLAlchemy](https://docs.sqlalchemy.org/en/20/)-style queries against the database freely (`db.session` is nonfunctional otherwise, since there is no connection).

# Usage
Once you have a suitable Python environment set up, with the required dependencies, here are some things you can do with the code:

## Querying the hypergraph
```python
from LIReC.db.access import db
```
This line imports a variable named `db`, granting access to a database wrapper accessing the hypergraph of integer relations. Everything to do with the hypergraph is done through this `db` variable.

### Named constants
```python
from LIReC.db.access import db
nameds = db.constants
```
Retrieves a list of all of the named constants in the hypergraph. These include $\pi$, $e$, $ln$ and many more. A full listing can be found inside `LIReC.lib.calculator.Constants`, and the specification of the list elements can be found in `LIReC.db.models.NamedConstant`.

For a more simplified API:
* `db.names()` retrieves the names of all of the named constants. 
* `db.names_with_descriptions()` is the same as `db.names()`, but each element of the list is a tuple of two strings. The first is the name of a constant, and the second is a short description of the constant.
* `db.describe(name)` retrieves the description of the constant with the name `name`, or `None` if `name` is not recognized.

### Continued Fractions
```python
from LIReC.db.access import db
cfs = db.cfs
```
Retrieves a list of all C-transform continued fractions in the hypergraph. The specification of the list elements can be found in `LIReC.db.models.PcfCanonicalConstant`.

### Relations
```python
from LIReC.db.access import db
rels = db.relations()
```
Retrieves a listing of all polynomial relations in the database. The specification of the list list elements can be found in `LIReC.lib.pslq_utils.PolyPSLQRelation`.
The `PolyPSLQRelation`s returned make use of `LIReC.db.access.DualConstant`s to backreference `LIReC.db.models.Constant` rows in the database, which allows for retrieving more information about each constant in the relations using its `const_id`.

### Numeric Identification
```python
from LIReC.db.access import db
db.identify(...)
```
The `db.identify` function allows one to check one (or more) numeric values against the known "famous" constants in the database for the existence of polynomial relations. This is done in three steps:
- First, any decimal expressions within `values` (see below) are tested against each other for relation. This can be skipped by inputting `first_step=False` (see below).
- If nothing was found yet, all expressions in `values` are tested together for a relation.
- Finally, if still nothing was found and `wide_search` is truthy (see below), executes an iterative search against all named constants in the database, stopping at the first relation found. 

The inputs are:
1. `values`: List that contains any combination of the following:
   - Numeric values. It is not recommended to use python's native `float` type as it cannot reach high precision. Instead, use `str` (or `decimal.Decimal` or `mpmath.mpf`) for better results. Each such value will be given a name `c{i}` where `i` is its position in `values`, and these are assumed to all be accurate up to the least accurate value among them (unless `min_prec` is specified, see below). If this is not the case, input less digits or explicitly specify `min_prec`.
   - Strings that represent famous constants. For a full list of possible names try `db.names`. Also try `db.names_with_descriptions` for a full list of names along with a short description of each name, or `db.describe` to conveniently fish out the description of any one name in the database (if it exists).
   - `Sympy` expressions that involve one or more famous constants. Can be inputted either in string form or as `sympy.Expr` form.
2. `degree`: The maximal degree of the relation. Each relation is defined as the coefficients of a multivariate polynomial, where each monomial (in multi-index notation) is of the form a_alpha \* x \*\* alpha. Then, the degree is the L1 norm on alpha. Defaults to 2.
3. `order`: The maximal order of the relation, which is the L-infinity norm on each alpha (see degree). Defaults to 1.
4. `min_prec`: The minimal digital precision expected of the numbers in `values`. Can be omitted, in which case it will be inferred from `values`, see above.
5. `min_roi`: Given a vector of random numbers, the total amount of digits in an integer relation on them is usually about the same as the working precision. To differentiate between such "garbage" cases and potentially substantive results, any relation found will be required to have `min_roi` times less digits than the working precision to be returned. Defaults to `2`.
6. `isolate`: Modifies the way results will be printed:
   - If set to `False`, results will be printed in the form `expr = 0 (precision)`, where `precision` specifies how many digits the computer knows for sure satisfy the equality. This is the default option.
   - If set to `True`, will take the first recognized named constant and isolate it as a function of all other constants in the relations it participates in, resulting in the form `const = expr (precision)`.
   - If set to a number `i`, will similarly isolate `values[i]` as a function of everything else.
7. `first_step`: Whether or not to perform the first step of identification (see above). Defaults to `True`.
8. `wide_search`: If truthy, enables the wide search (see above). Also, if this is a `list` (or any kind of collection) of positive integers, the search is limited to subsets of sizes contained in `wide_search`. Do note that the wide search may take a long time, so Ctrl+C it if you have to. Defaults to `False`.
9. `verbose`: If set to `True`, will print various messages regarding the input/output and progress. Defaults to `False`.

The result of calling `identify` is a list of `pslq_util.PolyPSLQRelation` objects. These can be casted into a string that represents the relation and its estimated precision. For instance, try this code snippet:
```python
from LIReC.db.access import db
import mpmath as mp
mp.mp.dps=400
results = db.identify([(100-mp.zeta(3))/(23+5*mp.zeta(3)), 'Zeta3']) # first run should take a few seconds to query the db...
print([str(x) for x in results])
results = db.identify([mp.mpf(1)/3 - mp.zeta(3), 'Zeta3']) # this works
print([str(x) for x in results])
results = db.identify([1/3 - mp.zeta(3), 'Zeta3']) # this doesn't! don't let bad floats pollute your numbers!
print([str(x) for x in results])
```
## Search job - Automated enrichment of the hypergraph
The code we used for automated enrichment of the hypergraph can be found in `LIReC.jobs.job_poly_pslq_v1`. However, it is nonfunctional without connection to a real PostgreSQL database (see [Optional Installation](#optional-installation) for more information).

## Experiments
### Return on Investment experiment
From a command line run `python LIReC/experiments/roi.py`. Alternatively for a shorter runtime consider investigating the code manually and skipping the calculations, directly viewing the results generated from the raw data in the `all_rois,all_rois2,all_rois3` variables (requires uncommenting the respective lines).

### PSLQ rebounding experiment
From a command line run `python LRIeC/experiments/rebound.py`. Note that experiment relies on manually transcribed errors that have been debug-printed by PSLQ. These can be found in the `errors` variable.
