# import sys
# sys.path.append("/workspace/Project_EvoWire/EVAttack/adversarial_training")

import os
import yaml
from ml_collections import ConfigDict
from tqdm import tqdm
from copy import deepcopy
import numpy as np
import logging

import torch
import torch_geometric

from utils.data import load_dataset, make_dataset_splits, load_dataset_splits, check_dataset_valid
from utils.split import SplitManager, node_induced_subgraph
from utils.storage import TensorHash
from utils.model import create_robust_model_instance, load_robust_model_instance
from utils.attack import load_attack_class, attack_storage_label, create_attack_instance

from robust_diffusion.data import SparseGraph
from robust_diffusion.data import count_edges_for_idx
from robust_diffusion.train import train, train_inductive

from robust_diffusion.helper.utils import calculate_loss
from robust_diffusion.models import create_model, GPRGNN, DenseGPRGNN, ChebNetII
from robust_diffusion.models.gprgnn import GPR_prop

from sacred import Experiment

experiment = Experiment("VanillaAttack")

@experiment.config
def default_config():
    ## Experiment configs
    dataset_name = "cora_ml"

    # model_name in ["GCN", "DenseGCN", "GAT", "GPRGNN", "DenseGPRGNN", "APPNP", "ChebNetII", "SoftMedian_GDC"]
    model_name = "GCN"
    n_splits = 10

    training_split = None
    validation_split = None
    training_split_type = None
    validation_split_type = None
    test_split = None
    test_split_type = None

    model_params = None

    # attack_name in ["PRBCD", "LRBCD", "EvaAttack", "Evafast", "PGD"] 
    train_attack_name = "PRBCD"
    loss_type = 'tanhMargin'
    
    inductive = False
    self_training = True
    robust_training = True
    robust_epsilon = 0.2

    make_undirected = True
    binary_attr = False
    data_device = 0

    device = 0
    seed = 0
    validate_every = 10

    train_params = None
    train_attack_params = None
    val_attack_params = None


@experiment.automain
def run(dataset_name, model_name, n_splits, training_split, 
        validation_split, training_split_type, validation_split_type, 
        test_split, test_split_type, train_params, loss_type, validate_every, make_undirected, binary_attr, data_device,
        model_params, train_attack_name, train_attack_params, val_attack_params , inductive, self_training ,robust_training, robust_epsilon):
    
    print("Experiment Started")

    ## Loading general configs (like dataset_root, etc.) and initial parameters
    general_config = yaml.safe_load(open("conf/general-config.yaml"))
    default_dataset_configs = yaml.safe_load(open("conf/data-configs.yaml")).get("configs").get("default")
    default_model_configs = yaml.safe_load(open("conf/model-configs.yaml")).get("configs")
    default_attack_configs = yaml.safe_load(open("conf/attack-configs.yaml")).get("configs")
    
    if train_params is None:
        file_config = yaml.safe_load(open("conf/model-configs.yaml"))

        # print(file_config.get("robust_configs"))
        train_params = file_config.get(
            "robust_configs", dict()).get("training", dict(
            lr=1e-2,
            weight_decay=1e-3,
            patience=200,
            max_epochs=3000
        ))
    
    val_attack_name = train_attack_name
    if train_attack_params is None:
        if train_attack_name in ["LRBCD", "EVAFAST", "PRBCD"]:
            train_attack_params = yaml.safe_load(open("conf/model-configs.yaml")).get(
                "robust_configs", dict()).get("train_attack_params").get(train_attack_name)
        else:
            raise(ValueError(f"{train_attack_name} Invalid attack name"))
    
    if val_attack_params is None:
        if val_attack_name in ["LRBCD", "EVAFAST", "PRBCD"]:
            val_attack_params = yaml.safe_load(open("conf/model-configs.yaml")).get(
                "robust_configs", dict()).get("val_attack_params").get(val_attack_name)
        else:
            raise(ValueError(f"{val_attack_name} Invalid attack name"))

    # extracting configs 
    dataset_root = general_config.get("dataset_root", "data/")
    splits_root = general_config.get("splits_root", "splits/")
    models_root = general_config.get("models_root", "models/")
    results_root = general_config.get("results_root", "results/")
    reports_root = general_config.get("reports_root", "reports/")
        
    device = 'cuda' if torch.cuda.is_available() else 'cpu'

    print("Experiment Started")
    # Trains the specified model on the given graph and saves the model artifacts, and the splits.

    print("Loading dataset =", dataset_name)

    dataset_splits = [
        split_record for split_record in os.listdir(splits_root) 
        if split_record.split("-")[0] == dataset_name 
        and check_dataset_valid(split_record=split_record, training_split=training_split,
                                validation_split=validation_split, training_split_type=training_split_type, 
                                validation_split_type=validation_split_type, test_split=test_split, 
                                test_split_type=test_split_type, splits_root=splits_root)]
    creating_splits = max(n_splits - len(dataset_splits), 0)

    if creating_splits > 0:
        raise ValueError("Not enough splits for the dataset. Create the splits by running training scripts.")

    # creating remaining needed dataset splits
    print(f"Found {len(dataset_splits)} splits!")

    print(f"Loading pretrained {model_name} model on {dataset_name} dataset for {n_splits} splits")

    clean_accs = []
    pert_accs = []
    for split_file in tqdm(dataset_splits[:n_splits]):
        split_code = split_file.split("-")[1].replace(".pt", "")

        # load dataset splits
        data = load_dataset_splits(
            dataset_name, split_code, inductive=inductive, 
            dataset_root=dataset_root, splits_root=splits_root, device=device)

        training_attr = data["training_attr"]
        training_adj = data["training_adj"]
        validation_attr = data["validation_attr"]
        validation_adj = data["validation_adj"]
        labels = data["labels"]
        training_idx = data["training_idx"]
        validation_idx = data["validation_idx"]
        test_attr = data["test_attr"]
        test_adj = data["test_adj"]
        unlabeled_mask = data["unlabeled_mask"]
        test_mask = data["test_mask"]
        dataset_info = data["dataset_info"]
        split_name = data["split_name"]
        data_config = data["config"]

        try:
            model_instance = load_robust_model_instance(
                model_name=model_name, model_params=model_params, 
                dataset_info=dataset_info, 
                test_attr=test_attr, test_adj=test_adj, labels=labels, test_mask=test_mask, unlabeled_mask=unlabeled_mask,
                split_name=split_name, inductive=inductive,
                models_root=models_root, self_training=self_training, 
                robust_training=robust_training, train_attack_name=train_attack_name, robust_epsilon=robust_epsilon,
                default_model_configs=default_model_configs, suffix='', device=device)
            print("model loaded successfully")

        except FileNotFoundError:
            print("Model not found. Run training scripts to train the model.")
            model_instance = create_robust_model_instance(
                model_name=model_name, model_params=model_params, dataset_info=dataset_info, 
                training_attr=training_attr, training_adj=training_adj, 
                validation_attr=validation_attr, validation_adj=validation_adj,
                labels=labels, training_idx=training_idx, validation_idx=validation_idx, train_attack_name=train_attack_name,
                test_attr=test_attr, test_adj=test_adj, test_mask=test_mask, unlabeled_mask=unlabeled_mask, inductive=inductive,
                split_name=split_name, train_params=train_params, train_attack_params=train_attack_params, val_attack_params=val_attack_params,
                make_undirected=make_undirected,
                models_root=models_root, default_model_configs=default_model_configs, suffix='',
                self_training=self_training, robust_training=robust_training, robust_epsilon=robust_epsilon,
                validate_every=validate_every, loss_type=loss_type, binary_attr=binary_attr,
                device=device)
            
        model = model_instance["model"]
        model_params = model_instance["model_params"]
        model_storage_name = model_instance["model_storage_name"]
        clean_acc = model_instance["clean_accuracy"]
        clean_accs.append(clean_acc)


    mean_clean_acc = torch.mean(torch.tensor(clean_accs))
    std_clean_acc = torch.std(torch.tensor(clean_accs))

    print(f"Mean clean accuracy on clean dateset: {mean_clean_acc:.4f} $\\pm$ {std_clean_acc:.4f}")
    # print(f"Mean perturbed accuracy: {mean_pert_acc:.4f} $\\pm$ {std_pert_acc:.4f}")
    print("Experiment Finished")

