import os
import yaml
from tqdm import tqdm
import torch

from utils.data import make_arxiv_dataset_splits, load_arxiv_dataset_splits
from utils.model import load_model_instance, create_model_instance
from sacred import Experiment

experiment = Experiment("Vanilla Training of the ogbn-arxiv dataset")

@experiment.config
def default_config():
    ## Experiment configs
    dataset_name = "ogbn-arxiv"
    # model_name in ["GPRGNN", "APPNP"]
    model_name = "GPRGNN"
    n_splits = 10
    model_params = None

    inductive = True

@experiment.automain
def run(dataset_name, model_name, n_splits, model_params, inductive):

    assert dataset_name == "ogbn-arxiv"

    general_config = yaml.safe_load(open("conf/general-config.yaml"))
    default_model_configs = yaml.safe_load(open("conf/model-configs.yaml")).get("arxiv_configs")
    dataset_root = general_config.get("dataset_root", "data/")
    splits_root = general_config.get("splits_root", "splits/")
    models_root = general_config.get("models_root", "models/")
    device = 'cuda' if torch.cuda.is_available() else 'cpu'

    try:
        dataset_splits = [split_record for split_record in os.listdir(splits_root) 
                          if split_record.split("-")[0] == dataset_name.replace("-", "_")]
    except FileNotFoundError:
        dataset_splits = []

    creating_splits = max(n_splits - len(dataset_splits), 0)
    print(f"Found {len(dataset_splits)} splits, creating {creating_splits} more splits")
    for i in tqdm(range(creating_splits)):
        torch.cuda.empty_cache()
        make_arxiv_dataset_splits(dataset_name)

    dataset_splits = [split_record for split_record in os.listdir(splits_root) 
                      if split_record.split("-")[0] == dataset_name.replace("-", "_")][:n_splits]
    print(f"Training {model_name} model on {dataset_name} dataset for {n_splits} splits")

    accs = []
    for split_file in tqdm(dataset_splits):
        split_code = split_file.split("-")[1].replace(".pt", "")

        dataset = load_arxiv_dataset_splits(
            dataset_name, split_code, inductive=inductive, 
            dataset_root=dataset_root, splits_root=splits_root, device=device)
        
        training_attr = dataset["training_attr"]
        training_adj = dataset["training_adj"]
        validation_attr = dataset["validation_attr"]
        validation_adj = dataset["validation_adj"]
        test_attr = dataset["test_attr"]
        test_adj = dataset["test_adj"]
        labels = dataset["labels"]
        training_idx = dataset["training_idx"]
        validation_idx = dataset["validation_idx"]
        unlabeled_mask = dataset["unlabeled_mask"]
        test_mask = dataset["test_mask"]
        split_name = dataset["split_name"]
        dataset_info = dataset["dataset_info"]

        try:
            model_instance = load_model_instance(
                model_name=model_name, model_params=model_params, 
                test_attr=test_attr, test_adj=test_adj, labels=labels, test_mask=test_mask, unlabeled_mask=unlabeled_mask,
                split_name=split_name, dataset_info=dataset_info, inductive=inductive, models_root=models_root,
                default_model_configs=default_model_configs, device=device)
        except FileNotFoundError as e:
            print(e)
            print("Creating model from scratch")
            model_instance = create_model_instance(
                model_name=model_name, model_params=model_params, dataset_info=dataset_info, 
                training_attr=training_attr, training_adj=training_adj, 
                validation_attr=validation_attr, validation_adj=validation_adj,
                labels=labels, training_idx=training_idx, validation_idx=validation_idx,
                test_attr=test_attr, test_adj=test_adj, test_mask=test_mask, unlabeled_mask=unlabeled_mask,
                inductive=inductive, split_name=split_name,
                models_root=models_root, 
                default_model_configs=default_model_configs, 
                device=device)

        acc = model_instance["accuracy"]
        accs.append(acc)
    
    acc_mean = torch.mean(torch.tensor(accs))
    acc_std = torch.std(torch.tensor(accs))

    print(f"Mean accuracy: {acc_mean}, std: {acc_std}")
    print("Experiment Finished")

