# Copyright (c) Facebook, Inc. and its affiliates.
# # https://pytorch.org/tutorials/beginner/transformer_tutorial.html
"""
Positional encodings for the transformer.
"""
import math

import torch
from torch import nn, Tensor


class PositionalEncoding(nn.Module):

    def __init__(self, d_model: int, dropout: float = 0.1, max_len: int = 5000):
        super().__init__()
        self.dropout = nn.Dropout(p=dropout)

        position = torch.arange(max_len).unsqueeze(1)
        div_term = torch.exp(torch.arange(0, d_model, 2) * (-math.log(10000.0) / d_model))
        self.pe = torch.zeros(max_len, 1, d_model)
        self.pe[:, 0, 0::2] = torch.sin(position * div_term)
        self.pe[:, 0, 1::2] = torch.cos(position * div_term)
        # self.register_buffer('pe', pe)

    # def forward(self, x: Tensor) -> Tensor:
    #     """
    #     Arguments:
    #         x: Tensor, shape ``[seq_len, batch_size, embedding_dim]``
    #     """
    #     x = x + self.pe[:x.size(0)]
    #     return self.dropout(x)

    def forward(self, x: Tensor) -> Tensor:
        """
        Arguments:
            x: Tensor, shape ``->[seq_len, batch_size, embedding_dim]``
        """
        x = x.permute(1, 0, 2) # b l c-> l b c

        self.pe = self.pe.to(x.device)
        pe = self.pe[:x.size(0)] #[L b c]
        # pe = self.pe[:x.size(0)].permute(1, 0, 2) #[1,L,c]
        return pe