# Copyright (c) Facebook, Inc. and its affiliates.
# # https://pytorch.org/tutorials/beginner/transformer_tutorial.html
"""
Positional encodings for the transformer.
"""
import math

import torch
from torch import nn, Tensor


class PositionalEncoding(nn.Module):

    def __init__(self, d_model: int, dropout: float = 0.1, max_len: int = 5000):
        super().__init__()
        self.dropout = nn.Dropout(p=dropout)

        position = torch.arange(max_len).unsqueeze(1)
        div_term = torch.exp(torch.arange(0, d_model, 2) * (-math.log(10000.0) / d_model))
        self.pe = torch.zeros(max_len, 1, d_model)
        self.pe[:, 0, 0::2] = torch.sin(position * div_term)
        self.pe[:, 0, 1::2] = torch.cos(position * div_term)
        # self.register_buffer('pe', pe)

    # def forward(self, x: Tensor) -> Tensor:
    #     """
    #     Arguments:
    #         x: Tensor, shape ``[seq_len, batch_size, embedding_dim]``
    #     """
    #     x = x + self.pe[:x.size(0)]
    #     return self.dropout(x)

    def forward(self, x: Tensor) -> Tensor:
        """
        Arguments:
            x: Tensor, shape ``->[seq_len, batch_size, embedding_dim]``
        """
        self.pe = self.pe.to(x.device)
        x = x.permute(1, 0, 2)
        pe = self.pe[:x.size(0)].permute(1, 0, 2) #[1,L,c]
        return pe


# # self code
# class PositionalEncoding(nn.Module):
#
#     def __init__(self, d_hid, n_position=200):
#         super().__init__()
#
#         # # Not a parameter
#         # self.register_buffer('pos_table', self._get_sinusoid_encoding_table(n_position, d_hid))
#         self.pos_table = self._get_sinusoid_encoding_table(n_position, d_hid) # [1, l, c]
#
#     def _get_sinusoid_encoding_table(self, n_position, d_hid):
#         ''' Sinusoid position encoding table '''
#         # TODO: make it with torch instead of numpy
#
#         def get_position_angle_vec(position):
#             return [position / np.power(10000, 2 * (hid_j // 2) / d_hid) for hid_j in range(d_hid)]
#
#         sinusoid_table = np.array([get_position_angle_vec(pos_i) for pos_i in range(n_position)])
#         sinusoid_table[:, 0::2] = np.sin(sinusoid_table[:, 0::2])  # dim 2i
#         sinusoid_table[:, 1::2] = np.cos(sinusoid_table[:, 1::2])  # dim 2i+1
#
#         return torch.FloatTensor(sinusoid_table).unsqueeze(0)
#
#     def forward(self, x):
#         return self.pos_table[:, :x.size(1)].clone().detach().to(x.device) # (1, l, c)
