"""
Input:\n
    EdgeList[e] = vertices in edge e
    delta[v] = edges incident to v
    EdgeColors[e] = color of e
    EdgeWeights[e] = weight of e
    n = |V|
    m = |E|
    k = |C|
    budget = global budget
Outputs:\n
    sigma = coloring
"""

function GloECC_PrimalDual(EdgeList::Vector{Vector{Int64}}, delta::Vector{Vector{Int64}}, EdgeColors::Vector{Int64}, EdgeWeights::Vector{Float64}, n::Int64, m::Int64, k::Int64, budget::Int64)
    
    sigma = [Int64[] for _=1:n]
    level = zeros(Float64, m) # level[e] = ∑ᵥβₑᵥ
    L = [w > 0 for w in EdgeWeights] # L[e] = false means e is tight

    # vertex2color2edges[v][color] = vector of {e ∈ δ(v) s.t. cₑ=color}
    vertex2color2edges = [Dict{Int64, Vector{Int64}}() for _=1:n]
    
    # vertex2color_in_L2num[v][color] = |{e ∈ δ(v) ∩ L s.t. cₑ = color}|
    vertex2color_in_L2num = [Dict{Int64, Int64}() for _=1:n]
    
    # v ∈ R if |χ(δ(v)∩L)| > 1
    R = ones(Bool,n)

    # global_slack = sum of |χ(δ(v)∩L)|-1 over all v ∈ R
    global_slack = 0

    for v = 1:n
        for edge in delta[v]
            color = EdgeColors[edge]
            if !haskey(vertex2color2edges[v], color)
                vertex2color2edges[v][color]= [edge]
            else
                push!(vertex2color2edges[v][color], edge)
            end

            if L[edge]
                if !haskey(vertex2color_in_L2num[v], color)
                    vertex2color_in_L2num[v][color] = 1
                else
                    vertex2color_in_L2num[v][color] += 1
                end
            end
        end
        global_slack += length(vertex2color_in_L2num[v]) - 1;
        if (length(vertex2color_in_L2num[v]) <= 1)
            R[v] = false
        end
    end
    
    # for each iteration, we compute the increase rate of each edge
    # the vector holds the value of the increase rate
    increase_rate_of_edge = Vector{Float64}(undef, m) # initialize with size m
    slack_increase_time_of_edge = Vector{Float64}(undef, m)

    ###########################
    # main wihle loop
    while global_slack > budget
        ###########################
        # compute the increase rate
        # here, set the increase rate of λ to be 1
        fill!(increase_rate_of_edge, 0.0) # reset the increase rate
        for v in 1:n
            if !R[v] # αᵥ for v ⁠∉ R does not increase
                continue
            end

            ###########################
            # increase-rate(λ) = increase-rate(αᵥ)
            # increase-rate(αᵥ) = increase-rate(sum of βₑᵥ over e ∈ δ(v)∩L s.t. cₑ=c)
            # for all c ∈ χ(δ(v)∩L)
            for (color_in_L, num) in vertex2color_in_L2num[v]
                ###########################
                # for each c ∈ χ(δ(v)∩L),
                # uniformly partition the increase rate among e ∈ δ(v)∩L s.t. cₑ=c
                for edge in vertex2color2edges[v][color_in_L]
                    if L[edge]
                        increase_rate_of_edge[edge] += 1 / num
                    end
                end
                ###########################
            end
            ###########################
        end
        # compute increase rate
        ###########################

        ###########################
        # compute the increase time
        fill!(slack_increase_time_of_edge, 0.0)
        min_increase_time = n+m+1000 # sufficiently large
        for e = 1:m
            if L[e]
                increase_time_of_e = max(0,(EdgeWeights[e] - level[e]))/increase_rate_of_edge[e] # max is for handling possible floating point error
                slack_increase_time_of_edge[e] = increase_time_of_e
                min_increase_time = min(increase_time_of_e, min_increase_time)
            end
        end
        ###########################
        
        ###########################
        # remove tight edges 
        # increase the level of e if untight
        for e = 1:m
            if !L[e]
                continue
            end
            ###########################
            # if slack_increase_time of e ≈ min_increase_time 
            # e becomes tight
            if slack_increase_time_of_edge[e] - min_increase_time < 1e-12
                level[e] = EdgeWeights[e]
                L[e] = false
                e_color = EdgeColors[e]

                ###########################
                # after the removal, check each v
                # if |χ(δ(v)∩L)|=1, assign such color to v
                # and remove v from R
                for v in EdgeList[e]
                    if vertex2color_in_L2num[v][e_color] == 1
                        delete!(vertex2color_in_L2num[v], e_color)
                        if R[v]
                            global_slack -= 1
                        end
                    else
                        vertex2color_in_L2num[v][e_color] -= 1
                    end
                    if R[v] && length(vertex2color_in_L2num[v]) <= 1
                        R[v] = false
                    end
                end
                ###########################
            else
                level[e] += min_increase_time * increase_rate_of_edge[e]
            end
            ###########################
            
            # heuristic
            if global_slack <= budget
                break
            end

        end
        ###########################
    end
    # main while loop
    ###########################

    # final step
    # coloring
    for v =1:n
        sigma[v] = collect(keys(vertex2color_in_L2num[v]))
    end

    # heuristic
    unsatisfied = zeros(Bool, m)
    
    potential = zeros(Int64,m) # how many uncolored vertices are there?
    
    for e=1:m
        for v in EdgeList[e]
            if length(sigma[v]) >= 1 && !(EdgeColors[e] in sigma[v])
                unsatisfied[e] = true
                break
            elseif length(sigma[v]) ==0
                potential[e] += 1
            end
        end
    end

    for v=1:n
        if length(sigma[v]) == 0
            min_e = -1
            min_potential = n+1
            for e in delta[v]
                if !unsatisfied[e] && potential[e] < min_potential
                    min_potential = potential[e]
                    min_e = e
                end
            end
            if min_e != -1 # it means there is non-unsatisfied edge
                sigma[v] = [EdgeColors[min_e]]
                for color in keys(vertex2color2edges[v])
                    if color == EdgeColors[min_e]
                        for e in vertex2color2edges[v][color]
                            potential[e] -= 1
                        end
                    else
                        for e in vertex2color2edges[v][color]
                            unsatisfied[e] = true
                        end
                    end
                end
            else # every incident edge is unsatisfied
                sigma[v] = [1]
            end
        end
    end
    
    return sigma
end