"""
Input:\n
    delta[v] = edges incident to v
    EdgeColors[e] = color of e
    EdgeWeights[e] = weight of e
    n = |V|
    m = |E|
    Budget[v] = bᵥ (local budget)
Outputs:\n
    sigma = coloring
"""
function LoECC_PrimalDual(delta::Vector{Vector{Int64}}, EdgeColors::Vector{Int64}, EdgeWeights::Vector{Float64}, n::Int64, m::Int64, Budget::Vector{Int64})

    sigma = [Int64[] for _=1:n]
    level = zeros(Float64, m) # level[e] = ∑ᵥβₑᵥ
    L = [w > 0 for w in EdgeWeights] # L[e] = false means e is tight
    
    # assign σ(v)
    for v = 1:n
        b_v = min(Budget[v],length(delta[v]))    
        # color2edges_in_L[color] = vector of {e ∈ δ(v) ∩ L s.t. cₑ=color}
        color2edges_in_L = Dict{Int64, Vector{Int64}}()
        
        for edge in delta[v]
            # update color2edges_in_L
            if L[edge] # if edge ∈ L
                color = EdgeColors[edge]
                if !haskey(color2edges_in_L, color)
                    color2edges_in_L[color] = [edge]
                else
                    push!(color2edges_in_L[color], edge)
                end
            end
        end

        untight_colors_num = length(color2edges_in_L)
        
        ###########################
        # if |χ(δ(v)∩L)| ≤ bᵥ,
        # σ(v)=χ(δ(v)∩L)
        if untight_colors_num <= b_v
            color_in_L =collect(keys(color2edges_in_L))
            sigma[v] = color_in_L
        ###########################

        ###########################
        # else |χ(δ(v)∩L)| > bᵥ, 
        # increase the dual
        else
            ###########################
            # computing color_slack_vec
            # color_slack_vec contains (color, slack) where
            # slack = sum of wₑ-βₑᵥ over e∈δ(v)∩L s.t. cₑ=color
            color_slack_vec = Vector{Tuple{Int64, Float64}}()
            for (color, edges) in color2edges_in_L
                slack = 0
                for edge in edges
                    # here, max is for handling possible floating-point error
                    slack += max(0, EdgeWeights[edge] - level[edge])
                end
                push!(color_slack_vec, (color,slack))
            end
            # computing color_slack_vec
            ###########################

            ###########################
            # here is to find the (bᵥ+1)-largest slack
            # x->x[2] : comparison based on the second elt (slack)
            # rev=True : descending order
            _, bv_plus_1_slack = partialsort!(color_slack_vec, 1:b_v+1, by = x->x[2]; rev=true)[b_v+1]
            # color_slack_vec[1:bᵥ+1] is sorted (w.r.t slack)
            ###########################

            ###########################
            # for (color,slack) in color_slack_vec[1:bᵥ],
            # we have slack ≥ bv_plus_1_slack;
            # increase the dual and add color to σ(v)
            for idx=1:b_v
                color, slack = color_slack_vec[idx]
                push!(sigma[v],color)
                ###########################
                # Increase Method. "Uniform"
                # increase edge ∈ color2edges_in_L[color]
                # proportional to EdgeWeights[edge] - level[edge]
                # so that no edge in color2edges_in_L[color] becomes tight
                for edge in color2edges_in_L[color]
                    level[edge] += bv_plus_1_slack * max(0, EdgeWeights[edge] - level[edge]) / slack
                end
            end
            # at this point, we know that 
            # there is exactly bᵥ number of colors
            # which are assigned to v
            ###########################

            ###########################
            # for (color, slack) in color_slack_vec[bᵥ+1:]
            # we have slack ≤ bv_plus_1_slack;
            # make edge of such color tight 
            for idx=b_v+1:untight_colors_num
                color, _ = color_slack_vec[idx]
                for edge in color2edges_in_L[color]
                    level[edge] = EdgeWeights[edge]
                    L[edge] = false
                end
            end
            ###########################

        end
        # |χ(δ(v)∩L)| > bᵥ case done
        ###########################
    end
    return sigma
end