import torch
from torch.optim.lr_scheduler import _LRScheduler

import math
from bisect import bisect_right


class WarmupMultiStepLR(_LRScheduler):
    def __init__(
        self,
        optimizer,
        milestones,
        gamma=0.1,
        warmup_factor=1.0 / 3,
        warmup_epochs = 5,
        warmup_method='linear',
        last_epoch=-1,
    ):
        if not list(milestones) == sorted(milestones):
            raise ValueError(
                "Milestones should be a list of" " increasing integers. Got {}",
                milestones,
            )

        if warmup_method not in ('constant', 'linear'):
            raise ValueError(
                "Only 'constant' or 'linear' warmup_method accepted"
                "got {}".format(warmup_method)
            )

        self.milestones = milestones
        self.gamma = gamma
        self.warmup_factor = warmup_factor
        self.warmup_epochs = warmup_epochs
        self.warmup_method = warmup_method
        super(WarmupMultiStepLR, self).__init__(optimizer, last_epoch)

    def get_lr(self):
        warmup_factor = 1
        if self.last_epoch < self.warmup_epochs:
            if self.warmup_method == 'constant':
                warmup_factor = self.warmup_factor
            elif self.warmup_method == 'linear':
                alpha = float(self.last_epoch) / self.warmup_epochs
                warmup_factor = self.warmup_factor * (1 - alpha) + alpha
        return [
            base_lr
            * warmup_factor
            * self.gamma ** bisect_right(self.milestones, self.last_epoch)
            for base_lr in self.base_lrs
        ]


class WarmupCustomLR(_LRScheduler):
    def __init__(
        self,
        optimizer,
        warmup_from=1.0 / 3,
        warmup_epochs = 5,
        last_epoch=-1,
        after_scheduler=None,
    ):
        self.warmup_epochs = warmup_epochs
        self.after_scheduler = after_scheduler
        
        self.warmup_to = [group['initial_lr'] for group in optimizer.param_groups]
        if (self.after_scheduler is not None):
            self.after_scheduler.step(epoch=self.warmup_epochs)
            self.warmup_to = self.after_scheduler.get_last_lr()  
        self.warmup_from = [warmup_from * v for v in self.warmup_to]
        super().__init__(optimizer, last_epoch)  

    def get_lr(self):
        if self.last_epoch <= self.warmup_epochs:
            alpha = float(self.last_epoch) / self.warmup_epochs
            return [
                lr_from + alpha * (lr_to - lr_from) 
                for lr_from, lr_to in zip(self.warmup_from, self.warmup_to)]
        else:
            self.after_scheduler.last_epoch = self.last_epoch
            return self.after_scheduler.get_lr()

