#!/bin/bash

# =========================== Preparations ==========================
# Create directories
CSV_PREFILL="csv_results/prefill"
CSV_AR="csv_results/ar"
CSV_SD="csv_results/sd"
mkdir -p $CSV_PREFILL
mkdir -p $CSV_AR
mkdir -p $CSV_SD

LOG_PREFILL="log_results/prefill"
LOG_AR="log_results/ar"
LOG_SD="log_results/sd"
mkdir -p $LOG_PREFILL
mkdir -p $LOG_AR
mkdir -p $LOG_SD

# Set the model path
MODEL_PATH="../models/Qwen2-57B-A14B-Instruct"
SPEC_MODEL_PATH="../models/Qwen2-0.5B-Instruct"

# ========================== Sweeping hyperparameters ===========================
# Set the seed for reproducibility
SEED=0

# Datasets to test
DATASETS=("humaneval" "mtbench")

# Temperatures to test
TEMPERATURES=(0.0 1.0)

# We use TP2 to fully hold the Qwen2-57B-17A-Instruct model in GPU memory
TP_SIZES=(2)

# Number of prompts in a batch, refering to batch size B in the paper
NUM_PROMPT=(1 2 4 8 12 16 20 24 28 32 36 40 44 48 52 56 60 80 100)

# Draf length, refering to \gamma in the paper
SPEC_TOKENS=(2 3 4)


# ========================== Run experiments ===========================
# Check the model's activated expert per token is not changed
python benchmark_utils.py --model "$MODEL_PATH"

# Run the experiments
for dataset in "${DATASETS[@]}"; do
    for temp in "${TEMPERATURES[@]}"; do
        for tp in "${TP_SIZES[@]}"; do
            for num_prompts in "${NUM_PROMPT[@]}"; do

                ## run PREFILL
                echo "====== Running PREFILL: dataset=$dataset, temp=$temp, tp=$tp, num_prompts=$num_prompts ======"
                FILENAME="${dataset}_temp${temp}_tp${tp}_np${num_prompts}_seed${SEED}.log"
                FILEPATH="$LOG_PREFILL/$FILENAME"
                # no need for speculative model
                python benchmark_prefill.py \
                    --model_name "$MODEL_PATH" \
                    --dataset "$dataset" \
                    --temperature "$temp" \
                    --tensor_parallel_size "$tp" \
                    --num_prompts "$num_prompts" \
                    --seed "$SEED" \
                    --csv_dir "$CSV_PREFILL" > $FILEPATH

                ## run AR
                echo "====== Running AR: dataset=$dataset, temp=$temp, tp=$tp, num_prompts=$num_prompts ======"
                FILENAME="${dataset}_temp${temp}_tp${tp}_np${num_prompts}_seed${SEED}.log"
                FILEPATH="$LOG_AR/$FILENAME"
                # no need for speculative model
                python benchmark_ar.py \
                    --model_name "$MODEL_PATH" \
                    --dataset "$dataset" \
                    --temperature "$temp" \
                    --tensor_parallel_size "$tp" \
                    --num_prompts "$num_prompts" \
                    --seed "$SEED" \
                    --csv_dir "$CSV_AR" > $FILEPATH

                ## run SD
                for spec_tokens in "${SPEC_TOKENS[@]}"; do
                    echo "====== Running SD: dataset=$dataset, temp=$temp, num_prompts=$num_prompts, spec_tokens=$spec_tokens ======"
                    FILENAME="${dataset}_temp${temp}_tp${tp}_np${num_prompts}_st${spec_tokens}_seed${SEED}.log"
                    FILEPATH="$LOG_SD/$FILENAME"
                    python benchmark_sd.py \
                        --model_name "$MODEL_PATH" \
                        --speculative_model_name "$SPEC_MODEL_PATH" \
                        --dataset "$dataset" \
                        --temperature "$temp" \
                        --tensor_parallel_size "$tp" \
                        --num_prompts "$num_prompts" \
                        --num_speculative_tokens "$spec_tokens" \
                        --seed "$SEED" \
                        --csv_dir "$CSV_SD" > $FILEPATH
                done
            done
        done
    done
done

