#!/bin/bash

# =========================== Preparations ==========================

# Create directories
CSV_PREFILL="csv_results/prefill"
CSV_AR="csv_results/ar"
CSV_SD="csv_results/sd"
mkdir -p $CSV_PREFILL
mkdir -p $CSV_AR
mkdir -p $CSV_SD

LOG_PREFILL="log_results/prefill"
LOG_AR="log_results/ar"
LOG_SD="log_results/sd"
mkdir -p $LOG_PREFILL
mkdir -p $LOG_AR
mkdir -p $LOG_SD

# Set the model path
MODEL_PATH="../models/Qwen2-57B-A14B-Instruct"
SPEC_MODEL_PATH="../models/Qwen2-0.5B-Instruct"

# ========================== Following ===========================
# set num_experts_per_tok, used for simulation
# the complete set should be (1 2 4 8 16 32), but 8 has been run in moe
NUM_EXPERTS=(1 2 4 16 32)

# Set the seed for reproducibility
SEED=0

# Dataset. Due to GPU resource and time constraints, we only run on one dataset.
DATASETS=("humaneval")

# Temperature. Due to GPU resource and time constraints, we only run on one dataset.
TEMPERATURES=(0)

# TP 
TP_SIZES=(2)

# Number of prompts to test.
NUM_PROMPT=(1 2 4 8 12 16 20 24 28 32 36 40 44 48 52 56 60 80 100)

# Speculative tokens to test (for SD only).
SPEC_TOKENS=(2 4)


# ========================== Run experiments ===========================
for n in "${NUM_EXPERTS[@]}"; do

    # first change the num_experts_per_tok in simulate_utils.py
    echo "=====Setting num_experts_per_tok to $n======"
    python simulate_utils.py --n "$n"

    # run experiments for each configuraton
    for dataset in "${DATASETS[@]}"; do
        for temp in "${TEMPERATURES[@]}"; do
            for tp in "${TP_SIZES[@]}"; do
                for num_prompts in "${NUM_PROMPT[@]}"; do

                    ## run PREFILL
                    echo "====== Running PREFILL: dataset=$dataset, temp=$temp, tp=$tp, num_prompts=$num_prompts, expum=$n ======"
                    FILENAME="${dataset}_temp${temp}_tp${tp}_np${num_prompts}_seed${SEED}_expnum${n}.log"
                    FILEPATH="$LOG_PREFILL/$FILENAME"
                    # no need for speculative model
                    python simulate_prefill.py \
                        --model_name "$MODEL_PATH" \
                        --dataset "$dataset" \
                        --temperature "$temp" \
                        --tensor_parallel_size "$tp" \
                        --num_prompts "$num_prompts" \
                        --seed "$SEED" \
                        --n "$n" \
                        --csv_dir "$CSV_PREFILL" > $FILEPATH


                    ## run AR
                    echo "====== Running AR: dataset=$dataset, temp=$temp, tp=$tp, num_prompts=$num_prompts, expnum=$n ======"
                    FILENAME="${dataset}_temp${temp}_tp${tp}_np${num_prompts}_seed${SEED}_expnum${n}.log"
                    FILEPATH="$LOG_AR/$FILENAME"
                    # no need for speculative model
                    python simulate_ar.py \
                        --model_name "$MODEL_PATH" \
                        --dataset "$dataset" \
                        --temperature "$temp" \
                        --tensor_parallel_size "$tp" \
                        --num_prompts "$num_prompts" \
                        --seed "$SEED" \
                        --n "$n" \
                        --csv_dir "$CSV_AR" > $FILEPATH

                    ## run SD
                    for spec_tokens in "${SPEC_TOKENS[@]}"; do
                        echo "====== Running SD: dataset=$dataset, temp=$temp, tp=$tp, num_prompts=$num_prompts, spec_tokens=$spec_tokens, expnum=$n ======"
                        FILENAME="${dataset}_temp${temp}_tp${tp}_np${num_prompts}_st${spec_tokens}_seed${SEED}_expnum${n}.log"
                        FILEPATH="$LOG_SD/$FILENAME"
                        python simulate_sd.py \
                            --model_name "$MODEL_PATH" \
                            --speculative_model_name "$SPEC_MODEL_PATH" \
                            --dataset "$dataset" \
                            --temperature "$temp" \
                            --tensor_parallel_size "$tp" \
                            --num_prompts "$num_prompts" \
                            --num_speculative_tokens "$spec_tokens" \
                            --seed "$SEED" \
                            --n "$n" \
                            --csv_dir "$CSV_SD" > $FILEPATH
                    done
                done
            done
        done
    done
done
