import os
import re

def analyze_results():
    # Define paths
    data_dir = "data"
    results_dir = "results/local-searcher"
    
    # Get list of all datasets by extracting base names from CSV files
    datasets = []
    for filename in os.listdir(data_dir):
        if filename.endswith(".csv") and filename != "weka":
            # Extract base name without extension
            base_name = os.path.splitext(filename)[0]
            datasets.append(base_name)
    
    # Total number of datasets
    total_instances = len(datasets)
    
    # Configurations to check
    configs = [
        {"name": "kadj0-kexch2", "pattern": r".*_data-kadj0-kexch2(?!.*\.timeout)"},
        {"name": "kadj2-kexch0", "pattern": r".*_data-kadj2-kexch0(?!.*\.timeout)"}
    ]
    
    # Get all result files
    result_files = []
    if os.path.exists(results_dir):
        result_files = os.listdir(results_dir)
    
    # Track solved instances for each configuration
    solved_counts = {config["name"]: 0 for config in configs}
    solved_instances = {config["name"]: [] for config in configs}
    
    # Check each dataset against result files
    for dataset in datasets:
        for config in configs:
            # Create pattern to match result files for this dataset and configuration
            # dataset_pattern = f"^{dataset}_data-{config['name'].replace('kexch', 'kexch').replace('kadj', 'kadj')}.*"
            dataset_pattern = f"^{dataset}_data-{config['name']}"
            pattern = re.compile(dataset_pattern)
            
            # Check if any matching result file exists
            found = False
            for result_file in result_files:
                if pattern.match(result_file) and not result_file.endswith(".timeout"):
                    found = True
                    break
            
            if found:
                solved_counts[config["name"]] += 1
                solved_instances[config["name"]].append(dataset)
    
    # Calculate percentages
    percentages = {
        config["name"]: (solved_counts[config["name"]] / total_instances) * 100 
        for config in configs
    }
    
    # Print statistics
    print(f"Total number of instances: {total_instances}")
    for config in configs:
        config_name = config["name"]
        print(f"\nConfiguration: {config_name}")
        print(f"Number of instances solved: {solved_counts[config_name]}")
        print(f"Percentage of instances solved: {percentages[config_name]:.2f}%")
    
    # Create LaTeX file
    create_latex_file(total_instances, solved_counts, percentages)

def create_latex_file(total, solved, percentages):
    # Create results directory if it doesn't exist
    os.makedirs("results", exist_ok=True)
    
    # Create LaTeX file
    with open("results/solver_stats.tex", "w") as f:
        f.write("% Solver statistics - auto-generated\n\n")
        
        # Total instances
        f.write("\\newcommand{\\totalInstances}{" + str(total) + "}\n")
        
        # For each configuration
        for config_name, count in solved.items():
            # Clean up config name for LaTeX command
            clean_name = config_name.replace("-", "").replace("2", "two").replace("0", "zero")
            
            # Instances solved
            f.write(f"\\newcommand{{\\{clean_name}Solved}}{{{count}}}\n")
            
            # Percentage solved (formatted to 2 decimal places)
            percentage = percentages[config_name]
            f.write(f"\\newcommand{{\\{clean_name}Percentage}}{{{percentage:.2f}}}\n")
    
    print(f"\nLaTeX file written to results/solver_stats.tex")

if __name__ == "__main__":
    analyze_results()
