from __future__ import print_function, division
import math
import gc

import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.utils.data

from .shared import conv_block, up_conv


class Head(nn.Module):
    """
    UNet - Basic Implementation
    Paper : https://arxiv.org/abs/1505.04597
    """

    def __init__(self, in_ch=3, out_ch=1, n_filters=32):
        super().__init__()

        n1 = n_filters
        filters = [n1 * i for i in [1, 2, 4, 8, 16]]

        self.Maxpool1 = nn.MaxPool2d(kernel_size=2, stride=2)
        self.Maxpool2 = nn.MaxPool2d(kernel_size=2, stride=2)
        self.Maxpool3 = nn.MaxPool2d(kernel_size=2, stride=2)
        self.Maxpool4 = nn.MaxPool2d(kernel_size=2, stride=2)

        self.Conv1 = conv_block(in_ch, filters[0])
        self.Conv2 = conv_block(filters[0], filters[1])
        self.Conv3 = conv_block(filters[1], filters[2])
        self.Conv4 = conv_block(filters[2], filters[3])
        self.Conv5 = conv_block(filters[3], filters[4])

        self.Up5 = up_conv(filters[4], filters[3])
        self.Up_conv5 = conv_block(filters[4], filters[3])

        self.Up4 = up_conv(filters[3], filters[2])
        self.Up_conv4 = conv_block(filters[3], filters[2])

        self.Up3 = up_conv(filters[2], filters[1])
        self.Up_conv3 = conv_block(filters[2], filters[1])

        self.Up2 = up_conv(filters[1], filters[0])
        self.Up_conv2 = conv_block(filters[1], filters[0])

        self.Conv = nn.Conv2d(filters[0], out_ch,
                              kernel_size=5, stride=1, padding=2)

        # self.active = torch.nn.Softmax(dim=1)

    def forward(self, x):
        e1 = self.Conv1(x)

        e2 = self.Maxpool1(e1)
        e2 = self.Conv2(e2)

        e3 = self.Maxpool2(e2)
        e3 = self.Conv3(e3)

        e4 = self.Maxpool3(e3)
        e4 = self.Conv4(e4)

        e5 = self.Maxpool4(e4)
        e5 = self.Conv5(e5)

        d5 = self.Up5(e5)
        d5 = torch.cat((e4, d5), dim=1)

        d5 = self.Up_conv5(d5)

        d4 = self.Up4(d5)
        d4 = torch.cat((e3, d4), dim=1)
        d4 = self.Up_conv4(d4)

        d3 = self.Up3(d4)
        d3 = torch.cat((e2, d3), dim=1)
        d3 = self.Up_conv3(d3)

        d2 = self.Up2(d3)
        d2 = torch.cat((e1, d2), dim=1)
        d2 = self.Up_conv2(d2)

        out = self.Conv(d2)

        # d1 = self.active(out)

        return d2, out


class UNet_m(nn.Module):
    def __init__(self):
        super().__init__()
        n_filters = 32
        self.filters = n_filters
        self.head1 = Head(in_ch=4, out_ch=1, n_filters=n_filters)
        self.head2 = Head(in_ch=4, out_ch=1, n_filters=n_filters)
        self.head3 = Head(in_ch=4, out_ch=1, n_filters=n_filters)
        self.head4 = Head(in_ch=4, out_ch=1, n_filters=n_filters)
        self.imgq = Head(in_ch=4, out_ch=n_filters, n_filters=8)

        self.conv = nn.Sequential(
            nn.Conv2d(in_channels=4+n_filters*4, out_channels=n_filters, kernel_size=5, stride=1, padding=2),
            nn.BatchNorm2d(n_filters),
            nn.ReLU(inplace=True),
            nn.Conv2d(in_channels=n_filters, out_channels=n_filters, kernel_size=5, stride=1, padding=2),
            nn.BatchNorm2d(n_filters),
            nn.ReLU(inplace=True),
            nn.Conv2d(in_channels=n_filters, out_channels=5, kernel_size=5, stride=1, padding=2)
        )

    def forward(self, x):
        d21, out1 = self.head1(x)
        d22, out2 = self.head2(x)
        d23, out3 = self.head3(x)
        d24, out4 = self.head4(x)

        imgq = self.imgq(x)[-1]
        att_1 = (imgq * d21).sum(dim=1, keepdim=True) / math.sqrt(self.filters)
        att_2 = (imgq * d22).sum(dim=1, keepdim=True) / math.sqrt(self.filters)
        att_3 = (imgq * d23).sum(dim=1, keepdim=True) / math.sqrt(self.filters)
        att_4 = (imgq * d24).sum(dim=1, keepdim=True) / math.sqrt(self.filters)

        att = torch.cat((att_1, att_2, att_3, att_4), dim=1).softmax(dim=1)

        out = self.conv(torch.cat((x, d21 * att[:, 0:1, :, :], d22 * att[:, 1:2, :, :], d23 * att[:, 2:3, :, :], d24 * att[:, 3:4, :, :]), dim=1))

        return out, out1.squeeze(), out2.squeeze(), out3.squeeze(), out4.squeeze()
