import math
import torch
import torch.nn as nn
import torch.utils.model_zoo as model_zoo
from ..sync_batchnorm.batchnorm import SynchronizedBatchNorm2d


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, inplanes, planes, stride=1, dilation=1, downsample=None, BatchNorm=None):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv2d(inplanes, planes, kernel_size=1, bias=False)
        self.bn1 = BatchNorm(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=stride,
                               dilation=dilation, padding=dilation, bias=False)
        self.bn2 = BatchNorm(planes)
        self.conv3 = nn.Conv2d(planes, planes * 4, kernel_size=1, bias=False)
        self.bn3 = BatchNorm(planes * 4)
        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        self.stride = stride
        self.dilation = dilation
        
        # self.att = nn.parameter.Parameter(torch.rand(1), requires_grad=True)
        self.att = nn.parameter.Parameter(torch.ones(1).float(), requires_grad=True)
        self.att_gate = nn.Sequential(
            nn.Tanh(),
            nn.ReLU()
        )

    def forward(self, input):
        x, s = input

        residual = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv3(out)
        out = self.bn3(out)

        if self.downsample is not None:
            residual = self.downsample(x)

        # out += residual
        a = self.att_gate(self.att)
        out = out * a + residual

        out = self.relu(out)

        return (out, s + a)

class conv_block_nested(nn.Module):

    def __init__(self, in_ch, mid_ch, out_ch, batchnorm):
        super().__init__()
        self.activation = nn.ReLU(inplace=True)
        self.conv1 = nn.Conv2d(in_ch, mid_ch, kernel_size=3, padding=1, bias=True)
        self.bn1 = batchnorm(mid_ch)
        self.conv2 = nn.Conv2d(mid_ch, out_ch, kernel_size=3, padding=1, bias=True)
        self.bn2 = batchnorm(out_ch)

    def forward(self, x):
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.activation(x)

        x = self.conv2(x)
        x = self.bn2(x)
        output = self.activation(x)

        return output


class ResNet101(nn.Module):

    def __init__(self, in_ch, block, layers, output_stride, BatchNorm, pretrained=True):
        self.inplanes = 64
        super().__init__()
        blocks = [1, 2, 4]
        if output_stride == 16:
            strides = [2, 2, 2, 2, 1]
            dilations = [1, 1, 1, 1, 2]
        elif output_stride == 8:
            raise NotImplementedError
            # strides = [1, 2, 1, 1]
            # dilations = [1, 1, 2, 4]
        else:
            raise NotImplementedError

        # Modules
        self.conv = nn.Sequential(
            nn.Conv2d(in_ch, 64, kernel_size=7, stride=1, padding=3, bias=False),
            BatchNorm(64),
            nn.ReLU(inplace=True),
            nn.Conv2d(64, 64, kernel_size=3, stride=1, padding=1, bias=False),
            BatchNorm(64),
            nn.ReLU(inplace=True)
        )

        self.layer1 = self._make_layer(block, 64, layers[0], stride=strides[0], dilation=dilations[0], BatchNorm=BatchNorm)
        self.layer2 = self._make_layer(block, 64, layers[1], stride=strides[1], dilation=dilations[1], BatchNorm=BatchNorm)
        self.layer3 = self._make_layer(block, 128, layers[2], stride=strides[2], dilation=dilations[2], BatchNorm=BatchNorm)
        self.layer4 = self._make_layer(block, 256, layers[3], stride=strides[3], dilation=dilations[3], BatchNorm=BatchNorm)
        self.layer5 = self._make_layer(block, 512, layers[4], stride=strides[4], dilation=dilations[4], BatchNorm=BatchNorm)
        # self.layer4 = self._make_MG_unit(block, 512, blocks=blocks, stride=strides[3], dilation=dilations[3], BatchNorm=BatchNorm)
        self._init_weight()

        filters = [64, 256, 256, 512, 1024, 2048]
        self.Up = nn.Upsample(scale_factor=2, mode='bilinear', align_corners=True)

        self.conv0_0 = self.conv
        self.conv1_0 = self.layer1
        self.conv2_0 = self.layer2
        self.conv3_0 = self.layer3
        self.conv4_0 = self.layer4
        self.conv5_0 = self.layer5

        self.conv0_1 = conv_block_nested(filters[0] + filters[1], filters[0], filters[0], BatchNorm)
        self.conv1_1 = conv_block_nested(filters[1] + filters[2], filters[1], filters[1], BatchNorm)
        self.conv2_1 = conv_block_nested(filters[2] + filters[3], filters[2], filters[2], BatchNorm)
        self.conv3_1 = conv_block_nested(filters[3] + filters[4], filters[3], filters[3], BatchNorm)
        self.conv4_1 = conv_block_nested(filters[4] + filters[5], filters[4], filters[4], BatchNorm)

        self.conv0_2 = conv_block_nested(filters[0]*2 + filters[1], filters[0], filters[0], BatchNorm)
        self.conv1_2 = conv_block_nested(filters[1]*2 + filters[2], filters[1], filters[1], BatchNorm)
        self.conv2_2 = conv_block_nested(filters[2]*2 + filters[3], filters[2], filters[2], BatchNorm)
        self.conv3_2 = conv_block_nested(filters[3]*2 + filters[4], filters[3], filters[3], BatchNorm)

        self.conv0_3 = conv_block_nested(filters[0]*3 + filters[1], filters[0], filters[0], BatchNorm)
        self.conv1_3 = conv_block_nested(filters[1]*3 + filters[2], filters[1], filters[1], BatchNorm)
        self.conv2_3 = conv_block_nested(filters[2]*3 + filters[3], filters[2], filters[2], BatchNorm)

        self.conv0_4 = conv_block_nested(filters[0]*4 + filters[1], filters[0], filters[0], BatchNorm)
        self.conv1_4 = conv_block_nested(filters[1]*4 + filters[2], filters[1], filters[1], BatchNorm)

        self.conv0_5 = conv_block_nested(filters[0]*5 + filters[1], filters[0], filters[0], BatchNorm)

        self.att_01 = nn.parameter.Parameter(torch.ones(2).float())
        self.att_02 = nn.parameter.Parameter(torch.ones(3).float())
        self.att_03 = nn.parameter.Parameter(torch.ones(4).float())
        self.att_04 = nn.parameter.Parameter(torch.ones(5).float())
        self.att_05 = nn.parameter.Parameter(torch.ones(6).float())

        self.att_11 = nn.parameter.Parameter(torch.ones(2).float())
        self.att_12 = nn.parameter.Parameter(torch.ones(3).float())
        self.att_13 = nn.parameter.Parameter(torch.ones(4).float())
        self.att_14 = nn.parameter.Parameter(torch.ones(5).float())

        self.att_21 = nn.parameter.Parameter(torch.ones(2).float())
        self.att_22 = nn.parameter.Parameter(torch.ones(3).float())
        self.att_23 = nn.parameter.Parameter(torch.ones(4).float())

        self.att_31 = nn.parameter.Parameter(torch.ones(2).float())
        self.att_32 = nn.parameter.Parameter(torch.ones(3).float())

        self.att_41 = nn.parameter.Parameter(torch.ones(2).float())

        self.att_gate = nn.Sequential(
            nn.Tanh(),
            nn.ReLU()
        )

        if pretrained:
            self._load_pretrained_model()

    def _make_layer(self, block, planes, blocks, stride=1, dilation=1, BatchNorm=None):
        downsample = None
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                nn.Conv2d(self.inplanes, planes * block.expansion,
                          kernel_size=1, stride=stride, bias=False),
                BatchNorm(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, dilation, downsample, BatchNorm))
        self.inplanes = planes * block.expansion
        for i in range(1, blocks):
            layers.append(block(self.inplanes, planes, dilation=dilation, BatchNorm=BatchNorm))

        return nn.Sequential(*layers)

    def _make_MG_unit(self, block, planes, blocks, stride=1, dilation=1, BatchNorm=None):
        downsample = None
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                nn.Conv2d(self.inplanes, planes * block.expansion,
                          kernel_size=1, stride=stride, bias=False),
                BatchNorm(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, dilation=blocks[0]*dilation,
                            downsample=downsample, BatchNorm=BatchNorm))
        self.inplanes = planes * block.expansion
        for i in range(1, len(blocks)):
            layers.append(block(self.inplanes, planes, stride=1,
                                dilation=blocks[i]*dilation, BatchNorm=BatchNorm))

        return nn.Sequential(*layers)

    def forward(self, x):

        att_0 = [self.att_gate(i) for i in [self.att_01, self.att_02, self.att_03, self.att_04, self.att_05]]
        att_1 = [self.att_gate(i) for i in [self.att_11, self.att_12, self.att_13, self.att_14]]
        att_2 = [self.att_gate(i) for i in [self.att_21, self.att_22, self.att_23]]
        att_3 = [self.att_gate(i) for i in [self.att_31, self.att_32]]
        att_4 = [self.att_gate(i) for i in [self.att_41]]

        a01, a02, a03, a04, a05 = att_0
        a11, a12, a13, a14 = att_1
        a21, a22, a23 = att_2
        a31, a32 = att_3
        [a41] = att_4


        x0_0 = self.conv0_0(x)

        x1_0, s1 = self.conv1_0((x0_0, 0))
        x0_1 = self.conv0_1(torch.cat([x0_0 * a01[0], self.Up(x1_0) * a01[1]], 1))

        x2_0, s2 = self.conv2_0((x1_0, s1))
        x1_1 = self.conv1_1(torch.cat([x1_0 * a11[0], self.Up(x2_0) * a11[1]], 1))
        x0_2 = self.conv0_2(torch.cat([x0_0 * a02[0], x0_1 * a02[1], self.Up(x1_1) * a02[2]], 1))

        x3_0, s3 = self.conv3_0((x2_0, s2))
        x2_1 = self.conv2_1(torch.cat([x2_0 * a21[0], self.Up(x3_0) * a21[1]], 1))
        x1_2 = self.conv1_2(torch.cat([x1_0 * a12[0], x1_1 * a12[1], self.Up(x2_1) * a12[2]], 1))
        x0_3 = self.conv0_3(torch.cat([x0_0 * a03[0], x0_1 * a03[1], x0_2 * a03[2], self.Up(x1_2) * a03[3]], 1))

        x4_0, s4 = self.conv4_0((x3_0, s3))
        x3_1 = self.conv3_1(torch.cat([x3_0 * a31[0], self.Up(x4_0) * a31[1]], 1))
        x2_2 = self.conv2_2(torch.cat([x2_0 * a22[0], x2_1 * a22[1], self.Up(x3_1) * a22[2]], 1))
        x1_3 = self.conv1_3(torch.cat([x1_0 * a13[0], x1_1 * a13[1], x1_2 * a13[2], self.Up(x2_2) * a13[3]], 1))
        x0_4 = self.conv0_4(torch.cat([x0_0 * a04[0], x0_1 * a04[1], x0_2 * a04[2], x0_3 * a04[3], self.Up(x1_3) * a04[4]], 1))

        x5_0, s5 = self.conv5_0((x4_0, s4))
        x4_1 = self.conv4_1(torch.cat([x4_0 * a41[0], x5_0 * a41[1]], 1))
        x3_2 = self.conv3_2(torch.cat([x3_0 * a32[0], x3_1 * a32[1], self.Up(x4_1) * a32[2]], 1))
        x2_3 = self.conv2_3(torch.cat([x2_0 * a23[0], x2_1 * a23[1], x2_2 * a23[2], self.Up(x3_2) * a23[3]], 1))
        x1_4 = self.conv1_4(torch.cat([x1_0 * a14[0], x1_1 * a14[1], x1_2 * a14[2], x1_3 * a14[3], self.Up(x2_3) * a14[4]], 1))
        x0_5 = self.conv0_5(torch.cat([x0_0 * a05[0], x0_1 * a05[1], x0_2 * a05[2], x0_3 * a05[3], x0_4 * a05[4], self.Up(x1_4) * a05[5]], 1))

        low_level_feat = x0_5  # bs * 64 * H * W

        mid_level_feat = x2_3  # bs * 512 * H/4 * W/4

        high_level_feat = x5_0  # bs * 2048 * H/16 * W/16

        nested_gate_loss = 0
        cnt = 0
        for atts in [att_0, att_1, att_2, att_3, att_4]:
            for a in atts:
                # gate_loss += a[-1]  # apply loss only on unsample edges
                # cnt += 1
                nested_gate_loss += a.sum()  # apply loss on all edges
                cnt += a.numel()
        nested_gate_loss /= cnt
        
        bb_gate_loss = s5[0] / 33

        return low_level_feat, mid_level_feat, high_level_feat, nested_gate_loss, bb_gate_loss

    def _init_weight(self):
        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                n = m.kernel_size[0] * m.kernel_size[1] * m.out_channels
                m.weight.data.normal_(0, math.sqrt(2. / n))
            elif isinstance(m, SynchronizedBatchNorm2d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()
            elif isinstance(m, nn.BatchNorm2d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()

    def _load_pretrained_model(self):
        pretrain_dict = model_zoo.load_url('https://download.pytorch.org/models/resnet101-5d3b4d8f.pth')
        model_dict = {}
        state_dict = self.state_dict()
        for k, v in pretrain_dict.items():
            if k in state_dict:
                model_dict[k] = v
        state_dict.update(model_dict)
        self.load_state_dict(state_dict)


def ResNet_101(in_ch, output_stride, BatchNorm, pretrained=False):
    """Constructs a ResNet-101 model.
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
    """
    model = ResNet101(in_ch, Bottleneck, [3, 3, 4, 20, 3], output_stride, BatchNorm, pretrained=pretrained)
    return model

# if __name__ == "__main__":
#     import torch
#     model = ResNet101(3, BatchNorm=nn.BatchNorm2d, pretrained=True, output_stride=8)
#     input = torch.rand(1, 3, 512, 512)
#     output, low_level_feat = model(input)
#     print(output.size())
#     print(low_level_feat.size())
