% State evolution (SE) recursion for the square model (1.1) with
% Marchenko-Pastur noise

clear;
close all;
clc;

alphagrid = [3, 4]; % values of the SNR \alpha
c = 2; % parameter of the Marchenko-Pastur distribution
niter = 10; % number of iterations of SE recursion
    
% These choices of alphagrid and niter ensure that the program 
% runs fast on a laptop. The parameters employed to obtain the results
% reported in Figures 1-2 are specified in Section 4 of the paper


for j = 1 : length(alphagrid)
    
    alpha = alphagrid(j);     
    fprintf('alpha=%f\n', alpha);

    % allocate vectors for SE recursion
    muSE = zeros(niter, 1); % contains \mu_i
    sigmaSE = zeros(niter, niter); % contains \sigma_{i, j}
    Mprod = zeros(niter, niter); % contains E[U_i U_j]
    avgder = zeros(niter, 1); % contains E[u_i'(F_{i-1})]
    scal = zeros(niter, 1); % contains the (limit) normalized scalar product between signal and AMP iteration 
        
    % initialization of SE recursion
    muSE(1) = alpha * sqrt(1-c/(alpha-1)^2);
    scal(1) = sqrt(1-c/(alpha-1)^2);
    sigmaSE(1, 1) = c * alpha^2/(alpha-1)^2;
    Mprod(1, 1) = 1;
    
    fprintf('Iteration %d, scal=%f\n', 1, scal(1));
    
    % computation of E[u_2'(F_{1})]
    fun = @(x) 1/sqrt(2*pi) * exp(-x.^2/2) .* ...
        ( (tanh( muSE(1)^2/sigmaSE(1, 1) + muSE(1)/sqrt(sigmaSE(1, 1)) * x )).^2 + ...
        (tanh( - muSE(1)^2/sigmaSE(1, 1) + muSE(1)/sqrt(sigmaSE(1, 1)) * x )).^2 ) ;    
    avgder(1) = muSE(1)/sigmaSE(1) * ( 1 - 1/2 * integral(fun,-Inf,Inf));
    
        
    for jj = 2 : niter
        
        % computation of \mu_t
        fun = @(x) 1/sqrt(2*pi) * exp(-x.^2/2) .* ...
            ( (tanh( muSE(jj-1)^2/sigmaSE(jj-1, jj-1) + muSE(jj-1)/sqrt(sigmaSE(jj-1, jj-1)) * x )) - ...
            (tanh( - muSE(jj-1)^2/sigmaSE(jj-1, jj-1) + muSE(jj-1)/sqrt(sigmaSE(jj-1, jj-1)) * x )) ) ;
        intf = integral(fun,-Inf,Inf);        
        muSE(jj) = alpha/2 * intf;

        % computation of E[U_t U_1]
        if jj == 2
        
            fun = @(x) 1/sqrt(2*pi) * exp(-x.^2/2) .* ...
                ( (tanh( muSE(1)^2/sigmaSE(1, 1) + muSE(1)/sqrt(sigmaSE(1, 1)) * x )) .* (muSE(1)+sqrt(sigmaSE(1, 1)) * x)/alpha + ...
                (tanh( -muSE(1)^2/sigmaSE(1, 1) + muSE(1)/sqrt(sigmaSE(1, 1)) * x )) .* (-muSE(1)+sqrt(sigmaSE(1, 1)) * x)/alpha );
            intf = integral(fun,-Inf,Inf);
        
            Mprod(jj, 1) = 1/2 * intf;        
            Mprod(1, jj) = Mprod(jj, 1);
            
        else
            
            Sigma = [sigmaSE(jj-1, jj-1), sigmaSE(jj-1, 1); ...
                    sigmaSE(jj-1, 1), sigmaSE(1, 1)];
            invS = inv(Sigma);
            
            % If we run the SE recursion for many iterations, because of
            % numerical issues, the matrix Sigma may have determinant < 0.
            % This is not possible since Sigma is a covariance matrix.  
            % Thus, in this case, we assume that Sigma has 0 determinant
            % and the 2D integral becomes a 1D integral
            
            if det(Sigma) < 0 
                fun = @(x) 1/sqrt(2*pi) * exp(-x.^2/2) .* ...
                    ( ( (tanh( muSE(jj-1)^2/sigmaSE(jj-1, jj-1) + muSE(jj-1)/sqrt(sigmaSE(jj-1, jj-1)) * x )) .* ... 
                    (muSE(1)+sqrt(sigmaSE(1, 1))*x)/alpha ) + ...
                    ( (tanh( - muSE(jj-1)^2/sigmaSE(jj-1, jj-1) + muSE(jj-1)/sqrt(sigmaSE(jj-1, jj-1)) * x )) .* ... 
                    (-muSE(1)+sqrt(sigmaSE(1, 1))*x)/alpha ) );
            
                Mprod(jj, 1) = 1/2 * integral(fun,-Inf,Inf);
                Mprod(1, jj) = Mprod(jj, 1);
            else
                       
                fun = @(x,y) 1/(2*pi*sqrt(det(Sigma))) * ...
                    exp(-1/2 * ( invS(1, 1) * x.^2 + invS(2, 2) * y.^2 + 2*invS(1, 2)*x.*y) ) .* ...
                    ( ( (tanh( muSE(jj-1)^2/sigmaSE(jj-1, jj-1) + muSE(jj-1)/sigmaSE(jj-1, jj-1) * x )) .* ... 
                    (muSE(1)+y)/alpha ) + ...
                    ( (tanh( - muSE(jj-1)^2/sigmaSE(jj-1, jj-1) + muSE(jj-1)/sigmaSE(jj-1, jj-1) * x )) .* ... 
                    (-muSE(1)+y)/alpha ) );
            
                Mprod(jj, 1) = 1/2 * integral2(fun,-Inf,Inf,-Inf,Inf);
                Mprod(1, jj) = Mprod(jj, 1);
            end   
        end

        % computation of E[U_t^2]
        fun = @(x) 1/sqrt(2*pi) * exp(-x.^2/2) .* ...
            ( (tanh( muSE(jj-1)^2/sigmaSE(jj-1, jj-1) + muSE(jj-1)/sqrt(sigmaSE(jj-1, jj-1)) * x )).^2 + ...
            (tanh( - muSE(jj-1)^2/sigmaSE(jj-1, jj-1) + muSE(jj-1)/sqrt(sigmaSE(jj-1, jj-1)) * x )).^2 ) ;
        Mprod(jj, jj) = 1/2 * integral(fun,-Inf,Inf);
        
        % normalized correlation between u^t and the signal
        scal(jj) = muSE(jj)/alpha/sqrt(Mprod(jj, jj));
        fprintf('Iteration %d, scal=%f\n', jj, scal(jj));
        
        % computation of E[U_i U_j] (all the remaining values)
        for ii = 2 : jj-1
                          
            Sigma = [sigmaSE(jj-1, jj-1), sigmaSE(jj-1, ii-1); ...
                    sigmaSE(jj-1, ii-1), sigmaSE(ii-1, ii-1)];
            invS = inv(Sigma);
            
            if det(Sigma) < 0 
                fun = @(x) 1/sqrt(2*pi) * exp(-x.^2/2) .* ...
                    ( ( (tanh( muSE(jj-1)^2/sigmaSE(jj-1, jj-1) + muSE(jj-1)/sqrt(sigmaSE(jj-1, jj-1)) * x )) .* ... 
                    (tanh( muSE(ii-1)^2/sigmaSE(ii-1, ii-1) + muSE(ii-1)/sqrt(sigmaSE(ii-1, ii-1)) * x )) ) + ...
                    ( (tanh( - muSE(jj-1)^2/sigmaSE(jj-1, jj-1) + muSE(jj-1)/sqrt(sigmaSE(jj-1, jj-1)) * x )) .* ... 
                    (tanh( - muSE(ii-1)^2/sigmaSE(ii-1, ii-1) + muSE(ii-1)/sqrt(sigmaSE(ii-1, ii-1)) * x )) ) );
                Mprod(jj, ii) = 1/2 * integral(fun,-Inf,Inf);
            else
                       
                fun = @(x,y) 1/(2*pi*sqrt(det(Sigma))) * ...
                    exp(-1/2 * ( invS(1, 1) * x.^2 + invS(2, 2) * y.^2 + 2*invS(1, 2)*x.*y) ) .* ...
                    ( ( (tanh( muSE(jj-1)^2/sigmaSE(jj-1, jj-1) + muSE(jj-1)/sigmaSE(jj-1, jj-1) * x )) .* ... 
                    (tanh( muSE(ii-1)^2/sigmaSE(ii-1, ii-1) + muSE(ii-1)/sigmaSE(ii-1, ii-1) * y )) ) + ...
                    ( (tanh( - muSE(jj-1)^2/sigmaSE(jj-1, jj-1) + muSE(jj-1)/sigmaSE(jj-1, jj-1) * x )) .* ... 
                    (tanh( - muSE(ii-1)^2/sigmaSE(ii-1, ii-1) + muSE(ii-1)/sigmaSE(ii-1, ii-1) * y )) ) );
            
                Mprod(jj, ii) = 1/2 * integral2(fun,-Inf,Inf,-Inf,Inf);
            end
            Mprod(ii, jj) = Mprod(jj, ii);
        end
            
        % computation of \sigma_{i, j}
        for ii = 1 : jj

            M = zeros(jj-1, ii-1);
                      
            for i1 = 1 : jj-1
                for i2 = 1 : ii-1       
                   M(i1, i2) = c * prod(avgder(jj-i1+1 : jj-1)) ...
                       * prod(avgder(ii-i2+1 : ii-1)) * Mprod(jj-i1+1, ii-i2+1);
                end
            end
            
            M1 = zeros(1, ii-1);
           
            for i2 = 1 : ii-1
               M1(i2) = c/(alpha-1) * prod(avgder(1 : jj-1)) ...
                   * prod(avgder(ii-i2+1 : ii-1)) * alpha * Mprod(1, ii-i2+1);
            end
            
            M2 = zeros(jj-1, 1);
           
            for i1 = 1 : jj-1
               M2(i1) = c/(alpha-1) * prod(avgder(1 : ii-1)) ...
                   * prod(avgder(jj-i1+1 : jj-1)) * alpha * Mprod(1, jj-i1+1);
            end
           
            M3 = c/(alpha-1)^2 * prod(avgder(1 : jj-1)) * prod(avgder(1 : ii-1)) ...
                   * alpha^2 * Mprod(1, 1);
                       
            sigmaSE(jj, ii) = sum(sum(M)) + sum(M1) + sum(M2) + M3;
            sigmaSE(ii, jj) = sigmaSE(jj, ii);
           
        end
        
        % computation of E[u_t'(F_{t-1})]
        fun = @(x) 1/sqrt(2*pi) * exp(-x.^2/2) .* ...
            ( (tanh( muSE(jj)^2/sigmaSE(jj, jj) + muSE(jj)/sqrt(sigmaSE(jj, jj)) * x )).^2 + ...
            (tanh( - muSE(jj)^2/sigmaSE(jj, jj) + muSE(jj)/sqrt(sigmaSE(jj, jj)) * x )).^2 ) ;
        avgder(jj) = muSE(jj)/sigmaSE(jj, jj) * ( 1 - 1/2 * integral(fun,-Inf,Inf));
        
    end
end

