import torch
from datetime import datetime

class Timer:
    def __init__(self, name, verbose=False):
        self.name = name
        self.verbose = verbose
        self.start_time = None
        self.end_time = None
        self.elapsed_time = None

    def __enter__(self):
        self.start_time = datetime.now()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.end_time = datetime.now()
        self.elapsed_time = (self.end_time - self.start_time).total_seconds()
        if self.verbose:
            print(f"{self.name}: {self.elapsed_time:.4f}s")

    def get_time(self):
        if self.elapsed_time is None:
            raise ValueError("Timer hasn't been used yet or didn't complete properly")
        return self.elapsed_time

    def get_time_dict(self, details=False):
        if details:
            return {
                self.name: {
                    'start': self.start_time.strftime("%Y-%m-%dT%H:%M:%S.%f"),
                    'end': self.end_time.strftime("%Y-%m-%dT%H:%M:%S.%f"),
                    'elapsed': self.get_time(),
                }
            }
        else:
            return {self.name: self.get_time()}

class CUDATimer(Timer):
    def __enter__(self):
        torch.cuda.synchronize()
        self.start_time = datetime.now()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        torch.cuda.synchronize()
        self.end_time = datetime.now()
        self.elapsed_time = (self.end_time - self.start_time).total_seconds()
        if self.verbose:
            print(f"{self.name}: {self.elapsed_time:.4f}s")
