# Copyright (c) OpenMMLab. All rights reserved.
import collections

from mmcv.utils import build_from_cfg

# from ..builder import
from mmdet.datasets.builder import PIPELINES
from mmdet3d.datasets.pipelines import Compose
from projects.mmdet3d_plugin.utils import CUDATimer


@PIPELINES.register_module()
class CustomCompose:
    """Compose multiple transforms sequentially.
    Args:
        transforms (Sequence[dict | callable]): Sequence of transform object or
            config dict to be composed.
    """
    def __init__(self, transforms):
        assert isinstance(transforms, collections.abc.Sequence)
        self.transforms = []
        for transform in transforms:
            if isinstance(transform, dict):
                transform = build_from_cfg(transform, PIPELINES)
                self.transforms.append(transform)
            elif callable(transform):
                self.transforms.append(transform)
            else:
                raise TypeError('transform must be callable or a dict')

    def __call__(self, data, seed=0):
        """Call function to apply transforms sequentially.
        Args:
            data (dict): A result dict contains the data to transform.
        Returns:
           dict: Transformed data.
        """

        for t in self.transforms:

            if hasattr(t, 'seed'):
                data = t(data, seed=seed)
            else:
                data = t(data)

            if data is None:
                return None
        return data

    def __repr__(self):
        format_string = self.__class__.__name__ + '('
        for t in self.transforms:
            str_ = t.__repr__()
            if 'Compose(' in str_:
                str_ = str_.replace('\n', '\n    ')
            format_string += '\n'
            format_string += f'    {str_}'
        format_string += '\n)'
        return format_string


@PIPELINES.register_module()
class TimeCompose(Compose):

    def __call__(self, data):
        """Call function to apply transforms sequentially.

        Args:
            data (dict): A result dict contains the data to transform.

        Returns:
           dict: Transformed data.
        """

        time_dict = dict()
        for t in self.transforms:
            name = type(t).__name__
            assert name not in time_dict, f'duplicated pipeline name: {name}'
            with CUDATimer(name) as timer:
                data = t(data)
            time_details = timer.get_time_dict(details=True)
            time_dict.update(time_details)
            if data is None:
                return None, time_dict
        return data, time_dict

    def __repr__(self):
        format_string = self.__class__.__name__ + '('
        for t in self.transforms:
            format_string += '\n'
            format_string += f'    {t}'
        format_string += '\n)'
        return format_string
