import re
import argparse

def parse_tegrastats_line(line):
    patterns = {
        "VDD_GPU_SOC": r"VDD_GPU_SOC\s+\d+mW/(\d+)mW",
        "VDD_CPU_CV": r"VDD_CPU_CV\s+\d+mW/(\d+)mW",
        "VIN_SYS_5V0": r"VIN_SYS_5V0\s+\d+mW/(\d+)mW",
        "VDDQ_VDD2_1V8AO": r"VDDQ_VDD2_1V8AO\s+\d+mW/(\d+)mW"
    }
    
    parsed_powers = {}
    all_components_found = True
    
    for component_name, pattern in patterns.items():
        match = re.search(pattern, line)
        if match:
            try:
                parsed_powers[component_name] = int(match.group(1))
            except ValueError:
                all_components_found = False
                break
        else:
            all_components_found = False
            break
            
    if all_components_found and len(parsed_powers) == len(patterns):
        return parsed_powers
    else:
        return None

def calculate_average_power_from_log(log_file_path):
    component_keys = ["VDD_GPU_SOC", "VDD_CPU_CV", "VIN_SYS_5V0", "VDDQ_VDD2_1V8AO"]
    component_average_power_sums = {key: 0 for key in component_keys}
    overall_total_average_power_sum = 0
    valid_lines_count = 0
    skipped_line_details = []

    with open(log_file_path, 'r') as f:
        for line_number, line_content in enumerate(f, 1):
            line_content_stripped = line_content.strip()
            if not line_content_stripped:
                continue

            average_powers_for_line = parse_tegrastats_line(line_content_stripped)
            
            if average_powers_for_line:
                current_line_total_avg_power = 0
                for component, avg_power_value in average_powers_for_line.items():
                    if component in component_average_power_sums:
                        component_average_power_sums[component] += avg_power_value
                        current_line_total_avg_power += avg_power_value
                    else:
                        pass 
                
                overall_total_average_power_sum += current_line_total_avg_power
                valid_lines_count += 1
            else:
                skipped_line_details.append((line_number, line_content_stripped[:100]))

    results = {
        "component_averages": {},
        "average_total_system_power": None,
        "valid_lines_count": valid_lines_count,
        "skipped_line_details": skipped_line_details
    }

    if valid_lines_count > 0:
        for component_name, total_sum in component_average_power_sums.items():
            results["component_averages"][component_name] = total_sum / valid_lines_count
        
        results["average_total_system_power"] = overall_total_average_power_sum / valid_lines_count
    
    return results

if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description="Calculate average power consumption from a tegrastats log file. "
                    "The script uses the 'average_mW' values (Y in XmW/YmW) provided by tegrastats."
    )
    parser.add_argument(
        "logfile", 
        help="Path to the tegrastats log file."
    )
    args = parser.parse_args()
    
    analysis_results = calculate_average_power_from_log(args.logfile)

    for line_num, line_preview in analysis_results["skipped_line_details"]:
        print(f"Info: Skipping malformed or incomplete line {line_num}: \"{line_preview}...\"")

    if analysis_results["valid_lines_count"] > 0:
        print("\n--- Average Power Consumption (from tegrastats 'average_mW' fields) ---")
        
        for component_name, avg_component_power in analysis_results["component_averages"].items():
            print(f"  Average {component_name}: {avg_component_power:.2f} mW")
        
        avg_total_system_power = analysis_results["average_total_system_power"]
        if avg_total_system_power is not None:
             print(f"  Average Total System Power (sum of component averages): {avg_total_system_power:.2f} mW")
        
        print(f"\nProcessed {analysis_results['valid_lines_count']} valid lines from the log file.")
    else:
        print("No valid tegrastats lines found in the file to calculate averages.")
        if analysis_results["skipped_line_details"]:
             print(f"(Note: {len(analysis_results['skipped_line_details'])} lines were skipped due to parsing issues or missing components.)")