# import dependencies
import tensorflow as tf
import numpy as np
import argparse



parser = argparse.ArgumentParser()
parser.add_argument('-m', '--model_type', type=str, default="lenet", dest="model_type")
parser.add_argument('-s', '--seed', type=int, default=1, dest="seed")
parser.add_argument('-w', '--weight_decay', type=float, default=1e-4, dest="weight_decay")
parser.add_argument('-l', '--init_lr', type=float, default=5e-3, dest="init_lr")
parser.add_argument('-b', '--momentum', type=float, default=0.9, dest="momentum")
parser.add_argument('--batch_size', type=int, default=50, dest="batch_size")
parser.add_argument('--dtype', type=str, default="float32", dest="dtype")
parser.add_argument('--para_str', type=str, default="Htop5e+03", dest="para_str")
parser.add_argument('--epochs', type=int, default=20, dest="epochs")
parser.add_argument('--wr', default=False, action=argparse.BooleanOptionalAction)

args = parser.parse_args()
model_type = args.model_type
seed = args.seed
weight_decay = args.weight_decay
init_lr = args.init_lr
momentum = args.momentum
batch_size = args.batch_size
dtype = args.dtype
para_str = args.para_str
epochs = args.epochs
with_replacement = args.wr



# network specific parameters
model_str = model_type+"_"+f"wd{weight_decay:.0e}_lr{init_lr:.0e}_b{batch_size:.0f}_m{momentum:.2f}_"+dtype[-2:]+"_"+str(seed)
initial_learning_rate = init_lr
learning_rate_now = 0.98**100*initial_learning_rate
tf.keras.backend.set_floatx(dtype)

# load dataset
(x_train, y_train), (x_test, y_test) = tf.keras.datasets.cifar10.load_data()
x_train = x_train.astype(dtype)
x_test = x_test.astype(dtype)
x_train, x_test = x_train / 255.0, x_test / 255.0

# load the model
model = tf.keras.models.load_model(model_str+"/data/trained_model")



@tf.function
def flatten_tf(params):
  return tf.concat([tf.reshape(_params, [-1]) for _params in params], axis=0)

# Defining what to recorde during training: one specific layer or projections onto Hessian eigenvectors
if para_str[:4] == "Htop":
  dim = int(float(para_str[4:]))
  H_eigvec = np.load(model_str+"/data/"+para_str+"_eigvec.npy").astype(dtype)
  def current_weights():
    weights = np.array(flatten_tf(model.trainable_variables))
    weights = np.matmul(H_eigvec, weights)
    return weights
elif para_str[:3] == "lyr":
  layer_idx = int(para_str[3:])
  parameters = model.trainable_variables[layer_idx]
  dim = len(np.array(parameters).flatten())
  def current_weights():
    return np.array(tf.reshape(parameters, [-1]))



optimizer = tf.keras.optimizers.SGD(learning_rate=learning_rate_now, momentum=momentum)
train_loss = tf.keras.metrics.Mean(name='train_loss')

# train function
@tf.function
def train_step(images, labels):
  with tf.GradientTape() as tape:
    predictions = model(images, training=True)
    regularization_loss=tf.math.add_n(model.losses)
    pred_loss=model.loss(labels, predictions)
    total_loss=pred_loss + regularization_loss
  gradients = tape.gradient(total_loss, model.trainable_variables)
  optimizer.apply_gradients(zip(gradients, model.trainable_variables))
  train_loss(total_loss)



# train the network further
num_batches = int(np.ceil(y_train.size/batch_size))
weights = np.zeros((num_batches*epochs, dim))
batch_index = np.array([0])
train_ds = tf.data.Dataset.from_tensor_slices((x_train, y_train)).shuffle(y_train.size).batch(batch_size)

for _epoch in range(epochs):
  # Draw the examples with replacement if necessary
  if with_replacement == True:
    rng = np.random.default_rng(seed+_epoch)
    idx_list = rng.integers(low=0, high=y_train.size, size=y_train.size)
    train_ds = tf.data.Dataset.from_tensor_slices((x_train[idx_list], y_train[idx_list])).batch(batch_size)

  # Reset the metrics at the start of the next epoch
  train_loss.reset_states()

  for images, labels in train_ds:
    train_step(images, labels)
    weights[batch_index[0]] = current_weights()
    np.add.at(batch_index, 0, 1)

  output_str = f"Epoch {_epoch+1}/{epochs}, Loss: {train_loss.result():.3e}"
  print(output_str)

if with_replacement == True:
  np.save(model_str+"/data/weights_timeseries_WR_"+para_str, weights)
else:
  np.save(model_str+"/data/weights_timeseries_"+para_str, weights)





