"""Generate raw data to investigate mesh size effect."""

import argparse
from glob import glob
import pathlib

import numpy as np
import siml

import generate_data


RAW_DIRECTORY = pathlib.Path('data/raw')
TIMEOUT = 60 * 10
CONDUCTIVITIES = (1.e-2,)
MESHING_SUCCESS_FILE = pathlib.Path('meshing_success')
GENERATION_SUCCESS_FILE = pathlib.Path('generation_success')
TEMPERATURE_GRADIENT_DIRECTION = 1
ALPHAS = (1., 5., 10.)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        'input_geo_directories',
        type=pathlib.Path,
        nargs='+',
        help='Input geo file directories')
    parser.add_argument(
        '--output-directory',
        '-o',
        type=pathlib.Path,
        default=RAW_DIRECTORY,
        help='Output base direcoty')
    parser.add_argument(
        '--conductivities',
        '-c',
        type=float,
        nargs='+',
        default=CONDUCTIVITIES,
        help='Thermal conductivities')
    parser.add_argument(
        '--alphas',
        '-a',
        type=float,
        default=ALPHAS,
        nargs='+',
        help='The number of repetition.')
    args = parser.parse_args()

    input_geo_files = [pathlib.Path(f) for f in np.unique(np.concatenate([
        glob(str(input_geo_directory / '**/*.geo'), recursive=True)
        for input_geo_directory in args.input_geo_directories]))]

    for input_geo_file in input_geo_files:
        output_directory = siml.prepost.determine_output_directory(
            input_geo_file.parent, args.output_directory, 'external')

        output_vtk_file = generate_data.generate_mesh_if_needed(
            output_directory, input_geo_file, n_thread=args.n_thread)
        if output_vtk_file is None:
            continue

        for alpha in args.alphas:
            generate_data.generate_heat(
                output_vtk_file, output_directory,
                conductivities=args.conductivities, alpha=alpha)

    return


def t_init_funtion(nodes):
    t_init = nodes[:, TEMPERATURE_GRADIENT_DIRECTION]
    return t_init


if __name__ == '__main__':
    main()
