# Copyright (c) ByteDance, Inc. and its affiliates.
# All rights reserved.
#
# This source code is licensed under the license found in the
# LICENSE file in the root directory of this source tree.

"""
Mostly copy-paste from DINO and timm library:
https://github.com/facebookresearch/dino
https://github.com/rwightman/pytorch-image-models/blob/master/timm/models/vision_transformer.py
"""

import math
import torch
import torch.nn as nn

from functools import partial
from timm.models import register_model
import warnings
from torch import Tensor

import torch.nn.functional as F

import random


def _no_grad_trunc_normal_(tensor, mean, std, a, b):
    # Cut & paste from PyTorch official master until it's in a few official releases - RW
    # Method based on https://people.sc.fsu.edu/~jburkardt/presentations/truncated_normal.pdf
    def norm_cdf(x):
        # Computes standard normal cumulative distribution function
        return (1. + math.erf(x / math.sqrt(2.))) / 2.

    if (mean < a - 2 * std) or (mean > b + 2 * std):
        warnings.warn("mean is more than 2 std from [a, b] in nn.init.trunc_normal_. "
                      "The distribution of values may be incorrect.",
                      stacklevel=2)

    with torch.no_grad():
        # Values are generated by using a truncated uniform distribution and
        # then using the inverse CDF for the normal distribution.
        # Get upper and lower cdf values
        l = norm_cdf((a - mean) / std)
        u = norm_cdf((b - mean) / std)

        # Uniformly fill tensor with values from [l, u], then translate to
        # [2l-1, 2u-1].
        tensor.uniform_(2 * l - 1, 2 * u - 1)

        # Use inverse cdf transform for normal distribution to get truncated
        # standard normal
        tensor.erfinv_()

        # Transform to proper mean, std
        tensor.mul_(std * math.sqrt(2.))
        tensor.add_(mean)

        # Clamp to ensure it's in the proper range
        tensor.clamp_(min=a, max=b)
        return tensor


def trunc_normal_(tensor, mean=0., std=1., a=-2., b=2.):
    # type: (Tensor, float, float, float, float) -> Tensor
    return _no_grad_trunc_normal_(tensor, mean, std, a, b)



def drop_path(x, drop_prob: float = 0., training: bool = False):
    if drop_prob == 0. or not training:
        return x
    keep_prob = 1 - drop_prob
    shape = (x.shape[0],) + (1,) * (x.ndim - 1)  # work with diff dim tensors, not just 2D ConvNets
    random_tensor = keep_prob + torch.rand(shape, dtype=x.dtype, device=x.device)
    random_tensor.floor_()  # binarize
    output = x.div(keep_prob) * random_tensor
    return output


class DropPath(nn.Module):
    """Drop paths (Stochastic Depth) per sample  (when applied in main path of residual blocks).
    """
    def __init__(self, drop_prob=None):
        super(DropPath, self).__init__()
        self.drop_prob = drop_prob

    def forward(self, x):
        return drop_path(x, self.drop_prob, self.training)


class Mlp(nn.Module):
    def __init__(self, in_features, hidden_features=None, out_features=None, act_layer=nn.GELU, drop=0.):
        super().__init__()
        out_features = out_features or in_features
        hidden_features = hidden_features or in_features
        self.fc1 = nn.Linear(in_features, hidden_features)
        self.act = act_layer()
        self.fc2 = nn.Linear(hidden_features, out_features)
        self.drop = nn.Dropout(drop)

    def forward(self, x):
        x = self.fc1(x)
        x = self.act(x)
        x = self.drop(x)
        x = self.fc2(x)
        x = self.drop(x)
        return x


class Attention(nn.Module):
    def __init__(self, dim, num_heads=8, qkv_bias=False, qk_scale=None, attn_drop=0., proj_drop=0.):
        super().__init__()
        self.num_heads = num_heads
        head_dim = dim // num_heads
        self.scale = qk_scale or head_dim ** -0.5

        self.qkv = nn.Linear(dim, dim * 3, bias=qkv_bias)
        self.attn_drop = nn.Dropout(attn_drop)
        self.proj = nn.Linear(dim, dim)
        self.proj_drop = nn.Dropout(proj_drop)

    def forward(self, x):
        B, N, C = x.shape
        qkv = self.qkv(x).reshape(B, N, 3, self.num_heads, C // self.num_heads).permute(2, 0, 3, 1, 4)
        q, k, v = qkv[0], qkv[1], qkv[2]

        attn = (q @ k.transpose(-2, -1)) * self.scale
        attn = attn.softmax(dim=-1)
        attn = self.attn_drop(attn)

        x = (attn @ v).transpose(1, 2).reshape(B, N, C)
        x = self.proj(x)
        x = self.proj_drop(x)
        return x, attn

class Block(nn.Module):
    def __init__(self, dim, num_heads, mlp_ratio=4., qkv_bias=False, qk_scale=None, drop=0., 
                 attn_drop=0., drop_path=0., act_layer=nn.GELU, norm_layer=nn.LayerNorm, init_values=0):
        super().__init__()
        self.norm1 = norm_layer(dim)
        self.attn = Attention(
            dim, num_heads=num_heads, qkv_bias=qkv_bias, qk_scale=qk_scale, attn_drop=attn_drop, proj_drop=drop)
        self.drop_path = DropPath(drop_path) if drop_path > 0. else nn.Identity()
        self.norm2 = norm_layer(dim)
        mlp_hidden_dim = int(dim * mlp_ratio)
        self.mlp = Mlp(in_features=dim, hidden_features=mlp_hidden_dim, act_layer=act_layer, drop=drop)

        if init_values > 0:
            self.gamma_1 = nn.Parameter(init_values * torch.ones((dim)), requires_grad=True)
            self.gamma_2 = nn.Parameter(init_values * torch.ones((dim)), requires_grad=True)
        else:
            self.gamma_1, self.gamma_2 = None, None

    def forward(self, x, return_attention=False):
        y, attn = self.attn(self.norm1(x))
        if return_attention:
            return attn
        if self.gamma_1 is None:
            x = x + self.drop_path(y)
            x = x + self.drop_path(self.mlp(self.norm2(x)))
        else:
            x = x + self.drop_path(self.gamma_1 * y)
            x = x + self.drop_path(self.gamma_2 * self.mlp(self.norm2(x)))
        return x

"""
STEM NETWORK IMPLEMENTATIONS ===========================================================================================================
"""  


from einops import rearrange

    
class GroupedPatchEmbedSimple(nn.Module):
    """ Image to Patch Embedding with Grouped Projections.
        Only simple grouped projection and then concatenate. No FFN.
    """
    def __init__(self, img_size=224, patch_size=16, embed_dim=768, in_groups=[[0,1,2,3]], reduce_group=True, keep_channels=[3],
                 aggregation="post", debug=False):
        super().__init__()

        self.img_size = img_size
        self.patch_size = patch_size
        self.num_patches = (img_size // patch_size) * (img_size // patch_size)
        self.in_groups = in_groups
        self.keep_channels=keep_channels
        self.aggregate = True if aggregation=="pre" else False

        self.debug=debug

        if reduce_group:
            self.proj = nn.ModuleList([nn.Conv2d(1, embed_dim//len(self.in_groups), \
                                        kernel_size=patch_size, stride=patch_size) \
                                        for _ in self.in_groups])
        else:
            self.proj = nn.ModuleList([nn.Conv2d(1, embed_dim, \
                            kernel_size=patch_size, stride=patch_size) \
                            for _ in self.in_groups])

        self.ins = nn.ModuleList([nn.InstanceNorm2d(1) \
                for _ in self.in_groups])

    def process_grp(self, x, i, gg):

        b = x.shape[0]

        x_ = x[:, gg]

        if self.training and x_.shape[1]>1:

            keep_idxs = random.sample(range(x_.shape[1]), random.choice(self.keep_channels))
            x_ = x[:, keep_idxs]
            gg = keep_idxs

        x_ = rearrange(x_, 'b (c n) h w -> (b n) c h w', c=1)
        x_ = self.ins[i](x_)
        x_ = rearrange(x_, '(b n) c h w -> b c (n h) w', b=b)

        x_ = self.proj[i](x_)
        x_ = rearrange(x_, 'b c (n h) w -> b c n h w', n=len(gg))

        if self.aggregate:
            return torch.mean(x_, dim=2, keepdim=True)

        return x_

    def forward(self, x):

        B, C, H, W = x.shape

        if self.debug:
            print(f"input.shape: {x.shape}")

        x = [self.process_grp(x,i,gg) for i,gg in enumerate(self.in_groups)]

        return x
    
class PatchEmbed(nn.Module):
    """ Image to Patch Embedding
    """
    def __init__(self, img_size=224, patch_size=16, in_chans=3, embed_dim=768, groups=1):
        super().__init__()
        num_patches = (img_size // patch_size) * (img_size // patch_size)
        self.img_size = img_size
        self.patch_size = patch_size
        self.num_patches = num_patches

        self.proj = nn.Conv2d(in_chans, embed_dim, kernel_size=patch_size, stride=patch_size, groups=groups)
            
    def forward(self, x):
        B, C, H, W = x.shape
        return self.proj(x)
    

class StreamCombiner(nn.Module):
    """ Image to Patch Embedding
    """
    def __init__(self, dim=768):
        super().__init__()

        self.norm = nn.LayerNorm(dim)
            
    def forward(self, x1, x2, all_chans=None):

        x1 = rearrange(x1, "(b c) ... -> b c ...", c=all_chans[0])
        x2 = rearrange(x2, "(b c) ... -> b c ...", c=all_chans[1])

        x1 = x1.mean(1)
        x2 = x2.mean(1)

        x = torch.cat([x1, x2], dim=-1)
        x = self.norm(x)

        return x
    


class VisionTransformer(nn.Module):
    """ Vision Transformer """
    def __init__(self, img_size=[224], patch_size=16, in_chans=3, num_classes=0, embed_dim=768, depth=12,
                 num_heads=12, mlp_ratio=4., qkv_bias=False, qk_scale=None, drop_rate=0., attn_drop_rate=0.,
                 drop_path_rate=0., norm_layer=partial(nn.LayerNorm, eps=1e-6), return_all_tokens=False, 
                 init_values=0, use_mean_pooling=False, masked_im_modeling=False, num_prefix_tokens=1,
                 **cce_kwargs
                 ):
        super().__init__()
        self.num_features = self.embed_dim = embed_dim
        self.return_all_tokens = return_all_tokens

        self.in_groups = cce_kwargs['in_groups']
        self.keep_channels = cce_kwargs['keep_channels']
        self.aggregation = cce_kwargs['aggregation']

        self.patch_embed = GroupedPatchEmbedSimple(
                    img_size=img_size[0], patch_size=patch_size, embed_dim=embed_dim, 
                    in_groups=self.in_groups, keep_channels=self.keep_channels, aggregation=self.aggregation)

        num_patches = self.patch_embed.num_patches

        small_dim = embed_dim//2

        self.cls_token = nn.Parameter(torch.zeros(1, num_prefix_tokens, embed_dim))
        self.pos_embed = nn.Parameter(torch.zeros(1, num_patches + num_prefix_tokens, embed_dim))
        self.pos_drop = nn.Dropout(p=drop_rate)
        self.num_prefix_tokens = num_prefix_tokens

        pre_layers = list(range(4)) if cce_kwargs['n_pre_layers'] == None else list(range(cce_kwargs['n_pre_layers']))

        if cce_kwargs['n_post_layers'] == -1:
            post_layers_all = self.solve_final_layers(len(pre_layers), small_dim, embed_dim)
        else:
            post_layers_all = cce_kwargs['n_post_layers']
            
        post_layers = list(range(pre_layers[-1]+1, pre_layers[-1]+1+post_layers_all))

        dpr = [x.item() for x in torch.linspace(0, drop_path_rate, len(pre_layers + post_layers))]  # stochastic depth decay rule

        self.cnt_blocks = nn.ModuleList([
            Block(
                dim=small_dim, num_heads=num_heads, mlp_ratio=mlp_ratio, qkv_bias=qkv_bias, qk_scale=qk_scale,
                drop=drop_rate, attn_drop=attn_drop_rate, drop_path=dpr[i], norm_layer=norm_layer, 
                init_values=init_values)
            for i in pre_layers])
        
        self.stn_blocks = nn.ModuleList([
            Block(
                dim=small_dim, num_heads=num_heads, mlp_ratio=mlp_ratio, qkv_bias=qkv_bias, qk_scale=qk_scale,
                drop=drop_rate, attn_drop=attn_drop_rate, drop_path=dpr[i], norm_layer=norm_layer, 
                init_values=init_values)
            for i in pre_layers])
        
        self.combiner = StreamCombiner(dim=embed_dim)

        self.cmb_blocks = nn.ModuleList([
            Block(
                dim=embed_dim, num_heads=num_heads, mlp_ratio=mlp_ratio, qkv_bias=qkv_bias, qk_scale=qk_scale,
                drop=drop_rate, attn_drop=attn_drop_rate, drop_path=dpr[i], norm_layer=norm_layer, 
                init_values=init_values)
            for i in post_layers])

        self.norm = nn.Identity() if use_mean_pooling else norm_layer(embed_dim)
        self.fc_norm = norm_layer(embed_dim) if use_mean_pooling else None
        # Classifier head
        self.head = nn.Linear(embed_dim, num_classes) if num_classes > 0 else nn.Identity()

        trunc_normal_(self.pos_embed, std=.02)
        trunc_normal_(self.cls_token, std=.02)
        self.apply(self._init_weights)

        # masked image modeling
        self.masked_im_modeling = masked_im_modeling
        if masked_im_modeling:
            self.masked_embed = nn.Parameter(torch.zeros(1, embed_dim))
            self.keep_mask=True
        else:
            self.keep_mask=False

        return
    
    def solve_final_layers(self, L_branch, D_branch=384, D_final=768):

        P_orig=12 * D_final**2 * 12
        P_branch_layer = 2 * 12 * D_branch**2
        P_final_layer = 12 * D_final**2

        P_branch = L_branch * P_branch_layer # Total used by branches
        P_remaining = P_orig - P_branch # Remaining budget
        L_final = P_remaining // P_final_layer # Solve for how many final layers we can fit
        return int(L_final)


    def _init_weights(self, m):
        if isinstance(m, nn.Linear):
            trunc_normal_(m.weight, std=.02)
            if isinstance(m, nn.Linear) and m.bias is not None:
                nn.init.constant_(m.bias, 0)
        elif isinstance(m, nn.LayerNorm):
            nn.init.constant_(m.bias, 0)
            nn.init.constant_(m.weight, 1.0)

    def interpolate_pos_encoding(self, x, w, h, c_idx):

        pos_embed = self.pos_embed[:, :, c_idx*(self.embed_dim//2):(c_idx+1)*(self.embed_dim//2)]

        npatch = x.shape[1] - 1
        N = pos_embed.shape[1] - self.num_prefix_tokens
        if npatch == N and w == h:
            return pos_embed
        class_pos_embed = pos_embed[:, 0:self.num_prefix_tokens]
        patch_pos_embed = pos_embed[:, self.num_prefix_tokens:]
        dim = x.shape[-1]
        w0 = w // self.patch_embed.patch_size
        h0 = h // self.patch_embed.patch_size
        # we add a small number to avoid floating point error in the interpolation
        # see discussion at https://github.com/facebookresearch/dino/issues/8
        w0, h0 = w0 + 0.1, h0 + 0.1
        patch_pos_embed = nn.functional.interpolate(
            patch_pos_embed.reshape(1, int(math.sqrt(N)), int(math.sqrt(N)), dim).permute(0, 3, 1, 2),
            scale_factor=(w0 / math.sqrt(N), h0 / math.sqrt(N)),
            mode='bicubic',
        )
        assert int(w0) == patch_pos_embed.shape[-2] and int(h0) == patch_pos_embed.shape[-1]
        patch_pos_embed = patch_pos_embed.permute(0, 2, 3, 1).view(1, -1, dim)
        return torch.cat((class_pos_embed, patch_pos_embed), dim=1)

    def prepare_tokens(self, x, mask=None):
        B, nc, w, h = x.shape
        # patch linear embedding

        x = self.patch_embed(x) # x = [x_context, x_stain]
        
        cnt_chans = x[0].shape[2]
        stn_chans = x[1].shape[2]

        all_chans = [cnt_chans, stn_chans]

        x_new = []

        for c_idx, xx in enumerate(x):
            xx = rearrange(xx, "b d c h w -> (b c) d h w")

            if mask is not None:
                msk = mask.unsqueeze(1).expand(-1, all_chans[c_idx], -1, -1)
                msk = rearrange(msk, "b c h w -> (b c) h w")
                mask_embed = self.masked_embed[:, c_idx*(self.embed_dim//2):(c_idx+1)*(self.embed_dim//2)]
                if self.keep_mask:
                    xx.permute(0, 2, 3, 1)[msk, :] = mask_embed.to(xx.dtype)
                else:
                    xx.permute(0, 2, 3, 1)[msk, :] += 0.1*mask_embed.to(xx.dtype)
                "masking done"

            xx = xx.flatten(2).transpose(1, 2)
            # add the [CLS] token to the embed patch tokens
            cls_tokens = self.cls_token[:,:,c_idx*(self.embed_dim//2):(c_idx+1)*(self.embed_dim//2)].expand(xx.shape[0], -1, -1)
            xx = torch.cat((cls_tokens, xx), dim=1)
            # add positional encoding to each token
            xx = xx + self.interpolate_pos_encoding(xx, w, h, c_idx)
            xx = self.pos_drop(xx)
            xx = rearrange(xx, "(b c) N d -> b c N d", c=all_chans[c_idx])
            x_new.append(xx)

        return tuple(x_new), all_chans

    def forward(self, x, return_all_tokens=None, mask=None):
        # mim
        if self.masked_im_modeling:
            assert mask is not None
            x, all_chans = self.prepare_tokens(x, mask=mask)
        else:
            x, all_chans = self.prepare_tokens(x)

        x_cnt, x_stn = x

        x_cnt = rearrange(x_cnt, "b c ... -> (b c) ...")
        x_stn = rearrange(x_stn, "b c ... -> (b c) ...")

        for cnt_blk, stn_blk in zip(self.cnt_blocks, self.stn_blocks):
            x_cnt = cnt_blk(x_cnt)
            x_stn = stn_blk(x_stn)

        x = self.combiner(x_cnt, x_stn, all_chans)

        for blk in self.cmb_blocks:
            x = blk(x)

        x = self.norm(x)

        if self.fc_norm is not None:
            x[:, 0] = self.fc_norm(x[:, 1:, :].mean(1))
        
        return_all_tokens = self.return_all_tokens if \
            return_all_tokens is None else return_all_tokens
        if return_all_tokens:
            return x
        
        return x[:, 0]

    def get_last_selfattention(self, x):
        x = self.prepare_tokens(x)
        for i, blk in enumerate(self.blocks):
            if i < len(self.blocks) - 1:
                x = blk(x)
            else:
                # return attention of the last block
                return blk(x, return_attention=True)

    def get_intermediate_layers(self, x, n=1):
        x = self.prepare_tokens(x)
        # we return the output tokens from the `n` last blocks
        output = []
        for i, blk in enumerate(self.blocks):
            x = blk(x)
            if len(self.blocks) - i <= n:
                output.append(self.norm(x))
        return output
        
    def get_num_layers(self):
        return len(self.blocks)

    def mask_model(self, x, mask):
        if self.keep_mask:
            x.permute(0, 2, 3, 1)[mask, :] = self.masked_embed.to(x.dtype)
        else:
            x.permute(0, 2, 3, 1)[mask, :] += 0.1*self.masked_embed.to(x.dtype)
            
        return x

def vit_tiny(patch_size=16, **kwargs):
    model = VisionTransformer(
        patch_size=patch_size, embed_dim=192, depth=12, num_heads=3, mlp_ratio=4,
        qkv_bias=True, **kwargs)
    return model

def vit_small(patch_size=16, **kwargs):
    model = VisionTransformer(
        patch_size=patch_size, embed_dim=384, depth=12, num_heads=6, mlp_ratio=4,
        qkv_bias=True, **kwargs)
    return model

def vit_base(patch_size=16, **kwargs):
    model = VisionTransformer(
        patch_size=patch_size, embed_dim=768, depth=12, num_heads=12, mlp_ratio=4,
        qkv_bias=True, **kwargs)
    return model

def vit_large(patch_size=16, **kwargs):
    model = VisionTransformer(
        patch_size=patch_size, embed_dim=1024, depth=24, num_heads=16, mlp_ratio=4,
        qkv_bias=True, **kwargs)
    return model



if __name__ == '__main__':

    cce_kwargs = {
        "in_groups" : [[0,1,2], [3]],  #  context and concept group channel indexes
        "n_pre_layers": 2,
        "n_post_layers": -1,
        "aggregation": "post", # "pre" for pre-aggregation
    }

    "we randomly keep from rate distribution of [1,2,3] channels. For the teacher network during training, it is set to [3] (all channels)."
    model = vit_small(masked_im_modeling=False, keep_channels=[1,2,3], **cce_kwargs).cuda()
    print(model)

    model.eval()

    img = torch.randn(8, 4, 224, 224).cuda()
    out = model(img, return_all_tokens=False)

    print(out.shape)

