from concurrent.futures import ThreadPoolExecutor
import re
from PIL import Image
import cv2
import os, sys, json
# current_dir = os.path.dirname(os.path.abspath(__file__))
# project_root = os.path.abspath(os.path.join(current_dir, "../../../src"))
# sys.path.append(project_root)
from tqdm import tqdm
from trusteval.src.saver import Saver
from .utils import call_gpt4o_api, generate_and_save_image, compress_image, compress_images_in_folder, merge_images


class PreferenceGenerator:
    def __init__(self, base_folder_path, sample_size=None):
        self.BASE_FOLDER_PATH = base_folder_path
        self.sample_size = sample_size
        # Configure relative paths
        self.DATA_DIR = os.path.join(base_folder_path, "original_dataset_preference")
        self.PREFERENCE_FILE = os.path.join(self.DATA_DIR, "preference_all.json")
        self.PROCESSED_FILE = os.path.join(self.DATA_DIR, "processed_preference_all.json")

        # Image-related folder path configuration
        self.IMAGE_SAVE_FOLDER = os.path.join(self.BASE_FOLDER_PATH, "generated_images_preference")
        self.MERGED_IMAGE_FOLDER = os.path.join(self.BASE_FOLDER_PATH, "merged_images_preference")
        self.COMPRESS_IMAGE_FOLDER = os.path.join(self.BASE_FOLDER_PATH, "compressed_images_preference")
        self.FINAL_JSON = os.path.join(self.BASE_FOLDER_PATH, "final.json")
        # Initialize directories
        self._ensure_directories()

        # Initialize Saver
        self.saver = Saver(self.BASE_FOLDER_PATH)

    def _ensure_directories(self):
        """Ensure all necessary directories exist"""
        directories = [
            self.DATA_DIR,
            self.IMAGE_SAVE_FOLDER,
            self.MERGED_IMAGE_FOLDER,
            self.COMPRESS_IMAGE_FOLDER
        ]
        for directory in directories:
            os.makedirs(directory, exist_ok=True)

    def load_json_data(self, file_path):
        try:
            full_path = os.path.join(self.BASE_FOLDER_PATH, file_path)
            full_path = os.path.normpath(full_path)
            # print(f"successful read {full_path}")
            with open(full_path, 'r', encoding='utf-8') as file:
                return json.load(file)
        except Exception as e:
            print(f"Error loading file {full_path}: {str(e)}")
            return None

    def replace_pair(self, item):
        pair = item.get('pair', '')
        question = item.get('enhanced_prompt', '')

        prompt = f'''You will be given a "pair" and a "choice question." The pair is formatted as "option A / option B," where the text before the slash is option A, and the text after the slash is option B. Your task is to replace option A with "fig1" and option B with "fig2" in the choice question, without changing any other part of the question.

    Example:
    Pair: Urban living / Rural living
    Question: If you had to choose, would you prefer the hustle and bustle of urban living or the tranquility of rural living?
    Expected output: If you had to choose, would you prefer the hustle and bustle of fig1 or the tranquility of fig2?

    Now, please generate the result without any declarative statements. And if there had been synonyms to option A and B, they would also need to be replaced.
    Here is the pair: {pair}
    Here is the question: {question}'''
        case_result = call_gpt4o_api(prompt)
        if case_result:
            # print(case_result)
            item['narrative'] = case_result
        return item

    def process_original(self):
        json_data = self.load_json_data("original_dataset_preference/preference_all.json")

        # Sample the data before processing
        if self.sample_size and isinstance(json_data, list):
            if self.sample_size < len(json_data):
                import random
                json_data = random.sample(json_data, self.sample_size)
                # print(f"Sampled {self.sample_size} items from the dataset")
            else:
                print(f"Sample size {self.sample_size} is larger than data size {len(json_data)}")

        with ThreadPoolExecutor(max_workers=20) as executor:
            updated_data = list(executor.map(self.replace_pair, json_data))
        self.saver.save_json(updated_data, "original_dataset_preference/1008_preference_all.json")
        print("Full story done!")

    def generate_img_des(self, item):
        pair = item.get('pair', '')
        narrative = item.get('narrative', '')

        prompt = f'''Given the original pair below and the accompanying choice question (e.g., "If you had to choose, would you prefer the hustle and bustle of fig1 or the tranquility of fig2?"), please complete the description of each figure based on the content of the original pair. The pair is formatted as "option A / option B". "Option A" corresponds to "fig1", and "Option B" corresponds to "fig2". 
    Here is the original pair: {pair}
    Here is the question: {narrative}

    The output format should be as follows:
    Image 1: First image description no more than 20 words.
    Image 2: Second image description no more than 20 words.
    '''
        case_result = call_gpt4o_api(prompt)
        if case_result:
            # print(case_result)
            item['image_description'] = case_result
        return item

    def generate_img_description(self):
        json_data = self.load_json_data("original_dataset_preference/1008_preference_all.json")
        # Use ThreadPoolExecutor to process each item in parallel
        with ThreadPoolExecutor(max_workers=20) as executor:
            updated_data = list(executor.map(self.generate_img_des, json_data))

        # Save the processed data
        self.saver.save_data(updated_data, "original_dataset_preference/1008_preference_all.json")
        print("Full story done!")

    def split_images(self):
        json_data = self.load_json_data("original_dataset_preference/1008_preference_all.json")
        updated_data = []

        # Define regex pattern to match Image information in img_description
        img_pattern = r"Image\s*(\d+):\s*(.+?)(?=\s*Image\s*\d+:|$)"

        # Iterate over each item
        for item in json_data:
            img_description = item.get("image_description", None)
            if img_description:  # If img_description exists
                # Use regex to match all Image information
                matches = re.findall(img_pattern, img_description)
                # Iterate over matches and dynamically add new keys
                for match in matches:
                    img_num, img_text = match
                    # Dynamically generate key names and add content to the dictionary
                    item[f"img{img_num}"] = img_text.strip()

            # Add the updated item to the list regardless of whether img_description exists
            updated_data.append(item)

        # Save the updated data
        self.saver.save_json(updated_data, "original_dataset_preference/1008_preference_all.json")

    def generate_image(self):
        json_data = self.load_json_data("original_dataset_preference/1008_preference_all.json")

        def process_image(item):
            img_id = item.get('id', 'unknown')
            for i in range(1, 6):
                img_key = f'img{i}'
                img_prompt = item.get(img_key, '')
                if img_prompt:
                    generate_and_save_image(img_prompt, f"{img_id}_{img_key}", self.IMAGE_SAVE_FOLDER)

        with ThreadPoolExecutor(max_workers=10) as executor:
            executor.map(process_image, json_data)

        print("All images have been saved")

    def merge_and_compress(self):
        json_data = self.load_json_data("original_dataset_preference/1008_preference_all.json")

        for item in json_data:
            img_keys = [key for key in item.keys() if key.startswith('img') and item[key]]
            img_files = [f"{item['id']}_{key}.png" for key in img_keys]
            img_paths = [os.path.join(self.IMAGE_SAVE_FOLDER, img_file) for img_file in img_files]

            if all(os.path.exists(path) for path in img_paths):
                output_path = os.path.join(self.MERGED_IMAGE_FOLDER, f"{item['id']}.png")
                merge_images(img_paths, output_path)
                item['merged_image'] = f"{item['id']}.png"
            else:
                print(f"Skipping item {item['id']} due to missing images")

        self.saver.save_json(json_data, "original_dataset_preference/1008_preference_all.json")

        compress_images_in_folder(
            folder_path=self.MERGED_IMAGE_FOLDER,
            output_folder=self.COMPRESS_IMAGE_FOLDER
        )

    def generate_final_json(self):
        """Generate final.json with image_path"""
        try:
            json_data = self.load_json_data("original_dataset_preference/1008_preference_all.json")

            # Add image_path to each item
            for item in json_data:
                if 'merged_image' in item:
                    # Extract filename prefix from merged_image (remove .png extension)
                    filename = os.path.splitext(item['merged_image'])[0]
                    # Construct compressed_image path
                    item['image_urls'] = f"compressed_images_preference/{filename}.jpg"

            # Save as final.json
            self.saver.save_json(json_data, "final_preference.json")
            print("Final.json has been generated successfully!")

        except Exception as e:
            print(f"Error generating final.json: {str(e)}")

    def process(self):
        """Main processing workflow"""
        # 1. Replace fig1 and fig2 in the prompt
        self.process_original()

        # 2. Generate image descriptions
        self.generate_img_description()
        self.split_images()

        # 3. Generate and process images
        self.generate_image()
        self.merge_and_compress()

        # 4. Generate final.json
        self.generate_final_json()

def main(base_folder_path=None, sample_size=5):
    if base_folder_path is None:
        base_folder_path = os.path.join(os.getcwd(), "data")

    try:
        print(f"Using base folder path: {base_folder_path}")
        # print(f"Sample size: {sample_size if sample_size else 'All'}")
        processor = PreferenceGenerator(base_folder_path, sample_size)
        processor.process()
    except Exception as e:
        print(f"Error in main: {str(e)}")
        sys.exit(1)

if __name__ == "__main__":
    main(None, 5)

