from __future__ import annotations
from collections import deque
from dataclasses import dataclass
from typing import Callable, Dict, Iterable, List, Optional, Tuple, Set

Symbol = Optional[str]  # None = epsilon


@dataclass(frozen=True)
class Rule:
    q_to: str
    sym: Symbol
    update: Callable[[Tuple[int, ...]], Tuple[int, ...]]


def _is_probable_prime(n: int) -> bool:
    """Fast Miller–Rabin. Deterministic for 64-bit, probabilistic for larger."""
    if n < 2:
        return False
    # small primes
    small = (2, 3, 5, 7, 11, 13, 17, 19, 23, 29)
    for p in small:
        if n == p:
            return True
        if n % p == 0 and n != p:
            return False

    # write n-1 = d*2^s
    d = n - 1
    s = 0
    while d % 2 == 0:
        d //= 2
        s += 1

    # Bases: enough for deterministic up to 2^64; otherwise use a few random bases.
    # Ref: https://miller-rabin.appspot.com/
    if n < (1 << 64):
        bases = (2, 3, 5, 7, 11, 13, 17)
    else:
        import random
        # 8 random bases is already extremely strong for 200-bit numbers
        rng = random.Random(0xC0FFEE)  # fixed seed for reproducibility
        bases = tuple(rng.randrange(2, n - 2) for _ in range(8))

    def check(a: int) -> bool:
        x = pow(a, d, n)
        if x == 1 or x == n - 1:
            return True
        for _ in range(s - 1):
            x = (x * x) % n
            if x == n - 1:
                return True
        return False

    return all(check(a % n) for a in bases)


class KCM:
    def __init__(self, k: int):
        assert k >= 0
        self.k: int = k
        self.Q: Set[str] = set()
        self.Sigma: Set[str] = set()
        self.q0: Optional[str] = None
        self.F: List[Tuple[str, Tuple[bool, ...]]] = []  # (q_final, zero-mask)
        self.transitions: Dict[Tuple[str, Symbol], List[Rule]] = {}

    # ----- build API -----

    def add_state(self, q: str, *, initial: bool = False) -> None:
        self.Q.add(q)
        if initial:
            self.q0 = q

    def add_final(self, q: str, mask: Optional[Tuple[bool, ...]] = None) -> None:
        if mask is None:
            mask = tuple(False for _ in range(self.k))
        assert len(mask) == self.k
        self.F.append((q, mask))

    def add_symbol(self, a: str) -> None:
        self.Sigma.add(a)

    def add_rule(
        self,
        q_from: str,
        sym: Symbol,
        q_to: str,
        update: Optional[Callable[[Tuple[int, ...]], Tuple[int, ...]]] = None,
    ) -> None:
        if update is None:
            update = lambda n: n
        self.Q.update([q_from, q_to])
        if sym is not None:
            self.Sigma.add(sym)
        self.transitions.setdefault((q_from, sym), []).append(Rule(q_to, sym, update))

    # ----- acceptance (canonical) -----

    @staticmethod
    def _z(n: Tuple[int, ...]) -> Tuple[bool, ...]:
        return tuple(c == 0 for c in n)

    @staticmethod
    def _mask_ok(z: Tuple[bool, ...], m: Tuple[bool, ...]) -> bool:
        return all((not m_i) or z_i for z_i, m_i in zip(z, m))

    def _is_accepting(self, q: str, n: Tuple[int, ...]) -> bool:
        z = self._z(n)
        return any(q == qf and self._mask_ok(z, m) for (qf, m) in self.F)

    # ----- deterministic runner (fixed ε policy) -----

    def run_dtm_trace(
        self, w: Iterable[str]
    ) -> Tuple[List[Tuple[int, str, Tuple[int, ...], Symbol]], bool]:
        """
        Deterministic stepping:
        - NO ε-closure before/during reading.
        - Apply ε-closure ONLY AFTER the entire input is consumed.
        - Bail out if branching (>=2 rules) is encountered.
        """
        assert self.q0 is not None, "Start state not set"
        w_list = list(w)
        N = len(w_list)

        q = self.q0
        n = tuple(0 for _ in range(self.k))
        i = 0
        trace: List[Tuple[int, str, Tuple[int, ...], Symbol]] = [(0, q, n, None)]

        # READ PHASE
        while i < N:
            a = w_list[i]
            rules = self.transitions.get((q, a), [])
            if len(rules) != 1:
                return trace, False
            r = rules[0]
            n = r.update(n)
            q = r.q_to
            i += 1
            trace.append((i, q, n, a))

        # FINAL ε-CLOSURE (only now)
        while True:
            eps = self.transitions.get((q, None), [])
            if len(eps) == 1:
                r = eps[0]
                n = r.update(n)
                q = r.q_to
                trace.append((i, q, n, None))
            elif len(eps) == 0:
                break
            else:
                return trace, False  # ε-branching → not deterministic

        accepted = (i == N) and self._is_accepting(q, n)
        return trace, accepted

    # ----- BFS runner with ε -----

    def computation_trace(
        self, w: Iterable[str], max_confs: int = 200_000
    ) -> Tuple[List[Tuple[int, str, Tuple[int, ...], Symbol]], bool]:
        assert self.q0 is not None, "Start state not set"
        w_list = list(w)
        N = len(w_list)

        start = (0, self.q0, tuple(0 for _ in range(self.k)))
        qd: deque[Tuple[int, str, Tuple[int, ...]]] = deque([start])
        parents: Dict[Tuple[int, str, Tuple[int, ...]],
                      Tuple[Tuple[int, str, Tuple[int, ...]], Symbol]] = {}
        seen: Set[Tuple[int, str, Tuple[int, ...]]] = {start}
        pushes = 1
        best = start

        def backtrace(end_cfg: Tuple[int, str, Tuple[int, ...]], end_sym: Symbol
                      ) -> List[Tuple[int, str, Tuple[int, ...], Symbol]]:
            path: List[Tuple[int, str, Tuple[int, ...], Symbol]] = []
            cur = end_cfg
            sym = end_sym
            path.append((*cur, sym))
            while cur in parents:
                cur, sym = parents[cur]
                path.append((*cur, sym))
            path.reverse()
            if path and path[0][3] is not None:
                i0, q0, n0, _ = path[0]
                path[0] = (i0, q0, n0, None)
            return path

        while qd:
            i, q, n = qd.popleft()
            if i > best[0]:
                best = (i, q, n)

            if i == N and self._is_accepting(q, n):
                return backtrace((i, q, n), None), True

            # ε moves
            for r in self.transitions.get((q, None), []):
                child = (i, r.q_to, r.update(n))
                if child not in seen:
                    seen.add(child)
                    parents[child] = ((i, q, n), None)
                    qd.append(child)
                    pushes += 1
                    if pushes > max_confs:
                        return backtrace(child, None), False

            # symbol moves
            if i < N:
                a = w_list[i]
                for r in self.transitions.get((q, a), []):
                    child = (i + 1, r.q_to, r.update(n))
                    if child not in seen:
                        seen.add(child)
                        parents[child] = ((i, q, n), a)
                        qd.append(child)
                        pushes += 1
                        if pushes > max_confs:
                            return backtrace(child, a), False

        return backtrace(best, None), False

    # ----- unified runner: choose a canonical trace + label -----

    def _run_with_trace(
        self,
        w: Iterable[str],
        max_confs: int = 200_000,
    ) -> Tuple[List[Tuple[int, str, Tuple[int, ...], Symbol]], bool]:
        """
        Return a single canonical run (as a trace) and its acceptance label.

        Semantics:
          1. Try the deterministic runner run_dtm_trace(w).
             - If it ACCEPTS, return that trace (no BFS needed).
          2. Otherwise, run computation_trace(w).
             - If BFS ACCEPTS, return the BFS trace.
             - If both reject, return the one that consumed more input symbols.
        """
        w_list = list(w)

        # 1) Deterministic fast path
        try:
            dtm_trace, dtm_acc = self.run_dtm_trace(w_list)
        except Exception:
            dtm_trace, dtm_acc = [], False

        if dtm_acc:
            # Deterministic run already found an accepting run → done.
            return dtm_trace, True

        # 2) Full BFS over the original input (handles nondeterminism)
        bfs_trace, bfs_acc = self.computation_trace(w_list, max_confs=max_confs)

        if bfs_acc:
            # There exists an accepting run → return that one.
            return bfs_trace, True

        # 3) Both reject: choose the more informative (longer) trace
        def consumed_len(tr: List[Tuple[int, str, Tuple[int, ...], Symbol]]) -> int:
            # number of actually consumed input symbols (non-ε steps)
            return sum(1 for *_, s in tr if s is not None)

        if consumed_len(bfs_trace) >= consumed_len(dtm_trace):
            return bfs_trace, False
        else:
            return dtm_trace, False

    # ----- API -----

    def accepts(self, w: Iterable[str]) -> bool:
        """
        Language semantics: there exists an accepting run on w?
        Uses the same semantics as _run_with_trace, so it is
        guaranteed to agree with output_generator.
        """
        _, acc = self._run_with_trace(w)
        return acc

    def output_generator(self, seq: Iterable[str], alphabet: List[str]) -> str:
        """
        Produce a textual encoding of ONE canonical run plus the accept bit.

        Format:
           {#states}_{#counters}_{alphabet_as_string}_{q_s1-...-q_sn}?{0/1}

        where each step is 'q_sym', with sym = actual symbol or '*' for ε.
        """
        trace, acc = self._run_with_trace(seq)

        step_tuples = [
            f"{q}_{('*' if s is None else s)}"
            for (_, q, _, s) in trace
        ]
        trace_str = "-".join(step_tuples)
        alphabet_str = "".join(alphabet)
        states = len(self.Q)
        acc_str = "1" if acc else "0"
        return f"{states}_{self.k}_{alphabet_str}_{trace_str}?{acc_str}"


# =====================================================================
#  Concrete KCMs
# =====================================================================

def build_prime_binary_kcm() -> KCM:
    """
    L = { bin(p) | p is prime } over alphabet {'0','1'} (no separators).
    Canonical encoding: must start with '1' (no leading zeros).
    Accepts e.g. '10'(2), '11'(3), '101'(5), '111'(7), '1011'(11).
    Rejects e.g. '', '0', '01', '1'(=1), '100'(=4), etc.

    Counters:
      c0 = numeric value parsed from binary
      c1 = flag_zero (0 if prime, 1 otherwise)
    """
    M = KCM(k=2)

    # States
    M.add_state("qS", initial=True)   # must see first '1'
    M.add_state("qR")                 # read remaining bits (0/1)
    M.add_state("qC")                 # ε: compute primality
    M.add_state("qA")                 # masked accept (flag must be 0)
    M.add_state("qJ")                 # sink (e.g., leading zero)

    # Alphabet & final
    M.add_symbol('0')
    M.add_symbol('1')
    M.add_final("qA", mask=(False, True))  # require flag_zero == 0

    # Append helpers
    def append0(n): v, f = n; return (v * 2 + 0, f)
    def append1(n): v, f = n; return (v * 2 + 1, f)

    # First symbol must be '1'
    M.add_rule("qS", '1', "qR", update=append1)
    M.add_rule("qS", '0', "qJ")           # leading zero -> reject
    M.add_rule("qJ", '0', "qJ")
    M.add_rule("qJ", '1', "qJ")

    # Subsequent symbols: either 0 or 1
    M.add_rule("qR", '0', "qR", update=append0)
    M.add_rule("qR", '1', "qR", update=append1)

    # ε: compute primality on c0
    def compute_prime_flag(n):
        v, _ = n
        ok = (v >= 2) and _is_probable_prime(v)
        return (v, 0 if ok else 1)

    M.add_rule("qR", None, "qC", update=compute_prime_flag)
    M.add_rule("qC", None, "qA")
    return M


def build_gcd_binary_kcm() -> KCM:
    """
    Accept L = { bin(i)/bin(j)/bin(k) | k = gcd(i, j) } over alphabet {'0','1','/'}.
    Canonical binaries:
      - 0  -> "0" (exactly one zero)
      - >0 -> starts with '1' (no leading zeros)
    """
    M = KCM(k=4)

    # States
    M.add_state("qI0", initial=True)   # first bit of i
    M.add_state("qI")                  # more bits of i (after leading '1')
    M.add_state("qIE")                 # i == "0" (must see '/')

    M.add_state("qJ0")                 # first bit of j
    M.add_state("qJ")                  # more bits of j (after leading '1')
    M.add_state("qJE")                 # j == "0" (must see '/')

    M.add_state("qK0")                 # first bit of k
    M.add_state("qK")                  # more bits of k (after leading '1')
    M.add_state("qKE")                 # k == "0" (no more bits)

    M.add_state("qC")                  # ε: check gcd
    M.add_state("qA")                  # masked accept
    M.add_state("qJ")                  # reject sink

    # Alphabet & final
    for a in ('0', '1', '/'):
        M.add_symbol(a)
    # c3 is flag_zero → must be zero at accept
    M.add_final("qA", mask=(False, False, False, True))

    # Helpers to append bits into i, j, k
    def app0_i(n): i, j, k, f = n; return (i * 2 + 0, j, k, f)
    def app1_i(n): i, j, k, f = n; return (i * 2 + 1, j, k, f)
    def app0_j(n): i, j, k, f = n; return (i, j * 2 + 0, k, f)
    def app1_j(n): i, j, k, f = n; return (i, j * 2 + 1, k, f)
    def app0_k(n): i, j, k, f = n; return (i, j, k * 2 + 0, f)
    def app1_k(n): i, j, k, f = n; return (i, j, k * 2 + 1, f)

    # ----- Parse i -----
    M.add_rule("qI0", '0', "qIE", update=app0_i)   # i == 0
    M.add_rule("qI0", '1', "qI", update=app1_i)    # i > 0
    M.add_rule("qI0", '/', "qJ")                   # empty i not allowed

    M.add_rule("qI", '0', "qI", update=app0_i)
    M.add_rule("qI", '1', "qI", update=app1_i)
    M.add_rule("qI", '/', "qJ0")

    M.add_rule("qIE", '0', "qJ")
    M.add_rule("qIE", '1', "qJ")
    M.add_rule("qIE", '/', "qJ0")

    # ----- Parse j -----
    M.add_rule("qJ0", '0', "qJE", update=app0_j)   # j == 0
    M.add_rule("qJ0", '1', "qJ", update=app1_j)    # j > 0
    M.add_rule("qJ0", '/', "qJ")                   # empty j not allowed

    M.add_rule("qJ", '0', "qJ", update=app0_j)
    M.add_rule("qJ", '1', "qJ", update=app1_j)
    M.add_rule("qJ", '/', "qK0")

    M.add_rule("qJE", '0', "qJ")
    M.add_rule("qJE", '1', "qJ")
    M.add_rule("qJE", '/', "qK0")

    # ----- Parse k -----
    M.add_rule("qK0", '0', "qKE", update=app0_k)   # k == 0
    M.add_rule("qK0", '1', "qK", update=app1_k)    # k > 0
    M.add_rule("qK0", '/', "qJ")                   # empty k not allowed

    M.add_rule("qK", '0', "qK", update=app0_k)
    M.add_rule("qK", '1', "qK", update=app1_k)
    M.add_rule("qK", '/', "qJ")                    # no extra separator after k

    # k == "0": forbid any more symbols
    M.add_rule("qKE", '0', "qJ")
    M.add_rule("qKE", '1', "qJ")
    M.add_rule("qKE", '/', "qJ")

    # ----- End-of-input check -----
    import math

    def compute_gcd_flag(n):
        i, j, k, _ = n
        return (i, j, k, 0 if k == math.gcd(i, j) else 1)

    # valid end states for k
    M.add_rule("qK", None, "qC", update=compute_gcd_flag)
    M.add_rule("qKE", None, "qC", update=compute_gcd_flag)
    M.add_rule("qC", None, "qA")

    # Reject sink self-loops
    for a in ('0', '1', '/'):
        M.add_rule("qJ", a, "qJ")

    return M


def build_exp_binary_kcm() -> KCM:
    """
    Accept L = { bin(i)/bin(j)/bin(k) | k = i ** j } over alphabet {'0','1','/'}.
    Canonical binaries:
      - 0  -> "0" (exactly one zero)
      - >0 -> starts with '1' (no leading zeros)

    Convention: 0^0 = 1 (accepts "0/0/1"). To change, edit compute_exp_flag().
    Counters:
      c0 = i, c1 = j, c2 = k, c3 = flag_zero (0 if accept, 1 otherwise)
    """
    M = KCM(k=4)

    # States
    M.add_state("qI0", initial=True)   # first bit of i
    M.add_state("qI")                  # more bits of i
    M.add_state("qIE")                 # i == "0"

    M.add_state("qJ0")                 # first bit of j
    M.add_state("qJ")                  # more bits of j
    M.add_state("qJE")                 # j == "0"

    M.add_state("qK0")                 # first bit of k
    M.add_state("qK")                  # more bits of k
    M.add_state("qKE")                 # k == "0"

    M.add_state("qC")                  # ε: check exponentiation
    M.add_state("qA")                  # masked accept
    M.add_state("qJ")                  # sink

    # Alphabet & final
    for a in ('0', '1', '/'):
        M.add_symbol(a)
    M.add_final("qA", mask=(False, False, False, True))  # only flag must be zero

    # Append helpers
    def app0_i(n): i, j, k, f = n; return (i * 2 + 0, j, k, f)
    def app1_i(n): i, j, k, f = n; return (i * 2 + 1, j, k, f)
    def app0_j(n): i, j, k, f = n; return (i, j * 2 + 0, k, f)
    def app1_j(n): i, j, k, f = n; return (i, j * 2 + 1, k, f)
    def app0_k(n): i, j, k, f = n; return (i, j, k * 2 + 0, f)
    def app1_k(n): i, j, k, f = n; return (i, j, k * 2 + 1, f)

    # ----- Parse i -----
    M.add_rule("qI0", '0', "qIE", update=app0_i)   # i == 0
    M.add_rule("qI0", '1', "qI", update=app1_i)    # i > 0
    M.add_rule("qI0", '/', "qJ")                   # empty i not allowed

    M.add_rule("qI", '0', "qI", update=app0_i)
    M.add_rule("qI", '1', "qI", update=app1_i)
    M.add_rule("qI", '/', "qJ0")

    M.add_rule("qIE", '0', "qJ")
    M.add_rule("qIE", '1', "qJ")
    M.add_rule("qIE", '/', "qJ0")

    # ----- Parse j -----
    M.add_rule("qJ0", '0', "qJE", update=app0_j)   # j == 0
    M.add_rule("qJ0", '1', "qJ", update=app1_j)    # j > 0
    M.add_rule("qJ0", '/', "qJ")                   # empty j not allowed

    M.add_rule("qJ", '0', "qJ", update=app0_j)
    M.add_rule("qJ", '1', "qJ", update=app1_j)
    M.add_rule("qJ", '/', "qK0")

    M.add_rule("qJE", '0', "qJ")
    M.add_rule("qJE", '1', "qJ")
    M.add_rule("qJE", '/', "qK0")

    # ----- Parse k -----
    M.add_rule("qK0", '0', "qKE", update=app0_k)   # k == 0
    M.add_rule("qK0", '1', "qK", update=app1_k)    # k > 0
    M.add_rule("qK0", '/', "qJ")                   # empty k not allowed

    M.add_rule("qK", '0', "qK", update=app0_k)
    M.add_rule("qK", '1', "qK", update=app1_k)
    M.add_rule("qK", '/', "qJ")                    # no extra separator after k

    # k == "0": forbid any more symbols
    M.add_rule("qKE", '0', "qJ")
    M.add_rule("qKE", '1', "qJ")
    M.add_rule("qKE", '/', "qJ")

    # ----- End-of-input check -----
    def compute_exp_flag(n):
        i, j, k, _ = n
        ok = (k == pow(i, j))
        return (i, j, k, 0 if ok else 1)

    M.add_rule("qK", None, "qC", update=compute_exp_flag)
    M.add_rule("qKE", None, "qC", update=compute_exp_flag)
    M.add_rule("qC", None, "qA")

    # Reject sink
    for a in ('0', '1', '/'):
        M.add_rule("qJ", a, "qJ")

    return M


def build_dvd_binary_kcm() -> KCM:
    """
    L = { bin(w)/bin(v) | w >= 1 and w divides v } over alphabet {'0','1','/'}.
    Canonical binaries (no leading zeros):
      - w: must start with '1' (so w >= 1)
      - v: either exactly '0' OR starts with '1'
    Counters:
      c0 = w (parsed value)
      c1 = v (parsed value)
      c2 = flag_zero (0 if accept, 1 otherwise)
    """
    M = KCM(k=3)

    # States
    M.add_state("qW0", initial=True)   # first bit of w (must be '1')
    M.add_state("qW")                  # more bits of w
    M.add_state("qV0")                 # first bit of v after '/'
    M.add_state("qV")                  # more bits of v (after leading '1')
    M.add_state("qVzeroEnd")           # v == "0" (no more bits allowed)
    M.add_state("qC")                  # ε: compute divisibility
    M.add_state("qA")                  # masked accept
    M.add_state("qJ")                  # reject sink

    # Alphabet & final
    for a in ('0', '1', '/'):
        M.add_symbol(a)
    M.add_final("qA", mask=(False, False, True))  # require c2 == 0

    # Helpers to append bits
    def app0_w(n): w, v, f = n; return (w * 2 + 0, v, f)
    def app1_w(n): w, v, f = n; return (w * 2 + 1, v, f)
    def app0_v(n): w, v, f = n; return (w, v * 2 + 0, f)
    def app1_v(n): w, v, f = n; return (w, v * 2 + 1, f)

    # w: first bit must be '1' (w >= 1)
    M.add_rule("qW0", '1', "qW", update=app1_w)
    M.add_rule("qW0", '0', "qJ")
    M.add_rule("qW0", '/', "qJ")  # need at least one bit for w

    # w: more bits or delimiter '/'
    M.add_rule("qW", '0', "qW", update=app0_w)
    M.add_rule("qW", '1', "qW", update=app1_w)
    M.add_rule("qW", '/', "qV0")

    # v: first bit — either exactly "0" OR starts with '1'
    M.add_rule("qV0", '0', "qVzeroEnd", update=app0_v)  # v == 0; next must be end-of-input
    M.add_rule("qV0", '1', "qV", update=app1_v)
    M.add_rule("qV0", '/', "qJ")                      # empty v not allowed

    # v == "0": forbid any more symbols
    M.add_rule("qVzeroEnd", '0', "qJ")
    M.add_rule("qVzeroEnd", '1', "qJ")
    M.add_rule("qVzeroEnd", '/', "qJ")

    # v after a leading '1': read arbitrary bits; no extra separator allowed
    M.add_rule("qV", '0', "qV", update=app0_v)
    M.add_rule("qV", '1', "qV", update=app1_v)
    M.add_rule("qV", '/', "qJ")

    # ε: divisibility check at end
    def compute_divides_flag(n):
        w, v, _ = n
        ok = (w >= 1) and (v % w == 0)
        return (w, v, 0 if ok else 1)

    # Valid end-of-input states for v
    M.add_rule("qV", None, "qC", update=compute_divides_flag)
    M.add_rule("qVzeroEnd", None, "qC", update=compute_divides_flag)
    M.add_rule("qC", None, "qA")

    # Reject sink
    for a in ('0', '1', '/'):
        M.add_rule("qJ", a, "qJ")

    return M


def build_mul_binary_kcm() -> KCM:
    """
    Accept L = { bin(i)/bin(j)/bin(k) | k = i * j } over alphabet {'0','1','/'}.
    Canonical binaries:
      - 0  -> "0"
      - >0 -> starts with '1' (no leading zeros)
    """
    M = KCM(k=4)

    # States
    M.add_state("qI0", initial=True)   # first bit of i
    M.add_state("qI")                  # more bits of i (after leading '1')
    M.add_state("qIE")                 # i == "0" (must see '/')

    M.add_state("qJ0")                 # first bit of j
    M.add_state("qJ")                  # more bits of j (after leading '1')
    M.add_state("qJE")                 # j == "0" (must see '/')

    M.add_state("qK0")                 # first bit of k
    M.add_state("qK")                  # more bits of k (after leading '1')
    M.add_state("qKE")                 # k == "0" (no more bits allowed)

    M.add_state("qC")                  # ε: compute flag
    M.add_state("qA")                  # masked accept
    M.add_state("qJ")                  # reject sink

    # Alphabet & final
    for a in ('0', '1', '/'):
        M.add_symbol(a)
    # c3 is flag_zero → must be zero at accept
    M.add_final("qA", mask=(False, False, False, True))

    # Helpers to append bits into i, j, k
    def app0_i(n): i, j, k, f = n; return (i * 2 + 0, j, k, f)
    def app1_i(n): i, j, k, f = n; return (i * 2 + 1, j, k, f)
    def app0_j(n): i, j, k, f = n; return (i, j * 2 + 0, k, f)
    def app1_j(n): i, j, k, f = n; return (i, j * 2 + 1, k, f)
    def app0_k(n): i, j, k, f = n; return (i, j, k * 2 + 0, f)
    def app1_k(n): i, j, k, f = n; return (i, j, k * 2 + 1, f)

    # ----- Parse i -----
    M.add_rule("qI0", '0', "qIE", update=app0_i)   # i == 0
    M.add_rule("qI0", '1', "qI", update=app1_i)    # i > 0
    M.add_rule("qI0", '/', "qJ")                   # empty i not allowed

    M.add_rule("qI", '0', "qI", update=app0_i)
    M.add_rule("qI", '1', "qI", update=app1_i)
    M.add_rule("qI", '/', "qJ0")

    M.add_rule("qIE", '0', "qJ")                   # no more bits after "0"
    M.add_rule("qIE", '1', "qJ")
    M.add_rule("qIE", '/', "qJ0")

    # ----- Parse j -----
    M.add_rule("qJ0", '0', "qJE", update=app0_j)   # j == 0
    M.add_rule("qJ0", '1', "qJ", update=app1_j)    # j > 0
    M.add_rule("qJ0", '/', "qJ")                   # empty j not allowed

    M.add_rule("qJ", '0', "qJ", update=app0_j)
    M.add_rule("qJ", '1', "qJ", update=app1_j)
    M.add_rule("qJ", '/', "qK0")

    M.add_rule("qJE", '0', "qJ")                   # no more bits after "0"
    M.add_rule("qJE", '1', "qJ")
    M.add_rule("qJE", '/', "qK0")

    # ----- Parse k -----
    M.add_rule("qK0", '0', "qKE", update=app0_k)   # k == 0
    M.add_rule("qK0", '1', "qK", update=app1_k)    # k > 0
    M.add_rule("qK0", '/', "qJ")                   # empty k not allowed

    M.add_rule("qK", '0', "qK", update=app0_k)
    M.add_rule("qK", '1', "qK", update=app1_k)
    M.add_rule("qK", '/', "qJ")                    # no separator allowed after k

    # k == "0": forbid any more symbols
    M.add_rule("qKE", '0', "qJ")
    M.add_rule("qKE", '1', "qJ")
    M.add_rule("qKE", '/', "qJ")

    # ----- End-of-input check -----
    def compute_mul_flag(n):
        i, j, k, _ = n
        return (i, j, k, 0 if k == i * j else 1)

    # valid end states for k
    M.add_rule("qK", None, "qC", update=compute_mul_flag)
    M.add_rule("qKE", None, "qC", update=compute_mul_flag)
    M.add_rule("qC", None, "qA")

    # Reject sink self-loops
    for a in ('0', '1', '/'):
        M.add_rule("qJ", a, "qJ")

    return M


def build_add_binary_kcm() -> KCM:
    """
    Accept L = { bin(i)/bin(j)/bin(k) | k = i + j } over alphabet {'0','1','/'}.
    Canonical binaries:
      - 0  -> "0" (exactly one zero)
      - >0 -> starts with '1' (no leading zeros)
    """
    M = KCM(k=4)

    # States
    M.add_state("qI0", initial=True)   # first bit of i
    M.add_state("qI")                  # more bits of i
    M.add_state("qIE")                 # i == 0

    M.add_state("qJ0")                 # first bit of j
    M.add_state("qJ")                  # more bits of j
    M.add_state("qJE")                 # j == 0

    M.add_state("qK0")                 # first bit of k
    M.add_state("qK")                  # more bits of k
    M.add_state("qKE")                 # k == 0

    M.add_state("qC")                  # ε: check addition
    M.add_state("qA")                  # masked accept
    M.add_state("qJ")                  # sink

    # Alphabet & final
    for a in ('0', '1', '/'):
        M.add_symbol(a)
    M.add_final("qA", mask=(False, False, False, True))  # c3=flag must be 0

    # Append helpers
    def app0_i(n): i, j, k, f = n; return (i * 2 + 0, j, k, f)
    def app1_i(n): i, j, k, f = n; return (i * 2 + 1, j, k, f)
    def app0_j(n): i, j, k, f = n; return (i, j * 2 + 0, k, f)
    def app1_j(n): i, j, k, f = n; return (i, j * 2 + 1, k, f)
    def app0_k(n): i, j, k, f = n; return (i, j, k * 2 + 0, f)
    def app1_k(n): i, j, k, f = n; return (i, j, k * 2 + 1, f)

    # ----- Parse i -----
    M.add_rule("qI0", '0', "qIE", update=app0_i)   # i == 0
    M.add_rule("qI0", '1', "qI", update=app1_i)    # i > 0
    M.add_rule("qI0", '/', "qJ")                   # empty i not allowed

    M.add_rule("qI", '0', "qI", update=app0_i)
    M.add_rule("qI", '1', "qI", update=app1_i)
    M.add_rule("qI", '/', "qJ0")

    M.add_rule("qIE", '0', "qJ")
    M.add_rule("qIE", '1', "qJ")
    M.add_rule("qIE", '/', "qJ0")

    # ----- Parse j -----
    M.add_rule("qJ0", '0', "qJE", update=app0_j)   # j == 0
    M.add_rule("qJ0", '1', "qJ", update=app1_j)    # j > 0
    M.add_rule("qJ0", '/', "qJ")                   # empty j not allowed

    M.add_rule("qJ", '0', "qJ", update=app0_j)
    M.add_rule("qJ", '1', "qJ", update=app1_j)
    M.add_rule("qJ", '/', "qK0")

    M.add_rule("qJE", '0', "qJ")
    M.add_rule("qJE", '1', "qJ")
    M.add_rule("qJE", '/', "qK0")

    # ----- Parse k -----
    M.add_rule("qK0", '0', "qKE", update=app0_k)   # k == 0
    M.add_rule("qK0", '1', "qK", update=app1_k)    # k > 0
    M.add_rule("qK0", '/', "qJ")                   # empty k not allowed

    M.add_rule("qK", '0', "qK", update=app0_k)
    M.add_rule("qK", '1', "qK", update=app1_k)

    # k == "0": forbid any more symbols
    M.add_rule("qKE", '0', "qJ")
    M.add_rule("qKE", '1', "qJ")
    M.add_rule("qKE", '/', "qJ")

    # ----- End-of-input check -----
    def compute_add_flag(n):
        i, j, k, _ = n
        return (i, j, k, 0 if k == i + j else 1)

    M.add_rule("qK", None, "qC", update=compute_add_flag)
    M.add_rule("qKE", None, "qC", update=compute_add_flag)
    M.add_rule("qC", None, "qA")

    # Reject sink
    for a in ('0', '1', '/'):
        M.add_rule("qJ", a, "qJ")

    return M


# Simple local sanity check if you run this file directly.
if __name__ == "__main__":
    A = build_prime_binary_kcm()
    tests = [f"1" * i for i in range(1, 20)]
    for i, s in enumerate(tests):
        acc = A.accepts(s)
        og = A.output_generator(s, ['0', '1'])
        print(f"{i+1:2} {s:12} -> accepts={acc} trace_ok={og.endswith('?1') == acc}")
