from __future__ import annotations
from collections import deque
from dataclasses import dataclass
from typing import Callable, Dict, Iterable, List, Optional, Tuple, Set

Symbol = Optional[str]  # None = epsilon

@dataclass(frozen=True)
class Rule:
    q_to: str
    sym: Symbol
    update: Callable[[Tuple[int, ...]], Tuple[int, ...]]

class KCM:
    def __init__(self, k: int):
        assert k >= 0
        self.k: int = k
        self.Q: Set[str] = set()
        self.Sigma: Set[str] = set()
        self.q0: Optional[str] = None
        self.F: List[Tuple[str, Tuple[bool, ...]]] = []  # (q_final, zero-mask)
        self.transitions: Dict[Tuple[str, Symbol], List[Rule]] = {}

    # ----- build API -----

    def add_state(self, q: str, *, initial: bool = False) -> None:
        self.Q.add(q)
        if initial:
            self.q0 = q

    def add_final(self, q: str, mask: Optional[Tuple[bool, ...]] = None) -> None:
        if mask is None:
            mask = tuple(False for _ in range(self.k))
        assert len(mask) == self.k
        self.F.append((q, mask))

    def add_symbol(self, a: str) -> None:
        self.Sigma.add(a)

    def add_rule(
        self,
        q_from: str,
        sym: Symbol,
        q_to: str,
        update: Optional[Callable[[Tuple[int, ...]], Tuple[int, ...]]] = None,
    ) -> None:
        if update is None:
            update = lambda n: n
        self.Q.update([q_from, q_to])
        if sym is not None:
            self.Sigma.add(sym)
        self.transitions.setdefault((q_from, sym), []).append(Rule(q_to, sym, update))

    # ----- acceptance (canonical) -----

    @staticmethod
    def _z(n: Tuple[int, ...]) -> Tuple[bool, ...]:
        return tuple(c == 0 for c in n)

    @staticmethod
    def _mask_ok(z: Tuple[bool, ...], m: Tuple[bool, ...]) -> bool:
        return all((not m_i) or z_i for z_i, m_i in zip(z, m))

    def _is_accepting(self, q: str, n: Tuple[int, ...]) -> bool:
        z = self._z(n)
        return any(q == qf and self._mask_ok(z, m) for (qf, m) in self.F)

    # ----- deterministic runner (fixed ε policy) -----

    def run_dtm_trace(
        self, w: Iterable[str]
    ) -> Tuple[List[Tuple[int, str, Tuple[int, ...], Symbol]], bool]:
        """
        Deterministic stepping:
        - NO ε-closure before/during reading.
        - Apply ε-closure ONLY AFTER the entire input is consumed.
        - Bail out if branching (>=2 rules) is encountered.
        """
        assert self.q0 is not None, "Start state not set"
        w_list = list(w)
        N = len(w_list)

        q = self.q0
        n = tuple(0 for _ in range(self.k))
        i = 0
        trace: List[Tuple[int, str, Tuple[int, ...], Symbol]] = [(0, q, n, None)]

        # READ PHASE
        while i < N:
            a = w_list[i]
            rules = self.transitions.get((q, a), [])
            if len(rules) != 1:
                return trace, False
            r = rules[0]
            n = r.update(n)
            q = r.q_to
            i += 1
            trace.append((i, q, n, a))

        # FINAL ε-CLOSURE (only now)
        while True:
            eps = self.transitions.get((q, None), [])
            if len(eps) == 1:
                r = eps[0]
                n = r.update(n)
                q = r.q_to
                trace.append((i, q, n, None))
            elif len(eps) == 0:
                break
            else:
                return trace, False  # ε-branching → not deterministic

        accepted = (i == N) and self._is_accepting(q, n)
        return trace, accepted


    # ----- BFS runner with ε -----

    def computation_trace(
        self, w: Iterable[str], max_confs: int = 200_000
    ) -> Tuple[List[Tuple[int, str, Tuple[int, ...], Symbol]], bool]:
        assert self.q0 is not None, "Start state not set"
        w_list = list(w)
        N = len(w_list)

        start = (0, self.q0, tuple(0 for _ in range(self.k)))
        qd: deque[Tuple[int, str, Tuple[int, ...]]] = deque([start])
        parents: Dict[Tuple[int, str, Tuple[int, ...]],
                      Tuple[Tuple[int, str, Tuple[int, ...]], Symbol]] = {}
        seen: Set[Tuple[int, str, Tuple[int, ...]]] = {start}
        pushes = 1
        best = start

        def backtrace(end_cfg: Tuple[int, str, Tuple[int, ...]], end_sym: Symbol
                      ) -> List[Tuple[int, str, Tuple[int, ...], Symbol]]:
            path: List[Tuple[int, str, Tuple[int, ...], Symbol]] = []
            cur = end_cfg; sym = end_sym
            path.append((*cur, sym))
            while cur in parents:
                cur, sym = parents[cur]
                path.append((*cur, sym))
            path.reverse()
            if path and path[0][3] is not None:
                i0, q0, n0, _ = path[0]
                path[0] = (i0, q0, n0, None)
            return path

        while qd:
            i, q, n = qd.popleft()
            if i > best[0]:
                best = (i, q, n)

            if i == N and self._is_accepting(q, n):
                return backtrace((i, q, n), None), True

            # ε moves
            for r in self.transitions.get((q, None), []):
                child = (i, r.q_to, r.update(n))
                if child not in seen:
                    seen.add(child)
                    parents[child] = ((i, q, n), None)
                    qd.append(child)
                    pushes += 1
                    if pushes > max_confs:
                        return backtrace(child, None), False

            # symbol moves
            if i < N:
                a = w_list[i]
                for r in self.transitions.get((q, a), []):
                    child = (i + 1, r.q_to, r.update(n))
                    if child not in seen:
                        seen.add(child)
                        parents[child] = ((i, q, n), a)
                        qd.append(child)
                        pushes += 1
                        if pushes > max_confs:
                            return backtrace(child, a), False

        return backtrace(best, None), False

    # ----- API -----

    def accepts(self, w: Iterable[str]) -> bool:
        # Try the deterministic fast path.
        try:
            trace, acc = self.run_dtm_trace(w)
        except Exception:
            # If DTM stepping itself fails, just use BFS.
            _, acc = self.computation_trace(w)
            return acc

        if acc:
            return True

        # IMPORTANT: fall back to BFS on the ORIGINAL input, not on a
        # "consumed symbols" reconstruction from the DTM trace.
        _, acc_bfs = self.computation_trace(w)
        return acc_bfs


    def output_generator(self, seq: Iterable[str], alphabet: List[str]) -> str:
        w = list(seq)

        # Try to get an accepting path
        trace, acc = self.computation_trace(w, max_confs=1_000_000)

        if not acc:
            dtm_trace, dtm_acc = self.run_dtm_trace(w)
            # Prefer the longer trace for readability
            def consumed_len(tr): return sum(1 for *_, s in tr if s is not None)
            if consumed_len(dtm_trace) >= consumed_len(trace):
                trace, acc = dtm_trace, dtm_acc

        step_tuples = [f'{q}_{("*" if s is None else s)}' for _, q, _, s in trace]
        trace_str = '-'.join(step_tuples)
        alphabet_str = ''.join(alphabet)
        states = len(self.Q)
        acc_str = '1' if acc else '0'
        return f"{states}_{self.k}_{alphabet_str}_{trace_str}?{acc_str}"

# ---------- PRIME BUILDER (deterministic) ----------

def build_prime_unary_kcm() -> KCM:
    """
    Accept L = { a^n | n is prime } using two counters:
      c0 = length n
      c1 = flag_zero := 0 if n is prime else 1
    Final mask requires c1 == 0 (True in mask means 'must be zero').
    """
    M = KCM(k=2)
    M.add_state("q0", initial=True)   # read phase
    M.add_state("qC")               # compute primality
    M.add_state("qF")

    M.add_symbol('a')
    M.add_final("qF", mask=(False, True))  # require c1 == 0

    def inc_len(n):
        c0, c1 = n
        return (c0 + 1, c1)

    # Read input: count length in c0
    M.add_rule("q0", 'a', "q0", update=inc_len)

    # After input is consumed, ε to compute primality
    def compute_prime_flag(n):
        c0, _ = n
        N = c0
        # compute flag_zero: 0 if prime else 1
        if N < 2:
            return (c0, 1)
        if N in (2, 3):
            return (c0, 0)
        if N % 2 == 0:
            return (c0, 1)
        d = 3
        while d * d <= N:
            if N % d == 0:
                return (c0, 1)
            d += 2
        return (c0, 0)

    # ε-steps used ONLY after reading because run_dtm_trace applies ε-closure at the end
    M.add_rule("q0", None, "qC", update=compute_prime_flag)
    M.add_rule("qC", None, "qF")

    return M

def build_exp_kcm() -> KCM:
    """
    Accept L = { a^{2^i} | i >= 0 } over alphabet {'a'}.
    Counters:
      c0 = n   (number of a's)
      c1 = flag_zero := 0 if n is a power of two else 1
    Accept in qF with mask (False, True) → require c1 == 0.
    Deterministic: only 'a' is allowed.
    """
    M = KCM(k=2)
    M.add_state("qA", initial=True)   # reading a's
    M.add_state("qC")               # compute flag
    M.add_state("qF")

    M.add_symbol('a')
    M.add_final("qF", mask=(False, True))  # require c1 == 0

    def inc_a(n):
        cnt, f = n
        return (cnt + 1, f)

    # Read a's
    M.add_rule("qA", 'a', "qA", update=inc_a)

    # After all input, ε-compute flag_zero := 0 iff n is a power of two
    def compute_pow2_flag(n):
        cnt, _ = n
        # A positive integer is a power of two iff cnt & (cnt - 1) == 0.
        is_pow2 = (cnt > 0) and ((cnt & (cnt - 1)) == 0)
        flag_zero = 0 if is_pow2 else 1
        return (cnt, flag_zero)

    M.add_rule("qA", None, "qC", update=compute_pow2_flag)
    M.add_rule("qC", None, "qF")
    return M 

def build_dvd_kcm() -> KCM:
    """
    L = { a^m b^n | m >= 1 and m divides n } over alphabet {'a','b'}.
    Counters:
      c0 = m  (count of 'a')
      c1 = n  (count of 'b')
      c2 = flag_zero (0 if accept, 1 if reject)
    Accept in qF with mask (False, False, True) → require c2 == 0.

    Notes:
    - Strings with only a's (m>=1, n=0) are ACCEPTED since m | 0 is true.
    - Strings with no a's (m=0) are REJECTED (division by zero not allowed).
    - Deterministic: no 'a' allowed after we've seen a 'b'.
    """
    M = KCM(k=3)
    M.add_state("qA", initial=True)   # reading a's
    M.add_state("qB")                 # reading b's
    M.add_state("qC")               # compute divides flag
    M.add_state("qF")

    M.add_symbol('a'); M.add_symbol('b')
    M.add_final("qF", mask=(False, False, True))  # require c2 == 0

    def inc_a(n):
        m, j, f = n
        return (m + 1, j, f)

    def inc_b(n):
        m, j, f = n
        return (m, j + 1, f)

    # Read phase
    M.add_rule("qA", 'a', "qA", update=inc_a)
    M.add_rule("qA", 'b', "qB", update=inc_b)
    M.add_rule("qB", 'b', "qB", update=inc_b)
    # (No rule for 'a' from qB → deterministic reject if seen)

    # After input is consumed, compute flag_zero := 0 if (m>=1 and j % m == 0) else 1
    def compute_divides_flag(n):
        m, j, _ = n
        if m >= 1 and (j % m == 0):
            return (m, j, 0)  # accept
        else:
            return (m, j, 1)  # reject

    M.add_rule("qA", None, "qC", update=compute_divides_flag)
    M.add_rule("qB", None, "qC", update=compute_divides_flag)
    M.add_rule("qC", None, "qF")

    return M


def build_gcd_kcm() -> KCM:
    """
    Accept L = { a^i b^j c^k | k = gcd(i, j) } over alphabet {'a','b','c'}.

    Counters:
      c0 = i  (# of 'a')
      c1 = j  (# of 'b')
      c2 = k  (# of 'c')
      c3 = flag_zero (0 if accept, 1 if reject)

    Acceptance mask: require c3 == 0  -> mask = (False, False, False, True)
    Deterministic: read a's in qA, then b's in qB, then c's in qC.
                   Any out-of-order symbol ⇒ no rule ⇒ reject.
    """
    M = KCM(k=4)
    # States
    M.add_state("qA", initial=True)  # reading a*
    M.add_state("qB")                # reading b*
    M.add_state("qC")                # reading c*
    M.add_state("qC")              # ε: compute flag
    M.add_state("qF")              # final state (mask-enforced)

    # Alphabet
    M.add_symbol('a'); M.add_symbol('b'); M.add_symbol('c')

    # Final with mask: only c3 must be zero
    M.add_final("qF", mask=(False, False, False, True))

    # Updates
    def inc_a(n):
        i, j, k, f = n
        return (i + 1, j, k, f)

    def inc_b(n):
        i, j, k, f = n
        return (i, j + 1, k, f)

    def inc_c(n):
        i, j, k, f = n
        return (i, j, k + 1, f)

    # Read blocks deterministically: a* b* c*
    M.add_rule("qA", 'a', "qA", update=inc_a)
    M.add_rule("qA", 'b', "qB", update=inc_b)
    M.add_rule("qB", 'b', "qB", update=inc_b)
    M.add_rule("qB", 'c', "qC", update=inc_c)
    M.add_rule("qC", 'c', "qC", update=inc_c)
    # No 'a' after qB/qC; no 'b' after qC ⇒ deterministic structure

    # ε: compute flag_zero := 0 iff k == gcd(i, j), else 1
    def compute_gcd_flag(n):
        i, j, k, _ = n
        # Python’s math.gcd handles all nonnegative cases, gcd(0,0)=0
        import math
        flag_zero = 0 if k == math.gcd(i, j) else 1
        return (i, j, k, flag_zero)

    # End-of-input ε-transitions to check and accept
    M.add_rule("qA", None, "qC", update=compute_gcd_flag)  # covers a* (possibly empty b*, c*)
    M.add_rule("qB", None, "qC", update=compute_gcd_flag)  # covers a* b* (no c yet)
    M.add_rule("qC", None, "qC", update=compute_gcd_flag)  # covers a* b* c*
    M.add_rule("qC", None, "qF")

    return M


def build_mul_kcm() -> KCM:
    """
    Accept L = { a^i b^j c^k | k = i * j } over alphabet {'a','b','c'}.

    Counters:
      c0 = i  (# of 'a')
      c1 = j  (# of 'b')
      c2 = k  (# of 'c')
      c3 = flag_zero (0 if accept, 1 if reject)

    Acceptance mask requires c3 == 0  -> mask = (False, False, False, True)

    Deterministic scanner:
      qA reads a* (counts i)
      qB reads b* (counts j)
      qC reads c* (counts k)
      then ε to qC to set flag, ε to qF (mask-enforced)
      Any out-of-order symbol ⇒ no rule ⇒ reject.
    """
    M = KCM(k=4)

    # States
    M.add_state("qA", initial=True)  # reading a*
    M.add_state("qB")                # reading b*
    M.add_state("qC")                # reading c*
    M.add_state("qC")              # ε: compute flag
    M.add_state("qF")              # final with mask

    # Alphabet
    M.add_symbol('a'); M.add_symbol('b'); M.add_symbol('c')

    # Final + mask: only c3 must be zero
    M.add_final("qF", mask=(False, False, False, True))

    # Updates
    def inc_a(n):
        i, j, k, f = n
        return (i + 1, j, k, f)

    def inc_b(n):
        i, j, k, f = n
        return (i, j + 1, k, f)

    def inc_c(n):
        i, j, k, f = n
        return (i, j, k + 1, f)

    # Deterministic block reading: a* b* c*
    M.add_rule("qA", 'a', "qA", update=inc_a)
    M.add_rule("qA", 'b', "qB", update=inc_b)
    M.add_rule("qB", 'b', "qB", update=inc_b)
    M.add_rule("qB", 'c', "qC", update=inc_c)
    M.add_rule("qC", 'c', "qC", update=inc_c)
    # No 'a' after qB/qC; no 'b' after qC.

    # ε: compute flag_zero := 0 iff k == i*j, else 1
    def compute_mul_flag(n):
        i, j, k, _ = n
        flag_zero = 0 if k == i * j else 1
        return (i, j, k, flag_zero)

    # End-of-input ε transitions to check and accept
    M.add_rule("qA", None, "qC", update=compute_mul_flag)  # (a* only)
    M.add_rule("qB", None, "qC", update=compute_mul_flag)  # (a* b*)
    M.add_rule("qC", None, "qC", update=compute_mul_flag)  # (a* b* c*)
    M.add_rule("qC", None, "qF")

    return M


if __name__ == "__main__":
    # Example: build and test the prime unary KCM
    M = build_dvd_kcm()
    test_strings = ["", "abb", 'abbbbbbbbbbbbbb']
    for i in range(100):
        for j in range(100):
            s = 'a' * i + 'b' * j
            result = M.accepts(list(s))
            if result != (j % i == 0 if i != 0 else False):
                print(1)
            # show expected divisibility: guard against i == 0 to avoid modulo-by-zero
            # print(f"Input: '{s}' | Accepted: {result}", {(j % i == 0) if i != 0 else False})
            
    # for s in test_strings:
    #     result = M.accepts(list(s))
    #     print(f"Input: '{s}' | Accepted: {result}")