from __future__ import annotations
from collections import deque
from dataclasses import dataclass
from typing import Callable, Dict, Iterable, List, Optional, Tuple, Set

Symbol = Optional[str]  # None = epsilon

@dataclass(frozen=True)
class Rule:
    q_to: str
    sym: Symbol
    update: Callable[[Tuple[int, ...]], Tuple[int, ...]]

def _is_probable_prime(n: int) -> bool:
        """Fast Miller–Rabin. Deterministic for 64-bit, probabilistic for larger."""
        if n < 2:
            return False
        # small primes
        small = (2, 3, 5, 7, 11, 13, 17, 19, 23, 29)
        for p in small:
            if n == p:
                return True
            if n % p == 0 and n != p:
                return False

        # write n-1 = d*2^s
        d = n - 1
        s = 0
        while d % 2 == 0:
            d //= 2
            s += 1

        # Bases: enough for deterministic up to 2^64; otherwise use a few random bases.
        # Ref: https://miller-rabin.appspot.com/
        if n < (1 << 64):
            bases = (2, 3, 5, 7, 11, 13, 17)
        else:
            import random
            # 8 random bases is already extremely strong for 200-bit numbers
            rng = random.Random(0xC0FFEE)  # fixed seed for reproducibility
            bases = tuple(rng.randrange(2, n - 2) for _ in range(8))

        def check(a: int) -> bool:
            x = pow(a, d, n)
            if x == 1 or x == n - 1:
                return True
            for _ in range(s - 1):
                x = (x * x) % n
                if x == n - 1:
                    return True
            return False

        return all(check(a % n) for a in bases)


class KCM:
    def __init__(self, k: int):
        assert k >= 0
        self.k: int = k
        self.Q: Set[str] = set()
        self.Sigma: Set[str] = set()
        self.q0: Optional[str] = None
        self.F: List[Tuple[str, Tuple[bool, ...]]] = []  # (q_final, zero-mask)
        self.transitions: Dict[Tuple[str, Symbol], List[Rule]] = {}

    # ----- build API -----

    def add_state(self, q: str, *, initial: bool = False) -> None:
        self.Q.add(q)
        if initial:
            self.q0 = q

    def add_final(self, q: str, mask: Optional[Tuple[bool, ...]] = None) -> None:
        if mask is None:
            mask = tuple(False for _ in range(self.k))
        assert len(mask) == self.k
        self.F.append((q, mask))

    def add_symbol(self, a: str) -> None:
        self.Sigma.add(a)

    def add_rule(
        self,
        q_from: str,
        sym: Symbol,
        q_to: str,
        update: Optional[Callable[[Tuple[int, ...]], Tuple[int, ...]]] = None,
    ) -> None:
        if update is None:
            update = lambda n: n
        self.Q.update([q_from, q_to])
        if sym is not None:
            self.Sigma.add(sym)
        self.transitions.setdefault((q_from, sym), []).append(Rule(q_to, sym, update))

    # ----- acceptance (canonical) -----

    @staticmethod
    def _z(n: Tuple[int, ...]) -> Tuple[bool, ...]:
        return tuple(c == 0 for c in n)

    @staticmethod
    def _mask_ok(z: Tuple[bool, ...], m: Tuple[bool, ...]) -> bool:
        return all((not m_i) or z_i for z_i, m_i in zip(z, m))

    def _is_accepting(self, q: str, n: Tuple[int, ...]) -> bool:
        z = self._z(n)
        return any(q == qf and self._mask_ok(z, m) for (qf, m) in self.F)

    # ----- deterministic runner (fixed ε policy) -----

    def run_dtm_trace(
        self, w: Iterable[str]
    ) -> Tuple[List[Tuple[int, str, Tuple[int, ...], Symbol]], bool]:
        """
        Deterministic stepping:
        - NO ε-closure before/during reading.
        - Apply ε-closure ONLY AFTER the entire input is consumed.
        - Bail out if branching (>=2 rules) is encountered.
        """
        assert self.q0 is not None, "Start state not set"
        w_list = list(w)
        N = len(w_list)

        q = self.q0
        n = tuple(0 for _ in range(self.k))
        i = 0
        trace: List[Tuple[int, str, Tuple[int, ...], Symbol]] = [(0, q, n, None)]

        # READ PHASE
        while i < N:
            a = w_list[i]
            rules = self.transitions.get((q, a), [])
            if len(rules) != 1:
                return trace, False
            r = rules[0]
            n = r.update(n)
            q = r.q_to
            i += 1
            trace.append((i, q, n, a))

        # FINAL ε-CLOSURE (only now)
        while True:
            eps = self.transitions.get((q, None), [])
            if len(eps) == 1:
                r = eps[0]
                n = r.update(n)
                q = r.q_to
                trace.append((i, q, n, None))
            elif len(eps) == 0:
                break
            else:
                return trace, False  # ε-branching → not deterministic

        accepted = (i == N) and self._is_accepting(q, n)
        return trace, accepted


    # ----- BFS runner with ε -----

    def computation_trace(
        self, w: Iterable[str], max_confs: int = 200_000
    ) -> Tuple[List[Tuple[int, str, Tuple[int, ...], Symbol]], bool]:
        assert self.q0 is not None, "Start state not set"
        w_list = list(w)
        N = len(w_list)

        start = (0, self.q0, tuple(0 for _ in range(self.k)))
        qd: deque[Tuple[int, str, Tuple[int, ...]]] = deque([start])
        parents: Dict[Tuple[int, str, Tuple[int, ...]],
                      Tuple[Tuple[int, str, Tuple[int, ...]], Symbol]] = {}
        seen: Set[Tuple[int, str, Tuple[int, ...]]] = {start}
        pushes = 1
        best = start

        def backtrace(end_cfg: Tuple[int, str, Tuple[int, ...]], end_sym: Symbol
                      ) -> List[Tuple[int, str, Tuple[int, ...], Symbol]]:
            path: List[Tuple[int, str, Tuple[int, ...], Symbol]] = []
            cur = end_cfg; sym = end_sym
            path.append((*cur, sym))
            while cur in parents:
                cur, sym = parents[cur]
                path.append((*cur, sym))
            path.reverse()
            if path and path[0][3] is not None:
                i0, q0, n0, _ = path[0]
                path[0] = (i0, q0, n0, None)
            return path

        while qd:
            i, q, n = qd.popleft()
            if i > best[0]:
                best = (i, q, n)

            if i == N and self._is_accepting(q, n):
                return backtrace((i, q, n), None), True

            # ε moves
            for r in self.transitions.get((q, None), []):
                child = (i, r.q_to, r.update(n))
                if child not in seen:
                    seen.add(child)
                    parents[child] = ((i, q, n), None)
                    qd.append(child)
                    pushes += 1
                    if pushes > max_confs:
                        return backtrace(child, None), False

            # symbol moves
            if i < N:
                a = w_list[i]
                for r in self.transitions.get((q, a), []):
                    child = (i + 1, r.q_to, r.update(n))
                    if child not in seen:
                        seen.add(child)
                        parents[child] = ((i, q, n), a)
                        qd.append(child)
                        pushes += 1
                        if pushes > max_confs:
                            return backtrace(child, a), False

        return backtrace(best, None), False

    # ----- API -----

    def accepts(self, w: Iterable[str]) -> bool:
        # Try the deterministic fast path.
        try:
            trace, acc = self.run_dtm_trace(w)
        except Exception:
            # If DTM stepping itself fails, just use BFS.
            _, acc = self.computation_trace(w)
            return acc

        if acc:
            return True

        # IMPORTANT: fall back to BFS on the ORIGINAL input, not on a
        # "consumed symbols" reconstruction from the DTM trace.
        _, acc_bfs = self.computation_trace(w)
        return acc_bfs


   
    def output_generator(self, seq: Iterable[str], alphabet: List[str]) -> str:
        w = list(seq)

        # 1) Try the deterministic runner first (fast path).
        dtm_trace, dtm_acc = self.run_dtm_trace(w)

        # If the machine is deterministic (no branching found), we're done.
        # For all your KCMs (mul/gcd/add/dvd/prime/pow2), this holds.
        if dtm_trace and not (len(self.transitions.get((dtm_trace[-1][1], None), [])) > 1):
            trace, acc = dtm_trace, dtm_acc
        else:
            # 2) Fall back to ε-BFS only when DTM encountered branching
            #    or returned an obviously too-short trace.
            trace, acc = self.computation_trace(w, max_confs=200_000)

            # If BFS didn’t accept, keep the longer/more informative trace for logging.
            def consumed_len(tr): return sum(1 for *_, s in tr if s is not None)
            if not acc and consumed_len(dtm_trace) >= consumed_len(trace):
                trace, acc = dtm_trace, dtm_acc

        # Assemble the output string
        step_tuples = [f'{q}_{("*" if s is None else s)}' for _, q, _, s in trace]
        trace_str = '-'.join(step_tuples)
        alphabet_str = ''.join(alphabet)
        states = len(self.Q)
        acc_str = '1' if acc else '0'
        return f"{states}_{self.k}_{alphabet_str}_{trace_str}?{acc_str}"


def build_prime_binary_kcm() -> KCM:
    """
    L = { bin(p) | p is prime } over alphabet {'0','1'} (no separators).
    Canonical encoding: must start with '1' (no leading zeros).
    Accepts e.g. '10'(2), '11'(3), '101'(5), '111'(7), '1011'(11).
    Rejects e.g. '', '0', '01', '1'(=1), '100'(=4), etc.

    Counters:
      c0 = numeric value parsed from binary
      c1 = flag_zero (0 if prime, 1 otherwise)
    """
    M = KCM(k=2)

    # States
    M.add_state("qS", initial=True)   # must see first '1'
    M.add_state("qR")                 # read remaining bits (0/1)
    M.add_state("F")                 # ε: compute primality
    M.add_state("G")                 # masked accept (flag must be 0)
    M.add_state("H")              # sink (e.g., leading zero)

    # Alphabet & final
    M.add_symbol('0'); M.add_symbol('1')
    M.add_final("G", mask=(False, True))  # require flag_zero == 0

    # Append helpers
    def append0(n): v, f = n; return (v*2 + 0, f)
    def append1(n): v, f = n; return (v*2 + 1, f)

    # First symbol must be '1'
    M.add_rule("qS", '1', "qR", update=append1)
    M.add_rule("qS", '0', "H")           # leading zero -> reject
    M.add_rule("H", '0', "H")
    M.add_rule("H", '1', "H")

    # Subsequent symbols: either 0 or 1
    M.add_rule("qR", '0', "qR", update=append0)
    M.add_rule("qR", '1', "qR", update=append1)

    # ε: compute primality on c0
    def compute_prime_flag(n):
        v, _ = n
        ok = (v >= 2) and _is_probable_prime(v)  # explicit guard for clarity
        return (v, 0 if ok else 1)

    M.add_rule("qR", None, "F", update=compute_prime_flag)
    M.add_rule("F", None, "G")
    return M


def build_gcd_binary_kcm() -> KCM:
    """
    Accept L = { bin(i)/bin(j)/bin(k) | k = gcd(i, j) } over alphabet {'0','1','/'}.
    Canonical binaries:
      - 0  -> "0" (exactly one zero)
      - >0 -> starts with '1' (no leading zeros)
    """
    M = KCM(k=4)

    # States
    M.add_state("qI0", initial=True)   # first bit of i
    M.add_state("qI")                  # more bits of i (after leading '1')
    M.add_state("qIzeroEnd")           # i == "0" (must see '/')

    M.add_state("qJ0")                 # first bit of j
    M.add_state("qJ")                  # more bits of j (after leading '1')
    M.add_state("qJzeroEnd")           # j == "0" (must see '/')

    M.add_state("qK0")                 # first bit of k
    M.add_state("qK")                  # more bits of k (after leading '1')
    M.add_state("qKzeroEnd")           # k == "0" (no more bits)

    M.add_state("F")                  # ε: check gcd
    M.add_state("G")                  # masked accept
    M.add_state("H")               # reject sink  <-- renamed

    # Alphabet & final
    for a in ('0','1','/'):
        M.add_symbol(a)
    # c3 is flag_zero → must be zero at accept
    M.add_final("G", mask=(False, False, False, True))

    # Helpers to append bits into i, j, k
    def app0_i(n): i,j,k,f = n; return (i*2 + 0, j, k, f)
    def app1_i(n): i,j,k,f = n; return (i*2 + 1, j, k, f)
    def app0_j(n): i,j,k,f = n; return (i, j*2 + 0, k, f)
    def app1_j(n): i,j,k,f = n; return (i, j*2 + 1, k, f)
    def app0_k(n): i,j,k,f = n; return (i, j, k*2 + 0, f)
    def app1_k(n): i,j,k,f = n; return (i, j, k*2 + 1, f)

    # ----- Parse i -----
    M.add_rule("qI0", '0', "qIzeroEnd", update=app0_i)   # i == 0
    M.add_rule("qI0", '1', "qI",        update=app1_i)   # i > 0
    M.add_rule("qI0", '/', "H")                      # empty i not allowed

    M.add_rule("qI", '0', "qI", update=app0_i)
    M.add_rule("qI", '1', "qI", update=app1_i)
    M.add_rule("qI", '/', "qJ0")

    M.add_rule("qIzeroEnd", '0', "H")                # no extra bits after "0"
    M.add_rule("qIzeroEnd", '1', "H")
    M.add_rule("qIzeroEnd", '/', "qJ0")

    # ----- Parse j -----
    M.add_rule("qJ0", '0', "qJzeroEnd", update=app0_j)   # j == 0
    M.add_rule("qJ0", '1', "qJ",        update=app1_j)   # j > 0
    M.add_rule("qJ0", '/', "H")                      # empty j not allowed

    M.add_rule("qJ", '0', "qJ", update=app0_j)
    M.add_rule("qJ", '1', "qJ", update=app1_j)
    M.add_rule("qJ", '/', "qK0")

    M.add_rule("qJzeroEnd", '0', "H")                # no extra bits after "0"
    M.add_rule("qJzeroEnd", '1', "H")
    M.add_rule("qJzeroEnd", '/', "qK0")

    # ----- Parse k -----
    M.add_rule("qK0", '0', "qKzeroEnd", update=app0_k)   # k == 0
    M.add_rule("qK0", '1', "qK",        update=app1_k)   # k > 0
    M.add_rule("qK0", '/', "H")                      # empty k not allowed

    M.add_rule("qK", '0', "qK", update=app0_k)
    M.add_rule("qK", '1', "qK", update=app1_k)
    M.add_rule("qK", '/', "H")                       # no extra separator after k

    # k == "0": forbid any more symbols
    M.add_rule("qKzeroEnd", '0', "H")
    M.add_rule("qKzeroEnd", '1', "H")
    M.add_rule("qKzeroEnd", '/', "H")

    # ----- End-of-input check -----
    def compute_gcd_flag(n):
        i, j, k, _ = n
        import math
        return (i, j, k, 0 if k == math.gcd(i, j) else 1)

    # valid end states for k
    M.add_rule("qK",        None, "F", update=compute_gcd_flag)
    M.add_rule("qKzeroEnd", None, "F", update=compute_gcd_flag)
    M.add_rule("F",        None, "G")

    # Reject sink self-loops
    for a in ('0','1','/'):
        M.add_rule("H", a, "H")

    return M

def build_exp_binary_kcm() -> KCM:
    """
    Accept L = { bin(a)/bin(i)/bin(b)/bin(result) | result = a^i * b^(2^i) } over alphabet {'0','1','/'}.
    Canonical binaries:
      - 0  -> "0" (exactly one zero)
      - >0 -> starts with '1' (no leading zeros)

    Counters:
      c0 = a, c1 = i, c2 = b, c3 = result, c4 = flag_zero (0 if accept, 1 otherwise)
    """
    M = KCM(k=5)

    # States
    M.add_state("qA0", initial=True)   # first bit of a
    M.add_state("qA")                  # more bits of a
    M.add_state("qAzeroEnd")           # a == "0"

    M.add_state("qI0")                 # first bit of i
    M.add_state("qI")                  # more bits of i
    M.add_state("qIzeroEnd")           # i == "0"

    M.add_state("qB0")                 # first bit of b
    M.add_state("qB")                  # more bits of b
    M.add_state("qBzeroEnd")           # b == "0"

    M.add_state("qR0")                 # first bit of result
    M.add_state("qR")                  # more bits of result
    M.add_state("qRzeroEnd")           # result == "0"

    M.add_state("F")                  # ε: check a^i * b^(2^i)
    M.add_state("G")                  # masked accept
    M.add_state("H")               # sink

    # Alphabet & final
    for a in ('0','1','/'):
        M.add_symbol(a)
    M.add_final("G", mask=(False, False, False, False, True))  # only flag must be zero

    # Append helpers
    def app0_a(n): a,i,b,r,f = n; return (a*2 + 0, i, b, r, f)
    def app1_a(n): a,i,b,r,f = n; return (a*2 + 1, i, b, r, f)
    def app0_i(n): a,i,b,r,f = n; return (a, i*2 + 0, b, r, f)
    def app1_i(n): a,i,b,r,f = n; return (a, i*2 + 1, b, r, f)
    def app0_b(n): a,i,b,r,f = n; return (a, i, b*2 + 0, r, f)
    def app1_b(n): a,i,b,r,f = n; return (a, i, b*2 + 1, r, f)
    def app0_r(n): a,i,b,r,f = n; return (a, i, b, r*2 + 0, f)
    def app1_r(n): a,i,b,r,f = n; return (a, i, b, r*2 + 1, f)

    # ----- Parse a -----
    M.add_rule("qA0", '0', "qAzeroEnd", update=app0_a)   # a == 0
    M.add_rule("qA0", '1', "qA",        update=app1_a)   # a > 0
    M.add_rule("qA0", '/', "H")                      # empty a not allowed

    M.add_rule("qA", '0', "qA", update=app0_a)
    M.add_rule("qA", '1', "qA", update=app1_a)
    M.add_rule("qA", '/', "qI0")

    M.add_rule("qAzeroEnd", '0', "H")
    M.add_rule("qAzeroEnd", '1', "H")
    M.add_rule("qAzeroEnd", '/', "qI0")

    # ----- Parse i -----
    M.add_rule("qI0", '0', "qIzeroEnd", update=app0_i)   # i == 0
    M.add_rule("qI0", '1', "qI",        update=app1_i)   # i > 0
    M.add_rule("qI0", '/', "H")                      # empty i not allowed

    M.add_rule("qI", '0', "qI", update=app0_i)
    M.add_rule("qI", '1', "qI", update=app1_i)
    M.add_rule("qI", '/', "qB0")

    M.add_rule("qIzeroEnd", '0', "H")
    M.add_rule("qIzeroEnd", '1', "H")
    M.add_rule("qIzeroEnd", '/', "qB0")

    # ----- Parse b -----
    M.add_rule("qB0", '0', "qBzeroEnd", update=app0_b)   # b == 0
    M.add_rule("qB0", '1', "qB",        update=app1_b)   # b > 0
    M.add_rule("qB0", '/', "H")                      # empty b not allowed

    M.add_rule("qB", '0', "qB", update=app0_b)
    M.add_rule("qB", '1', "qB", update=app1_b)
    M.add_rule("qB", '/', "qR0")

    M.add_rule("qBzeroEnd", '0', "H")
    M.add_rule("qBzeroEnd", '1', "H")
    M.add_rule("qBzeroEnd", '/', "qR0")

    # ----- Parse result -----
    M.add_rule("qR0", '0', "qRzeroEnd", update=app0_r)   # result == 0
    M.add_rule("qR0", '1', "qR",        update=app1_r)   # result > 0
    M.add_rule("qR0", '/', "H")                      # empty result not allowed

    M.add_rule("qR", '0', "qR", update=app0_r)
    M.add_rule("qR", '1', "qR", update=app1_r)
    M.add_rule("qR", '/', "H")                       # no extra separator after result

    # result == "0": forbid any more symbols
    M.add_rule("qRzeroEnd", '0', "H")
    M.add_rule("qRzeroEnd", '1', "H")
    M.add_rule("qRzeroEnd", '/', "H")

    # ----- End-of-input check -----
    def compute_exp_flag(n):
        a, i, b, result, _ = n
        try:
            # Calculate a^i * b^(2^i)
            if i > 50:  # Prevent overflow for very large exponents
                expected = float('inf')
            else:
                two_power_i = 2 ** i
                if i > 30 or (a > 0 and two_power_i > 60):  # Prevent b^(2^i) overflow
                    expected = float('inf')
                else:
                    a_power_i = pow(a, i) if i < 100 else float('inf')
                    b_power_two_i = pow(b, two_power_i) if two_power_i < 100 else float('inf')
                    if a_power_i == float('inf') or b_power_two_i == float('inf'):
                        expected = float('inf')
                    else:
                        expected = a_power_i * b_power_two_i
            
            ok = (result == expected and expected != float('inf'))
        except (OverflowError, ValueError):
            ok = False
        return (a, i, b, result, 0 if ok else 1)

    M.add_rule("qR",        None, "F", update=compute_exp_flag)
    M.add_rule("qRzeroEnd", None, "F", update=compute_exp_flag)
    M.add_rule("F",        None, "G")

    # Reject sink
    for a in ('0','1','/'):
        M.add_rule("H", a, "H")

    return M

def build_dvd_binary_kcm() -> KCM:
    """
    L = { bin(w)/bin(v) | w >= 1 and w divides v } over alphabet {'0','1','/'}.
    Canonical binaries (no leading zeros):
      - w: must start with '1' (so w >= 1)
      - v: either exactly '0' OR starts with '1'
    Counters:
      c0 = w (parsed value)
      c1 = v (parsed value)
      c2 = flag_zero (0 if accept, 1 otherwise)
    """
    M = KCM(k=3)

    # States
    M.add_state("A", initial=True)   # first bit of w (must be '1')
    M.add_state("B")                  # more bits of w
    M.add_state("C")                 # first bit of v after '/'
    M.add_state("D")                  # more bits of v (after leading '1')
    M.add_state("E")           # v == "0" (no more bits allowed)
    M.add_state("F")                  # ε: compute divisibility
    M.add_state("G")                  # masked accept
    M.add_state("H")               # reject sink

    # Alphabet & final
    for a in ('0','1','/'):
        M.add_symbol(a)
    M.add_final("G", mask=(False, False, True))  # require c2 == 0

    # Helpers to append bits
    def app0_w(n): w, v, f = n; return (w*2 + 0, v, f)
    def app1_w(n): w, v, f = n; return (w*2 + 1, v, f)
    def app0_v(n): w, v, f = n; return (w, v*2 + 0, f)
    def app1_v(n): w, v, f = n; return (w, v*2 + 1, f)

    # w: first bit must be '1' (w >= 1)
    M.add_rule("A", '1', "B",    update=app1_w)
    M.add_rule("A", '0', "H")
    M.add_rule("A", '/',  "H")  # need at least one bit for w

    # w: more bits or delimiter '/'
    M.add_rule("B", '0', "B", update=app0_w)
    M.add_rule("B", '1', "B", update=app1_w)
    M.add_rule("B", '/', "C")

    # v: first bit — either exactly "0" OR starts with '1'
    M.add_rule("C", '0', "E", update=app0_v)  # v == 0; next must be end-of-input
    M.add_rule("C", '1', "D",        update=app1_v)
    M.add_rule("C", '/', "H")                      # empty v not allowed

    # v == "0": forbid any more symbols
    M.add_rule("E", '0', "H")
    M.add_rule("E", '1', "H")
    M.add_rule("E", '/', "H")

    # v after a leading '1': read arbitrary bits; no extra separator allowed
    M.add_rule("D", '0', "D", update=app0_v)
    M.add_rule("D", '1', "D", update=app1_v)
    M.add_rule("D", '/', "H")

    # ε: divisibility check at end
    def compute_divides_flag(n):
        w, v, _ = n
        ok = (w >= 1) and (v % w == 0)
        return (w, v, 0 if ok else 1)

    # Valid end-of-input states for v
    M.add_rule("D",        None, "F", update=compute_divides_flag)
    M.add_rule("E", None, "F", update=compute_divides_flag)
    M.add_rule("F",        None, "G")

    # Reject sink
    for a in ('0','1','/'):
        M.add_rule("H", a, "H")

    return M

def build_mul_binary_kcm() -> KCM:
    """
    Accept L = { bin(i)/bin(j)/bin(k) | k = i * j } over alphabet {'0','1','/'}.
    Canonical binaries:
      - 0  -> "0"
      - >0 -> starts with '1' (no leading zeros)
    """
    M = KCM(k=4)

    # States
    M.add_state("qI0", initial=True)   # first bit of i
    M.add_state("qI")                  # more bits of i (after leading '1')
    M.add_state("qIzeroEnd")           # i == "0" (must see '/')

    M.add_state("qJ0")                 # first bit of j
    M.add_state("qJ")                  # more bits of j (after leading '1')
    M.add_state("qJzeroEnd")           # j == "0" (must see '/')

    M.add_state("qK0")                 # first bit of k
    M.add_state("qK")                  # more bits of k (after leading '1')
    M.add_state("qKzeroEnd")           # k == "0" (no more bits allowed)

    M.add_state("F")                  # ε: compute flag
    M.add_state("G")                  # masked accept
    M.add_state("H")               # reject sink  <-- renamed and used consistently

    # Alphabet & final
    for a in ('0', '1', '/'):
        M.add_symbol(a)
    # c3 is flag_zero → must be zero at accept
    M.add_final("G", mask=(False, False, False, True))

    # Helpers to append bits into i, j, k
    def app0_i(n): i, j, k, f = n; return (i*2 + 0, j, k, f)
    def app1_i(n): i, j, k, f = n; return (i*2 + 1, j, k, f)
    def app0_j(n): i, j, k, f = n; return (i, j*2 + 0, k, f)
    def app1_j(n): i, j, k, f = n; return (i, j*2 + 1, k, f)
    def app0_k(n): i, j, k, f = n; return (i, j, k*2 + 0, f)
    def app1_k(n): i, j, k, f = n; return (i, j, k*2 + 1, f)

    # ----- Parse i -----
    M.add_rule("qI0", '0', "qIzeroEnd", update=app0_i)   # i == 0
    M.add_rule("qI0", '1', "qI",        update=app1_i)   # i > 0
    M.add_rule("qI0", '/', "H")                      # empty i not allowed  <-- sink

    M.add_rule("qI", '0', "qI", update=app0_i)
    M.add_rule("qI", '1', "qI", update=app1_i)
    M.add_rule("qI", '/', "qJ0")

    M.add_rule("qIzeroEnd", '0', "H")                # no more bits after "0"  <-- sink
    M.add_rule("qIzeroEnd", '1', "H")                # no more bits after "0"  <-- sink
    M.add_rule("qIzeroEnd", '/', "qJ0")

    # ----- Parse j -----
    M.add_rule("qJ0", '0', "qJzeroEnd", update=app0_j)   # j == 0
    M.add_rule("qJ0", '1', "qJ",        update=app1_j)   # j > 0
    M.add_rule("qJ0", '/', "H")                      # empty j not allowed    <-- sink

    M.add_rule("qJ", '0', "qJ", update=app0_j)
    M.add_rule("qJ", '1', "qJ", update=app1_j)
    M.add_rule("qJ", '/', "qK0")

    M.add_rule("qJzeroEnd", '0', "H")                # no more bits after "0"  <-- sink
    M.add_rule("qJzeroEnd", '1', "H")                # no more bits after "0"  <-- sink
    M.add_rule("qJzeroEnd", '/', "qK0")

    # ----- Parse k -----
    M.add_rule("qK0", '0', "qKzeroEnd", update=app0_k)   # k == 0
    M.add_rule("qK0", '1', "qK",        update=app1_k)   # k > 0
    M.add_rule("qK0", '/', "H")                      # empty k not allowed    <-- sink

    M.add_rule("qK", '0', "qK", update=app0_k)
    M.add_rule("qK", '1', "qK", update=app1_k)
    M.add_rule("qK", '/', "H")                       # no separator allowed after k  <-- sink

    # k == "0": forbid any more symbols
    M.add_rule("qKzeroEnd", '0', "H")
    M.add_rule("qKzeroEnd", '1', "H")
    M.add_rule("qKzeroEnd", '/', "H")

    # ----- End-of-input check -----
    def compute_mul_flag(n):
        i, j, k, _ = n
        return (i, j, k, 0 if k == i * j else 1)

    # valid end states for k
    M.add_rule("qK",       None, "F", update=compute_mul_flag)
    M.add_rule("qKzeroEnd", None, "F", update=compute_mul_flag)
    M.add_rule("F",       None, "G")

    # Reject sink self-loops
    for a in ('0', '1', '/'):
        M.add_rule("H", a, "H")

    return M


def build_add_binary_kcm() -> KCM:
    """
    Accept L = { bin(i)/bin(j)/bin(k) | k = i + j } over alphabet {'0','1','/'}.
    Canonical binaries:
      - 0  -> "0" (exactly one zero)
      - >0 -> starts with '1' (no leading zeros)
    """
    M = KCM(k=4)

    # States
    M.add_state("qI0", initial=True)   # first bit of i
    M.add_state("qI")                  # more bits of i
    M.add_state("qIzeroEnd")           # i == 0

    M.add_state("qJ0")                 # first bit of j
    M.add_state("qJ")                  # more bits of j
    M.add_state("qJzeroEnd")           # j == 0

    M.add_state("qK0")                 # first bit of k
    M.add_state("qK")                  # more bits of k
    M.add_state("qKzeroEnd")           # k == 0

    M.add_state("F")                  # ε: check addition
    M.add_state("G")                  # masked accept
    M.add_state("H")               # sink

    # Alphabet & final
    for a in ('0','1','/'):
        M.add_symbol(a)
    M.add_final("G", mask=(False, False, False, True))  # c3=flag must be 0

    # Append helpers
    def app0_i(n): i,j,k,f = n; return (i*2 + 0, j, k, f)
    def app1_i(n): i,j,k,f = n; return (i*2 + 1, j, k, f)
    def app0_j(n): i,j,k,f = n; return (i, j*2 + 0, k, f)
    def app1_j(n): i,j,k,f = n; return (i, j*2 + 1, k, f)
    def app0_k(n): i,j,k,f = n; return (i, j, k*2 + 0, f)
    def app1_k(n): i,j,k,f = n; return (i, j, k*2 + 1, f)

    # ----- Parse i -----
    M.add_rule("qI0", '0', "qIzeroEnd", update=app0_i)   # i == 0
    M.add_rule("qI0", '1', "qI",        update=app1_i)   # i > 0
    M.add_rule("qI0", '/', "H")                      # empty i not allowed

    M.add_rule("qI", '0', "qI", update=app0_i)
    M.add_rule("qI", '1', "qI", update=app1_i)
    M.add_rule("qI", '/', "qJ0")

    M.add_rule("qIzeroEnd", '0', "H")
    M.add_rule("qIzeroEnd", '1', "H")
    M.add_rule("qIzeroEnd", '/', "qJ0")

    # ----- Parse j -----
    M.add_rule("qJ0", '0', "qJzeroEnd", update=app0_j)   # j == 0
    M.add_rule("qJ0", '1', "qJ",        update=app1_j)   # j > 0
    M.add_rule("qJ0", '/', "H")                      # empty j not allowed

    M.add_rule("qJ", '0', "qJ", update=app0_j)
    M.add_rule("qJ", '1', "qJ", update=app1_j)
    M.add_rule("qJ", '/', "qK0")

    M.add_rule("qJzeroEnd", '0', "H")
    M.add_rule("qJzeroEnd", '1', "H")
    M.add_rule("qJzeroEnd", '/', "qK0")

    # ----- Parse k -----
    M.add_rule("qK0", '0', "qKzeroEnd", update=app0_k)   # k == 0
    M.add_rule("qK0", '1', "qK",        update=app1_k)   # k > 0
    M.add_rule("qK0", '/', "H")                      # empty k not allowed

    M.add_rule("qK", '0', "qK", update=app0_k)
    M.add_rule("qK", '1', "qK", update=app1_k)

    # k == "0": forbid any more symbols
    M.add_rule("qKzeroEnd", '0', "H")
    M.add_rule("qKzeroEnd", '1', "H")
    M.add_rule("qKzeroEnd", '/', "H")

    # ----- End-of-input check -----
    def compute_add_flag(n):
        i, j, k, _ = n
        return (i, j, k, 0 if k == i + j else 1)

    M.add_rule("qK", None, "F", update=compute_add_flag)
    M.add_rule("qKzeroEnd", None, "F", update=compute_add_flag)
    M.add_rule("F", None, "G")

    # Reject sink
    for a in ('0','1','/'):
        M.add_rule("H", a, "H")

    return M

 