'''
utilities to start jobs in AWS
'''

import sys, os, json, glob, shutil, random, uuid

def script_creator(output_folder, commands, time_in_mins=15000, num_gpus=8, code_dir="/fsx/ganayu/code/SuperShaper", wandb="online", wandb_entity="ganayu", wandb_project="effbert", generate_port=False, part_size=-1, torch_distributed_debug=None):
    if os.path.exists(output_folder):
        print("%s already exists"%(output_folder))
        sys.exit(0)
    print("ensure you are in login node and 'basic' conda environment")
    # create output folder
    os.makedirs(output_folder)
    # copy the current codebase
    cur_code_dir = output_folder + "/code"
    shutil.copytree(code_dir, cur_code_dir)
    # create environment script
    cur_envscript = output_folder + "/custom_config.yaml"
    w_envscript = open(cur_envscript, "w")
    for line in open(code_dir + "/default_config.yaml"):
        line = line.strip()
        if "num_processes" in line:
            line = "num_processes: %d"%num_gpus
        w_envscript.write(line.strip()+"\n")
    w_envscript.close()
    # create job scripts
    cur_bsz, cur_master_commands = 0, []
    master_shell_f = output_folder + "/master%d.sh"%len(cur_master_commands)
    cur_master_commands.append('bash %s'%(master_shell_f))
    w_master = open(master_shell_f, "w")
    for command in commands:
        cur_out_dir = output_folder + "/" + command["exp_name"]
        os.makedirs(cur_out_dir, exist_ok=True)
        cur_sbatch_f = cur_out_dir + "/sbatch.sh"
        cur_shell_f = cur_out_dir + "/shell.sh"
        w_sbatch = open(cur_sbatch_f, 'w')
        w_sbatch.write("#!/bin/bash\n")
        w_sbatch.write("#SBATCH --job-name=%s\n"%command["exp_name"])
        w_sbatch.write("#SBATCH --output=%s\n"%(cur_out_dir + "/sys.out"))
        w_sbatch.write("#SBATCH --error=%s\n"%(cur_out_dir + "/sys.err"))
        w_sbatch.write("#SBATCH --partition=a100\n")
        w_sbatch.write("#SBATCH --nodes=1\n")
        w_sbatch.write("#SBATCH --ntasks-per-node=1\n")
        w_sbatch.write("#SBATCH --cpus-per-task=10\n")
        w_sbatch.write("#SBATCH --gres=gpu:%d\n"%num_gpus)
        w_sbatch.write("#SBATCH --time %d\n"%time_in_mins)
        w_sbatch.write("module purge\n")
        w_sbatch.write("echo $SLURMD_NODENAME $SLURM_JOB_ID $CUDA_VISIBLE_DEVICES $SLURM_LOCALID\n")
        w_sbatch.write("nvidia-smi\n")
        # w_sbatch.write("wandb %s\n"%(wandb))
        w_sbatch.write("srun --label %s"%cur_shell_f)
        w_sbatch.close()
        w_shell = open(cur_shell_f, "w")
        w_shell.write("#!/bin/bash\n")
        if wandb != "no":
            w_shell.write("wandb %s\n"%(wandb))
        w_shell.write("TOKENIZERS_PARALLELISM=false\n")
        w_shell.write("cd %s\n"%cur_code_dir)
        for run in command["runs"]:
            if "<<OUTPUT_DIR>>" in run["params"]:
                run["params"] = run["params"].replace("<<OUTPUT_DIR>>", cur_out_dir)      
            if wandb != "no":
                w_shell.write("WANDB_MODE=%s "%wandb)
            if torch_distributed_debug is not None:
                w_shell.write("TORCH_DISTRIBUTED_DEBUG=DETAIL ")
            w_shell.write("accelerate launch --config_file %s"%(cur_envscript))
            if generate_port:
                w_shell.write(" --main_process_port %d"%(random.randint(17000, 22000)))
            w_shell.write(" %s %s"%(run["pyfile"], run["params"]))
            if wandb != "no":
                w_shell.write(" --wandb_suffix %s --wandb_entity %s --wandb_project %s"%(command["exp_name"], wandb_entity, wandb_project))
            w_shell.write("\n")
        w_shell.close()
        w_master.write("sbatch %s\n"%(cur_sbatch_f))
        os.system("chmod 777 %s/*.sh"%(cur_out_dir))
        cur_bsz += 1
        if part_size != -1 and cur_bsz % part_size == 0:
            w_master.close()
            cur_bsz = 0
            master_shell_f = output_folder + "/master%d.sh"%len(cur_master_commands)
            cur_master_commands.append('bash %s'%(master_shell_f))
            w_master = open(master_shell_f, "w")
    w_master.close()
    os.system("chmod 777 %s/*.sh"%(output_folder))
    for master_command in cur_master_commands:
        print(master_command)

def dataset_factory(name):
    datasets = {}
    
    # bert-pretraining data
    datasets["wikibooks"] = "/fsx/ganayu/data/bert_pretraining_data/wikibooks_datasets_final"
    datasets["wikibooks_dummy"] = "/fsx/ganayu/data/bert_pretraining_data/wikibooks_datasets_dummy"
    datasets["wikibooks_tokenized"] = "/fsx/ganayu/data/bert_pretraining_data/wikibooks_datasets_final_tokenized"
    datasets["wikibooks_dummy_tokenized"] = "/fsx/ganayu/data/bert_pretraining_data/wikibooks_datasets_dummy_tokenized"
    datasets["wikibooks_graphcore_128len"] = "/fsx/ganayu/data/bert_pretraining_data/wikibooks_graphcore_128"
    datasets["wikibooks_graphcore_128len_next_sentence_label_removed_w_splits"] = "/fsx/ganayu/data/bert_pretraining_data/wikibooks_graphcore_128_next_sentence_label_removed_w_splits"

    # academic bert preprocessing data
    datasets["wikibooks_acabert_bertbaseuncased_128len"] = "/fsx/ganayu/data/academic_bert_dataset/final_bert_preproc_128" 
    datasets["wikibooks_acabert_robertabase_128len"] = "/fsx/ganayu/data/academic_bert_dataset/final_roberta_preproc_128" 

    # c4-pretraining data
    datasets["c4"] = "/fsx/ganayu/data/supershaper_pretraining_data/c4_datasets_cleaned"

    return datasets[name]

def task_specific_trained_teacher(name):
    teacher_models = {}

    teacher_models["bertbase"] = {"cola": "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mnli_cola_ckptneeded/32_bertbase_cola_2e-5_16_4", "mnli": "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mnli_cola_ckptneeded/6_bertbase_mnli_3e-5_16_2", "sst2": "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mnli_cola_ckptneeded/43_bertbase_sst2_3e-5_16_3", "qnli": "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mnli_cola_ckptneeded/60_bertbase_qnli_3e-5_16_2", "qqp": "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mnli_cola_ckptneeded/77_bertbase_qqp_5e-5_32_4", "mrpc": "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mrpc_rte_stsb_ckptneeded/1_bertbase_mrpc_5e-5_16_3", "rte": "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mrpc_rte_stsb_ckptneeded/22_bertbase_rte_5e-5_32_3"}

    return teacher_models[name]

def config_factory(name):
    configs = {}
    # std. supershaper
    configs["supershaper.standard.train_mlm"] = {"per_device_train_batch_size": 128, "per_device_eval_batch_size": 256, "gradient_accumulation_steps": 2, "fp16": 1, "max_seq_length": 128, "mixing": "bert-bottleneck", "max_train_steps": 175214, "c4_dir": dataset_factory("c4"), "model_name_or_path": "bert-base-cased", "sampling_type": "random", "sampling_rule": "sandwich", "learning_rate": 2e-5, "weight_decay": 0.01, "num_warmup_steps": 10000, "eval_random_subtransformers": 1, "output_dir": "<<OUTPUT_DIR>>", "preprocessing_num_workers": 1}
    configs["supershaper.standard.train_glue"] = {"learning_rate": 5e-05, "mixing": "bert-bottleneck", "model_name_or_path": "", "num_train_epochs": 10, "per_device_train_batch_size": 4, "sampling_type": "none", "task": "", "subtransformer_config_path": "", "output_dir": "<<OUTPUT_DIR>>"}

    # phase-2 changes (didn't work)
    # 125000*2048 == 1000000*256
    configs["bertbase.standard.train_mlm"] = {"per_device_train_batch_size": 128, "per_device_eval_batch_size": 256, "gradient_accumulation_steps": 2, "fp16": 1, "max_seq_length": 512, "mixing": "attention", "max_train_steps": 125000, "c4_dir": dataset_factory("wikibooks"), "model_name_or_path": "bert-base-cased", "sampling_type": "none", "sampling_rule": "none", "learning_rate": 1e-4, "weight_decay": 0.01, "num_warmup_steps": 10000, "eval_random_subtransformers": 0, "output_dir": "<<OUTPUT_DIR>>", "preprocessing_num_workers": 1, "betas_2": 0.98}
    configs["supernetbase.standard.train_mlm"] = {"per_device_train_batch_size": 128, "per_device_eval_batch_size": 256, "gradient_accumulation_steps": 2, "fp16": 1, "max_seq_length": 512, "mixing": "bert-bottleneck", "max_train_steps": 125000, "tokenized_c4_dir": dataset_factory("wikibooks_tokenized"), "model_name_or_path": "bert-base-cased", "sampling_type": "random", "sampling_rule": "sandwich", "learning_rate": 1e-4, "weight_decay": 0.01, "num_warmup_steps": 10000, "eval_random_subtransformers": 0, "output_dir": "<<OUTPUT_DIR>>", "preprocessing_num_workers": 1, "betas_2": 0.98}
    # CUDA OOM issue
    configs["supernetbase.standard.train_mlm"] = modify_config(configs["supernetbase.standard.train_mlm"], {"per_device_train_batch_size": "16", "gradient_accumulation_steps": "16", "per_device_eval_batch_size": "16"})
    configs["supernetbase.standard.train_mlm.32bsz"] = modify_config(configs["supernetbase.standard.train_mlm"], {"per_device_train_batch_size": "32", "gradient_accumulation_steps": "8", "per_device_eval_batch_size": "32"})
    configs["supernetbase.standard.train_glue"] = {"learning_rate": 5e-05, "mixing": "bert-bottleneck", "model_name_or_path": "", "num_train_epochs": 10, "per_device_train_batch_size": 4, "sampling_type": "none", "task": "", "subtransformer_config_path": "", "output_dir": "<<OUTPUT_DIR>>"}

    # phase-3 after reading academic BERT paper https://arxiv.org/pdf/2104.07705.pdf
    configs["supernetbasev3.standard.train_mlm"] = {"per_device_train_batch_size": 128, "per_device_eval_batch_size": 256, "gradient_accumulation_steps": 2, "fp16": 1, "max_seq_length": 128, "mixing": "bert-bottleneck", "max_train_steps": 125000, "tokenized_c4_dir": dataset_factory("wikibooks_graphcore_128len_next_sentence_label_removed_w_splits"), "model_name_or_path": "bert-base-uncased", "sampling_type": "random", "sampling_rule": "sandwich", "learning_rate": 5e-4, "weight_decay": 0.01, "num_warmup_steps": 10000, "eval_random_subtransformers": 0, "output_dir": "<<OUTPUT_DIR>>", "preprocessing_num_workers": 1, "betas_2": 0.98, "tokenizer_name": "Graphcore/bert-base-uncased"}
    configs["bertbasev3.standard.train_mlm"] = {"per_device_train_batch_size": 128, "per_device_eval_batch_size": 256, "gradient_accumulation_steps": 2, "fp16": 1, "max_seq_length": 128, "mixing": "bert-bottleneck", "max_train_steps": 125000, "tokenized_c4_dir": dataset_factory("wikibooks_graphcore_128len_next_sentence_label_removed_w_splits"), "model_name_or_path": "bert-base-uncased", "sampling_type": "none", "sampling_rule": "none", "learning_rate": 5e-4, "weight_decay": 0.01, "num_warmup_steps": 10000, "eval_random_subtransformers": 0, "output_dir": "<<OUTPUT_DIR>>", "preprocessing_num_workers": 1, "betas_2": 0.98, "tokenizer_name": "Graphcore/bert-base-uncased"}
    configs["supernetbasev3.standard.train_glue"] = {"learning_rate": 5e-05, "mixing": "bert-bottleneck", "model_name_or_path": "", "num_train_epochs": 10, "per_device_train_batch_size": 4, "sampling_type": "none", "task_name": "", "output_dir": "<<OUTPUT_DIR>>", "tokenizer_name": "Graphcore/bert-base-uncased", "skip_saving_checkpoints": "yes"} # "subtransformer_config_path": ""
    # configs["supernetbasev3.standard.train_glue_original"] = {"learning_rate": 5e-05, "model_name_or_path": "", "num_train_epochs": 10, "per_device_train_batch_size": 4, "task_name": "", "output_dir": "<<OUTPUT_DIR>>", "tokenizer_name": "Graphcore/bert-base-uncased"} 
    # configs["supernetbasev3.standard.nockptsavings.train_glue_original"] =  modify_config(configs["supernetbasev3.standard.train_glue_original"], {"skip_saving_checkpoints" : "yes"})
    configs["supernetbasev3.evosearch"] =  {"output_dir": "<<OUTPUT_DIR>>", "mixing": "bert-bottleneck", "supernet_ckpt_dir": "", "data_dir": dataset_factory("wikibooks_graphcore_128len_next_sentence_label_removed_w_splits")}
    configs["supernetbasev3.standard.train_glue_original"] = {"learning_rate": "", "model_name_or_path": "", "num_train_epochs": 10, "per_device_train_batch_size": "", "task_name": "", "output_dir": "<<OUTPUT_DIR>>", "tokenizer_name": "Graphcore/bert-base-uncased",  "skip_saving_checkpoints": "yes", "seed": "333"} 
    configs["supernetbasev3.standard.ckpt_needed.train_glue_original"] = {"learning_rate": "", "model_name_or_path": "", "num_train_epochs": 10, "per_device_train_batch_size": "", "task_name": "", "output_dir": "<<OUTPUT_DIR>>", "tokenizer_name": "Graphcore/bert-base-uncased",  "skip_saving_checkpoints": "no", "seed": "333"} 
    configs["supernetbasev3.standard.mnlickpt.train_glue_original"] = {"learning_rate": "", "model_name_or_path": "", "num_train_epochs": 10, "per_device_train_batch_size": "", "task_name": "", "output_dir": "<<OUTPUT_DIR>>", "tokenizer_name": "Graphcore/bert-base-uncased",  "skip_saving_checkpoints": "yes", "seed": "333", "is_mnli_checkpoint": 1} 
    configs["supernetbasev3.standard.supernet_finetune"] = {"learning_rate": "", "model_name_or_path": "", "num_train_epochs": 10, "per_device_train_batch_size": "", "task_name": "", "output_dir": "<<OUTPUT_DIR>>", "tokenizer_name": "Graphcore/bert-base-uncased",  "skip_saving_checkpoints": "no", "seed": "333", "sampling_type": "random", "sampling_rule": "sandwich"}
    configs["supernetbasev3.standard.distill.supernet_finetune"] = {"learning_rate": "", "model_name_or_path": "", "num_train_epochs": 10, "per_device_train_batch_size": "", "task_name": "", "output_dir": "<<OUTPUT_DIR>>", "tokenizer_name": "bert-base-uncased",  "skip_saving_checkpoints": "yes", "seed": "333", "sampling_type": "none", "sampling_rule": "none", "freeze_largest_model": "yes", "freeze_smallest_model": "yes", "inplace_distillation": "1", "distillation_type": "hard+hiddenlastlayer", "inplace_kd_hard_loss_contrib": "1.0", "inplace_kd_distill_loss_contrib": "1.0", "inplace_kd_layers": "0+1+2+3+4+5+6+7+8+9+10+11+12", "teacher_model_path": "", "subtransformer_config_path": ""}  

    return configs[name]

def modify_config_and_to_string(prev_config, updates):
    for update_key in updates:
        prev_config[update_key] = updates[update_key]
    out_str = ""
    for k in prev_config:
        out_str += "--%s %s "%(k, str(prev_config[k]))
    return out_str.strip()

def modify_config(prev_config, updates):
    for update_key in updates:
        prev_config[update_key] = updates[update_key]
    return prev_config

def get_experiments_dir():
    return "/fsx/ganayu/experiments/supershaper"

def get_glue_datasets():
    return ["mnli", "cola", "mrpc", "qnli", "qqp", "rte", "sst2", "stsb"] #, "wnli"]

def get_model_configs(model_name):
    configs = {}

    # bert 12L_120H vs. 12L_768H
    configs["bert.bottleneck.12L_120H"] = "/fsx/ganayu/experiments/supershaper/configs/bert/bertbase_12L_120H.csv"
    configs["bert.bottleneck.12L_768H"] = "/fsx/ganayu/experiments/supershaper/configs/bert/bertbase_12L_768H.csv"
    configs["bertuncased.bottleneck.12L_120H"] = "/fsx/ganayu/experiments/supershaper/configs/bert/bertbase_uncased_12L_120H.csv"
    configs["bertuncased.bottleneck.12L_768H"] = "/fsx/ganayu/experiments/supershaper/configs/bert/bertbase_uncased_12L_768H.csv"
    configs["bertuncased.nobottleneck.12L_768H"] = "/fsx/ganayu/experiments/supershaper/configs/bert/bertbase_uncased_nobottleneck_12L_768H.csv"
    configs["robertabase.nobottleneck.12L_768H"] = "/fsx/ganayu/experiments/supershaper/configs/bert/robertabase_uncased_nobottleneck_12L_768H.csv"

    return configs[model_name]

def get_finetuning_sweeps(method):
    sweeps = {}
    sweeps["roberta"] = {"ft_lrs":["1e-5", "2e-5", "3e-5"], "ft_bsz":["16", "32"], "ft_epochs":["10"], "warmup_ratio": 0.06}
    sweeps["wellreadstudents"] = {"ft_lrs":["3e-4", "1e-4", "5e-5", "3e-5"], "ft_bsz":["8", "16", "32", "64", "128"], "ft_epochs":["4"]}
    sweeps["supershaper"] = {"ft_lrs":["5e-5"], "ft_bsz":["32"], "ft_epochs":["10"]}
    sweeps["summer"] = {"ft_lrs": [
    "1e-6", "2e-6", "5e-6", "1e-5", "2e-5", "5e-5", "1e-4", "2e-4", "5e-4", "1e-3", "2e-3", "5e-3"], "ft_bsz":["8", "16", "32"], "ft_epochs":["10"]}
    sweeps["bert"] = {"ft_lrs":["5e-5", "3e-5", "2e-5"], "ft_bsz":["16", "32"], "ft_epochs":["2", "3", "4"]}
    sweeps["autodistill"] = {"ft_lrs":["1e-5", "2e-5"], "ft_bsz":["32"], "ft_epochs":["5", "10", "20"]}
    return sweeps[method]

def finetuning_kd_config(name):
    kd_config = {}
    kd_config["std_logits"] = {"inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5", "distillation_type": "hard+logits", "inplace_distillation": "1"}
    return kd_config[name]

'''
phase-1 slide upto 27
'''

# script_creator(get_experiments_dir() + "/jul8_std_supershaper", [ {"exp_name": "seqlen_128", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supershaper.standard.train_mlm"), {"eval_random_subtransformers": 0})}]}, {"exp_name": "seqlen_512", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supershaper.standard.train_mlm"), {"max_seq_length": 512, "eval_random_subtransformers": 0})}]}  ] )
# script_creator(get_experiments_dir() + "/jul9_wikibooks_inplace_distillation", [ {"exp_name": "seqlen_128", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supershaper.standard.train_mlm"), {"eval_random_subtransformers": 0, "inplace_distillation": 1, "layerwise_distillation": 1, "c4_dir": dataset_factory("wikibooks")})}]}  ] )
# script_creator(get_experiments_dir() + "/jul10_wikibooks_lr5e-5", [ {"exp_name": "seqlen_128", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supershaper.standard.train_mlm"), {"eval_random_subtransformers": 0, "learning_rate": 5e-5,  "c4_dir": dataset_factory("wikibooks")})}]}  ] )
# script_creator(get_experiments_dir() + "/jul10_wikibooks_supernet_train_more_steps", [ {"exp_name": "seqlen_128", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supershaper.standard.train_mlm"), {"eval_random_subtransformers": 0, "max_train_steps": 350000, "resume_from_checkpoint_dir": "/fsx/ganayu/experiments/supershaper/jul6_bertdata_bertbottleneck_128seqlen/c4_realnews_bert-bottleneck_random_K=1_pretraining_128seqlen_08-07-2022-06-40-22/best_model", "c4_dir": dataset_factory("wikibooks")})}]}  ] )
# script_creator(get_experiments_dir() + "/jul11_wikibooks_finetune_9tasks", [ {"exp_name": "seqlen_128_%s"%TASK, "runs": [{"pyfile": "train_glue.py", "params": modify_config_and_to_string(config_factory("supershaper.standard.train_glue"), {"model_name_or_path": "/fsx/ganayu/experiments/supershaper/jul6_bertdata_bertbottleneck_128seqlen/c4_realnews_bert-bottleneck_random_K=1_pretraining_128seqlen_08-07-2022-06-40-22/best_model", "subtransformer_config_path": "/fsx/ganayu/experiments/supershaper/jul10_search_results/gpu_params60000000.0_/best_configs_iter_299.csv", "task": TASK})}]}  for TASK in get_glue_datasets() ] )
# script_creator(get_experiments_dir() + "/jul11_wikibooks_efficient_subnet_train_more_steps", [ {"exp_name": "seqlen_128", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supershaper.standard.train_mlm"), {"eval_random_subtransformers": 0, "max_train_steps": 50000, "sampling_type": "none", "sampling_rule": "none", "model_name_or_path": "/fsx/ganayu/experiments/supershaper/jul6_bertdata_bertbottleneck_128seqlen/c4_realnews_bert-bottleneck_random_K=1_pretraining_128seqlen_08-07-2022-06-40-22/best_model", "subtransformer_config_path": "/fsx/ganayu/experiments/supershaper/jul10_search_results/gpu_params60000000.0_/best_configs_iter_299.csv", "c4_dir": dataset_factory("wikibooks")})}]}  ] )
# script_creator(get_experiments_dir() + "/jul11_wikibooks_efficient_subnet_train_more_steps_100K", [ {"exp_name": "seqlen_128", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supershaper.standard.train_mlm"), {"eval_random_subtransformers": 0, "max_train_steps": 100000, "sampling_type": "none", "sampling_rule": "none", "model_name_or_path": "/fsx/ganayu/experiments/supershaper/jul6_bertdata_bertbottleneck_128seqlen/c4_realnews_bert-bottleneck_random_K=1_pretraining_128seqlen_08-07-2022-06-40-22/best_model", "subtransformer_config_path": "/fsx/ganayu/experiments/supershaper/jul10_search_results/gpu_params60000000.0_/best_configs_iter_299.csv", "c4_dir": dataset_factory("wikibooks")})}]}  ] )
# script_creator(get_experiments_dir() + "/jul12_wikibooks_efficient_subnet_train_more_steps_finetune_9tasks", [ {"exp_name": "seqlen_128_%s"%TASK, "runs": [{"pyfile": "train_glue.py", "params": modify_config_and_to_string(config_factory("supershaper.standard.train_glue"), {"model_name_or_path": "/fsx/ganayu/experiments/supershaper/jul11_wikibooks_efficient_subnet_train_more_steps/seqlen_128/c4_realnews_bert-bottleneck_none_K=1_pretraining_seqlen_128_13-07-2022-00-00-19/best_model", "subtransformer_config_path": "/fsx/ganayu/experiments/supershaper/jul10_search_results/gpu_params60000000.0_/best_configs_iter_299.csv", "task": TASK})}]}  for TASK in get_glue_datasets() ] )
# script_creator(get_experiments_dir() + "/jul12_wikibooks_efficient_supernet_train_more_steps_finetune_9tasks", [ {"exp_name": "seqlen_128_%s"%TASK, "runs": [{"pyfile": "train_glue.py", "params": modify_config_and_to_string(config_factory("supershaper.standard.train_glue"), {"model_name_or_path": "/fsx/ganayu/experiments/supershaper/jul6_bertdata_bertbottleneck_128seqlen/c4_realnews_bert-bottleneck_random_K=1_pretraining_128seqlen_08-07-2022-06-40-22/best_model/best_model", "subtransformer_config_path": "/fsx/ganayu/experiments/supershaper/jul10_search_results/gpu_params60000000.0_/best_configs_iter_299.csv", "task": TASK})}]}  for TASK in get_glue_datasets() ] )
# script_creator(get_experiments_dir() + "/jul12_wikibooks_lr5e-5_finetune_9tasks", [ {"exp_name": "seqlen_128_%s"%TASK, "runs": [{"pyfile": "train_glue.py", "params": modify_config_and_to_string(config_factory("supershaper.standard.train_glue"), {"model_name_or_path": "/fsx/ganayu/experiments/supershaper/jul10_wikibooks_lr5e-5/seqlen_128/c4_realnews_bert-bottleneck_random_K=1_pretraining_seqlen_128_11-07-2022-05-19-41/best_model", "subtransformer_config_path": "/fsx/ganayu/experiments/supershaper/jul10_search_results/gpu_params60000000.0_/best_configs_iter_299.csv", "task": TASK})}]}  for TASK in get_glue_datasets() ] )
# script_creator(get_experiments_dir() + "/jul12_wikibooks_efficient_subnet_train_more_steps_100K_finetune_9tasks", [ {"exp_name": "seqlen_128_%s"%TASK, "runs": [{"pyfile": "train_glue.py", "params": modify_config_and_to_string(config_factory("supershaper.standard.train_glue"), {"model_name_or_path": "/fsx/ganayu/experiments/supershaper/jul11_wikibooks_efficient_subnet_train_more_steps_100K/seqlen_128/c4_realnews_bert-bottleneck_none_K=1_pretraining_seqlen_128_13-07-2022-00-07-28/best_model", "subtransformer_config_path": "/fsx/ganayu/experiments/supershaper/jul10_search_results/gpu_params60000000.0_/best_configs_iter_299.csv", "task": TASK})}]}  for TASK in get_glue_datasets() ] )

'''
phase-2
changes:
1) BERT lr 1e-4
2) max seq len 512
3) NO line by line & Merge sentences from different docs
4) max steps is 125K
5) validation set back to 5% (from 50K)
6) roberts' AdamW change B_2=0.98 for stability with bigger batches

train independent models
'''
# supernetbase - base, base_longer250K, base_lr6e-4
# script_creator(get_experiments_dir() + "/jul16_supernetbase_initial", [ {"exp_name": "base", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbase.standard.train_mlm"), {"experiment_name": "jul14_supernetbase_initial_base"})}]}  ]) #, {"exp_name": "base_longer250K", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbase.standard.train_mlm"), {"max_train_steps": 250000, "experiment_name": "jul14_supernetbase_initial_base_longer250K"})}]}, {"exp_name": "base_lr6e-4", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbase.standard.train_mlm"), {"learning_rate": 6e-4, "experiment_name": "jul14_supernetbase_initial_base_lr6e-4"})}]} ] )
# script_creator(get_experiments_dir() + "/jul17_supernetbase_initial_largebsz", [ {"exp_name": "base", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbase.standard.train_mlm.32bsz"), {"experiment_name": "jul17_supernetbase_initial_largebsz"})}]}  ]) 

# train largest transformer bert-base 768H 12L (110M) w bottleneck
# train smallest transformer bert-base 120H 12L w bottleneck
# script_creator(get_experiments_dir() + "/jul16_bertstandalone", [ {"exp_name": "12L_768H", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbase.standard.train_mlm"), {"sampling_type": "none", "sampling_rule": "none", "subtransformer_config_path": "/fsx/ganayu/experiments/supershaper/configs/bert/bertbase_12L_768H.csv"})}]}, {"exp_name": "12L_120H", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbase.standard.train_mlm"), {"sampling_type": "none", "sampling_rule": "none", "subtransformer_config_path": "/fsx/ganayu/experiments/supershaper/configs/bert/bertbase_12L_120H.csv"})}]}  ] )
# script_creator(get_experiments_dir() + "/jul17_bertstandalone_largebsz", [ {"exp_name": "12L_768H", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbase.standard.train_mlm.32bsz"), {"sampling_type": "none", "sampling_rule": "none", "subtransformer_config_path": "/fsx/ganayu/experiments/supershaper/configs/bert/bertbase_12L_768H.csv", "experiment_name": "jul17_bertstandalone_largebsz_12L_768H"})}]}, {"exp_name": "12L_120H", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbase.standard.train_mlm.32bsz"), {"sampling_type": "none", "sampling_rule": "none", "subtransformer_config_path": "/fsx/ganayu/experiments/supershaper/configs/bert/bertbase_12L_120H.csv", "experiment_name": "jul17_bertstandalone_largebsz_12L_120H"})}]}  ] )

# supernet - train with sample_one_arch=True
# script_creator(get_experiments_dir() + "/jul18_supernetbase_sample1arch", [ {"exp_name": "base", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbase.standard.train_mlm.32bsz"), {"experiment_name": "jul18_supernetbase_sample1arch", "sampling_rule": "none", "sample_one_arch": "yes" })}]}  ]) 

# finetuning on mnli - standalone vs supernet
def create_finetuning_experiments_standalone_vs_supernet(epochs_to_consider, supernet_master_dir, standalone_master_dir, configs=["bert.bottleneck.12L_120H", "bert.bottleneck.12L_768H"], tasks=["mnli"]):
    experiments = []
    for task in tasks:
        for epoch in epochs_to_consider:
            # supernet
            if supernet_master_dir:
                supernet_ckpt = get_experiments_dir() + "/%s/base/epoch_%d"%(supernet_master_dir, epoch)
                for model_name in configs:
                    config = get_model_configs(model_name)
                    experiment_name = "supernet_%s_ep%d_%s"%(task, epoch, model_name)
                    experiments.append({"exp_name": experiment_name, "experiment_name": experiment_name, "runs": [{"pyfile": "train_glue.py", "params": modify_config_and_to_string(config_factory("supernetbase.standard.train_glue"), {"model_name_or_path": supernet_ckpt, "subtransformer_config_path": config, "task": task })}]})
            # standalone
            if standalone_master_dir:
                for model_name in configs:
                    config = get_model_configs(model_name)
                    effnet_ckpt =  get_experiments_dir() + "/%s/%s/epoch_%d"%(standalone_master_dir, model_name.split(".")[-1], epoch)
                    experiment_name = "standalone_%s_ep%d_%s"%(task, epoch, model_name)
                    experiments.append({"exp_name": experiment_name, "experiment_name": experiment_name, "runs": [{"pyfile": "train_glue.py", "params": modify_config_and_to_string(config_factory("supernetbase.standard.train_glue"), {"model_name_or_path": effnet_ckpt, "subtransformer_config_path": config, "task": task})}]})
    return experiments
experiments = []
epochs_to_consider = [2]
for epoch in epochs_to_consider:
    # supernet
    supernet_ckpt = get_experiments_dir() + "/jul17_supernetbase_initial_largebsz/base/epoch_%d"%epoch
    for model_name, config in [ ("bert.bottleneck.12L_120H", get_model_configs("bert.bottleneck.12L_120H")), ("bert.bottleneck.12L_768H", get_model_configs("bert.bottleneck.12L_768H"))]:
        experiment_name = "supernet_mnli_ep%d_%s"%(epoch, model_name)
        experiments.append({"exp_name": experiment_name, "experiment_name": experiment_name, "runs": [{"pyfile": "train_glue.py", "params": modify_config_and_to_string(config_factory("supernetbase.standard.train_glue"), {"model_name_or_path": supernet_ckpt, "subtransformer_config_path": config, "task": "mnli"})}]})
    # standalone
    for model_name, config in [ ("bert.bottleneck.12L_120H", get_model_configs("bert.bottleneck.12L_120H")), ("bert.bottleneck.12L_768H", get_model_configs("bert.bottleneck.12L_768H"))]:
        effnet_ckpt =  get_experiments_dir() + "/jul17_bertstandalone_largebsz/%s/epoch_%d"%(model_name.split(".")[-1], epoch)
        experiment_name = "standalone_mnli_ep%d_%s"%(epoch, model_name)
        experiments.append({"exp_name": experiment_name, "experiment_name": experiment_name, "runs": [{"pyfile": "train_glue.py", "params": modify_config_and_to_string(config_factory("supernetbase.standard.train_glue"), {"model_name_or_path": effnet_ckpt, "subtransformer_config_path": config, "task": "mnli"})}]})
# experiments = create_finetuning_experiments_standalone_vs_supernet(epochs_to_consider=[2], supernet_master_dir="jul17_supernetbase_initial_largebsz", standalone_master_dir="jul17_bertstandalone_largebsz")
# script_creator(get_experiments_dir() + "/jul18_supernetbase_vs_standalone_finetune_mnli", experiments, time_in_mins=600, wandb="offline")

'''
phase-3
bert-base uncased
128 seq len
lr = 5e-4 (supershaper)
dataset - graphcore

todo:
mixing attention for standard bert model
turn off pretrained weights initialization
'''
# script_creator(get_experiments_dir() + "/jul19_v3_bertstandalone", [ {"exp_name": "12L_768H", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("bertbasev3.standard.train_mlm"), {"subtransformer_config_path": "/fsx/ganayu/experiments/supershaper/configs/bert/bertbase_uncased_12L_768H.csv", "experiment_name": "jul19_v3_bertstandalone_12L_768H"})}]}, {"exp_name": "12L_120H", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("bertbasev3.standard.train_mlm"), {"subtransformer_config_path": "/fsx/ganayu/experiments/supershaper/configs/bert/bertbase_uncased_12L_120H.csv", "experiment_name": "jul19_v3_bertstandalone_12L_120H"})}]}  ], time_in_mins=10000)
# script_creator(get_experiments_dir() + "/jul19_v3_supernetbase", [ {"exp_name": "base", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "jul19_v3_supernetbase"})}]}  ], time_in_mins=10000) 
# experiments = create_finetuning_experiments_standalone_vs_supernet([7], None, "/fsx/ganayu/experiments/supershaper/jul19_v3_bertstandalone", configs=["bertuncased.bottleneck.12L_120H", "bertuncased.bottleneck.12L_768H"], tasks=["mrpc"])
# script_creator(get_experiments_dir() + "/jul20_v3_standalone_finetune_mrpc", experiments, time_in_mins=120, wandb="offline")

# bert base 110M - standalone - no preinit - no bottleneck 
# script_creator(get_experiments_dir() + "/jul21_v3_bertstandalone_noinit_nobottleneck", [ {"exp_name": "12L_768H_5e-4", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("bertbasev3.standard.train_mlm"), {"subtransformer_config_path": get_model_configs("bertuncased.nobottleneck.12L_768H"), "initialize_pretrained_weights": "no", "mixing": "attention", "experiment_name": "jul21_v3_bertstandalone_noinit_nobottleneck_12L_768H_5e-4"})}]},  {"exp_name": "12L_768H_1e-3", "runs": [{"pyfile": "train_mlm.py", "params": # modify_config_and_to_string(config_factory("bertbasev3.standard.train_mlm"), {"subtransformer_config_path": get_model_configs("bertuncased.nobottleneck.12L_768H"), "initialize_pretrained_weights": "no", "mixing": "attention", "learning_rate": 1e-3, "experiment_name": "jul21_v3_bertstandalone_noinit_nobottleneck12L_768H_1e-3"})}]}  ], time_in_mins=5000, wandb="online")

# compute test PPL for baselines and ours.
# script_creator(get_experiments_dir() + "/jul21_v3_test_ppl_baselines", [ {"exp_name": exp_name, "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("bertbasev3.standard.train_mlm"), {"subtransformer_config_path": get_model_configs(model_config), "mixing": mixing, "experiment_name": "jul21_v3_%s"%exp_name, "tokenizer_name": tokenizer_name, "model_name_or_path": model_name_or_path, "check_test_loss_only": "yes"})}]} for exp_name, model_name_or_path, tokenizer_name, mixing, model_config in [("google-bert", "bert-base-uncased", "bert-base-uncased", "attention", "bertuncased.nobottleneck.12L_768H"), ("graphcore-bert", "Graphcore/bert-base-uncased", "Graphcore/bert-base-uncased", "attention", "bertuncased.nobottleneck.12L_768H"), ("ourbert-w-initbottle", "/fsx/ganayu/experiments/supershaper/jul19_v3_bertstandalone/12L_768H/best_model", "Graphcore/bert-base-uncased", "bert-bottleneck", "bertuncased.bottleneck.12L_768H")]], time_in_mins=300, wandb="offline", num_gpus=1)
# compute valid MNLI score for baseline and ours following RoBERTa sweep
def create_finetuning_experiments_standalone_vs_supernet_v2(models=[ ("graphcore-bert", "Graphcore/bert-base-uncased", "Graphcore/bert-base-uncased", "attention", "bertuncased.nobottleneck.12L_768H"), ("ourbert-w-initbottle", "/fsx/ganayu/experiments/supershaper/jul19_v3_bertstandalone/12L_768H/best_model", "Graphcore/bert-base-uncased", "bert-bottleneck", "bertuncased.bottleneck.12L_768H"), ("google-bert", "bert-base-uncased", "bert-base-uncased", "attention", "bertuncased.nobottleneck.12L_768H")], configs=["bert.bottleneck.12L_120H", "bert.bottleneck.12L_768H"], tasks=["mnli"], sweep={"ft_lrs":["1e-5", "2e-5", "3e-5"], "ft_bsz":["16", "32"], "ft_epochs":["10"]}, glue_config="supernetbasev3.standard.train_glue", num_gpus=1, search_space_id=None, is_mnli_checkpoint=0, ckpt_needed=[]):
    experiments = []
    for task in tasks:
        for lr in sweep["ft_lrs"]:
            for bsz in sweep["ft_bsz"]:
                for epoch in sweep["ft_epochs"]:
                    for model_info in models:
                        if isinstance(model_info, tuple):
                            exp_name, model_name_or_path, tokenizer_name, mixing, model_config = model_info
                            cur_exp_name = "%d_%s_%s_%s_%s_%s"%(len(experiments), exp_name, task, lr, bsz, epoch)
                        elif isinstance(model_info, dict):
                            exp_name, model_name_or_path, tokenizer_name, mixing = model_info["exp_name"], model_info["model_name_or_path"], model_info["tokenizer_name"], model_info["mixing"]
                            cur_exp_name = "%d_%s_%s_%s_%s_%s"%(len(experiments), exp_name, task, lr, bsz, epoch)
                        run_config = None
                        if glue_config.endswith("train_glue"):
                            run_config = {"exp_name": cur_exp_name, "experiment_name": cur_exp_name, "runs": [{"pyfile": "train_glue.py", "params": modify_config_and_to_string(config_factory(glue_config), {"model_name_or_path": model_name_or_path,  "task_name": task, "tokenizer_name":tokenizer_name, "mixing": mixing, "learning_rate": lr, "per_device_train_batch_size": int(bsz) // num_gpus, "num_train_epochs": epoch})}]}
                            if model_config != "none":
                                run_config["subtransformer_config_path"] = get_model_configs(model_config)
                        elif glue_config.endswith("train_glue_original"):
                            if model_config == "none":
                                run_config = {"exp_name": cur_exp_name, "experiment_name": cur_exp_name, "runs": [{"pyfile": "train_glue_original.py", "params": modify_config_and_to_string(config_factory(glue_config), {"model_name_or_path": model_name_or_path, "task_name": task, "tokenizer_name": tokenizer_name, "learning_rate": lr, "per_device_train_batch_size": int(bsz) // num_gpus, "num_train_epochs": epoch, "is_mnli_checkpoint": is_mnli_checkpoint, "skip_saving_checkpoints": "yes" if task not in ckpt_needed else "no"})}]}
                            else:
                                run_config = {"exp_name": cur_exp_name, "experiment_name": cur_exp_name, "runs": [{"pyfile": "train_glue_original.py", "params": modify_config_and_to_string(config_factory(glue_config), {"model_name_or_path": model_name_or_path, "task_name": task, "tokenizer_name": tokenizer_name, "learning_rate": lr, "subtransformer_config_path": model_config, "per_device_train_batch_size": int(bsz) // num_gpus, "num_train_epochs": epoch, "is_mnli_checkpoint": is_mnli_checkpoint, "skip_saving_checkpoints": "yes" if task not in ckpt_needed else "no"})}]}
                        elif glue_config.endswith("supernet_finetune"):
                            if "distill" in glue_config:
                                new_configs = {"model_name_or_path": model_name_or_path, "task_name": task, "tokenizer_name": tokenizer_name, "mixing": mixing, "learning_rate": lr, "per_device_train_batch_size": int(bsz) // num_gpus, "num_train_epochs": epoch, "is_mnli_checkpoint": is_mnli_checkpoint, "skip_saving_checkpoints": "yes" if task not in ckpt_needed else "no"}
                                for key in ["subtransformer_config_path", "teacher_model_path", "inplace_distillation", "distillation_type", "inplace_kd_hard_loss_contrib", "inplace_kd_layers", "inplace_kd_distill_loss_contrib"]:
                                    if key in model_info:
                                        new_configs[key] = model_info[key]
                                if search_space_id:
                                    new_configs["search_space_id"] = search_space_id
                                run_config = {"exp_name": cur_exp_name, "experiment_name": cur_exp_name, "runs": [{"pyfile": "supernet_finetune.py", "params": modify_config_and_to_string(config_factory(glue_config), new_configs)}]}
                            else:
                                run_config = {"exp_name": cur_exp_name, "experiment_name": cur_exp_name, "runs": [{"pyfile": "supernet_finetune.py", "params": modify_config_and_to_string(config_factory(glue_config), {"model_name_or_path": model_name_or_path, "task_name": task, "tokenizer_name": tokenizer_name, "mixing": mixing, "search_space_id": search_space_id, "learning_rate": lr, "per_device_train_batch_size": int(bsz) // num_gpus, "num_train_epochs": epoch, "skip_saving_checkpoints": "yes" if task not in ckpt_needed else "no"})}]}
                        assert(run_config!=None)
                        experiments.append(run_config)
    return experiments

# script_creator(get_experiments_dir() + "/jul21_v3_finetune_mnli_graphcore_vs_ourbert-w-initbottle", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=1), time_in_mins=1000, wandb="offline", num_gpus=1, generate_port=True)
# script_creator(get_experiments_dir() + "/jul21_v3_8gpu1setting_finetune_mnli_graphcore_vs_ourbert-w-initbottle", create_finetuning_experiments_standalone_vs_supernet_v2(sweep={"ft_lrs":["5e-5"], "ft_bsz":["32"], "ft_epochs":["10"]}, num_gpus=8), time_in_mins=1000, wandb="offline", num_gpus=8)
#script_creator(get_experiments_dir() + "/jul22_v3_finetune_mnli_ourbert-no-initbottle", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=1, models=[ ("ourbert-no-initbottle", "/fsx/ganayu/experiments/supershaper/jul21_v3_bertstandalone_noinit_nobottleneck/12L_768H_5e-4/best_model", "Graphcore/bert-base-uncased", "attention", "bertuncased.nobottleneck.12L_768H") ]), time_in_mins=1000, wandb="offline", num_gpus=1, generate_port=True)
# script_creator(get_experiments_dir() + "/jul22_v3_finetune_mnli_oursupernet_base", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=1, models=[ ("supernetbase-12L_768H", "/fsx/ganayu/experiments/supershaper/jul19_v3_supernetbase/base/best_model", "Graphcore/bert-base-uncased", "bert-bottleneck", "bertuncased.bottleneck.12L_768H") ]), time_in_mins=1000, wandb="offline", num_gpus=1, generate_port=True)
# script_creator(get_experiments_dir() + "/jul23_v3_finetunecorrected_mnli_allmodels_slide36", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=1, tasks=["mnli"], sweep=get_finetuning_sweeps("wellreadstudents"), models=[ ("google-bert", "bert-base-uncased", "bert-base-uncased", "attention", "none"), ("graphcore-bert", "Graphcore/bert-base-uncased", "Graphcore/bert-base-uncased", "attention", "none"), ("ourbert-w-initbottle", "/fsx/ganayu/experiments/supershaper/jul19_v3_bertstandalone/12L_768H/best_model", "Graphcore/bert-base-uncased", "bert-bottleneck", "none"), ("supernetbase-12L_768H", "/fsx/ganayu/experiments/supershaper/jul19_v3_supernetbase/base/best_model", "Graphcore/bert-base-uncased", "bert-bottleneck", "none"), ("ourbert-no-initbottle", "/fsx/ganayu/experiments/supershaper/jul21_v3_bertstandalone_noinit_nobottleneck/12L_768H_5e-4/best_model", "Graphcore/bert-base-uncased", "attention", "none")]), time_in_mins=1000, wandb="offline", num_gpus=1, generate_port=True, part_size=25)
# script_creator(get_experiments_dir() + "/jul23_v3_finetunecorrected_mnli_allmodels_slide36_supershapersweep", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=1, tasks=["mnli"], sweep=get_finetuning_sweeps("supershaper"), models=[ ("google-bert", "bert-base-uncased", "bert-base-uncased", "attention", "none"), ("graphcore-bert", "Graphcore/bert-base-uncased", "Graphcore/bert-base-uncased", "attention", "none"), ("ourbert-w-initbottle", "/fsx/ganayu/experiments/supershaper/jul19_v3_bertstandalone/12L_768H/best_model", "Graphcore/bert-base-uncased", "bert-bottleneck", "none"), ("supernetbase-12L_768H", "/fsx/ganayu/experiments/supershaper/jul19_v3_supernetbase/base/best_model", "Graphcore/bert-base-uncased", "bert-bottleneck", "none"), ("ourbert-no-initbottle", "/fsx/ganayu/experiments/supershaper/jul21_v3_bertstandalone_noinit_nobottleneck/12L_768H_5e-4/best_model", "Graphcore/bert-base-uncased", "attention", "none")]), time_in_mins=1000, wandb="offline", num_gpus=1, generate_port=True, part_size=25)

'''
# bringing back "token_type_ids" and using "train_glue_original"
'''
# script_creator(get_experiments_dir() + "/jul23_v3_bertstandalone_noinit_nobottleneck_toktypecorrected", [ {"exp_name": "12L_768H_5e-4", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("bertbasev3.standard.train_mlm"), {"subtransformer_config_path": get_model_configs("bertuncased.nobottleneck.12L_768H"), "initialize_pretrained_weights": "no", "mixing": "attention", "experiment_name": "jul23_v3_bertstandalone_noinit_nobottleneck_12L_768H_5e-4_toktypecorrected"})}]}], time_in_mins=5000, wandb="online")
# script_creator(get_experiments_dir() + "/jul23_v3_supernetbase_toktypecorrected", [ {"exp_name": "base", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "jul23_v3_supernetbase_toktypecorrected"})}]}  ], time_in_mins=10000, wandb="online") 
# script_creator(get_experiments_dir() + "/jul23_v3_bertstandalone_winit_wbottleneck_toktypecorrected", [ {"exp_name": "12L_768H", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("bertbasev3.standard.train_mlm"), {"subtransformer_config_path": "/fsx/ganayu/experiments/supershaper/configs/bert/bertbase_uncased_12L_768H.csv", "experiment_name": "jul23_v3_bertstandalone_winit_wbottleneck_12L_768H_toktypecorrected"})}]}, {"exp_name": "12L_120H", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("bertbasev3.standard.train_mlm"), {"subtransformer_config_path": "/fsx/ganayu/experiments/supershaper/configs/bert/bertbase_uncased_12L_120H.csv", "experiment_name": "jul23_v3_bertstandalone__winit_wbottleneck_12L_120H_toktypecorrected"})}]}  ], time_in_mins=10000)
# script_creator(get_experiments_dir() + "/jul23_v3_supernetbase_toktypecorrected_inplacekd_logits_only", [ {"exp_name": "base", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "jul23_v3_supernetbase_toktypecorrected_inplacekd_logits_only", "inplace_distillation": 1})}]}  ], time_in_mins=10000, wandb="online") 
# script_creator(get_experiments_dir() + "/jul23_v3_supernetbase_toktypecorrected_inplacekd_logits_last_attn_hid_layer", [ {"exp_name": "base", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "jul23_v3_supernetbase_toktypecorrected_inplacekd_logits_last_attn_hid_layer", "inplace_distillation": 1, "distillation_type": "hiddenlastlayer+attentionlastlayer"})}]}  ], time_in_mins=10000, wandb="online") 
# script_creator(get_experiments_dir() + "/jul23_v3_supernetbase_toktypecorrected_2xtrainbudget", [ {"exp_name": "base", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "jul23_v3_supernetbase_toktypecorrected_2xtrainbudget", "max_train_steps": 250000})}]} ], time_in_mins=10000, wandb="online") 
# script_creator(get_experiments_dir() + "/jul23_v3_supernetbase_toktypecorrected_noinit", [ {"exp_name": "base", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "jul23_v3_supernetbase_toktypecorrected_noinit", "initialize_pretrained_weights": "no"})}]} ], time_in_mins=8000, wandb="online") 


# fine-tuning - reproduce standalone baseline? can we?
# script_creator(get_experiments_dir() + "/jul24_v3_finetune_glueoriginal_mnli_toktypecorrected", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=1, tasks=["mnli"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("wellreadstudents"), models=[ ("google-bert", "bert-base-uncased", "bert-base-uncased", "attention", "none"), ("graphcore-bert", "Graphcore/bert-base-uncased", "Graphcore/bert-base-uncased", "attention", "none"), ("ourbert-no-initbottle", "/fsx/ganayu/experiments/supershaper/jul23_v3_bertstandalone_noinit_nobottleneck_toktypecorrected/12L_768H_5e-4/best_model", "Graphcore/bert-base-uncased", "attention", "none")]), time_in_mins=1000, wandb="no", num_gpus=1, generate_port=True, part_size=25)
# script_creator(get_experiments_dir() + "/jul24_v3_finetune_glueoriginal_sst2_cola_toktypecorrected", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=1, tasks=["sst2", "cola"], glue_config="supernetbasev3.standard.nockptsavings.train_glue_original", sweep=get_finetuning_sweeps("wellreadstudents"), models=[ ("google-bert", "bert-base-uncased", "bert-base-uncased", "attention", "none"), ("graphcore-bert", "Graphcore/bert-base-uncased", "Graphcore/bert-base-uncased", "attention", "none"), ("ourbert-no-initbottle", "/fsx/ganayu/experiments/supershaper/jul23_v3_bertstandalone_noinit_nobottleneck_toktypecorrected/12L_768H_5e-4/best_model", "Graphcore/bert-base-uncased", "attention", "none")]), time_in_mins=1000, wandb="no", num_gpus=1, generate_port=True, part_size=25)

# changing search space
# script_creator(get_experiments_dir() + "/jul24_v3_supernetbase_toktypecorrected_extending_search_space", [ {"exp_name": "hidden_layer_elastic", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "jul24_v3_supernetbase_toktypecorrected_hidden_layer_elastic", "search_space_id": "hidden_layer_elastic"})}]}, {"exp_name": "ffn_intermediate_elastic", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "jul24_v3_supernetbase_toktypecorrected_ffn_intermediate_elastic", "search_space_id": "ffn_intermediate_elastic"})}]} ], time_in_mins=8000, wandb="online")

# 2 samples during sandwich rule
# script_creator(get_experiments_dir() + "/jul25_v3_supernetbase_sandwch_2random", [ {"exp_name": "base", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "jul23_v3_supernetbase_toktypecorrected", "pop_size": 2})}]} ], time_in_mins=10000, wandb="online") 

# inplace kd loss
# script_creator(get_experiments_dir() + "/jul27_v3_supernetbase_inplacekd_logitshard_logitshidden", [ {"exp_name": "logits", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "jul27_v3_supernetbase_toktypecorrected_inplacekd_logitshard", "inplace_distillation": 1, "distillation_type": "logits+hard"})}]}, {"exp_name": "logitshidden", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "jul27_v3_supernetbase_toktypecorrected_inplacekd_logits_hiddenlastlayer", "inplace_distillation": 1, "distillation_type": "logits+hiddenlastlayer"})}]}  ], time_in_mins=10000, wandb="online")

# hypernet-rank
# script_creator(get_experiments_dir() + "/jul27_v3_supernetbase_hypernet", [ {"exp_name": "rank%s_hyphid%s"%(rank, hyphid), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "jul27_v3_supernetbase_hypernet_rank%s_hyphid%s"%(rank, hyphid), "use_hypernet_w_low_rank": 1, "bottleneck_rank": rank, "hypernet_hidden_size": hyphid })}]}  for rank, hyphid in [("64", "50"), ("64", "16"), ("32", "16")] ], time_in_mins=10000, wandb="online")

# search space -> FFN expansion ratio
# search space -> v2 -> Intermediate expansion ratio - [2, 3, 4]; Number of layers - [6, 9, 12]; Hidden size - [120, 360, 540, 768]
# search space ->  hidden_layer_elastic corrected with val. smallest equal to v2
# script_creator(get_experiments_dir() + "/jul27_v3_supernetbase_search_space_ffn_v2_hidlayer", [ {"exp_name": search_space_id, "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "jul27_v3_supernetbase_search_space_%s"%search_space_id, "search_space_id": search_space_id})}]}  for search_space_id in ["hidden_layer_elastic", "ffn_intermediate_ratio_elastic", "v2"] ], time_in_mins=10000, wandb="online")

# inplace-kd -> mult distill loss by 1.0, 1.25 and 1.5
# script_creator(get_experiments_dir() + "/jul27_v3_supernetbase_inplacekd_distill_contrib", [ {"exp_name": inplace_kd_distill_loss_weights, "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "jul27_v3_supernetbase_inplacekd_distill_contrib_%s"%inplace_kd_distill_loss_weights, "inplace_distillation": 1, "distillation_type": "logits", "inplace_kd_distill_loss_weights": inplace_kd_distill_loss_weights})}]} for inplace_kd_distill_loss_weights in ["1.0", "1.5", "2"] ], time_in_mins=10000, wandb="online")

# playing with batch size
# script_creator(get_experiments_dir() + "/jul30_v3_supernetbase_nasbert_bsz_1024_250Ksteps", [ {"exp_name": "base", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "jul30_v3_supernetbase_nasbert_bsz_1024_250Ksteps", "max_train_steps": 250000, "gradient_accumulation_steps": 1})}]}  ], time_in_mins=10000, wandb="online")

# search - different spaces
# script_creator(get_experiments_dir() + "/aug1_v3_supernetbase_search_different_spaces", [ {"exp_name": "hiddenonly", "runs": [{"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/jul23_v3_supernetbase_toktypecorrected/base/best_model"})}]},   {"exp_name": "hidden_layer_elastic", "runs": [{"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/jul27_v3_supernetbase_search_space_ffn_v2_hidlayer/hidden_layer_elastic/best_model", "search_space_id": "hidden_layer_elastic"})}]},    {"exp_name": "ffn_intermediate_ratio_elastic", "runs": [{"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/jul27_v3_supernetbase_search_space_ffn_v2_hidlayer/ffn_intermediate_ratio_elastic/best_model", "search_space_id": "ffn_intermediate_ratio_elastic"})}]},  {"exp_name": "v2", "runs": [{"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/jul27_v3_supernetbase_search_space_ffn_v2_hidlayer/v2/best_model", "search_space_id": "v2"})}]}, ], time_in_mins=8000, wandb="no")

# script_creator(get_experiments_dir() + "/aug1_v3_supernetbase_search_diff_configs", [ {"exp_name": "2xtrainbudget", "runs": [{"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/jul23_v3_supernetbase_toktypecorrected_2xtrainbudget/base/best_model"})}]}, {"exp_name": "noinit", "runs": [{"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/jul23_v3_supernetbase_toktypecorrected_noinit/base/best_model"})}]},  {"exp_name": "sandwch_2random", "runs": [{"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/jul25_v3_supernetbase_sandwch_2random/base/best_model"})}]}   ], time_in_mins=8000, wandb="no")

# inplace search
# script_creator(get_experiments_dir() + "/aug1_v3_supernetbase_search_inplace_configs", [ {"exp_name": "logitshard", "runs": [{"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/jul27_v3_supernetbase_inplacekd_logitshard_logitshidden/logits/best_model"})}]}  ], time_in_mins=8000, wandb="no")

# train for more steps
# script_creator(get_experiments_dir() + "/aug1_v3_evosearch_hiddenlayeronly_subnet_train_more_steps", [ {"exp_name": "stage1", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug1_v3_evosearch_hiddenlayeronly_subnet_train_more_steps_stage1", "max_train_steps": 100000, "sampling_type": "none", "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/jul23_v3_supernetbase_toktypecorrected/base/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug1_v3_supernetbase_search_different_spaces/hiddenonly/evo_results_29.xls" })}]},  {"exp_name": "nowarmup", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug1_v3_evosearch_hiddenlayeronly_subnet_train_more_steps_nowarmup", "max_train_steps": 100000, "num_warmup_steps": 0, "sampling_type": "none", "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/jul23_v3_supernetbase_toktypecorrected/base/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug1_v3_supernetbase_search_different_spaces/hiddenonly/evo_results_29.xls" })}]},   {"exp_name": "5timeslowerlr", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug1_v3_evosearch_hiddenlayeronly_subnet_train_more_steps_5timeslowerlr", "max_train_steps": 100000, "learning_rate": 1e-05, "sampling_type": "none", "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/jul23_v3_supernetbase_toktypecorrected/base/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug1_v3_supernetbase_search_different_spaces/hiddenonly/evo_results_29.xls" })}]}  ], time_in_mins=8000, wandb="online") 

# direct finetuning
def create_finetuning_experiments_standalone_vs_supernet_v3(models=None, tasks=["mnli"], sweep=None, glue_config=None, num_gpus=4, ckpt_needed=[], is_mnli_checkpoint=0, all_experts_average=0, last_expert_averaging_expert="no", use_expert_id=0, kd_teacher=None, kd_config=None, seed=None, mnli_ckpt=None, collapse_experts_before_ft=0):
    experiments = []
    if kd_teacher is not None:
        mnli_ckpt_folder = "/fsx/ganayu/experiments/supershaper/sep23_mnlikd_models_master/%s"%(str(uuid.uuid4())) if mnli_ckpt is None else mnli_ckpt
    for model_info in models:
        search_space_id = None # not using currently
        if isinstance(model_info, dict):
            exp_name, model_name_or_path, tokenizer_name, mixing = model_info["exp_name"], model_info["model_name_or_path"], model_info["tokenizer_name"], model_info["mixing"]
        elif len(model_info) == 4:
            exp_name, model_name_or_path, subtransformer_config_path, tokenizer_name = model_info
        elif len(model_info) == 5:
            exp_name, model_name_or_path, subtransformer_config_path, tokenizer_name, search_space_id = model_info
        run_config = {"exp_name": exp_name, "runs": []}
        for task in tasks:
            for lr in sweep["ft_lrs"]:
                for bsz in sweep["ft_bsz"]:
                    for epoch in sweep["ft_epochs"]:
                        if glue_config.endswith("supernet_finetune"):
                            if "distill" in glue_config:
                                updated_params_config = {"model_name_or_path": model_name_or_path, "task_name": task, "tokenizer_name": tokenizer_name, "mixing": mixing, "learning_rate": lr, "per_device_train_batch_size": int(bsz) // num_gpus, "num_train_epochs": epoch, "skip_saving_checkpoints": "yes" if task not in ckpt_needed else "no"}
                                for key in ["subtransformer_config_path", "teacher_model_path", "inplace_distillation", "distillation_type", "inplace_kd_hard_loss_contrib", "inplace_kd_layers", "inplace_kd_distill_loss_contrib", "search_space_id"]:
                                    if key in model_info:
                                        updated_params_config[key] = model_info[key]  
                                if collapse_experts_before_ft == 1:
                                    updated_params_config["collapse_experts_before_ft"] = 1
                                if seed is not None:
                                    updated_params_config["seed"] = seed
                                if kd_teacher is not None and task in kd_teacher:
                                    updated_params_config["teacher_model_path"] = kd_teacher[task]
                                    for param in kd_config:
                                        updated_params_config[param] = kd_config[param]
                                    # for task in ["mrpc", "rte"]: # , "stsb"
                                    #    updated_params_config["is_mnli_checkpoint"] = 1
                                    updated_params_config["output_dir"] = mnli_ckpt_folder
                                    if task in ["mrpc", "rte"]:
                                        updated_params_config["model_name_or_path"] = mnli_ckpt_folder + "/mnli_best"
                                        updated_params_config["is_mnli_checkpoint"] = 1
                                run_info = { "pyfile": "supernet_finetune.py", "params": modify_config_and_to_string(config_factory(glue_config), updated_params_config) }
                                run_config["runs"].append(run_info)
                        else:
                            updated_params_config = {"model_name_or_path": model_name_or_path, "task_name": task, "tokenizer_name": tokenizer_name, "learning_rate": lr, "per_device_train_batch_size": int(bsz) // num_gpus, "num_train_epochs": epoch, "skip_saving_checkpoints": "yes" if task not in ckpt_needed else "no", "all_experts_average": all_experts_average, "last_expert_averaging_expert": last_expert_averaging_expert, "use_expert_id": use_expert_id}
                            if subtransformer_config_path:
                                assert(os.path.exists(subtransformer_config_path))
                                updated_params_config["subtransformer_config_path"] = subtransformer_config_path
                            for param in sweep:
                                if not param.startswith("ft"):
                                    updated_params_config[param] = sweep[param]
                            if seed is not None:
                                updated_params_config["seed"] = seed
                            run_info = {"pyfile": "train_glue_original.py", "params": modify_config_and_to_string(config_factory(glue_config), updated_params_config) }
                            run_config["runs"].append(run_info)
        experiments.append(run_config)
    return experiments

models = []
# exp_name, model_name_or_path, subtransformer_config_path, tokenizer_name
models.append(("supershaper", "/fsx/ganayu/experiments/supershaper/jul23_v3_supernetbase_toktypecorrected/base/best_model", "/fsx/ganayu/experiments/supershaper/aug1_v3_supernetbase_search_different_spaces/hiddenonly/evo_results_29.xls", "Graphcore/bert-base-uncased"))
models.append(("2xtrainbudget", "/fsx/ganayu/experiments/supershaper/jul23_v3_supernetbase_toktypecorrected_2xtrainbudget/base/best_model", "/fsx/ganayu/experiments/supershaper/aug1_v3_supernetbase_search_diff_configs/2xtrainbudget/evo_results_29.xls", "Graphcore/bert-base-uncased"))
models.append(("sandwch_2random", "/fsx/ganayu/experiments/supershaper/jul25_v3_supernetbase_sandwch_2random/base/best_model", "/fsx/ganayu/experiments/supershaper/aug1_v3_supernetbase_search_diff_configs/sandwch_2random/evo_results_29.xls", "Graphcore/bert-base-uncased"))
# for sweep in ["roberta", "bert", "wellreadstudents", "supershaper", "summer"]:
#    experiments = create_finetuning_experiments_standalone_vs_supernet_v3(models=[models[0]], sweep=get_finetuning_sweeps(sweep), glue_config="supernetbasev3.standard.train_glue_original", tasks=["mrpc", "cola"], num_gpus=4)
#    script_creator(get_experiments_dir() + "/aug2_supershaper_directfinetune_sweepcheck_%s_mrpc_cola"%(sweep), experiments, time_in_mins=10000, wandb="no", num_gpus=4, generate_port=True, part_size=25)
# experiments = create_finetuning_experiments_standalone_vs_supernet_v3(models=models, sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.train_glue_original", tasks=["mnli", "cola", "mrpc", "sst2"], num_gpus=8)
# script_creator(get_experiments_dir() + "/aug2_directfinetune_mnli_supershaper_2xtrainbudget_sandwch_2random", experiments, time_in_mins=10000, wandb="no", num_gpus=8, generate_port=True, part_size=25)

# directfinetuning - one
# script_creator(get_experiments_dir() + "/aug2_v3_finetune_supershaper_60M_direct", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("supershaper-60M-direct", "/fsx/ganayu/experiments/supershaper/jul23_v3_supernetbase_toktypecorrected/base/best_model", "Graphcore/bert-base-uncased", None, "/fsx/ganayu/experiments/supershaper/aug1_v3_supernetbase_search_different_spaces/hiddenonly/evo_results_29.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)
# script_creator(get_experiments_dir() + "/aug2_v3_finetune_v2_60M_direct", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("v2-60M-direct", get_experiments_dir() + "/jul27_v3_supernetbase_search_space_ffn_v2_hidlayer/v2/best_model", "Graphcore/bert-base-uncased", None, get_experiments_dir() + "/aug1_v3_supernetbase_search_different_spaces/v2/evo_results_29.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)
# 100Kfinetuning - one
# script_creator(get_experiments_dir() + "/aug2_v3_finetune_supershaper_60M_100Ksteps", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("supershaper-60M-100Ksteps", "/fsx/ganayu/experiments/supershaper/aug1_v3_evosearch_hiddenlayeronly_subnet_train_more_steps/nowarmup/best_model", "Graphcore/bert-base-uncased", None, "/fsx/ganayu/experiments/supershaper/aug1_v3_supernetbase_search_different_spaces/hiddenonly/evo_results_29.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)

# continue pretraining - 2xtrainbudget, v2
# script_creator(get_experiments_dir() + "/aug1_v3_evosearch_v2_2xbudget_subnet_train_more_steps", [ {"exp_name": "v2", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug1_v3_evosearch_v2_2xbudget_subnet_train_more_steps_v2", "max_train_steps": 100000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/jul27_v3_supernetbase_search_space_ffn_v2_hidlayer/v2/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug1_v3_supernetbase_search_different_spaces/v2/evo_results_29.xls" })}]},  {"exp_name": "2xbudget", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug1_v3_evosearch_hiddenlayeronly_subnet_train_more_steps_2xbudget", "max_train_steps": 100000, "num_warmup_steps": 0, "sampling_type": "none", "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/jul23_v3_supernetbase_toktypecorrected_2xtrainbudget/base/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug1_v3_supernetbase_search_diff_configs/2xtrainbudget/evo_results_29.xls" })}]}  ], time_in_mins=8000, wandb="online") 
# finetune the above two
# script_creator(get_experiments_dir() + "/aug4_v3_finetune_v2_2xbudget_60M_100Ksteps", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("v2", get_experiments_dir() + "/aug1_v3_evosearch_v2_2xbudget_subnet_train_more_steps/v2/best_model", "Graphcore/bert-base-uncased", None, get_experiments_dir() + "/aug1_v3_supernetbase_search_different_spaces/v2/evo_results_29.xls"), ("2xbudget", get_experiments_dir() + "/aug1_v3_evosearch_v2_2xbudget_subnet_train_more_steps/2xbudget/best_model", "Graphcore/bert-base-uncased", None, get_experiments_dir() + "/aug1_v3_supernetbase_search_diff_configs/2xtrainbudget/evo_results_29.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)

# continue pretraining - sandwich2rand
# script_creator(get_experiments_dir() + "/aug4_v3_sandwch_2rand_subnet_train_more_steps", [ {"exp_name": "sandwch_2rand", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug4_v3_sandwch_2rand_subnet_train_more_steps_sandwch_2rand", "max_train_steps": 100000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/jul25_v3_supernetbase_sandwch_2random/base/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug1_v3_supernetbase_search_diff_configs/sandwch_2random/evo_results_29.xls" })}]}], time_in_mins=8000, wandb="online")
# script_creator(get_experiments_dir() + "/aug5_v3_finetune_sandwch_2rand_60M_100Ksteps", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("sandwch_2rand", get_experiments_dir() + "/aug4_v3_sandwch_2rand_subnet_train_more_steps/sandwch_2rand/best_model", "Graphcore/bert-base-uncased", None, get_experiments_dir() + "/aug1_v3_supernetbase_search_diff_configs/sandwch_2random/evo_results_29.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)

# v1 - other tasks - cola, mrpc, sst2
# script_creator(get_experiments_dir() + "/aug5_v3_finetune_v1_60M_100Ksteps_cola_mrpc_sst2", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["cola", "mrpc", "sst2"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("v1", get_experiments_dir() + "/aug1_v3_evosearch_hiddenlayeronly_subnet_train_more_steps/nowarmup/best_model", "Graphcore/bert-base-uncased", None, get_experiments_dir() + "/aug1_v3_supernetbase_search_different_spaces/hiddenonly/evo_results_29.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)
# 2x budget - other tasks - cola, mrpc, sst2
# script_creator(get_experiments_dir() + "/aug5_v3_finetune_2xbudget_60M_100Ksteps_cola_mrpc_sst2", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["cola", "mrpc", "sst2"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("2xtrainbudget", get_experiments_dir() + "/jul23_v3_supernetbase_toktypecorrected_2xtrainbudget/base/best_model", "Graphcore/bert-base-uncased", None, get_experiments_dir() + "/aug1_v3_supernetbase_search_diff_configs/2xtrainbudget/evo_results_29.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)
# v2 - other tasks - cola, mrpc, sst2
# script_creator(get_experiments_dir() + "/aug6_finetune_v2_60M_directFT_cola_mrpc_sst2", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["cola", "mrpc", "sst2"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("v2", get_experiments_dir() + "/jul27_v3_supernetbase_search_space_ffn_v2_hidlayer/v2/best_model", "Graphcore/bert-base-uncased", None, get_experiments_dir() + "/aug1_v3_supernetbase_search_different_spaces/v2/evo_results_29.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)

# v1 - search again - duplicate issue resolved
# script_creator(get_experiments_dir() + "/aug6_v3_supernetbase_search_v1_duplissueresolved", [ {"exp_name": "hiddenonly", "runs": [{"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/jul23_v3_supernetbase_toktypecorrected/base/best_model"})}]}  ], time_in_mins=8000, wandb="no")
# script_creator(get_experiments_dir() + "/aug8_finetune_search_v1_duplissueresolved_mnli", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("logitshard", get_experiments_dir() + "/jul23_v3_supernetbase_toktypecorrected/base/best_model", "Graphcore/bert-base-uncased", None, get_experiments_dir() + "/aug6_v3_supernetbase_search_v1_duplissueresolved/hiddenonly/evo_results_29.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)

# inplace kd -> grad_accum_steps/2 fix
# script_creator(get_experiments_dir() + "/aug6_v3_supernetbase_inplacekd_logitshard_hiddenhard", [ {"exp_name": "logitshard", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug6_v3_inplacekd_logitshard", "inplace_distillation": 1, "distillation_type": "hard+logits"})}]}, {"exp_name": "hiddenhard", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug6_v3_inplacekd_hiddenhard", "inplace_distillation": 1, "distillation_type": "hard+hiddenlastlayer"})}]}, {"exp_name": "attentionhard", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug6_v3_inplacekd_attentionhard", "inplace_distillation": 1, "distillation_type": "hard+attentionlastlayer"})}]}  ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/aug8_finetune_inplacekd_logitshard_mnli", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("logitshard", get_experiments_dir() + "/aug6_v3_supernetbase_inplacekd_logitshard_hiddenhard/logitshard/best_model", "Graphcore/bert-base-uncased", None, get_experiments_dir() + "/aug1_v3_supernetbase_search_different_spaces/hiddenonly/evo_results_29.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)

# 100K steps -> continue pretrain with hard + soft loss
# script_creator(get_experiments_dir() + "/aug6_v3_evosearch_v1_subnet_train_100Ksteps_logitshard_logitsonly", [ {"exp_name": "logitshard", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug6_v3_evosearch_v1_subnet_train_100Ksteps_logitshard", "inplace_distillation": 1, "distillation_type": "hard+logits", "freeze_largest_model": "yes", "freeze_smallest_model": "yes", "max_train_steps": 100000, "num_warmup_steps": 0, "sampling_type": "none", "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/jul23_v3_supernetbase_toktypecorrected/base/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug1_v3_supernetbase_search_different_spaces/hiddenonly/evo_results_29.xls" })}]}, {"exp_name": "logits", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug6_v3_evosearch_v1_subnet_train_100Ksteps_logits", "inplace_distillation": 1, "distillation_type": "logits", "freeze_largest_model": "yes", "freeze_smallest_model": "yes", "max_train_steps": 100000, "num_warmup_steps": 0, "sampling_type": "none", "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/jul23_v3_supernetbase_toktypecorrected/base/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug1_v3_supernetbase_search_different_spaces/hiddenonly/evo_results_29.xls" })}]} ], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)
# script_creator(get_experiments_dir() + "/aug7_finetune_v1_60M_100Ksteps_logitshard_mnli", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("v1", get_experiments_dir() + "/aug6_v3_evosearch_v1_subnet_train_100Ksteps_logitshard_logitsonly/logitshard/best_model", "Graphcore/bert-base-uncased", None, get_experiments_dir() + "/aug1_v3_supernetbase_search_different_spaces/hiddenonly/evo_results_29.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)

# v3 run 
# v4 Linear -> relu -> Linear -> Normal Convnet -> relu  (not good supernet)
# script_creator(get_experiments_dir() + "/aug6_supernetbase_v3_v4_space", [ {"exp_name": "v3", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug6_supernetbase_v3space", "search_space_id": "v3", "mixing": "attention"})}]}, {"exp_name": "v4", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug6_supernetbase_v4space", "search_space_id": "v4", "mixing": "attention"})}]}  ], time_in_mins=10000, wandb="online", generate_port=True)
# 4p2 70 hr training time; Linear -> relu -> depthwise Convnet -> relu -> Linear  (Bad supernet)
# script_creator(get_experiments_dir() + "/aug7_supernetbase_v4_conv1dfix", [ {"exp_name": "v4", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug6_supernetbase_v4space", "search_space_id": "v4", "mixing": "attention"})}]}  ], time_in_mins=10000, wandb="online", generate_port=True)
# 4p3  Linear -> relu -> Linear -> Depthwise Convnet
# script_creator(get_experiments_dir() + "/aug9_supernetbase_v4p3", [ {"exp_name": "v4p3", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug6_supernetbase_v4p3space", "search_space_id": "v4", "mixing": "attention"})}]}  ], time_in_mins=10000, wandb="online", generate_port=True)
# script_creator(get_experiments_dir() + "/aug13_supernetbase_v4p5", [ {"exp_name": "v4p5", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug13_supernetbase_v4p5", "search_space_id": "v4.5", "mixing": "attention"})}]}  ], time_in_mins=10000, wandb="online", generate_port=True)

# bert & roberta standalone - academic bert preprocessing
# script_creator(get_experiments_dir() + "/aug9_bert_roberta_standalone_noinit_nobottle_acabertpreproc", [ {"exp_name": "bertbase", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("bertbasev3.standard.train_mlm"), {"subtransformer_config_path": get_model_configs("bertuncased.nobottleneck.12L_768H"), "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "initialize_pretrained_weights": "no", "mixing": "attention", "experiment_name": "bertbase"})}]},   {"exp_name": "robertabase", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("bertbasev3.standard.train_mlm"), {"subtransformer_config_path": get_model_configs("robertabase.nobottleneck.12L_768H"), "tokenized_c4_dir": dataset_factory("wikibooks_acabert_robertabase_128len"), "tokenizer_name": "roberta-base", "model_name_or_path": "roberta-base", "initialize_pretrained_weights": "no", "mixing": "attention", "experiment_name": "robertabase"})}]}  ], time_in_mins=8000, wandb="online", wandb_entity="ganayu", wandb_project="effbert", num_gpus=8, generate_port=True)
# finetune direct - bert
# script_creator(get_experiments_dir() + "/aug10_finetune_acabertdata_bertbasestandalone", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli", "cola", "mrpc", "sst2"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("bertbase", get_experiments_dir() + "/aug9_bert_roberta_standalone_noinit_nobottle_acabertpreproc/bertbase/best_model", "bert-base-uncased", None, "none")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)
# script_creator(get_experiments_dir() + "/aug10_finetune_acabertdata_robertbasestandalone", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli", "cola", "mrpc", "sst2"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("robertabase", get_experiments_dir() + "/aug9_bert_roberta_standalone_noinit_nobottle_acabertpreproc/robertabase/best_model", "roberta-base", None, "none")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)

# search - v3, v4.1
# script_creator(get_experiments_dir() + "/aug9_evosearch_v3_valppl", [ {"exp_name": "v3search", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/aug6_supernetbase_v3_v4_space/v3/best_model", "mixing": "attention", "search_space_id": "v3"})}]} ], time_in_mins=8000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/aug9_evosearch_v4p1_valppl", [ {"exp_name": "v4p1search", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/aug6_supernetbase_v3_v4_space/v4/best_model", "mixing": "attention", "search_space_id": "v4.1"})}]} ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/aug11_evosearch_supernetbase_v4_conv1dfix", [ {"exp_name": "v4_conv1dfix", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/aug7_supernetbase_v4_conv1dfix/v4/best_model", "mixing": "attention", "search_space_id": "v4.2"})}]} ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/aug11_evosearch_supernetbase_v4p3", [ {"exp_name": "v4p3", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/aug9_supernetbase_v4p3/v4p3/best_model", "mixing": "attention", "search_space_id": "v4.3"})}]} ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/aug12_directfinetune_v4_conv1dfix", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli", "cola"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("v4_conv1dfix", get_experiments_dir() + "/aug7_supernetbase_v4_conv1dfix/v4/best_model", "bert-base-uncased", None, get_experiments_dir() + "/aug11_evosearch_supernetbase_v4_conv1dfix/v4_conv1dfix/evo_results_9.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)

# supernetbase - acadbert dataset
# script_creator(get_experiments_dir() + "/aug9_acadbertdata_supernet_v1", [ {"exp_name": "acav1", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug9_acadbertdata_supernet_v1", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased"})}]}  ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1", [ {"exp_name": "v1search", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "mixing": "bert-bottleneck", "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len")})}]} ], time_in_mins=8000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/aug12_directfinetune_v1_acabert_evo15", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli", "cola"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("v1", get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "bert-base-uncased", None, get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)
# continue pretrain
# script_creator(get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps", [ {"exp_name": "supernet_continue", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_supernet_continue", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased" })}]}, {"exp_name": "retrain_scratch", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("bertbasev3.standard.train_mlm"), {"subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "initialize_pretrained_weights": "no", "mixing": "bert-bottleneck", "experiment_name": "aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_retrain_scratch"})}]}], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)
# script_creator(get_experiments_dir() + "/aug13_finetune_acadbertdata_supernet_retrain_subnet_125Ksteps_retrain_scratch", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli", "cola"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("v1", get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/retrain_scratch/best_model", "bert-base-uncased", None, get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)
# script_creator(get_experiments_dir() + "/aug13_finetune_acadbertdata_supernet_retrain_subnet_125Ksteps_supernet_continue", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli", "cola"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("v1", get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/supernet_continue/best_model", "bert-base-uncased", None, get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)
# get ckpt for mnli
# script_creator(get_experiments_dir() + "/aug13_finetune_acadbertdata_supernet_retrain_subnet_125Ksteps_supernet_continue_mnli_bestckpt", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=8, tasks=["mnli"], glue_config="supernetbasev3.standard.ckpt_needed.train_glue_original", sweep={"ft_lrs":["3e-5"], "ft_bsz":["32"], "ft_epochs":["2"]}, models=[ ("v1", get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/supernet_continue/best_model", "bert-base-uncased", None, get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls")]), time_in_mins=1000, wandb="no", num_gpus=8, generate_port=True, part_size=5)
# use mnli ckpt for finetune rte, stsb, mrpc
# script_creator(get_experiments_dir() + "/aug13_finetune_acadbertdata_supernet_retrain_subnet_125Ksteps_supernet_continue_mnli_bestckpt_rte_stsb_mrpc", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["rte", "stsb", "mrpc"], glue_config="supernetbasev3.standard.mnlickpt.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("v1", get_experiments_dir() + "/aug13_finetune_acadbertdata_supernet_retrain_subnet_125Ksteps_supernet_continue_mnli_bestckpt/0_v1_mnli_3e-5_32_2", "bert-base-uncased", None, get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)

# script_creator(get_experiments_dir() + "/aug16_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_tinybert", [ {"exp_name": "tbhard", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug16_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_tinybert_tbhard", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": "bert-base-uncased", "teacher_model_path":  "bert-base-uncased", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "inplace_distillation": "1", "distillation_type": "hard+tinybert", "inplace_kd_hard_loss_contrib": "1.0", "inplace_kd_distill_loss_contrib": "1.0", "inplace_kd_layers": "0+1+2+3+4+5+6+7+8+9+10+11+12 ", "freeze_largest_model": "yes", "freeze_smallest_model": "yes"})}]} ], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)
# script_creator(get_experiments_dir() + "/aug19_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_gendistill", [ {"exp_name": "hahid", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug19_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_gendistill_hahid", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": "bert-base-uncased", "teacher_model_path":  "bert-base-uncased", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "inplace_distillation": "1", "distillation_type": "hard+hiddenlastlayer", "inplace_kd_hard_loss_contrib": "1.0", "inplace_kd_distill_loss_contrib": "1.0", "inplace_kd_layers": "0+1+2+3+4+5+6+7+8+9+10+11+12 ", "freeze_largest_model": "yes", "freeze_smallest_model": "yes"})}]},  {"exp_name": "haatt", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug19_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_gendistill_haatt", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": "bert-base-uncased", "teacher_model_path":  "bert-base-uncased", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "inplace_distillation": "1", "distillation_type": "hard+attentionlastlayer", "inplace_kd_hard_loss_contrib": "1.0", "inplace_kd_distill_loss_contrib": "1.0", "inplace_kd_layers": "0+1+2+3+4+5+6+7+8+9+10+11+12 ", "freeze_largest_model": "yes", "freeze_smallest_model": "yes"})}]} ], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)

# supernet - finetuning - mnli - v3 search space
# bertbase as initial point
# script_creator(get_experiments_dir() + "/aug11_supernet_finetune_mnli_v3_bertbaseinit", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli"], glue_config="supernetbasev3.standard.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ ("bertbase", get_experiments_dir() + "/aug9_bert_roberta_standalone_noinit_nobottle_acabertpreproc/bertbase/best_model", "bert-base-uncased", "attention", "none")], search_space_id="v3"), time_in_mins=1000, num_gpus=4, generate_port=True, part_size=5, wandb="offline", wandb_entity="ganayu", wandb_project="effbert")
# script_creator(get_experiments_dir() + "/aug15_evosearch_supernet_finetune_mnli_v3_bertbaseinit", [ {"exp_name": "v3_superft", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/aug11_supernet_finetune_mnli_v3_bertbaseinit/1_bertbase_mnli_5e-5_16_3", "mixing": "attention", "search_space_id": "v3", "finetune": "mnli", "per_device_eval_batch_size": 128, "fitness_metric": "accuracy"})} ]} ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# supernet-v3 as initial point
# script_creator(get_experiments_dir() + "/aug16_supernet_finetune_mnli_v3_supernetbertbaseinit", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli"], glue_config="supernetbasev3.standard.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ ("bertbase", get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "bert-base-uncased", "attention", "none")], search_space_id="v3"), time_in_mins=1000, num_gpus=4, generate_port=True, part_size=5, wandb="offline", wandb_entity="ganayu", wandb_project="effbert")
# script_creator(get_experiments_dir() + "/aug16_evosearch_supernet_finetune_mnli_v3_supernetbertbaseinit", [ {"exp_name": "v3_superft", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/aug16_supernet_finetune_mnli_v3_supernetbertbaseinit/2_bertbase_mnli_5e-5_16_4", "mixing": "attention", "search_space_id": "v3", "finetune": "mnli", "per_device_eval_batch_size": 128, "fitness_metric": "accuracy"})} ]} ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)

# inplace logits+hard - scaling
# script_creator(get_experiments_dir() + "/aug13_supernetbase_inplacekd_distill_contrib", [ {"exp_name": "%s_%s"%(inplace_kd_hard_loss_contrib, inplace_kd_distill_loss_contrib), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug13_supernetbase_inplacekd_distill_contrib_%s_%s"%(inplace_kd_hard_loss_contrib, inplace_kd_distill_loss_contrib), "inplace_distillation": 1, "distillation_type": "logits+hard", "inplace_kd_hard_loss_contrib": inplace_kd_hard_loss_contrib, "inplace_kd_distill_loss_contrib": inplace_kd_distill_loss_contrib, "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased"})}]} for inplace_kd_hard_loss_contrib, inplace_kd_distill_loss_contrib in [("1.0", "1.0"), ("1.0", "1.5"), ("1.0", "2.0"), ("1.0", "2.5")] ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/aug13_supernetbase_inplacekd_distill_contrib_itemcorrected", [ {"exp_name": "%s_%s"%(inplace_kd_hard_loss_contrib, inplace_kd_distill_loss_contrib), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug13_supernetbase_inplacekd_distill_contrib_%s_%s"%(inplace_kd_hard_loss_contrib, inplace_kd_distill_loss_contrib), "inplace_distillation": 1, "distillation_type": "logits+hard", "inplace_kd_hard_loss_contrib": inplace_kd_hard_loss_contrib, "inplace_kd_distill_loss_contrib": inplace_kd_distill_loss_contrib, "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased"})}]} for inplace_kd_hard_loss_contrib, inplace_kd_distill_loss_contrib in [("1.0", "1.0"), ("1.0", "2.0")] ], time_in_mins=10000, wandb="online") # ("1.0", "1.5"), , ("1.0", "2.5")
# script_creator(get_experiments_dir() + "/aug17_supernetbase_inplacekd_distill_contrib_itemcorrected_0p5equal", [ {"exp_name": "%s_%s"%(inplace_kd_hard_loss_contrib, inplace_kd_distill_loss_contrib), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug17_supernetbase_inplacekd_distill_contrib_itemcorrected_0p5equal_%s_%s"%(inplace_kd_hard_loss_contrib, inplace_kd_distill_loss_contrib), "inplace_distillation": 1, "distillation_type": "logits+hard", "inplace_kd_hard_loss_contrib": inplace_kd_hard_loss_contrib, "inplace_kd_distill_loss_contrib": inplace_kd_distill_loss_contrib, "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased"})}]} for inplace_kd_hard_loss_contrib, inplace_kd_distill_loss_contrib in [("0.5", "0.5")] ], time_in_mins=10000, wandb="online")

# v4 - continue pretrain - wait until search is done
# script_creator(get_experiments_dir() + "/aug13_v4p1_valppl_retrain_subnet_125Ksteps", [ {"exp_name": "supernet_continue", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug13_v4p1_valppl_retrain_subnet_125Ksteps_supernet_continue", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/aug6_supernetbase_v3_v4_space/v4/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug9_evosearch_v4p1_valppl/v4p1search/evo_results_29.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "mixing": "attention" })}]}], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)
# script_creator(get_experiments_dir() + "/aug13_v4_conv1dfix_retrain_subnet_125Ksteps", [ {"exp_name": "supernet_continue", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug13_v4_conv1dfix_retrain_subnet_125Ksteps_supernet_continue", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/aug7_supernetbase_v4_conv1dfix/v4/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_supernetbase_v4_conv1dfix/v4_conv1dfix/evo_results_29.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "mixing": "attention" })}]}], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8) # nan error: need to debug
# script_creator(get_experiments_dir() + "/aug11_evosearch_supernetbase_v4_conv1dfix", [ {"exp_name": "v4_conv1dfix", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/aug7_supernetbase_v4_conv1dfix/v4/best_model", "mixing": "attention", "search_space_id": "v4"})}]} ], time_in_mins=8000, wandb="no", num_gpus=8, generate_port=True)

# v3 - continue pretrain
# script_creator(get_experiments_dir() + "/aug17_v3_retrain_subnet_125Ksteps", [ {"exp_name": "v3subrt", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug17_v3_retrain_subnet_125Ksteps_supernet_continue", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/aug6_supernetbase_v3_v4_space/v3/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug9_evosearch_v3_valppl/v3search/evo_results_29.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "mixing": "attention", "search_space_id": "v3" })}]}], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8) # nan error todo: debug; wrong data: used new aca data instead of graphcore's
# script_creator(get_experiments_dir() + "/aug19_finetune_v3_retrain_subnet_125Ksteps_supernet_continue", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli", "cola"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("v3", get_experiments_dir() + "/aug17_v3_retrain_subnet_125Ksteps/v3subrt/best_model", "bert-base-uncased", None, get_experiments_dir() + "/aug9_evosearch_v3_valppl/v3search/evo_results_29.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5) # not started this due to nan error

# direct finetune v3, v4.1 v4.2 v4.3 using FT v3
model_to_directfinetune = []
model_to_directfinetune.append(("dftv3", get_experiments_dir() + "/aug6_supernetbase_v3_v4_space/v3/best_model", get_experiments_dir() + "/aug9_evosearch_v3_valppl/v3search/evo_results_29.xls", "Graphcore/bert-base-uncased", "v3"))
model_to_directfinetune.append(("dftv4.1", get_experiments_dir() + "/aug6_supernetbase_v3_v4_space/v4/best_model", get_experiments_dir() + "/aug9_evosearch_v4p1_valppl/v4p1search/evo_results_29.xls", "Graphcore/bert-base-uncased", "v4.1"))
model_to_directfinetune = []
model_to_directfinetune.append(("dftv4.2",  get_experiments_dir() + "/aug7_supernetbase_v4_conv1dfix/v4/best_model", get_experiments_dir() + "/aug11_evosearch_supernetbase_v4_conv1dfix/v4_conv1dfix/evo_results_29.xls", "Graphcore/bert-base-uncased", "v4.2"))
# model_to_directfinetune.append(("dftv4.3", get_experiments_dir() + "/aug9_supernetbase_v4p3/v4p3/best_model", get_experiments_dir() + "/aug11_evosearch_supernetbase_v4p3/v4p3/evo_results_29.xls", "Graphcore/bert-base-uncased", "v4.3"))
# experiments = create_finetuning_experiments_standalone_vs_supernet_v3(models=model_to_directfinetune, sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.train_glue_original", tasks=["mnli", "cola", "mrpc", "sst2"], num_gpus=8)
# script_creator(get_experiments_dir() + "/aug14_directfinetune_v3_v4.1-3_mnli_cola_mrpc_sst2", experiments, time_in_mins=10000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/aug14_directfinetune_v4.2_mnli_cola_mrpc_sst2", experiments, time_in_mins=10000, wandb="no", num_gpus=8, generate_port=True)

# best v1 model - check on other GLUE benchmark
# script_creator(get_experiments_dir() + "/aug14_finetune_acadbertdata_supernet_retrain_subnet_125Ksteps_supernet_continue_otherglue", create_finetuning_experiments_standalone_vs_supernet_v3(models=[("acav1", get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/supernet_continue/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased", None)], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.train_glue_original", tasks=["mrpc", "sst2", "qnli", "qqp", "rte", "stsb"], num_gpus=8), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)

# block-wise inplace distillation - attention/hidden 0-5/0,3,6,9,11
# script_creator(get_experiments_dir() + "/aug14_v3_supernetbase_inplacekd_blockwise_layerpatterns", [ {"exp_name": "hid0-5", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug14_v3_supernetbase_inplacekd_blockwise_layerpatterns_hid0-5", "inplace_distillation": 1, "distillation_type": "hard+hiddenlastlayer", "inplace_kd_layers": "0+1+2+3+4+5", "inplace_kd_distill_loss_contrib": "2.0", "inplace_kd_hard_loss_contrib": "1.0"})}]}, {"exp_name": "hid036911", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug14_v3_supernetbase_inplacekd_blockwise_layerpatterns_hid036911", "inplace_distillation": 1, "distillation_type": "hard+hiddenlastlayer", "inplace_kd_layers": "0+3+6+9+11", "inplace_kd_distill_loss_contrib": "2.0", "inplace_kd_hard_loss_contrib": "1.0"})}]}, {"exp_name": "att0-5", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug14_v3_supernetbase_inplacekd_blockwise_layerpatterns_att0-5", "inplace_distillation": 1, "distillation_type": "hard+attentionlastlayer", "inplace_kd_layers": "0+1+2+3+4+5", "inplace_kd_distill_loss_contrib": "1.0", "inplace_kd_hard_loss_contrib": "1.0"})}]}, {"exp_name": "att036911", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug14_v3_supernetbase_inplacekd_blockwise_layerpatterns_att036911", "inplace_distillation": 1, "distillation_type": "hard+attentionlastlayer", "inplace_kd_layers": "0+3+6+9+11", "inplace_kd_distill_loss_contrib": "1.0", "inplace_kd_hard_loss_contrib": "1.0"})}]}  ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/aug16_v3_supernetbase_inplacekd_blockwise_layerpatterns_itemcorrected", [ {"exp_name": "hidfull", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug16_v3_supernetbase_inplacekd_blockwise_layerpatterns_itemcorrected_hidfull", "inplace_distillation": 1, "distillation_type": "hard+hiddenlastlayer", "inplace_kd_layers": "0+1+2+3+4+5+6+7+8+9+10+11+12", "inplace_kd_distill_loss_contrib": "1.0", "inplace_kd_hard_loss_contrib": "1.0"})}]}, {"exp_name": "attfull", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug16_v3_supernetbase_inplacekd_blockwise_layerpatterns_itemcorrected_attfull", "inplace_distillation": 1, "distillation_type": "hard+attentionlastlayer", "inplace_kd_layers": "0+1+2+3+4+5+6+7+8+9+10+11", "inplace_kd_distill_loss_contrib": "1.0", "inplace_kd_hard_loss_contrib": "1.0"})}]}  ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/aug17_v3_supernetbase_inplacekd_blockwise_layerpatterns_itemcorrected_0p5equal", [ {"exp_name": "hidfull", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug16_v3_supernetbase_inplacekd_blockwise_layerpatterns_itemcorrected_hidfull_0p5equal", "inplace_distillation": 1, "distillation_type": "hard+hiddenlastlayer", "inplace_kd_layers": "0+1+2+3+4+5+6+7+8+9+10+11+12", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_hard_loss_contrib": "0.5"})}]}, {"exp_name": "attfull", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug16_v3_supernetbase_inplacekd_blockwise_layerpatterns_itemcorrected_attfull_0p5equal", "inplace_distillation": 1, "distillation_type": "hard+attentionlastlayer", "inplace_kd_layers": "0+1+2+3+4+5+6+7+8+9+10+11", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_hard_loss_contrib": "0.5"})}]}  ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/aug20_v3_supernetbase_inplacekd_logits_itemcorrected_0p5equal", [ {"exp_name": "loghard", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug20_v3_supernetbase_inplacekd_logits_itemcorrected_0p5equal_loghard", "inplace_distillation": 1, "distillation_type": "hard+logits", "inplace_kd_layers": "0+1+2+3+4+5+6+7+8+9+10+11+12", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_hard_loss_contrib": "0.5"})}]} ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/aug20_v1_supernetbase_search_inplace_configs", [ {"exp_name": "seloghard", "runs": [{"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/aug16_v3_supernetbase_inplacekd_blockwise_layerpatterns_itemcorrected/attfull/best_model"})}]}  ], time_in_mins=8000, wandb="no", num_gpus=4) # naming mistake should be atthard
# script_creator(get_experiments_dir() + "/aug21_directfinetune_atthard_mnli_cola_mrpc_sst2", create_finetuning_experiments_standalone_vs_supernet_v3(models=[("dftatthard", get_experiments_dir() + "/aug16_v3_supernetbase_inplacekd_blockwise_layerpatterns_itemcorrected/attfull/best_model", get_experiments_dir() + "/aug20_v1_supernetbase_search_inplace_configs/seloghard/evo_results_29.xls", "Graphcore/bert-base-uncased", None)], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.train_glue_original", tasks=["mnli", "cola", "mrpc", "sst2"], num_gpus=8), time_in_mins=10000, wandb="no", num_gpus=8, generate_port=True)

# acabertdata - supernet - v3 space
# script_creator(get_experiments_dir() + "/aug18_acadbertdata_supernet_v3_v4p2_space", [ {"exp_name": "acav3", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug18_acadbertdata_supernet_v3", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "search_space_id": "v3", "mixing": "attention"})}]}, {"exp_name": "acav4p2", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug18_acadbertdata_supernet_v4p2", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "search_space_id": "v4.2", "mixing": "attention"})}]}  ], time_in_mins=10000, wandb="online", generate_port=True)
# search
# script_creator(get_experiments_dir() + "/aug20_evosearch_v3_valppl_acabert", [ {"exp_name": "v3search", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/aug18_acadbertdata_supernet_v3_v4p2_space/acav3/best_model", "mixing": "attention", "search_space_id": "v3",  "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len")})}]} ], time_in_mins=8000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/aug20_evosearch_v4_conv1dfix_acabert", [ {"exp_name": "v4_conv1dfix", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/aug18_acadbertdata_supernet_v3_v4p2_space/acav4p2/best_model", "mixing": "attention", "search_space_id": "v4.2",  "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len") })}]} ], time_in_mins=8000, wandb="no", num_gpus=8, generate_port=True) # need to start once supernet training is done
# script_creator(get_experiments_dir() + "/aug21_v3_acadbertdata_supernet_retrain_subnet_125Ksteps", [ {"exp_name": "v3acrs", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug21_v3_acadbertdata_supernet_retrain_subnet_125Ksteps_supernet_continue", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/aug18_acadbertdata_supernet_v3_v4p2_space/acav3/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug20_evosearch_v3_valppl_acabert/v3search/evo_results_29.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "mixing": "attention", "search_space_id": "v3" })}]}], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)

# v4p5 - search
# script_creator(get_experiments_dir() + "/aug18_evosearch_supernetbase_v4p5", [ {"exp_name": "v4p5", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/aug13_supernetbase_v4p5/v4p5/best_model", "mixing": "attention", "search_space_id": "v4.5"})}]} ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/aug19_directfinetune_v4.5_mnli_cola_mrpc_sst2", create_finetuning_experiments_standalone_vs_supernet_v3(models=[("dftv4.5", get_experiments_dir() + "/aug13_supernetbase_v4p5/v4p5/best_model", get_experiments_dir() + "/aug18_evosearch_supernetbase_v4p5/v4p5/evo_results_29.xls", "Graphcore/bert-base-uncased", "v4.5")], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.train_glue_original", tasks=["mnli", "cola", "mrpc", "sst2"], num_gpus=8), time_in_mins=10000, wandb="no", num_gpus=8, generate_port=True)

# v1 - acabert - search - different model sizes - 55M-60M, 25M-30M
# script_creator(get_experiments_dir() + "/aug18_evosearch_acadbertdata_supernet_v1_model_sizes", [ {"exp_name": "50M-60M", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "mixing": "bert-bottleneck", "params_constraints": "50000000,60000000", "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len")})}]}], time_in_mins=8000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/aug19_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_60M", [ {"exp_name": "50Mrsub", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug19_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_supernet_continue", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug18_evosearch_acadbertdata_supernet_v1_model_sizes/50M-60M/evo_results_29.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck" })}]}], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)
# script_creator(get_experiments_dir() + "/aug21_directfinetune_v1_aca_60M_mnli_cola_mrpc_sst2", create_finetuning_experiments_standalone_vs_supernet_v3(models=[("df60M", get_experiments_dir() + "/aug19_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_60M/50Mrsub/best_model", get_experiments_dir() + "/aug18_evosearch_acadbertdata_supernet_v1_model_sizes/50M-60M/evo_results_29.xls", "Graphcore/bert-base-uncased", None)], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.train_glue_original", tasks=["mnli", "cola", "mrpc", "sst2"], num_gpus=8), time_in_mins=10000, wandb="no", num_gpus=8, generate_port=True)

# mnli, cola - get best checkpoint - for kd during finetuning
# script_creator(get_experiments_dir() + "/aug18_finetune_acabertdata_bertbasestandalone_mnli_ckptneeded", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli", "cola"], glue_config="supernetbasev3.standard.ckpt_needed.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("bertbase", get_experiments_dir() + "/aug9_bert_roberta_standalone_noinit_nobottle_acabertpreproc/bertbase/best_model", "bert-base-uncased", None, "none")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)
# qnli, qqp, sst2 - get best checkpoint - for kd during finetuning
# script_creator(get_experiments_dir() + "/aug18_finetune_acabertdata_bertbasestandalone_sst2_qnli_qqp_ckptneeded", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["sst2", "qnli", "qqp"], glue_config="supernetbasev3.standard.ckpt_needed.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("bertbase", get_experiments_dir() + "/aug9_bert_roberta_standalone_noinit_nobottle_acabertpreproc/bertbase/best_model", "bert-base-uncased", None, "none")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)
# mrpc, rte, stsb - get best checkpoint - for kd during finetuning - uses mnli checkpoint (error: set is_mnli_checkpoint to 1)
# script_creator(get_experiments_dir() + "/aug18_finetune_acabertdata_bertbasestandalone_mrpc_rte_stsb_ckptneeded", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mrpc", "rte", "stsb"], glue_config="supernetbasev3.standard.ckpt_needed.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("bertbase", get_experiments_dir() + "/aug18_finetune_acabertdata_bertbasestandalone_mnli_ckptneeded/8_bertbase_mnli_3e-5_16_4", "bert-base-uncased", None, "none")], is_mnli_checkpoint=1), time_in_mins=1000, wandb="no", num_gpus=2, generate_port=True, part_size=5)
# previous problem gpu problem - sts problem fix next
# script_creator(get_experiments_dir() + "/aug25_finetune_acabertdata_bertbasestandalone_mrpc_rte_ckptneeded", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=2, tasks=["mrpc", "rte"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("bertbase", get_experiments_dir() + "/aug18_finetune_acabertdata_bertbasestandalone_mnli_ckptneeded/8_bertbase_mnli_3e-5_16_4", "bert-base-uncased", None, "none")], is_mnli_checkpoint=1, ckpt_needed=["mrpc", "rte"]), time_in_mins=1000, wandb="no", num_gpus=2, generate_port=True, part_size=5)
# cola - kd during finetuning
# script_creator(get_experiments_dir() + "/aug19_finetune_v1sub_cola_hardhidden", create_finetuning_experiments_standalone_vs_supernet_v2(models=[{"exp_name": "acav1", "model_name_or_path": get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/supernet_continue/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "teacher_model_path": "/fsx/ganayu/experiments/supershaper/aug18_finetune_acabertdata_bertbasestandalone_mnli_ckptneeded/23_bertbase_cola_5e-5_32_4", "inplace_distillation": "1", "distillation_type": "hard+hiddenlastlayer", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_layers": "0+1+2+3+4+5+6+7+8+9+10+11+12"}], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.distill.supernet_finetune", tasks=["cola"], num_gpus=4), time_in_mins=1000, wandb="offline", num_gpus=4, generate_port=True, part_size=5)

# bert - academic bert preprocessing - for teacher for kd during finetuning
# script_creator(get_experiments_dir() + "/aug20_bert_standalone_yesbottle_acabertpreproc", [ {"exp_name": "bertbase", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("bertbasev3.standard.train_mlm"), {"subtransformer_config_path": get_model_configs("bertuncased.bottleneck.12L_768H"), "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "initialize_pretrained_weights": "no", "mixing": "bert-bottleneck", "experiment_name": "bertbase"})}]} ], time_in_mins=8000, wandb="online", wandb_entity="ganayu", wandb_project="effbert", num_gpus=8, generate_port=True)
# mnli, cola - get best checkpoint - for kd during finetuning
# script_creator(get_experiments_dir() + "/aug20_finetune_acabertdata_bertbasestandaloneyesbottle_mnli_cola_ckptneeded", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli", "cola"], glue_config="supernetbasev3.standard.ckpt_needed.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("bertbase", get_experiments_dir() + "/aug20_bert_standalone_yesbottle_acabertpreproc/bertbase/best_model", "bert-base-uncased", "bert-bottleneck", "none")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)
# bertbase cola 60.76107288534243 /fsx/ganayu/experiments/supershaper/aug18_finetune_acabertdata_bertbasestandalone_mnli_ckptneeded/23_bertbase_cola_5e-5_32_4/sys.out
# bertbase mnli 83.69299674267101 /fsx/ganayu/experiments/supershaper/aug18_finetune_acabertdata_bertbasestandalone_mnli_ckptneeded/8_bertbase_mnli_3e-5_16_4/sys.out
# for exp_suffix, distill_type in [("hardlogits", "hard+logits"),("hardattn", "hard+attentionlastlayer")]: # ("hardhid", "hard+hiddenlastlayer"), 
  # script_creator(get_experiments_dir() + "/aug22_finetune_v1sub_cola_%s"%exp_suffix, create_finetuning_experiments_standalone_vs_supernet_v2(models=[{"exp_name": exp_suffix, "model_name_or_path": get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/supernet_continue/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "teacher_model_path": "/fsx/ganayu/experiments/supershaper/aug18_finetune_acabertdata_bertbasestandalone_mnli_ckptneeded/23_bertbase_cola_5e-5_32_4", "inplace_distillation": "1", "distillation_type": distill_type, "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_layers": "0+1+2+3+4+5+6+7+8+9+10+11+12"}], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.distill.supernet_finetune", tasks=["cola"], num_gpus=4), time_in_mins=1000, wandb="offline", num_gpus=4, generate_port=True, part_size=5) # this only chose hard+hiddenlastlayer
  # script_creator(get_experiments_dir() + "/aug23_finetune_v1sub_cola_%s"%exp_suffix, create_finetuning_experiments_standalone_vs_supernet_v2(models=[{"exp_name": exp_suffix, "model_name_or_path": get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/supernet_continue/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "teacher_model_path": "/fsx/ganayu/experiments/supershaper/aug18_finetune_acabertdata_bertbasestandalone_mnli_ckptneeded/23_bertbase_cola_5e-5_32_4", "inplace_distillation": "1", "distillation_type": distill_type, "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_layers": "0+1+2+3+4+5+6+7+8+9+10+11+12"}], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.distill.supernet_finetune", tasks=["cola"], num_gpus=4), time_in_mins=1000, wandb="offline", num_gpus=4, generate_port=True, part_size=5) # corrected for different distillation choices
# for exp_suffix, distill_type in [("hardlogits", "hard+logits"),("hardattn", "hard+attentionlastlayer")]: # ("hardhid", "hard+hiddenlastlayer"), 
  # script_creator(get_experiments_dir() + "/aug22_finetune_v1sub_mnli_%s"%exp_suffix, create_finetuning_experiments_standalone_vs_supernet_v2(models=[{"exp_name": exp_suffix, "model_name_or_path": get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/supernet_continue/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "teacher_model_path": "/fsx/ganayu/experiments/supershaper/aug18_finetune_acabertdata_bertbasestandalone_mnli_ckptneeded/8_bertbase_mnli_3e-5_16_4", "inplace_distillation": "1", "distillation_type": distill_type, "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_layers": "0+1+2+3+4+5+6+7+8+9+10+11+12"}], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.distill.supernet_finetune", tasks=["mnli"], num_gpus=4), time_in_mins=1000, wandb="offline", num_gpus=4, generate_port=True, part_size=5) # this only chose hard+hiddenlastlayer
# script_creator(get_experiments_dir() + "/aug23_finetune_v1sub_mnli_%s"%exp_suffix, create_finetuning_experiments_standalone_vs_supernet_v2(models=[{"exp_name": exp_suffix, "model_name_or_path": get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/supernet_continue/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "teacher_model_path": "/fsx/ganayu/experiments/supershaper/aug18_finetune_acabertdata_bertbasestandalone_mnli_ckptneeded/8_bertbase_mnli_3e-5_16_4", "inplace_distillation": "1", "distillation_type": distill_type, "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_layers": "0+1+2+3+4+5+6+7+8+9+10+11+12"}], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.distill.supernet_finetune", tasks=["mnli"], num_gpus=4), time_in_mins=1000, wandb="offline", num_gpus=4, generate_port=True, part_size=4)
# finetune w. kd - v1 - aca - sst2, qnli, qqp
# for task, teacher_model_path in [("sst2", "/fsx/ganayu/experiments/supershaper/aug18_finetune_acabertdata_bertbasestandalone_sst2_qnli_qqp_ckptneeded/7_bertbase_sst2_3e-5_16_3"), ("qnli", "/fsx/ganayu/experiments/supershaper/aug18_finetune_acabertdata_bertbasestandalone_sst2_qnli_qqp_ckptneeded/23_bertbase_qnli_5e-5_32_4"), ("qqp", "/fsx/ganayu/experiments/supershaper/aug18_finetune_acabertdata_bertbasestandalone_sst2_qnli_qqp_ckptneeded/41_bertbase_qqp_5e-5_32_4")]:
#  script_creator(get_experiments_dir() + "/aug23_finetune_v1sub_%s_hardlogits"%task, create_finetuning_experiments_standalone_vs_supernet_v2(models=[{"exp_name": "hl%s"%task, "model_name_or_path": get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/supernet_continue/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "teacher_model_path": teacher_model_path, "inplace_distillation": "1", "distillation_type": "hard+logits", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5"}], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.distill.supernet_finetune", tasks=[task], num_gpus=4), time_in_mins=1000, wandb="offline", num_gpus=4, generate_port=True, part_size=5)
# finetune w. kd - v1 - aca - mrpc, rte, stsb
# for task, teacher_model_path in [("mrpc", "/fsx/ganayu/experiments/supershaper/aug25_finetune_acabertdata_bertbasestandalone_mrpc_rte_ckptneeded/0_bertbase_mrpc_5e-5_16_2"), ("rte", "/fsx/ganayu/experiments/supershaper/aug25_finetune_acabertdata_bertbasestandalone_mrpc_rte_ckptneeded/29_bertbase_rte_3e-5_32_4")]: # ("stsb", "/fsx/ganayu/experiments/supershaper/aug18_finetune_acabertdata_bertbasestandalone_sst2_qnli_qqp_ckptneeded/41_bertbase_qqp_5e-5_32_4"), 
#  script_creator(get_experiments_dir() + "/aug25_finetune_v1sub_%s_hardlogits"%task, create_finetuning_experiments_standalone_vs_supernet_v2(models=[{"exp_name": "hl%s"%task, "model_name_or_path": get_experiments_dir() + "/aug23_finetune_v1sub_mnli_hardlogits/10_hardlogits_mnli_3e-5_32_3", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "teacher_model_path": teacher_model_path, "inplace_distillation": "1", "distillation_type": "hard+logits", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5"}], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.distill.supernet_finetune", tasks=[task], num_gpus=2), time_in_mins=1000, wandb="offline", num_gpus=2, generate_port=True, part_size=5) # is_mnli_checkpoint=1

# search - v1 - acabert - val. accuracy instead of val. ppl as search fitness metric
# script_creator(get_experiments_dir() + "/aug21_evosearch_acadbertdata_supernet_v1_valacc", [ {"exp_name": "sevalacc", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "fitness_metric": "accuracy", "mixing": "bert-bottleneck", "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len")})}]} ], time_in_mins=8000, wandb="no", num_gpus=8, generate_port=True)

# acabert - v1 - subnet training - logits+hard
# script_creator(get_experiments_dir() + "/aug23_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_logitshard", [ {"exp_name": "hardlog", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug23_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_logitshard", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "teacher_model_path":  "bert-base-uncased", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "inplace_distillation": "1", "distillation_type": "hard+logits", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_layers": "0+1+2+3+4+5+6+7+8+9+10+11+12 ", "freeze_largest_model": "yes", "freeze_smallest_model": "yes"})}]} ], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)

# acabert - v1 - inplace distillation - hard+hidden/logits - differnt patterns
# script_creator(get_experiments_dir() + "/aug23_v3_supernetbase_inplacekd_blockwise_layerpatterns", [ {"exp_name": "loghard", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug23_v3_supernetbase_inplacekd_blockwise_layerpatterns_loghard", "inplace_distillation": 1, "distillation_type": "hard+logits",  "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_hard_loss_contrib": "0.5", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased"})}]}, {"exp_name": "hid12", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug23_v3_supernetbase_inplacekd_blockwise_layerpatterns_hid12", "inplace_distillation": 1, "distillation_type": "hard+hiddenlastlayer", "inplace_kd_layers": "12", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_hard_loss_contrib": "0.5", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased"})}]}, {"exp_name": "hid24681012", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug23_v3_supernetbase_inplacekd_blockwise_layerpatterns_hid24681012", "inplace_distillation": 1, "distillation_type": "hard+hiddenlastlayer", "inplace_kd_layers": "2+4+6+8+10+12", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_hard_loss_contrib": "0.5", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased"})}]}, {"exp_name": "hid1-12", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug23_v3_supernetbase_inplacekd_blockwise_layerpatterns_hid1-12", "inplace_distillation": 1, "distillation_type": "hard+hiddenlastlayer", "inplace_kd_layers": "1+2+3+4+5+6+7+8+9+10+11+12", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_hard_loss_contrib": "0.5", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased"})}]} ], time_in_mins=10000, wandb="online")

# acabert - v1 - distillation - subnet - hard+hidden (run later)
# script_creator(get_experiments_dir() + "/aug23_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_hardhidden", [ {"exp_name": "hid12", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug23_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_hardhidden_hid12", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "teacher_model_path":  "bert-base-uncased", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "inplace_distillation": "1", "distillation_type": "hard+hiddenlastlayer", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_layers": "12", "freeze_largest_model": "yes", "freeze_smallest_model": "yes"})}]}, {"exp_name": "hid1-12", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug23_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_hardhidden_hid1-12", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "teacher_model_path": "bert-base-uncased", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "inplace_distillation": "1", "distillation_type": "hard+hiddenlastlayer", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_layers": "1+2+3+4+5+6+7+8+9+10+11+12 ", "freeze_largest_model": "yes", "freeze_smallest_model": "yes"})}]}, {"exp_name": "hid24681012", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug23_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_hardhidden_hid24681012", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "teacher_model_path": "bert-base-uncased", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "inplace_distillation": "1", "distillation_type": "hard+hiddenlastlayer", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_layers": "2+4+6+8+10+12 ", "freeze_largest_model": "yes", "freeze_smallest_model": "yes"})}]} ], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)

# acabert - v3 - subnet training - debug: nan loss issue
# lower learning rate or increase no. of warump steps
# script_creator(get_experiments_dir() + "/aug23_v3_acadbertdata_supernet_retrain_subnet_125Ksteps_v3acrs", [ {"exp_name": "v3warm10k", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug23_v3_acadbertdata_supernet_retrain_subnet_125Ksteps_v3acrs_v3warm", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 10000, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/aug18_acadbertdata_supernet_v3_v4p2_space/acav3/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug20_evosearch_v3_valppl_acabert/v3search/evo_results_29.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "mixing": "attention", "search_space_id": "v3" })}]}, {"exp_name": "v3lr", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug23_v3_acadbertdata_supernet_retrain_subnet_125Ksteps_v3acrs_v3lr", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/aug18_acadbertdata_supernet_v3_v4p2_space/acav3/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug20_evosearch_v3_valppl_acabert/v3search/evo_results_29.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "mixing": "attention", "search_space_id": "v3", "learning_rate": 1e-05})}]}], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)
# subnet finetuning
# script_creator(get_experiments_dir() + "/aug24_finetune_acadbertdata_supernet_retrain_subnet_125Ksteps_supernet_continue_v3", create_finetuning_experiments_standalone_vs_supernet_v2(models=[("acav3", get_experiments_dir() + "/aug23_v3_acadbertdata_supernet_retrain_subnet_125Ksteps_v3acrs/v3lr/best_model", "bert-base-uncased", "attention", get_experiments_dir() + "/aug20_evosearch_v3_valppl_acabert/v3search/evo_results_29.xls")], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.train_glue_original", tasks=["mrpc", "sst2", "rte"], num_gpus=2), time_in_mins=300, wandb="no", num_gpus=2, generate_port=True, part_size=5)

# v1 - acabert - subnet - supernetcontinue - 60M
# script_creator(get_experiments_dir() + "/aug24_v1_acabert_supernet_rsub_125Ksteps_60M", [ {"exp_name": "v1_60M", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug24_v1_acabert_supernet_rsub_125Ksteps_60M", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug18_evosearch_acadbertdata_supernet_v1_model_sizes/50M-60M/evo_results_29.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased" })}]} ], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)

# v1 - acabert - subnet - MLM + logits - scratch 67M
# script_creator(get_experiments_dir() + "/aug24_v1_acadbertdata_supernet_rsub_scratch_125Ksteps_logitshard", [ {"exp_name": "rsscratch", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug24_v1_acadbertdata_supernet_rsub_scratch_125Ksteps_logitshard", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "teacher_model_path": "bert-base-uncased", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "inplace_distillation": "1", "distillation_type": "hard+logits", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_layers": "0+1+2+3+4+5+6+7+8+9+10+11+12 ", "freeze_largest_model": "yes", "freeze_smallest_model": "yes", "initialize_pretrained_weights": "no"})}]} ], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)

# v1 - finetune - subnet - fix: hidden_dropout as 0.1
# script_creator(get_experiments_dir() + "/aug25_finetune_acadbertdata_supernet_retrain_subnet_125Ksteps_supernet_continue", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli", "cola", "sst2", "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("v1", get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/supernet_continue/best_model", "bert-base-uncased", None, get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls")], ckpt_needed=["mnli"]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)
# script_creator(get_experiments_dir() + "/aug13_finetune_acadbertdata_supernet_retrain_subnet_125Ksteps_supernet_continue_mnli_bestckpt_rte_stsb_mrpc", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["rte", "stsb", "mrpc"], glue_config="supernetbasev3.standard.mnlickpt.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("v1", get_experiments_dir() + "/aug13_finetune_acadbertdata_supernet_retrain_subnet_125Ksteps_supernet_continue_mnli_bestckpt/0_v1_mnli_3e-5_32_2", "bert-base-uncased", None, get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls")]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)

# v1 - finetune - subnet - 60M
# script_creator(get_experiments_dir() + "/aug25_finetune_aca_v1subnet_rsub_60M", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["cola", "mnli", "sst2", "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("v1", get_experiments_dir() + "/aug24_v1_acabert_supernet_rsub_125Ksteps_60M/v1_60M/best_model", "bert-base-uncased", None, get_experiments_dir() + "/aug18_evosearch_acadbertdata_supernet_v1_model_sizes/50M-60M/evo_results_29.xls")], ckpt_needed=["mnli"]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)

# v1 - direct finetune w. kd - subnet 67M
# for task, teacher_model_path in [("cola", "/fsx/ganayu/experiments/supershaper/aug18_finetune_acabertdata_bertbasestandalone_mnli_ckptneeded/23_bertbase_cola_5e-5_32_4"), ("mnli", "/fsx/ganayu/experiments/supershaper/aug18_finetune_acabertdata_bertbasestandalone_mnli_ckptneeded/8_bertbase_mnli_3e-5_16_4")]:
#  script_creator(get_experiments_dir() + "/aug25_directfinetune_v1sub_%s_hardlogits"%task, create_finetuning_experiments_standalone_vs_supernet_v2(models=[{"exp_name": "hl%s"%task, "model_name_or_path": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "teacher_model_path": teacher_model_path, "inplace_distillation": "1", "distillation_type": "hard+logits", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5"}], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.distill.supernet_finetune", tasks=[task], num_gpus=4), time_in_mins=1000, wandb="offline", num_gpus=4, generate_port=True, part_size=5)

# v1 - scratch finetune w. kd - subnet 67M
# for task, teacher_model_path in [("cola", "/fsx/ganayu/experiments/supershaper/aug18_finetune_acabertdata_bertbasestandalone_mnli_ckptneeded/23_bertbase_cola_5e-5_32_4"), ("mnli", "/fsx/ganayu/experiments/supershaper/aug18_finetune_acabertdata_bertbasestandalone_mnli_ckptneeded/8_bertbase_mnli_3e-5_16_4")]:
#  script_creator(get_experiments_dir() + "/aug25_directfinetune_v1scratch_%s_hardlogits"%task, create_finetuning_experiments_standalone_vs_supernet_v2(models=[{"exp_name": "hl%s"%task, "model_name_or_path": get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/retrain_scratch/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "teacher_model_path": teacher_model_path, "inplace_distillation": "1", "distillation_type": "hard+logits", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5"}], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.distill.supernet_finetune", tasks=[task], num_gpus=4), time_in_mins=1000, wandb="offline", num_gpus=4, generate_port=True, part_size=5)

# mnli, cola - get best checkpoint - for kd during finetuning - use google trained bert as teacher
# script_creator(get_experiments_dir() + "/aug25_finetune_googlebert_mnli_cola_ckptneeded", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mnli", "cola", "sst2", "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("bertbase", "bert-base-uncased", "bert-base-uncased", "bert-bottleneck", "none") ], ckpt_needed=["mnli", "cola", "sst2", "qnli", "qqp"]), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)
# mrpc, rte, sts-b - use google trained bert as teacher
# script_creator(get_experiments_dir() + "/aug25_finetune_googlebert_mrpc_rte_stsb_ckptneeded", create_finetuning_experiments_standalone_vs_supernet_v2(num_gpus=4, tasks=["mrpc", "rte", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("bertbase", "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mnli_cola_ckptneeded/6_bertbase_mnli_3e-5_16_2", "bert-base-uncased", "bert-bottleneck", "none") ], ckpt_needed=["mrpc", "rte", "stsb"], is_mnli_checkpoint=1), time_in_mins=1000, wandb="no", num_gpus=4, generate_port=True, part_size=5)

# small, big - bert standalone
# script_creator(get_experiments_dir() + "/aug26_bert_standalone_yesbottle_acabertpreproc_small_big", [ {"exp_name": "big", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("bertbasev3.standard.train_mlm"), {"subtransformer_config_path": get_model_configs("bertuncased.bottleneck.12L_768H"), "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "experiment_name": "big"})}]}, {"exp_name": "small", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("bertbasev3.standard.train_mlm"), {"subtransformer_config_path": get_model_configs("bertuncased.bottleneck.12L_120H"), "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "experiment_name": "small"})}]} ], time_in_mins=8000, wandb="online", wandb_entity="ganayu", wandb_project="effbert", num_gpus=8, generate_port=True)

# v3 fix - different lr
# script_creator(get_experiments_dir() + "/aug26_v3_acadbertdata_supernet_retrain_subnet_125Ksteps_v3acrs_difflr", [ {"exp_name": "v3lr4e-5", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug23_v3_acadbertdata_supernet_retrain_subnet_125Ksteps_v3acrs_v3warm", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/aug18_acadbertdata_supernet_v3_v4p2_space/acav3/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug20_evosearch_v3_valppl_acabert/v3search/evo_results_29.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "mixing": "attention", "search_space_id": "v3", "learning_rate": 4e-05 })}]}, {"exp_name": "v3lr3e-5", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug23_v3_acadbertdata_supernet_retrain_subnet_125Ksteps_v3acrs_v3lr", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/aug18_acadbertdata_supernet_v3_v4p2_space/acav3/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug20_evosearch_v3_valppl_acabert/v3search/evo_results_29.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "mixing": "attention", "search_space_id": "v3", "learning_rate": 3e-05})}]}], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)
# script_creator(get_experiments_dir() + "/aug28_finetune_aca_v3rsub_4e5_mnli_cola_qnli_qqp", create_finetuning_experiments_standalone_vs_supernet_v3(models=[("acav34e5", get_experiments_dir() + "/aug26_v3_acadbertdata_supernet_retrain_subnet_125Ksteps_v3acrs_difflr/v3lr4e-5/best_model", get_experiments_dir() + "/aug20_evosearch_v3_valppl_acabert/v3search/evo_results_29.xls", "bert-base-uncased")], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.train_glue_original", tasks=["mnli", "cola", "qnli", "qqp"], num_gpus=8, ckpt_needed=["mnli"]), time_in_mins=10000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/aug28_finetune_aca_v3rsub_3e5_mnli_cola_qnli_qqp", create_finetuning_experiments_standalone_vs_supernet_v3(models=[("acav33e5", get_experiments_dir() + "/aug26_v3_acadbertdata_supernet_retrain_subnet_125Ksteps_v3acrs_difflr/v3lr3e-5/best_model", get_experiments_dir() + "/aug20_evosearch_v3_valppl_acabert/v3search/evo_results_29.xls", "bert-base-uncased")], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.train_glue_original", tasks=["mnli", "cola", "qnli", "qqp"], num_gpus=8, ckpt_needed=["mnli"]), time_in_mins=10000, wandb="no", num_gpus=8, generate_port=True)

# v3 - subnet - finetune - 1e-5 - mnli, cola, qnli, qqp
# script_creator(get_experiments_dir() + "/aug27_finetune_aca_v3rsub_1e5_mnli_cola_qnli_qqp", create_finetuning_experiments_standalone_vs_supernet_v3(models=[("acav3", get_experiments_dir() + "/aug23_v3_acadbertdata_supernet_retrain_subnet_125Ksteps_v3acrs/v3lr/best_model", get_experiments_dir() + "/aug20_evosearch_v3_valppl_acabert/v3search/evo_results_29.xls", "bert-base-uncased")], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.train_glue_original", tasks=["mnli", "cola", "qnli", "qqp"], num_gpus=8, ckpt_needed=["mnli"]), time_in_mins=10000, wandb="no", num_gpus=8, generate_port=True)

# v3 - subnet - finetune - 1e-5 - mprc, rte, sts-b (use mnli checkpoint)

# v1 - inplacekd - fixed google teacher
# script_creator(get_experiments_dir() + "/aug27_v3_supernetbase_inplacekd_fixedteacher_loghard", [ {"exp_name": "goobert", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug27_v3_supernetbase_inplacekd_fixedteacher_loghard_goobert", "inplace_distillation": 1, "distillation_type": "hard+logits", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_hard_loss_contrib": "0.5", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "teacher_model_path": "bert-base-uncased"})}]} ], time_in_mins=10000, wandb="online")

# v1 - inplacekd - direct finetune
# for exp in ["loghard", "hid1-12"]: #  "hid12", "hid24681012", 
# for exp in ["hid12", "hid24681012"]:
#    script_creator(get_experiments_dir() + "/aug27_finetune_aca_v1_supernet_inplace_kd_%s"%exp, create_finetuning_experiments_standalone_vs_supernet_v3(models=[(exp, get_experiments_dir() + "/aug23_v3_supernetbase_inplacekd_blockwise_layerpatterns/%s/best_model"%exp, get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.train_glue_original", tasks=["mnli"], num_gpus=8), time_in_mins=8000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/aug29_finetune_aca_v1_supernet_inplace_kd_hid1-12_nonmnli_8tasks", create_finetuning_experiments_standalone_vs_supernet_v3(models=[("hid1-12", get_experiments_dir() + "/aug23_v3_supernetbase_inplacekd_blockwise_layerpatterns/hid1-12/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.train_glue_original", tasks=["cola", "mrpc", "rte", "sst2", "qnli", "qqp", "stsb"], num_gpus=8), time_in_mins=10000, wandb="no", num_gpus=8, generate_port=True)

# v1 - moe 2e/4e, pop_size=1/2
# script_creator(get_experiments_dir() + "/aug27_acadbertdata_supernet_v1_moe", [ {"exp_name": "%de_%drand"%(nexp, popsize), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug27_acadbertdata_supernet_v1_moe_%de_%drand"%(nexp, popsize), "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": nexp, "pop_size": popsize, "expert_routing_type": "sentence"})}]} for nexp, popsize in [(2, 1), (2, 2), (4, 1), (4, 2)] ], time_in_mins=10000, wandb="online")

# v1 - scratch - direct ft - non mnli/cola tasks
# script_creator(get_experiments_dir() + "/aug29_finetune_acadbertdata_supernet_retrain_subnet_125Ksteps_retrain_scratch_nonmnlicola_tasks", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["mrpc", "rte", "sst2", "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("v1scratch", get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/retrain_scratch/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=10000, wandb="no", num_gpus=8, generate_port=True)

# v1 - supernet - direct ft - non mnli/cola tasks
# script_creator(get_experiments_dir() + "/aug29_finetune_acadbertdata_supernet_noretrain_125Ksteps_nonmnlicola_tasks", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["mrpc", "rte", "sst2", "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("v1scratch", get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=10000, wandb="no", num_gpus=8, generate_port=True)

# v1 - google-trained bert as teacher - kd in finetuning
# for task, teacher_model_path in [("cola", "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mnli_cola_ckptneeded/32_bertbase_cola_2e-5_16_4"), ("mnli", "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mnli_cola_ckptneeded/6_bertbase_mnli_3e-5_16_2"), ("sst2", "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mnli_cola_ckptneeded/43_bertbase_sst2_3e-5_16_3"), ("qnli", "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mnli_cola_ckptneeded/60_bertbase_qnli_3e-5_16_2"), ("qqp", "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mnli_cola_ckptneeded/77_bertbase_qqp_5e-5_32_4")]:
#    script_creator(get_experiments_dir() + "/aug29_directfinetune_v1scratch_%s_hardlogits"%task, create_finetuning_experiments_standalone_vs_supernet_v3(models=[{"exp_name": "hl%s"%task, "model_name_or_path": get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/supernet_continue/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "teacher_model_path": teacher_model_path, "inplace_distillation": "1", "distillation_type": "hard+logits", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5"}], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.distill.supernet_finetune", tasks=[task], num_gpus=8), time_in_mins=8000, wandb="offline", num_gpus=8, generate_port=True)

# v1 - inpl kd with google trained bert - loghard
# script_creator(get_experiments_dir() + "/aug29_finetune_acadbertdata_supernet_inplkd_googlebert_loghard_125Ksteps_mnli", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["mnli"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("gooft", get_experiments_dir() + "/aug27_v3_supernetbase_inplacekd_fixedteacher_loghard/goobert/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=10000, wandb="no", num_gpus=4, generate_port=True)

# v1 - moe 2e/4e rand1 - direct finetune
# script_creator(get_experiments_dir() + "/aug29_finetune_acadbertdata_supernet_moe2e1rand_125Ksteps_mnli", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["mnli"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("moe2e1rand", get_experiments_dir() + "/aug27_acadbertdata_supernet_v1_moe/2e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=10000, wandb="no", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/aug29_finetune_acadbertdata_supernet_moe4e1rand_125Ksteps_mnli", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["mnli"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("moe4e1rand", get_experiments_dir() + "/aug27_acadbertdata_supernet_v1_moe/4e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=10000, wandb="no", num_gpus=4, generate_port=True)

# v1 - moe 2e/4e all exp. average - direct finetune
# script_creator(get_experiments_dir() + "/aug30_finetune_acadbertdata_supernet_moe2e1rand_allexpavg_mnli", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["mnli"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("allexpavg", get_experiments_dir() + "/aug27_acadbertdata_supernet_v1_moe/2e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")], all_experts_average=1), time_in_mins=10000, wandb="no", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/aug30_finetune_acadbertdata_supernet_moe4e1rand_allexpavg_mnli", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["mnli"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("allexpavg", get_experiments_dir() + "/aug27_acadbertdata_supernet_v1_moe/4e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")], all_experts_average=1), time_in_mins=10000, wandb="no", num_gpus=4, generate_port=True)

# v1 - moe 4e rand1 - direct finetune - non mnli tasks
# script_creator(get_experiments_dir() + "/aug31_finetune_acadbertdata_supernet_moe4e1rand_125Ksteps_nonmnlitasks", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["cola", "mrpc", "rte", "sst2", "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("moe4e1rand", get_experiments_dir() + "/aug27_acadbertdata_supernet_v1_moe/4e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=10000, wandb="no", num_gpus=4, generate_port=True)

# v1 - moe 2e/4e rand2 - direct finetune
# script_creator(get_experiments_dir() + "/aug31_finetune_acadbertdata_supernet_moe2e2rand_125Ksteps_alltasks", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["mnli", "cola", "mrpc", "rte", "sst2", "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("2e_2rand", get_experiments_dir() + "/aug27_acadbertdata_supernet_v1_moe/2e_2rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/aug31_finetune_acadbertdata_supernet_moe4e2rand_125Ksteps_alltasks", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["mnli", "cola", "mrpc", "rte", "sst2", "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("4e_2rand", get_experiments_dir() + "/aug27_acadbertdata_supernet_v1_moe/4e_2rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=4, generate_port=True)

# v1 - inplacekd - fixed google teacher - hidden1-12
# script_creator(get_experiments_dir() + "/aug31_v3_supernetbase_inplacekd_fixedteacher_loghard_hid1-12", [ {"exp_name": "goikh112", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "aug31_v3_supernetbase_inplacekd_fixedteacher_loghard_hid1-12", "inplace_distillation": 1, "distillation_type": "hard+hiddenlastlayer", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_hard_loss_contrib": "0.5", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "inplace_kd_layers": "1+2+3+4+5+6+7+8+9+10+11+12", "teacher_model_path": "bert-base-uncased"})}]} ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/sep6_finetune_supernetbase_inplacekd_fixedteacher_loghard_hid1-12", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["mnli", "cola", "mrpc", "rte", "sst2", "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("ikd1-12", get_experiments_dir() + "/aug31_v3_supernetbase_inplacekd_fixedteacher_loghard_hid1-12/goikh112/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=12000, wandb="no", num_gpus=8, generate_port=True)

# v1 - subnet-kd - logits+hard, hidden1-12, hidden24681012 - mnli+cola+mrpc
# script_creator(get_experiments_dir() + "/aug31_finetune_v1_subnetkd_logitshard_fthard_mnli_cola_mrpc", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["mnli", "cola", "mrpc"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("31log", get_experiments_dir() + "/aug23_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_logitshard/hardlog/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/aug31_finetune_v1_subnetkd_logitshid1-12_fthard_mnli_cola_mrpc", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["mnli", "cola", "mrpc"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("31hid1-12", get_experiments_dir() + "/aug23_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_hardhidden/hid1-12/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/aug31_finetune_v1_subnetkd_logitshid24681012_fthard_mnli_cola_mrpc", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["mnli", "cola", "mrpc"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("31hid24681012", get_experiments_dir() + "/aug23_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_hardhidden/hid24681012/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)

# v3 - google-trained bert as teacher - kd in finetuning
# for task, teacher_model_path in [("cola", "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mnli_cola_ckptneeded/32_bertbase_cola_2e-5_16_4"), ("mnli", "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mnli_cola_ckptneeded/6_bertbase_mnli_3e-5_16_2")]: # ("sst2", "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mnli_cola_ckptneeded/43_bertbase_sst2_3e-5_16_3"), ("qnli", "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mnli_cola_ckptneeded/60_bertbase_qnli_3e-5_16_2"), ("qqp", "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mnli_cola_ckptneeded/77_bertbase_qqp_5e-5_32_4")
#    script_creator(get_experiments_dir() + "/aug31_directfinetune_v3rsub_%s_hardlogits"%task, create_finetuning_experiments_standalone_vs_supernet_v3(models=[{"exp_name": "31hl%s"%task, "model_name_or_path": get_experiments_dir() + "/aug26_v3_acadbertdata_supernet_retrain_subnet_125Ksteps_v3acrs_difflr/v3lr4e-5/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug20_evosearch_v3_valppl_acabert/v3search/evo_results_29.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "teacher_model_path": teacher_model_path, "inplace_distillation": "1", "distillation_type": "hard+logits", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5"}], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.distill.supernet_finetune", tasks=[task], num_gpus=4), time_in_mins=8000, wandb="offline", num_gpus=4, generate_port=True)

# v1 - moe 2e - max/min/rand dynamic moe (not just rand) (bug: only min/rand dynamic moe)
# script_creator(get_experiments_dir() + "/sep3_acadbertdata_supernet_v1_moe_minmaxrand", [ {"exp_name": "%de_%drand"%(nexp, popsize), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep3_acadbertdata_supernet_v1_moe_minmaxrand_%de_%drand"%(nexp, popsize), "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": nexp, "pop_size": popsize, "expert_routing_type": "sentence"})}]} for nexp, popsize in [(2, 1), (4, 1), (8, 1)] ], time_in_mins=10000, wandb="online")
# direct finetune
#for nexp, popsize in [(2, 1), (4, 1), (8, 1)]:
    # script_creator(get_experiments_dir() + "/sep4_finetune_acadbertdata_supernet_v1_moe_%de_%drand"%(nexp, popsize), create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["mnli", "cola", "mrpc", "rte", "sst2", "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("%de_%drand"%(nexp, popsize), get_experiments_dir() + "/sep3_acadbertdata_supernet_v1_moe_minmaxrand/%de_%drand/best_model"%(nexp, popsize), get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/sep4_finetune_acadbertdata_supernet_v1_moe_%de_%drand_allexpavg"%(nexp, popsize), create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["mnli", "cola", "mrpc", "rte", "sst2", "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("%de_%drand"%(nexp, popsize), get_experiments_dir() + "/sep3_acadbertdata_supernet_v1_moe_minmaxrand/%de_%drand/best_model"%(nexp, popsize), get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")], all_experts_average=1), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)

# v1 - moe 3e, 5e, 9e - averaging moe as last expert (bug: only min/rand dynamic moe)
# script_creator(get_experiments_dir() + "/sep4_acadbertdata_supernet_v1_moe_averagingexpert", [ {"exp_name": "%de_%drand"%(nexp, popsize), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep3_acadbertdata_supernet_v1_moe_minmaxrand_%de_%drand"%(nexp, popsize), "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": nexp, "pop_size": popsize, "expert_routing_type": "sentence", "last_expert_averaging_expert": "yes"})}]} for nexp, popsize in [(3, 1), (5, 1), (9, 1)] ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/sep6_finetune_supernetbase_moe_averagingexpert_3e_1rand_allexpavg", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["mnli", "cola", "mrpc", "rte", "sst2", "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("ft3e_aleavg", get_experiments_dir() + "/sep4_acadbertdata_supernet_v1_moe_averagingexpert/3e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")], last_expert_averaging_expert="yes", all_experts_average=1), time_in_mins=12000, wandb="no", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/sep6_finetune_supernetbase_moe_averagingexpert_3e_1rand_firstexp", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["mnli", "cola", "mrpc", "rte", "sst2", "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("ft3e_1stexp", get_experiments_dir() + "/sep4_acadbertdata_supernet_v1_moe_averagingexpert/3e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")], last_expert_averaging_expert="yes"), time_in_mins=12000, wandb="no", num_gpus=4, generate_port=True)

# v1 - moe 2e - rand-min-max dynamic moe (corrected bug)
# script_creator(get_experiments_dir() + "/sep6_acadbertdata_supernet_v1_moe_minmaxrand_corrected", [ {"exp_name": "%de_%drand"%(nexp, popsize), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep3_acadbertdata_supernet_v1_moe_minmaxrand_%de_%drand"%(nexp, popsize), "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": nexp, "pop_size": popsize, "expert_routing_type": "sentence"})}]} for nexp, popsize in [(2, 1)]], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/sep6_finetune_acadbertdata_supernet_v1_moe_minmaxrand_corrected_2e_1rand", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("ft2e", get_experiments_dir() + "/sep6_acadbertdata_supernet_v1_moe_minmaxrand_corrected/2e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=12000, wandb="no", num_gpus=8, generate_port=True)
# --collapsed_training yes --initialize_other_experts no
# script_creator(get_experiments_dir() + "/sep8_acadbertdata_supernet_v1_moe_minmaxrand_collapsedtraining", [ {"exp_name": "ct_%de_%drand"%(nexp, popsize), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep8_acadbertdata_supernet_v1_moe_minmaxrand_collapsedtraining_%de_%drand"%(nexp, popsize), "model_name_or_path": "/fsx/ganayu/experiments/supershaper/sep6_acadbertdata_supernet_v1_moe_minmaxrand_corrected/2e_1rand/best_model", "num_warmup_steps": 0, "max_train_steps": 25000, "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": nexp, "pop_size": popsize, "expert_routing_type": "sentence", "collapsed_training": "yes", "initialize_other_experts": "no"})}]} for nexp, popsize in [(2, 1)]], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/sep8_acadbertdata_supernet_v1_moe_minmaxrand_collapsedtraining_partialcollapsing", [ {"exp_name": "ctpc_%de_%drand"%(nexp, popsize), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep8_acadbertdata_supernet_v1_moe_minmaxrand_collapsedtraining_partialcollapsing%de_%drand"%(nexp, popsize), "model_name_or_path": "/fsx/ganayu/experiments/supershaper/sep6_acadbertdata_supernet_v1_moe_minmaxrand_corrected/2e_1rand/best_model", "num_warmup_steps": 0, "max_train_steps": 25000, "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": nexp, "pop_size": popsize, "expert_routing_type": "sentence", "collapsed_training": "yes", "initialize_other_experts": "no", "partial_collapsing": "yes"})}]} for nexp, popsize in [(2, 1)]], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/sep9_finetune_acadbertdata_supernet_v1_moe_minmaxrand_collapsedtraining", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("ft2e", get_experiments_dir() + "/sep8_acadbertdata_supernet_v1_moe_minmaxrand_collapsedtraining/ct_2e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=12000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/sep9_acadbertdata_supernet_v1_moe_minmaxrand_collapsedtraining_partialcollapsing", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("ft2e", get_experiments_dir() + "/sep8_acadbertdata_supernet_v1_moe_minmaxrand_collapsedtraining_partialcollapsing/ctpc_2e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=12000, wandb="no", num_gpus=8, generate_port=True)

# v1 - moe 2e, 8e - rand-min-max dynamic moe (corrected bug) + averaging moe as last expert (nan error)
# script_creator(get_experiments_dir() + "/sep6_acadbertdata_supernet_v1_moe_minmaxrand_corrected_averagingexpert", [ {"exp_name": "%de_%drand"%(nexp, popsize), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep3_acadbertdata_supernet_v1_moe_minmaxrand_%de_%drand"%(nexp, popsize), "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": nexp, "pop_size": popsize, "expert_routing_type": "sentence", "last_expert_averaging_expert": "yes"})}]} for nexp, popsize in [(3, 1), (9, 1)]], time_in_mins=10000, wandb="online")

# v1 - moe 8e - rand-min-max dynamic moe (corrected bug)
# script_creator(get_experiments_dir() + "/sep6_acadbertdata_supernet_v1_moe_8e_minmaxrand_corrected", [ {"exp_name": "%de_%drand"%(nexp, popsize), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep3_acadbertdata_supernet_v1_moe_8e_minmaxrand_%de_%drand"%(nexp, popsize), "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": nexp, "pop_size": popsize, "expert_routing_type": "sentence"})}]} for nexp, popsize in [(8, 1)]], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/sep9_finetune_acadbertdata_supernet_v1_moe_8e_minmaxrand_corrected", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("ft2e", get_experiments_dir() + "/sep6_acadbertdata_supernet_v1_moe_8e_minmaxrand_corrected/8e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=12000, wandb="no", num_gpus=8, generate_port=True)

# v1 - moe 2e, 8e - rand-min-max dynamic moe (corrected bug) + initialize_other_experts=No
# script_creator(get_experiments_dir() + "/sep7_acadbertdata_supernet_v1_moe_2e_8e_initialize_other_experts_no", [ {"exp_name": "%de_%drand"%(nexp, popsize), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep3_acadbertdata_supernet_v1_moe_8e_minmaxrand_%de_%drand"%(nexp, popsize), "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": nexp, "pop_size": popsize, "expert_routing_type": "sentence", "initialize_other_experts": "no"})}]} for nexp, popsize in [(2, 1), (8, 1)]], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/sep9_finetune_acadbertdata_supernet_v1_moe_2e_8e_initialize_other_experts_no", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("ft2e", get_experiments_dir() + "/sep7_acadbertdata_supernet_v1_moe_2e_8e_initialize_other_experts_no/2e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=12000, wandb="no", num_gpus=8, generate_port=True)

# v1 - moe 4e, 6e - rand-min-max dynamic moe (corrected bug)
# script_creator(get_experiments_dir() + "/sep9_acadbertdata_supernet_v1_moe_minmaxrand_corrected_4e_6e", [ {"exp_name": "%de_%drand"%(nexp, popsize), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep3_acadbertdata_supernet_v1_moe_minmaxrand_%de_%drand"%(nexp, popsize), "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": nexp, "pop_size": popsize, "expert_routing_type": "sentence"})}]} for nexp, popsize in [(4, 1), (6, 1)]], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/sep11_finetune_acadbertdata_supernet_v1_moe_minmaxrand_corrected_4e", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("11-4e", get_experiments_dir() + "/sep9_acadbertdata_supernet_v1_moe_minmaxrand_corrected_4e_6e/4e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/sep11_finetune_acadbertdata_supernet_v1_moe_minmaxrand_corrected_6e", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("11-6e", get_experiments_dir() + "/sep9_acadbertdata_supernet_v1_moe_minmaxrand_corrected_4e_6e/6e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)

# v1 - moe 2e - rand-min-max dynamic moe + finetune 2nd expert
# script_creator(get_experiments_dir() + "/sep10_finetune_acadbertdata_supernet_v1_moe_minmaxrand_corrected_2e_1rand_useexpert2", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("ft2e", get_experiments_dir() + "/sep6_acadbertdata_supernet_v1_moe_minmaxrand_corrected/2e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")], use_expert_id=1), time_in_mins=12000, wandb="no", num_gpus=8, generate_port=True)

# v1 - moe 2e - consistency loss for max
# script_creator(get_experiments_dir() + "/sep10_acadbertdata_supernet_v1_moe_2e_consistencyloss_for_max", [ {"exp_name": "%de_%drand"%(nexp, popsize), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep10_acadbertdata_supernet_v1_moe_2e_consistencyloss_for_max_%de_%drand"%(nexp, popsize), "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": nexp, "pop_size": popsize, "expert_routing_type": "sentence", "consistency_loss_max": "yes"})}]} for nexp, popsize in [(2, 1)]], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/sep12_finetune_acadbertdata_supernet_v1_moe_2e_consistencyloss_for_max", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("12-clmax", get_experiments_dir() + "/sep10_acadbertdata_supernet_v1_moe_2e_consistencyloss_for_max/2e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)

# v1 - moe 2e - inplace kd
# script_creator(get_experiments_dir() + "/sep10_acadbertdata_supernet_v1_inplacekd_logits_moe2e", [ {"exp_name": "2e_ikd", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep10_acadbertdata_supernet_v1_inplacekd_logits_moe2e", "inplace_distillation": 1, "distillation_type": "logits+hard", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "pop_size": 1, "expert_routing_type": "sentence"})}]} ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/sep12_finetune_acadbertdata_supernet_v1_inplacekd_logits_moe2e", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("12-ikd", get_experiments_dir() + "/sep10_acadbertdata_supernet_v1_inplacekd_logits_moe2e/2e_ikd/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)

# v1 - moe 2e - inplace kd + consistency loss for max
# script_creator(get_experiments_dir() + "/sep10_acadbertdata_supernet_v1_inplacekd_logits_moe2e_consistencyloss_for_max", [ {"exp_name": "2eikdconst", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep10_acadbertdata_supernet_v1_inplacekd_logits_moe2e_consistencyloss_for_max", "inplace_distillation": 1, "distillation_type": "logits+hard", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "pop_size": 1, "expert_routing_type": "sentence", "consistency_loss_max": "yes"})}]} ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/sep14_finetune_acadbertdata_supernet_v1_inplacekd_logits_moe2e_consistencyloss_for_max", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("14-ikdconst", get_experiments_dir() + "/sep10_acadbertdata_supernet_v1_inplacekd_logits_moe2e_consistencyloss_for_max/2eikdconst/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)

# v1 - standard supernet - cosine_with_restarts - no momentum in adam optimizer (didn't start)
# script_creator(get_experiments_dir() + "/sep10_acadbertdata_supernet_v1_cosine_with_restarts", [ {"exp_name": "cosrest", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep10_acadbertdata_supernet_v1_cosine_with_restarts", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "lr_scheduler_type": "cosine_with_restarts"})}]} ], time_in_mins=10000, wandb="online")

# v1 - 67M - subnet+kd -> finetune+kd
# aug23_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_logitshard/hardlog/best_model
# script_creator(get_experiments_dir() + "/sep10_finetune_v1_subnetkdlogits_finetunekdlogits_67M", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli", "qnli", "qqp"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "10s-f-kd", "model_name_or_path": get_experiments_dir() + "/aug23_v1_acadbertdata_supernet_retrain_subnet_125Ksteps_logitshard/hardlog/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=12000, wandb="offline", num_gpus=8, generate_port=True)

# v1 - 67M - moe 2e - search
# script_creator(get_experiments_dir() + "/sep10_evosearch_supernet_v1_moe2e_67M", [ {"exp_name": "evomoe2e", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/sep6_acadbertdata_supernet_v1_moe_minmaxrand_corrected/2e_1rand/best_model", "mixing": "bert-bottleneck", "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len")})}]} ], time_in_mins=8000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/sep11_subnethardlog_evosearch_supernet_v1_moe2e_67M", [ {"exp_name": "hardlog", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep11_subnethardlog_evosearch_supernet_v1_moe2e_67M", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/sep6_acadbertdata_supernet_v1_moe_minmaxrand_corrected/2e_1rand/best_model", "teacher_model_path":  "bert-base-uncased", "subtransformer_config_path": get_experiments_dir() + "/sep10_evosearch_supernet_v1_moe2e_67M/evomoe2e/evo_results_29.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "inplace_distillation": "1", "distillation_type": "hard+logits", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5", "inplace_kd_layers": "0+1+2+3+4+5+6+7+8+9+10+11+12 ", "freeze_largest_model": "yes", "freeze_smallest_model": "yes"})}]} ], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)
# script_creator(get_experiments_dir() + "/sep12_finetune_subnethardlog_evosearch_supernet_v1_moe2e_67M", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("12-fsubmoe", get_experiments_dir() + "/sep11_subnethardlog_evosearch_supernet_v1_moe2e_67M/hardlog/best_model", get_experiments_dir() + "/sep10_evosearch_supernet_v1_moe2e_67M/evomoe2e/evo_results_29.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/sep14_subnethard_evosearch_supernet_v1_moe2e_67M", [ {"exp_name": "submoe_%dit"%(it), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep14_subnethard_evosearch_supernet_v1_moe2e_67M_%dit"%(it), "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/sep6_acadbertdata_supernet_v1_moe_minmaxrand_corrected/2e_1rand/best_model", "subtransformer_config_path": get_experiments_dir() + "/sep10_evosearch_supernet_v1_moe2e_67M/evomoe2e/evo_results_%d.xls"%it, "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "freeze_largest_model": "yes", "freeze_smallest_model": "yes"})}]} for it in [14, 29] ], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)
# for it in [14, 29]:
# script_creator(get_experiments_dir() + "/sep15_finetune_v1_subnet_finetunekdlogits_67M_moe2e_%devoit"%it, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli", "qnli", "qqp"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "15moe%dft"%it, "model_name_or_path": get_experiments_dir() + "/sep14_subnethard_evosearch_supernet_v1_moe2e_67M/submoe_%dit/best_model"%it, "subtransformer_config_path": get_experiments_dir() + "/sep10_evosearch_supernet_v1_moe2e_67M/evomoe2e/evo_results_%d.xls"%it, "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=8, generate_port=True)
# v1 - mnli - 67M checkpoint - 29.xls
#for task, teacher_model_path in [("mnli", "/fsx/ganayu/experiments/supershaper/aug18_finetune_acabertdata_bertbasestandalone_mnli_ckptneeded/8_bertbase_mnli_3e-5_16_4")]:
#  script_creator(get_experiments_dir() + "/sep18_finetune_v1sub_moe_%s_hardlogits"%task, create_finetuning_experiments_standalone_vs_supernet_v2(models=[{"exp_name": "hl%s"%task, "model_name_or_path": get_experiments_dir() + "/sep14_subnethard_evosearch_supernet_v1_moe2e_67M/submoe_29it/best_model", "subtransformer_config_path": get_experiments_dir() + "/sep10_evosearch_supernet_v1_moe2e_67M/evomoe2e/evo_results_29.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "teacher_model_path": teacher_model_path, "inplace_distillation": "1", "distillation_type": "hard+logits", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5"}], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.distill.supernet_finetune", tasks=[task], num_gpus=4, ckpt_needed=["mnli"]), time_in_mins=1000, wandb="offline", num_gpus=4, generate_port=True, part_size=5)
# v1 - mrpc, rte # 7799
# mnli /fsx/ganayu/experiments/supershaper/sep18_finetune_v1sub_moe_mnli_hardlogits/14_hlmnli_mnli_2e-5_16_4/sys.out
# script_creator(get_experiments_dir() + "/sep19_finetune_v1_subnet_finetunekdlogits_67M_moe", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["mrpc", "rte"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "19-67Mmoeft", "model_name_or_path": get_experiments_dir() + "/sep18_finetune_v1sub_moe_mnli_hardlogits/14_hlmnli_mnli_2e-5_16_4", "subtransformer_config_path": get_experiments_dir() + "/sep10_evosearch_supernet_v1_moe2e_67M/evomoe2e/evo_results_29.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits"), is_mnli_checkpoint=1), time_in_mins=4000, wandb="offline", num_gpus=8, generate_port=True)

# v1 - moe 2e - token - sentence-single routing (didn't start, too much training latency)
# script_creator(get_experiments_dir() + "/sep10_supernet_v1_moe_token_sentencesingle_routing", [ {"exp_name": "2e_%s"%(routing_type), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep3_acadbertdata_supernet_v1_moe_minmaxrand_2e_%s"%(routing_type), "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "pop_size": 1, "expert_routing_type": routing_type})}]} for routing_type in ["token", "sentence_single"] ], time_in_mins=10000, wandb="online")

# v1 - 60M - subnet finetuning + logits-kd w. google bert - 29.xls
# script_creator(get_experiments_dir() + "/sep14_finetune_v1_subnet_finetunekdlogits_60M", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli", "qnli", "qqp"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "14-60Mft", "model_name_or_path": get_experiments_dir() + "/aug24_v1_acabert_supernet_rsub_125Ksteps_60M/v1_60M/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug18_evosearch_acadbertdata_supernet_v1_model_sizes/50M-60M/evo_results_29.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=8, generate_port=True)

# v1 - acabert - subnet - supernetcontinue - 60M - 14.xls
# script_creator(get_experiments_dir() + "/sep14_v1_acabert_supernet_rsub_60M_14evoit", [ {"exp_name": "v1_60M", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep14_v1_acabert_supernet_rsub_60M_14evoit", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug18_evosearch_acadbertdata_supernet_v1_model_sizes/50M-60M/evo_results_14.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased" })}]} ], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)
# script_creator(get_experiments_dir() + "/sep15_finetune_v1_subnet_finetunekdlogits_60M_14evoit", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli", "qnli", "qqp"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "15-60Mft", "model_name_or_path": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug18_evosearch_acadbertdata_supernet_v1_model_sizes/50M-60M/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=8, generate_port=True)

# v1 - google-trained bert as teacher - kd in finetuning - mnli only (save ckpt, didn't save all ckpt)
# for task, teacher_model_path in [("mnli", "/fsx/ganayu/experiments/supershaper/aug25_finetune_googlebert_mnli_cola_ckptneeded/6_bertbase_mnli_3e-5_16_2")]:
#  script_creator(get_experiments_dir() + "/sep14_directfinetune_v1scratch_%s_hardlogits_saveckpt"%task, create_finetuning_experiments_standalone_vs_supernet_v3(models=[{"exp_name": "sack%s"%task, "model_name_or_path": get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/supernet_continue/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "teacher_model_path": teacher_model_path, "inplace_distillation": "1", "distillation_type": "hard+logits", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5"}], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.distill.supernet_finetune", tasks=[task], num_gpus=8, ckpt_needed=["mnli"]), time_in_mins=8000, wandb="offline", num_gpus=8, generate_port=True)
# v1 - google-trained bert as teacher - kd in finetuning - mnli only
# for task, teacher_model_path in [("mnli", "/fsx/ganayu/experiments/supershaper/aug18_finetune_acabertdata_bertbasestandalone_mnli_ckptneeded/8_bertbase_mnli_3e-5_16_4")]:
#  script_creator(get_experiments_dir() + "/sep17_directfinetune_v1sub_%s_hardlogits"%task, create_finetuning_experiments_standalone_vs_supernet_v2(models=[{"exp_name": "hl%s"%task, "model_name_or_path": get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/supernet_continue/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "teacher_model_path": teacher_model_path, "inplace_distillation": "1", "distillation_type": "hard+logits", "inplace_kd_hard_loss_contrib": "0.5", "inplace_kd_distill_loss_contrib": "0.5"}], sweep=get_finetuning_sweeps("bert"), glue_config="supernetbasev3.standard.distill.supernet_finetune", tasks=[task], num_gpus=4, ckpt_needed=["mnli"]), time_in_mins=1000, wandb="offline", num_gpus=4, generate_port=True, part_size=5)
# script_creator(get_experiments_dir() + "/sep18_finetune_v1_subnet_finetunekdlogits_67M_14evoit", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["mrpc", "rte"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "18-60Mft", "model_name_or_path": get_experiments_dir() + "/sep17_directfinetune_v1sub_mnli_hardlogits/10_hlmnli_mnli_3e-5_32_3", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits"), is_mnli_checkpoint=1), time_in_mins=8000, wandb="offline", num_gpus=8, generate_port=True)

# v1 - moe2e - supernet - different seed - 7620
# script_creator(get_experiments_dir() + "/sep16_supernet_v1_moe_seed333", [ {"exp_name": "%de_%drand"%(nexp, popsize), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep16_supernet_v1_moe_seed333_%de_%drand"%(nexp, popsize), "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": nexp, "pop_size": popsize, "expert_routing_type": "sentence", "seed": 333})}]} for nexp, popsize in [(2, 1)]], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/sep19_finetune_supernet_v1_moe_seed333", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("19-sdmoe", get_experiments_dir() + "/sep16_supernet_v1_moe_seed333/2e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True) # 7802
# script_creator(get_experiments_dir() + "/sep19_finetune_supernet_v1_moe_seed333_useexpert2", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("19-2ndexp", get_experiments_dir() + "/sep16_supernet_v1_moe_seed333/2e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")], use_expert_id=1), time_in_mins=12000, wandb="no", num_gpus=8, generate_port=True) # 7803

# v1 - sandwich - supernet - longer steps - 7621
# script_creator(get_experiments_dir() + "/sept16_supernet_v1_sandwich_2xtrainbudget", [ {"exp_name": "sand250K", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sept16_supernet_v1_sandwich_2xtrainbudget", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_train_steps": 250000})}]}  ], time_in_mins=20000, wandb="online")
# script_creator(get_experiments_dir() + "/sep19_finetune_supernet_v1_sandwich_2xtrainbudget", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("19-sand250K", get_experiments_dir() + "/sept16_supernet_v1_sandwich_2xtrainbudget/sand250K/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=12000, wandb="no", num_gpus=8, generate_port=True) # 7804

# v1 - moe2e - supernet - longer steps - 7622
# script_creator(get_experiments_dir() + "/sep16_supernet_v1_moe_2xtrainbudget", [ {"exp_name": "%de_%drand"%(nexp, popsize), "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep16_supernet_v1_moe_2xtrainbudget_%de_%drand"%(nexp, popsize), "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": nexp, "pop_size": popsize, "expert_routing_type": "sentence", "max_train_steps": 250000})}]} for nexp, popsize in [(2, 1)]], time_in_mins=20000, wandb="online")
# script_creator(get_experiments_dir() + "/sep19_finetune_supernet_v1_moe_2xtrainbudget", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp", "stsb"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("19-moe250K", get_experiments_dir() + "/sep16_supernet_v1_moe_2xtrainbudget/2e_1rand/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=12000, wandb="no", num_gpus=8, generate_port=True) # 7805

# v1.1 - supernet - train
# script_creator(get_experiments_dir() + "/sept16_supernet_v1p1", [ {"exp_name": "v1p1", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sept16_supernet_v1p1", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_train_steps": 125000, "search_space_id": "v1.1"})}]}  ], time_in_mins=8000, wandb="online")
# search 
# script_creator(get_experiments_dir() + "/sep22_evosearch_supernet_v1p1", [ {"exp_name": "1p1_60M-67M", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/sept16_supernet_v1p1/v1p1/best_model", "mixing": "bert-bottleneck", "params_constraints": "60000000,67000000", "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "search_space_id": "v1.1"})}]}, {"exp_name": "1p1_50M-60M", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/sept16_supernet_v1p1/v1p1/best_model", "mixing": "bert-bottleneck", "search_space_id": "v1.1", "params_constraints": "50000000,60000000", "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len")})}]}  ], time_in_mins=8000, wandb="no", num_gpus=8, generate_port=True)
# subnet train
# script_creator(get_experiments_dir() + "/sep24_v1_supernet_v1p1_67M", [ {"exp_name": "v1p1_67M", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep24_v1_supernet_v1p1_67M", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/sept16_supernet_v1p1/v1p1/best_model", "subtransformer_config_path": get_experiments_dir() + "/sep22_evosearch_supernet_v1p1/1p1_60M-67M/evo_results_14.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased" })}]} ], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)
# script_creator(get_experiments_dir() + "/sep24_v1_supernet_v1p1_60M", [ {"exp_name": "v1p1_60M", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep24_v1_supernet_v1p1_60M", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/sept16_supernet_v1p1/v1p1/best_model", "subtransformer_config_path": get_experiments_dir() + "/sep22_evosearch_supernet_v1p1/1p1_50M-60M/evo_results_14.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased" })}]} ], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)
# ft train
# script_creator(get_experiments_dir() + "/sep25_finetune_supernet_v1p1_67M", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["cola", "mnli", "mrpc", "rte", "sst2", "qnli", "qqp"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "25-1p16-7M", "model_name_or_path": get_experiments_dir() + "/sep24_v1_supernet_v1p1_67M/v1p1_67M/best_model", "subtransformer_config_path": get_experiments_dir() + "/sep22_evosearch_supernet_v1p1/1p1_60M-67M/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=25000, wandb="offline", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/sep25_finetune_supernet_v1p1_60M", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mnli", "mrpc", "rte", "sst2", "qnli", "qqp"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "25-1p16-7M", "model_name_or_path": get_experiments_dir() + "/sep24_v1_supernet_v1p1_60M/v1p1_60M/best_model", "subtransformer_config_path": get_experiments_dir() + "/sept16_supernet_v1p1/v1p1/best_model", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=8, generate_port=True)

# moe - supernet - arch routing 
# script_creator(get_experiments_dir() + "/sep19_supernet_v1_moe_archrouting", [ {"exp_name": routing_type, "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "sep19_supernet_v1_moe_archrouting", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "pop_size": 1, "expert_routing_type": routing_type, "max_train_steps": 125000})}]} for routing_type in ["archrouting_1L", "archrouting_2L"] ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/sep23_finetune_v1_moe_archrouting_1L", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("archrouting_1L", get_experiments_dir() + "/sep19_supernet_v1_moe_archrouting/archrouting_1L/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/sep23_finetune_v1_moe_archrouting_2L", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("archrouting_2L", get_experiments_dir() + "/sep19_supernet_v1_moe_archrouting/archrouting_2L/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/sep23_finetune_v1_moe_archrouting_1L_setreqsgrad", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("archrouting_1L", get_experiments_dir() + "/sep19_supernet_v1_moe_archrouting/archrouting_1L/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True, torch_distributed_debug="detail")
# script_creator(get_experiments_dir() + "/sep23_finetune_v1_moe_archrouting_2L_setreqsgrad", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("archrouting_2L", get_experiments_dir() + "/sep19_supernet_v1_moe_archrouting/archrouting_2L/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)

# v1 - moe 2e - kd during FT
# script_creator(get_experiments_dir() + "/sep23_finetune_moe_2e_kd", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mnli", "mrpc", "rte", "sst2", "qnli", "qqp"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "23-2ekd", "model_name_or_path": get_experiments_dir() + "/sep6_acadbertdata_supernet_v1_moe_minmaxrand_corrected/2e_1rand/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=8, generate_port=True)

# v1 - supernet - kd during FT
# script_creator(get_experiments_dir() + "/sep23_finetune_supernet_kd", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mnli", "mrpc", "rte", "sst2", "qnli", "qqp"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "23-kd", "model_name_or_path": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=8, generate_port=True)

# v1 - scratch - kd during FT
# script_creator(get_experiments_dir() + "/sep23_finetune_scratch_kd", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mnli", "mrpc", "rte", "sst2", "qnli", "qqp"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "23-sckd", "model_name_or_path": get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/retrain_scratch/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=8, generate_port=True)

# v1 - moe 2e - kd during FT
# script_creator(get_experiments_dir() + "/sep23_finetune_moe_2e_kd_seed333", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mnli", "mrpc", "rte", "sst2", "qnli", "qqp"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "23-2ekd", "model_name_or_path": get_experiments_dir() + "/sep16_supernet_v1_moe_seed333/2e_1rand/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=8, generate_port=True)

# moe - supernet - arch jack routing 
# script_creator(get_experiments_dir() + "/oct14_supernet_v1_moe_archrouting_jack", [ {"exp_name": routing_type, "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "oct14_supernet_v1_moe_archrouting_jack_%s"%routing_type, "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "pop_size": 1, "expert_routing_type": routing_type, "max_train_steps": 125000})}]} for routing_type in ["archrouting_jack_1L", "archrouting_jack_2L"] ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/oct16_finetune_v1_moe_archrouting_jack_1L", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("ftjack1L", get_experiments_dir() + "/oct14_supernet_v1_moe_archrouting_jack/archrouting_jack_1L/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/oct16_finetune_v1_moe_archrouting_jack_2L", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("ftjack2L", get_experiments_dir() + "/oct14_supernet_v1_moe_archrouting_jack/archrouting_jack_2L/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)

# moe - evo search - rand moe - ffn cropping
# script_creator(get_experiments_dir() + "/oct16_rand2moe_ffncropping_evosearch", [ {"exp_name": "ffncrop", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/sep6_acadbertdata_supernet_v1_moe_minmaxrand_corrected/2e_1rand/best_model", "mixing": "bert-bottleneck", "params_constraints": "0,250000000", "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "expert_search": "yes", "max_experts": 2, "expert_routing_type": "sentence", "last_expert_averaging_expert": "no", "fixed_arch": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls" })}]} ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/oct16_rand2moe_ffncropping_evosearch_seed333", [ {"exp_name": "ffncrop", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/sep16_supernet_v1_moe_seed333/2e_1rand/best_model", "mixing": "bert-bottleneck", "params_constraints": "0,250000000", "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "expert_search": "yes", "max_experts": 2, "expert_routing_type": "sentence", "last_expert_averaging_expert": "no", "fixed_arch": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls" })}]} ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# moe - finetune - ffn cropping
# script_creator(get_experiments_dir() + "/oct17_finetune_rand2moe_ffncropping_evosearch", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("fcrps1", get_experiments_dir() + "/sep6_acadbertdata_supernet_v1_moe_minmaxrand_corrected/2e_1rand/best_model", get_experiments_dir() + "/oct16_rand2moe_ffncropping_evosearch/ffncrop/evo_results_29.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/oct17_finetune_rand2moe_ffncropping_evosearch_seed333", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("fcrps2", get_experiments_dir() + "/sep16_supernet_v1_moe_seed333/2e_1rand/best_model", get_experiments_dir() + "/oct16_rand2moe_ffncropping_evosearch_seed333/ffncrop/evo_results_29.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)

# moe - supernet - arch jack routing - relu layer bug corrected
# script_creator(get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L", [ {"exp_name": "archrouting_jack_2L_hyp%d"%hypernet_hidden_size, "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "oct18_supernet_v1_moe_archrouting_jack_2L_hyp%d"%hypernet_hidden_size, "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "hypernet_hidden_size": hypernet_hidden_size, "pop_size": 1, "expert_routing_type": "archrouting_jack_2L", "max_train_steps": 125000})}]} for hypernet_hidden_size in [64, 128] ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/oct20_finetune_moe_archrouting_jack_2L_64", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("2L_64", get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp64/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/oct20_finetune_moe_archrouting_jack_2L_128", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("2L_128", get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L_seed333", [ {"exp_name": "archrouting_jack_2L_hyp%d"%hypernet_hidden_size, "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "oct18_supernet_v1_moe_archrouting_jack_2L_hyp%d"%hypernet_hidden_size, "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "hypernet_hidden_size": hypernet_hidden_size, "pop_size": 1, "expert_routing_type": "archrouting_jack_2L", "max_train_steps": 125000, "seed": 333})}]} for hypernet_hidden_size in [64] ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/oct20_finetune_moe_archrouting_jack_2L_seed333", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("2L_64", get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L_seed333/archrouting_jack_2L_hyp64/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)

# moe - supernet - arch jack routing - fixed arch
# script_creator(get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L_fixedarch", [ {"exp_name": "archrouting_jack_2L_fixedarch_hyp%d"%hypernet_hidden_size, "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "oct18_supernet_v1_moe_archrouting_jack_2L_fixedarch_hyp%d"%hypernet_hidden_size, "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "hypernet_hidden_size": hypernet_hidden_size, "pop_size": 1, "expert_routing_type": "archrouting_jack_2L", "max_train_steps": 125000, "fixed_hypernet_input": "yes"})}]} for hypernet_hidden_size in [64, 128] ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/oct20_finetune_moe_archrouting_jack_2L_fixedarch_64", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("2L_64", get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L_fixedarch/archrouting_jack_2L_fixedarch_hyp64/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/oct20_finetune_moe_archrouting_jack_2L_fixedarch_128", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("2L_128", get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L_fixedarch/archrouting_jack_2L_fixedarch_hyp128/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L_moreexperts", [ {"exp_name": "archrouting_jack_2L_exp%d"%n_experts, "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "oct18_supernet_v1_moe_archrouting_jack_2L_moreexperts_exp%d"%n_experts, "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": n_experts, "hypernet_hidden_size": 64, "pop_size": 1, "expert_routing_type": "archrouting_jack_2L", "max_train_steps": 125000})}]} for n_experts in [4, 6] ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/oct21_finetune_moe_archrouting_jack_2L_moreexperts_exp4", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("exp4", get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L_moreexperts/archrouting_jack_2L_exp4/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/oct21_finetune_moe_archrouting_jack_2L_moreexperts_exp6", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("exp6", get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L_moreexperts/archrouting_jack_2L_exp6/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")]), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)

# ft - different finetuning seeds (same seed 333)
# script_creator(get_experiments_dir() + "/oct21_finetune_moe_archrouting_jack_2L_128_seed333", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("2L_128", get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")], seed=333), time_in_mins=15000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/oct21_finetune_scratch_seed333", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("scratch", get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/retrain_scratch/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")], seed=333), time_in_mins=10000, wandb="no", num_gpus=8, generate_port=True)
# script_creator(get_experiments_dir() + "/oct21_finetune_supernet_seed333", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=8, tasks=["cola", "mrpc", "rte", "sst2", "mnli",  "qnli", "qqp"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("supernet", get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")], seed=333), time_in_mins=10000, wandb="no", num_gpus=8, generate_port=True)

# ft - different finetuning seeds (same seed 444)
# script_creator(get_experiments_dir() + "/oct27_finetune_moe_archrouting_jack_2L_128_seed444", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["cola", "mrpc", "rte", "sst2"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("2L_128", get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")], seed=444), time_in_mins=10000, wandb="no", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/oct27_finetune_scratch_seed444", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["cola", "mrpc", "rte", "sst2"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("scratch", get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/retrain_scratch/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")], seed=444), time_in_mins=10000, wandb="no", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/oct27_finetune_supernet_seed444", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["cola", "mrpc", "rte", "sst2"], glue_config="supernetbasev3.standard.train_glue_original", sweep=get_finetuning_sweeps("bert"), models=[ ("supernet", get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "bert-base-uncased")], seed=444), time_in_mins=10000, wandb="no", num_gpus=4, generate_port=True)

# archmoe_pavg - ft w kd 
# script_creator(get_experiments_dir() + "/oct28_finetune_archmoe_pavg_2e_kd_seed333", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["cola", "mnli", "mrpc", "rte", "sst2", "qnli", "qqp"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "28-archkd", "model_name_or_path": get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=25000, wandb="offline", num_gpus=4, generate_port=True)

# moe - supernet - arch jack routing + jack - onehot input embedding
# script_creator(get_experiments_dir() + "/oct28_supernet_v1_archpavg_2e_onehot", [{"exp_name": "archpavg_2e_onehot", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "archpavg_2e_onehot", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "hypernet_hidden_size": 128, "pop_size": 1, "expert_routing_type": "archrouting_jack_2L", "max_train_steps": 125000, "hypernet_input_format": "onehot"})}]}] , time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/oct28_finetune_archpavg_2e_onehot_kd_seed333", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["cola", "mnli", "mrpc", "rte", "sst2", "qnli", "qqp"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "28-onehot", "model_name_or_path": get_experiments_dir() + "/oct28_supernet_v1_archpavg_2e_onehot/archpavg_2e_onehot/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=25000, wandb="offline", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/oct28_finetune_archpavg_2e_onehot_kd_seed333_part2", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=["mrpc", "rte"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "28-onehot", "model_name_or_path": get_experiments_dir() + "/oct28_supernet_v1_archpavg_2e_onehot/archpavg_2e_onehot/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits"), mnli_ckpt="/fsx/ganayu/experiments/supershaper/sep23_mnlikd_models_master/f875d060-8699-42c8-9002-4d42cef1d2c2"), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# ft kd - debug
# script_creator(get_experiments_dir() + "/sep23_finetune_scratch_kd_corrected", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["cola", "mnli", "mrpc", "rte", "sst2", "qnli", "qqp"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "23-sckd", "model_name_or_path": get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/retrain_scratch/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=25000, wandb="offline", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/sep23_finetune_scratch_kd_corrected_cola_seed444", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=["cola"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "23-sckd", "model_name_or_path": get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/retrain_scratch/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits"), seed=444), time_in_mins=5000, wandb="offline", num_gpus=2, generate_port=True)
# script_creator(get_experiments_dir() + "/sep23_finetune_scratch_kd_corrected_part2", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=["mrpc", "rte"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "scratch", "model_name_or_path": get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/retrain_scratch/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits"), mnli_ckpt="/fsx/ganayu/experiments/supershaper/sep23_mnlikd_models_master/bfe5aba3-ae3e-43e9-be71-d74aeb3c88a0"), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)
# script_creator(get_experiments_dir() + "/sep23_finetune_scratch_kd_corrected_part2_mrpc_rte_seed444", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=["mrpc", "rte"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "scratch", "model_name_or_path": get_experiments_dir() + "/aug13_v1_acadbertdata_supernet_retrain_subnet_125Ksteps/retrain_scratch/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits"), mnli_ckpt="/fsx/ganayu/experiments/supershaper/sep23_mnlikd_models_master/bfe5aba3-ae3e-43e9-be71-d74aeb3c88a0", seed=444), time_in_mins=5000, wandb="offline", num_gpus=2, generate_port=True)
# script_creator(get_experiments_dir() + "/sep23_finetune_supernet_kd_corrected", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["cola", "mnli", "mrpc", "rte", "sst2", "qnli", "qqp"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "23-kd", "model_name_or_path": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=25000, wandb="offline", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/sep23_finetune_supernet_kd_corrected_cola_seed444", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=["cola"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "23-kd", "model_name_or_path": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits"), seed=444), time_in_mins=5000, wandb="offline", num_gpus=2, generate_port=True)
# script_creator(get_experiments_dir() + "/sep23_finetune_supernet_kd_corrected_part2", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=["mrpc", "rte"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "supernet", "model_name_or_path": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits"), mnli_ckpt="/fsx/ganayu/experiments/supershaper/sep23_mnlikd_models_master/ba1b4cba-f28e-4ce5-8f97-0d48157d3cae"), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)
# script_creator(get_experiments_dir() + "/sep23_finetune_supernet_kd_corrected_part2_mrpc_rte_seed444", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=["mrpc", "rte"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "supernet", "model_name_or_path": get_experiments_dir() + "/aug9_acadbertdata_supernet_v1/acav1/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits"), mnli_ckpt="/fsx/ganayu/experiments/supershaper/sep23_mnlikd_models_master/ba1b4cba-f28e-4ce5-8f97-0d48157d3cae", seed=444), time_in_mins=5000, wandb="offline", num_gpus=2, generate_port=True)
# script_creator(get_experiments_dir() + "/oct28_finetune_archmoe_pavg_2e_kd_seed333_corrected", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["cola", "mnli", "mrpc", "rte", "sst2", "qnli", "qqp"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "28-archkd", "model_name_or_path": get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=25000, wandb="offline", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/oct28_finetune_archmoe_pavg_2e_kd_seed333_corrected_cola_seed444", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=["cola",], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "28-archkd", "model_name_or_path": get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits"), seed=444), time_in_mins=5000, wandb="offline", num_gpus=2, generate_port=True)
# script_creator(get_experiments_dir() + "/oct28_finetune_archmoe_pavg_2e_kd_seed333_corrected_part2", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["mrpc", "rte"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "archmoe", "model_name_or_path": get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits"), mnli_ckpt="/fsx/ganayu/experiments/supershaper/sep23_mnlikd_models_master/979e3173-b0d7-4aeb-b265-a8cece5651c5"), time_in_mins=15000, wandb="offline", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/oct28_finetune_archmoe_pavg_2e_kd_seed333_corrected_part2_mrpc_rte_seed444", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=["mrpc", "rte"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "archmoe", "model_name_or_path": get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits"), mnli_ckpt="/fsx/ganayu/experiments/supershaper/sep23_mnlikd_models_master/979e3173-b0d7-4aeb-b265-a8cece5651c5", seed=444), time_in_mins=5000, wandb="offline", num_gpus=2, generate_port=True) # error, forgot 444 seed

# evo_ours -> 50M (autodistill, archmoe)
# script_creator(get_experiments_dir() + "/nov6_archmoe_50M", [ {"exp_name": name, "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", "mixing": "bert-bottleneck", "params_constraints": constraint, "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "max_experts": 2, "expert_routing_type": "archrouting_jack_2L", "last_expert_averaging_expert": "no", "hypernet_hidden_size": 128})}]} for name, constraint in [("50M", "45000000,50000000")] ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/nov6_finetune_archmoe_50M", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["cola", "sst2", "qnli", "qqp"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "6-p1", "model_name_or_path": get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", "subtransformer_config_path": get_experiments_dir() + "/nov6_archmoe_50M/50M/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=25000, wandb="offline", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/nov6_finetune_archmoe_50M_part2", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["mnli", "mrpc", "rte"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "6-p2", "model_name_or_path": get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", "subtransformer_config_path": get_experiments_dir() + "/nov6_archmoe_50M/50M/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=20000, wandb="offline", num_gpus=4, generate_port=True)
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:
#  script_creator(get_experiments_dir() + "/nov19_finetune_archmoe_50M_%s_largerepochs"%exp_name, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("autodistill"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", "subtransformer_config_path": get_experiments_dir() + "/nov6_archmoe_50M/50M/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# evo_ours -> 27M
# script_creator(get_experiments_dir() + "/nov6_archmoe_27M", [ {"exp_name": name, "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", "mixing": "bert-bottleneck", "params_constraints": constraint, "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "max_experts": 2, "expert_routing_type": "archrouting_jack_2L", "last_expert_averaging_expert": "no", "hypernet_hidden_size": 128})}]} for name, constraint in [("27M", "25000000,27000000")] ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)

# neuronrouting_jack_2L
# script_creator(get_experiments_dir() + "/nov10_neuronrouting_jack_2L", [ {"exp_name": "neuron", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "nov10_neuronrouting_jack_2L", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "hypernet_hidden_size": 128, "pop_size": 1, "expert_routing_type": "neuronrouting_jack_2L", "max_train_steps": 125000})}]}  ], time_in_mins=10000, wandb="online")
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:
#   script_creator(get_experiments_dir() + "/nov12_finetune_neuron_2L_%s"%exp_name, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/nov10_neuronrouting_jack_2L/neuron/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# comparison to nas-bert
# evo_ours -> 60M
# script_creator(get_experiments_dir() + "/nov6_archmoe_60M", [ {"exp_name": name, "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", "mixing": "bert-bottleneck", "params_constraints": constraint, "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "max_experts": 2, "expert_routing_type": "archrouting_jack_2L", "last_expert_averaging_expert": "no", "hypernet_hidden_size": 128})}]} for name, constraint in [("60M", "55000000,60000000")] ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/nov6_standalone_archmoe_60M", [ {"exp_name": "stand_60M", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "nov6_standalone_archmoe_60M", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", "subtransformer_config_path": get_experiments_dir() + "/nov6_archmoe_60M/60M/evo_results_29.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased" })}]} ], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:
#  script_creator(get_experiments_dir() + "/nov16_finetune_standalone_archmoe_60M_%s"%exp_name, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/nov6_standalone_archmoe_60M/stand_60M/best_model", "subtransformer_config_path": get_experiments_dir() + "/nov6_archmoe_60M/60M/evo_results_29.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# standalone - 67M - 3x budget
# script_creator(get_experiments_dir() + "/nov12_standalone_67M_3xbudget", [ {"exp_name": "stand67M", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("bertbasev3.standard.train_mlm"), {"subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "initialize_pretrained_weights": "no", "max_train_steps": 375000, "mixing": "bert-bottleneck", "experiment_name": "nov12_standalone_67M_3xbudget_stand67M"})}]}], time_in_mins=15000, wandb="online", generate_port=True, num_gpus=8)
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:
#  script_creator(get_experiments_dir() + "/nov12_finetune_standalone_67M_3xbudget_%s"%exp_name, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/nov12_standalone_67M_3xbudget/stand67M/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# mnli as proxy task (archmoe)
# script_creator(get_experiments_dir() + "/nov13_finetune_archmoe_supernet_proxy_mnli", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["mnli"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "archmoe", "model_name_or_path": get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", "subtransformer_config_path": get_experiments_dir() + "/nov13_mnli_proxy/supernet_config.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=25000, wandb="offline", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/nov13_evosearch_archmoe_supernet_proxy_mnli", [ {"exp_name": "evomnli", "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/sep23_mnlikd_models_master/de93c039-bebf-45da-acaa-620f9df2f430/mnli_best", "mixing": "bert-bottleneck", "finetune": "mnli", "per_device_eval_batch_size": 128, "fitness_metric": "accuracy", "evo_iter": "150", "params_constraints": "45000000,50000000"})} ]} ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:
#  script_creator(get_experiments_dir() + "/nov16_finetune_evosearch_archmoe_supernet_proxy_mnli_%s"%exp_name, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", "subtransformer_config_path": get_experiments_dir() + "/nov13_evosearch_archmoe_supernet_proxy_mnli/evomnli/evo_results_149.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# mnli as proxy task (neuronmoe)
# script_creator(get_experiments_dir() + "/nov13_finetune_neuromoe_supernet_proxy_mnli", create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=["mnli"], glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": "neuromoe", "model_name_or_path": get_experiments_dir() + "/nov10_neuronrouting_jack_2L/neuron/best_model", "subtransformer_config_path": get_experiments_dir() + "/nov13_mnli_proxy/supernet_config.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=25000, wandb="offline", num_gpus=4, generate_port=True)

# collapse_experts_before_ft==1
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:
#  script_creator(get_experiments_dir() + "/nov19_collapse_and_finetune_neuronmoe_2L_%s"%exp_name, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/nov10_neuronrouting_jack_2L/neuron/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits"), collapse_experts_before_ft=1), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:
#  script_creator(get_experiments_dir() + "/nov19_collapse_and_finetune_archmoe_2L_%s"%exp_name, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits"), collapse_experts_before_ft=1), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:
#  script_creator(get_experiments_dir() + "/nov20_collapse_and_finetune_archmoe_50M_%s"%exp_name, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L/archrouting_jack_2L_hyp128/best_model", "subtransformer_config_path": get_experiments_dir() + "/nov6_archmoe_50M/50M/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits"), collapse_experts_before_ft=1), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:
#   script_creator(get_experiments_dir() + "/nov20_collapse_and_finetune_standalone_archmoe_60M_%s"%exp_name, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/nov6_standalone_archmoe_60M/stand_60M/best_model", "subtransformer_config_path": get_experiments_dir() + "/nov6_archmoe_60M/60M/evo_results_29.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits"), collapse_experts_before_ft=1), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# comparison to nas-bert (neuron-moe)
# evo_ours -> 60M
# script_creator(get_experiments_dir() + "/nov21_neuronmoe_60M", [ {"exp_name": name, "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/nov10_neuronrouting_jack_2L/neuron/best_model", "mixing": "bert-bottleneck", "params_constraints": constraint, "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "max_experts": 2, "expert_routing_type": "neuronrouting_jack_2L", "last_expert_averaging_expert": "no", "hypernet_hidden_size": 128})}]} for name, constraint in [("60M", "55000000,60000000")] ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# script_creator(get_experiments_dir() + "/nov21_standalone_neuronmoe_60M", [ {"exp_name": "stand_60M", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "nov21_standalone_neuronmoe_60M", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/nov10_neuronrouting_jack_2L/neuron/best_model", "subtransformer_config_path": get_experiments_dir() + "/nov21_neuronmoe_60M/60M/evo_results_29.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased" })}]} ], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:
#  script_creator(get_experiments_dir() + "/nov21_finetune_standalone_neuronmoe_60M_%s"%exp_name, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/nov21_standalone_neuronmoe_60M/stand_60M/best_model", "subtransformer_config_path": get_experiments_dir() + "/nov21_neuronmoe_60M/60M/evo_results_29.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# evo_ours -> 50M (autodistill, neuronmoe)
# script_creator(get_experiments_dir() + "/nov21_neuronmoe_50M", [ {"exp_name": name, "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/nov10_neuronrouting_jack_2L/neuron/best_model", "mixing": "bert-bottleneck", "params_constraints": constraint, "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "max_experts": 2, "expert_routing_type": "neuronrouting_jack_2L", "last_expert_averaging_expert": "no", "hypernet_hidden_size": 128})}]} for name, constraint in [("50M", "45000000,50000000")] ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:
#  script_creator(get_experiments_dir() + "/nov22_finetune_neuronmoe_50M_%s"%exp_name, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("autodistill"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/nov10_neuronrouting_jack_2L/neuron/best_model", "subtransformer_config_path": get_experiments_dir() + "/nov21_neuronmoe_50M/50M/evo_results_29.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# neuronrouting_jack_2L (larger lr: 8e-4)
# script_creator(get_experiments_dir() + "/nov23_neuronrouting_jack_2L_lr8e-4", [  {"exp_name": "neuron", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "nov23_neuronrouting_jack_2L_lr8e-4", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "hypernet_hidden_size": 128, "pop_size": 1, "learning_rate": 8e-4, "expert_routing_type": "neuronrouting_jack_2L", "max_train_steps": 125000})}]}  ], time_in_mins=10000, wandb="online")
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:      
# script_creator(get_experiments_dir() + "/nov26_finetune_neuronmoe_67M_%s_lr8e-4"%exp_name, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/nov23_neuronrouting_jack_2L_lr8e-4/neuron/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# neuronrouting_jack_drop_2L
# script_creator(get_experiments_dir() + "/nov26_neuronrouting_jack_drop_2L", [ {"exp_name": "neuron", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "nov26_neuronrouting_jack_drop_2L", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "hypernet_hidden_size": 128, "pop_size": 1, "expert_routing_type": "neuronrouting_jack_drop_2L", "max_train_steps": 125000})}]}  ], time_in_mins=10000, wandb="online")
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:      
#  script_creator(get_experiments_dir() + "/nov26_finetune_neuronmoe_drop_67M_%s"%exp_name, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/nov26_neuronrouting_jack_drop_2L/neuron/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# v1.2 - supernet training -- ffn elastic # bug: smallest transformer has inter_size as 3072 (not 2*120)
# script_creator(get_experiments_dir() + "/dec2_neuronrouting_v1p2", [ {"exp_name": "neuron", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "dec2_neuronrouting_v1p2", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "hypernet_hidden_size": 128, "pop_size": 1, "expert_routing_type": "neuronrouting_jack_2L", "max_train_steps": 125000, "search_space_id": "v1.2"})}]}  ], time_in_mins=10000, wandb="online")
# evosearch
# script_creator(get_experiments_dir() + "/dec5_evosearch_neuronmoe_v1p2_27M", [ {"exp_name": name, "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/dec2_neuronrouting_v1p2/neuron/best_model", "mixing": "bert-bottleneck", "params_constraints": constraint, "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "max_experts": 2, "expert_routing_type": "neuronrouting_jack_2L", "last_expert_averaging_expert": "no", "hypernet_hidden_size": 128, "search_space_id": "v1.2", "include_smallest_arch_into_initial_popu": "yes"})}]} for name, constraint in [("27M", "26000000,27000000")] ], time_in_mins=8000, wandb="no", num_gpus=8, generate_port=True)
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:      
# script_creator(get_experiments_dir() + "/dec6_finetune_neuronmoe_27M_%s"%exp_name, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/dec2_neuronrouting_v1p2/neuron/best_model", "subtransformer_config_path": get_experiments_dir() + "/dec5_evosearch_neuronmoe_v1p2_27M/27M/evo_results_9.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "search_space_id": "v1.2"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# v1.2 - supernet training -- ffn elastic # bug corrected
# script_creator(get_experiments_dir() + "/dec5_neuronrouting_v1p2_corrected", [ {"exp_name": "neuron", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "dec2_neuronrouting_v1p2", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "hypernet_hidden_size": 128, "pop_size": 1, "expert_routing_type": "neuronrouting_jack_2L", "max_train_steps": 125000, "search_space_id": "v1.2"})}]}  ], time_in_mins=10000, wandb="online")
# evosearch
# script_creator(get_experiments_dir() + "/dec5_evosearch_neuronmoe_v1p2_corrected_27M", [ {"exp_name": name, "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/dec5_neuronrouting_v1p2_corrected/neuron/best_model", "mixing": "bert-bottleneck", "params_constraints": constraint, "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "max_experts": 2, "expert_routing_type": "neuronrouting_jack_2L", "last_expert_averaging_expert": "no", "hypernet_hidden_size": 128, "search_space_id": "v1.2", "mutation_prob": 0.1, "include_smallest_arch_into_initial_popu": "yes"})}]} for name, constraint in [("27M", "26000000,27000000")] ], time_in_mins=8000, wandb="no", num_gpus=8, generate_port=True)
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:      
#  script_creator(get_experiments_dir() + "/dec8_finetune_neuronmoe_corrected_27M_%s"%exp_name, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/dec5_neuronrouting_v1p2_corrected/neuron/best_model", "subtransformer_config_path": get_experiments_dir() + "/dec5_evosearch_neuronmoe_v1p2_corrected_27M/27M/evo_results_29.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck", "search_space_id": "v1.2"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# evo_ours -> 50M larger lr
# script_creator(get_experiments_dir() + "/dec7_neuronmoe_50M_lr8e-4", [ {"exp_name": name, "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/nov23_neuronrouting_jack_2L_lr8e-4/neuron/best_model", "mixing": "bert-bottleneck", "params_constraints": constraint, "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "max_experts": 2, "expert_routing_type": "neuronrouting_jack_2L", "last_expert_averaging_expert": "no", "hypernet_hidden_size": 128})}]} for name, constraint in [("50M", "45000000,50000000")] ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:
#  script_creator(get_experiments_dir() + "/dec8_finetune_neuronmoe_50M_%s_lr8e-4"%exp_name, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/nov23_neuronrouting_jack_2L_lr8e-4/neuron/best_model", "subtransformer_config_path": get_experiments_dir() + "/dec7_neuronmoe_50M_lr8e-4/50M/evo_results_29.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# neuron jack drop 2L - dropout layer removed
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:       
#  script_creator(get_experiments_dir() + "/dec7_finetune_neuronmoe_drop_corrected_67M_%s"%exp_name, create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/nov26_neuronrouting_jack_drop_2L/neuron/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "bert-bottleneck"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# neuron jack drop 2L - v5.1
# script_creator(get_experiments_dir() + "/dec15_neuronrouting_v5p1", [ {"exp_name": "5p1", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "dec15_neuronrouting_v5p1", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "hypernet_hidden_size": 128, "pop_size": 1, "expert_routing_type": "neuronrouting_jack_2L", "max_train_steps": 125000, "search_space_id": "v5.1", "mixing": "attention"})}]}  ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/dec15_neuronrouting_v5p2", [ {"exp_name": "5p1", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "dec15_neuronrouting_v5p2", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "hypernet_hidden_size": 128, "pop_size": 1, "expert_routing_type": "neuronrouting_jack_2L", "max_train_steps": 125000, "search_space_id": "v5.2", "mixing": "attention"})}]}  ], time_in_mins=10000, wandb="online")
# script_creator(get_experiments_dir() + "/dec18_evosearch_neuronmoe_v5_27M_10M_5M_50M", [ {"exp_name": name, "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/dec15_neuronrouting_v5p1/5p1/best_model", "mixing": "attention", "params_constraints": constraint, "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "max_experts": 2, "expert_routing_type": "neuronrouting_jack_2L", "last_expert_averaging_expert": "no", "hypernet_hidden_size": 128, "search_space_id": "v5.1", "enumerate_all": "yes"})}]} for name, constraint in [("27M", "25000000,26800000"), ("10M", "8000000,10000000"), ("5M", "4000000,5137560"), ("50M", "48000000,50000000")] ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
#for model_size in ["50M", "10M"]: # : # done: "5M", "27M"
#  for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:       
#    script_creator(get_experiments_dir() + "/dec19_finetune_neuronmoe_v5/%s_%s"%(model_size, exp_name), create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/dec15_neuronrouting_v5p1/5p1/best_model", "subtransformer_config_path": get_experiments_dir() + "/dec18_evosearch_neuronmoe_v5_27M_10M_5M_50M/%s/evo_results_0.xls"%model_size, "tokenizer_name": "bert-base-uncased", "mixing": "attention"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)
# script_creator(get_experiments_dir() + "/dec20_evosearch_neuronmoe_v5_30M_60M_67M", [ {"exp_name": name, "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/dec15_neuronrouting_v5p1/5p1/best_model", "mixing": "attention", "params_constraints": constraint, "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "max_experts": 2, "expert_routing_type": "neuronrouting_jack_2L", "last_expert_averaging_expert": "no", "hypernet_hidden_size": 128, "search_space_id": "v5.1", "enumerate_all": "yes"})}]} for name, constraint in [("30M", "28000000,30000000"), ("60M", "58000000,60000000"), ("67M", "65000000,66500000")] ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# for model_size in ["30M", "60M", "67M"]: # : # done: "5M", "27M"
#  for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:       
#    script_creator(get_experiments_dir() + "/dec21_finetune_neuronmoe_v5/%s_%s"%(model_size, exp_name), create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/dec15_neuronrouting_v5p1/5p1/best_model", "subtransformer_config_path": get_experiments_dir() + "/dec20_evosearch_neuronmoe_v5_30M_60M_67M/%s/evo_results_0.xls"%model_size, "tokenizer_name": "bert-base-uncased", "mixing": "attention"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# neuron jack drop 2L - v5.2
# script_creator(get_experiments_dir() + "/dec22_neuronrouting_v5p2", [ {"exp_name": "5p2", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "dec22_neuronrouting_v5p2", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "hypernet_hidden_size": 128, "pop_size": 1, "expert_routing_type": "neuronrouting_jack_2L", "max_train_steps": 125000, "search_space_id": "v5.2", "mixing": "attention"})}]}  ], time_in_mins=8000, wandb="online")
# script_creator(get_experiments_dir() + "/dec24_evosearch_neuronmoe_v5p2_27M_10M_5M_50M_30M_60M_67M", [ {"exp_name": name, "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/dec22_neuronrouting_v5p2/5p2/best_model", "mixing": "attention", "params_constraints": constraint, "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "max_experts": 2, "expert_routing_type": "neuronrouting_jack_2L", "last_expert_averaging_expert": "no", "hypernet_hidden_size": 128, "search_space_id": "v5.2", "enumerate_all": "yes"})}]} for name, constraint in [("27M", "25000000,26800000"), ("10M", "8000000,10000000"), ("5M", "4000000,5137560"), ("50M", "48000000,50000000"), ("30M", "28000000,30000000"), ("60M", "58000000,60000000"), ("67M", "65000000,66500000")] ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# for model_size in ["10M", "5M", "50M", "30M", "60M", "67M"]:  # "27M"
#  for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:       
    # script_creator(get_experiments_dir() + "/dec24_finetune_neuronmoe_v5p2/%s_%s"%(model_size, exp_name), create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=4, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/dec22_neuronrouting_v5p2/5p2/best_model", "subtransformer_config_path": get_experiments_dir() + "/dec24_evosearch_neuronmoe_v5p2_27M_10M_5M_50M_30M_60M_67M/%s/evo_results_0.xls"%model_size, "tokenizer_name": "bert-base-uncased", "mixing": "attention"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=4, generate_port=True) # 4gpu
#     script_creator(get_experiments_dir() + "/dec24_finetune_neuronmoe_v5p2/%s_%s"%(model_size, exp_name), create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/dec22_neuronrouting_v5p2/5p2/best_model", "subtransformer_config_path": get_experiments_dir() + "/dec24_evosearch_neuronmoe_v5p2_27M_10M_5M_50M_30M_60M_67M/%s/evo_results_0.xls"%model_size, "tokenizer_name": "bert-base-uncased", "mixing": "attention"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True) 

# neuron jack drop 2L - v5.1 - long training
# script_creator(get_experiments_dir() + "/dec26_neuronrouting_v5p1_250Ksteps", [ {"exp_name": "5p1", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "dec26_neuronrouting_v5p1", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "hypernet_hidden_size": 128, "pop_size": 1, "expert_routing_type": "neuronrouting_jack_2L", "max_train_steps": 250000, "search_space_id": "v5.1", "mixing": "attention"})}]}  ], time_in_mins=15000, wandb="online")
# script_creator(get_experiments_dir() + "/dec29_evosearch_neuronmoe_v5_250Ksteps_27M_50M", [ {"exp_name": name, "runs": [ {"pyfile": "evo_ours.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.evosearch"), {"supernet_ckpt_dir": get_experiments_dir() + "/dec26_neuronrouting_v5p1_250Ksteps/5p1/best_model", "mixing": "attention", "params_constraints": constraint, "data_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "max_experts": 2, "expert_routing_type": "neuronrouting_jack_2L", "last_expert_averaging_expert": "no", "hypernet_hidden_size": 128, "search_space_id": "v5.1", "enumerate_all": "yes"})}]} for name, constraint in [("27M", "25000000,26800000"), ("50M", "48000000,50000000")] ], time_in_mins=8000, wandb="no", num_gpus=4, generate_port=True)
# for model_size in ["27M", "50M"]:  
#  for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:
#    script_creator(get_experiments_dir() + "/dec26_finetune_neuronmoe_v5p1_250Ksteps/%s_%s"%(model_size, exp_name), create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/dec26_neuronrouting_v5p1_250Ksteps/5p1/best_model", "subtransformer_config_path": get_experiments_dir() + "/dec29_evosearch_neuronmoe_v5_250Ksteps_27M_50M/%s/evo_results_0.xls"%model_size, "tokenizer_name": "bert-base-uncased", "mixing": "attention"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True) 

# fixed - archrouting
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:
# script_creator(get_experiments_dir() + "/jan3_finetune_archmoe_jack_2L_fixedarch/%s"%(exp_name), create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/oct18_supernet_v1_moe_archrouting_jack_2L_fixedarch/archrouting_jack_2L_fixedarch_hyp128/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "attention"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True) 

# fixed - neuronrouting
# script_creator(get_experiments_dir() + "/jan3_supernet_neuronmoe_jack_2L_fixedarch", [ {"exp_name": "neuronfixed", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "jan3_supernet_neuronmoe_jack_2L_fixedarch%d"%hypernet_hidden_size, "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "max_experts": 2, "hypernet_hidden_size": hypernet_hidden_size, "pop_size": 1, "expert_routing_type": "neuronrouting_jack_2L", "max_train_steps": 125000, "fixed_hypernet_input": "yes"})}]} for hypernet_hidden_size in [128] ], time_in_mins=10000, wandb="online")
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:
#  script_creator(get_experiments_dir() + "/jan3_finetune_neuronmoe_jack_2L_fixedarch/%s"%(exp_name), create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/jan3_supernet_neuronmoe_jack_2L_fixedarch/neuronfixed/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "attention"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# neuron moe - 60M - collapse and subnet train - v1
# script_creator(get_experiments_dir() + "/jan4_standalone_neuronmoe_60M", [ {"exp_name": "stand_60M", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "nov21_standalone_neuronmoe_60M", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/nov10_neuronrouting_jack_2L/neuron/best_model", "subtransformer_config_path": get_experiments_dir() + "/nov21_neuronmoe_60M/60M/evo_results_29.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "tokenizer_name": "bert-base-uncased", "collapse_experts_before_ft": 1})}]} ], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)
# for exp_name, tasks in [("mnli_mrpc_rte", ["mnli", "mrpc", "rte"]), ("cola_qqp", ["cola", "qqp"]), ("sst2_qnli", ["sst2", "qnli"])]:
#  script_creator(get_experiments_dir() + "/jan3_finetune_neuronmoe_jack_2L_fixedarch/%s"%(exp_name), create_finetuning_experiments_standalone_vs_supernet_v3(num_gpus=2, tasks=tasks, glue_config="supernetbasev3.standard.distill.supernet_finetune", sweep=get_finetuning_sweeps("bert"), models=[ {"exp_name": exp_name, "model_name_or_path": get_experiments_dir() + "/jan3_supernet_neuronmoe_jack_2L_fixedarch/neuronfixed/best_model", "subtransformer_config_path": get_experiments_dir() + "/aug11_evosearch_acadbertdata_supernet_v1/v1search/evo_results_14.xls", "tokenizer_name": "bert-base-uncased", "mixing": "attention"} ], kd_teacher=task_specific_trained_teacher("bertbase"), kd_config=finetuning_kd_config("std_logits")), time_in_mins=15000, wandb="offline", num_gpus=2, generate_port=True)

# neuron moe - 60M - collapse and subnet train - v5.1
# script_creator(get_experiments_dir() + "/jan4_standalone_neuronmoe_v5p1_60M", [ {"exp_name": "stand_60M", "runs": [{"pyfile": "train_mlm.py", "params": modify_config_and_to_string(config_factory("supernetbasev3.standard.train_mlm"), {"experiment_name": "jan4_standalone_neuronmoe_v5p1_60M", "max_train_steps": 125000, "sampling_type": "none", "num_warmup_steps": 0, "sampling_rule": "none", "model_name_or_path": get_experiments_dir() + "/dec15_neuronrouting_v5p1/5p1/best_model", "subtransformer_config_path": get_experiments_dir() + "/dec20_evosearch_neuronmoe_v5_30M_60M_67M/60M/evo_results_0.xls", "tokenized_c4_dir": dataset_factory("wikibooks_acabert_bertbaseuncased_128len"), "search_space_id": "v5.1", "tokenizer_name": "bert-base-uncased", "collapse_experts_before_ft": 1, "mixing": "attention"})}]} ], time_in_mins=8000, wandb="online", generate_port=True, num_gpus=8)

