#!/usr/bin/env python3
"""Load recorded episodes and save the downscaled observations, as well as an MP4 video."""

import multiprocessing
import sys
import imageio
import numpy as np
from pathlib import Path
from load_data import downscale

# Separate folders for downscaled .npz data files and MP4 video files
input_folder = Path(sys.argv[1])
mp4_output_folder = Path(sys.argv[2])
downscaled_output_folder = Path(sys.argv[3])
# Create the output folders if they don't exist
mp4_output_folder.mkdir(exist_ok=True)
downscaled_output_folder.mkdir(exist_ok=True)

def downscale_file(path):
    try:
        data = dict(np.load(path))
    except:
        print('Failed to load file', path)
        return
    obs = data['observ']
    # Save the observations to a compressed MP4 file
    mp4_path = mp4_output_folder / path.with_suffix('.mp4').name
    downscaled_path = downscaled_output_folder / path.name
    if mp4_path.exists() and downscaled_path.exists():
        print('Skipping file', path, 'because MP4 and downscaled files already exist')
        return
    # macro_block_size allows non-multiple-of-16 image sizes
    macro_block_size = int(np.gcd(obs.shape[1], obs.shape[2]))
    imageio.mimwrite(mp4_path, obs, macro_block_size=macro_block_size)
    print('Saved video to', mp4_path)
    # Convert the observations to grayscale, downscale, and save
    obs_grayscale = (obs[:, :, :, 0] * 0.299) + (obs[:, :, :, 1] * 0.587) + (obs[:, :, :, 2] * 0.114)
    data['observ'] = downscale(obs_grayscale, 8)
    np.savez_compressed(downscaled_path, **data)
    print('Saved downscaled data to', downscaled_path)
    # Delete the original episode once the downscaled version is saved
    path.unlink()
    print('Deleted file', path)

pool = multiprocessing.Pool(64)
pool.map(downscale_file, input_folder.iterdir())
