import torch
import torch.nn as nn
import torch.nn.functional as F
import numpy as np
from lib.sync_batchnorm.batchnorm import SynchronizedBatchNorm1d

class SegTime(nn.Module):
    def __init__(self, input_channel, output_stride=8, num_classes=21,
                 sync_bn=True, channel_last_data=True, 
                 skip_list=[1,4,16,64], dim_list=[1,1,1,1], predict_resolution=1):
        super(SegTime, self).__init__()
        if sync_bn == True:
            BatchNorm = SynchronizedBatchNorm1d
        else:
            BatchNorm = nn.BatchNorm1d
        
        self.resnet1d = ResNet33(input_channel, output_stride, BatchNorm=nn.BatchNorm1d)
        
        self.amsp = AMSP(input_channel, output_stride, BatchNorm)
        
        self.decoder = Decoder(input_channel, BatchNorm)
        
        self.mslstm = MultiSkipLSTM(input_channel, skip_list=skip_list, dim_list=dim_list)
        self.mslstm_dropout = nn.Dropout(0.2)
        
        self.predict = StepwisePredict(min(256,input_channel*8), sum(dim_list), num_classes)
        self.averagepredict = AveragePredict(predict_resolution)

        self.channel_last_data = channel_last_data

    def forward(self, input):
        if self.channel_last_data:# swap data axis
            input = input.permute(0,2,1)

        x, low_level_feat = self.resnet1d(input)
        x = self.amsp(x) 
        x = self.decoder(x, low_level_feat)
        
        rnn_feat = self.mslstm(input.permute(0,2,1))
        rnn_feat = self.mslstm_dropout(rnn_feat).permute(0,2,1)
        
        x = self.predict(x, rnn_feat)
        x = self.averagepredict(x)

        return x
                                
# deeplab
class DeepLab1D(nn.Module):
    def __init__(self, input_channel, output_stride=8, num_classes=21,
                 sync_bn=True, freeze_bn=False, channel_last_data=True, predict_resolution=1):
        super(DeepLab1D, self).__init__()

        if sync_bn == True:
            BatchNorm = SynchronizedBatchNorm1d
        else:
            BatchNorm = nn.BatchNorm1d
        
        self.backbone = ResNet33(input_channel, output_stride, BatchNorm=nn.BatchNorm1d)
        self.amsp = AMSP(input_channel, output_stride, BatchNorm)
        self.decoder = DecoderPredict(input_channel, num_classes, BatchNorm)
        self.averagepredict = AveragePredict(predict_resolution)
        self.channel_last_data = channel_last_data

    def forward(self, input):
        if self.channel_last_data:# swap data axis
            input = input.permute(0,2,1)

        x, low_level_feat = self.backbone(input)
        x = self.amsp(x)
        x = self.decoder(x, low_level_feat)
        x = F.interpolate(x, size=input.size()[2:], mode='linear', align_corners=True)
        x = self.averagepredict(x)

        return x   
                                
# prediction layer                                
class StepwisePredict(nn.Module):
    def __init__(self, cnn_feat_dim, rnn_feat_dim, num_classes):
        super(StepwisePredict, self).__init__()
        self.predict = nn.Conv1d(cnn_feat_dim+rnn_feat_dim, num_classes, kernel_size=1, stride=1)
    
    def forward(self, conv_feat, rnn_feat):
        
        conv_feat = F.interpolate(
            conv_feat, size=rnn_feat.size()[2], mode='linear', align_corners=True)
        x = torch.cat((conv_feat, rnn_feat), dim=1)
        out = self.predict(x)
        
        return out
    
    def _init_weight(self):
        for m in self.modules():
            if isinstance(m, nn.Conv1d):
                torch.nn.init.kaiming_normal_(m.weight)
                
class AveragePredict(nn.Module):
    def __init__(self, resoultion):
        super(AveragePredict, self).__init__()
        
        self.averagepool = nn.AvgPool1d(kernel_size=(resoultion))
        self.upsample = nn.Upsample(scale_factor=resoultion)
    
    def forward(self, x):
        x = self.averagepool(x)
        x = self.upsample(x)
        return x
    
# pyramid
class AMSP(nn.Module):
    def __init__(self, input_channel, output_stride, BatchNorm):
        super(AMSP, self).__init__()
        
        inplanes = min(input_channel*32, 1024)
        amsp_channel = min(256, input_channel*8)
        if output_stride == 16:
            dilations = [1, 6, 12, 18]
        elif output_stride == 8:
            dilations = [1, 12, 24, 36]
        else:
            raise NotImplementedError
            
        self.amsp1 = _AMSPModule(inplanes, amsp_channel, 1, padding=0, dilation=dilations[0], BatchNorm=BatchNorm)
        self.amsp2 = _AMSPModule(inplanes, amsp_channel, 3, padding=dilations[1], dilation=dilations[1], BatchNorm=BatchNorm)
        self.amsp3 = _AMSPModule(inplanes, amsp_channel, 3, padding=dilations[2], dilation=dilations[2], BatchNorm=BatchNorm)
        self.amsp4 = _AMSPModule(inplanes, amsp_channel, 3, padding=dilations[3], dilation=dilations[3], BatchNorm=BatchNorm)
        self.global_avg_pool = nn.Sequential(nn.AdaptiveAvgPool1d(1),
                                             nn.Conv1d(inplanes, amsp_channel, 1, stride=1, bias=False),
                                             nn.ReLU())
        self.conv1 = nn.Conv1d(amsp_channel*5, amsp_channel, 1, bias=False)
        self.bn1 = BatchNorm(amsp_channel)
        self.relu = nn.ReLU()
        self.dropout = nn.Dropout(0.5)
        self._init_weight()

    def forward(self, x):
        x1 = self.amsp1(x)
        x2 = self.amsp2(x)
        x3 = self.amsp3(x)
        x4 = self.amsp4(x)
        x5 = self.global_avg_pool(x)
        x5 = F.interpolate(x5, size=x4.size()[2:], mode='linear', align_corners=True)
        x = torch.cat((x1, x2, x3, x4, x5), dim=1)

        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)

        return self.dropout(x)

    def _init_weight(self):
        for m in self.modules():
            if isinstance(m, nn.Conv1d):
                torch.nn.init.kaiming_normal_(m.weight)
            elif isinstance(m, SynchronizedBatchNorm1d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()
            elif isinstance(m, nn.BatchNorm1d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()

class _AMSPModule(nn.Module):
    def __init__(self, inplanes, planes, kernel_size, padding, dilation, BatchNorm):
        super(_AMSPModule, self).__init__()
        self.atrous_conv = nn.Conv1d(inplanes, planes, kernel_size=kernel_size,
                                            stride=1, padding=padding, dilation=dilation, bias=False)
        self.bn = BatchNorm(planes)
        self.relu = nn.ReLU()

        self._init_weight()

    def forward(self, x):
        x = self.atrous_conv(x)
        x = self.bn(x)

        return self.relu(x)

    def _init_weight(self):
        for m in self.modules():
            if isinstance(m, nn.Conv1d):
                torch.nn.init.kaiming_normal_(m.weight)
            elif isinstance(m, SynchronizedBatchNorm1d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()
            elif isinstance(m, nn.BatchNorm1d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()
                
# decoder
class Decoder(nn.Module):
    def __init__(self, input_channel, BatchNorm):
        super(Decoder, self).__init__()
        inplanes = min(256, input_channel*8)
        low_level_inplanes = min(128, input_channel*4)
        low_level_feat_num = min(48, int(input_channel*1.5))
        
        self.conv1 = nn.Conv1d(low_level_inplanes, low_level_feat_num, 1, bias=False)
        self.bn1 = BatchNorm(low_level_feat_num)
        self.relu = nn.ReLU()
        self.last_conv = nn.Sequential(
            nn.Conv1d(inplanes+low_level_feat_num, inplanes, kernel_size=3, stride=1, padding=1, bias=False),
            BatchNorm(inplanes),
            nn.ReLU(),
            nn.Dropout(0.5),#0.5
            nn.Conv1d(inplanes, inplanes, kernel_size=3, stride=1, padding=1, bias=False),
            BatchNorm(inplanes),
            nn.ReLU(),
            nn.Dropout(0.2))
        
        self._init_weight()

    def forward(self, x, low_level_feat):
        low_level_feat = self.conv1(low_level_feat)
        low_level_feat = self.bn1(low_level_feat)
        low_level_feat = self.relu(low_level_feat)

        x = F.interpolate(x, size=low_level_feat.size()[2:], mode='linear', align_corners=True)
        x = torch.cat((x, low_level_feat), dim=1)
        x = self.last_conv(x)

        return x

    def _init_weight(self):
        for m in self.modules():
            if isinstance(m, nn.Conv1d):
                torch.nn.init.kaiming_normal_(m.weight)
            elif isinstance(m, SynchronizedBatchNorm1d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()
            elif isinstance(m, nn.BatchNorm1d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()

# decoder for deeplab
class DecoderPredict(nn.Module):
    def __init__(self, input_channel, num_classes, BatchNorm):
        super(DecoderPredict, self).__init__()
        inplanes = min(256, input_channel*8)
        low_level_inplanes = min(128, input_channel*4)
        low_level_feat_num = min(48, int(input_channel*1.5))
        self.conv1 = nn.Conv1d(low_level_inplanes, low_level_feat_num, 1, bias=False)
        self.bn1 = BatchNorm(low_level_feat_num)
        self.relu = nn.ReLU()
        self.last_conv = nn.Sequential(
            nn.Conv1d(inplanes+low_level_feat_num, inplanes, kernel_size=3, stride=1, padding=1, bias=False),
            BatchNorm(inplanes),
            nn.ReLU(),
            nn.Dropout(0.5),
            nn.Conv1d(inplanes, inplanes, kernel_size=3, stride=1, padding=1, bias=False),
            BatchNorm(inplanes),
            nn.ReLU(),
            nn.Dropout(0.1),
            nn.Conv1d(inplanes, num_classes, kernel_size=1, stride=1))
        self._init_weight()

    def forward(self, x, low_level_feat):
        low_level_feat = self.conv1(low_level_feat)
        low_level_feat = self.bn1(low_level_feat)
        low_level_feat = self.relu(low_level_feat)
        x = F.interpolate(x, size=low_level_feat.size()[2:], mode='linear', align_corners=True)
        x = torch.cat((x, low_level_feat), dim=1)
        x = self.last_conv(x)

        return x

    def _init_weight(self):
        for m in self.modules():
            if isinstance(m, nn.Conv1d):
                torch.nn.init.kaiming_normal_(m.weight)
            elif isinstance(m, SynchronizedBatchNorm1d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()
            elif isinstance(m, nn.BatchNorm1d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()

# multiskip layer
class _SkipLSTM(nn.Module):
    def __init__(self,n_input, n_hidden, skip_const=1, core='lstm', bidirectional=False):
        super(_SkipLSTM, self).__init__()
        self.skip_const = skip_const
        if core == 'lstm':
            self.lstm = nn.LSTM(n_input, n_hidden, 1, batch_first=True, bidirectional=bidirectional)
        elif core== 'gru':
            self.lstm = nn.GRU(n_input, n_hidden, 1, batch_first=True, bidirectional=bidirectional)
        else:
            raise ValueError('please assign core')
    
    def forward(self, x):
        skip_s = x[:,::self.skip_const,:]
        out, _ = self.lstm(skip_s)
        out = F.interpolate(out.permute(0,2,1), size=x.size()[1], mode='linear', align_corners=True)
        
        return out.permute(0,2,1)

class MultiSkipLSTM(nn.Module):
    def __init__(self, n_input, skip_list=[1,4,16,64], dim_list=[1,1,1,1], core='lstm', bidirectional=False):
        super(MultiSkipLSTM, self).__init__()
        self.skiplstm1 = _SkipLSTM(n_input, dim_list[0], skip_const=skip_list[0], core=core, bidirectional=bidirectional)
        self.skiplstm2 = _SkipLSTM(n_input, dim_list[1], skip_const=skip_list[1], core=core, bidirectional=bidirectional)
        self.skiplstm3 = _SkipLSTM(n_input, dim_list[2], skip_const=skip_list[2], core=core, bidirectional=bidirectional)
        self.skiplstm4 = _SkipLSTM(n_input, dim_list[3], skip_const=skip_list[3], core=core, bidirectional=bidirectional)
 
    def forward(self, x):
        x1 = self.skiplstm1(x)
        x2 = self.skiplstm2(x)
        x3 = self.skiplstm3(x)
        x4 = self.skiplstm4(x)
        out = torch.cat((x1, x2, x3, x4), dim=2)
        return out

class OneMSLSTM(nn.Module):
    def __init__(self, n_input, n_hidden=8):
        super(OneMSLSTM, self).__init__()
        self.mslstm1 = MultiSkipLSTMOLD(n_input, n_hidden)
        self.dropout = nn.Dropout(0.2)
    
    def forward(self, x):
        x = self.mslstm1(x)
        x = self.dropout(x)
        
        return x

class OneLSTM(nn.Module):
    def __init__(self, n_input, n_hidden=8):
        super(OneLSTM, self).__init__()
        self.lstm1 = nn.LSTM(n_input, n_hidden, 1, batch_first=True)
        self.dropout = nn.Dropout(0.2)
    
    def forward(self, x):
        x,_ = self.lstm1(x)
        x = self.dropout(x)
        
        return x
    
    
# 1d resnet    
class Bottleneck1D(nn.Module):
    expansion = 2

    def __init__(self, inplanes, planes, stride=1, dilation=1, downsample=None, BatchNorm=None):
        super(Bottleneck1D, self).__init__()
        self.conv1 = nn.Conv1d(inplanes, planes, kernel_size=1, bias=False)
        self.bn1 = BatchNorm(planes)
        self.conv2 = nn.Conv1d(planes, planes, kernel_size=3, stride=stride,
                               dilation=dilation, padding=dilation, bias=False)
        self.bn2 = BatchNorm(planes)
        self.conv3 = nn.Conv1d(planes, planes * 2, kernel_size=1, bias=False)
        self.bn3 = BatchNorm(planes * 2)
        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        self.stride = stride
        self.dilation = dilation

    def forward(self, x):
        residual = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv3(out)
        out = self.bn3(out)

        if self.downsample is not None:
            residual = self.downsample(x)

        out += residual
        out = self.relu(out)

        return out
    
def ResNet33(input_channel, output_stride, BatchNorm, pretrained=False):
    """Constructs a ResNet-33 model.
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
    """
    model = Res1DNet(input_channel, Bottleneck1D, [2, 2, 4, 2], output_stride, BatchNorm, pretrained=pretrained)
    return model

class Res1DNet(nn.Module):

    def __init__(self, input_channel, block, layers, output_stride, BatchNorm, pretrained=True):
        self.inplanes = min(64, input_channel*2)
        init_inplanes = self.inplanes
        super(Res1DNet, self).__init__()
        blocks = [1, 2, 4]
        if output_stride == 16:
            strides = [1, 2, 2, 1]
            dilations = [1, 1, 1, 2]
        elif output_stride == 8:
            strides = [1, 2, 1, 1]
            dilations = [1, 1, 2, 4]
        else:
            raise NotImplementedError

        # Modules
        self.conv1 = nn.Conv1d(input_channel, init_inplanes, kernel_size=7, stride=2, padding=3,
                                bias=False)
        self.bn1 = BatchNorm(init_inplanes)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool1d(kernel_size=3, stride=2, padding=1)

        self.layer1 = self._make_layer(block, init_inplanes, layers[0], stride=strides[0], dilation=dilations[0], BatchNorm=BatchNorm)
        self.layer2 = self._make_layer(block, init_inplanes*2, layers[1], stride=strides[1], dilation=dilations[1], BatchNorm=BatchNorm)
        self.layer3 = self._make_layer(block, init_inplanes*4, layers[2], stride=strides[2], dilation=dilations[2], BatchNorm=BatchNorm)
        self.layer4 = self._make_MG_unit(block, init_inplanes*8, blocks=blocks, stride=strides[3], dilation=dilations[3], BatchNorm=BatchNorm)
        # self.layer4 = self._make_layer(block, 512, layers[3], stride=strides[3], dilation=dilations[3], BatchNorm=BatchNorm)
        self._init_weight()

        if pretrained:
            self._load_pretrained_model()

    def _make_layer(self, block, planes, blocks, stride=1, dilation=1, BatchNorm=None):
        downsample = None
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                nn.Conv1d(self.inplanes, planes * block.expansion,
                          kernel_size=1, stride=stride, bias=False),
                BatchNorm(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, dilation, downsample, BatchNorm))
        self.inplanes = planes * block.expansion
        for i in range(1, blocks):
            layers.append(block(self.inplanes, planes, dilation=dilation, BatchNorm=BatchNorm))

        return nn.Sequential(*layers)

    def _make_MG_unit(self, block, planes, blocks, stride=1, dilation=1, BatchNorm=None):
        downsample = None
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                nn.Conv1d(self.inplanes, planes * block.expansion,
                          kernel_size=1, stride=stride, bias=False),
                BatchNorm(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, dilation=blocks[0]*dilation,
                            downsample=downsample, BatchNorm=BatchNorm))
        self.inplanes = planes * block.expansion
        for i in range(1, len(blocks)):
            layers.append(block(self.inplanes, planes, stride=1,
                                dilation=blocks[i]*dilation, BatchNorm=BatchNorm))

        return nn.Sequential(*layers)

    def forward(self, input):
        x = self.conv1(input)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        low_level_feat = x
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)
        return x, low_level_feat

    def _init_weight(self):
        for m in self.modules():
            if isinstance(m, nn.Conv1d):
                torch.nn.init.kaiming_normal_(m.weight)
            elif isinstance(m, SynchronizedBatchNorm1d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()
            elif isinstance(m, nn.BatchNorm1d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()
