;; gorilla-repl.fileformat = 1

;; **
;;; # Gorilla REPL
;;; 
;;; Welcome to gorilla :-)
;;; 
;;; Shift + enter evaluates code. Hit alt+g twice in quick succession or click the menu icon (upper-right corner) for more commands ...
;;; 
;;; It's a good habit to run each worksheet in its own namespace: feel free to use the declaration we've provided below if you'd like.
;; **

;; @@

(ns model
  (:require [gorilla-plot.core :as plot])
  (:use [anglican core emit runtime stat
          [state :only [get-predicts get-log-weight get-result]]]))

(defdist geometric
"Geometric distribution on support {0,1,2....}"
[p] []
(sample* [this]
        (loop [value 0]
            (if (sample* (flip p))
            value
            (recur (inc value)))))
(observe* [this value] (+ (log p) (* value (log (- 1 p))))))

(defdist dirac [x]
    (sample* [this] x)
    (observe* [this value]
              (if (= value x)
                0
                (- (/ 1.0 0.0)))))


(def data [4 5 4 0 1 4 3 4 0 6 3 3 4 0 2 6 3 3 5 4 5 3 1 4 4 1 5 5 3 4 2 5 2 2 3 4 2 1 3 -1 2 1 1 1 1 3 0 0 1 0 1 1 0 0 3 1 0 3 2 2 0 1 1 1 0 1 0 1 0 0 0 2 1 0 0 0 1 1 0 2 3 3 1 -1 2 1 1 1 1 2 4 2 0 0 1 4 0 0 0 1 0 0 0 0 0 1 0 0 1 0 1]
  )

(with-primitive-procedures [dirac geometric]
  (defquery model [method- options- ]
    (let [
      rate1 (sample (geometric 0.1))
      rate2 (sample (geometric 0.1))]
      (doall (map
        (fn [i]
          (let [x (nth data i)
                rate (if (flip 0.5) rate1 rate2)
                dist (if (zero? rate) (dirac 0) (poisson (* 0.1 rate)))]
            (if (<= 0 (nth data i)) (observe dist (nth data i)))))
        (range 0 (count data))))
      rate1
    )
  )
)


(def model_name "mixture")
(def outfile "mixture_anglican.json")

; (def configurations [:rmh []])
(def configurations
  [
    [:importance []]
    [:lmh []]
    [:rmh []]
    [:rmh [:alpha 0.8 :sigma 2]]
    [:smc []]
    [:smc [:number-of-particles 1000]]
    [:pgibbs []]
    [:pgibbs [:number-of-particles 1000]]
    [:ipmcmc []]
    [:ipmcmc [:number-of-particles 1000 :number-of-nodes 32]]
  ])

; (def num_samples_options [1000])
(def num_samples_options [1000 10000])
(def thinning 1)

(spit outfile "[\n" :append false)

(def num-chains 20)

(doall
  (for [ num_samples num_samples_options
         [method options] configurations
         chain (range 0 num-chains)]
    (do
      (println (format "\nMethod %s with options %s" method options))
      (println (format "Chain no. %s" chain))
      (let [start (. System (nanoTime))
            warmup (/ num_samples 5)
            samples (take-nth thinning (take (* num_samples thinning) (drop warmup (apply doquery method model [method options] options))))
            results (collect-results samples)
            values (map (fn [s] (get-result s)) samples)
            max-value (apply max values)
            mean (empirical-mean results)
            variance (empirical-variance results)
            std (empirical-std results)
            skewness (if (zero? std) (/ 0.0 0.0) (empirical-skew results))
            kurtosis (if (zero? std) (/ 0.0 0.0) (empirical-kurtosis results))
            distribution (empirical-distribution (collect-results samples))
            masses (for [n (range 0 (inc max-value))] (get distribution n 0.0))
            end (. System (nanoTime))
            elapsed_ms (/ (- end start) 1e6)]
        (println (format "Elapsed time: %s ms" elapsed_ms))
        (println (format "Empirical mean: %s" mean))
        (println (format "Empirical variance: %s" variance))
        (println (format "Empirical std: %s" std))
        (println (format "Empirical skewness: %s" skewness))
        (println (format "Empirical kurtosis: %s" kurtosis))
        (spit outfile (format
                   "{\"model\": \"%s\", \"system\": \"anglican\", \"method\": \"%s\", \"options\": \"%s\", \"num_samples\": %s, \"time_ms\": %s, \"total\": 1.0, \"mean\": %s, \"variance\": %s, \"stddev\": %s, \"skewness\": %s, \"kurtosis\": %s, \"masses\": [%s] },\n"
                   model_name method options num_samples elapsed_ms mean variance std skewness kurtosis
                   (clojure.string/join ", " masses)) :append true)
        (if false (do
          (println "Empirical distribution:")
          (doall (for [n (range 0 (inc max-value))]
            (println (format "p(%s) = %s" n (get distribution n 0.0)))))))
        ; (println "List of samples (format: sample log-weight):")
        ; (doall (map (fn [s] (println (format "%s %s" (get-result s) (get-log-weight s)))) samples))
        ; values need to be adjusted if they are weighted!
        ; (plot/histogram values :normalize :probability)
      )
    )
  )
)

(spit outfile "]\n" :append true)



;; @@
;; ->
;;; 
;;; Method :importance with options []
;;; Chain no. 0
;;; Elapsed time: 1780.897834 ms
;;; Empirical mean: 17.129419922482974
;;; Empirical variance: 1.5193278039359526
;;; Empirical std: 1.2326101589456224
;;; Empirical skewness: 0.2124596875011093
;;; Empirical kurtosis: 3.2324984937589116
;;; 
;;; Method :importance with options []
;;; Chain no. 1
;;; Elapsed time: 1758.402739 ms
;;; Empirical mean: 17.12741570383119
;;; Empirical variance: 1.540647537319779
;;; Empirical std: 1.2412282374002692
;;; Empirical skewness: 0.030507646832139173
;;; Empirical kurtosis: 3.234044300720508
;;; 
;;; Method :importance with options []
;;; Chain no. 2
;;; Elapsed time: 1792.54715 ms
;;; Empirical mean: 17.336218117328976
;;; Empirical variance: 1.4829835328543415
;;; Empirical std: 1.2177781131447312
;;; Empirical skewness: 0.13597872777513723
;;; Empirical kurtosis: 2.9903995881615484
;;; 
;;; Method :importance with options []
;;; Chain no. 3
;;; Elapsed time: 1833.869255 ms
;;; Empirical mean: 17.088983898566187
;;; Empirical variance: 1.3813327221978757
;;; Empirical std: 1.175301119797763
;;; Empirical skewness: 0.32883649412261945
;;; Empirical kurtosis: 3.2200666349147684
;;; 
;;; Method :importance with options []
;;; Chain no. 4
;;; Elapsed time: 1997.669184 ms
;;; Empirical mean: 17.27841788115641
;;; Empirical variance: 1.6952914158660046
;;; Empirical std: 1.3020335694082563
;;; Empirical skewness: 0.04407483649179952
;;; Empirical kurtosis: 2.4669064137889705
;;; 
;;; Method :importance with options []
;;; Chain no. 5
;;; Elapsed time: 1935.570502 ms
;;; Empirical mean: 17.162314153055497
;;; Empirical variance: 1.4579775153781085
;;; Empirical std: 1.2074673972319536
;;; Empirical skewness: 0.31403830306475616
;;; Empirical kurtosis: 3.235581349202665
;;; 
;;; Method :importance with options []
;;; Chain no. 6
;;; Elapsed time: 2186.443637 ms
;;; Empirical mean: 17.217670858066043
;;; Empirical variance: 1.557584935860603
;;; Empirical std: 1.248032425804956
;;; Empirical skewness: 0.17516648012288527
;;; Empirical kurtosis: 3.084327942005642
;;; 
;;; Method :importance with options []
;;; Chain no. 7
;;; Elapsed time: 1833.087127 ms
;;; Empirical mean: 17.11806843484984
;;; Empirical variance: 1.4496975125215035
;;; Empirical std: 1.2040338502390635
;;; Empirical skewness: 0.24811107030912927
;;; Empirical kurtosis: 3.2614627057020997
;;; 
;;; Method :importance with options []
;;; Chain no. 8
;;; Elapsed time: 1922.743098 ms
;;; Empirical mean: 17.23943869091643
;;; Empirical variance: 1.4638347397141984
;;; Empirical std: 1.2098903833464412
;;; Empirical skewness: -0.03249100508463736
;;; Empirical kurtosis: 3.1764370061358873
;;; 
;;; Method :importance with options []
;;; Chain no. 9
;;; Elapsed time: 1947.321711 ms
;;; Empirical mean: 17.068057529911027
;;; Empirical variance: 1.7911317323383753
;;; Empirical std: 1.338331697427202
;;; Empirical skewness: 0.31366339648784614
;;; Empirical kurtosis: 2.7434474169081158
;;; 
;;; Method :importance with options []
;;; Chain no. 10
;;; Elapsed time: 2338.067877 ms
;;; Empirical mean: 17.257091000276723
;;; Empirical variance: 1.6381176784739189
;;; Empirical std: 1.2798897134026506
;;; Empirical skewness: -0.0445127053774519
;;; Empirical kurtosis: 2.794766678905241
;;; 
;;; Method :importance with options []
;;; Chain no. 11
;;; Elapsed time: 1830.193998 ms
;;; Empirical mean: 17.212138761566997
;;; Empirical variance: 1.8696295215464724
;;; Empirical std: 1.3673439660694278
;;; Empirical skewness: 0.18126378038794766
;;; Empirical kurtosis: 2.77869240702107
;;; 
;;; Method :importance with options []
;;; Chain no. 12
;;; Elapsed time: 1705.815049 ms
;;; Empirical mean: 17.405426732015492
;;; Empirical variance: 1.297584709894647
;;; Empirical std: 1.139115757899366
;;; Empirical skewness: -0.18565615277926942
;;; Empirical kurtosis: 3.6936273841986815
;;; 
;;; Method :importance with options []
;;; Chain no. 13
;;; Elapsed time: 1805.369072 ms
;;; Empirical mean: 17.188155857612344
;;; Empirical variance: 1.5417934252828063
;;; Empirical std: 1.241689745984401
;;; Empirical skewness: 0.11216613755507047
;;; Empirical kurtosis: 2.8877881592780876
;;; 
;;; Method :importance with options []
;;; Chain no. 14
;;; Elapsed time: 1825.190464 ms
;;; Empirical mean: 17.056912544742435
;;; Empirical variance: 1.5728099921702368
;;; Empirical std: 1.2541172162801357
;;; Empirical skewness: 0.3208519210687742
;;; Empirical kurtosis: 3.0827974055811853
;;; 
;;; Method :importance with options []
;;; Chain no. 15
;;; Elapsed time: 2410.170112 ms
;;; Empirical mean: 17.247553856311196
;;; Empirical variance: 1.3004587710951474
;;; Empirical std: 1.1403765917867428
;;; Empirical skewness: -0.07336112326196388
;;; Empirical kurtosis: 3.347304522592278
;;; 
;;; Method :importance with options []
;;; Chain no. 16
;;; Elapsed time: 1987.416791 ms
;;; Empirical mean: 17.05487896183676
;;; Empirical variance: 1.7175165300571562
;;; Empirical std: 1.3105405488031099
;;; Empirical skewness: 0.20392919808002216
;;; Empirical kurtosis: 3.0580032646462136
;;; 
;;; Method :importance with options []
;;; Chain no. 17
;;; Elapsed time: 1707.893012 ms
;;; Empirical mean: 17.22975296212052
;;; Empirical variance: 1.4730329585115187
;;; Empirical std: 1.2136856918129664
;;; Empirical skewness: -0.0014069370375321387
;;; Empirical kurtosis: 3.0555976162391625
;;; 
;;; Method :importance with options []
;;; Chain no. 18
;;; Elapsed time: 1918.652641 ms
;;; Empirical mean: 17.227224786482793
;;; Empirical variance: 1.7554365643694347
;;; Empirical std: 1.3249288903067344
;;; Empirical skewness: 0.08780398328222083
;;; Empirical kurtosis: 2.7880222831963257
;;; 
;;; Method :importance with options []
;;; Chain no. 19
;;; Elapsed time: 1736.102772 ms
;;; Empirical mean: 17.160616855237016
;;; Empirical variance: 1.5053474387648629
;;; Empirical std: 1.2269260119358718
;;; Empirical skewness: 0.12158440546033662
;;; Empirical kurtosis: 2.951076618312999
;;; 
;;; Method :lmh with options []
;;; Chain no. 0
;;; Elapsed time: 1622.891068 ms
;;; Empirical mean: 16.913
;;; Empirical variance: 0.8014310000000002
;;; Empirical std: 0.8952267869093284
;;; Empirical skewness: -0.004415570157539152
;;; Empirical kurtosis: 2.116552599517621
;;; 
;;; Method :lmh with options []
;;; Chain no. 1
;;; Elapsed time: 2081.465012 ms
;;; Empirical mean: 17.067000000000004
;;; Empirical variance: 1.6985110000000019
;;; Empirical std: 1.3032693505181505
;;; Empirical skewness: 1.0116014623933962
;;; Empirical kurtosis: 3.5983769251283997
;;; 
;;; Method :lmh with options []
;;; Chain no. 2
;;; Elapsed time: 2011.064998 ms
;;; Empirical mean: 17.111
;;; Empirical variance: 1.714679000000001
;;; Empirical std: 1.3094575212659634
;;; Empirical skewness: 0.43853585856112665
;;; Empirical kurtosis: 3.534172886156323
;;; 
;;; Method :lmh with options []
;;; Chain no. 3
;;; Elapsed time: 1650.350616 ms
;;; Empirical mean: 17.043999999999993
;;; Empirical variance: 1.1160640000000004
;;; Empirical std: 1.0564393025630958
;;; Empirical skewness: -0.5609630286713749
;;; Empirical kurtosis: 2.5702010228452288
;;; 
;;; Method :lmh with options []
;;; Chain no. 4
;;; Elapsed time: 1815.823527 ms
;;; Empirical mean: 17.197000000000003
;;; Empirical variance: 1.0561910000000005
;;; Empirical std: 1.0277115354028097
;;; Empirical skewness: 0.25164062053157404
;;; Empirical kurtosis: 2.5031089383897602
;;; 
;;; Method :lmh with options []
;;; Chain no. 5
;;; Elapsed time: 2538.893092 ms
;;; Empirical mean: 17.138
;;; Empirical variance: 0.7949560000000007
;;; Empirical std: 0.8916030506901603
;;; Empirical skewness: 0.7086210994671165
;;; Empirical kurtosis: 4.464803169459866
;;; 
;;; Method :lmh with options []
;;; Chain no. 6
;;; Elapsed time: 1840.136583 ms
;;; Empirical mean: 17.403999999999996
;;; Empirical variance: 1.032784000000001
;;; Empirical std: 1.0162598093007522
;;; Empirical skewness: -0.681868562322546
;;; Empirical kurtosis: 3.268655316076103
;;; 
;;; Method :lmh with options []
;;; Chain no. 7
;;; Elapsed time: 1717.025244 ms
;;; Empirical mean: 17.616000000000003
;;; Empirical variance: 0.9525440000000013
;;; Empirical std: 0.9759836064196987
;;; Empirical skewness: 0.5442310131241554
;;; Empirical kurtosis: 3.20569086956093
;;; 
;;; Method :lmh with options []
;;; Chain no. 8
;;; Elapsed time: 1593.066059 ms
;;; Empirical mean: 17.681999999999995
;;; Empirical variance: 1.8328760000000006
;;; Empirical std: 1.3538375087136567
;;; Empirical skewness: -0.7645951552388891
;;; Empirical kurtosis: 2.8523696180452327
;;; 
;;; Method :lmh with options []
;;; Chain no. 9
;;; Elapsed time: 2126.965156 ms
;;; Empirical mean: 16.826
;;; Empirical variance: 1.0957240000000008
;;; Empirical std: 1.046768360240221
;;; Empirical skewness: 0.37772238908105105
;;; Empirical kurtosis: 2.531686394981193
;;; 
;;; Method :lmh with options []
;;; Chain no. 10
;;; Elapsed time: 2096.105065 ms
;;; Empirical mean: 17.239
;;; Empirical variance: 1.4278790000000012
;;; Empirical std: 1.1949389105724197
;;; Empirical skewness: 0.6854644964716317
;;; Empirical kurtosis: 2.810555056974762
;;; 
;;; Method :lmh with options []
;;; Chain no. 11
;;; Elapsed time: 1791.051697 ms
;;; Empirical mean: 17.308
;;; Empirical variance: 0.9051360000000007
;;; Empirical std: 0.9513863568498345
;;; Empirical skewness: 0.718164878173121
;;; Empirical kurtosis: 3.8105801185916395
;;; 
;;; Method :lmh with options []
;;; Chain no. 12
;;; Elapsed time: 2073.149302 ms
;;; Empirical mean: 16.729999999999997
;;; Empirical variance: 1.629100000000001
;;; Empirical std: 1.2763620176109916
;;; Empirical skewness: -0.1003914083342305
;;; Empirical kurtosis: 2.614453030782248
;;; 
;;; Method :lmh with options []
;;; Chain no. 13
;;; Elapsed time: 1651.269796 ms
;;; Empirical mean: 16.831999999999997
;;; Empirical variance: 2.453776000000001
;;; Empirical std: 1.5664533188065328
;;; Empirical skewness: 0.1606370503183197
;;; Empirical kurtosis: 2.5764077573855073
;;; 
;;; Method :lmh with options []
;;; Chain no. 14
;;; Elapsed time: 1711.265566 ms
;;; Empirical mean: 17.336000000000002
;;; Empirical variance: 0.819104000000001
;;; Empirical std: 0.9050436453563999
;;; Empirical skewness: -0.37984681124664693
;;; Empirical kurtosis: 4.666453358822381
;;; 
;;; Method :lmh with options []
;;; Chain no. 15
;;; Elapsed time: 1674.058181 ms
;;; Empirical mean: 17.223
;;; Empirical variance: 1.447271000000001
;;; Empirical std: 1.20302576863507
;;; Empirical skewness: -0.010518534658883147
;;; Empirical kurtosis: 2.526043999968368
;;; 
;;; Method :lmh with options []
;;; Chain no. 16
;;; Elapsed time: 2451.237146 ms
;;; Empirical mean: 17.192
;;; Empirical variance: 1.469136000000001
;;; Empirical std: 1.2120792053327212
;;; Empirical skewness: 0.9359752733911633
;;; Empirical kurtosis: 5.235300492053784
;;; 
;;; Method :lmh with options []
;;; Chain no. 17
;;; Elapsed time: 1736.406001 ms
;;; Empirical mean: 17.224
;;; Empirical variance: 1.0798240000000008
;;; Empirical std: 1.0391458030517184
;;; Empirical skewness: 0.7567266297552421
;;; Empirical kurtosis: 3.6067862064406997
;;; 
;;; Method :lmh with options []
;;; Chain no. 18
;;; Elapsed time: 1554.760923 ms
;;; Empirical mean: 17.23
;;; Empirical variance: 2.637100000000001
;;; Empirical std: 1.6239150224072691
;;; Empirical skewness: -0.11623457462659134
;;; Empirical kurtosis: 1.895250071746654
;;; 
;;; Method :lmh with options []
;;; Chain no. 19
;;; Elapsed time: 1611.501141 ms
;;; Empirical mean: 17.172
;;; Empirical variance: 0.9464160000000008
;;; Empirical std: 0.9728391439492969
;;; Empirical skewness: -0.5055216468123664
;;; Empirical kurtosis: 2.5179704073652136
;;; 
;;; Method :rmh with options []
;;; Chain no. 0
;;; Elapsed time: 1590.918605 ms
;;; Empirical mean: 17.296999999999997
;;; Empirical variance: 0.8507910000000004
;;; Empirical std: 0.9223833259551044
;;; Empirical skewness: 0.1742544751859632
;;; Empirical kurtosis: 2.4625136886530687
;;; 
;;; Method :rmh with options []
;;; Chain no. 1
;;; Elapsed time: 1958.928238 ms
;;; Empirical mean: 17.043999999999997
;;; Empirical variance: 1.1900640000000007
;;; Empirical std: 1.0909005454210758
;;; Empirical skewness: -0.07330965343676621
;;; Empirical kurtosis: 2.699323392215569
;;; 
;;; Method :rmh with options []
;;; Chain no. 2
;;; Elapsed time: 1890.686088 ms
;;; Empirical mean: 16.969
;;; Empirical variance: 1.8940390000000005
;;; Empirical std: 1.3762408946111144
;;; Empirical skewness: 0.6794823812698374
;;; Empirical kurtosis: 3.4732007292701454
;;; 
;;; Method :rmh with options []
;;; Chain no. 3
;;; Elapsed time: 1999.983003 ms
;;; Empirical mean: 16.829
;;; Empirical variance: 1.2917590000000019
;;; Empirical std: 1.1365557619404345
;;; Empirical skewness: 0.4649863607577944
;;; Empirical kurtosis: 2.8840584355346883
;;; 
;;; Method :rmh with options []
;;; Chain no. 4
;;; Elapsed time: 1655.188044 ms
;;; Empirical mean: 17.256
;;; Empirical variance: 0.8004640000000001
;;; Empirical std: 0.8946865372855457
;;; Empirical skewness: -0.23951703900833535
;;; Empirical kurtosis: 1.7645153094580786
;;; 
;;; Method :rmh with options []
;;; Chain no. 5
;;; Elapsed time: 2205.834657 ms
;;; Empirical mean: 17.104000000000003
;;; Empirical variance: 0.957184000000001
;;; Empirical std: 0.9783578077574692
;;; Empirical skewness: -0.27952492669052953
;;; Empirical kurtosis: 3.7662691702561104
;;; 
;;; Method :rmh with options []
;;; Chain no. 6
;;; Elapsed time: 1730.194529 ms
;;; Empirical mean: 17.507
;;; Empirical variance: 1.9039510000000013
;;; Empirical std: 1.3798373092506238
;;; Empirical skewness: -0.11846837361008349
;;; Empirical kurtosis: 2.4114714059188835
;;; 
;;; Method :rmh with options []
;;; Chain no. 7
;;; Elapsed time: 1983.879602 ms
;;; Empirical mean: 16.828
;;; Empirical variance: 1.4324160000000008
;;; Empirical std: 1.196835828340713
;;; Empirical skewness: -0.012707255021787768
;;; Empirical kurtosis: 2.5401294425825314
;;; 
;;; Method :rmh with options []
;;; Chain no. 8
;;; Elapsed time: 1682.837965 ms
;;; Empirical mean: 16.919000000000008
;;; Empirical variance: 3.264439000000001
;;; Empirical std: 1.8067758577089748
;;; Empirical skewness: 0.19612896360338694
;;; Empirical kurtosis: 2.0461804733178077
;;; 
;;; Method :rmh with options []
;;; Chain no. 9
;;; Elapsed time: 1830.399683 ms
;;; Empirical mean: 17.019
;;; Empirical variance: 1.8886390000000017
;;; Empirical std: 1.3742776284288418
;;; Empirical skewness: -0.36472897589562725
;;; Empirical kurtosis: 3.4182790126620044
;;; 
;;; Method :rmh with options []
;;; Chain no. 10
;;; Elapsed time: 1787.403108 ms
;;; Empirical mean: 17.117999999999995
;;; Empirical variance: 1.2280760000000006
;;; Empirical std: 1.108185904981651
;;; Empirical skewness: 0.17606850990664052
;;; Empirical kurtosis: 2.136434619990352
;;; 
;;; Method :rmh with options []
;;; Chain no. 11
;;; Elapsed time: 1596.4387 ms
;;; Empirical mean: 17.331
;;; Empirical variance: 1.6274390000000014
;;; Empirical std: 1.275711174208332
;;; Empirical skewness: -0.6942258941486543
;;; Empirical kurtosis: 3.251380896098851
;;; 
;;; Method :rmh with options []
;;; Chain no. 12
;;; Elapsed time: 1551.019253 ms
;;; Empirical mean: 17.125000000000004
;;; Empirical variance: 2.1773750000000014
;;; Empirical std: 1.4755931010952854
;;; Empirical skewness: 0.06241402818257445
;;; Empirical kurtosis: 2.7649159876281835
;;; 
;;; Method :rmh with options []
;;; Chain no. 13
;;; Elapsed time: 1630.39839 ms
;;; Empirical mean: 17.662
;;; Empirical variance: 0.6357560000000007
;;; Empirical std: 0.7973430880116794
;;; Empirical skewness: -0.7747421928038078
;;; Empirical kurtosis: 3.736378988709172
;;; 
;;; Method :rmh with options []
;;; Chain no. 14
;;; Elapsed time: 1577.878918 ms
;;; Empirical mean: 17.012
;;; Empirical variance: 2.785856000000001
;;; Empirical std: 1.6690883739335078
;;; Empirical skewness: 0.3642507068225092
;;; Empirical kurtosis: 2.8824008938476258
;;; 
;;; Method :rmh with options []
;;; Chain no. 15
;;; Elapsed time: 1581.464786 ms
;;; Empirical mean: 17.593999999999998
;;; Empirical variance: 1.7571640000000008
;;; Empirical std: 1.3255806274987578
;;; Empirical skewness: -0.002082561274112654
;;; Empirical kurtosis: 2.935941729746378
;;; 
;;; Method :rmh with options []
;;; Chain no. 16
;;; Elapsed time: 1597.672168 ms
;;; Empirical mean: 16.756999999999998
;;; Empirical variance: 2.0019510000000014
;;; Empirical std: 1.4149031768993952
;;; Empirical skewness: 0.39426119596290465
;;; Empirical kurtosis: 2.7540581070699752
;;; 
;;; Method :rmh with options []
;;; Chain no. 17
;;; Elapsed time: 1557.701821 ms
;;; Empirical mean: 17.262
;;; Empirical variance: 1.4433560000000007
;;; Empirical std: 1.2013975195579525
;;; Empirical skewness: 0.5625793099628801
;;; Empirical kurtosis: 2.3494040620089875
;;; 
;;; Method :rmh with options []
;;; Chain no. 18
;;; Elapsed time: 1563.277795 ms
;;; Empirical mean: 17.057
;;; Empirical variance: 1.1917510000000004
;;; Empirical std: 1.0916734859837902
;;; Empirical skewness: -0.16370032900850934
;;; Empirical kurtosis: 2.8785473030916804
;;; 
;;; Method :rmh with options []
;;; Chain no. 19
;;; Elapsed time: 1514.362942 ms
;;; Empirical mean: 17.083
;;; Empirical variance: 0.9981110000000007
;;; Empirical std: 0.9990550535380924
;;; Empirical skewness: 0.681829625586542
;;; Empirical kurtosis: 3.2823593360970285
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 0
;;; Elapsed time: 1532.851235 ms
;;; Empirical mean: 16.467
;;; Empirical variance: 1.6469110000000007
;;; Empirical std: 1.2833203029641511
;;; Empirical skewness: 0.13997072208794442
;;; Empirical kurtosis: 2.463499040548545
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 1
;;; Elapsed time: 1509.206141 ms
;;; Empirical mean: 16.794999999999995
;;; Empirical variance: 1.608975000000001
;;; Empirical std: 1.268453783154909
;;; Empirical skewness: -0.1346777626944312
;;; Empirical kurtosis: 2.479194987259833
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 2
;;; Elapsed time: 1606.080023 ms
;;; Empirical mean: 17.024999999999995
;;; Empirical variance: 0.8663750000000005
;;; Empirical std: 0.9307926729406504
;;; Empirical skewness: 0.6274758467975472
;;; Empirical kurtosis: 3.5932984408153383
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 3
;;; Elapsed time: 1523.623553 ms
;;; Empirical mean: 16.939
;;; Empirical variance: 1.3412790000000003
;;; Empirical std: 1.1581360023762324
;;; Empirical skewness: 0.7407578609168812
;;; Empirical kurtosis: 2.3878470457024985
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 4
;;; Elapsed time: 1535.899803 ms
;;; Empirical mean: 17.780999999999995
;;; Empirical variance: 1.517039000000001
;;; Empirical std: 1.231681371134597
;;; Empirical skewness: -0.21717957390430756
;;; Empirical kurtosis: 3.322603924127472
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 5
;;; Elapsed time: 1534.370547 ms
;;; Empirical mean: 17.177999999999997
;;; Empirical variance: 1.656316000000001
;;; Empirical std: 1.286979409314695
;;; Empirical skewness: 0.5216101739543377
;;; Empirical kurtosis: 4.122086195995287
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 6
;;; Elapsed time: 1571.189082 ms
;;; Empirical mean: 17.023
;;; Empirical variance: 1.610471000000001
;;; Empirical std: 1.2690433404734454
;;; Empirical skewness: -0.524589961026105
;;; Empirical kurtosis: 2.6559530164013703
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 7
;;; Elapsed time: 1544.276097 ms
;;; Empirical mean: 16.828
;;; Empirical variance: 1.3284160000000005
;;; Empirical std: 1.1525693037730966
;;; Empirical skewness: -0.4411551459393783
;;; Empirical kurtosis: 2.582107450390401
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 8
;;; Elapsed time: 1624.897306 ms
;;; Empirical mean: 17.121999999999996
;;; Empirical variance: 2.329116000000002
;;; Empirical std: 1.5261441609494177
;;; Empirical skewness: 0.6008405035093206
;;; Empirical kurtosis: 3.316178633442149
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 9
;;; Elapsed time: 1569.922484 ms
;;; Empirical mean: 16.935999999999996
;;; Empirical variance: 0.9579040000000006
;;; Empirical std: 0.9787257021249625
;;; Empirical skewness: 0.07058979045460421
;;; Empirical kurtosis: 2.5974668041760625
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 10
;;; Elapsed time: 1561.256643 ms
;;; Empirical mean: 17.194999999999997
;;; Empirical variance: 2.1269750000000007
;;; Empirical std: 1.4584152357953482
;;; Empirical skewness: 0.36534496881241924
;;; Empirical kurtosis: 2.883784643810889
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 11
;;; Elapsed time: 1566.025701 ms
;;; Empirical mean: 16.983
;;; Empirical variance: 0.8807110000000004
;;; Empirical std: 0.9384620397224388
;;; Empirical skewness: 0.6217937900331559
;;; Empirical kurtosis: 2.7724027995376237
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 12
;;; Elapsed time: 1537.501619 ms
;;; Empirical mean: 16.926000000000002
;;; Empirical variance: 1.2185240000000006
;;; Empirical std: 1.1038677457014499
;;; Empirical skewness: 0.48986898634420367
;;; Empirical kurtosis: 2.278402546806677
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 13
;;; Elapsed time: 1537.994154 ms
;;; Empirical mean: 16.753000000000004
;;; Empirical variance: 2.0099910000000016
;;; Empirical std: 1.4177415138169587
;;; Empirical skewness: -0.6262273952616064
;;; Empirical kurtosis: 2.7654017454439663
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 14
;;; Elapsed time: 1524.934695 ms
;;; Empirical mean: 17.379
;;; Empirical variance: 2.0313590000000015
;;; Empirical std: 1.4252575205905778
;;; Empirical skewness: -0.6131145274369374
;;; Empirical kurtosis: 3.311998020076948
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 15
;;; Elapsed time: 1556.800282 ms
;;; Empirical mean: 17.071
;;; Empirical variance: 0.7059590000000002
;;; Empirical std: 0.840213663302377
;;; Empirical skewness: -0.3468346271847362
;;; Empirical kurtosis: 2.0066063326840515
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 16
;;; Elapsed time: 1516.218592 ms
;;; Empirical mean: 17.241
;;; Empirical variance: 1.564919000000001
;;; Empirical std: 1.250967225789709
;;; Empirical skewness: 0.4819860930220667
;;; Empirical kurtosis: 3.1052532228737784
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 17
;;; Elapsed time: 1562.655204 ms
;;; Empirical mean: 16.550000000000004
;;; Empirical variance: 2.535500000000002
;;; Empirical std: 1.592325343640552
;;; Empirical skewness: 0.5018278232545522
;;; Empirical kurtosis: 2.9920623109253395
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 18
;;; Elapsed time: 1501.233983 ms
;;; Empirical mean: 17.159
;;; Empirical variance: 1.789719000000001
;;; Empirical std: 1.3378037972737262
;;; Empirical skewness: -0.21664669664183991
;;; Empirical kurtosis: 2.8474932116654146
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 19
;;; Elapsed time: 1518.037945 ms
;;; Empirical mean: 17.380999999999997
;;; Empirical variance: 0.3458390000000002
;;; Empirical std: 0.5880807767645532
;;; Empirical skewness: 0.7493299834438655
;;; Empirical kurtosis: 3.1809572036100326
;;; 
;;; Method :smc with options []
;;; Chain no. 0
;;; Elapsed time: 2786.646519 ms
;;; Empirical mean: 17.25970583203204
;;; Empirical variance: 1.5494573970498349
;;; Empirical std: 1.2447720261356434
;;; Empirical skewness: 0.10247984416843219
;;; Empirical kurtosis: 2.9839206716857944
;;; 
;;; Method :smc with options []
;;; Chain no. 1
;;; Elapsed time: 2861.634546 ms
;;; Empirical mean: 17.050095298328564
;;; Empirical variance: 1.4708991567273537
;;; Empirical std: 1.2128063145974108
;;; Empirical skewness: 0.3075092472847769
;;; Empirical kurtosis: 3.2503869673184735
;;; 
;;; Method :smc with options []
;;; Chain no. 2
;;; Elapsed time: 2880.581151 ms
;;; Empirical mean: 17.36803535424265
;;; Empirical variance: 1.680686605744518
;;; Empirical std: 1.2964129765412402
;;; Empirical skewness: -0.10470844769105667
;;; Empirical kurtosis: 2.8299934262396254
;;; 
;;; Method :smc with options []
;;; Chain no. 3
;;; Elapsed time: 2820.96466 ms
;;; Empirical mean: 17.196475925624185
;;; Empirical variance: 1.6627614268909041
;;; Empirical std: 1.2894810688377336
;;; Empirical skewness: 0.133347283167512
;;; Empirical kurtosis: 2.8978823191517025
;;; 
;;; Method :smc with options []
;;; Chain no. 4
;;; Elapsed time: 2851.848458 ms
;;; Empirical mean: 17.26594914950581
;;; Empirical variance: 1.5847314862800164
;;; Empirical std: 1.25886118626321
;;; Empirical skewness: 0.07208221727532622
;;; Empirical kurtosis: 2.9865452086508735
;;; 
;;; Method :smc with options []
;;; Chain no. 5
;;; Elapsed time: 2832.585628 ms
;;; Empirical mean: 17.23729862224398
;;; Empirical variance: 1.7641147007413212
;;; Empirical std: 1.328199796996416
;;; Empirical skewness: 0.03525950979755357
;;; Empirical kurtosis: 2.573286819823881
;;; 
;;; Method :smc with options []
;;; Chain no. 6
;;; Elapsed time: 2816.202992 ms
;;; Empirical mean: 17.167460948461795
;;; Empirical variance: 1.5446432059177975
;;; Empirical std: 1.2428367575501609
;;; Empirical skewness: 0.10097308918764433
;;; Empirical kurtosis: 3.2822110655327785
;;; 
;;; Method :smc with options []
;;; Chain no. 7
;;; Elapsed time: 2857.199492 ms
;;; Empirical mean: 17.06069105552567
;;; Empirical variance: 1.2972142481705817
;;; Empirical std: 1.1389531369510255
;;; Empirical skewness: 0.25271490289544235
;;; Empirical kurtosis: 3.640649614846602
;;; 
;;; Method :smc with options []
;;; Chain no. 8
;;; Elapsed time: 2762.519575 ms
;;; Empirical mean: 17.25056958592033
;;; Empirical variance: 1.4929161475422663
;;; Empirical std: 1.221849478267379
;;; Empirical skewness: 0.09968968398070634
;;; Empirical kurtosis: 3.2392953912987386
;;; 
;;; Method :smc with options []
;;; Chain no. 9
;;; Elapsed time: 2798.432938 ms
;;; Empirical mean: 17.253620469428643
;;; Empirical variance: 1.4601944995863227
;;; Empirical std: 1.2083850791806074
;;; Empirical skewness: 0.060525331134811684
;;; Empirical kurtosis: 2.911264955725158
;;; 
;;; Method :smc with options []
;;; Chain no. 10
;;; Elapsed time: 2781.449571 ms
;;; Empirical mean: 16.96464077262942
;;; Empirical variance: 1.3887374175223888
;;; Empirical std: 1.1784470363670947
;;; Empirical skewness: 0.4384753426004804
;;; Empirical kurtosis: 3.4783811830045983
;;; 
;;; Method :smc with options []
;;; Chain no. 11
;;; Elapsed time: 2949.151049 ms
;;; Empirical mean: 17.411640658416612
;;; Empirical variance: 1.4359321190662433
;;; Empirical std: 1.1983038508935215
;;; Empirical skewness: -0.16545382595756608
;;; Empirical kurtosis: 3.2603368456991344
;;; 
;;; Method :smc with options []
;;; Chain no. 12
;;; Elapsed time: 2885.597846 ms
;;; Empirical mean: 17.232593238393864
;;; Empirical variance: 1.6640148834693727
;;; Empirical std: 1.289967008674785
;;; Empirical skewness: 0.053155100258648634
;;; Empirical kurtosis: 2.7313549321658823
;;; 
;;; Method :smc with options []
;;; Chain no. 13
;;; Elapsed time: 2812.954719 ms
;;; Empirical mean: 17.20891140916813
;;; Empirical variance: 1.2523353502976402
;;; Empirical std: 1.1190779018002457
;;; Empirical skewness: -0.0051193620082727225
;;; Empirical kurtosis: 3.4249704697353285
;;; 
;;; Method :smc with options []
;;; Chain no. 14
;;; Elapsed time: 2845.335972 ms
;;; Empirical mean: 16.961660805833173
;;; Empirical variance: 1.3766745427397509
;;; Empirical std: 1.1733177501170562
;;; Empirical skewness: 0.45954463320183114
;;; Empirical kurtosis: 3.882003658985583
;;; 
;;; Method :smc with options []
;;; Chain no. 15
;;; Elapsed time: 2782.346823 ms
;;; Empirical mean: 17.091808729616254
;;; Empirical variance: 1.4224623665213156
;;; Empirical std: 1.1926702673083267
;;; Empirical skewness: 0.2572659696770232
;;; Empirical kurtosis: 3.5622070836282687
;;; 
;;; Method :smc with options []
;;; Chain no. 16
;;; Elapsed time: 2762.192995 ms
;;; Empirical mean: 17.25571977082251
;;; Empirical variance: 1.7371859489466779
;;; Empirical std: 1.3180235009083403
;;; Empirical skewness: 0.014384773106328673
;;; Empirical kurtosis: 2.513398679542546
;;; 
;;; Method :smc with options []
;;; Chain no. 17
;;; Elapsed time: 2811.67441 ms
;;; Empirical mean: 17.232377072295797
;;; Empirical variance: 1.4621001850455442
;;; Empirical std: 1.2091733478064857
;;; Empirical skewness: -0.09935443023666113
;;; Empirical kurtosis: 2.983981300424761
;;; 
;;; Method :smc with options []
;;; Chain no. 18
;;; Elapsed time: 2861.386555 ms
;;; Empirical mean: 17.2715277777989
;;; Empirical variance: 1.9457976672851716
;;; Empirical std: 1.3949185163604259
;;; Empirical skewness: 0.1884764803969658
;;; Empirical kurtosis: 2.7074378327452986
;;; 
;;; Method :smc with options []
;;; Chain no. 19
;;; Elapsed time: 2800.138465 ms
;;; Empirical mean: 17.162205325786893
;;; Empirical variance: 1.5339560687864007
;;; Empirical std: 1.2385298013315629
;;; Empirical skewness: 0.2491033016658732
;;; Empirical kurtosis: 3.1827326400872917
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 0
;;; Elapsed time: 3578.16857 ms
;;; Empirical mean: 24.0
;;; Empirical variance: 0.0
;;; Empirical std: 0.0
;;; Empirical skewness: NaN
;;; Empirical kurtosis: NaN
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 1
;;; Elapsed time: 3572.425201 ms
;;; Empirical mean: 24.130020661333553
;;; Empirical variance: 0.12945769701922477
;;; Empirical std: 0.3598023026874964
;;; Empirical skewness: 2.910530833838619
;;; Empirical kurtosis: 12.876066045630328
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 2
;;; Elapsed time: 3605.63086 ms
;;; Empirical mean: 23.00357051414179
;;; Empirical variance: 0.007219036736409558
;;; Empirical std: 0.08496491473784669
;;; Empirical skewness: 23.927321655367496
;;; Empirical kurtosis: 580.2170491902741
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 3
;;; Elapsed time: 3670.073274 ms
;;; Empirical mean: 22.005268120197915
;;; Empirical variance: 0.010508487305407398
;;; Empirical std: 0.10251091310395882
;;; Empirical skewness: 19.407336247084306
;;; Empirical kurtosis: 377.6447002073928
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 4
;;; Elapsed time: 3620.34169 ms
;;; Empirical mean: 23.013366500235296
;;; Empirical variance: 0.015678866496061834
;;; Empirical std: 0.12521528060129816
;;; Empirical skewness: 10.386311611146553
;;; Empirical kurtosis: 125.25189158758657
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 5
;;; Elapsed time: 3607.230058 ms
;;; Empirical mean: 22.004633573562153
;;; Empirical variance: 0.00978588942266512
;;; Empirical std: 0.09892365451531358
;;; Empirical skewness: 21.81847266221523
;;; Empirical kurtosis: 492.25764841134185
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 6
;;; Elapsed time: 3588.360404 ms
;;; Empirical mean: 23.007518979617213
;;; Empirical variance: 0.007462444562731145
;;; Empirical std: 0.08638544184485684
;;; Empirical skewness: 11.401944815360318
;;; Empirical kurtosis: 131.0043455725215
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 7
;;; Elapsed time: 3596.285244 ms
;;; Empirical mean: 23.155988613292166
;;; Empirical variance: 0.13165616581535242
;;; Empirical std: 0.3628445477271946
;;; Empirical skewness: 1.8961915721907383
;;; Empirical kurtosis: 4.5955424784471965
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 8
;;; Elapsed time: 3599.987981 ms
;;; Empirical mean: 21.0002789014113
;;; Empirical variance: 5.57725036603947E-4
;;; Empirical std: 0.023616202840506494
;;; Empirical skewness: 84.66399999528976
;;; Empirical kurtosis: 7168.992895202413
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 9
;;; Elapsed time: 3587.965181 ms
;;; Empirical mean: 24.018325671413724
;;; Empirical variance: 0.017989841180959813
;;; Empirical std: 0.13412621362343682
;;; Empirical skewness: 7.182404029365816
;;; Empirical kurtosis: 52.5869276410503
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 10
;;; Elapsed time: 3613.876677 ms
;;; Empirical mean: 22.094195565304222
;;; Empirical variance: 0.09866986759554076
;;; Empirical std: 0.3141176015372917
;;; Empirical skewness: 3.4044818990116266
;;; Empirical kurtosis: 14.640514819157215
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 11
;;; Elapsed time: 3631.24356 ms
;;; Empirical mean: 23.007894084861626
;;; Empirical variance: 0.010087221103431236
;;; Empirical std: 0.10043515870167795
;;; Empirical skewness: 14.234426393160962
;;; Empirical kurtosis: 228.23244706342464
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 12
;;; Elapsed time: 3637.698051 ms
;;; Empirical mean: 23.00793288562065
;;; Empirical variance: 0.007869954946375313
;;; Empirical std: 0.08871276653546159
;;; Empirical skewness: 11.093490454559317
;;; Empirical kurtosis: 124.06553046540009
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 13
;;; Elapsed time: 3596.156422 ms
;;; Empirical mean: 24.061160606229343
;;; Empirical variance: 0.06391773809857475
;;; Empirical std: 0.25281957617750794
;;; Empirical skewness: 4.251163639051875
;;; Empirical kurtosis: 21.63518105618209
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 14
;;; Elapsed time: 3540.066817 ms
;;; Empirical mean: 24.008805102935284
;;; Empirical variance: 0.01092884883140754
;;; Empirical std: 0.10454113463803394
;;; Empirical skewness: 13.233584147869335
;;; Empirical kurtosis: 198.22905467109558
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 15
;;; Elapsed time: 3607.133765 ms
;;; Empirical mean: 23.057735242951093
;;; Empirical variance: 0.06115102575322387
;;; Empirical std: 0.24728733439710143
;;; Empirical skewness: 4.458672029642263
;;; Empirical kurtosis: 23.94045206460706
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 16
;;; Elapsed time: 3526.947618 ms
;;; Empirical mean: 23.024316596622974
;;; Empirical variance: 0.0242987055686951
;;; Empirical std: 0.15588042073555966
;;; Empirical skewness: 6.40227542710821
;;; Empirical kurtosis: 43.841460268289275
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 17
;;; Elapsed time: 3553.901621 ms
;;; Empirical mean: 23.063989629779996
;;; Empirical variance: 0.05989495706061217
;;; Empirical std: 0.2447344623476885
;;; Empirical skewness: 3.563130145525416
;;; Empirical kurtosis: 13.695896433952077
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 18
;;; Elapsed time: 3588.838535 ms
;;; Empirical mean: 24.068882082238254
;;; Empirical variance: 0.0661339230786391
;;; Empirical std: 0.25716516692320346
;;; Empirical skewness: 3.579552650900638
;;; Empirical kurtosis: 14.673886371395142
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 19
;;; Elapsed time: 3667.6286 ms
;;; Empirical mean: 23.126237336480028
;;; Empirical variance: 0.1254677944691271
;;; Empirical std: 0.354214334081961
;;; Empirical skewness: 2.789958750162551
;;; Empirical kurtosis: 10.68763994560332
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 0
;;; Elapsed time: 2414.234767 ms
;;; Empirical mean: 31.887999999999998
;;; Empirical variance: 62.68745600000002
;;; Empirical std: 7.917541032416568
;;; Empirical skewness: -0.037788549491439824
;;; Empirical kurtosis: 1.8248690594552233
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 1
;;; Elapsed time: 2469.339145 ms
;;; Empirical mean: 31.402
;;; Empirical variance: 69.872396
;;; Empirical std: 8.35897098930245
;;; Empirical skewness: 0.1842116912794503
;;; Empirical kurtosis: 2.3734362186223628
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 2
;;; Elapsed time: 2430.120693 ms
;;; Empirical mean: 30.741999999999997
;;; Empirical variance: 105.08743600000003
;;; Empirical std: 10.251216318076603
;;; Empirical skewness: 1.3902542854523412
;;; Empirical kurtosis: 5.06494105686868
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 3
;;; Elapsed time: 2396.416132 ms
;;; Empirical mean: 31.535999999999987
;;; Empirical variance: 65.94470399999999
;;; Empirical std: 8.120634457971864
;;; Empirical skewness: -0.06358965168103102
;;; Empirical kurtosis: 2.9608870591221983
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 4
;;; Elapsed time: 2338.762697 ms
;;; Empirical mean: 33.486999999999995
;;; Empirical variance: 56.987831000000014
;;; Empirical std: 7.549028480539732
;;; Empirical skewness: 0.03198556334126778
;;; Empirical kurtosis: 3.2537342449398428
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 5
;;; Elapsed time: 2404.518445 ms
;;; Empirical mean: 28.944000000000003
;;; Empirical variance: 22.712864000000007
;;; Empirical std: 4.76580150656739
;;; Empirical skewness: -0.03122604449266453
;;; Empirical kurtosis: 3.3919035319072472
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 6
;;; Elapsed time: 2418.980098 ms
;;; Empirical mean: 29.53
;;; Empirical variance: 47.18510000000002
;;; Empirical std: 6.869141139909706
;;; Empirical skewness: 0.9923969852238784
;;; Empirical kurtosis: 4.065265916703224
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 7
;;; Elapsed time: 2361.113125 ms
;;; Empirical mean: 28.91399999999999
;;; Empirical variance: 107.70660400000004
;;; Empirical std: 10.378179223736698
;;; Empirical skewness: 1.3610494393201586
;;; Empirical kurtosis: 6.190275677677711
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 8
;;; Elapsed time: 2389.628425 ms
;;; Empirical mean: 31.219999999999995
;;; Empirical variance: 20.747600000000016
;;; Empirical std: 4.55495334773036
;;; Empirical skewness: -1.1817452202842265
;;; Empirical kurtosis: 4.841074533567021
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 9
;;; Elapsed time: 2372.500772 ms
;;; Empirical mean: 30.783
;;; Empirical variance: 88.675911
;;; Empirical std: 9.416788783868947
;;; Empirical skewness: 0.8782252290753616
;;; Empirical kurtosis: 2.9872151757144585
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 10
;;; Elapsed time: 2339.855276 ms
;;; Empirical mean: 29.810000000000006
;;; Empirical variance: 80.29390000000002
;;; Empirical std: 8.960686357640245
;;; Empirical skewness: 1.211022048726673
;;; Empirical kurtosis: 4.13834166430325
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 11
;;; Elapsed time: 2393.76794 ms
;;; Empirical mean: 30.082
;;; Empirical variance: 65.36727600000005
;;; Empirical std: 8.085003153988255
;;; Empirical skewness: 1.062476445492388
;;; Empirical kurtosis: 7.765410329511138
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 12
;;; Elapsed time: 2370.679927 ms
;;; Empirical mean: 29.78799999999999
;;; Empirical variance: 69.76705600000004
;;; Empirical std: 8.352667597839629
;;; Empirical skewness: 0.5396382966159942
;;; Empirical kurtosis: 2.93044150206931
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 13
;;; Elapsed time: 2324.227702 ms
;;; Empirical mean: 29.51600000000001
;;; Empirical variance: 41.117744000000016
;;; Empirical std: 6.412311907572807
;;; Empirical skewness: -0.05132240802550782
;;; Empirical kurtosis: 2.600927622663963
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 14
;;; Elapsed time: 2462.17951 ms
;;; Empirical mean: 27.420999999999996
;;; Empirical variance: 31.78175900000003
;;; Empirical std: 5.637531285944232
;;; Empirical skewness: 2.0551915057662007
;;; Empirical kurtosis: 16.1309626352245
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 15
;;; Elapsed time: 2425.563665 ms
;;; Empirical mean: 29.189999999999994
;;; Empirical variance: 113.27790000000002
;;; Empirical std: 10.643209102521665
;;; Empirical skewness: 1.0502690211708952
;;; Empirical kurtosis: 4.238143284007632
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 16
;;; Elapsed time: 2382.712809 ms
;;; Empirical mean: 29.392000000000014
;;; Empirical variance: 67.05833600000003
;;; Empirical std: 8.188915434903453
;;; Empirical skewness: 0.70679729099247
;;; Empirical kurtosis: 2.963095036320844
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 17
;;; Elapsed time: 2389.82546 ms
;;; Empirical mean: 29.444000000000003
;;; Empirical variance: 56.98286400000003
;;; Empirical std: 7.548699490640757
;;; Empirical skewness: 1.1867753829058467
;;; Empirical kurtosis: 3.947699027916939
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 18
;;; Elapsed time: 2362.856834 ms
;;; Empirical mean: 29.471999999999987
;;; Empirical variance: 36.977216000000006
;;; Empirical std: 6.080889408630945
;;; Empirical skewness: -0.28875976243573576
;;; Empirical kurtosis: 2.402365011996215
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 19
;;; Elapsed time: 2398.484725 ms
;;; Empirical mean: 29.352
;;; Empirical variance: 42.464096000000005
;;; Empirical std: 6.516448112277117
;;; Empirical skewness: 0.9794995710415547
;;; Empirical kurtosis: 5.475112731235138
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 0
;;; Elapsed time: 3644.628316 ms
;;; Empirical mean: 24.025
;;; Empirical variance: 0.026375000000000013
;;; Empirical std: 0.16240381768911719
;;; Empirical skewness: 6.771780668854656
;;; Empirical kurtosis: 51.75097145167469
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 1
;;; Elapsed time: 3630.528828 ms
;;; Empirical mean: 23.839000000000002
;;; Empirical variance: 0.219079
;;; Empirical std: 0.4680587569953157
;;; Empirical skewness: -0.26341989606087074
;;; Empirical kurtosis: 5.518425226471973
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 2
;;; Elapsed time: 3621.790042 ms
;;; Empirical mean: 25.015
;;; Empirical variance: 0.014775000000000017
;;; Empirical std: 0.12155245781143226
;;; Empirical skewness: 7.980093676960333
;;; Empirical kurtosis: 64.68189509306237
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 3
;;; Elapsed time: 3698.796255 ms
;;; Empirical mean: 23.027
;;; Empirical variance: 0.02827100000000003
;;; Empirical std: 0.16813982276664868
;;; Empirical skewness: 6.456398259631137
;;; Empirical kurtosis: 46.99565586183478
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 4
;;; Elapsed time: 3747.845787 ms
;;; Empirical mean: 24.008000000000003
;;; Empirical variance: 0.41793600000000003
;;; Empirical std: 0.6464796980571007
;;; Empirical skewness: 0.10351722918599313
;;; Empirical kurtosis: 2.7305383533559935
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 5
;;; Elapsed time: 3644.128025 ms
;;; Empirical mean: 22.433999999999994
;;; Empirical variance: 0.6616439999999997
;;; Empirical std: 0.813415023220004
;;; Empirical skewness: 1.4173272455098263
;;; Empirical kurtosis: 3.1581809247053587
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 6
;;; Elapsed time: 3655.542349 ms
;;; Empirical mean: 22.8
;;; Empirical variance: 0.15999999999999998
;;; Empirical std: 0.39999999999999997
;;; Empirical skewness: -1.500000000000005
;;; Empirical kurtosis: 3.25000000000001
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 7
;;; Elapsed time: 3682.93061 ms
;;; Empirical mean: 22.49
;;; Empirical variance: 1.0119000000000005
;;; Empirical std: 1.0059324032955697
;;; Empirical skewness: 1.6925561687809545
;;; Empirical kurtosis: 4.168018106032441
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 8
;;; Elapsed time: 3638.612646 ms
;;; Empirical mean: 23.433000000000003
;;; Empirical variance: 1.5135109999999992
;;; Empirical std: 1.2302483489117144
;;; Empirical skewness: -1.3986258571338968
;;; Empirical kurtosis: 3.194539089689296
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 9
;;; Elapsed time: 3642.724351 ms
;;; Empirical mean: 23.415000000000003
;;; Empirical variance: 0.7027750000000003
;;; Empirical std: 0.8383167659065398
;;; Empirical skewness: 1.5930999446550818
;;; Empirical kurtosis: 3.768221105049439
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 10
;;; Elapsed time: 3588.615769 ms
;;; Empirical mean: 23.844000000000005
;;; Empirical variance: 0.17766400000000004
;;; Empirical std: 0.42150207591422373
;;; Empirical skewness: -0.7619184684720195
;;; Empirical kurtosis: 5.19135501674245
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 11
;;; Elapsed time: 4089.364701 ms
;;; Empirical mean: 23.011
;;; Empirical variance: 0.010879000000000019
;;; Empirical std: 0.10430244484191163
;;; Empirical skewness: 9.37657790747216
;;; Empirical kurtosis: 88.92021325489486
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 12
;;; Elapsed time: 3661.929485 ms
;;; Empirical mean: 24.024
;;; Empirical variance: 0.023424000000000025
;;; Empirical std: 0.15304901175767202
;;; Empirical skewness: 6.220229644522845
;;; Empirical kurtosis: 39.691256830600885
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 13
;;; Elapsed time: 3562.475138 ms
;;; Empirical mean: 24.595000000000002
;;; Empirical variance: 0.252975
;;; Empirical std: 0.5029662016477847
;;; Empirical skewness: -0.2452512668408178
;;; Empirical kurtosis: 1.4153611454465702
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 14
;;; Elapsed time: 3587.777878 ms
;;; Empirical mean: 24.006000000000004
;;; Empirical variance: 0.005964000000000012
;;; Empirical std: 0.07722693830523136
;;; Empirical skewness: 12.79346328731851
;;; Empirical kurtosis: 164.6727028839677
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 15
;;; Elapsed time: 3627.137158 ms
;;; Empirical mean: 24.037
;;; Empirical variance: 0.04163100000000003
;;; Empirical std: 0.20403676139362736
;;; Empirical skewness: 5.9249821761342725
;;; Empirical kurtosis: 41.0858747142033
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 16
;;; Elapsed time: 3594.517017 ms
;;; Empirical mean: 23.062
;;; Empirical variance: 0.062156000000000045
;;; Empirical std: 0.24931105069771786
;;; Empirical skewness: 4.013935393357544
;;; Empirical kurtosis: 18.928015304828975
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 17
;;; Elapsed time: 3620.221136 ms
;;; Empirical mean: 22.431999999999995
;;; Empirical variance: 0.6473759999999996
;;; Empirical std: 0.8045967934313433
;;; Empirical skewness: 1.379173228125738
;;; Empirical kurtosis: 2.9704837995531075
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 18
;;; Elapsed time: 3519.709073 ms
;;; Empirical mean: 24.824
;;; Empirical variance: 0.19502399999999998
;;; Empirical std: 0.4416152171291202
;;; Empirical skewness: -0.7149509116102932
;;; Empirical kurtosis: 4.139544240723992
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 19
;;; Elapsed time: 3785.029435 ms
;;; Empirical mean: 22.404
;;; Empirical variance: 0.644784
;;; Empirical std: 0.8029844332239574
;;; Empirical skewness: 1.484462152291243
;;; Empirical kurtosis: 3.2036278815851493
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 0
;;; Elapsed time: 671.481893 ms
;;; Empirical mean: 16.103585657370516
;;; Empirical variance: 17.351819812383933
;;; Empirical std: 4.165551561604289
;;; Empirical skewness: 0.40437526115599454
;;; Empirical kurtosis: 2.3044076534656255
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 1
;;; Elapsed time: 680.802613 ms
;;; Empirical mean: 16.46825396825397
;;; Empirical variance: 39.955341395817584
;;; Empirical std: 6.321023761687468
;;; Empirical skewness: 0.42253166753710797
;;; Empirical kurtosis: 1.7115992980383132
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 2
;;; Elapsed time: 705.207815 ms
;;; Empirical mean: 15.569721115537847
;;; Empirical variance: 31.886573228996376
;;; Empirical std: 5.646819744687834
;;; Empirical skewness: 0.6261215579395402
;;; Empirical kurtosis: 2.690955137926415
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 3
;;; Elapsed time: 836.591964 ms
;;; Empirical mean: 15.16733067729084
;;; Empirical variance: 45.10347454802307
;;; Empirical std: 6.715912041415006
;;; Empirical skewness: 1.0554662249881654
;;; Empirical kurtosis: 2.950946839879203
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 4
;;; Elapsed time: 755.007691 ms
;;; Empirical mean: 14.892430278884465
;;; Empirical variance: 37.60197457183221
;;; Empirical std: 6.132044893168364
;;; Empirical skewness: 0.9467513964182553
;;; Empirical kurtosis: 2.4118896494329483
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 5
;;; Elapsed time: 696.756662 ms
;;; Empirical mean: 17.063745019920315
;;; Empirical variance: 17.382390755702282
;;; Empirical std: 4.169219442018168
;;; Empirical skewness: 0.3561023308038003
;;; Empirical kurtosis: 2.328062246747887
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 6
;;; Elapsed time: 716.07086 ms
;;; Empirical mean: 15.119047619047626
;;; Empirical variance: 29.747732426303852
;;; Empirical std: 5.4541481852168125
;;; Empirical skewness: 0.5691904079400153
;;; Empirical kurtosis: 2.2392131788067573
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 7
;;; Elapsed time: 734.555674 ms
;;; Empirical mean: 14.944444444444445
;;; Empirical variance: 34.54453262786596
;;; Empirical std: 5.8774597087403295
;;; Empirical skewness: 1.0777553478020707
;;; Empirical kurtosis: 3.0194244260508394
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 8
;;; Elapsed time: 712.271705 ms
;;; Empirical mean: 17.645418326693232
;;; Empirical variance: 15.487817653688044
;;; Empirical std: 3.935456473357067
;;; Empirical skewness: 0.7068132747783032
;;; Empirical kurtosis: 2.833368707624909
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 9
;;; Elapsed time: 714.337038 ms
;;; Empirical mean: 14.595238095238097
;;; Empirical variance: 8.16156462585034
;;; Empirical std: 2.8568452225926313
;;; Empirical skewness: 0.6178226891639348
;;; Empirical kurtosis: 2.213682268638453
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 10
;;; Elapsed time: 732.212672 ms
;;; Empirical mean: 15.444000000000006
;;; Empirical variance: 34.75086400000001
;;; Empirical std: 5.894986344343811
;;; Empirical skewness: 0.8483443595825547
;;; Empirical kurtosis: 2.549317421205896
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 11
;;; Elapsed time: 744.536726 ms
;;; Empirical mean: 18.800796812749002
;;; Empirical variance: 23.808923667878293
;;; Empirical std: 4.8794388681362015
;;; Empirical skewness: 0.26753762923027824
;;; Empirical kurtosis: 2.2811353413827717
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 12
;;; Elapsed time: 746.660436 ms
;;; Empirical mean: 16.236000013039764
;;; Empirical variance: 38.94030389189632
;;; Empirical std: 6.24021665424337
;;; Empirical skewness: 0.5142354381773193
;;; Empirical kurtosis: 2.3074532601631783
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 13
;;; Elapsed time: 728.294335 ms
;;; Empirical mean: 15.040000000000001
;;; Empirical variance: 25.110400000000016
;;; Empirical std: 5.011027838677412
;;; Empirical skewness: 1.1841053373687525
;;; Empirical kurtosis: 3.051402963478485
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 14
;;; Elapsed time: 737.376941 ms
;;; Empirical mean: 15.865079365079369
;;; Empirical variance: 34.22782816830436
;;; Empirical std: 5.850455381276261
;;; Empirical skewness: 0.6145117815296725
;;; Empirical kurtosis: 2.226823905586047
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 15
;;; Elapsed time: 757.367506 ms
;;; Empirical mean: 16.713147410358566
;;; Empirical variance: 30.148791289027162
;;; Empirical std: 5.490791499322039
;;; Empirical skewness: 0.7370765556896582
;;; Empirical kurtosis: 2.4352390743201213
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 16
;;; Elapsed time: 814.293421 ms
;;; Empirical mean: 15.615999999999996
;;; Empirical variance: 20.68454400000001
;;; Empirical std: 4.548026385147739
;;; Empirical skewness: 0.9511380262097554
;;; Empirical kurtosis: 2.8507791863145697
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 17
;;; Elapsed time: 786.330376 ms
;;; Empirical mean: 14.908000000000003
;;; Empirical variance: 12.731536000000009
;;; Empirical std: 3.568127800401775
;;; Empirical skewness: 1.3321964195156553
;;; Empirical kurtosis: 4.65066877922056
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 18
;;; Elapsed time: 782.640731 ms
;;; Empirical mean: 18.509960159356407
;;; Empirical variance: 37.07061792662056
;;; Empirical std: 6.088564521019758
;;; Empirical skewness: 0.6892838068447272
;;; Empirical kurtosis: 2.841973625784831
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 19
;;; Elapsed time: 733.226458 ms
;;; Empirical mean: 15.119999999999996
;;; Empirical variance: 20.0896
;;; Empirical std: 4.482142344906061
;;; Empirical skewness: 0.5145682784418755
;;; Empirical kurtosis: 2.1393874588070374
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 0
;;; Elapsed time: 16448.609162 ms
;;; Empirical mean: 24.06076574710153
;;; Empirical variance: 0.05707327108072157
;;; Empirical std: 0.2389001278373906
;;; Empirical skewness: 3.6771370268787504
;;; Empirical kurtosis: 14.52133671444277
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 1
;;; Elapsed time: 17296.029624 ms
;;; Empirical mean: 23.0
;;; Empirical variance: 0.0
;;; Empirical std: 0.0
;;; Empirical skewness: NaN
;;; Empirical kurtosis: NaN
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 2
;;; Elapsed time: 19008.506674 ms
;;; Empirical mean: 24.01350373221362
;;; Empirical variance: 0.016322210810728946
;;; Empirical std: 0.1277584079844804
;;; Empirical skewness: 10.474526027917989
;;; Empirical kurtosis: 125.0375476175644
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 3
;;; Elapsed time: 18662.753607 ms
;;; Empirical mean: 21.00814930867521
;;; Empirical variance: 0.03381340231613746
;;; Empirical std: 0.1838842089907055
;;; Empirical skewness: 22.693888128592018
;;; Empirical kurtosis: 520.5597119231552
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 4
;;; Elapsed time: 22547.262003 ms
;;; Empirical mean: 22.0
;;; Empirical variance: 0.0
;;; Empirical std: 0.0
;;; Empirical skewness: NaN
;;; Empirical kurtosis: NaN
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 5
;;; Elapsed time: 22930.542722 ms
;;; Empirical mean: 22.066737104459687
;;; Empirical variance: 0.062283263348023914
;;; Empirical std: 0.2495661502448277
;;; Empirical skewness: 3.472128693056142
;;; Empirical kurtosis: 13.055677661143749
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 6
;;; Elapsed time: 22858.803964 ms
;;; Empirical mean: 23.0
;;; Empirical variance: 0.0
;;; Empirical std: 0.0
;;; Empirical skewness: NaN
;;; Empirical kurtosis: NaN
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 7
;;; Elapsed time: 22204.0248 ms
;;; Empirical mean: 23.018458482973767
;;; Empirical variance: 0.01811776738007289
;;; Empirical std: 0.1346022562220741
;;; Empirical skewness: 7.155028905782375
;;; Empirical kurtosis: 52.194438642581424
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 8
;;; Elapsed time: 22786.983582 ms
;;; Empirical mean: 22.208771262437367
;;; Empirical variance: 0.3891149739520852
;;; Empirical std: 0.6237908094482358
;;; Empirical skewness: 2.7153461804243597
;;; Empirical kurtosis: 8.673707501746106
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 9
;;; Elapsed time: 22993.056021 ms
;;; Empirical mean: 23.105842986554112
;;; Empirical variance: 0.10763000874408324
;;; Empirical std: 0.3280701277838067
;;; Empirical skewness: 3.099696953270574
;;; Empirical kurtosis: 12.350670052350244
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 10
;;; Elapsed time: 22312.874935 ms
;;; Empirical mean: 23.797128844615752
;;; Empirical variance: 0.18333816681389195
;;; Empirical std: 0.42818006354090327
;;; Empirical skewness: -1.0565315036800684
;;; Empirical kurtosis: 3.2791962983905387
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 11
;;; Elapsed time: 22482.710216 ms
;;; Empirical mean: 24.052158348436286
;;; Empirical variance: 0.05322672387630383
;;; Empirical std: 0.23070917596901913
;;; Empirical skewness: 4.483296516537238
;;; Empirical kurtosis: 23.408420149710572
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 12
;;; Elapsed time: 22639.262929 ms
;;; Empirical mean: 24.049245450582077
;;; Empirical variance: 0.04922255328060508
;;; Empirical std: 0.22186156332408072
;;; Empirical skewness: 4.492495717929782
;;; Empirical kurtosis: 22.9789606937189
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 13
;;; Elapsed time: 22458.020225 ms
;;; Empirical mean: 23.032683759749755
;;; Empirical variance: 0.03666381877867557
;;; Empirical std: 0.19147798510187944
;;; Empirical skewness: 6.598481368732386
;;; Empirical kurtosis: 55.4050023353241
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 14
;;; Elapsed time: 23139.760675 ms
;;; Empirical mean: 23.04879274535655
;;; Empirical variance: 0.09710108379539133
;;; Empirical std: 0.31161046804526854
;;; Empirical skewness: 6.290064013229072
;;; Empirical kurtosis: 41.13368694137826
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 15
;;; Elapsed time: 17201.70979 ms
;;; Empirical mean: 23.01771689053311
;;; Empirical variance: 0.022212147418797762
;;; Empirical std: 0.149037402751114
;;; Empirical skewness: 9.491138595702656
;;; Empirical kurtosis: 105.15717594293463
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 16
;;; Elapsed time: 16629.272479 ms
;;; Empirical mean: 24.002978071521422
;;; Empirical variance: 0.0029692026114373072
;;; Empirical std: 0.054490390083365224
;;; Empirical skewness: 18.242553511479052
;;; Empirical kurtosis: 333.7907586191754
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 17
;;; Elapsed time: 16751.589289 ms
;;; Empirical mean: 23.31884886681101
;;; Empirical variance: 0.22738115567384695
;;; Empirical std: 0.4768450017289129
;;; Empirical skewness: 0.9178943795728723
;;; Empirical kurtosis: 2.209969237341941
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 18
;;; Elapsed time: 17741.45127 ms
;;; Empirical mean: 24.04068600668004
;;; Empirical variance: 0.11697844829083738
;;; Empirical std: 0.3420211225799327
;;; Empirical skewness: 0.83656097413172
;;; Empirical kurtosis: 9.228417450219618
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 19
;;; Elapsed time: 18586.456314 ms
;;; Empirical mean: 24.003774548474684
;;; Empirical variance: 0.00476684751841107
;;; Empirical std: 0.06904236031894528
;;; Empirical skewness: 20.479673042580668
;;; Empirical kurtosis: 471.6929163068164
;;; 
;;; Method :importance with options []
;;; Chain no. 0
;;; Elapsed time: 16881.268644 ms
;;; Empirical mean: 17.214731759272265
;;; Empirical variance: 1.5366208455021326
;;; Empirical std: 1.2396051167618392
;;; Empirical skewness: 0.0688375942354829
;;; Empirical kurtosis: 3.0299098312386197
;;; 
;;; Method :importance with options []
;;; Chain no. 1
;;; Elapsed time: 16776.455728 ms
;;; Empirical mean: 17.186141783124135
;;; Empirical variance: 1.5721665225289496
;;; Empirical std: 1.2538606471729423
;;; Empirical skewness: 0.1068049872336521
;;; Empirical kurtosis: 3.005101042124169
;;; 
;;; Method :importance with options []
;;; Chain no. 2
;;; Elapsed time: 17035.788895 ms
;;; Empirical mean: 17.18003055383077
;;; Empirical variance: 1.5429895953096076
;;; Empirical std: 1.2421713228494722
;;; Empirical skewness: 0.10669048669331216
;;; Empirical kurtosis: 3.0102759350363963
;;; 
;;; Method :importance with options []
;;; Chain no. 3
;;; Elapsed time: 16657.406455 ms
;;; Empirical mean: 17.128305994197163
;;; Empirical variance: 1.5417031575077313
;;; Empirical std: 1.2416533966883558
;;; Empirical skewness: 0.1986506048516132
;;; Empirical kurtosis: 2.9850882098985205
;;; 
;;; Method :importance with options []
;;; Chain no. 4
;;; Elapsed time: 16608.034827 ms
;;; Empirical mean: 17.191144550655697
;;; Empirical variance: 1.4929613024943813
;;; Empirical std: 1.2218679562433827
;;; Empirical skewness: 0.11212866248748571
;;; Empirical kurtosis: 3.183018122951635
;;; 
;;; Method :importance with options []
;;; Chain no. 5
;;; Elapsed time: 16758.527923 ms
;;; Empirical mean: 17.204920758011312
;;; Empirical variance: 1.518990456571378
;;; Empirical std: 1.2324733086648887
;;; Empirical skewness: 0.11748150980955778
;;; Empirical kurtosis: 3.095753003345666
;;; 
;;; Method :importance with options []
;;; Chain no. 6
;;; Elapsed time: 16797.144686 ms
;;; Empirical mean: 17.180816272777193
;;; Empirical variance: 1.5847390815411935
;;; Empirical std: 1.2588642029786983
;;; Empirical skewness: 0.12554981592293715
;;; Empirical kurtosis: 2.984174889856054
;;; 
;;; Method :importance with options []
;;; Chain no. 7
;;; Elapsed time: 16643.058566 ms
;;; Empirical mean: 17.136412312190554
;;; Empirical variance: 1.5576210920508065
;;; Empirical std: 1.2480469109976622
;;; Empirical skewness: 0.15829506442996102
;;; Empirical kurtosis: 3.082346028383592
;;; 
;;; Method :importance with options []
;;; Chain no. 8
;;; Elapsed time: 17100.223709 ms
;;; Empirical mean: 17.164856594807002
;;; Empirical variance: 1.5365840215137339
;;; Empirical std: 1.2395902635603968
;;; Empirical skewness: 0.13601140999240763
;;; Empirical kurtosis: 3.057625330346894
;;; 
;;; Method :importance with options []
;;; Chain no. 9
;;; Elapsed time: 17978.157542 ms
;;; Empirical mean: 17.155399235426692
;;; Empirical variance: 1.5754778822640492
;;; Empirical std: 1.2551804182124773
;;; Empirical skewness: 0.14165821716371063
;;; Empirical kurtosis: 2.96464393378308
;;; 
;;; Method :importance with options []
;;; Chain no. 10
;;; Elapsed time: 17438.37002 ms
;;; Empirical mean: 17.137687552888057
;;; Empirical variance: 1.4780385438614576
;;; Empirical std: 1.2157460852749877
;;; Empirical skewness: 0.1735459149976692
;;; Empirical kurtosis: 3.283533446214002
;;; 
;;; Method :importance with options []
;;; Chain no. 11
;;; Elapsed time: 20100.185579 ms
;;; Empirical mean: 17.174831993466444
;;; Empirical variance: 1.5247998627775285
;;; Empirical std: 1.234827867671251
;;; Empirical skewness: 0.15659884896080908
;;; Empirical kurtosis: 3.0724701026597683
;;; 
;;; Method :importance with options []
;;; Chain no. 12
;;; Elapsed time: 21461.375808 ms
;;; Empirical mean: 17.21115147288754
;;; Empirical variance: 1.5266721775549268
;;; Empirical std: 1.2355857629298448
;;; Empirical skewness: 0.08493176471302417
;;; Empirical kurtosis: 3.147452269891611
;;; 
;;; Method :importance with options []
;;; Chain no. 13
;;; Elapsed time: 17947.408476 ms
;;; Empirical mean: 17.162259383531538
;;; Empirical variance: 1.6613890044118267
;;; Empirical std: 1.288948798211871
;;; Empirical skewness: 0.17361839271270693
;;; Empirical kurtosis: 2.930853601453965
;;; 
;;; Method :importance with options []
;;; Chain no. 14
;;; Elapsed time: 19700.070994 ms
;;; Empirical mean: 17.18846326333386
;;; Empirical variance: 1.6145843398424615
;;; Empirical std: 1.2706629528881612
;;; Empirical skewness: 0.11787695296296002
;;; Empirical kurtosis: 2.9719526572936767
;;; 
;;; Method :importance with options []
;;; Chain no. 15
;;; Elapsed time: 21072.421941 ms
;;; Empirical mean: 17.202474591391884
;;; Empirical variance: 1.5936070499190484
;;; Empirical std: 1.2623814993570877
;;; Empirical skewness: 0.09662579539287271
;;; Empirical kurtosis: 3.017288907211825
;;; 
;;; Method :importance with options []
;;; Chain no. 16
;;; Elapsed time: 18292.181255 ms
;;; Empirical mean: 17.187319559783152
;;; Empirical variance: 1.5667307329389366
;;; Empirical std: 1.2516911491813532
;;; Empirical skewness: 0.11945952663484971
;;; Empirical kurtosis: 3.0274127579940777
;;; 
;;; Method :importance with options []
;;; Chain no. 17
;;; Elapsed time: 19097.018563 ms
;;; Empirical mean: 17.182578243321007
;;; Empirical variance: 1.6071390981847624
;;; Empirical std: 1.267729899538842
;;; Empirical skewness: 0.14367004343369527
;;; Empirical kurtosis: 2.954959618603737
;;; 
;;; Method :importance with options []
;;; Chain no. 18
;;; Elapsed time: 19967.377157 ms
;;; Empirical mean: 17.179929486523584
;;; Empirical variance: 1.5544980136636584
;;; Empirical std: 1.2467950969039212
;;; Empirical skewness: 0.12976377387723614
;;; Empirical kurtosis: 3.0773649656511415
;;; 
;;; Method :importance with options []
;;; Chain no. 19
;;; Elapsed time: 18806.906018 ms
;;; Empirical mean: 17.25524470212356
;;; Empirical variance: 1.4799867373601407
;;; Empirical std: 1.2165470551360276
;;; Empirical skewness: -0.007322834097783841
;;; Empirical kurtosis: 3.137055319502677
;;; 
;;; Method :lmh with options []
;;; Chain no. 0
;;; Elapsed time: 17782.543583 ms
;;; Empirical mean: 17.268200000000004
;;; Empirical variance: 1.4240687600000017
;;; Empirical std: 1.1933435213717807
;;; Empirical skewness: 0.4498651669065833
;;; Empirical kurtosis: 3.6898232499311607
;;; 
;;; Method :lmh with options []
;;; Chain no. 1
;;; Elapsed time: 18597.739616 ms
;;; Empirical mean: 17.1329
;;; Empirical variance: 1.491637590000002
;;; Empirical std: 1.2213261603683112
;;; Empirical skewness: 0.21223067839238904
;;; Empirical kurtosis: 2.9956981016497424
;;; 
;;; Method :lmh with options []
;;; Chain no. 2
;;; Elapsed time: 17497.43031 ms
;;; Empirical mean: 17.229000000000003
;;; Empirical variance: 1.552959000000001
;;; Empirical std: 1.2461777561808753
;;; Empirical skewness: -0.04665609985314975
;;; Empirical kurtosis: 2.7731375313167073
;;; 
;;; Method :lmh with options []
;;; Chain no. 3
;;; Elapsed time: 23908.158422 ms
;;; Empirical mean: 17.1435
;;; Empirical variance: 1.6519077500000041
;;; Empirical std: 1.28526563402279
;;; Empirical skewness: -0.013565412879317697
;;; Empirical kurtosis: 3.0501727141416595
;;; 
;;; Method :lmh with options []
;;; Chain no. 4
;;; Elapsed time: 17059.282834 ms
;;; Empirical mean: 17.2317
;;; Empirical variance: 1.5410151100000014
;;; Empirical std: 1.241376296696534
;;; Empirical skewness: -0.025038631469305104
;;; Empirical kurtosis: 2.705419722856331
;;; 
;;; Method :lmh with options []
;;; Chain no. 5
;;; Elapsed time: 16035.476891 ms
;;; Empirical mean: 17.228299999999997
;;; Empirical variance: 1.4385791100000014
;;; Empirical std: 1.1994078163827353
;;; Empirical skewness: -0.03841450022472036
;;; Empirical kurtosis: 3.071177460315982
;;; 
;;; Method :lmh with options []
;;; Chain no. 6
;;; Elapsed time: 16312.365266 ms
;;; Empirical mean: 17.190300000000004
;;; Empirical variance: 1.555685910000001
;;; Empirical std: 1.2472713858659634
;;; Empirical skewness: 0.09383162717757511
;;; Empirical kurtosis: 3.0673907913128913
;;; 
;;; Method :lmh with options []
;;; Chain no. 7
;;; Elapsed time: 16895.022335 ms
;;; Empirical mean: 17.122899999999998
;;; Empirical variance: 1.63839559
;;; Empirical std: 1.2799982773425909
;;; Empirical skewness: 0.11213691876558733
;;; Empirical kurtosis: 2.876528810624062
;;; 
;;; Method :lmh with options []
;;; Chain no. 8
;;; Elapsed time: 15738.015338 ms
;;; Empirical mean: 17.273
;;; Empirical variance: 1.3554710000000014
;;; Empirical std: 1.1642469669275508
;;; Empirical skewness: 0.045575500443784035
;;; Empirical kurtosis: 2.8251073283356156
;;; 
;;; Method :lmh with options []
;;; Chain no. 9
;;; Elapsed time: 17007.989995 ms
;;; Empirical mean: 17.171599999999998
;;; Empirical variance: 1.2085534400000029
;;; Empirical std: 1.0993422760905736
;;; Empirical skewness: 0.04860980296990981
;;; Empirical kurtosis: 3.040717457397425
;;; 
;;; Method :lmh with options []
;;; Chain no. 10
;;; Elapsed time: 16874.872749 ms
;;; Empirical mean: 17.2271
;;; Empirical variance: 1.3525255900000004
;;; Empirical std: 1.1629813369095827
;;; Empirical skewness: 0.07292723260091509
;;; Empirical kurtosis: 2.994055317220534
;;; 
;;; Method :lmh with options []
;;; Chain no. 11
;;; Elapsed time: 18442.672725 ms
;;; Empirical mean: 17.23
;;; Empirical variance: 1.4685000000000012
;;; Empirical std: 1.2118168178400568
;;; Empirical skewness: 0.31567927504695176
;;; Empirical kurtosis: 2.809225551355445
;;; 
;;; Method :lmh with options []
;;; Chain no. 12
;;; Elapsed time: 19100.228323 ms
;;; Empirical mean: 17.1756
;;; Empirical variance: 1.4471646400000002
;;; Empirical std: 1.20298156261848
;;; Empirical skewness: -0.05272006944332692
;;; Empirical kurtosis: 2.512825078574957
;;; 
;;; Method :lmh with options []
;;; Chain no. 13
;;; Elapsed time: 18447.852191 ms
;;; Empirical mean: 17.2222
;;; Empirical variance: 1.8092271600000016
;;; Empirical std: 1.345075150316889
;;; Empirical skewness: 0.09590880250400112
;;; Empirical kurtosis: 2.8544738598766743
;;; 
;;; Method :lmh with options []
;;; Chain no. 14
;;; Elapsed time: 17018.736138 ms
;;; Empirical mean: 17.2847
;;; Empirical variance: 1.5880459100000004
;;; Empirical std: 1.260176935989546
;;; Empirical skewness: 0.008831562340522505
;;; Empirical kurtosis: 3.1096114806638018
;;; 
;;; Method :lmh with options []
;;; Chain no. 15
;;; Elapsed time: 17027.491731 ms
;;; Empirical mean: 17.1704
;;; Empirical variance: 1.624363840000004
;;; Empirical std: 1.2745053314914003
;;; Empirical skewness: 0.3010623451265734
;;; Empirical kurtosis: 3.0909078452262255
;;; 
;;; Method :lmh with options []
;;; Chain no. 16
;;; Elapsed time: 15677.437836 ms
;;; Empirical mean: 17.2808
;;; Empirical variance: 1.5111513600000004
;;; Empirical std: 1.2292889652152583
;;; Empirical skewness: 0.1012387895214585
;;; Empirical kurtosis: 2.8124431239927494
;;; 
;;; Method :lmh with options []
;;; Chain no. 17
;;; Elapsed time: 19643.54278 ms
;;; Empirical mean: 17.207200000000004
;;; Empirical variance: 1.602068160000001
;;; Empirical std: 1.2657283120796505
;;; Empirical skewness: -0.15422806147731688
;;; Empirical kurtosis: 3.2182678163425416
;;; 
;;; Method :lmh with options []
;;; Chain no. 18
;;; Elapsed time: 19552.890598 ms
;;; Empirical mean: 17.0311
;;; Empirical variance: 1.6339327900000002
;;; Empirical std: 1.2782538050011822
;;; Empirical skewness: 0.23260962136670676
;;; Empirical kurtosis: 3.1680208364098563
;;; 
;;; Method :lmh with options []
;;; Chain no. 19
;;; Elapsed time: 17818.613388 ms
;;; Empirical mean: 17.1818
;;; Empirical variance: 1.8925487600000006
;;; Empirical std: 1.3756993712290488
;;; Empirical skewness: -0.07082745332776287
;;; Empirical kurtosis: 2.602784732413019
;;; 
;;; Method :rmh with options []
;;; Chain no. 0
;;; Elapsed time: 19221.31413 ms
;;; Empirical mean: 17.115299999999998
;;; Empirical variance: 1.6104059100000006
;;; Empirical std: 1.2690176949120924
;;; Empirical skewness: 0.015623048681486383
;;; Empirical kurtosis: 2.3698341305785853
;;; 
;;; Method :rmh with options []
;;; Chain no. 1
;;; Elapsed time: 20349.025935 ms
;;; Empirical mean: 17.244600000000002
;;; Empirical variance: 1.86057084
;;; Empirical std: 1.3640274337417118
;;; Empirical skewness: 0.007281266372217052
;;; Empirical kurtosis: 2.9890432670004534
;;; 
;;; Method :rmh with options []
;;; Chain no. 2
;;; Elapsed time: 18623.044259 ms
;;; Empirical mean: 17.212799999999994
;;; Empirical variance: 1.5753161600000005
;;; Empirical std: 1.2551159946395394
;;; Empirical skewness: 0.23228844573559745
;;; Empirical kurtosis: 3.1650242061959126
;;; 
;;; Method :rmh with options []
;;; Chain no. 3
;;; Elapsed time: 19050.368456 ms
;;; Empirical mean: 17.149199999999997
;;; Empirical variance: 1.4451393600000007
;;; Empirical std: 1.202139492737844
;;; Empirical skewness: 0.022814291224994163
;;; Empirical kurtosis: 3.0883674425994316
;;; 
;;; Method :rmh with options []
;;; Chain no. 4
;;; Elapsed time: 19940.93135 ms
;;; Empirical mean: 17.120299999999997
;;; Empirical variance: 1.4894279099999994
;;; Empirical std: 1.2204212018807274
;;; Empirical skewness: 0.033901896978046145
;;; Empirical kurtosis: 2.8980883054374837
;;; 
;;; Method :rmh with options []
;;; Chain no. 5
;;; Elapsed time: 17940.537623 ms
;;; Empirical mean: 17.2766
;;; Empirical variance: 1.6188924400000024
;;; Empirical std: 1.2723570410855605
;;; Empirical skewness: 0.290940084987769
;;; Empirical kurtosis: 3.42495167721972
;;; 
;;; Method :rmh with options []
;;; Chain no. 6
;;; Elapsed time: 17570.502692 ms
;;; Empirical mean: 17.145899999999997
;;; Empirical variance: 1.58601319
;;; Empirical std: 1.259370156070089
;;; Empirical skewness: 0.09432181163064841
;;; Empirical kurtosis: 2.8077544689618827
;;; 
;;; Method :rmh with options []
;;; Chain no. 7
;;; Elapsed time: 17671.817149 ms
;;; Empirical mean: 17.279799999999998
;;; Empirical variance: 1.384311960000002
;;; Empirical std: 1.1765678730953018
;;; Empirical skewness: 0.2078319284556389
;;; Empirical kurtosis: 3.4636014698178856
;;; 
;;; Method :rmh with options []
;;; Chain no. 8
;;; Elapsed time: 31992.564265 ms
;;; Empirical mean: 17.099800000000002
;;; Empirical variance: 1.3936399600000025
;;; Empirical std: 1.1805252898604088
;;; Empirical skewness: 0.02051399564757542
;;; Empirical kurtosis: 3.4957421583685124
;;; 
;;; Method :rmh with options []
;;; Chain no. 9
;;; Elapsed time: 23732.167605 ms
;;; Empirical mean: 17.0657
;;; Empirical variance: 1.4199835100000016
;;; Empirical std: 1.1916306097109126
;;; Empirical skewness: 0.1870683999275819
;;; Empirical kurtosis: 3.1017974931959307
;;; 
;;; Method :rmh with options []
;;; Chain no. 10
;;; Elapsed time: 18221.74452 ms
;;; Empirical mean: 17.217499999999998
;;; Empirical variance: 1.6015937500000008
;;; Empirical std: 1.2655408922670184
;;; Empirical skewness: 0.09550182602257608
;;; Empirical kurtosis: 2.761083777093354
;;; 
;;; Method :rmh with options []
;;; Chain no. 11
;;; Elapsed time: 17201.238387 ms
;;; Empirical mean: 17.2764
;;; Empirical variance: 1.6126030400000004
;;; Empirical std: 1.2698830812322843
;;; Empirical skewness: 0.02984320865612197
;;; Empirical kurtosis: 3.0951140395931542
;;; 
;;; Method :rmh with options []
;;; Chain no. 12
;;; Elapsed time: 16351.447891 ms
;;; Empirical mean: 17.1604
;;; Empirical variance: 1.5876718400000014
;;; Empirical std: 1.2600285076140147
;;; Empirical skewness: 0.0201376038612486
;;; Empirical kurtosis: 2.8135828776574496
;;; 
;;; Method :rmh with options []
;;; Chain no. 13
;;; Elapsed time: 16380.038919 ms
;;; Empirical mean: 17.269299999999994
;;; Empirical variance: 1.837777510000001
;;; Empirical std: 1.3556465284136572
;;; Empirical skewness: 0.202229250528516
;;; Empirical kurtosis: 2.802203194768213
;;; 
;;; Method :rmh with options []
;;; Chain no. 14
;;; Elapsed time: 17851.176366 ms
;;; Empirical mean: 17.042800000000003
;;; Empirical variance: 1.509568160000001
;;; Empirical std: 1.2286448469757243
;;; Empirical skewness: 0.16471086102302251
;;; Empirical kurtosis: 3.235965138994656
;;; 
;;; Method :rmh with options []
;;; Chain no. 15
;;; Elapsed time: 18069.768284 ms
;;; Empirical mean: 17.1435
;;; Empirical variance: 1.5301077500000009
;;; Empirical std: 1.2369752422744769
;;; Empirical skewness: 0.20934775414759318
;;; Empirical kurtosis: 2.8696832326223096
;;; 
;;; Method :rmh with options []
;;; Chain no. 16
;;; Elapsed time: 20556.426281 ms
;;; Empirical mean: 17.1137
;;; Empirical variance: 1.6937723100000013
;;; Empirical std: 1.3014500797187734
;;; Empirical skewness: 0.1581794124564022
;;; Empirical kurtosis: 2.9545387826935685
;;; 
;;; Method :rmh with options []
;;; Chain no. 17
;;; Elapsed time: 19817.840165 ms
;;; Empirical mean: 17.1959
;;; Empirical variance: 1.6155231900000018
;;; Empirical std: 1.271032332397568
;;; Empirical skewness: 0.2286661500603793
;;; Empirical kurtosis: 2.929483978575865
;;; 
;;; Method :rmh with options []
;;; Chain no. 18
;;; Elapsed time: 17474.966908 ms
;;; Empirical mean: 17.183000000000003
;;; Empirical variance: 1.446711000000001
;;; Empirical std: 1.2027929996470719
;;; Empirical skewness: 0.1962119292519359
;;; Empirical kurtosis: 2.8816717566106917
;;; 
;;; Method :rmh with options []
;;; Chain no. 19
;;; Elapsed time: 16588.692434 ms
;;; Empirical mean: 17.141599999999997
;;; Empirical variance: 1.3383494400000002
;;; Empirical std: 1.15687053726854
;;; Empirical skewness: 0.17001259611144837
;;; Empirical kurtosis: 2.809934652649963
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 0
;;; Elapsed time: 16221.219894 ms
;;; Empirical mean: 17.191
;;; Empirical variance: 1.6739190000000004
;;; Empirical std: 1.2938002164167388
;;; Empirical skewness: -0.06340785658274939
;;; Empirical kurtosis: 2.984715361271935
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 1
;;; Elapsed time: 16423.550836 ms
;;; Empirical mean: 17.404000000000003
;;; Empirical variance: 1.5191840000000005
;;; Empirical std: 1.2325518244682454
;;; Empirical skewness: 0.22996845641707167
;;; Empirical kurtosis: 3.1234030302927684
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 2
;;; Elapsed time: 16849.481329 ms
;;; Empirical mean: 17.1564
;;; Empirical variance: 1.7497390400000006
;;; Empirical std: 1.3227770182460838
;;; Empirical skewness: -0.03525717749335929
;;; Empirical kurtosis: 2.811303972847716
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 3
;;; Elapsed time: 15996.479398 ms
;;; Empirical mean: 17.0368
;;; Empirical variance: 1.786445759999999
;;; Empirical std: 1.3365798741564228
;;; Empirical skewness: 0.07904024060035912
;;; Empirical kurtosis: 2.5992141291479647
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 4
;;; Elapsed time: 16217.485947 ms
;;; Empirical mean: 17.080500000000004
;;; Empirical variance: 1.739619750000002
;;; Empirical std: 1.3189464545613676
;;; Empirical skewness: 0.3682165817370113
;;; Empirical kurtosis: 3.1432596554861916
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 5
;;; Elapsed time: 16367.345135 ms
;;; Empirical mean: 17.124699999999997
;;; Empirical variance: 1.7265499100000017
;;; Empirical std: 1.313982461831208
;;; Empirical skewness: 0.2790456104376071
;;; Empirical kurtosis: 3.1943246216464765
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 6
;;; Elapsed time: 16252.890152 ms
;;; Empirical mean: 17.3722
;;; Empirical variance: 1.5076671600000018
;;; Empirical std: 1.2278709867082949
;;; Empirical skewness: 0.15285963278621184
;;; Empirical kurtosis: 3.1429629178337
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 7
;;; Elapsed time: 16176.703448 ms
;;; Empirical mean: 17.147
;;; Empirical variance: 1.4049910000000008
;;; Empirical std: 1.185323162686025
;;; Empirical skewness: 0.08179718552994393
;;; Empirical kurtosis: 3.0041906893257115
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 8
;;; Elapsed time: 16462.041857 ms
;;; Empirical mean: 16.946199999999997
;;; Empirical variance: 1.401905560000001
;;; Empirical std: 1.1840209288690808
;;; Empirical skewness: 0.2626827405417653
;;; Empirical kurtosis: 2.7119376872980667
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 9
;;; Elapsed time: 15950.047604 ms
;;; Empirical mean: 17.1355
;;; Empirical variance: 1.3301397500000005
;;; Empirical std: 1.1533168471846758
;;; Empirical skewness: 0.21844492986220151
;;; Empirical kurtosis: 2.8472748030433843
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 10
;;; Elapsed time: 15550.666427 ms
;;; Empirical mean: 17.057600000000004
;;; Empirical variance: 1.7896822400000023
;;; Empirical std: 1.3377900582677398
;;; Empirical skewness: 0.10030485728581191
;;; Empirical kurtosis: 3.324057503774951
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 11
;;; Elapsed time: 22916.631162 ms
;;; Empirical mean: 17.293899999999997
;;; Empirical variance: 1.8007227899999994
;;; Empirical std: 1.341910127393038
;;; Empirical skewness: 0.2076704030085358
;;; Empirical kurtosis: 3.1782048153020233
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 12
;;; Elapsed time: 25663.820325 ms
;;; Empirical mean: 17.059799999999996
;;; Empirical variance: 1.5050239600000008
;;; Empirical std: 1.226794179966632
;;; Empirical skewness: 0.08329107067640133
;;; Empirical kurtosis: 3.427980924226774
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 13
;;; Elapsed time: 36838.494488 ms
;;; Empirical mean: 17.176899999999996
;;; Empirical variance: 1.6524063899999994
;;; Empirical std: 1.285459602632459
;;; Empirical skewness: 0.11046179826199887
;;; Empirical kurtosis: 2.933306098073759
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 14
;;; Elapsed time: 22336.191166 ms
;;; Empirical mean: 17.2924
;;; Empirical variance: 1.7897022400000007
;;; Empirical std: 1.337797533261293
;;; Empirical skewness: -0.01851359667970164
;;; Empirical kurtosis: 2.814219497161868
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 15
;;; Elapsed time: 18212.875034 ms
;;; Empirical mean: 17.087099999999996
;;; Empirical variance: 1.4085135900000023
;;; Empirical std: 1.1868081521459155
;;; Empirical skewness: 0.33152840563706065
;;; Empirical kurtosis: 3.0199182725482268
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 16
;;; Elapsed time: 16526.588002 ms
;;; Empirical mean: 17.219300000000004
;;; Empirical variance: 1.4742075100000016
;;; Empirical std: 1.2141694733438169
;;; Empirical skewness: 0.12451393651253104
;;; Empirical kurtosis: 3.382107892672586
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 17
;;; Elapsed time: 16881.27539 ms
;;; Empirical mean: 17.105100000000004
;;; Empirical variance: 1.3310539900000031
;;; Empirical std: 1.1537131315886124
;;; Empirical skewness: 0.3492046894565705
;;; Empirical kurtosis: 3.9642713149677604
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 18
;;; Elapsed time: 17331.5102 ms
;;; Empirical mean: 17.092099999999995
;;; Empirical variance: 1.6502175900000007
;;; Empirical std: 1.2846079518670281
;;; Empirical skewness: 0.09545983643339206
;;; Empirical kurtosis: 2.843580675026749
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 19
;;; Elapsed time: 17583.337709 ms
;;; Empirical mean: 17.1663
;;; Empirical variance: 1.6322443100000013
;;; Empirical std: 1.2775931707707275
;;; Empirical skewness: 0.1796222299706656
;;; Empirical kurtosis: 2.9318596085144746
;;; 
;;; Method :smc with options []
;;; Chain no. 0
;;; Elapsed time: 30456.711937 ms
;;; Empirical mean: 17.22039127814564
;;; Empirical variance: 1.6538740520516928
;;; Empirical std: 1.2860303464738663
;;; Empirical skewness: 0.03563373821739495
;;; Empirical kurtosis: 2.9154622815233537
;;; 
;;; Method :smc with options []
;;; Chain no. 1
;;; Elapsed time: 33609.597615 ms
;;; Empirical mean: 17.184593006802835
;;; Empirical variance: 1.5244618295979033
;;; Empirical std: 1.2346909854687946
;;; Empirical skewness: 0.11098186026781551
;;; Empirical kurtosis: 3.0622676053887545
;;; 
;;; Method :smc with options []
;;; Chain no. 2
;;; Elapsed time: 31442.525775 ms
;;; Empirical mean: 17.130771444437453
;;; Empirical variance: 1.54639090649082
;;; Empirical std: 1.2435396682417574
;;; Empirical skewness: 0.23442320252059867
;;; Empirical kurtosis: 3.100389620100416
;;; 
;;; Method :smc with options []
;;; Chain no. 3
;;; Elapsed time: 31066.98388 ms
;;; Empirical mean: 17.08106059059962
;;; Empirical variance: 1.5856028729632599
;;; Empirical std: 1.2592072398788294
;;; Empirical skewness: 0.23386251981386139
;;; Empirical kurtosis: 3.0626330324263034
;;; 
;;; Method :smc with options []
;;; Chain no. 4
;;; Elapsed time: 31073.702251 ms
;;; Empirical mean: 17.18140522105954
;;; Empirical variance: 1.5849059900298097
;;; Empirical std: 1.2589304945189825
;;; Empirical skewness: 0.09538311951237934
;;; Empirical kurtosis: 2.958235381150921
;;; 
;;; Method :smc with options []
;;; Chain no. 5
;;; Elapsed time: 31547.023246 ms
;;; Empirical mean: 17.163923790660586
;;; Empirical variance: 1.5346445005193687
;;; Empirical std: 1.2388076931143788
;;; Empirical skewness: 0.1155104674690543
;;; Empirical kurtosis: 3.108920692585612
;;; 
;;; Method :smc with options []
;;; Chain no. 6
;;; Elapsed time: 30942.650646 ms
;;; Empirical mean: 17.172143156552643
;;; Empirical variance: 1.610090105999243
;;; Empirical std: 1.2688932602860032
;;; Empirical skewness: 0.13787038091745005
;;; Empirical kurtosis: 3.04435642891192
;;; 
;;; Method :smc with options []
;;; Chain no. 7
;;; Elapsed time: 30566.975241 ms
;;; Empirical mean: 17.209481858889323
;;; Empirical variance: 1.468949721980773
;;; Empirical std: 1.2120023605508254
;;; Empirical skewness: 0.0828118579201194
;;; Empirical kurtosis: 3.197065290078068
;;; 
;;; Method :smc with options []
;;; Chain no. 8
;;; Elapsed time: 28966.943428 ms
;;; Empirical mean: 17.16124561699951
;;; Empirical variance: 1.4794949721327804
;;; Empirical std: 1.2163449231746644
;;; Empirical skewness: 0.13058535970204577
;;; Empirical kurtosis: 3.142521532149427
;;; 
;;; Method :smc with options []
;;; Chain no. 9
;;; Elapsed time: 31393.80019 ms
;;; Empirical mean: 17.21674953354612
;;; Empirical variance: 1.4816106347100433
;;; Empirical std: 1.2172142928465979
;;; Empirical skewness: 0.1262577103898835
;;; Empirical kurtosis: 3.1294242061332347
;;; 
;;; Method :smc with options []
;;; Chain no. 10
;;; Elapsed time: 28787.951628 ms
;;; Empirical mean: 17.175348693731262
;;; Empirical variance: 1.6111304777385835
;;; Empirical std: 1.2693031465093685
;;; Empirical skewness: 0.10375955975480869
;;; Empirical kurtosis: 2.8778740629177886
;;; 
;;; Method :smc with options []
;;; Chain no. 11
;;; Elapsed time: 28838.208742 ms
;;; Empirical mean: 17.21440633656138
;;; Empirical variance: 1.5559536442473767
;;; Empirical std: 1.2473787092328363
;;; Empirical skewness: 0.11657823614403803
;;; Empirical kurtosis: 2.993535154105599
;;; 
;;; Method :smc with options []
;;; Chain no. 12
;;; Elapsed time: 29732.760763 ms
;;; Empirical mean: 17.155625119312383
;;; Empirical variance: 1.5525678981480904
;;; Empirical std: 1.2460208257280816
;;; Empirical skewness: 0.18698768386009065
;;; Empirical kurtosis: 3.0698491576246036
;;; 
;;; Method :smc with options []
;;; Chain no. 13
;;; Elapsed time: 32546.188153 ms
;;; Empirical mean: 17.163502525738913
;;; Empirical variance: 1.6525420642173672
;;; Empirical std: 1.2855123741984622
;;; Empirical skewness: 0.16217090193469452
;;; Empirical kurtosis: 2.863358402581668
;;; 
;;; Method :smc with options []
;;; Chain no. 14
;;; Elapsed time: 37637.195468 ms
;;; Empirical mean: 17.167161105459392
;;; Empirical variance: 1.5482935213573936
;;; Empirical std: 1.2443044327484305
;;; Empirical skewness: 0.15123676909312442
;;; Empirical kurtosis: 3.02694225503495
;;; 
;;; Method :smc with options []
;;; Chain no. 15
;;; Elapsed time: 40005.821125 ms
;;; Empirical mean: 17.212049620069056
;;; Empirical variance: 1.513042324618957
;;; Empirical std: 1.2300578541755494
;;; Empirical skewness: 0.07000946822904744
;;; Empirical kurtosis: 2.998541087868097
;;; 
;;; Method :smc with options []
;;; Chain no. 16
;;; Elapsed time: 28487.825436 ms
;;; Empirical mean: 17.17930039181176
;;; Empirical variance: 1.576040113273151
;;; Empirical std: 1.2554043624558389
;;; Empirical skewness: 0.09355365264898863
;;; Empirical kurtosis: 3.0533192661563047
;;; 
;;; Method :smc with options []
;;; Chain no. 17
;;; Elapsed time: 27247.703939 ms
;;; Empirical mean: 17.178019213216295
;;; Empirical variance: 1.6979625595376355
;;; Empirical std: 1.3030589240466586
;;; Empirical skewness: 0.1398777651485555
;;; Empirical kurtosis: 2.833748245709557
;;; 
;;; Method :smc with options []
;;; Chain no. 18
;;; Elapsed time: 27340.145042 ms
;;; Empirical mean: 17.203612753905443
;;; Empirical variance: 1.5915901283078944
;;; Empirical std: 1.2615823906142216
;;; Empirical skewness: 0.1330763677692063
;;; Empirical kurtosis: 3.066416116931395
;;; 
;;; Method :smc with options []
;;; Chain no. 19
;;; Elapsed time: 27222.297265 ms
;;; Empirical mean: 17.188849889994316
;;; Empirical variance: 1.570459406799411
;;; Empirical std: 1.25317971847593
;;; Empirical skewness: 0.1641629617531048
;;; Empirical kurtosis: 3.0472786347777325
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 0
;;; Elapsed time: 20643.456895 ms
;;; Empirical mean: 22.039330430097888
;;; Empirical variance: 0.04377557010583901
;;; Empirical std: 0.20922612194905063
;;; Empirical skewness: 5.723889866346376
;;; Empirical kurtosis: 38.97531444368566
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 1
;;; Elapsed time: 20638.513941 ms
;;; Empirical mean: 21.062171175368725
;;; Empirical variance: 0.12300637357528449
;;; Empirical std: 0.35072264479968285
;;; Empirical skewness: 5.522743283653443
;;; Empirical kurtosis: 32.00881539361319
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 2
;;; Elapsed time: 20580.876946 ms
;;; Empirical mean: 20.001824846201455
;;; Empirical variance: 0.0037298660362290238
;;; Empirical std: 0.061072629190407575
;;; Empirical skewness: 33.826422077385686
;;; Empirical kurtosis: 1165.2318257439524
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 3
;;; Elapsed time: 20857.134593 ms
;;; Empirical mean: 22.103727380240297
;;; Empirical variance: 0.09801785773654334
;;; Empirical std: 0.3130780377741999
;;; Empirical skewness: 2.8670804501637583
;;; Empirical kurtosis: 10.469740969926876
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 4
;;; Elapsed time: 20572.134106 ms
;;; Empirical mean: 22.14088098631485
;;; Empirical variance: 0.13390360835921192
;;; Empirical std: 0.365928419720595
;;; Empirical skewness: 2.4644924061131865
;;; Empirical kurtosis: 8.397946252977
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 5
;;; Elapsed time: 20776.631429 ms
;;; Empirical mean: 22.125865333156288
;;; Empirical variance: 0.12107605804962376
;;; Empirical std: 0.34795985120358897
;;; Empirical skewness: 2.6672528751612847
;;; Empirical kurtosis: 9.638441502895533
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 6
;;; Elapsed time: 20801.319337 ms
;;; Empirical mean: 22.102502254410314
;;; Empirical variance: 0.10039047157766191
;;; Empirical std: 0.31684455428121516
;;; Empirical skewness: 3.0319472817170294
;;; Empirical kurtosis: 11.859016553365642
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 7
;;; Elapsed time: 20782.448918 ms
;;; Empirical mean: 23.027065942525294
;;; Empirical variance: 0.028208289681029747
;;; Empirical std: 0.1679532365899203
;;; Empirical skewness: 6.427704126240246
;;; Empirical kurtosis: 46.75282008114617
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 8
;;; Elapsed time: 20651.959211 ms
;;; Empirical mean: 22.006497913519635
;;; Empirical variance: 0.008726295614040118
;;; Empirical std: 0.09341464346685759
;;; Empirical skewness: 16.384284244718767
;;; Empirical kurtosis: 306.64696596488153
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 9
;;; Elapsed time: 20701.651906 ms
;;; Empirical mean: 23.027753920864143
;;; Empirical variance: 0.029987087781205967
;;; Empirical std: 0.17316780238025187
;;; Empirical skewness: 6.617411580507232
;;; Empirical kurtosis: 50.62657637856332
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 10
;;; Elapsed time: 20647.003463 ms
;;; Empirical mean: 21.056165372596528
;;; Empirical variance: 0.05892989097912694
;;; Empirical std: 0.24275479599613875
;;; Empirical skewness: 4.502027308721437
;;; Empirical kurtosis: 24.64692179963663
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 11
;;; Elapsed time: 20702.000006 ms
;;; Empirical mean: 21.00283093026778
;;; Empirical variance: 0.006085382335999429
;;; Empirical std: 0.07800886062492791
;;; Empirical skewness: 28.375130168798393
;;; Empirical kurtosis: 838.7119581675306
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 12
;;; Elapsed time: 20685.207574 ms
;;; Empirical mean: 22.035349757188936
;;; Empirical variance: 0.03810145426573624
;;; Empirical std: 0.19519593813841576
;;; Empirical skewness: 5.857640808638706
;;; Empirical kurtosis: 40.44253722369576
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 13
;;; Elapsed time: 20683.414792 ms
;;; Empirical mean: 22.08795056182941
;;; Empirical variance: 0.08532271103393144
;;; Empirical std: 0.2921005152921361
;;; Empirical skewness: 3.2411061312419442
;;; Empirical kurtosis: 13.11217121356021
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 14
;;; Elapsed time: 20492.243637 ms
;;; Empirical mean: 22.102142730857313
;;; Empirical variance: 0.10146348956696104
;;; Empirical std: 0.318533341374119
;;; Empirical skewness: 3.11563568232621
;;; Empirical kurtosis: 12.782109730304773
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 15
;;; Elapsed time: 22145.248657 ms
;;; Empirical mean: 23.036567379026714
;;; Empirical variance: 0.03802031573751459
;;; Empirical std: 0.19498798870062378
;;; Empirical skewness: 5.539627990186472
;;; Empirical kurtosis: 35.893231616885245
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 16
;;; Elapsed time: 22514.671634 ms
;;; Empirical mean: 24.037595792131814
;;; Empirical variance: 0.03804481325101551
;;; Empirical std: 0.19505079659159433
;;; Empirical skewness: 5.24094449900185
;;; Empirical kurtosis: 30.93345345456955
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 17
;;; Elapsed time: 22395.718261 ms
;;; Empirical mean: 21.035129276156766
;;; Empirical variance: 0.03579105237978071
;;; Empirical std: 0.18918523298550738
;;; Empirical skewness: 5.504036217950083
;;; Empirical kurtosis: 34.760991629459085
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 18
;;; Elapsed time: 22877.383708 ms
;;; Empirical mean: 22.08918308627303
;;; Empirical variance: 0.08444287087298485
;;; Empirical std: 0.29059055537471423
;;; Empirical skewness: 3.0914933479186173
;;; Empirical kurtosis: 11.581053149365351
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 19
;;; Elapsed time: 22666.853298 ms
;;; Empirical mean: 21.002127915118884
;;; Empirical variance: 0.004639095619688865
;;; Empirical std: 0.0681109067601428
;;; Empirical skewness: 33.124051185410494
;;; Empirical kurtosis: 1148.218764848666
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 0
;;; Elapsed time: 26034.051905 ms
;;; Empirical mean: 30.9917
;;; Empirical variance: 74.88303110999995
;;; Empirical std: 8.653498200727839
;;; Empirical skewness: 0.5112459854798442
;;; Empirical kurtosis: 3.4486019165102584
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 1
;;; Elapsed time: 26034.486965 ms
;;; Empirical mean: 29.62899999999999
;;; Empirical variance: 72.74015899999998
;;; Empirical std: 8.5287841454688
;;; Empirical skewness: 1.0774221460390891
;;; Empirical kurtosis: 4.853158221844831
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 2
;;; Elapsed time: 26021.328623 ms
;;; Empirical mean: 29.146599999999992
;;; Empirical variance: 73.49150843999998
;;; Empirical std: 8.572718847600216
;;; Empirical skewness: 1.3091654452672556
;;; Empirical kurtosis: 6.518273266151067
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 3
;;; Elapsed time: 26186.693992 ms
;;; Empirical mean: 29.93229999999999
;;; Empirical variance: 74.64091670999996
;;; Empirical std: 8.639497480177882
;;; Empirical skewness: 0.8245869967344829
;;; Empirical kurtosis: 3.9387044102083837
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 4
;;; Elapsed time: 26005.697242 ms
;;; Empirical mean: 30.21489999999999
;;; Empirical variance: 67.65871798999996
;;; Empirical std: 8.22549196036322
;;; Empirical skewness: 0.7753783529043801
;;; Empirical kurtosis: 4.604063109464008
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 5
;;; Elapsed time: 25934.049593 ms
;;; Empirical mean: 29.266399999999994
;;; Empirical variance: 54.81003104000001
;;; Empirical std: 7.403379703892001
;;; Empirical skewness: 0.5134377950610701
;;; Empirical kurtosis: 3.547517318213328
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 6
;;; Elapsed time: 26010.608171 ms
;;; Empirical mean: 28.986199999999997
;;; Empirical variance: 50.01040955999996
;;; Empirical std: 7.071803840605306
;;; Empirical skewness: 0.9182631293054221
;;; Empirical kurtosis: 6.107782486617698
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 7
;;; Elapsed time: 26077.281743 ms
;;; Empirical mean: 28.557700000000004
;;; Empirical variance: 55.06987070999999
;;; Empirical std: 7.420907674267346
;;; Empirical skewness: 1.1740310911057577
;;; Empirical kurtosis: 7.107911832005685
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 8
;;; Elapsed time: 26303.927047 ms
;;; Empirical mean: 29.2942
;;; Empirical variance: 56.45864635999998
;;; Empirical std: 7.5138968824438885
;;; Empirical skewness: 0.5255471161221374
;;; Empirical kurtosis: 4.198529564890827
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 9
;;; Elapsed time: 25711.975945 ms
;;; Empirical mean: 29.078599999999998
;;; Empirical variance: 51.694822039999984
;;; Empirical std: 7.189911128797072
;;; Empirical skewness: 0.6909633625817415
;;; Empirical kurtosis: 4.248774858801991
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 10
;;; Elapsed time: 26627.020562 ms
;;; Empirical mean: 30.091499999999993
;;; Empirical variance: 68.03532774999994
;;; Empirical std: 8.24835303257565
;;; Empirical skewness: 0.9252720870543346
;;; Empirical kurtosis: 4.328955842318371
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 11
;;; Elapsed time: 26035.968259 ms
;;; Empirical mean: 29.560499999999998
;;; Empirical variance: 63.34013974999997
;;; Empirical std: 7.958651880186743
;;; Empirical skewness: 0.706613903064155
;;; Empirical kurtosis: 3.7020029473977427
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 12
;;; Elapsed time: 25850.577821 ms
;;; Empirical mean: 29.969500000000007
;;; Empirical variance: 78.44996974999998
;;; Empirical std: 8.857198752991827
;;; Empirical skewness: 0.5509755597948996
;;; Empirical kurtosis: 2.9888280702716514
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 13
;;; Elapsed time: 26131.608068 ms
;;; Empirical mean: 30.110799999999998
;;; Empirical variance: 74.16292335999998
;;; Empirical std: 8.611789788423774
;;; Empirical skewness: 0.8875250086076434
;;; Empirical kurtosis: 4.345778832659621
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 14
;;; Elapsed time: 25795.145887 ms
;;; Empirical mean: 30.105500000000003
;;; Empirical variance: 71.48036975
;;; Empirical std: 8.454606421945376
;;; Empirical skewness: 0.7162470457815989
;;; Empirical kurtosis: 3.625920831308368
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 15
;;; Elapsed time: 26546.543813 ms
;;; Empirical mean: 29.666599999999992
;;; Empirical variance: 67.98464443999997
;;; Empirical std: 8.245280131081051
;;; Empirical skewness: 0.8099097779976199
;;; Empirical kurtosis: 3.8935157035508907
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 16
;;; Elapsed time: 26358.915356 ms
;;; Empirical mean: 29.410600000000006
;;; Empirical variance: 60.66760764
;;; Empirical std: 7.788941368376064
;;; Empirical skewness: 0.6064476584552304
;;; Empirical kurtosis: 3.4499118833573563
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 17
;;; Elapsed time: 26176.097478 ms
;;; Empirical mean: 29.747799999999998
;;; Empirical variance: 58.40459515999997
;;; Empirical std: 7.642289915987221
;;; Empirical skewness: 0.7571427313869328
;;; Empirical kurtosis: 4.019515427468484
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 18
;;; Elapsed time: 26081.17003 ms
;;; Empirical mean: 29.937700000000007
;;; Empirical variance: 53.00321870999996
;;; Empirical std: 7.280330947834718
;;; Empirical skewness: 0.5603210788418775
;;; Empirical kurtosis: 4.211554268907976
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 19
;;; Elapsed time: 26073.373161 ms
;;; Empirical mean: 30.50530000000001
;;; Empirical variance: 72.30757191
;;; Empirical std: 8.503385908566068
;;; Empirical skewness: 0.2913526185114976
;;; Empirical kurtosis: 2.932264371888868
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 0
;;; Elapsed time: 23677.815299 ms
;;; Empirical mean: 23.033499999999997
;;; Empirical variance: 0.6469777500000012
;;; Empirical std: 0.8043492711502891
;;; Empirical skewness: 0.09615971855533076
;;; Empirical kurtosis: 1.9519241126482625
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 1
;;; Elapsed time: 23730.732758 ms
;;; Empirical mean: 23.778100000000002
;;; Empirical variance: 0.5008603900000017
;;; Empirical std: 0.7077149072896527
;;; Empirical skewness: 0.6213538241036966
;;; Empirical kurtosis: 3.238427086371179
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 2
;;; Elapsed time: 23898.279303 ms
;;; Empirical mean: 23.439599999999995
;;; Empirical variance: 0.6949518400000021
;;; Empirical std: 0.8336377150777201
;;; Empirical skewness: -0.6756924877924262
;;; Empirical kurtosis: 2.2639637496149105
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 3
;;; Elapsed time: 23437.577887 ms
;;; Empirical mean: 23.848100000000002
;;; Empirical variance: 0.4094263900000021
;;; Empirical std: 0.639864352812377
;;; Empirical skewness: 0.2762852846339265
;;; Empirical kurtosis: 2.909171474068706
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 4
;;; Elapsed time: 23602.756911 ms
;;; Empirical mean: 23.127199999999995
;;; Empirical variance: 0.33282016000000036
;;; Empirical std: 0.5769056768658117
;;; Empirical skewness: 0.36216477275961717
;;; Empirical kurtosis: 3.848067121837447
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 5
;;; Elapsed time: 23453.055633 ms
;;; Empirical mean: 23.4452
;;; Empirical variance: 0.4755969600000005
;;; Empirical std: 0.6896353819229408
;;; Empirical skewness: 1.2864863194902616
;;; Empirical kurtosis: 3.4391666555135045
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 6
;;; Elapsed time: 23695.681022 ms
;;; Empirical mean: 23.339800000000004
;;; Empirical variance: 0.8631359600000017
;;; Empirical std: 0.9290511073132639
;;; Empirical skewness: 0.184418253515093
;;; Empirical kurtosis: 2.233406969886991
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 7
;;; Elapsed time: 23617.891819 ms
;;; Empirical mean: 23.623299999999997
;;; Empirical variance: 0.26959711000000025
;;; Empirical std: 0.5192274164564119
;;; Empirical skewness: -0.10696707266387508
;;; Empirical kurtosis: 2.0150883618951765
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 8
;;; Elapsed time: 23820.912805 ms
;;; Empirical mean: 23.4589
;;; Empirical variance: 0.557710790000002
;;; Empirical std: 0.7468003682377252
;;; Empirical skewness: -0.16840351715732021
;;; Empirical kurtosis: 2.725083599046196
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 9
;;; Elapsed time: 24316.269771 ms
;;; Empirical mean: 23.7246
;;; Empirical variance: 0.24255483999999994
;;; Empirical std: 0.4924985685258384
;;; Empirical skewness: -0.4128101145693841
;;; Empirical kurtosis: 2.677826549057336
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 10
;;; Elapsed time: 23513.188735 ms
;;; Empirical mean: 23.2282
;;; Empirical variance: 0.40412476000000186
;;; Empirical std: 0.6357080776582926
;;; Empirical skewness: 0.10841586950215407
;;; Empirical kurtosis: 3.0161922703553463
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 11
;;; Elapsed time: 23864.350588 ms
;;; Empirical mean: 23.629899999999996
;;; Empirical variance: 0.657125990000002
;;; Empirical std: 0.8106330797592719
;;; Empirical skewness: -0.3450404547360391
;;; Empirical kurtosis: 2.715950734759274
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 12
;;; Elapsed time: 23928.975119 ms
;;; Empirical mean: 23.3304
;;; Empirical variance: 0.24603584
;;; Empirical std: 0.49601999959679044
;;; Empirical skewness: 1.0231298738337495
;;; Empirical kurtosis: 2.74089830389549
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 13
;;; Elapsed time: 23497.67288 ms
;;; Empirical mean: 23.342399999999994
;;; Empirical variance: 1.0905622400000041
;;; Empirical std: 1.0442998803025902
;;; Empirical skewness: -0.5311981247125511
;;; Empirical kurtosis: 3.4781789723298604
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 14
;;; Elapsed time: 24463.035666 ms
;;; Empirical mean: 23.331900000000008
;;; Empirical variance: 1.275342390000003
;;; Empirical std: 1.1293105817267466
;;; Empirical skewness: -0.5499847370980516
;;; Empirical kurtosis: 2.725641751606624
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 15
;;; Elapsed time: 23542.786764 ms
;;; Empirical mean: 23.433199999999996
;;; Empirical variance: 0.47453776000000114
;;; Empirical std: 0.6888670118390059
;;; Empirical skewness: -0.5292384325917303
;;; Empirical kurtosis: 2.579414537470754
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 16
;;; Elapsed time: 23698.503803 ms
;;; Empirical mean: 23.038499999999996
;;; Empirical variance: 1.275817750000004
;;; Empirical std: 1.1295210268073828
;;; Empirical skewness: 0.09970833040805047
;;; Empirical kurtosis: 2.672847858253997
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 17
;;; Elapsed time: 24041.840631 ms
;;; Empirical mean: 23.322899999999997
;;; Empirical variance: 0.6576355900000009
;;; Empirical std: 0.8109473410770892
;;; Empirical skewness: -0.42605519723523894
;;; Empirical kurtosis: 2.0326656367226184
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 18
;;; Elapsed time: 23808.616712 ms
;;; Empirical mean: 23.632199999999994
;;; Empirical variance: 0.6963231600000019
;;; Empirical std: 0.8344598013086082
;;; Empirical skewness: -0.2204997690373096
;;; Empirical kurtosis: 2.6820370226211883
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 19
;;; Elapsed time: 24225.308279 ms
;;; Empirical mean: 23.223100000000002
;;; Empirical variance: 1.0015263900000007
;;; Empirical std: 1.0007629039887522
;;; Empirical skewness: 0.26107591124773516
;;; Empirical kurtosis: 2.0363859176916983
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 0
;;; Elapsed time: 10140.119295 ms
;;; Empirical mean: 14.937500000000009
;;; Empirical variance: 40.55859374999993
;;; Empirical std: 6.368562926594973
;;; Empirical skewness: 0.7855885196792445
;;; Empirical kurtosis: 2.3404816838406033
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 1
;;; Elapsed time: 10176.966837 ms
;;; Empirical mean: 13.62644364794903
;;; Empirical variance: 21.737795357139042
;;; Empirical std: 4.662380867876308
;;; Empirical skewness: 0.9999430117397888
;;; Empirical kurtosis: 2.9606225104993307
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 2
;;; Elapsed time: 9823.176659 ms
;;; Empirical mean: 14.93001988071571
;;; Empirical variance: 26.325122663620604
;;; Empirical std: 5.130801366611322
;;; Empirical skewness: 0.39579373289015446
;;; Empirical kurtosis: 2.2093441396381204
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 3
;;; Elapsed time: 9869.070374 ms
;;; Empirical mean: 16.67965023847377
;;; Empirical variance: 42.71454614157783
;;; Empirical std: 6.535636628636711
;;; Empirical skewness: 0.29559417759699974
;;; Empirical kurtosis: 1.560462570315322
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 4
;;; Elapsed time: 9954.586338 ms
;;; Empirical mean: 16.74920508744038
;;; Empirical variance: 22.094892054918475
;;; Empirical std: 4.700520402563792
;;; Empirical skewness: 0.4168084537237208
;;; Empirical kurtosis: 2.3055783411613753
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 5
;;; Elapsed time: 10562.000167 ms
;;; Empirical mean: 16.12749003984064
;;; Empirical variance: 9.48414469611593
;;; Empirical std: 3.0796338574765554
;;; Empirical skewness: 0.49752241562683086
;;; Empirical kurtosis: 2.315448869723718
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 6
;;; Elapsed time: 10291.213426 ms
;;; Empirical mean: 15.564541832669313
;;; Empirical variance: 27.50559530801095
;;; Empirical std: 5.2445777054030716
;;; Empirical skewness: 1.1395834773389268
;;; Empirical kurtosis: 3.2630623880164906
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 7
;;; Elapsed time: 10221.985835 ms
;;; Empirical mean: 15.749401436552276
;;; Empirical variance: 19.897296130149567
;;; Empirical std: 4.460638533904038
;;; Empirical skewness: 0.8564278324533492
;;; Empirical kurtosis: 3.2108529286707075
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 8
;;; Elapsed time: 10239.610602 ms
;;; Empirical mean: 17.249503377036156
;;; Empirical variance: 7.83047750465685
;;; Empirical std: 2.79829903774719
;;; Empirical skewness: 0.17258107613593476
;;; Empirical kurtosis: 1.8973945781576662
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 9
;;; Elapsed time: 10025.16805 ms
;;; Empirical mean: 16.502188619180256
;;; Empirical variance: 25.00208434643634
;;; Empirical std: 5.000208430299315
;;; Empirical skewness: 0.13114743616176971
;;; Empirical kurtosis: 1.6441231646334376
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 10
;;; Elapsed time: 10008.318145 ms
;;; Empirical mean: 15.996427153632395
;;; Empirical variance: 47.9035203828434
;;; Empirical std: 6.921236911336254
;;; Empirical skewness: 0.5715022441241406
;;; Empirical kurtosis: 1.7439329304171596
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 11
;;; Elapsed time: 10493.111247 ms
;;; Empirical mean: 16.750994431185365
;;; Empirical variance: 46.78922932136917
;;; Empirical std: 6.840265296124791
;;; Empirical skewness: 0.6366097085922726
;;; Empirical kurtosis: 1.8441927566554417
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 12
;;; Elapsed time: 10240.877293 ms
;;; Empirical mean: 17.254675686430552
;;; Empirical variance: 29.355753108111582
;;; Empirical std: 5.418094970385032
;;; Empirical skewness: 0.5741666388189981
;;; Empirical kurtosis: 2.388785051938214
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 13
;;; Elapsed time: 10721.890365 ms
;;; Empirical mean: 16.62763839107925
;;; Empirical variance: 30.35397924956556
;;; Empirical std: 5.509444550003708
;;; Empirical skewness: 0.2779114700808771
;;; Empirical kurtosis: 1.6042166034427954
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 14
;;; Elapsed time: 9964.977491 ms
;;; Empirical mean: 17.117249602543716
;;; Empirical variance: 30.109066521164394
;;; Empirical std: 5.487172907897508
;;; Empirical skewness: 0.172783976388977
;;; Empirical kurtosis: 1.7681638226786407
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 15
;;; Elapsed time: 9774.850971 ms
;;; Empirical mean: 14.060207336523126
;;; Empirical variance: 17.769501073438985
;;; Empirical std: 4.2153886028976
;;; Empirical skewness: 1.4986848543195659
;;; Empirical kurtosis: 4.5227096642488895
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 16
;;; Elapsed time: 10076.686352 ms
;;; Empirical mean: 19.875000011971142
;;; Empirical variance: 77.60937481638446
;;; Empirical std: 8.809618312752514
;;; Empirical skewness: 0.22283672616004158
;;; Empirical kurtosis: 1.585411856921191
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 17
;;; Elapsed time: 10624.851856 ms
;;; Empirical mean: 17.433571996817818
;;; Empirical variance: 17.127051123098077
;;; Empirical std: 4.138484157647348
;;; Empirical skewness: 0.38218039338495885
;;; Empirical kurtosis: 1.5380595854580656
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 18
;;; Elapsed time: 10295.341552 ms
;;; Empirical mean: 17.191818903891974
;;; Empirical variance: 25.31467492828441
;;; Empirical std: 5.031369090842413
;;; Empirical skewness: 0.24480043276715668
;;; Empirical kurtosis: 1.8791352288462864
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 19
;;; Elapsed time: 10421.496329 ms
;;; Empirical mean: 15.434142473715108
;;; Empirical variance: 42.38255087237582
;;; Empirical std: 6.51018823632434
;;; Empirical skewness: 0.9172237699566892
;;; Empirical kurtosis: 2.6948004372130443
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 0
;;; Elapsed time: 22108.121656 ms
;;; Empirical mean: 22.60545783414614
;;; Empirical variance: 0.3231688913261105
;;; Empirical std: 0.5684794555004696
;;; Empirical skewness: 0.32851719458548573
;;; Empirical kurtosis: 2.5942206769031433
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 1
;;; Elapsed time: 22461.917686 ms
;;; Empirical mean: 23.08403042673824
;;; Empirical variance: 0.08150997668946394
;;; Empirical std: 0.2854995213471713
;;; Empirical skewness: 3.2964983917741977
;;; Empirical kurtosis: 13.205901840181696
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 2
;;; Elapsed time: 22419.026239 ms
;;; Empirical mean: 23.0782834584921
;;; Empirical variance: 0.5344704435112773
;;; Empirical std: 0.7310748549302439
;;; Empirical skewness: 0.01739510840036591
;;; Empirical kurtosis: 2.209050315531488
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 3
;;; Elapsed time: 21659.279039 ms
;;; Empirical mean: 22.665387074418387
;;; Empirical variance: 0.34836199053440875
;;; Empirical std: 0.5902219841164922
;;; Empirical skewness: 0.30640254752024443
;;; Empirical kurtosis: 2.6440641244992427
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 4
;;; Elapsed time: 22508.73359 ms
;;; Empirical mean: 22.900974182060487
;;; Empirical variance: 0.14885910055285578
;;; Empirical std: 0.3858226283577154
;;; Empirical skewness: -0.8618629260301808
;;; Empirical kurtosis: 6.287544059909041
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 5
;;; Elapsed time: 22692.313506 ms
;;; Empirical mean: 22.873534200435248
;;; Empirical variance: 0.401972780286667
;;; Empirical std: 0.6340132335264517
;;; Empirical skewness: 0.1989266138227777
;;; Empirical kurtosis: 2.8044622669977777
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 6
;;; Elapsed time: 22231.453653 ms
;;; Empirical mean: 22.26980378974369
;;; Empirical variance: 1.2868090307545532
;;; Empirical std: 1.1343760535001404
;;; Empirical skewness: -0.03046460871469294
;;; Empirical kurtosis: 1.460607833923881
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 7
;;; Elapsed time: 22192.56104 ms
;;; Empirical mean: 23.507627357856148
;;; Empirical variance: 0.2920468606225382
;;; Empirical std: 0.5404136014410983
;;; Empirical skewness: 0.38125043634146166
;;; Empirical kurtosis: 2.0310492629297885
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 8
;;; Elapsed time: 22266.881014 ms
;;; Empirical mean: 23.095870890340052
;;; Empirical variance: 0.09659962957443488
;;; Empirical std: 0.31080480944546995
;;; Empirical skewness: 3.2438156316897477
;;; Empirical kurtosis: 13.406192425601805
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 9
;;; Elapsed time: 22244.483559 ms
;;; Empirical mean: 22.986162850737923
;;; Empirical variance: 0.3680911242465774
;;; Empirical std: 0.6067051378112579
;;; Empirical skewness: 0.24051317660697402
;;; Empirical kurtosis: 3.5222830102572034
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 10
;;; Elapsed time: 22174.097566 ms
;;; Empirical mean: 23.691912170001935
;;; Empirical variance: 0.2635574951430033
;;; Empirical std: 0.5133785105972039
;;; Empirical skewness: -0.23022241000884064
;;; Empirical kurtosis: 2.4378851777059722
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 11
;;; Elapsed time: 22058.034153 ms
;;; Empirical mean: 22.54807933944041
;;; Empirical variance: 0.6215371657458729
;;; Empirical std: 0.7883762843629132
;;; Empirical skewness: -0.9911350223140581
;;; Empirical kurtosis: 2.848897636734572
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 12
;;; Elapsed time: 22398.206987 ms
;;; Empirical mean: 22.55052615387822
;;; Empirical variance: 1.0643920741768325
;;; Empirical std: 1.0316937889591236
;;; Empirical skewness: -0.13158592419372087
;;; Empirical kurtosis: 2.101606100039996
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 13
;;; Elapsed time: 22327.968415 ms
;;; Empirical mean: 22.971126212563597
;;; Empirical variance: 0.27674392892127425
;;; Empirical std: 0.5260645672550797
;;; Empirical skewness: 0.19021260581737526
;;; Empirical kurtosis: 4.513544434856769
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 14
;;; Elapsed time: 22658.146808 ms
;;; Empirical mean: 23.240238065645844
;;; Empirical variance: 0.5434439395387692
;;; Empirical std: 0.7371865025478758
;;; Empirical skewness: -0.1717604390196929
;;; Empirical kurtosis: 2.298077216448339
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 15
;;; Elapsed time: 21745.456246 ms
;;; Empirical mean: 22.79174450013095
;;; Empirical variance: 0.866962664150724
;;; Empirical std: 0.931108298830337
;;; Empirical skewness: 0.5125169840933721
;;; Empirical kurtosis: 1.5676772495787905
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 16
;;; Elapsed time: 22507.970858 ms
;;; Empirical mean: 23.155213729248302
;;; Empirical variance: 0.5967864707565304
;;; Empirical std: 0.7725195601125776
;;; Empirical skewness: -0.11094355433651275
;;; Empirical kurtosis: 2.0225468895829555
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 17
;;; Elapsed time: 22660.893626 ms
;;; Empirical mean: 22.852203915228447
;;; Empirical variance: 0.6701483635165054
;;; Empirical std: 0.8186258996125797
;;; Empirical skewness: 0.4087671514308005
;;; Empirical kurtosis: 1.9680211506491463
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 18
;;; Elapsed time: 21973.310927 ms
;;; Empirical mean: 22.560217834824318
;;; Empirical variance: 0.4595779193285836
;;; Empirical std: 0.6779217649025465
;;; Empirical skewness: 0.8327501388052835
;;; Empirical kurtosis: 2.6254430477208985
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 19
;;; Elapsed time: 22306.026499 ms
;;; Empirical mean: 22.8396380761991
;;; Empirical variance: 0.28355494631035794
;;; Empirical std: 0.5324987758768633
;;; Empirical skewness: 0.01988839222671973
;;; Empirical kurtosis: 3.893929343729641
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-unkown'>#multifn[print-method 0x1b74665d]</span>","value":"#multifn[print-method 0x1b74665d]"}],"value":"[nil,#multifn[print-method 0x1b74665d]]"},{"type":"html","content":"<span class='clj-unkown'>#multifn[print-method 0x1b74665d]</span>","value":"#multifn[print-method 0x1b74665d]"}],"value":"[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/data</span>","value":"#'model/data"}],"value":"[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/model</span>","value":"#'model/model"}],"value":"[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/model_name</span>","value":"#'model/model_name"}],"value":"[[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model],#'model/model_name]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/outfile</span>","value":"#'model/outfile"}],"value":"[[[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model],#'model/model_name],#'model/outfile]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/configurations</span>","value":"#'model/configurations"}],"value":"[[[[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/num_samples_options</span>","value":"#'model/num_samples_options"}],"value":"[[[[[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/thinning</span>","value":"#'model/thinning"}],"value":"[[[[[[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning]"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"[[[[[[[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/num-chains</span>","value":"#'model/num-chains"}],"value":"[[[[[[[[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains]"},{"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)"}],"value":"[[[[[[[[[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains],(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)]"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"[[[[[[[[[[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains],(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)],nil]"}
;; <=

;; @@

;; @@
