;; gorilla-repl.fileformat = 1

;; @@

(ns model
  (:require [gorilla-plot.core :as plot])
  (:use [anglican core emit runtime stat
          [state :only [get-predicts get-log-weight get-result]]]))

(defdist geometric
"Geometric distribution on support {0,1,2....}"
[p] []
(sample* [this]
        (loop [value 0]
            (if (sample* (flip p))
            value
            (recur (inc value)))))
(observe* [this value] (+ (log p) (* value (log (- 1 p))))))

(defdist dirac [x]
    (sample* [this] x)
    (observe* [this value]
              (if (= value x)
                0
                (- (/ 1.0 0.0)))))


(with-primitive-procedures [dirac geometric]
  (defquery model [method- options- ]
    (let [[ a] [ 0 ]
          [ a]
          (let [
                a (sample (poisson 5.140000000000001))
                a (sample (binomial a 0.2636))
                a (+ a (sample (poisson 23.26)))
                _unused (observe (binomial a 0.2) 5)
                a (sample (binomial a 0.2636))
                a (+ a (sample (poisson 42.08)))
                _unused (observe (binomial a 0.2) 11)
                a (sample (binomial a 0.2636))
                a (+ a (sample (poisson 30.080000000000002)))
                _unused (observe (binomial a 0.2) 7)
                a (sample (binomial a 0.2636))
                a (+ a (sample (poisson 8.559999999999999)))
                _unused (observe (binomial a 0.2) 5)
               ]
            [ a ]
          )
         ]
    a
    )
  )

)


(def model_name "population200")
(def outfile "population200_anglican.json")

; (def configurations [:rmh []])
(def configurations
  [
    [:importance []]
    [:lmh []]
    [:rmh []]
    [:smc []]
    [:smc [:number-of-particles 100]]
    [:pgibbs []]
    [:ipmcmc []]
  ])

; (def num_samples_options [1000])
(def num_samples_options [1000 10000])
(def thinning 1)

(spit outfile "[\n" :append false)

(def num-chains 20)

(doall
  (for [ num_samples num_samples_options
         [method options] configurations
         chain (range 0 num-chains)]
    (do
      (println (format "\nMethod %s with %s samples and options %s" method num_samples options))
      (println (format "Chain no. %s" chain))
      (let [start (. System (nanoTime))
            warmup (/ num_samples 5)
            samples (take-nth thinning (take (* num_samples thinning) (drop warmup (apply doquery method model [method options] options))))
            results (collect-results samples)
            values (map (fn [s] (get-result s)) samples)
            max-value (apply max values)
            mean (empirical-mean results)
            variance (empirical-variance results)
            std (empirical-std results)
            skewness (if (zero? std) (/ 0.0 0.0) (empirical-skew results))
            kurtosis (if (zero? std) (/ 0.0 0.0) (empirical-kurtosis results))
            distribution (empirical-distribution (collect-results samples))
            masses (for [n (range 0 (inc max-value))] (get distribution n 0.0))
            end (. System (nanoTime))
            elapsed_ms (/ (- end start) 1e6)]
        (println (format "Elapsed time: %s ms" elapsed_ms))
        (println (format "Empirical mean: %s" mean))
        (println (format "Empirical variance: %s" variance))
        (println (format "Empirical std: %s" std))
        (println (format "Empirical skewness: %s" skewness))
        (println (format "Empirical kurtosis: %s" kurtosis))
        (spit outfile (format
                   "{\"model\": \"%s\", \"system\": \"anglican\", \"method\": \"%s\", \"options\": \"%s\", \"num_samples\": %s, \"time_ms\": %s, \"total\": 1.0, \"mean\": %s, \"variance\": %s, \"stddev\": %s, \"skewness\": %s, \"kurtosis\": %s, \"masses\": [%s] },\n"
                   model_name method options num_samples elapsed_ms mean variance std skewness kurtosis
                   (clojure.string/join ", " masses)) :append true)
        (if false (do
          (println "Empirical distribution:")
          (doall (for [n (range 0 (inc max-value))]
            (println (format "p(%s) = %s" n (get distribution n 0.0)))))))
        ; (println "List of samples (format: sample log-weight):")
        ; (doall (map (fn [s] (println (format "%s %s" (get-result s) (get-log-weight s)))) samples))
        ; values need to be adjusted if they are weighted!
        ; (plot/histogram values :normalize :probability)
      )
    )
  )
)

(spit outfile "]\n" :append true)




;; @@
;; ->
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 101.400549 ms
;;; Empirical mean: 20.70971952967483
;;; Empirical variance: 15.17123185355235
;;; Empirical std: 3.8950265536389286
;;; Empirical skewness: 0.17897534657707675
;;; Empirical kurtosis: 3.1225162910658497
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 80.270306 ms
;;; Empirical mean: 20.96576018733313
;;; Empirical variance: 15.409123812874514
;;; Empirical std: 3.925445683342786
;;; Empirical skewness: 0.14809980479783966
;;; Empirical kurtosis: 2.8154962904833356
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 82.495506 ms
;;; Empirical mean: 20.64031956074375
;;; Empirical variance: 13.91181048661962
;;; Empirical std: 3.7298539497706367
;;; Empirical skewness: 0.1466348722469556
;;; Empirical kurtosis: 3.1465017195784633
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 83.656961 ms
;;; Empirical mean: 20.641508450155182
;;; Empirical variance: 15.05954689118048
;;; Empirical std: 3.880663202492646
;;; Empirical skewness: 0.2521273597150444
;;; Empirical kurtosis: 3.1665799796499297
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 83.095612 ms
;;; Empirical mean: 20.622109170278513
;;; Empirical variance: 13.73617182871724
;;; Empirical std: 3.706234184278867
;;; Empirical skewness: 0.1932585494972549
;;; Empirical kurtosis: 2.799955752811059
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 76.352065 ms
;;; Empirical mean: 20.790647164430528
;;; Empirical variance: 16.3490864492916
;;; Empirical std: 4.043400357284893
;;; Empirical skewness: 0.20734150876020202
;;; Empirical kurtosis: 2.7111998097719865
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 79.918906 ms
;;; Empirical mean: 20.499246978901585
;;; Empirical variance: 15.46490989496698
;;; Empirical std: 3.9325449641379793
;;; Empirical skewness: 0.21726865587342029
;;; Empirical kurtosis: 2.9433529888230154
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 83.928325 ms
;;; Empirical mean: 20.56100603197042
;;; Empirical variance: 14.98020150587859
;;; Empirical std: 3.8704265276424756
;;; Empirical skewness: 0.10744536796995131
;;; Empirical kurtosis: 3.0688495336141446
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 83.698174 ms
;;; Empirical mean: 20.543193513136572
;;; Empirical variance: 14.960209320608552
;;; Empirical std: 3.8678429803455767
;;; Empirical skewness: 0.09975449882152826
;;; Empirical kurtosis: 2.895788805418954
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 84.75474 ms
;;; Empirical mean: 20.725876865705622
;;; Empirical variance: 15.70500139310919
;;; Empirical std: 3.962953619853403
;;; Empirical skewness: 0.1918295888318269
;;; Empirical kurtosis: 2.9014792169633465
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 86.679698 ms
;;; Empirical mean: 20.68284807100957
;;; Empirical variance: 14.553544626467255
;;; Empirical std: 3.8149108281147615
;;; Empirical skewness: 0.36383962395868685
;;; Empirical kurtosis: 3.4165411125404557
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 75.889089 ms
;;; Empirical mean: 20.82037423995684
;;; Empirical variance: 15.647975488817144
;;; Empirical std: 3.9557522026559178
;;; Empirical skewness: 0.16118985304901343
;;; Empirical kurtosis: 3.04955793044088
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 80.344946 ms
;;; Empirical mean: 20.575172727222423
;;; Empirical variance: 15.926724624387917
;;; Empirical std: 3.9908300670897923
;;; Empirical skewness: 0.26289178841202104
;;; Empirical kurtosis: 3.025711269929849
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 80.906137 ms
;;; Empirical mean: 20.540634125490282
;;; Empirical variance: 15.057979009179249
;;; Empirical std: 3.8804611851143735
;;; Empirical skewness: 0.32366132321224533
;;; Empirical kurtosis: 3.1889431315768624
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 83.007604 ms
;;; Empirical mean: 20.682477421374067
;;; Empirical variance: 15.055924589761462
;;; Empirical std: 3.8801964627788452
;;; Empirical skewness: 0.27129070585170667
;;; Empirical kurtosis: 3.1368327541420418
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 85.1636 ms
;;; Empirical mean: 20.607289791934527
;;; Empirical variance: 16.17575288845124
;;; Empirical std: 4.021909109919223
;;; Empirical skewness: 0.3430590582748957
;;; Empirical kurtosis: 3.1309573957853107
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 80.675745 ms
;;; Empirical mean: 20.62376572032315
;;; Empirical variance: 16.71586474655763
;;; Empirical std: 4.0885039741398845
;;; Empirical skewness: 0.25405901038849654
;;; Empirical kurtosis: 2.9790715344302576
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 84.735293 ms
;;; Empirical mean: 20.561242222368982
;;; Empirical variance: 16.5382718639997
;;; Empirical std: 4.066727414518915
;;; Empirical skewness: 0.2519844072941536
;;; Empirical kurtosis: 3.066816743088654
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 78.809815 ms
;;; Empirical mean: 20.78112257418607
;;; Empirical variance: 14.933005080668146
;;; Empirical std: 3.864324660360222
;;; Empirical skewness: 0.18477985383799692
;;; Empirical kurtosis: 3.1204826168350146
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 82.370758 ms
;;; Empirical mean: 20.526337278593832
;;; Empirical variance: 15.51343086521025
;;; Empirical std: 3.938709289248224
;;; Empirical skewness: 0.3147662135454356
;;; Empirical kurtosis: 3.0464950907850046
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 160.052643 ms
;;; Empirical mean: 20.769999999999996
;;; Empirical variance: 14.715100000000001
;;; Empirical std: 3.8360265901059654
;;; Empirical skewness: 1.0231732278525238
;;; Empirical kurtosis: 4.717730127112426
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 172.523145 ms
;;; Empirical mean: 20.849000000000004
;;; Empirical variance: 16.600199000000007
;;; Empirical std: 4.074334178733993
;;; Empirical skewness: -0.2572888168087518
;;; Empirical kurtosis: 3.049443748350558
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 135.242427 ms
;;; Empirical mean: 19.666999999999998
;;; Empirical variance: 16.574111
;;; Empirical std: 4.071131415221081
;;; Empirical skewness: 0.02465288566129037
;;; Empirical kurtosis: 2.6911904156138298
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 130.167165 ms
;;; Empirical mean: 20.257000000000005
;;; Empirical variance: 13.988951000000007
;;; Empirical std: 3.7401806106122746
;;; Empirical skewness: 0.4815742490648782
;;; Empirical kurtosis: 3.1259977924733953
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 131.775035 ms
;;; Empirical mean: 20.447999999999997
;;; Empirical variance: 12.867296000000005
;;; Empirical std: 3.587101336734161
;;; Empirical skewness: 0.12731608601855787
;;; Empirical kurtosis: 3.185617043409753
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 127.217634 ms
;;; Empirical mean: 21.270999999999997
;;; Empirical variance: 13.657559000000003
;;; Empirical std: 3.6956134808716135
;;; Empirical skewness: 0.2136486712372298
;;; Empirical kurtosis: 2.6057629407652345
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 135.425635 ms
;;; Empirical mean: 20.314
;;; Empirical variance: 15.865404000000005
;;; Empirical std: 3.9831399674126446
;;; Empirical skewness: 0.26692086248489855
;;; Empirical kurtosis: 2.740327482501374
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 132.145882 ms
;;; Empirical mean: 20.888999999999996
;;; Empirical variance: 19.644678999999996
;;; Empirical std: 4.432231830579262
;;; Empirical skewness: 0.19260468284817853
;;; Empirical kurtosis: 2.7210209869678534
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 127.834892 ms
;;; Empirical mean: 20.017999999999997
;;; Empirical variance: 18.251675999999996
;;; Empirical std: 4.272198029118032
;;; Empirical skewness: 0.034836954523305055
;;; Empirical kurtosis: 2.4774130301351045
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 125.637516 ms
;;; Empirical mean: 20.940999999999992
;;; Empirical variance: 15.063519
;;; Empirical std: 3.8811749509652356
;;; Empirical skewness: 0.9226756196422365
;;; Empirical kurtosis: 4.513281958819088
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 135.21451 ms
;;; Empirical mean: 20.367
;;; Empirical variance: 12.018311000000004
;;; Empirical std: 3.466743572864887
;;; Empirical skewness: 0.16684287600247125
;;; Empirical kurtosis: 3.381878038558406
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 131.499469 ms
;;; Empirical mean: 20.320000000000007
;;; Empirical variance: 20.17960000000002
;;; Empirical std: 4.492170967360884
;;; Empirical skewness: 0.7332450042287388
;;; Empirical kurtosis: 3.583150381272334
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 127.88946 ms
;;; Empirical mean: 20.064999999999998
;;; Empirical variance: 16.888775000000006
;;; Empirical std: 4.109595478876237
;;; Empirical skewness: 0.39765151391663844
;;; Empirical kurtosis: 2.9698939897721206
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 137.365775 ms
;;; Empirical mean: 21.218999999999998
;;; Empirical variance: 12.957039000000004
;;; Empirical std: 3.599588726507516
;;; Empirical skewness: -0.015314706382436078
;;; Empirical kurtosis: 3.094262419780435
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 151.608111 ms
;;; Empirical mean: 21.382000000000005
;;; Empirical variance: 15.324076000000005
;;; Empirical std: 3.914597808204568
;;; Empirical skewness: 0.14247722437527335
;;; Empirical kurtosis: 2.7751125088899453
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 129.593415 ms
;;; Empirical mean: 20.399
;;; Empirical variance: 17.861799000000012
;;; Empirical std: 4.226322159987335
;;; Empirical skewness: 0.23426906494004573
;;; Empirical kurtosis: 2.895955553783278
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 134.299607 ms
;;; Empirical mean: 19.926000000000002
;;; Empirical variance: 16.558524000000006
;;; Empirical std: 4.06921663222788
;;; Empirical skewness: 0.23841414068076405
;;; Empirical kurtosis: 2.460366836128774
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 130.26982 ms
;;; Empirical mean: 19.439999999999998
;;; Empirical variance: 19.090400000000002
;;; Empirical std: 4.369256229611626
;;; Empirical skewness: -0.024942942336704896
;;; Empirical kurtosis: 2.730533431358755
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 131.405941 ms
;;; Empirical mean: 19.534999999999997
;;; Empirical variance: 14.538775
;;; Empirical std: 3.8129745606284864
;;; Empirical skewness: 0.21868714003213813
;;; Empirical kurtosis: 2.973315151306407
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 134.052302 ms
;;; Empirical mean: 20.068
;;; Empirical variance: 14.489376000000009
;;; Empirical std: 3.806491297770167
;;; Empirical skewness: 0.5135441730714605
;;; Empirical kurtosis: 3.4785790299721517
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 198.667364 ms
;;; Empirical mean: 20.597
;;; Empirical variance: 15.624591000000004
;;; Empirical std: 3.952795340009397
;;; Empirical skewness: 0.0684994466901754
;;; Empirical kurtosis: 2.273591026496044
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 204.719266 ms
;;; Empirical mean: 19.514000000000003
;;; Empirical variance: 11.947804000000005
;;; Empirical std: 3.4565595611821887
;;; Empirical skewness: 0.2060784652425823
;;; Empirical kurtosis: 2.5067096289995443
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 203.352488 ms
;;; Empirical mean: 19.194999999999993
;;; Empirical variance: 11.350975000000002
;;; Empirical std: 3.369120805195326
;;; Empirical skewness: 0.23212901120879254
;;; Empirical kurtosis: 2.7267380350983568
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 210.835272 ms
;;; Empirical mean: 20.665999999999993
;;; Empirical variance: 21.412444000000008
;;; Empirical std: 4.627358209605132
;;; Empirical skewness: 0.7225103552352147
;;; Empirical kurtosis: 3.5062924803149937
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 202.368081 ms
;;; Empirical mean: 20.265000000000004
;;; Empirical variance: 14.92477500000001
;;; Empirical std: 3.863259634039629
;;; Empirical skewness: -0.20171960710052025
;;; Empirical kurtosis: 3.2081242168406976
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 200.767011 ms
;;; Empirical mean: 21.119999999999997
;;; Empirical variance: 9.605600000000004
;;; Empirical std: 3.099290241329457
;;; Empirical skewness: -0.47475601343029417
;;; Empirical kurtosis: 3.3573890425610164
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 199.019797 ms
;;; Empirical mean: 20.961000000000002
;;; Empirical variance: 14.311479000000007
;;; Empirical std: 3.783051546040578
;;; Empirical skewness: 0.8027456453340244
;;; Empirical kurtosis: 3.7449675848293893
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 203.098931 ms
;;; Empirical mean: 19.869000000000003
;;; Empirical variance: 20.053839000000004
;;; Empirical std: 4.478151292665312
;;; Empirical skewness: 0.25235373996145377
;;; Empirical kurtosis: 2.462536462650397
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 218.916582 ms
;;; Empirical mean: 21.140000000000004
;;; Empirical variance: 19.892400000000006
;;; Empirical std: 4.460089685196925
;;; Empirical skewness: -0.042847961009289545
;;; Empirical kurtosis: 2.582748464169822
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 207.019586 ms
;;; Empirical mean: 20.823999999999995
;;; Empirical variance: 12.201024000000002
;;; Empirical std: 3.4929964214124243
;;; Empirical skewness: -0.27885800404993477
;;; Empirical kurtosis: 2.6147504044934187
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 203.575295 ms
;;; Empirical mean: 20.77
;;; Empirical variance: 14.817100000000009
;;; Empirical std: 3.8492986374143547
;;; Empirical skewness: 0.09422151792587509
;;; Empirical kurtosis: 2.666607892514203
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 211.013515 ms
;;; Empirical mean: 19.957000000000004
;;; Empirical variance: 16.99115100000001
;;; Empirical std: 4.1220323870634505
;;; Empirical skewness: 0.4555941623984228
;;; Empirical kurtosis: 2.8284553586640198
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 207.249149 ms
;;; Empirical mean: 20.334000000000003
;;; Empirical variance: 9.590444000000007
;;; Empirical std: 3.096844200149566
;;; Empirical skewness: 0.3300699936166197
;;; Empirical kurtosis: 3.1290275267582137
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 207.647745 ms
;;; Empirical mean: 22.289000000000005
;;; Empirical variance: 11.525479000000008
;;; Empirical std: 3.3949195866765396
;;; Empirical skewness: -0.054327433548353565
;;; Empirical kurtosis: 2.4359512909226972
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 209.663695 ms
;;; Empirical mean: 20.566
;;; Empirical variance: 10.669644000000007
;;; Empirical std: 3.2664421011247096
;;; Empirical skewness: -0.3127065180002172
;;; Empirical kurtosis: 3.157384718847893
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 201.526112 ms
;;; Empirical mean: 20.294999999999998
;;; Empirical variance: 11.211975000000004
;;; Empirical std: 3.3484287359894647
;;; Empirical skewness: 0.12212105848757332
;;; Empirical kurtosis: 2.413133512594488
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 319.277476 ms
;;; Empirical mean: 19.623000000000005
;;; Empirical variance: 9.708871000000007
;;; Empirical std: 3.1159061282394256
;;; Empirical skewness: 0.46434443797412683
;;; Empirical kurtosis: 3.975917958225275
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 240.885991 ms
;;; Empirical mean: 19.378
;;; Empirical variance: 10.379116000000003
;;; Empirical std: 3.2216635454373574
;;; Empirical skewness: 0.09946977538467945
;;; Empirical kurtosis: 2.4930507320618895
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 212.158997 ms
;;; Empirical mean: 20.40800000000001
;;; Empirical variance: 18.519536000000013
;;; Empirical std: 4.303433048160505
;;; Empirical skewness: 0.010667458023622565
;;; Empirical kurtosis: 3.012928572528931
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 224.25051 ms
;;; Empirical mean: 20.256
;;; Empirical variance: 13.76446400000001
;;; Empirical std: 3.710049056279446
;;; Empirical skewness: 0.3675205995631767
;;; Empirical kurtosis: 3.6886362168430633
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 135.647202 ms
;;; Empirical mean: 20.698291473932866
;;; Empirical variance: 15.636171554020082
;;; Empirical std: 3.954259924944247
;;; Empirical skewness: 0.29376922483088463
;;; Empirical kurtosis: 2.8585232178103768
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 132.394663 ms
;;; Empirical mean: 20.608790151697963
;;; Empirical variance: 15.729204629123178
;;; Empirical std: 3.966006130747049
;;; Empirical skewness: 0.2875102925927687
;;; Empirical kurtosis: 3.1114938864347423
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 121.493433 ms
;;; Empirical mean: 20.760452351211317
;;; Empirical variance: 16.468221747276544
;;; Empirical std: 4.058105684586904
;;; Empirical skewness: 0.27220940382812336
;;; Empirical kurtosis: 3.0720258136803027
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 126.475562 ms
;;; Empirical mean: 20.53603518076923
;;; Empirical variance: 14.68522742004355
;;; Empirical std: 3.8321309241782893
;;; Empirical skewness: 0.22591754992258975
;;; Empirical kurtosis: 3.0728419561773466
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 125.306598 ms
;;; Empirical mean: 20.645382587146592
;;; Empirical variance: 14.636827774777954
;;; Empirical std: 3.8258107343121344
;;; Empirical skewness: 0.13094035095035644
;;; Empirical kurtosis: 3.0640624247026214
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 114.22406 ms
;;; Empirical mean: 20.60275366677647
;;; Empirical variance: 15.751341027519363
;;; Empirical std: 3.9687959165872164
;;; Empirical skewness: 0.32089880502435536
;;; Empirical kurtosis: 3.078442995102682
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 122.695073 ms
;;; Empirical mean: 20.520901133553394
;;; Empirical variance: 15.482786198245876
;;; Empirical std: 3.934817174691332
;;; Empirical skewness: 0.21180264476225658
;;; Empirical kurtosis: 2.8707682332864737
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 135.231932 ms
;;; Empirical mean: 20.662413695111276
;;; Empirical variance: 14.002255314531602
;;; Empirical std: 3.741958753718646
;;; Empirical skewness: 0.14108959413351913
;;; Empirical kurtosis: 2.99738179723667
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 117.311826 ms
;;; Empirical mean: 20.58990120056714
;;; Empirical variance: 14.92868197892703
;;; Empirical std: 3.8637652592939737
;;; Empirical skewness: 0.21744971290738707
;;; Empirical kurtosis: 2.868256557044399
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 121.799198 ms
;;; Empirical mean: 20.362429162343503
;;; Empirical variance: 15.487145513146977
;;; Empirical std: 3.9353710769312436
;;; Empirical skewness: 0.3186466947990646
;;; Empirical kurtosis: 3.200901369104807
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 126.088582 ms
;;; Empirical mean: 20.493671345238408
;;; Empirical variance: 14.667540513285095
;;; Empirical std: 3.8298225172043017
;;; Empirical skewness: 0.22776234878043305
;;; Empirical kurtosis: 3.176911556473395
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 121.411918 ms
;;; Empirical mean: 20.54664245486877
;;; Empirical variance: 14.420884794571617
;;; Empirical std: 3.797484008468188
;;; Empirical skewness: 0.1660271355689147
;;; Empirical kurtosis: 2.991640942036053
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 118.167744 ms
;;; Empirical mean: 20.608318047033613
;;; Empirical variance: 15.314213888773699
;;; Empirical std: 3.9133379471716596
;;; Empirical skewness: 0.22123346472562055
;;; Empirical kurtosis: 2.979445097911473
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 116.753333 ms
;;; Empirical mean: 20.61649561544083
;;; Empirical variance: 15.640687010493172
;;; Empirical std: 3.954830844738264
;;; Empirical skewness: 0.3395686034975377
;;; Empirical kurtosis: 3.1934370384933106
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 117.996026 ms
;;; Empirical mean: 20.516413818584287
;;; Empirical variance: 15.605145289215196
;;; Empirical std: 3.950334832544603
;;; Empirical skewness: 0.2476277767109123
;;; Empirical kurtosis: 2.802047306695653
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 118.082938 ms
;;; Empirical mean: 20.675930941562694
;;; Empirical variance: 15.628800908003873
;;; Empirical std: 3.953327827034317
;;; Empirical skewness: 0.352126913918369
;;; Empirical kurtosis: 3.2077730342000352
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 116.772742 ms
;;; Empirical mean: 20.705683195851307
;;; Empirical variance: 14.448079603722062
;;; Empirical std: 3.8010629570847763
;;; Empirical skewness: 0.3186856031936704
;;; Empirical kurtosis: 3.4739057709207586
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 123.921166 ms
;;; Empirical mean: 20.56193893034279
;;; Empirical variance: 13.798295065053821
;;; Empirical std: 3.7146056405833745
;;; Empirical skewness: 0.19731490103675642
;;; Empirical kurtosis: 3.0232172546390355
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 115.957003 ms
;;; Empirical mean: 20.39046089749747
;;; Empirical variance: 15.256260632090502
;;; Empirical std: 3.905926347499464
;;; Empirical skewness: 0.3476017933492414
;;; Empirical kurtosis: 2.964089205080832
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 123.345384 ms
;;; Empirical mean: 20.40280772194034
;;; Empirical variance: 14.872964708365668
;;; Empirical std: 3.8565482893859464
;;; Empirical skewness: 0.2327498170383059
;;; Empirical kurtosis: 3.0477313013725538
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 0
;;; Elapsed time: 115.733892 ms
;;; Empirical mean: 20.73395407846164
;;; Empirical variance: 16.683031264255767
;;; Empirical std: 4.084486658596863
;;; Empirical skewness: 0.3882998487107272
;;; Empirical kurtosis: 3.099762050967748
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 1
;;; Elapsed time: 102.909811 ms
;;; Empirical mean: 20.662783505255042
;;; Empirical variance: 15.870757826863635
;;; Empirical std: 3.9838119718259337
;;; Empirical skewness: 0.20319574449695796
;;; Empirical kurtosis: 2.951084112984016
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 2
;;; Elapsed time: 107.889059 ms
;;; Empirical mean: 20.806161992403094
;;; Empirical variance: 16.014338570496665
;;; Empirical std: 4.001791919939949
;;; Empirical skewness: 0.2657969870609819
;;; Empirical kurtosis: 3.053581521354221
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 3
;;; Elapsed time: 106.444048 ms
;;; Empirical mean: 20.627283863688145
;;; Empirical variance: 14.992769290939641
;;; Empirical std: 3.872049753159125
;;; Empirical skewness: 0.12078839865771611
;;; Empirical kurtosis: 2.749503129692819
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 4
;;; Elapsed time: 99.662333 ms
;;; Empirical mean: 20.69752186521217
;;; Empirical variance: 15.07663921861087
;;; Empirical std: 3.882864821058141
;;; Empirical skewness: 0.37342690128531836
;;; Empirical kurtosis: 3.4917228747396845
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 5
;;; Elapsed time: 107.251856 ms
;;; Empirical mean: 20.693229531520505
;;; Empirical variance: 14.520611918067855
;;; Empirical std: 3.8105920692285937
;;; Empirical skewness: 0.2689174833739087
;;; Empirical kurtosis: 2.8673983988291716
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 6
;;; Elapsed time: 106.86242 ms
;;; Empirical mean: 20.688383044274325
;;; Empirical variance: 15.489069095537758
;;; Empirical std: 3.935615465913528
;;; Empirical skewness: 0.4208298946318166
;;; Empirical kurtosis: 3.2669740389254707
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 7
;;; Elapsed time: 113.135415 ms
;;; Empirical mean: 20.503784926861677
;;; Empirical variance: 15.044613269770005
;;; Empirical std: 3.878738618387427
;;; Empirical skewness: 0.24713728677098679
;;; Empirical kurtosis: 3.2033809696351976
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 8
;;; Elapsed time: 110.342835 ms
;;; Empirical mean: 20.55221662218122
;;; Empirical variance: 14.618169767133292
;;; Empirical std: 3.8233715183242776
;;; Empirical skewness: 0.22547316913719428
;;; Empirical kurtosis: 3.1362379440506882
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 9
;;; Elapsed time: 106.780873 ms
;;; Empirical mean: 20.264430829990403
;;; Empirical variance: 15.083873047654542
;;; Empirical std: 3.8837962160307202
;;; Empirical skewness: 0.1301957606537808
;;; Empirical kurtosis: 2.9339294779693286
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 10
;;; Elapsed time: 123.758743 ms
;;; Empirical mean: 20.52177151332815
;;; Empirical variance: 15.50699263496178
;;; Empirical std: 3.937891902396735
;;; Empirical skewness: 0.28177904798672576
;;; Empirical kurtosis: 2.969703868290134
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 11
;;; Elapsed time: 116.323044 ms
;;; Empirical mean: 20.677819431514592
;;; Empirical variance: 16.70318599966166
;;; Empirical std: 4.0869531438055
;;; Empirical skewness: 0.24420531286952665
;;; Empirical kurtosis: 3.0377762253846967
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 12
;;; Elapsed time: 98.552677 ms
;;; Empirical mean: 20.67520514503846
;;; Empirical variance: 16.508669986651036
;;; Empirical std: 4.0630862637471825
;;; Empirical skewness: 0.19278580911320126
;;; Empirical kurtosis: 2.860310786657924
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 13
;;; Elapsed time: 97.962631 ms
;;; Empirical mean: 20.657335627101844
;;; Empirical variance: 14.659416553507965
;;; Empirical std: 3.82876175199084
;;; Empirical skewness: 0.15199907473503654
;;; Empirical kurtosis: 2.9477718172022565
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 14
;;; Elapsed time: 100.828059 ms
;;; Empirical mean: 20.580102349256958
;;; Empirical variance: 14.871405713827286
;;; Empirical std: 3.856346161047694
;;; Empirical skewness: 0.3539791241529294
;;; Empirical kurtosis: 3.1113010799536145
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 15
;;; Elapsed time: 108.830187 ms
;;; Empirical mean: 20.669585737219712
;;; Empirical variance: 13.719235557231508
;;; Empirical std: 3.7039486439786806
;;; Empirical skewness: 0.21250230127326328
;;; Empirical kurtosis: 3.1653393867734265
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 16
;;; Elapsed time: 98.732905 ms
;;; Empirical mean: 20.73775285977559
;;; Empirical variance: 15.334701172213094
;;; Empirical std: 3.915954694862173
;;; Empirical skewness: 0.34397090735486596
;;; Empirical kurtosis: 3.405760023108743
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 17
;;; Elapsed time: 108.117487 ms
;;; Empirical mean: 20.693761827953917
;;; Empirical variance: 15.645602194458215
;;; Empirical std: 3.9554522111205204
;;; Empirical skewness: 0.15003757839233725
;;; Empirical kurtosis: 2.761226968154764
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 18
;;; Elapsed time: 110.258902 ms
;;; Empirical mean: 20.62546130577144
;;; Empirical variance: 15.91869180670822
;;; Empirical std: 3.989823530772786
;;; Empirical skewness: 0.24022259224105452
;;; Empirical kurtosis: 2.978661281372109
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 19
;;; Elapsed time: 99.253179 ms
;;; Empirical mean: 20.63414446416666
;;; Empirical variance: 15.214687374318428
;;; Empirical std: 3.9006008991331615
;;; Empirical skewness: 0.2119854090205405
;;; Empirical kurtosis: 3.2788707082752153
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 135.205828 ms
;;; Empirical mean: 20.685000000000002
;;; Empirical variance: 14.767775000000007
;;; Empirical std: 3.8428862850727197
;;; Empirical skewness: 0.15167700040430393
;;; Empirical kurtosis: 3.097142144929194
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 106.710009 ms
;;; Empirical mean: 20.193000000000005
;;; Empirical variance: 14.335751000000004
;;; Empirical std: 3.7862581792582506
;;; Empirical skewness: 0.44277668245652285
;;; Empirical kurtosis: 3.285648004926384
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 128.758368 ms
;;; Empirical mean: 20.467000000000002
;;; Empirical variance: 16.09691100000001
;;; Empirical std: 4.012095587096599
;;; Empirical skewness: 0.3512009345456134
;;; Empirical kurtosis: 2.6718297672354847
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 135.299588 ms
;;; Empirical mean: 20.635
;;; Empirical variance: 15.755775000000005
;;; Empirical std: 3.969354481524673
;;; Empirical skewness: 0.1378344331588533
;;; Empirical kurtosis: 2.8459302000788185
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 147.69132 ms
;;; Empirical mean: 20.618000000000006
;;; Empirical variance: 13.396076000000003
;;; Empirical std: 3.6600650267447437
;;; Empirical skewness: 0.14590193272511723
;;; Empirical kurtosis: 2.4603994709491834
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 110.547099 ms
;;; Empirical mean: 20.708000000000006
;;; Empirical variance: 13.442736000000002
;;; Empirical std: 3.666433689568107
;;; Empirical skewness: 0.12491623139649953
;;; Empirical kurtosis: 2.9079685176436283
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 104.345453 ms
;;; Empirical mean: 20.363999999999997
;;; Empirical variance: 15.955504000000003
;;; Empirical std: 3.9944341276330997
;;; Empirical skewness: 0.3408640674124986
;;; Empirical kurtosis: 3.104349933525955
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 120.846187 ms
;;; Empirical mean: 20.766000000000005
;;; Empirical variance: 16.761244000000012
;;; Empirical std: 4.09404982871484
;;; Empirical skewness: 0.23915591658807672
;;; Empirical kurtosis: 3.404519115915205
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 112.660064 ms
;;; Empirical mean: 20.459
;;; Empirical variance: 15.248318999999997
;;; Empirical std: 3.9049096020266587
;;; Empirical skewness: 0.2515037011316118
;;; Empirical kurtosis: 3.0852731493748022
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 112.338588 ms
;;; Empirical mean: 20.681
;;; Empirical variance: 15.755239000000001
;;; Empirical std: 3.9692869636749624
;;; Empirical skewness: 0.24035706723258046
;;; Empirical kurtosis: 2.9143536155701626
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 106.177216 ms
;;; Empirical mean: 20.576999999999998
;;; Empirical variance: 14.20407100000001
;;; Empirical std: 3.76882886318814
;;; Empirical skewness: 0.1404683730445787
;;; Empirical kurtosis: 3.16865064266285
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 110.850509 ms
;;; Empirical mean: 20.559000000000008
;;; Empirical variance: 16.346519000000008
;;; Empirical std: 4.043082858413863
;;; Empirical skewness: 0.06102573116820767
;;; Empirical kurtosis: 3.1765555890343045
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 116.317127 ms
;;; Empirical mean: 20.311000000000007
;;; Empirical variance: 14.150279000000006
;;; Empirical std: 3.761685659381975
;;; Empirical skewness: 0.0017417128520298925
;;; Empirical kurtosis: 3.13987562789432
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 104.398697 ms
;;; Empirical mean: 20.847000000000005
;;; Empirical variance: 16.365591000000006
;;; Empirical std: 4.045440767085832
;;; Empirical skewness: 0.34755728731174695
;;; Empirical kurtosis: 3.3125329033202644
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 109.806535 ms
;;; Empirical mean: 20.433000000000003
;;; Empirical variance: 13.837511000000006
;;; Empirical std: 3.7198805088335845
;;; Empirical skewness: 0.08507013342310131
;;; Empirical kurtosis: 3.21787083558196
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 102.73591 ms
;;; Empirical mean: 20.942000000000007
;;; Empirical variance: 18.628636000000007
;;; Empirical std: 4.316090360499882
;;; Empirical skewness: 0.5949485888544717
;;; Empirical kurtosis: 4.192497133375619
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 111.416636 ms
;;; Empirical mean: 20.833000000000006
;;; Empirical variance: 13.83711100000001
;;; Empirical std: 3.7198267432771663
;;; Empirical skewness: 0.40709881683647736
;;; Empirical kurtosis: 3.298984119526701
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 109.081663 ms
;;; Empirical mean: 20.719
;;; Empirical variance: 15.710039000000005
;;; Empirical std: 3.9635891563077026
;;; Empirical skewness: 0.21932164456056205
;;; Empirical kurtosis: 2.5797582360559077
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 120.150395 ms
;;; Empirical mean: 20.927
;;; Empirical variance: 14.25967100000001
;;; Empirical std: 3.7761979556162055
;;; Empirical skewness: 0.4064129587784837
;;; Empirical kurtosis: 3.3120543976331844
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 117.447882 ms
;;; Empirical mean: 20.307999999999996
;;; Empirical variance: 15.425136000000004
;;; Empirical std: 3.927484691249605
;;; Empirical skewness: 0.2877085668292047
;;; Empirical kurtosis: 2.7978483872770457
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 78.296626 ms
;;; Empirical mean: 20.784420718103203
;;; Empirical variance: 16.5015641471701
;;; Empirical std: 4.062211730962592
;;; Empirical skewness: 0.4144639714905511
;;; Empirical kurtosis: 3.7105748337866737
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 68.245392 ms
;;; Empirical mean: 20.516050087642114
;;; Empirical variance: 14.96245844032291
;;; Empirical std: 3.868133715414051
;;; Empirical skewness: 0.07378694961106065
;;; Empirical kurtosis: 2.318037392948837
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 69.731155 ms
;;; Empirical mean: 20.534256390954617
;;; Empirical variance: 13.436484419430943
;;; Empirical std: 3.665581047996476
;;; Empirical skewness: 0.2574944857173021
;;; Empirical kurtosis: 2.89064143724938
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 68.489232 ms
;;; Empirical mean: 20.398568850169802
;;; Empirical variance: 13.214530056458326
;;; Empirical std: 3.6351795081478886
;;; Empirical skewness: 0.0629799469946507
;;; Empirical kurtosis: 2.544932031233618
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 66.770362 ms
;;; Empirical mean: 21.119578352374045
;;; Empirical variance: 15.720644708344292
;;; Empirical std: 3.9649268225711674
;;; Empirical skewness: 0.17261978274417925
;;; Empirical kurtosis: 2.5981552520391875
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 71.429913 ms
;;; Empirical mean: 20.94065109394082
;;; Empirical variance: 13.641520129012214
;;; Empirical std: 3.693442855793523
;;; Empirical skewness: 0.43563517640125116
;;; Empirical kurtosis: 3.55445219604688
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 69.507458 ms
;;; Empirical mean: 20.653607170308263
;;; Empirical variance: 17.007033310771835
;;; Empirical std: 4.123958451630161
;;; Empirical skewness: 0.5938996108029483
;;; Empirical kurtosis: 3.4527029864236836
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 72.668322 ms
;;; Empirical mean: 21.118028438527254
;;; Empirical variance: 16.186297105227112
;;; Empirical std: 4.02321974359183
;;; Empirical skewness: 0.40385223749060983
;;; Empirical kurtosis: 3.2169351949223866
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 72.760768 ms
;;; Empirical mean: 20.884286280876992
;;; Empirical variance: 15.69738165169585
;;; Empirical std: 3.9619921317054443
;;; Empirical skewness: 0.5897758406104809
;;; Empirical kurtosis: 3.6976909804709863
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 66.081868 ms
;;; Empirical mean: 20.002881047391657
;;; Empirical variance: 15.784169021742889
;;; Empirical std: 3.9729295264002467
;;; Empirical skewness: 0.5589341089127322
;;; Empirical kurtosis: 2.7525783790925193
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 70.623787 ms
;;; Empirical mean: 20.483846577931125
;;; Empirical variance: 17.735446952997634
;;; Empirical std: 4.2113474035037335
;;; Empirical skewness: 0.09927478547929941
;;; Empirical kurtosis: 2.323098590479529
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 67.376125 ms
;;; Empirical mean: 20.466264033134443
;;; Empirical variance: 16.029755202202054
;;; Empirical std: 4.003717672639025
;;; Empirical skewness: 0.1392076667143497
;;; Empirical kurtosis: 2.669641111569848
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 68.610651 ms
;;; Empirical mean: 20.793473662228376
;;; Empirical variance: 16.778029071148204
;;; Empirical std: 4.096099250646669
;;; Empirical skewness: 0.1659334099493405
;;; Empirical kurtosis: 2.563400233464087
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 67.345715 ms
;;; Empirical mean: 20.208666432062444
;;; Empirical variance: 14.71551513873187
;;; Empirical std: 3.8360807002371407
;;; Empirical skewness: 0.4984176623587825
;;; Empirical kurtosis: 3.5824759854493196
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 77.815118 ms
;;; Empirical mean: 20.396489103476068
;;; Empirical variance: 12.839798185194335
;;; Empirical std: 3.5832664128130824
;;; Empirical skewness: 0.35242360565885417
;;; Empirical kurtosis: 3.5792174010281776
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 85.302696 ms
;;; Empirical mean: 20.582385964875943
;;; Empirical variance: 14.402184996555858
;;; Empirical std: 3.7950210798565873
;;; Empirical skewness: 0.22164986535629352
;;; Empirical kurtosis: 2.778736252892497
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 87.057945 ms
;;; Empirical mean: 20.675221252939444
;;; Empirical variance: 13.253501306596132
;;; Empirical std: 3.6405358543209174
;;; Empirical skewness: 0.3282319302638908
;;; Empirical kurtosis: 2.8839373903872
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 83.87518 ms
;;; Empirical mean: 20.448492581719936
;;; Empirical variance: 11.479709458082747
;;; Empirical std: 3.3881719935804244
;;; Empirical skewness: 0.10865859628654986
;;; Empirical kurtosis: 3.186076578785272
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 85.561429 ms
;;; Empirical mean: 20.956951804136363
;;; Empirical variance: 18.495028159930488
;;; Empirical std: 4.300584630016074
;;; Empirical skewness: 0.6224667895113353
;;; Empirical kurtosis: 3.560857421186074
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 85.344151 ms
;;; Empirical mean: 20.685637757122294
;;; Empirical variance: 15.121474853487548
;;; Empirical std: 3.888634060115139
;;; Empirical skewness: 0.1800709690193057
;;; Empirical kurtosis: 2.9241603176102444
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 816.090921 ms
;;; Empirical mean: 20.650128166919544
;;; Empirical variance: 15.220017055897696
;;; Empirical std: 3.9012840265607034
;;; Empirical skewness: 0.23220612480421854
;;; Empirical kurtosis: 3.0835883505034123
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 765.275876 ms
;;; Empirical mean: 20.633020003111586
;;; Empirical variance: 15.184238924823104
;;; Empirical std: 3.89669589842768
;;; Empirical skewness: 0.2554955826815642
;;; Empirical kurtosis: 3.0668447933720917
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 780.642053 ms
;;; Empirical mean: 20.691607903382074
;;; Empirical variance: 15.361995072265245
;;; Empirical std: 3.9194381067016795
;;; Empirical skewness: 0.26063802248194623
;;; Empirical kurtosis: 3.0629232734276086
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 812.437662 ms
;;; Empirical mean: 20.55124312100396
;;; Empirical variance: 15.151821628047445
;;; Empirical std: 3.8925340882319124
;;; Empirical skewness: 0.23744626764592588
;;; Empirical kurtosis: 3.0344290303822916
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 750.032299 ms
;;; Empirical mean: 20.621259461991084
;;; Empirical variance: 15.413691255218817
;;; Empirical std: 3.926027413966797
;;; Empirical skewness: 0.22888252063082945
;;; Empirical kurtosis: 2.982878539116183
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 767.955856 ms
;;; Empirical mean: 20.604460277249547
;;; Empirical variance: 15.09552406457853
;;; Empirical std: 3.8852958786402008
;;; Empirical skewness: 0.26518389129304215
;;; Empirical kurtosis: 3.1097440553780253
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 750.015362 ms
;;; Empirical mean: 20.61518565910053
;;; Empirical variance: 15.385709551600382
;;; Empirical std: 3.922462179753985
;;; Empirical skewness: 0.2298724080465847
;;; Empirical kurtosis: 2.9608378225755274
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 822.427757 ms
;;; Empirical mean: 20.646986134723985
;;; Empirical variance: 15.124725825458006
;;; Empirical std: 3.8890520471521084
;;; Empirical skewness: 0.24085933770107054
;;; Empirical kurtosis: 3.011270747322124
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 763.444006 ms
;;; Empirical mean: 20.589123306214777
;;; Empirical variance: 14.90473407259877
;;; Empirical std: 3.860664978031475
;;; Empirical skewness: 0.2156068390051497
;;; Empirical kurtosis: 3.009194308668594
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 776.834184 ms
;;; Empirical mean: 20.54909678599263
;;; Empirical variance: 15.19917479261262
;;; Empirical std: 3.8986119058727327
;;; Empirical skewness: 0.25219704027417056
;;; Empirical kurtosis: 3.088647299113069
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 777.157407 ms
;;; Empirical mean: 20.611980678213666
;;; Empirical variance: 15.183424352822437
;;; Empirical std: 3.8965913761674367
;;; Empirical skewness: 0.24643549347749302
;;; Empirical kurtosis: 3.1021148868375508
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 841.802156 ms
;;; Empirical mean: 20.575396222345166
;;; Empirical variance: 15.242893419754505
;;; Empirical std: 3.9042148275619395
;;; Empirical skewness: 0.2509131294424303
;;; Empirical kurtosis: 3.0439823633186283
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 765.491084 ms
;;; Empirical mean: 20.640528470003456
;;; Empirical variance: 15.189345634986676
;;; Empirical std: 3.897351104915578
;;; Empirical skewness: 0.24889710904220688
;;; Empirical kurtosis: 3.042323100008577
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 772.207768 ms
;;; Empirical mean: 20.593224800833816
;;; Empirical variance: 14.783361771212753
;;; Empirical std: 3.8449137534166815
;;; Empirical skewness: 0.17415904340989094
;;; Empirical kurtosis: 3.0389108056618785
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 845.872211 ms
;;; Empirical mean: 20.57060246045327
;;; Empirical variance: 15.174061963811779
;;; Empirical std: 3.895389834639375
;;; Empirical skewness: 0.2741237686435965
;;; Empirical kurtosis: 3.190671884292957
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 780.949861 ms
;;; Empirical mean: 20.620763486625357
;;; Empirical variance: 14.902600185351663
;;; Empirical std: 3.8603886054841245
;;; Empirical skewness: 0.211433742220141
;;; Empirical kurtosis: 3.0028091726189423
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 802.61622 ms
;;; Empirical mean: 20.562175119836695
;;; Empirical variance: 15.247731371523543
;;; Empirical std: 3.9048343590379786
;;; Empirical skewness: 0.22403703071641276
;;; Empirical kurtosis: 3.004433211874018
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 805.827556 ms
;;; Empirical mean: 20.63134174928879
;;; Empirical variance: 15.74242915727225
;;; Empirical std: 3.967673015417507
;;; Empirical skewness: 0.2572663642888695
;;; Empirical kurtosis: 3.036852035282726
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 831.001296 ms
;;; Empirical mean: 20.622926449597582
;;; Empirical variance: 15.022704704876478
;;; Empirical std: 3.8759134026544606
;;; Empirical skewness: 0.20761596839125346
;;; Empirical kurtosis: 3.0369267913666276
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 777.123525 ms
;;; Empirical mean: 20.565377104035985
;;; Empirical variance: 15.26436399053604
;;; Empirical std: 3.9069635256214053
;;; Empirical skewness: 0.2560582916341554
;;; Empirical kurtosis: 3.093328268052201
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 1270.322104 ms
;;; Empirical mean: 20.8
;;; Empirical variance: 15.425199999999993
;;; Empirical std: 3.92749283894955
;;; Empirical skewness: 0.13027752086874525
;;; Empirical kurtosis: 2.8196091650608954
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 1299.120783 ms
;;; Empirical mean: 20.65320000000001
;;; Empirical variance: 15.917129759999991
;;; Empirical std: 3.9896277721110764
;;; Empirical skewness: 0.2211737709664647
;;; Empirical kurtosis: 2.926435884179347
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 1271.701152 ms
;;; Empirical mean: 20.494600000000002
;;; Empirical variance: 16.389370839999987
;;; Empirical std: 4.048378791565828
;;; Empirical skewness: 0.28211831784340896
;;; Empirical kurtosis: 3.036756530514828
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 1325.007835 ms
;;; Empirical mean: 20.7028
;;; Empirical variance: 15.068272159999992
;;; Empirical std: 3.8817872378583544
;;; Empirical skewness: 0.2189113119450821
;;; Empirical kurtosis: 2.991444498788376
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 1253.520737 ms
;;; Empirical mean: 20.595899999999993
;;; Empirical variance: 15.100803189999993
;;; Empirical std: 3.885975191634655
;;; Empirical skewness: 0.18795737645493488
;;; Empirical kurtosis: 3.015821384642145
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 1379.082825 ms
;;; Empirical mean: 20.703999999999997
;;; Empirical variance: 14.80878399999999
;;; Empirical std: 3.848218289026753
;;; Empirical skewness: 0.3154262869376541
;;; Empirical kurtosis: 2.9718483934353612
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 1347.628252 ms
;;; Empirical mean: 20.357100000000003
;;; Empirical variance: 15.928179589999989
;;; Empirical std: 3.9910123515218525
;;; Empirical skewness: 0.24277190698329915
;;; Empirical kurtosis: 2.9862493133285066
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 1289.985984 ms
;;; Empirical mean: 20.666199999999993
;;; Empirical variance: 15.440777559999988
;;; Empirical std: 3.929475481536943
;;; Empirical skewness: 0.1290827572599398
;;; Empirical kurtosis: 2.9397675580903297
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 1345.238173 ms
;;; Empirical mean: 20.26110000000001
;;; Empirical variance: 13.92132678999999
;;; Empirical std: 3.7311294255225174
;;; Empirical skewness: 0.21879529439970247
;;; Empirical kurtosis: 2.752990561699693
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 1382.803807 ms
;;; Empirical mean: 21.101699999999997
;;; Empirical variance: 15.444357109999986
;;; Empirical std: 3.9299309294184783
;;; Empirical skewness: 0.2382027124013035
;;; Empirical kurtosis: 3.0823375921910765
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 1268.123569 ms
;;; Empirical mean: 20.553800000000003
;;; Empirical variance: 15.970705559999983
;;; Empirical std: 3.996336517361868
;;; Empirical skewness: 0.3300908562118834
;;; Empirical kurtosis: 3.1113590835706195
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 1280.785886 ms
;;; Empirical mean: 20.48630000000001
;;; Empirical variance: 16.529812309999986
;;; Empirical std: 4.065687187917928
;;; Empirical skewness: 0.24834374528425784
;;; Empirical kurtosis: 3.3884953537307845
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 1270.460057 ms
;;; Empirical mean: 20.5749
;;; Empirical variance: 16.54338998999999
;;; Empirical std: 4.06735663422818
;;; Empirical skewness: 0.24175985653728357
;;; Empirical kurtosis: 2.756896476303875
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 1288.509921 ms
;;; Empirical mean: 20.291500000000003
;;; Empirical variance: 14.190727749999988
;;; Empirical std: 3.7670582355466697
;;; Empirical skewness: 0.07636261720600526
;;; Empirical kurtosis: 2.8966054186671175
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 1335.581673 ms
;;; Empirical mean: 20.593199999999996
;;; Empirical variance: 15.916513759999985
;;; Empirical std: 3.9895505711796644
;;; Empirical skewness: 0.2196140322792186
;;; Empirical kurtosis: 2.9321713782721965
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 1308.950142 ms
;;; Empirical mean: 20.917299999999997
;;; Empirical variance: 14.766660709999982
;;; Empirical std: 3.8427413014669596
;;; Empirical skewness: 0.16779566913931282
;;; Empirical kurtosis: 3.191211072026395
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 1282.188755 ms
;;; Empirical mean: 20.7845
;;; Empirical variance: 13.835259749999988
;;; Empirical std: 3.7195778994396647
;;; Empirical skewness: 0.1348508440771618
;;; Empirical kurtosis: 2.932780227612667
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 1290.032523 ms
;;; Empirical mean: 20.579899999999995
;;; Empirical variance: 15.04161598999998
;;; Empirical std: 3.878352226139341
;;; Empirical skewness: 0.25907522172686753
;;; Empirical kurtosis: 3.124462055033173
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 1323.39464 ms
;;; Empirical mean: 20.9422
;;; Empirical variance: 15.790059159999982
;;; Empirical std: 3.973670741266818
;;; Empirical skewness: 0.12832343649292322
;;; Empirical kurtosis: 2.955988936998176
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 1343.90839 ms
;;; Empirical mean: 20.5527
;;; Empirical variance: 14.113422709999991
;;; Empirical std: 3.7567835591101053
;;; Empirical skewness: 0.21356407116544968
;;; Empirical kurtosis: 2.9250747465465134
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 2007.025474 ms
;;; Empirical mean: 21.130899999999997
;;; Empirical variance: 16.347565189999987
;;; Empirical std: 4.043212236576258
;;; Empirical skewness: 0.19787331331988578
;;; Empirical kurtosis: 2.966132030013289
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 1954.628666 ms
;;; Empirical mean: 20.503100000000003
;;; Empirical variance: 13.920990389999986
;;; Empirical std: 3.731084345066456
;;; Empirical skewness: 0.17618677643366248
;;; Empirical kurtosis: 2.980340988250374
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 2051.795217 ms
;;; Empirical mean: 20.806199999999997
;;; Empirical variance: 15.95944155999998
;;; Empirical std: 3.994926978056042
;;; Empirical skewness: 0.21216399467656902
;;; Empirical kurtosis: 2.8880324632301284
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 2005.166711 ms
;;; Empirical mean: 20.491400000000002
;;; Empirical variance: 13.39492603999999
;;; Empirical std: 3.6599079278036477
;;; Empirical skewness: 0.22463070534616758
;;; Empirical kurtosis: 2.857223981881194
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 1973.659495 ms
;;; Empirical mean: 20.438499999999998
;;; Empirical variance: 14.07241774999999
;;; Empirical std: 3.751322133595033
;;; Empirical skewness: 0.08307831746405601
;;; Empirical kurtosis: 3.2344655175798036
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 2021.459286 ms
;;; Empirical mean: 20.519799999999996
;;; Empirical variance: 14.490207959999987
;;; Empirical std: 3.806600577943526
;;; Empirical skewness: 0.20809372299823722
;;; Empirical kurtosis: 3.1496396981071615
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 1962.467917 ms
;;; Empirical mean: 20.766600000000004
;;; Empirical variance: 15.405324439999987
;;; Empirical std: 3.9249617119151603
;;; Empirical skewness: 0.14304272930650663
;;; Empirical kurtosis: 2.721254967689806
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 1979.123601 ms
;;; Empirical mean: 20.3464
;;; Empirical variance: 15.666007039999986
;;; Empirical std: 3.958030702255856
;;; Empirical skewness: 0.2477693595568091
;;; Empirical kurtosis: 3.069929624486141
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 2023.788896 ms
;;; Empirical mean: 20.6303
;;; Empirical variance: 15.564021909999992
;;; Empirical std: 3.945126349053981
;;; Empirical skewness: 0.22128072868159449
;;; Empirical kurtosis: 2.827964389676758
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 2040.517184 ms
;;; Empirical mean: 20.7129
;;; Empirical variance: 14.973673589999994
;;; Empirical std: 3.8695831287103775
;;; Empirical skewness: 0.2565879989678103
;;; Empirical kurtosis: 2.9473974158787217
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 2212.550372 ms
;;; Empirical mean: 20.12
;;; Empirical variance: 14.492799999999988
;;; Empirical std: 3.8069410292254315
;;; Empirical skewness: 0.09680171925751574
;;; Empirical kurtosis: 2.97294362193038
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 2517.469143 ms
;;; Empirical mean: 20.388299999999997
;;; Empirical variance: 14.765123109999992
;;; Empirical std: 3.842541230748213
;;; Empirical skewness: 0.25655960064180205
;;; Empirical kurtosis: 2.9269253292843227
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 2434.992727 ms
;;; Empirical mean: 20.6073
;;; Empirical variance: 14.951086709999982
;;; Empirical std: 3.866663511349285
;;; Empirical skewness: 0.2600574501181905
;;; Empirical kurtosis: 3.15803081910479
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 2128.636086 ms
;;; Empirical mean: 20.547500000000003
;;; Empirical variance: 16.28454374999999
;;; Empirical std: 4.035411224398326
;;; Empirical skewness: 0.2935870055194895
;;; Empirical kurtosis: 2.961001289043258
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 2116.650038 ms
;;; Empirical mean: 20.4855
;;; Empirical variance: 14.938589749999997
;;; Empirical std: 3.865047185999156
;;; Empirical skewness: 0.24705253347809902
;;; Empirical kurtosis: 2.9079294413092764
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 1992.039725 ms
;;; Empirical mean: 20.9126
;;; Empirical variance: 14.473161239999989
;;; Empirical std: 3.8043608188498617
;;; Empirical skewness: 0.12904440612427417
;;; Empirical kurtosis: 2.8002726544527703
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 2247.772739 ms
;;; Empirical mean: 20.620800000000006
;;; Empirical variance: 12.675007359999992
;;; Empirical std: 3.5601976574342036
;;; Empirical skewness: 0.19506506066685506
;;; Empirical kurtosis: 2.8082382143907663
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 2259.513939 ms
;;; Empirical mean: 20.6305
;;; Empirical variance: 13.611169749999986
;;; Empirical std: 3.6893318839594773
;;; Empirical skewness: 0.07257289912397065
;;; Empirical kurtosis: 2.6888781211332353
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 2172.680705 ms
;;; Empirical mean: 21.087100000000007
;;; Empirical variance: 15.623713589999992
;;; Empirical std: 3.95268435243696
;;; Empirical skewness: 0.3179462306998039
;;; Empirical kurtosis: 3.109040264184404
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 1963.451218 ms
;;; Empirical mean: 20.6715
;;; Empirical variance: 16.858787749999983
;;; Empirical std: 4.105945414883153
;;; Empirical skewness: 0.2043112771925136
;;; Empirical kurtosis: 3.1349353423273647
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 1230.979321 ms
;;; Empirical mean: 20.5536696963674
;;; Empirical variance: 15.239602862134676
;;; Empirical std: 3.9037933938842966
;;; Empirical skewness: 0.2839629084952142
;;; Empirical kurtosis: 3.0939754742983614
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 1156.463746 ms
;;; Empirical mean: 20.537739231264162
;;; Empirical variance: 15.186060038723474
;;; Empirical std: 3.8969295655327763
;;; Empirical skewness: 0.2442615448459393
;;; Empirical kurtosis: 3.00216008847792
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 1137.388559 ms
;;; Empirical mean: 20.63926108311234
;;; Empirical variance: 15.23866500092121
;;; Empirical std: 3.9036732702572854
;;; Empirical skewness: 0.2481436067683727
;;; Empirical kurtosis: 3.109507055332358
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 1161.426334 ms
;;; Empirical mean: 20.61733590190614
;;; Empirical variance: 15.36756218537757
;;; Empirical std: 3.9201482351280506
;;; Empirical skewness: 0.27213680142959407
;;; Empirical kurtosis: 3.112523532055973
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 1135.607998 ms
;;; Empirical mean: 20.575268782404034
;;; Empirical variance: 15.349134558243312
;;; Empirical std: 3.9177971563422362
;;; Empirical skewness: 0.2725526176729917
;;; Empirical kurtosis: 3.096448245790127
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 1138.140512 ms
;;; Empirical mean: 20.646229782584207
;;; Empirical variance: 15.743369156811832
;;; Empirical std: 3.9677914709334
;;; Empirical skewness: 0.20511088915243983
;;; Empirical kurtosis: 2.9474940760888337
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 1163.361501 ms
;;; Empirical mean: 20.5983626720768
;;; Empirical variance: 15.002482853584883
;;; Empirical std: 3.873303867964
;;; Empirical skewness: 0.2418072030553118
;;; Empirical kurtosis: 3.0948121718457595
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 1115.767807 ms
;;; Empirical mean: 20.594433270565574
;;; Empirical variance: 14.968043034608533
;;; Empirical std: 3.868855519996648
;;; Empirical skewness: 0.21228216171406825
;;; Empirical kurtosis: 3.006785245899165
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 1136.347601 ms
;;; Empirical mean: 20.490619260567147
;;; Empirical variance: 15.140405157830736
;;; Empirical std: 3.891067354573901
;;; Empirical skewness: 0.2587299737110374
;;; Empirical kurtosis: 3.0228290109673006
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 1158.606819 ms
;;; Empirical mean: 20.627918333655746
;;; Empirical variance: 15.643655906612992
;;; Empirical std: 3.9552061775099654
;;; Empirical skewness: 0.24067154355410308
;;; Empirical kurtosis: 2.99306795019165
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 1358.055403 ms
;;; Empirical mean: 20.628154065111378
;;; Empirical variance: 15.449401698614258
;;; Empirical std: 3.9305726934651974
;;; Empirical skewness: 0.24292742961627645
;;; Empirical kurtosis: 3.047612320295299
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 1227.333277 ms
;;; Empirical mean: 20.538374998528305
;;; Empirical variance: 15.21158855234484
;;; Empirical std: 3.9002036552396646
;;; Empirical skewness: 0.29993249244648695
;;; Empirical kurtosis: 3.1632391670354
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 1223.127477 ms
;;; Empirical mean: 20.616721444419607
;;; Empirical variance: 15.252646646306935
;;; Empirical std: 3.90546369158733
;;; Empirical skewness: 0.2564849597226129
;;; Empirical kurtosis: 3.090388129721311
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 1180.774595 ms
;;; Empirical mean: 20.619810868533204
;;; Empirical variance: 15.052201308106824
;;; Empirical std: 3.87971665306976
;;; Empirical skewness: 0.2437842471818024
;;; Empirical kurtosis: 3.0312363175670045
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 1191.82802 ms
;;; Empirical mean: 20.562098259782868
;;; Empirical variance: 14.823650937327168
;;; Empirical std: 3.8501494694787066
;;; Empirical skewness: 0.24036772398169876
;;; Empirical kurtosis: 3.185991846217282
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 1179.473153 ms
;;; Empirical mean: 20.648377554592958
;;; Empirical variance: 15.593683508677392
;;; Empirical std: 3.948883830739693
;;; Empirical skewness: 0.2543967286273098
;;; Empirical kurtosis: 3.010020371922584
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 1195.192911 ms
;;; Empirical mean: 20.582467830962326
;;; Empirical variance: 15.05621631042522
;;; Empirical std: 3.8802340535623907
;;; Empirical skewness: 0.2472108985804246
;;; Empirical kurtosis: 3.09516139721434
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 1208.534616 ms
;;; Empirical mean: 20.5975085528691
;;; Empirical variance: 15.647015298229292
;;; Empirical std: 3.9556308344218993
;;; Empirical skewness: 0.25192554403695316
;;; Empirical kurtosis: 3.0053963146150235
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 1204.968138 ms
;;; Empirical mean: 20.6191004481754
;;; Empirical variance: 15.642653562185114
;;; Empirical std: 3.95507946344762
;;; Empirical skewness: 0.24373937440105553
;;; Empirical kurtosis: 3.038170174830454
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 1254.838369 ms
;;; Empirical mean: 20.705849581840013
;;; Empirical variance: 15.66883002374915
;;; Empirical std: 3.9583873008776123
;;; Empirical skewness: 0.21211184625102675
;;; Empirical kurtosis: 3.0112132454002207
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 0
;;; Elapsed time: 1065.721653 ms
;;; Empirical mean: 20.57166769146857
;;; Empirical variance: 15.200211207799686
;;; Empirical std: 3.898744824658275
;;; Empirical skewness: 0.27610204689573903
;;; Empirical kurtosis: 3.1425416518422358
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 1
;;; Elapsed time: 982.580853 ms
;;; Empirical mean: 20.574389267903893
;;; Empirical variance: 15.418384287523423
;;; Empirical std: 3.9266250505393843
;;; Empirical skewness: 0.30300603389702685
;;; Empirical kurtosis: 3.126694652435431
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 2
;;; Elapsed time: 935.980811 ms
;;; Empirical mean: 20.567742574705832
;;; Empirical variance: 15.28454844234489
;;; Empirical std: 3.9095458102374105
;;; Empirical skewness: 0.2510540244447007
;;; Empirical kurtosis: 3.1341947745679595
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 3
;;; Elapsed time: 945.737102 ms
;;; Empirical mean: 20.585088849129068
;;; Empirical variance: 15.413182716409416
;;; Empirical std: 3.9259626483716596
;;; Empirical skewness: 0.2646741136543499
;;; Empirical kurtosis: 3.051835560355058
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 4
;;; Elapsed time: 977.608021 ms
;;; Empirical mean: 20.557597499709264
;;; Empirical variance: 15.521733345612232
;;; Empirical std: 3.9397631078038478
;;; Empirical skewness: 0.2988578854536247
;;; Empirical kurtosis: 3.1224940069735334
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 5
;;; Elapsed time: 1023.93874 ms
;;; Empirical mean: 20.579081689487474
;;; Empirical variance: 15.005940818891537
;;; Empirical std: 3.87375022670429
;;; Empirical skewness: 0.2910836715990596
;;; Empirical kurtosis: 3.121820941466017
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 6
;;; Elapsed time: 1339.587638 ms
;;; Empirical mean: 20.587506098482663
;;; Empirical variance: 15.005570128512424
;;; Empirical std: 3.873702379960601
;;; Empirical skewness: 0.2335013711326203
;;; Empirical kurtosis: 3.0126965864532713
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 7
;;; Elapsed time: 1044.589053 ms
;;; Empirical mean: 20.594688934293895
;;; Empirical variance: 15.522175924491252
;;; Empirical std: 3.9398192756129378
;;; Empirical skewness: 0.2171998243360898
;;; Empirical kurtosis: 2.9480889043783307
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 8
;;; Elapsed time: 956.489224 ms
;;; Empirical mean: 20.565266707980847
;;; Empirical variance: 15.140320616265885
;;; Empirical std: 3.8910564910144756
;;; Empirical skewness: 0.28856613728433345
;;; Empirical kurtosis: 3.1153632438710157
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 9
;;; Elapsed time: 960.219562 ms
;;; Empirical mean: 20.66084028097845
;;; Empirical variance: 15.16856663252547
;;; Empirical std: 3.8946844073076665
;;; Empirical skewness: 0.24627628422247191
;;; Empirical kurtosis: 3.0383087415855865
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 10
;;; Elapsed time: 974.484307 ms
;;; Empirical mean: 20.563590490703994
;;; Empirical variance: 15.498992202805363
;;; Empirical std: 3.9368759445536714
;;; Empirical skewness: 0.24993169255216588
;;; Empirical kurtosis: 3.129106535023706
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 11
;;; Elapsed time: 970.71576 ms
;;; Empirical mean: 20.56467943026627
;;; Empirical variance: 15.412753851120595
;;; Empirical std: 3.9259080288667736
;;; Empirical skewness: 0.251242990149882
;;; Empirical kurtosis: 2.9559167428842876
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 12
;;; Elapsed time: 1007.995187 ms
;;; Empirical mean: 20.58160655648236
;;; Empirical variance: 14.850770085553831
;;; Empirical std: 3.8536696907692845
;;; Empirical skewness: 0.21950800439576512
;;; Empirical kurtosis: 3.1221058052498187
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 13
;;; Elapsed time: 957.026325 ms
;;; Empirical mean: 20.573629341950276
;;; Empirical variance: 15.228438321799967
;;; Empirical std: 3.902363171438554
;;; Empirical skewness: 0.22212693967040328
;;; Empirical kurtosis: 2.9788044560409395
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 14
;;; Elapsed time: 931.953817 ms
;;; Empirical mean: 20.59618199134896
;;; Empirical variance: 14.742997265070436
;;; Empirical std: 3.839661087266744
;;; Empirical skewness: 0.20913250129390767
;;; Empirical kurtosis: 3.0229201500437606
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 15
;;; Elapsed time: 1012.969272 ms
;;; Empirical mean: 20.581771305784713
;;; Empirical variance: 15.106450761732107
;;; Empirical std: 3.88670178451243
;;; Empirical skewness: 0.227721172005749
;;; Empirical kurtosis: 3.059957338042601
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 16
;;; Elapsed time: 1016.452463 ms
;;; Empirical mean: 20.712040618116657
;;; Empirical variance: 15.91055043557536
;;; Empirical std: 3.9888031332187053
;;; Empirical skewness: 0.28463575627236315
;;; Empirical kurtosis: 3.065258678766735
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 17
;;; Elapsed time: 969.919618 ms
;;; Empirical mean: 20.596622944734424
;;; Empirical variance: 15.315289814845379
;;; Empirical std: 3.9134754138547208
;;; Empirical skewness: 0.2786410449172542
;;; Empirical kurtosis: 3.0985951162186276
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 18
;;; Elapsed time: 1024.917365 ms
;;; Empirical mean: 20.63477849128784
;;; Empirical variance: 15.72226391402098
;;; Empirical std: 3.9651310084309923
;;; Empirical skewness: 0.25973546801595876
;;; Empirical kurtosis: 3.0237128592400278
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 19
;;; Elapsed time: 1034.18207 ms
;;; Empirical mean: 20.633779547047038
;;; Empirical variance: 15.451553877013536
;;; Empirical std: 3.9308464580817115
;;; Empirical skewness: 0.23550903284667477
;;; Empirical kurtosis: 3.0357386340077994
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 1171.442809 ms
;;; Empirical mean: 20.606300000000005
;;; Empirical variance: 15.160500309999989
;;; Empirical std: 3.8936487142524796
;;; Empirical skewness: 0.20797324202705766
;;; Empirical kurtosis: 3.0231425584810365
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 1100.274848 ms
;;; Empirical mean: 20.649399999999996
;;; Empirical variance: 15.213079639999986
;;; Empirical std: 3.900394805657497
;;; Empirical skewness: 0.1898580961765866
;;; Empirical kurtosis: 2.9625887658273258
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 1083.563523 ms
;;; Empirical mean: 20.546600000000005
;;; Empirical variance: 14.421028439999988
;;; Empirical std: 3.797502921657861
;;; Empirical skewness: 0.26919356850492604
;;; Empirical kurtosis: 3.0265768812593095
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 1099.389621 ms
;;; Empirical mean: 20.4678
;;; Empirical variance: 14.94676315999999
;;; Empirical std: 3.8661043907271813
;;; Empirical skewness: 0.23185199598939915
;;; Empirical kurtosis: 2.875269342928771
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 1178.296112 ms
;;; Empirical mean: 20.6841
;;; Empirical variance: 15.301107189999989
;;; Empirical std: 3.9116629698888925
;;; Empirical skewness: 0.18876458020771947
;;; Empirical kurtosis: 3.084274672905182
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 1090.450921 ms
;;; Empirical mean: 20.6034
;;; Empirical variance: 14.906108439999985
;;; Empirical std: 3.860842970129708
;;; Empirical skewness: 0.2422699352208733
;;; Empirical kurtosis: 3.0702701667630627
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 1096.850201 ms
;;; Empirical mean: 20.5355
;;; Empirical variance: 14.590939749999986
;;; Empirical std: 3.819808863019194
;;; Empirical skewness: 0.17674828571654178
;;; Empirical kurtosis: 2.957405677019314
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 1066.554151 ms
;;; Empirical mean: 20.468899999999998
;;; Empirical variance: 15.418832789999986
;;; Empirical std: 3.926682160552339
;;; Empirical skewness: 0.28717439313388193
;;; Empirical kurtosis: 3.1322789038163137
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 1102.243675 ms
;;; Empirical mean: 20.684800000000006
;;; Empirical variance: 15.642448959999996
;;; Empirical std: 3.9550535976140697
;;; Empirical skewness: 0.2805840369798071
;;; Empirical kurtosis: 3.1517581796038967
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 1068.66052 ms
;;; Empirical mean: 20.648000000000007
;;; Empirical variance: 15.111295999999994
;;; Empirical std: 3.8873250442945975
;;; Empirical skewness: 0.16526262210726841
;;; Empirical kurtosis: 2.890421149712062
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 1069.221864 ms
;;; Empirical mean: 20.604400000000002
;;; Empirical variance: 15.131300639999996
;;; Empirical std: 3.889897253141784
;;; Empirical skewness: 0.17078278208494066
;;; Empirical kurtosis: 2.9145688431980097
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 1112.019285 ms
;;; Empirical mean: 20.5541
;;; Empirical variance: 15.813473189999975
;;; Empirical std: 3.9766157961261452
;;; Empirical skewness: 0.2996534542065986
;;; Empirical kurtosis: 2.8709223219809443
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 1046.407021 ms
;;; Empirical mean: 20.6186
;;; Empirical variance: 16.10073403999999
;;; Empirical std: 4.012571998107945
;;; Empirical skewness: 0.2338589534738094
;;; Empirical kurtosis: 2.971180695614579
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 1061.98427 ms
;;; Empirical mean: 20.573400000000003
;;; Empirical variance: 15.81941243999999
;;; Empirical std: 3.9773624979375453
;;; Empirical skewness: 0.33212131989035815
;;; Empirical kurtosis: 3.202642817904811
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 1059.064843 ms
;;; Empirical mean: 20.5365
;;; Empirical variance: 16.010867749999985
;;; Empirical std: 4.001358238148639
;;; Empirical skewness: 0.323826050089321
;;; Empirical kurtosis: 3.193197782021877
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 1047.969243 ms
;;; Empirical mean: 20.477899999999995
;;; Empirical variance: 15.40471158999998
;;; Empirical std: 3.924883640313427
;;; Empirical skewness: 0.20653104474937672
;;; Empirical kurtosis: 2.947565440007137
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 1075.120284 ms
;;; Empirical mean: 20.695199999999996
;;; Empirical variance: 14.925696959999984
;;; Empirical std: 3.8633789563023693
;;; Empirical skewness: 0.19628274207040552
;;; Empirical kurtosis: 3.0183762071877496
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 1055.184757 ms
;;; Empirical mean: 20.667799999999993
;;; Empirical variance: 15.75824315999998
;;; Empirical std: 3.9696653712876078
;;; Empirical skewness: 0.217852067655536
;;; Empirical kurtosis: 3.0077241625982123
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 1050.498267 ms
;;; Empirical mean: 20.52229999999999
;;; Empirical variance: 15.462902709999977
;;; Empirical std: 3.932289754074587
;;; Empirical skewness: 0.2381134389601998
;;; Empirical kurtosis: 2.9243496939248335
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 1057.366875 ms
;;; Empirical mean: 20.7122
;;; Empirical variance: 15.779571159999989
;;; Empirical std: 3.9723508354625463
;;; Empirical skewness: 0.19436076229678736
;;; Empirical kurtosis: 3.035317584898346
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 624.868469 ms
;;; Empirical mean: 20.620725464458644
;;; Empirical variance: 13.959244603278531
;;; Empirical std: 3.7362072484377165
;;; Empirical skewness: 0.25339178891552827
;;; Empirical kurtosis: 3.2909101605958595
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 610.532517 ms
;;; Empirical mean: 20.878082359632234
;;; Empirical variance: 14.923150177370367
;;; Empirical std: 3.863049336647199
;;; Empirical skewness: 0.33065121684703014
;;; Empirical kurtosis: 3.2502420636498885
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 614.678422 ms
;;; Empirical mean: 20.804204323943047
;;; Empirical variance: 14.61450390652558
;;; Empirical std: 3.822892086696351
;;; Empirical skewness: 0.3155887042704996
;;; Empirical kurtosis: 3.3918047137242837
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 628.861997 ms
;;; Empirical mean: 20.66123189817404
;;; Empirical variance: 13.654307643911439
;;; Empirical std: 3.6951735607291085
;;; Empirical skewness: 0.19755985060121284
;;; Empirical kurtosis: 3.051523477463636
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 583.485486 ms
;;; Empirical mean: 20.70715036215756
;;; Empirical variance: 15.311240026680007
;;; Empirical std: 3.912957963827366
;;; Empirical skewness: 0.17821887834673963
;;; Empirical kurtosis: 2.838053685070892
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 581.191109 ms
;;; Empirical mean: 20.766125520189618
;;; Empirical variance: 15.178590042379115
;;; Empirical std: 3.8959710012241
;;; Empirical skewness: 0.2820680678717348
;;; Empirical kurtosis: 3.221735705758317
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 702.770521 ms
;;; Empirical mean: 21.060905483945316
;;; Empirical variance: 16.06983558139074
;;; Empirical std: 4.008719942998106
;;; Empirical skewness: 0.17340294913374268
;;; Empirical kurtosis: 3.1102654605088627
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 603.468902 ms
;;; Empirical mean: 20.59209462304029
;;; Empirical variance: 14.016556361220669
;;; Empirical std: 3.743869169885704
;;; Empirical skewness: 0.154950440884208
;;; Empirical kurtosis: 2.8945044462381193
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 592.672965 ms
;;; Empirical mean: 20.82465977183634
;;; Empirical variance: 14.29272947317698
;;; Empirical std: 3.7805726382622225
;;; Empirical skewness: 0.22015916642626124
;;; Empirical kurtosis: 3.0146491002233637
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 669.648629 ms
;;; Empirical mean: 20.80132725824743
;;; Empirical variance: 14.848777868049673
;;; Empirical std: 3.853411198931367
;;; Empirical skewness: 0.4305664614387019
;;; Empirical kurtosis: 3.469559532140609
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 668.11034 ms
;;; Empirical mean: 20.81813664640198
;;; Empirical variance: 14.423539967071292
;;; Empirical std: 3.7978335886490986
;;; Empirical skewness: 0.22538869426966673
;;; Empirical kurtosis: 2.972998827112211
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 642.198343 ms
;;; Empirical mean: 20.66535454726774
;;; Empirical variance: 14.039075622835538
;;; Empirical std: 3.7468754480013793
;;; Empirical skewness: 0.26360547231522674
;;; Empirical kurtosis: 3.0678753009231103
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 635.794217 ms
;;; Empirical mean: 20.68710883718152
;;; Empirical variance: 15.403857544119123
;;; Empirical std: 3.9247748399263775
;;; Empirical skewness: 0.447443385756417
;;; Empirical kurtosis: 3.403094050410855
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 635.842329 ms
;;; Empirical mean: 20.659993473835723
;;; Empirical variance: 14.633704524759558
;;; Empirical std: 3.8254025310755937
;;; Empirical skewness: 0.24339002784025618
;;; Empirical kurtosis: 3.058031083336069
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 617.166037 ms
;;; Empirical mean: 20.900414281017575
;;; Empirical variance: 15.03518813297407
;;; Empirical std: 3.8775234535685366
;;; Empirical skewness: 0.13382864350033719
;;; Empirical kurtosis: 2.9604602489337672
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 621.021578 ms
;;; Empirical mean: 20.488958008605717
;;; Empirical variance: 13.878722459961164
;;; Empirical std: 3.725415743237413
;;; Empirical skewness: 0.16543164989315484
;;; Empirical kurtosis: 3.082500452078916
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 657.675748 ms
;;; Empirical mean: 20.741412116228286
;;; Empirical variance: 14.89994919533289
;;; Empirical std: 3.860045232291053
;;; Empirical skewness: 0.11213588268611333
;;; Empirical kurtosis: 2.8548661276161442
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 683.75836 ms
;;; Empirical mean: 20.824398604814352
;;; Empirical variance: 14.207965593501894
;;; Empirical std: 3.7693455126191195
;;; Empirical skewness: 0.15289766304802196
;;; Empirical kurtosis: 3.1737317481039007
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 626.129987 ms
;;; Empirical mean: 20.69818632853637
;;; Empirical variance: 15.35719134608885
;;; Empirical std: 3.918825250772079
;;; Empirical skewness: 0.2716457635842692
;;; Empirical kurtosis: 3.1339172000023803
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 580.069298 ms
;;; Empirical mean: 20.753397742329952
;;; Empirical variance: 14.672733480169011
;;; Empirical std: 3.830500421638015
;;; Empirical skewness: 0.2142520281678434
;;; Empirical kurtosis: 3.1565908649521615
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-unkown'>#multifn[print-method 0x4369e02b]</span>","value":"#multifn[print-method 0x4369e02b]"}],"value":"[nil,#multifn[print-method 0x4369e02b]]"},{"type":"html","content":"<span class='clj-unkown'>#multifn[print-method 0x4369e02b]</span>","value":"#multifn[print-method 0x4369e02b]"}],"value":"[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/model</span>","value":"#'model/model"}],"value":"[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/model_name</span>","value":"#'model/model_name"}],"value":"[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/outfile</span>","value":"#'model/outfile"}],"value":"[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/configurations</span>","value":"#'model/configurations"}],"value":"[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/num_samples_options</span>","value":"#'model/num_samples_options"}],"value":"[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/thinning</span>","value":"#'model/thinning"}],"value":"[[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning]"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"[[[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/num-chains</span>","value":"#'model/num-chains"}],"value":"[[[[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains]"},{"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)"}],"value":"[[[[[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains],(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)]"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"[[[[[[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains],(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)],nil]"}
;; <=

;; @@

;; @@
