;; gorilla-repl.fileformat = 1

;; @@

(ns model
  (:require [gorilla-plot.core :as plot])
  (:use [anglican core emit runtime stat
          [state :only [get-predicts get-log-weight get-result]]]))

(defdist geometric
"Geometric distribution on support {0,1,2....}"
[p] []
(sample* [this]
        (loop [value 0]
            (if (sample* (flip p))
            value
            (recur (inc value)))))
(observe* [this value] (+ (log p) (* value (log (- 1 p))))))

(defdist dirac [x]
    (sample* [this] x)
    (observe* [this value]
              (if (= value x)
                0
                (- (/ 1.0 0.0)))))


(with-primitive-procedures [dirac geometric]
  (defquery model [method- options- ]
    (let [[ a] [ 0 ]
          [ a]
          (let [
                a (sample (poisson 51.4))
                a (sample (binomial a 0.2636))
                a (+ a (sample (poisson 232.6)))
                _unused (observe (binomial a 0.2) 45)
                a (sample (binomial a 0.2636))
                a (+ a (sample (poisson 420.8)))
                _unused (observe (binomial a 0.2) 98)
                a (sample (binomial a 0.2636))
                a (+ a (sample (poisson 300.8)))
                _unused (observe (binomial a 0.2) 73)
                a (sample (binomial a 0.2636))
                a (+ a (sample (poisson 85.6)))
                _unused (observe (binomial a 0.2) 38)
               ]
            [ a ]
          )
         ]
    a
    )
  )

)


(def model_name "population2000")
(def outfile "output/population2000_anglican.json")

; (def configurations [:rmh []])
(def configurations
  [
    [:importance []]
    [:lmh []]
    [:rmh []]
    [:rmh [:alpha 0.8 :sigma 2]]
    [:smc []]
    [:smc [:number-of-particles 1000]]
    [:pgibbs []]
    [:pgibbs [:number-of-particles 1000]]
    [:ipmcmc []]
    [:ipmcmc [:number-of-particles 1000 :number-of-nodes 32]]
  ])

; (def num_samples_options [1000])
(def num_samples_options [1000 10000])
(def thinning 1)

(spit outfile "[\n" :append false)

(def num-chains 20)

(doall
  (for [ num_samples num_samples_options
         [method options] configurations
         chain (range 0 num-chains)]
    (do
      (println (format "\nMethod %s with %s samples and options %s" method num_samples options))
      (println (format "Chain no. %s" chain))
      (let [start (. System (nanoTime))
            warmup (/ num_samples 5)
            samples (take-nth thinning (take (* num_samples thinning) (drop warmup (apply doquery method model [method options] options))))
            results (collect-results samples)
            values (map (fn [s] (get-result s)) samples)
            max-value (apply max values)
            mean (empirical-mean results)
            variance (empirical-variance results)
            std (empirical-std results)
            skewness (if (zero? std) (/ 0.0 0.0) (empirical-skew results))
            kurtosis (if (zero? std) (/ 0.0 0.0) (empirical-kurtosis results))
            distribution (empirical-distribution (collect-results samples))
            masses (for [n (range 0 (inc max-value))] (get distribution n 0.0))
            end (. System (nanoTime))
            elapsed_ms (/ (- end start) 1e6)]
        (println (format "Elapsed time: %s ms" elapsed_ms))
        (println (format "Empirical mean: %s" mean))
        (println (format "Empirical variance: %s" variance))
        (println (format "Empirical std: %s" std))
        (println (format "Empirical skewness: %s" skewness))
        (println (format "Empirical kurtosis: %s" kurtosis))
        (spit outfile (format
                   "{\"model\": \"%s\", \"system\": \"anglican\", \"method\": \"%s\", \"options\": \"%s\", \"num_samples\": %s, \"time_ms\": %s, \"total\": 1.0, \"mean\": %s, \"variance\": %s, \"stddev\": %s, \"skewness\": %s, \"kurtosis\": %s, \"masses\": [%s] },\n"
                   model_name method options num_samples elapsed_ms mean variance std skewness kurtosis
                   (clojure.string/join ", " masses)) :append true)
        (if false (do
          (println "Empirical distribution:")
          (doall (for [n (range 0 (inc max-value))]
            (println (format "p(%s) = %s" n (get distribution n 0.0)))))))
        ; (println "List of samples (format: sample log-weight):")
        ; (doall (map (fn [s] (println (format "%s %s" (get-result s) (get-log-weight s)))) samples))
        ; values need to be adjusted if they are weighted!
        ; (plot/histogram values :normalize :probability)
      )
    )
  )
)

(spit outfile "]\n" :append true)




;; @@
;; ->
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 171.3421 ms
;;; Empirical mean: 193.9501496222223
;;; Empirical variance: 149.3902650892723
;;; Empirical std: 12.222531042679838
;;; Empirical skewness: 0.16551813186695408
;;; Empirical kurtosis: 3.0012458111977915
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 150.761198 ms
;;; Empirical mean: 194.602757833903
;;; Empirical variance: 148.50271100112252
;;; Empirical std: 12.186168840169683
;;; Empirical skewness: 0.13531958068979813
;;; Empirical kurtosis: 2.769541904404621
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 159.15685 ms
;;; Empirical mean: 193.7010581607637
;;; Empirical variance: 160.31584732555186
;;; Empirical std: 12.661589447046206
;;; Empirical skewness: 0.08802032025641303
;;; Empirical kurtosis: 2.764427584549639
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 154.797297 ms
;;; Empirical mean: 194.5168377167643
;;; Empirical variance: 150.66600247780275
;;; Empirical std: 12.27460803764433
;;; Empirical skewness: 0.012857580151593752
;;; Empirical kurtosis: 3.0310029883538836
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 158.108954 ms
;;; Empirical mean: 194.5153665799451
;;; Empirical variance: 144.98051341443494
;;; Empirical std: 12.040785415180977
;;; Empirical skewness: 0.13288708059837293
;;; Empirical kurtosis: 3.274013542586061
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 161.368007 ms
;;; Empirical mean: 194.48251514628063
;;; Empirical variance: 161.50897934796586
;;; Empirical std: 12.708618309948799
;;; Empirical skewness: 0.027970922403576598
;;; Empirical kurtosis: 2.96332252514696
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 154.695458 ms
;;; Empirical mean: 194.5488313266986
;;; Empirical variance: 144.16235792841914
;;; Empirical std: 12.006763007922624
;;; Empirical skewness: 0.12316155166766536
;;; Empirical kurtosis: 3.236138852297595
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 156.195474 ms
;;; Empirical mean: 194.90459648722586
;;; Empirical variance: 152.5608504184733
;;; Empirical std: 12.351552550933558
;;; Empirical skewness: 0.06605501826191318
;;; Empirical kurtosis: 3.100052696309037
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 156.751624 ms
;;; Empirical mean: 193.94529380188752
;;; Empirical variance: 144.3694878313103
;;; Empirical std: 12.015385463284575
;;; Empirical skewness: 0.03265663475942757
;;; Empirical kurtosis: 2.8244666624588226
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 159.787998 ms
;;; Empirical mean: 193.7677841666242
;;; Empirical variance: 144.65629612839712
;;; Empirical std: 12.027314585076633
;;; Empirical skewness: 0.1865130922168561
;;; Empirical kurtosis: 2.969309818886934
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 155.508392 ms
;;; Empirical mean: 194.19052588113593
;;; Empirical variance: 157.07086722793787
;;; Empirical std: 12.53279167735337
;;; Empirical skewness: 0.036237778255600914
;;; Empirical kurtosis: 2.906330251255483
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 156.592117 ms
;;; Empirical mean: 194.58865814310525
;;; Empirical variance: 149.0094684641161
;;; Empirical std: 12.206943452974464
;;; Empirical skewness: -0.032073600781142235
;;; Empirical kurtosis: 2.8336430421961056
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 160.892465 ms
;;; Empirical mean: 194.36052272593054
;;; Empirical variance: 148.65275438314742
;;; Empirical std: 12.192323584253636
;;; Empirical skewness: 0.05881808485148541
;;; Empirical kurtosis: 3.103906033281919
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 161.720759 ms
;;; Empirical mean: 194.06339510394886
;;; Empirical variance: 144.16526024876313
;;; Empirical std: 12.006883869212825
;;; Empirical skewness: 0.1878114722151944
;;; Empirical kurtosis: 3.01412706480417
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 167.677685 ms
;;; Empirical mean: 194.82241202358074
;;; Empirical variance: 151.65090825254313
;;; Empirical std: 12.314662327995158
;;; Empirical skewness: -0.03515687307569949
;;; Empirical kurtosis: 2.930511442243332
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 167.099137 ms
;;; Empirical mean: 194.13491306816465
;;; Empirical variance: 144.0911583855512
;;; Empirical std: 12.003797665137114
;;; Empirical skewness: 0.22249438766915444
;;; Empirical kurtosis: 3.14660989005971
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 169.057939 ms
;;; Empirical mean: 193.85586500881953
;;; Empirical variance: 157.54730249215976
;;; Empirical std: 12.55178483292953
;;; Empirical skewness: -0.13630440419623407
;;; Empirical kurtosis: 3.1178986602240966
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 174.372783 ms
;;; Empirical mean: 194.47469774354917
;;; Empirical variance: 154.4495934482337
;;; Empirical std: 12.427775080368718
;;; Empirical skewness: -0.059358943336776125
;;; Empirical kurtosis: 3.2138843108085378
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 167.482203 ms
;;; Empirical mean: 194.04191681481547
;;; Empirical variance: 162.16318215593765
;;; Empirical std: 12.734330848377454
;;; Empirical skewness: 0.14963390585069883
;;; Empirical kurtosis: 2.8688122734538997
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 155.720909 ms
;;; Empirical mean: 194.26951293023856
;;; Empirical variance: 157.56036803290016
;;; Empirical std: 12.552305287591604
;;; Empirical skewness: 0.07730139010939716
;;; Empirical kurtosis: 2.8073882504547947
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 138.486646 ms
;;; Empirical mean: 194.46300000000005
;;; Empirical variance: 157.874631
;;; Empirical std: 12.564817189278958
;;; Empirical skewness: -0.07190732922083971
;;; Empirical kurtosis: 2.4517472899084694
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 140.061808 ms
;;; Empirical mean: 194.80899999999994
;;; Empirical variance: 163.416519
;;; Empirical std: 12.783447070332791
;;; Empirical skewness: 0.08230610753655987
;;; Empirical kurtosis: 2.5691309534514857
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 137.01587 ms
;;; Empirical mean: 198.682
;;; Empirical variance: 135.99487599999995
;;; Empirical std: 11.661684097933708
;;; Empirical skewness: -0.11485047462942957
;;; Empirical kurtosis: 2.851547459337309
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 150.027685 ms
;;; Empirical mean: 194.70599999999996
;;; Empirical variance: 180.97556399999993
;;; Empirical std: 13.452715859632209
;;; Empirical skewness: 0.19772747917749015
;;; Empirical kurtosis: 2.5750641686167457
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 150.456814 ms
;;; Empirical mean: 192.36999999999998
;;; Empirical variance: 141.3851
;;; Empirical std: 11.890546665313584
;;; Empirical skewness: 0.23759498298889256
;;; Empirical kurtosis: 3.0855674102604573
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 143.103859 ms
;;; Empirical mean: 195.669
;;; Empirical variance: 144.98743899999997
;;; Empirical std: 12.041073000360058
;;; Empirical skewness: 0.21987790009051658
;;; Empirical kurtosis: 3.9701930273336816
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 143.060212 ms
;;; Empirical mean: 194.34499999999997
;;; Empirical variance: 123.20597500000001
;;; Empirical std: 11.099818692212951
;;; Empirical skewness: -0.13023613505111667
;;; Empirical kurtosis: 3.0406630290209313
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 141.298347 ms
;;; Empirical mean: 193.67999999999995
;;; Empirical variance: 147.04760000000002
;;; Empirical std: 12.126318485014322
;;; Empirical skewness: 0.20015997455273735
;;; Empirical kurtosis: 3.1216951575267498
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 148.159373 ms
;;; Empirical mean: 191.364
;;; Empirical variance: 182.071504
;;; Empirical std: 13.493387417546419
;;; Empirical skewness: -0.21348443665382297
;;; Empirical kurtosis: 3.0666106614068522
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 141.453881 ms
;;; Empirical mean: 192.83499999999987
;;; Empirical variance: 185.61177500000002
;;; Empirical std: 13.62394124326731
;;; Empirical skewness: 0.18864683951217862
;;; Empirical kurtosis: 3.0709099380165807
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 143.520703 ms
;;; Empirical mean: 192.52400000000003
;;; Empirical variance: 138.45542400000008
;;; Empirical std: 11.766708290766797
;;; Empirical skewness: 0.15897116715883847
;;; Empirical kurtosis: 3.315891540013041
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 151.997752 ms
;;; Empirical mean: 193.84599999999995
;;; Empirical variance: 128.56228399999995
;;; Empirical std: 11.338530945409108
;;; Empirical skewness: 0.3860414056686613
;;; Empirical kurtosis: 2.960556162259652
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 148.407631 ms
;;; Empirical mean: 191.2750000000001
;;; Empirical variance: 156.5013750000001
;;; Empirical std: 12.510050959128828
;;; Empirical skewness: 0.28968583624938776
;;; Empirical kurtosis: 2.7789082823514364
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 133.625091 ms
;;; Empirical mean: 195.18099999999995
;;; Empirical variance: 127.27423900000001
;;; Empirical std: 11.281588496306716
;;; Empirical skewness: -0.046099929514018756
;;; Empirical kurtosis: 2.454813374862297
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 152.832656 ms
;;; Empirical mean: 192.30299999999997
;;; Empirical variance: 138.163191
;;; Empirical std: 11.754283942461148
;;; Empirical skewness: 0.3969883734797252
;;; Empirical kurtosis: 2.9119031249553204
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 140.895485 ms
;;; Empirical mean: 196.94899999999998
;;; Empirical variance: 143.42439900000002
;;; Empirical std: 11.975992610218162
;;; Empirical skewness: 0.29712991681680784
;;; Empirical kurtosis: 2.608363982611752
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 136.709837 ms
;;; Empirical mean: 190.84000000000003
;;; Empirical variance: 133.72840000000005
;;; Empirical std: 11.564099619079734
;;; Empirical skewness: 0.1847211340716981
;;; Empirical kurtosis: 2.7980977914230163
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 134.802329 ms
;;; Empirical mean: 192.39799999999997
;;; Empirical variance: 117.36159600000002
;;; Empirical std: 10.833355712797397
;;; Empirical skewness: 0.365755800701906
;;; Empirical kurtosis: 2.5422473788331623
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 155.422151 ms
;;; Empirical mean: 196.90500000000006
;;; Empirical variance: 153.57997500000002
;;; Empirical std: 12.392738801411092
;;; Empirical skewness: 0.4010411225658197
;;; Empirical kurtosis: 2.725186113150073
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 138.521511 ms
;;; Empirical mean: 192.60099999999997
;;; Empirical variance: 149.28779899999998
;;; Empirical std: 12.218338635019082
;;; Empirical skewness: 0.01400621269007762
;;; Empirical kurtosis: 3.5212406642250778
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 201.161335 ms
;;; Empirical mean: 194.4159999999999
;;; Empirical variance: 176.54094400000002
;;; Empirical std: 13.28687111399821
;;; Empirical skewness: 0.6315903921630732
;;; Empirical kurtosis: 3.68693963870899
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 214.406638 ms
;;; Empirical mean: 195.971
;;; Empirical variance: 171.04015900000007
;;; Empirical std: 13.07823225822206
;;; Empirical skewness: 0.24839053677329612
;;; Empirical kurtosis: 3.0155107370878933
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 213.973042 ms
;;; Empirical mean: 197.91700000000006
;;; Empirical variance: 238.24611100000004
;;; Empirical std: 15.435223062852057
;;; Empirical skewness: -0.02658770563037279
;;; Empirical kurtosis: 2.54976463884571
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 213.628554 ms
;;; Empirical mean: 194.0880000000001
;;; Empirical variance: 179.40825600000008
;;; Empirical std: 13.394336713701058
;;; Empirical skewness: -0.313814119815881
;;; Empirical kurtosis: 2.573864538186934
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 207.55094 ms
;;; Empirical mean: 193.051
;;; Empirical variance: 113.15239900000003
;;; Empirical std: 10.637311643455785
;;; Empirical skewness: 0.13346590880449766
;;; Empirical kurtosis: 3.127615563387889
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 212.969253 ms
;;; Empirical mean: 192.91200000000003
;;; Empirical variance: 149.9862560000001
;;; Empirical std: 12.246887604612043
;;; Empirical skewness: 0.23046282370330315
;;; Empirical kurtosis: 3.890922272045941
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 228.828925 ms
;;; Empirical mean: 192.54299999999998
;;; Empirical variance: 159.28015099999996
;;; Empirical std: 12.620624033699759
;;; Empirical skewness: -0.6648684515212508
;;; Empirical kurtosis: 3.430078171950206
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 261.346637 ms
;;; Empirical mean: 192.11500000000004
;;; Empirical variance: 171.05977500000003
;;; Empirical std: 13.078982185170222
;;; Empirical skewness: -0.13734940589298877
;;; Empirical kurtosis: 2.7014127399832266
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 220.915913 ms
;;; Empirical mean: 197.28799999999998
;;; Empirical variance: 98.71905600000001
;;; Empirical std: 9.93574637357456
;;; Empirical skewness: 0.022764851501962707
;;; Empirical kurtosis: 3.0593991242039715
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 229.720935 ms
;;; Empirical mean: 193.08999999999992
;;; Empirical variance: 123.64590000000001
;;; Empirical std: 11.119617799187164
;;; Empirical skewness: -0.08550406331904356
;;; Empirical kurtosis: 2.75350054510861
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 211.960464 ms
;;; Empirical mean: 193.22299999999996
;;; Empirical variance: 123.65327100000003
;;; Empirical std: 11.119949235495639
;;; Empirical skewness: 0.5338445174626937
;;; Empirical kurtosis: 2.749722386837131
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 214.781669 ms
;;; Empirical mean: 192.78199999999998
;;; Empirical variance: 167.38047600000002
;;; Empirical std: 12.937560666524428
;;; Empirical skewness: 0.054710441773752554
;;; Empirical kurtosis: 2.580624470505146
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 214.649664 ms
;;; Empirical mean: 196.36
;;; Empirical variance: 134.84240000000003
;;; Empirical std: 11.61216603394905
;;; Empirical skewness: -0.04165023256615312
;;; Empirical kurtosis: 2.2316281069565918
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 214.554047 ms
;;; Empirical mean: 190.33300000000008
;;; Empirical variance: 166.530111
;;; Empirical std: 12.90465462536677
;;; Empirical skewness: 0.26832892164143773
;;; Empirical kurtosis: 2.7915939369208975
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 247.606454 ms
;;; Empirical mean: 196.76800000000003
;;; Empirical variance: 172.772176
;;; Empirical std: 13.144283015820985
;;; Empirical skewness: -0.0038362867975936877
;;; Empirical kurtosis: 2.2550614277847085
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 242.915784 ms
;;; Empirical mean: 194.122
;;; Empirical variance: 147.78111600000005
;;; Empirical std: 12.156525654972315
;;; Empirical skewness: 0.3826008406617483
;;; Empirical kurtosis: 3.5723205173847368
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 207.069021 ms
;;; Empirical mean: 189.07499999999996
;;; Empirical variance: 106.71337500000004
;;; Empirical std: 10.330216599858884
;;; Empirical skewness: 0.18210369834515208
;;; Empirical kurtosis: 2.8279575330410127
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 212.323537 ms
;;; Empirical mean: 197.278
;;; Empirical variance: 176.330716
;;; Empirical std: 13.278957639814957
;;; Empirical skewness: 0.02406619008772975
;;; Empirical kurtosis: 2.251906491615663
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 207.046596 ms
;;; Empirical mean: 195.9279999999999
;;; Empirical variance: 127.39481599999992
;;; Empirical std: 11.286931203830381
;;; Empirical skewness: -0.025010385586029273
;;; Empirical kurtosis: 3.4611912236581777
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 206.846342 ms
;;; Empirical mean: 191.28999999999994
;;; Empirical variance: 150.99189999999996
;;; Empirical std: 12.287876138698662
;;; Empirical skewness: -0.5000433657096791
;;; Empirical kurtosis: 2.91839555277817
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 0
;;; Elapsed time: 217.545803 ms
;;; Empirical mean: 195.27000000000004
;;; Empirical variance: 139.5191
;;; Empirical std: 11.811820350818074
;;; Empirical skewness: -0.2561885165980334
;;; Empirical kurtosis: 2.045741563394401
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 1
;;; Elapsed time: 210.129512 ms
;;; Empirical mean: 195.95600000000005
;;; Empirical variance: 187.84206399999994
;;; Empirical std: 13.705548657386903
;;; Empirical skewness: 0.2533379965913751
;;; Empirical kurtosis: 3.014485297702383
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 2
;;; Elapsed time: 200.759066 ms
;;; Empirical mean: 190.00099999999992
;;; Empirical variance: 120.72499899999993
;;; Empirical std: 10.987492844138735
;;; Empirical skewness: -0.15498725205149774
;;; Empirical kurtosis: 4.242528726605368
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 3
;;; Elapsed time: 209.464081 ms
;;; Empirical mean: 193.97399999999993
;;; Empirical variance: 189.87132399999996
;;; Empirical std: 13.779380392455966
;;; Empirical skewness: 0.17436699799081293
;;; Empirical kurtosis: 2.322937185324017
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 4
;;; Elapsed time: 200.123484 ms
;;; Empirical mean: 195.96699999999998
;;; Empirical variance: 102.75191100000002
;;; Empirical std: 10.136661728596847
;;; Empirical skewness: -0.06868968777294879
;;; Empirical kurtosis: 2.507142074732741
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 5
;;; Elapsed time: 207.403205 ms
;;; Empirical mean: 193.08600000000004
;;; Empirical variance: 121.89260399999999
;;; Empirical std: 11.040498358316983
;;; Empirical skewness: 0.0980141726124409
;;; Empirical kurtosis: 2.2025926424235664
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 6
;;; Elapsed time: 209.768247 ms
;;; Empirical mean: 192.24899999999982
;;; Empirical variance: 216.46499899999992
;;; Empirical std: 14.712749539090234
;;; Empirical skewness: 0.37267582069760197
;;; Empirical kurtosis: 2.548461065207549
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 7
;;; Elapsed time: 242.750167 ms
;;; Empirical mean: 199.93200000000002
;;; Empirical variance: 106.37537599999999
;;; Empirical std: 10.313843900311852
;;; Empirical skewness: 0.13506482591802652
;;; Empirical kurtosis: 2.5567629287025606
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 8
;;; Elapsed time: 206.120487 ms
;;; Empirical mean: 191.17500000000004
;;; Empirical variance: 129.59637500000005
;;; Empirical std: 11.384040363596752
;;; Empirical skewness: 0.3661766591003601
;;; Empirical kurtosis: 2.5414431961953814
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 9
;;; Elapsed time: 206.994887 ms
;;; Empirical mean: 193.268
;;; Empirical variance: 88.636176
;;; Empirical std: 9.41467875182154
;;; Empirical skewness: -0.42952048228543993
;;; Empirical kurtosis: 4.122828746656984
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 10
;;; Elapsed time: 211.870837 ms
;;; Empirical mean: 192.608
;;; Empirical variance: 121.98433599999998
;;; Empirical std: 11.044651918462618
;;; Empirical skewness: 0.2517599848537935
;;; Empirical kurtosis: 2.2793474573438663
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 11
;;; Elapsed time: 207.440378 ms
;;; Empirical mean: 196.83799999999994
;;; Empirical variance: 168.63375599999995
;;; Empirical std: 12.985906052332272
;;; Empirical skewness: -0.07034129782185351
;;; Empirical kurtosis: 2.953745965319117
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 12
;;; Elapsed time: 205.4616 ms
;;; Empirical mean: 191.4200000000001
;;; Empirical variance: 157.4996000000001
;;; Empirical std: 12.54988446161956
;;; Empirical skewness: 0.2618061060734478
;;; Empirical kurtosis: 2.103171867043124
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 13
;;; Elapsed time: 223.345246 ms
;;; Empirical mean: 196.0229999999999
;;; Empirical variance: 150.388471
;;; Empirical std: 12.263297721249371
;;; Empirical skewness: -0.2403880985976932
;;; Empirical kurtosis: 2.7873612180458096
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 14
;;; Elapsed time: 210.795997 ms
;;; Empirical mean: 193.82700000000006
;;; Empirical variance: 132.22107100000005
;;; Empirical std: 11.498742148600432
;;; Empirical skewness: -0.1443781578631708
;;; Empirical kurtosis: 2.7333298974065845
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 15
;;; Elapsed time: 212.003543 ms
;;; Empirical mean: 196.67000000000007
;;; Empirical variance: 106.30310000000009
;;; Empirical std: 10.310339470647904
;;; Empirical skewness: -0.19808378172967311
;;; Empirical kurtosis: 2.7075369295849523
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 16
;;; Elapsed time: 210.931678 ms
;;; Empirical mean: 188.7019999999999
;;; Empirical variance: 199.36719599999995
;;; Empirical std: 14.11974489854544
;;; Empirical skewness: 0.600842044716047
;;; Empirical kurtosis: 2.300424547483749
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 17
;;; Elapsed time: 213.651805 ms
;;; Empirical mean: 193.94299999999993
;;; Empirical variance: 124.36575100000006
;;; Empirical std: 11.15193933807031
;;; Empirical skewness: 0.22520108139138909
;;; Empirical kurtosis: 2.9510944113537008
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 18
;;; Elapsed time: 210.188428 ms
;;; Empirical mean: 194.26899999999995
;;; Empirical variance: 130.64663900000005
;;; Empirical std: 11.43007607148789
;;; Empirical skewness: 0.2565723129551272
;;; Empirical kurtosis: 2.487378444205876
;;; 
;;; Method :rmh with 1000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 19
;;; Elapsed time: 206.16404 ms
;;; Empirical mean: 194.10200000000006
;;; Empirical variance: 149.513596
;;; Empirical std: 12.227575229782886
;;; Empirical skewness: -0.29323899058390807
;;; Empirical kurtosis: 2.679356395916203
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 193.540264 ms
;;; Empirical mean: 194.405668202393
;;; Empirical variance: 162.4617648549793
;;; Empirical std: 12.746048989980357
;;; Empirical skewness: 0.10779238609461618
;;; Empirical kurtosis: 2.910394663617667
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 193.562838 ms
;;; Empirical mean: 194.3610763471998
;;; Empirical variance: 135.2117365511878
;;; Empirical std: 11.628058159090356
;;; Empirical skewness: 0.1691704735156661
;;; Empirical kurtosis: 3.042557460783047
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 193.601869 ms
;;; Empirical mean: 194.6785739953968
;;; Empirical variance: 149.68842840177206
;;; Empirical std: 12.23472224456984
;;; Empirical skewness: 0.13184676708275506
;;; Empirical kurtosis: 3.1241395562444487
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 189.702827 ms
;;; Empirical mean: 194.71861468302478
;;; Empirical variance: 156.33737126312775
;;; Empirical std: 12.50349436210245
;;; Empirical skewness: 0.031707035188857716
;;; Empirical kurtosis: 2.959594390621249
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 210.089241 ms
;;; Empirical mean: 194.50519039716653
;;; Empirical variance: 144.88289886998913
;;; Empirical std: 12.03673123692596
;;; Empirical skewness: 0.10620798418198701
;;; Empirical kurtosis: 2.9406333354235215
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 194.010721 ms
;;; Empirical mean: 194.06700098012155
;;; Empirical variance: 161.05279300242896
;;; Empirical std: 12.690657705667936
;;; Empirical skewness: 0.15961538678241385
;;; Empirical kurtosis: 3.077709515640261
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 225.320041 ms
;;; Empirical mean: 194.87365985337044
;;; Empirical variance: 160.33114766836948
;;; Empirical std: 12.662193635716108
;;; Empirical skewness: 0.015082748748598818
;;; Empirical kurtosis: 2.9057256834256546
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 199.325218 ms
;;; Empirical mean: 194.3467276087234
;;; Empirical variance: 158.45985942821568
;;; Empirical std: 12.58808402530805
;;; Empirical skewness: 6.293266733737075E-5
;;; Empirical kurtosis: 3.0836854994734346
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 215.295047 ms
;;; Empirical mean: 194.62592017658943
;;; Empirical variance: 150.5108504147047
;;; Empirical std: 12.268286368303631
;;; Empirical skewness: 0.06896042162446982
;;; Empirical kurtosis: 3.0302205075317685
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 201.992338 ms
;;; Empirical mean: 194.45324180703435
;;; Empirical variance: 160.49129166267537
;;; Empirical std: 12.668515763998377
;;; Empirical skewness: 0.013842338324911866
;;; Empirical kurtosis: 3.0039695789116063
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 218.506054 ms
;;; Empirical mean: 194.24205840519207
;;; Empirical variance: 141.84380256959844
;;; Empirical std: 11.909819585938253
;;; Empirical skewness: 0.010503645091031875
;;; Empirical kurtosis: 3.3969164400690612
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 218.695184 ms
;;; Empirical mean: 194.4123100120818
;;; Empirical variance: 136.59586169382453
;;; Empirical std: 11.687423227291143
;;; Empirical skewness: 0.10175164347075959
;;; Empirical kurtosis: 3.160998715113908
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 212.407457 ms
;;; Empirical mean: 194.85811791972787
;;; Empirical variance: 155.73520331557978
;;; Empirical std: 12.479391143624747
;;; Empirical skewness: 0.08954948470738475
;;; Empirical kurtosis: 2.9610663343557326
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 205.787203 ms
;;; Empirical mean: 194.22746848644945
;;; Empirical variance: 142.2325304554495
;;; Empirical std: 11.926128057984682
;;; Empirical skewness: 0.08807000057530814
;;; Empirical kurtosis: 3.1731795548917545
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 243.526979 ms
;;; Empirical mean: 194.48420585431117
;;; Empirical variance: 153.4115824885026
;;; Empirical std: 12.38594293901367
;;; Empirical skewness: 0.05203313941987394
;;; Empirical kurtosis: 2.875106610040692
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 193.062818 ms
;;; Empirical mean: 194.91356664310482
;;; Empirical variance: 154.1315021338715
;;; Empirical std: 12.414970887354972
;;; Empirical skewness: 0.025772638060662025
;;; Empirical kurtosis: 3.0568997451570654
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 191.279842 ms
;;; Empirical mean: 193.37852991424654
;;; Empirical variance: 156.03233495345518
;;; Empirical std: 12.491290363827718
;;; Empirical skewness: 0.060951880621033176
;;; Empirical kurtosis: 3.058647207539141
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 195.995273 ms
;;; Empirical mean: 194.00922544046702
;;; Empirical variance: 146.00420055890214
;;; Empirical std: 12.083219792708487
;;; Empirical skewness: 0.04564112029381257
;;; Empirical kurtosis: 3.2068873983062653
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 212.907404 ms
;;; Empirical mean: 194.24706368035595
;;; Empirical variance: 154.61806903082046
;;; Empirical std: 12.434551420570847
;;; Empirical skewness: 0.17055654402110354
;;; Empirical kurtosis: 2.8819053981834886
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 192.55261 ms
;;; Empirical mean: 193.2973915589823
;;; Empirical variance: 151.55017592947314
;;; Empirical std: 12.310571714159872
;;; Empirical skewness: 0.13253447440265043
;;; Empirical kurtosis: 2.9079051143126007
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 0
;;; Elapsed time: 323.321756 ms
;;; Empirical mean: 194.91118424704518
;;; Empirical variance: 150.9437798692252
;;; Empirical std: 12.2859179498003
;;; Empirical skewness: 0.18537025937276097
;;; Empirical kurtosis: 3.010175082691669
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 1
;;; Elapsed time: 311.119585 ms
;;; Empirical mean: 193.655610771574
;;; Empirical variance: 152.51429536332438
;;; Empirical std: 12.34966782400743
;;; Empirical skewness: 0.22235922974953456
;;; Empirical kurtosis: 2.9537925159925082
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 2
;;; Elapsed time: 281.974203 ms
;;; Empirical mean: 194.32831381938243
;;; Empirical variance: 146.31374687868467
;;; Empirical std: 12.096021944370168
;;; Empirical skewness: 0.055123925640221325
;;; Empirical kurtosis: 2.9445112783678042
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 3
;;; Elapsed time: 279.67762 ms
;;; Empirical mean: 194.48766732827445
;;; Empirical variance: 164.50760709569064
;;; Empirical std: 12.826051890417824
;;; Empirical skewness: 0.04456473178065176
;;; Empirical kurtosis: 2.7556916952905657
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 4
;;; Elapsed time: 283.227534 ms
;;; Empirical mean: 194.09964812507414
;;; Empirical variance: 166.06186365569746
;;; Empirical std: 12.886499278535558
;;; Empirical skewness: 0.10102735504884224
;;; Empirical kurtosis: 2.9609526331425404
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 5
;;; Elapsed time: 282.242794 ms
;;; Empirical mean: 194.51566157242297
;;; Empirical variance: 157.0206469908147
;;; Empirical std: 12.530787963684276
;;; Empirical skewness: 0.17713810546707853
;;; Empirical kurtosis: 3.129442806735164
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 6
;;; Elapsed time: 284.249061 ms
;;; Empirical mean: 194.4288389079103
;;; Empirical variance: 158.23132946248288
;;; Empirical std: 12.579003516275957
;;; Empirical skewness: 0.028056377408888566
;;; Empirical kurtosis: 3.159831196105892
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 7
;;; Elapsed time: 288.62221 ms
;;; Empirical mean: 194.20233194493977
;;; Empirical variance: 170.31565238603105
;;; Empirical std: 13.05050391310738
;;; Empirical skewness: 0.134140103241502
;;; Empirical kurtosis: 3.085373893770719
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 8
;;; Elapsed time: 287.565433 ms
;;; Empirical mean: 194.51306237025528
;;; Empirical variance: 152.1910827989851
;;; Empirical std: 12.336575002770626
;;; Empirical skewness: -0.06499094575296083
;;; Empirical kurtosis: 3.078495781933698
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 9
;;; Elapsed time: 293.881512 ms
;;; Empirical mean: 194.23841050050572
;;; Empirical variance: 159.55987484520753
;;; Empirical std: 12.631701185715546
;;; Empirical skewness: 0.050649643932727306
;;; Empirical kurtosis: 2.9403478181548
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 10
;;; Elapsed time: 286.116854 ms
;;; Empirical mean: 194.6892893737245
;;; Empirical variance: 152.88628231698655
;;; Empirical std: 12.364719257507893
;;; Empirical skewness: 0.004097794594250115
;;; Empirical kurtosis: 3.043713159022087
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 11
;;; Elapsed time: 287.664814 ms
;;; Empirical mean: 194.41009247889883
;;; Empirical variance: 159.9892606977227
;;; Empirical std: 12.648686125353997
;;; Empirical skewness: 0.11825674190194264
;;; Empirical kurtosis: 2.975046920159165
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 12
;;; Elapsed time: 289.800496 ms
;;; Empirical mean: 194.40997496509095
;;; Empirical variance: 148.86330485913703
;;; Empirical std: 12.200955079793427
;;; Empirical skewness: 0.12597055697247164
;;; Empirical kurtosis: 3.1552140122600427
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 13
;;; Elapsed time: 305.745989 ms
;;; Empirical mean: 194.22352958806343
;;; Empirical variance: 162.35729156234035
;;; Empirical std: 12.741950069056948
;;; Empirical skewness: -0.09700976746499491
;;; Empirical kurtosis: 2.7764530166139565
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 14
;;; Elapsed time: 280.295292 ms
;;; Empirical mean: 194.10691794294445
;;; Empirical variance: 153.57935242467101
;;; Empirical std: 12.392713682832788
;;; Empirical skewness: 0.07219757935265864
;;; Empirical kurtosis: 3.1755941654551973
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 15
;;; Elapsed time: 280.124711 ms
;;; Empirical mean: 193.88257365948232
;;; Empirical variance: 156.79849007610557
;;; Empirical std: 12.521920382916734
;;; Empirical skewness: 0.009563260734835642
;;; Empirical kurtosis: 2.8597961580993028
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 16
;;; Elapsed time: 276.41013 ms
;;; Empirical mean: 193.83450097497467
;;; Empirical variance: 150.94528021044877
;;; Empirical std: 12.285979009035005
;;; Empirical skewness: 0.18699104311476575
;;; Empirical kurtosis: 3.252876561632113
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 17
;;; Elapsed time: 294.708715 ms
;;; Empirical mean: 194.06950290893062
;;; Empirical variance: 142.81763117810496
;;; Empirical std: 11.950633086916566
;;; Empirical skewness: 0.12614740793287382
;;; Empirical kurtosis: 2.908037370310749
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 18
;;; Elapsed time: 277.146261 ms
;;; Empirical mean: 194.13093282662678
;;; Empirical variance: 145.13353818151992
;;; Empirical std: 12.047138173919976
;;; Empirical skewness: 9.54443880370059E-4
;;; Empirical kurtosis: 2.9706664888412946
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 19
;;; Elapsed time: 282.827198 ms
;;; Empirical mean: 194.53247636446267
;;; Empirical variance: 151.97653949215552
;;; Empirical std: 12.32787651999141
;;; Empirical skewness: 0.10452066142698177
;;; Empirical kurtosis: 3.097559125225714
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 145.641072 ms
;;; Empirical mean: 195.4350000000001
;;; Empirical variance: 144.8817750000001
;;; Empirical std: 12.03668455181908
;;; Empirical skewness: 0.368185883017626
;;; Empirical kurtosis: 3.6606523289569677
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 142.812943 ms
;;; Empirical mean: 193.644
;;; Empirical variance: 162.03726400000002
;;; Empirical std: 12.729385845357976
;;; Empirical skewness: 0.1035767021354651
;;; Empirical kurtosis: 2.765645858442165
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 145.775724 ms
;;; Empirical mean: 193.11599999999996
;;; Empirical variance: 150.77854400000007
;;; Empirical std: 12.279191504329594
;;; Empirical skewness: 0.13116476216126172
;;; Empirical kurtosis: 2.766425210470528
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 145.086093 ms
;;; Empirical mean: 195.483
;;; Empirical variance: 147.79971100000003
;;; Empirical std: 12.157290446476964
;;; Empirical skewness: -0.017840980945886358
;;; Empirical kurtosis: 2.5317025708147947
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 144.207508 ms
;;; Empirical mean: 193.8879999999999
;;; Empirical variance: 147.401456
;;; Empirical std: 12.140900131374115
;;; Empirical skewness: 0.08020752595787563
;;; Empirical kurtosis: 3.0926469255220455
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 146.689478 ms
;;; Empirical mean: 194.8269999999999
;;; Empirical variance: 160.08107099999992
;;; Empirical std: 12.652314847489368
;;; Empirical skewness: 0.011245332919162377
;;; Empirical kurtosis: 2.5468996339801073
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 144.47453 ms
;;; Empirical mean: 193.77100000000007
;;; Empirical variance: 147.18055900000004
;;; Empirical std: 12.131799495540637
;;; Empirical skewness: -0.07169156246050723
;;; Empirical kurtosis: 3.1180418574914035
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 141.88369 ms
;;; Empirical mean: 194.60299999999998
;;; Empirical variance: 140.70739099999997
;;; Empirical std: 11.862014626529508
;;; Empirical skewness: 0.189215240337945
;;; Empirical kurtosis: 3.0376460905651275
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 145.143579 ms
;;; Empirical mean: 195.04200000000006
;;; Empirical variance: 150.370236
;;; Empirical std: 12.262554220063617
;;; Empirical skewness: 0.20682709668176497
;;; Empirical kurtosis: 2.8937752521013174
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 144.68847 ms
;;; Empirical mean: 193.37500000000009
;;; Empirical variance: 136.9263750000001
;;; Empirical std: 11.701554383926952
;;; Empirical skewness: -0.009073423830610817
;;; Empirical kurtosis: 2.9211452722287996
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 144.332637 ms
;;; Empirical mean: 193.95899999999995
;;; Empirical variance: 152.5793189999999
;;; Empirical std: 12.352300150174457
;;; Empirical skewness: 0.12227603710815474
;;; Empirical kurtosis: 2.724679653649876
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 149.040706 ms
;;; Empirical mean: 194.10600000000002
;;; Empirical variance: 158.37076400000007
;;; Empirical std: 12.584544648099115
;;; Empirical skewness: 0.23900611224953966
;;; Empirical kurtosis: 3.232556466311096
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 146.447357 ms
;;; Empirical mean: 193.75999999999996
;;; Empirical variance: 158.2944
;;; Empirical std: 12.581510243210074
;;; Empirical skewness: 0.08847827384566306
;;; Empirical kurtosis: 2.8186132210175554
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 144.078304 ms
;;; Empirical mean: 195.73200000000003
;;; Empirical variance: 180.5881760000001
;;; Empirical std: 13.438310012795512
;;; Empirical skewness: 0.2020422629826742
;;; Empirical kurtosis: 2.857952260161436
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 150.550609 ms
;;; Empirical mean: 194.84899999999993
;;; Empirical variance: 153.18219899999997
;;; Empirical std: 12.376679643587773
;;; Empirical skewness: -0.08237939316124493
;;; Empirical kurtosis: 2.9399825777050124
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 147.876361 ms
;;; Empirical mean: 194.0769999999999
;;; Empirical variance: 143.795071
;;; Empirical std: 11.991458251605598
;;; Empirical skewness: -0.058671564819856796
;;; Empirical kurtosis: 2.9337305960716393
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 143.824479 ms
;;; Empirical mean: 193.68600000000006
;;; Empirical variance: 152.89740400000005
;;; Empirical std: 12.365168983883724
;;; Empirical skewness: -0.04482787791103789
;;; Empirical kurtosis: 3.1273072180361057
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 147.283212 ms
;;; Empirical mean: 193.83699999999996
;;; Empirical variance: 153.00243100000003
;;; Empirical std: 12.36941514381339
;;; Empirical skewness: 0.06290955725072389
;;; Empirical kurtosis: 3.1532227226450362
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 145.823613 ms
;;; Empirical mean: 195.05799999999996
;;; Empirical variance: 146.57663599999998
;;; Empirical std: 12.106883826980416
;;; Empirical skewness: -0.016709545042955094
;;; Empirical kurtosis: 2.981890666131616
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 141.573107 ms
;;; Empirical mean: 195.21099999999998
;;; Empirical variance: 150.18847900000006
;;; Empirical std: 12.255140921262393
;;; Empirical skewness: 0.06840030434911185
;;; Empirical kurtosis: 2.7662945996122565
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 0
;;; Elapsed time: 323.629429 ms
;;; Empirical mean: 195.2979999999999
;;; Empirical variance: 153.11119599999998
;;; Empirical std: 12.373810892364567
;;; Empirical skewness: 0.004584751071987473
;;; Empirical kurtosis: 2.8577145998378217
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 1
;;; Elapsed time: 304.43888 ms
;;; Empirical mean: 194.69900000000013
;;; Empirical variance: 163.70039900000012
;;; Empirical std: 12.79454567384087
;;; Empirical skewness: 0.06741255818630426
;;; Empirical kurtosis: 3.0291584019892044
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 2
;;; Elapsed time: 303.428925 ms
;;; Empirical mean: 194.15100000000015
;;; Empirical variance: 157.86619900000014
;;; Empirical std: 12.564481644699878
;;; Empirical skewness: 0.10130449400438492
;;; Empirical kurtosis: 3.03404523649438
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 3
;;; Elapsed time: 306.095423 ms
;;; Empirical mean: 195.03999999999996
;;; Empirical variance: 151.38039999999998
;;; Empirical std: 12.30367424796349
;;; Empirical skewness: 0.10992452426711821
;;; Empirical kurtosis: 2.9027621132401245
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 4
;;; Elapsed time: 300.501621 ms
;;; Empirical mean: 193.91299999999993
;;; Empirical variance: 157.02543099999997
;;; Empirical std: 12.530978852428088
;;; Empirical skewness: -0.14134532602653765
;;; Empirical kurtosis: 2.954776838730985
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 5
;;; Elapsed time: 308.880631 ms
;;; Empirical mean: 193.848
;;; Empirical variance: 155.78289599999997
;;; Empirical std: 12.481301855175204
;;; Empirical skewness: 0.0413493182425957
;;; Empirical kurtosis: 2.7998224265501093
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 6
;;; Elapsed time: 307.704828 ms
;;; Empirical mean: 194.65600000000003
;;; Empirical variance: 148.18566399999997
;;; Empirical std: 12.173153412325007
;;; Empirical skewness: 0.12457894263020321
;;; Empirical kurtosis: 2.933831666777182
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 7
;;; Elapsed time: 304.119806 ms
;;; Empirical mean: 194.02400000000014
;;; Empirical variance: 147.7894240000001
;;; Empirical std: 12.156867359644922
;;; Empirical skewness: 0.1741151749119691
;;; Empirical kurtosis: 3.2506403355077658
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 8
;;; Elapsed time: 302.223625 ms
;;; Empirical mean: 194.465
;;; Empirical variance: 156.306775
;;; Empirical std: 12.50227079373983
;;; Empirical skewness: 0.1087645237751994
;;; Empirical kurtosis: 3.059622907437248
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 9
;;; Elapsed time: 306.020303 ms
;;; Empirical mean: 194.09799999999996
;;; Empirical variance: 151.82239599999994
;;; Empirical std: 12.321623107366982
;;; Empirical skewness: 0.14256989931855296
;;; Empirical kurtosis: 3.2570313571530476
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 10
;;; Elapsed time: 303.948964 ms
;;; Empirical mean: 193.94600000000014
;;; Empirical variance: 160.23708400000007
;;; Empirical std: 12.658478739564249
;;; Empirical skewness: 0.05488803277330632
;;; Empirical kurtosis: 3.133622578599911
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 11
;;; Elapsed time: 307.537463 ms
;;; Empirical mean: 194.181
;;; Empirical variance: 145.03623900000002
;;; Empirical std: 12.04309922735838
;;; Empirical skewness: 0.07950965827028149
;;; Empirical kurtosis: 2.9547014993048535
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 12
;;; Elapsed time: 299.853716 ms
;;; Empirical mean: 194.67800000000003
;;; Empirical variance: 156.45031600000007
;;; Empirical std: 12.508010073548872
;;; Empirical skewness: 0.062454651571325084
;;; Empirical kurtosis: 2.882679349197101
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 13
;;; Elapsed time: 314.835738 ms
;;; Empirical mean: 194.48100000000008
;;; Empirical variance: 150.271639
;;; Empirical std: 12.258533313573855
;;; Empirical skewness: 0.0556556048499224
;;; Empirical kurtosis: 2.9395676686712244
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 14
;;; Elapsed time: 304.149209 ms
;;; Empirical mean: 194.45500000000004
;;; Empirical variance: 149.303975
;;; Empirical std: 12.219000572878292
;;; Empirical skewness: 0.124001125663396
;;; Empirical kurtosis: 3.0857149223189135
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 15
;;; Elapsed time: 301.44957 ms
;;; Empirical mean: 194.22400000000002
;;; Empirical variance: 162.379824
;;; Empirical std: 12.742834221632172
;;; Empirical skewness: 0.09222184374423029
;;; Empirical kurtosis: 3.172139674128916
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 16
;;; Elapsed time: 302.707232 ms
;;; Empirical mean: 194.38100000000003
;;; Empirical variance: 155.90183900000002
;;; Empirical std: 12.486065793515587
;;; Empirical skewness: 0.08604224720774757
;;; Empirical kurtosis: 2.9044131071740584
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 17
;;; Elapsed time: 301.439863 ms
;;; Empirical mean: 194.14399999999998
;;; Empirical variance: 141.20326400000008
;;; Empirical std: 11.882897963039154
;;; Empirical skewness: 0.28527329942497054
;;; Empirical kurtosis: 3.1654489730641764
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 18
;;; Elapsed time: 306.323093 ms
;;; Empirical mean: 194.99900000000002
;;; Empirical variance: 153.93099900000004
;;; Empirical std: 12.406893204988911
;;; Empirical skewness: 0.07111760619047051
;;; Empirical kurtosis: 3.021694254927906
;;; 
;;; Method :pgibbs with 1000 samples and options [:number-of-particles 1000]
;;; Chain no. 19
;;; Elapsed time: 302.203351 ms
;;; Empirical mean: 194.36499999999998
;;; Empirical variance: 154.599775
;;; Empirical std: 12.43381578599265
;;; Empirical skewness: 0.03321478316866591
;;; Empirical kurtosis: 3.0810685651504865
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 83.211748 ms
;;; Empirical mean: 194.53671509476257
;;; Empirical variance: 179.66999067197008
;;; Empirical std: 13.404103501240584
;;; Empirical skewness: -0.2687607576980913
;;; Empirical kurtosis: 2.984964729566026
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 78.3561 ms
;;; Empirical mean: 193.93516653100943
;;; Empirical variance: 139.05402796670884
;;; Empirical std: 11.792117196106425
;;; Empirical skewness: 0.5096571058269231
;;; Empirical kurtosis: 3.4667476972497036
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 78.349239 ms
;;; Empirical mean: 194.09617546166666
;;; Empirical variance: 155.79359527926647
;;; Empirical std: 12.481730460127173
;;; Empirical skewness: -0.3779336451205923
;;; Empirical kurtosis: 3.5827924045398283
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 79.99883 ms
;;; Empirical mean: 193.48412846127846
;;; Empirical variance: 154.34084792051144
;;; Empirical std: 12.423399209576719
;;; Empirical skewness: -0.1431376179921443
;;; Empirical kurtosis: 2.918215463173178
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 77.16139 ms
;;; Empirical mean: 194.81206649999288
;;; Empirical variance: 131.4020510845685
;;; Empirical std: 11.463073369937423
;;; Empirical skewness: -0.12714759623397504
;;; Empirical kurtosis: 3.3004816725143127
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 75.368704 ms
;;; Empirical mean: 194.0246697104507
;;; Empirical variance: 150.6025736682907
;;; Empirical std: 12.272024024923137
;;; Empirical skewness: -0.06041835964481207
;;; Empirical kurtosis: 2.583958723766577
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 72.003371 ms
;;; Empirical mean: 191.4729489128332
;;; Empirical variance: 146.37390109755262
;;; Empirical std: 12.098508217856969
;;; Empirical skewness: 0.2152586979414164
;;; Empirical kurtosis: 2.963600749674508
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 77.461029 ms
;;; Empirical mean: 193.43509627587986
;;; Empirical variance: 155.44437245289652
;;; Empirical std: 12.467733252395822
;;; Empirical skewness: 0.05597218581506626
;;; Empirical kurtosis: 3.0893791199573615
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 72.545084 ms
;;; Empirical mean: 193.309603127483
;;; Empirical variance: 149.55182982600857
;;; Empirical std: 12.2291385561702
;;; Empirical skewness: 0.1926002929206084
;;; Empirical kurtosis: 3.2041334636497045
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 72.909781 ms
;;; Empirical mean: 194.48554390672118
;;; Empirical variance: 138.3959441760512
;;; Empirical std: 11.764180556930057
;;; Empirical skewness: 0.08108360798900982
;;; Empirical kurtosis: 3.2015025294102895
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 72.700186 ms
;;; Empirical mean: 193.50657044350652
;;; Empirical variance: 139.55282863384068
;;; Empirical std: 11.813248013727668
;;; Empirical skewness: 0.006684484249416198
;;; Empirical kurtosis: 2.5466456103799096
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 75.79203 ms
;;; Empirical mean: 193.3886303564959
;;; Empirical variance: 139.76165801971672
;;; Empirical std: 11.822083488950529
;;; Empirical skewness: -0.11537580352978676
;;; Empirical kurtosis: 3.085886033835079
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 77.924315 ms
;;; Empirical mean: 193.8740546866368
;;; Empirical variance: 134.0579950454475
;;; Empirical std: 11.57834163623822
;;; Empirical skewness: -0.06046266634852232
;;; Empirical kurtosis: 3.402733181589744
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 79.47246 ms
;;; Empirical mean: 194.3865987155637
;;; Empirical variance: 141.29656807523043
;;; Empirical std: 11.8868232962062
;;; Empirical skewness: 0.256091261571073
;;; Empirical kurtosis: 3.0084520966523227
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 71.017717 ms
;;; Empirical mean: 193.6216310959545
;;; Empirical variance: 166.1957171741167
;;; Empirical std: 12.89169178867214
;;; Empirical skewness: 0.22494688364329682
;;; Empirical kurtosis: 3.397284765370927
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 77.277115 ms
;;; Empirical mean: 194.38286207682992
;;; Empirical variance: 131.70143493789342
;;; Empirical std: 11.476124560926193
;;; Empirical skewness: 0.2009788799243137
;;; Empirical kurtosis: 3.4801094618006774
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 76.741389 ms
;;; Empirical mean: 194.67706349989632
;;; Empirical variance: 168.0737085197539
;;; Empirical std: 12.96432445288816
;;; Empirical skewness: -0.14420139294764564
;;; Empirical kurtosis: 2.825109827542403
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 76.300393 ms
;;; Empirical mean: 194.92547265650646
;;; Empirical variance: 150.66661145190787
;;; Empirical std: 12.274632843873901
;;; Empirical skewness: 0.2620365242750897
;;; Empirical kurtosis: 3.026420822618421
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 75.915667 ms
;;; Empirical mean: 192.10033538092085
;;; Empirical variance: 126.79198459687248
;;; Empirical std: 11.260194696224062
;;; Empirical skewness: 0.26005285394966043
;;; Empirical kurtosis: 3.244127057275095
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 79.2334 ms
;;; Empirical mean: 192.52142333567318
;;; Empirical variance: 128.69195751730115
;;; Empirical std: 11.344247772210423
;;; Empirical skewness: 0.25875852805432925
;;; Empirical kurtosis: 2.962063387756957
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 0
;;; Elapsed time: 1248.339704 ms
;;; Empirical mean: 194.20150496767937
;;; Empirical variance: 147.0414050832159
;;; Empirical std: 12.126063049614078
;;; Empirical skewness: 0.13042859730940526
;;; Empirical kurtosis: 3.1043092483412598
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 1
;;; Elapsed time: 1304.813833 ms
;;; Empirical mean: 194.36467010270925
;;; Empirical variance: 161.12594817415632
;;; Empirical std: 12.693539623531189
;;; Empirical skewness: 0.06663244123438235
;;; Empirical kurtosis: 2.948120423624023
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 2
;;; Elapsed time: 1257.771161 ms
;;; Empirical mean: 194.71149454028077
;;; Empirical variance: 150.37491500259347
;;; Empirical std: 12.26274500275503
;;; Empirical skewness: 0.051134090264482394
;;; Empirical kurtosis: 2.7351411240310393
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 3
;;; Elapsed time: 1283.396489 ms
;;; Empirical mean: 194.34330731287602
;;; Empirical variance: 154.63849341481105
;;; Empirical std: 12.435372668915518
;;; Empirical skewness: 0.03687226986647022
;;; Empirical kurtosis: 3.0260090580181096
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 4
;;; Elapsed time: 1390.043817 ms
;;; Empirical mean: 194.26498664383405
;;; Empirical variance: 149.6856401848082
;;; Empirical std: 12.234608297154765
;;; Empirical skewness: 0.09126543356493089
;;; Empirical kurtosis: 2.8684680948565724
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 5
;;; Elapsed time: 1331.047618 ms
;;; Empirical mean: 193.63742541502887
;;; Empirical variance: 161.25313737564426
;;; Empirical std: 12.698548632644766
;;; Empirical skewness: 0.08590504946926576
;;; Empirical kurtosis: 3.183701625420441
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 6
;;; Elapsed time: 1312.91248 ms
;;; Empirical mean: 194.08252879481375
;;; Empirical variance: 141.69378346351456
;;; Empirical std: 11.90351979304922
;;; Empirical skewness: -0.1044002918790357
;;; Empirical kurtosis: 2.890843896731922
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 7
;;; Elapsed time: 1348.623671 ms
;;; Empirical mean: 193.96919685116066
;;; Empirical variance: 139.95031800102433
;;; Empirical std: 11.8300599322668
;;; Empirical skewness: -0.003201009666406889
;;; Empirical kurtosis: 3.2372410453351357
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 8
;;; Elapsed time: 1584.091417 ms
;;; Empirical mean: 193.44533259547367
;;; Empirical variance: 146.744064116546
;;; Empirical std: 12.11379643697821
;;; Empirical skewness: 0.10626859192068962
;;; Empirical kurtosis: 2.9124651167418585
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 9
;;; Elapsed time: 1324.917951 ms
;;; Empirical mean: 194.18478391518718
;;; Empirical variance: 152.03219205083923
;;; Empirical std: 12.330133496878256
;;; Empirical skewness: 0.19584314123642121
;;; Empirical kurtosis: 3.4859711497980843
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 10
;;; Elapsed time: 1407.607191 ms
;;; Empirical mean: 194.86213104072763
;;; Empirical variance: 147.31242029441037
;;; Empirical std: 12.137232810423074
;;; Empirical skewness: -0.0628046598150596
;;; Empirical kurtosis: 2.8846875433983143
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 11
;;; Elapsed time: 1361.314808 ms
;;; Empirical mean: 194.2128131777311
;;; Empirical variance: 155.75380380603093
;;; Empirical std: 12.480136369688871
;;; Empirical skewness: 0.054807683080258206
;;; Empirical kurtosis: 2.6229874056592557
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 12
;;; Elapsed time: 1567.648638 ms
;;; Empirical mean: 193.9894019965522
;;; Empirical variance: 162.84772812326065
;;; Empirical std: 12.761180514484568
;;; Empirical skewness: -0.027380650590410453
;;; Empirical kurtosis: 3.0082821325108875
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 13
;;; Elapsed time: 1439.032534 ms
;;; Empirical mean: 194.46414133303708
;;; Empirical variance: 151.4730570811988
;;; Empirical std: 12.307439095165119
;;; Empirical skewness: 0.06668420057867762
;;; Empirical kurtosis: 2.9782816600465054
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 14
;;; Elapsed time: 1356.784573 ms
;;; Empirical mean: 194.3289774234577
;;; Empirical variance: 151.4566226620885
;;; Empirical std: 12.30677141504174
;;; Empirical skewness: 0.0764086639921888
;;; Empirical kurtosis: 3.0551987639592078
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 15
;;; Elapsed time: 1348.872059 ms
;;; Empirical mean: 193.79314287535976
;;; Empirical variance: 138.08231542351808
;;; Empirical std: 11.750843179258162
;;; Empirical skewness: 0.14412108359673
;;; Empirical kurtosis: 3.2205276164474164
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 16
;;; Elapsed time: 1373.010175 ms
;;; Empirical mean: 194.678479050753
;;; Empirical variance: 154.9600309473855
;;; Empirical std: 12.448294298713598
;;; Empirical skewness: 0.1347544228977835
;;; Empirical kurtosis: 3.0089449907743653
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 17
;;; Elapsed time: 1308.077274 ms
;;; Empirical mean: 193.98555962523568
;;; Empirical variance: 150.68074655478944
;;; Empirical std: 12.275208615530305
;;; Empirical skewness: -0.019959714771778596
;;; Empirical kurtosis: 2.720013326473521
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 18
;;; Elapsed time: 1337.216006 ms
;;; Empirical mean: 195.01780951662863
;;; Empirical variance: 149.0986222024143
;;; Empirical std: 12.210594670302275
;;; Empirical skewness: 0.01799878553311754
;;; Empirical kurtosis: 2.6764290894948157
;;; 
;;; Method :ipmcmc with 1000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 19
;;; Elapsed time: 1315.019754 ms
;;; Empirical mean: 194.24511862351488
;;; Empirical variance: 164.96746050746293
;;; Empirical std: 12.84396591818364
;;; Empirical skewness: 0.13462299961686847
;;; Empirical kurtosis: 2.982349428986982
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 1500.365652 ms
;;; Empirical mean: 194.54398826578014
;;; Empirical variance: 154.61629048335837
;;; Empirical std: 12.434479904015221
;;; Empirical skewness: 0.07751047318795903
;;; Empirical kurtosis: 2.9486199666300754
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 1483.430171 ms
;;; Empirical mean: 194.34630696226375
;;; Empirical variance: 148.5731855217659
;;; Empirical std: 12.189060075402283
;;; Empirical skewness: 0.07607463487314979
;;; Empirical kurtosis: 3.11297810687355
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 1488.037113 ms
;;; Empirical mean: 194.21888679768585
;;; Empirical variance: 156.28978571745301
;;; Empirical std: 12.501591327405205
;;; Empirical skewness: 0.07284467766264655
;;; Empirical kurtosis: 2.930845677628694
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 1491.151192 ms
;;; Empirical mean: 194.01687104813226
;;; Empirical variance: 151.25600834304973
;;; Empirical std: 12.298618147704632
;;; Empirical skewness: 0.08341263494286992
;;; Empirical kurtosis: 3.0652816007020203
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 1496.882098 ms
;;; Empirical mean: 194.2991254790624
;;; Empirical variance: 151.14806820460873
;;; Empirical std: 12.294229061011054
;;; Empirical skewness: 0.08541564711125055
;;; Empirical kurtosis: 3.04834113642884
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 1507.878171 ms
;;; Empirical mean: 194.51253862275618
;;; Empirical variance: 151.98945370623753
;;; Empirical std: 12.32840028982826
;;; Empirical skewness: 0.11030928451853228
;;; Empirical kurtosis: 3.00155025625582
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 1497.894278 ms
;;; Empirical mean: 194.25999821367367
;;; Empirical variance: 154.49476868192275
;;; Empirical std: 12.429592458400347
;;; Empirical skewness: 0.0828936351384377
;;; Empirical kurtosis: 3.0387860681531977
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 1487.242052 ms
;;; Empirical mean: 194.43383721534323
;;; Empirical variance: 155.87684314652134
;;; Empirical std: 12.485064803457023
;;; Empirical skewness: 0.05472316324561075
;;; Empirical kurtosis: 2.9400387673185557
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 1495.516212 ms
;;; Empirical mean: 194.37600608794565
;;; Empirical variance: 148.397709395556
;;; Empirical std: 12.18185984961065
;;; Empirical skewness: 0.10233696450469858
;;; Empirical kurtosis: 3.0179221373907392
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 1490.236786 ms
;;; Empirical mean: 194.59985840788548
;;; Empirical variance: 151.5323638386623
;;; Empirical std: 12.309848245963973
;;; Empirical skewness: 0.0571190008146788
;;; Empirical kurtosis: 2.9572697759305915
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 1527.465469 ms
;;; Empirical mean: 194.38891015807437
;;; Empirical variance: 152.33903777972924
;;; Empirical std: 12.342570144817053
;;; Empirical skewness: 0.0500268547780071
;;; Empirical kurtosis: 3.0379315607958546
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 1485.939465 ms
;;; Empirical mean: 194.0521470572409
;;; Empirical variance: 154.21299964765964
;;; Empirical std: 12.418252680939442
;;; Empirical skewness: 0.02977856387461767
;;; Empirical kurtosis: 2.988689377511409
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 1495.735947 ms
;;; Empirical mean: 194.31079101111092
;;; Empirical variance: 150.64895727297565
;;; Empirical std: 12.273913690138759
;;; Empirical skewness: 0.0767193098277963
;;; Empirical kurtosis: 2.9715427465628528
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 1494.728 ms
;;; Empirical mean: 194.41344984285485
;;; Empirical variance: 153.14820171883338
;;; Empirical std: 12.375306126267478
;;; Empirical skewness: 0.07040842186816292
;;; Empirical kurtosis: 3.016502825966889
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 1500.736002 ms
;;; Empirical mean: 194.3087958706918
;;; Empirical variance: 153.96744143195875
;;; Empirical std: 12.408361754557237
;;; Empirical skewness: 0.10892745523304374
;;; Empirical kurtosis: 3.007858989210533
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 1496.170205 ms
;;; Empirical mean: 194.31099896078
;;; Empirical variance: 151.81094970158742
;;; Empirical std: 12.321158618473646
;;; Empirical skewness: 0.11996460792903497
;;; Empirical kurtosis: 2.9488982057680015
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 1503.47818 ms
;;; Empirical mean: 194.42667444041166
;;; Empirical variance: 148.25555150019377
;;; Empirical std: 12.17602363254087
;;; Empirical skewness: 0.09972901480423775
;;; Empirical kurtosis: 2.9811094071234208
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 1479.366427 ms
;;; Empirical mean: 194.3642932640407
;;; Empirical variance: 151.4417337420373
;;; Empirical std: 12.306166492536873
;;; Empirical skewness: 0.07109964865106581
;;; Empirical kurtosis: 2.9796283452154904
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 1488.240294 ms
;;; Empirical mean: 194.14825996136378
;;; Empirical variance: 149.38307735585073
;;; Empirical std: 12.222237002932431
;;; Empirical skewness: 0.07969242519993604
;;; Empirical kurtosis: 3.0934974693771715
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 1512.331507 ms
;;; Empirical mean: 194.14434391501138
;;; Empirical variance: 152.5923849898214
;;; Empirical std: 12.352829027790412
;;; Empirical skewness: 0.07406320168899105
;;; Empirical kurtosis: 3.0078048700989557
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 1318.272584 ms
;;; Empirical mean: 195.5999000000001
;;; Empirical variance: 159.1838199900001
;;; Empirical std: 12.616807044177227
;;; Empirical skewness: 0.12031423264273154
;;; Empirical kurtosis: 2.8900430586585264
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 1298.146328 ms
;;; Empirical mean: 193.6945
;;; Empirical variance: 152.3929697500001
;;; Empirical std: 12.344754746450013
;;; Empirical skewness: 0.12964043682799425
;;; Empirical kurtosis: 2.9071393257479823
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 1323.381885 ms
;;; Empirical mean: 194.32340000000005
;;; Empirical variance: 168.05561244000006
;;; Empirical std: 12.963626515755537
;;; Empirical skewness: -0.023042111413445168
;;; Empirical kurtosis: 2.817895494853766
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 1312.078322 ms
;;; Empirical mean: 194.2973999999999
;;; Empirical variance: 164.13615323999989
;;; Empirical std: 12.811563262927748
;;; Empirical skewness: 0.1905149628896556
;;; Empirical kurtosis: 3.042635864344492
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 1304.885451 ms
;;; Empirical mean: 194.2549999999999
;;; Empirical variance: 157.005175
;;; Empirical std: 12.53017058942136
;;; Empirical skewness: 0.08138063712251964
;;; Empirical kurtosis: 3.111396526863841
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 1326.597176 ms
;;; Empirical mean: 194.64170000000007
;;; Empirical variance: 160.19112111000013
;;; Empirical std: 12.656663111183773
;;; Empirical skewness: 0.024748559381937974
;;; Empirical kurtosis: 3.0050011067506857
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 1316.725089 ms
;;; Empirical mean: 193.63450000000006
;;; Empirical variance: 152.20770975000008
;;; Empirical std: 12.337248872824123
;;; Empirical skewness: 0.23004604796982747
;;; Empirical kurtosis: 3.31519636869312
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 1316.684226 ms
;;; Empirical mean: 194.3814
;;; Empirical variance: 151.28373404000007
;;; Empirical std: 12.299745283541448
;;; Empirical skewness: -0.00553740238359694
;;; Empirical kurtosis: 2.937045014992913
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 1323.703982 ms
;;; Empirical mean: 193.9937999999999
;;; Empirical variance: 156.03636156
;;; Empirical std: 12.491451539352822
;;; Empirical skewness: 0.2137058456689434
;;; Empirical kurtosis: 3.09475431807351
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 1310.541679 ms
;;; Empirical mean: 194.2884999999999
;;; Empirical variance: 144.96246774999994
;;; Empirical std: 12.040036036075637
;;; Empirical skewness: 0.05944554007644871
;;; Empirical kurtosis: 2.759889468443443
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 1334.597296 ms
;;; Empirical mean: 194.74440000000004
;;; Empirical variance: 140.38266864
;;; Empirical std: 11.84831923270132
;;; Empirical skewness: 0.09423390493439535
;;; Empirical kurtosis: 2.961289423930898
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 1316.202522 ms
;;; Empirical mean: 194.3556000000001
;;; Empirical variance: 141.5299486400001
;;; Empirical std: 11.896636022002191
;;; Empirical skewness: 0.06647114782700728
;;; Empirical kurtosis: 2.7452256705140132
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 1320.964861 ms
;;; Empirical mean: 195.11589999999998
;;; Empirical variance: 157.56566719000008
;;; Empirical std: 12.552516368840157
;;; Empirical skewness: 0.004675168060591104
;;; Empirical kurtosis: 2.899129282120895
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 1321.695946 ms
;;; Empirical mean: 194.42700000000008
;;; Empirical variance: 151.76167100000006
;;; Empirical std: 12.319158696924074
;;; Empirical skewness: 0.20278620544441558
;;; Empirical kurtosis: 3.134858638742715
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 1339.361059 ms
;;; Empirical mean: 193.5434
;;; Empirical variance: 148.03331644000002
;;; Empirical std: 12.166894280793272
;;; Empirical skewness: 0.04185219667336188
;;; Empirical kurtosis: 2.868247444664872
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 1360.058078 ms
;;; Empirical mean: 194.72059999999996
;;; Empirical variance: 148.17893563999996
;;; Empirical std: 12.172877048586335
;;; Empirical skewness: 0.048260302754404134
;;; Empirical kurtosis: 3.076860908691931
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 1346.705722 ms
;;; Empirical mean: 195.13869999999991
;;; Empirical variance: 151.63106231000003
;;; Empirical std: 12.313856516542655
;;; Empirical skewness: 0.04502035451064881
;;; Empirical kurtosis: 3.0301887666053844
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 1328.344057 ms
;;; Empirical mean: 193.83780000000002
;;; Empirical variance: 156.70689115999997
;;; Empirical std: 12.518262305927287
;;; Empirical skewness: -1.6208380330633878E-4
;;; Empirical kurtosis: 2.8337282491896825
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 1341.893787 ms
;;; Empirical mean: 194.77669999999995
;;; Empirical variance: 153.94683711000005
;;; Empirical std: 12.407531467217806
;;; Empirical skewness: 0.060937650808043316
;;; Empirical kurtosis: 3.253924023284029
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 1325.931903 ms
;;; Empirical mean: 194.29700000000003
;;; Empirical variance: 145.93579100000002
;;; Empirical std: 12.080388694077687
;;; Empirical skewness: 0.06350259993699148
;;; Empirical kurtosis: 2.8620585114608517
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 2020.428783 ms
;;; Empirical mean: 193.63160000000005
;;; Empirical variance: 146.2108814400001
;;; Empirical std: 12.091769160879647
;;; Empirical skewness: 0.13198206898682763
;;; Empirical kurtosis: 2.931478904550353
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 2000.958738 ms
;;; Empirical mean: 194.51140000000004
;;; Empirical variance: 128.16847004
;;; Empirical std: 11.32115144497237
;;; Empirical skewness: 0.09677999177698439
;;; Empirical kurtosis: 2.744017231807773
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 2011.973237 ms
;;; Empirical mean: 194.46460000000005
;;; Empirical variance: 160.16054684
;;; Empirical std: 12.65545522057583
;;; Empirical skewness: 0.0904329398860518
;;; Empirical kurtosis: 2.686839347374228
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 2029.84834 ms
;;; Empirical mean: 193.44140000000004
;;; Empirical variance: 145.40476604000014
;;; Empirical std: 12.058389861005496
;;; Empirical skewness: 0.3149215168542739
;;; Empirical kurtosis: 3.45145477028734
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 2008.805803 ms
;;; Empirical mean: 193.37510000000003
;;; Empirical variance: 153.58919999000003
;;; Empirical std: 12.393110989174591
;;; Empirical skewness: 0.03242946306159643
;;; Empirical kurtosis: 2.8279475333136896
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 2018.849869 ms
;;; Empirical mean: 193.90660000000005
;;; Empirical variance: 147.7716764400001
;;; Empirical std: 12.156137398038906
;;; Empirical skewness: 0.23761679236842254
;;; Empirical kurtosis: 3.3313450119866994
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 2003.98154 ms
;;; Empirical mean: 194.64870000000002
;;; Empirical variance: 141.00728831
;;; Empirical std: 11.874648976285574
;;; Empirical skewness: 0.185000361976286
;;; Empirical kurtosis: 2.874182016585092
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 2002.055208 ms
;;; Empirical mean: 192.5195
;;; Empirical variance: 140.89841975000004
;;; Empirical std: 11.870064016255348
;;; Empirical skewness: 0.21893426836811872
;;; Empirical kurtosis: 2.9291476426352365
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 2021.406117 ms
;;; Empirical mean: 193.30919999999983
;;; Empirical variance: 156.06079536000001
;;; Empirical std: 12.492429521914463
;;; Empirical skewness: 0.03060848571227083
;;; Empirical kurtosis: 2.85448803271402
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 2061.102647 ms
;;; Empirical mean: 194.58709999999996
;;; Empirical variance: 159.29621359000004
;;; Empirical std: 12.621260380405756
;;; Empirical skewness: 0.16008618437349703
;;; Empirical kurtosis: 3.005505557848098
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 1990.541141 ms
;;; Empirical mean: 195.06129999999993
;;; Empirical variance: 174.27854230999998
;;; Empirical std: 13.201459855258431
;;; Empirical skewness: 0.010862914338170602
;;; Empirical kurtosis: 3.0629163270994306
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 2023.157369 ms
;;; Empirical mean: 193.25969999999992
;;; Empirical variance: 147.74885590999997
;;; Empirical std: 12.155198719478015
;;; Empirical skewness: 0.12535962865576522
;;; Empirical kurtosis: 3.327460508217344
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 2030.819948 ms
;;; Empirical mean: 194.47709999999995
;;; Empirical variance: 170.90007559000006
;;; Empirical std: 13.072875566989845
;;; Empirical skewness: -0.003668490434966352
;;; Empirical kurtosis: 3.293334322684411
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 2010.031786 ms
;;; Empirical mean: 194.86200000000008
;;; Empirical variance: 159.09175600000012
;;; Empirical std: 12.61315805022676
;;; Empirical skewness: -0.002768587661526719
;;; Empirical kurtosis: 3.073029520689078
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 2005.312758 ms
;;; Empirical mean: 195.35529999999994
;;; Empirical variance: 151.33846191000004
;;; Empirical std: 12.30196983860715
;;; Empirical skewness: -0.10855507119395072
;;; Empirical kurtosis: 3.120080887394296
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 1995.151907 ms
;;; Empirical mean: 194.09290000000001
;;; Empirical variance: 151.3490695900001
;;; Empirical std: 12.302400968510176
;;; Empirical skewness: 0.046200376967891456
;;; Empirical kurtosis: 2.9563746794691386
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 1980.27423 ms
;;; Empirical mean: 193.96879999999996
;;; Empirical variance: 141.70582656000005
;;; Empirical std: 11.904025645133668
;;; Empirical skewness: -0.12306421341501439
;;; Empirical kurtosis: 3.137570637172502
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 2016.531712 ms
;;; Empirical mean: 194.71669999999997
;;; Empirical variance: 150.2344411100001
;;; Empirical std: 12.257015995339163
;;; Empirical skewness: 0.07125211252955613
;;; Empirical kurtosis: 2.75941880862697
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 2006.134723 ms
;;; Empirical mean: 193.23559999999998
;;; Empirical variance: 144.92189263999992
;;; Empirical std: 12.038350910319899
;;; Empirical skewness: 0.17820225731081052
;;; Empirical kurtosis: 2.864535127964694
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 2042.924962 ms
;;; Empirical mean: 194.64869999999996
;;; Empirical variance: 160.77208831000007
;;; Empirical std: 12.679593381098625
;;; Empirical skewness: 0.15768666296965478
;;; Empirical kurtosis: 3.1480466743968467
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 0
;;; Elapsed time: 2007.125204 ms
;;; Empirical mean: 194.96770000000006
;;; Empirical variance: 159.65985671000013
;;; Empirical std: 12.635658143128126
;;; Empirical skewness: 0.07832963053246574
;;; Empirical kurtosis: 2.426761349633635
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 1
;;; Elapsed time: 2016.109644 ms
;;; Empirical mean: 194.3853000000001
;;; Empirical variance: 143.58204391000004
;;; Empirical std: 11.982572508021807
;;; Empirical skewness: 0.05558523791899666
;;; Empirical kurtosis: 3.03057447036132
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 2
;;; Elapsed time: 2039.282071 ms
;;; Empirical mean: 193.74360000000013
;;; Empirical variance: 148.4758590400001
;;; Empirical std: 12.185067051108094
;;; Empirical skewness: 0.27171659651652763
;;; Empirical kurtosis: 3.2013255206331555
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 3
;;; Elapsed time: 2028.943186 ms
;;; Empirical mean: 191.92489999999995
;;; Empirical variance: 152.84065999000012
;;; Empirical std: 12.36287426086669
;;; Empirical skewness: 0.10986527982185244
;;; Empirical kurtosis: 2.5142007442111303
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 4
;;; Elapsed time: 1978.516522 ms
;;; Empirical mean: 195.0293
;;; Empirical variance: 163.62704151000003
;;; Empirical std: 12.79167860407695
;;; Empirical skewness: 0.1372342105743854
;;; Empirical kurtosis: 2.759799493699109
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 5
;;; Elapsed time: 2032.528889 ms
;;; Empirical mean: 193.02169999999992
;;; Empirical variance: 123.24342910999994
;;; Empirical std: 11.101505713640828
;;; Empirical skewness: -0.0472324525682292
;;; Empirical kurtosis: 2.6572758513042
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 6
;;; Elapsed time: 2038.39734 ms
;;; Empirical mean: 191.81569999999994
;;; Empirical variance: 161.90813350999997
;;; Empirical std: 12.724312693029827
;;; Empirical skewness: -0.13039645742218714
;;; Empirical kurtosis: 2.743817419447834
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 7
;;; Elapsed time: 1989.964822 ms
;;; Empirical mean: 194.81469999999996
;;; Empirical variance: 115.91656391000001
;;; Empirical std: 10.76645549426551
;;; Empirical skewness: 0.0366571269936069
;;; Empirical kurtosis: 3.1934485783300746
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 8
;;; Elapsed time: 2036.07725 ms
;;; Empirical mean: 192.74400000000006
;;; Empirical variance: 140.58406399999998
;;; Empirical std: 11.856815086691704
;;; Empirical skewness: 0.07798149643489258
;;; Empirical kurtosis: 2.7511434297980886
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 9
;;; Elapsed time: 2011.15083 ms
;;; Empirical mean: 193.1316
;;; Empirical variance: 125.31768144000002
;;; Empirical std: 11.194538018158678
;;; Empirical skewness: 0.03943912142007314
;;; Empirical kurtosis: 2.988818240049711
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 10
;;; Elapsed time: 2013.473408 ms
;;; Empirical mean: 193.11300000000003
;;; Empirical variance: 146.93863100000007
;;; Empirical std: 12.12182457388326
;;; Empirical skewness: 0.1662468531656315
;;; Empirical kurtosis: 3.0506055644108496
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 11
;;; Elapsed time: 1994.336285 ms
;;; Empirical mean: 193.89860000000004
;;; Empirical variance: 145.11291804000004
;;; Empirical std: 12.04628233273652
;;; Empirical skewness: 0.0726468084655918
;;; Empirical kurtosis: 3.0087044850220495
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 12
;;; Elapsed time: 2019.942488 ms
;;; Empirical mean: 193.44650000000016
;;; Empirical variance: 154.8867377500001
;;; Empirical std: 12.445350045298047
;;; Empirical skewness: 0.22695218701536488
;;; Empirical kurtosis: 3.0792049410608717
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 13
;;; Elapsed time: 1992.509907 ms
;;; Empirical mean: 194.61950000000004
;;; Empirical variance: 141.8615197500001
;;; Empirical std: 11.910563368287837
;;; Empirical skewness: 0.017399759052532097
;;; Empirical kurtosis: 2.7510540315904883
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 14
;;; Elapsed time: 2011.400408 ms
;;; Empirical mean: 193.36790000000005
;;; Empirical variance: 171.69534959000015
;;; Empirical std: 13.103257212998612
;;; Empirical skewness: 0.2863840419638861
;;; Empirical kurtosis: 3.158204512433524
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 15
;;; Elapsed time: 1997.692784 ms
;;; Empirical mean: 192.9415
;;; Empirical variance: 155.03587775000005
;;; Empirical std: 12.451340399732073
;;; Empirical skewness: 0.02591202882958102
;;; Empirical kurtosis: 2.8224645714360914
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 16
;;; Elapsed time: 1992.578191 ms
;;; Empirical mean: 193.02579999999995
;;; Empirical variance: 160.34153436000008
;;; Empirical std: 12.662603774895592
;;; Empirical skewness: -0.088010843460613
;;; Empirical kurtosis: 2.6171207124116176
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 17
;;; Elapsed time: 2014.755084 ms
;;; Empirical mean: 195.28729999999987
;;; Empirical variance: 153.91255871000004
;;; Empirical std: 12.406150035768551
;;; Empirical skewness: 0.002434272488987545
;;; Empirical kurtosis: 2.9730864889391255
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 18
;;; Elapsed time: 2019.264146 ms
;;; Empirical mean: 195.01900000000006
;;; Empirical variance: 153.18303900000004
;;; Empirical std: 12.376713578329266
;;; Empirical skewness: 0.08255780651403558
;;; Empirical kurtosis: 3.144863318242446
;;; 
;;; Method :rmh with 10000 samples and options [:alpha 0.8 :sigma 2]
;;; Chain no. 19
;;; Elapsed time: 2395.545585 ms
;;; Empirical mean: 194.59510000000003
;;; Empirical variance: 133.5795559900001
;;; Empirical std: 11.557662219930123
;;; Empirical skewness: 0.4565318065801844
;;; Empirical kurtosis: 3.6780109089812925
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 2084.793554 ms
;;; Empirical mean: 194.48490099636538
;;; Empirical variance: 152.04160262427078
;;; Empirical std: 12.33051509971383
;;; Empirical skewness: 0.09447820025042805
;;; Empirical kurtosis: 3.052169944918929
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 1993.747695 ms
;;; Empirical mean: 194.19911282065462
;;; Empirical variance: 151.22981172091684
;;; Empirical std: 12.297553078597256
;;; Empirical skewness: 0.08279890678362609
;;; Empirical kurtosis: 3.0288556869844285
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 1928.547866 ms
;;; Empirical mean: 194.394447994193
;;; Empirical variance: 149.8043523988663
;;; Empirical std: 12.239458827859437
;;; Empirical skewness: 0.07576698783050805
;;; Empirical kurtosis: 3.00833282160955
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 1871.232845 ms
;;; Empirical mean: 194.10192012673602
;;; Empirical variance: 152.79547132232804
;;; Empirical std: 12.361046530222595
;;; Empirical skewness: 0.06514798336225705
;;; Empirical kurtosis: 3.05399002231595
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 1886.025234 ms
;;; Empirical mean: 194.00388224538386
;;; Empirical variance: 155.4590655790135
;;; Empirical std: 12.468322484561165
;;; Empirical skewness: 0.0937654061402664
;;; Empirical kurtosis: 2.913313689282067
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 1862.928932 ms
;;; Empirical mean: 194.18951509259986
;;; Empirical variance: 152.63502225804308
;;; Empirical std: 12.354554717109115
;;; Empirical skewness: 0.04225609083591155
;;; Empirical kurtosis: 2.9740522194384793
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 1873.102304 ms
;;; Empirical mean: 194.12860627420912
;;; Empirical variance: 149.0950221005127
;;; Empirical std: 12.21044725227183
;;; Empirical skewness: 0.06786834972451172
;;; Empirical kurtosis: 3.0996941055733753
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 1886.264754 ms
;;; Empirical mean: 194.4667696905618
;;; Empirical variance: 149.3824358204157
;;; Empirical std: 12.22221075830456
;;; Empirical skewness: 0.11556468069776044
;;; Empirical kurtosis: 2.957517496013697
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 1865.959469 ms
;;; Empirical mean: 194.1780852899462
;;; Empirical variance: 147.7943445300328
;;; Empirical std: 12.157069734522082
;;; Empirical skewness: 0.0944206884805826
;;; Empirical kurtosis: 3.025848313540301
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 1879.3131 ms
;;; Empirical mean: 194.34693118227432
;;; Empirical variance: 152.77431461320086
;;; Empirical std: 12.360190719127308
;;; Empirical skewness: 0.05586786509018536
;;; Empirical kurtosis: 3.0376337091220527
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 1877.527828 ms
;;; Empirical mean: 194.61697639428473
;;; Empirical variance: 154.03857755409905
;;; Empirical std: 12.411227882610932
;;; Empirical skewness: 0.0618274928851954
;;; Empirical kurtosis: 2.985398410213718
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 1873.653029 ms
;;; Empirical mean: 194.35359386738196
;;; Empirical variance: 152.69784318603132
;;; Empirical std: 12.357096875319515
;;; Empirical skewness: 0.07045970558515038
;;; Empirical kurtosis: 3.047610584815572
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 1847.299081 ms
;;; Empirical mean: 193.89996156723896
;;; Empirical variance: 155.67758295363433
;;; Empirical std: 12.477082309323535
;;; Empirical skewness: 0.040186467980159735
;;; Empirical kurtosis: 2.916060101513094
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 1870.545165 ms
;;; Empirical mean: 194.08170463355302
;;; Empirical variance: 153.83670148251704
;;; Empirical std: 12.40309241610805
;;; Empirical skewness: 0.07795519512718543
;;; Empirical kurtosis: 2.9869065149583647
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 1890.360041 ms
;;; Empirical mean: 194.23443179956365
;;; Empirical variance: 148.85573443823745
;;; Empirical std: 12.200644836984537
;;; Empirical skewness: 0.07060000611216655
;;; Empirical kurtosis: 3.0222703734662253
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 1940.942382 ms
;;; Empirical mean: 194.48457947014353
;;; Empirical variance: 155.55763640513993
;;; Empirical std: 12.472274708534123
;;; Empirical skewness: 0.09931953742981543
;;; Empirical kurtosis: 2.9751888621241833
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 1894.452058 ms
;;; Empirical mean: 194.31246134399316
;;; Empirical variance: 154.37349068909063
;;; Empirical std: 12.424712901676667
;;; Empirical skewness: 0.09051714405890783
;;; Empirical kurtosis: 2.982126872383717
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 1907.791554 ms
;;; Empirical mean: 194.094813996814
;;; Empirical variance: 152.63066676245026
;;; Empirical std: 12.354378445006866
;;; Empirical skewness: 0.07881880640674092
;;; Empirical kurtosis: 2.961107268105917
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 1863.994194 ms
;;; Empirical mean: 194.19079394082954
;;; Empirical variance: 152.65988271128697
;;; Empirical std: 12.355560801165076
;;; Empirical skewness: 0.06352023058078013
;;; Empirical kurtosis: 2.9039640736991132
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 1867.313754 ms
;;; Empirical mean: 194.34043203020624
;;; Empirical variance: 154.86922959914202
;;; Empirical std: 12.444646624116814
;;; Empirical skewness: 0.04727817454387181
;;; Empirical kurtosis: 2.9730853723823993
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 0
;;; Elapsed time: 1686.170061 ms
;;; Empirical mean: 194.24955639471966
;;; Empirical variance: 150.71398844644528
;;; Empirical std: 12.276562566388252
;;; Empirical skewness: 0.08149291424494298
;;; Empirical kurtosis: 3.0422934216436732
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 1
;;; Elapsed time: 1692.646755 ms
;;; Empirical mean: 194.6371732836097
;;; Empirical variance: 152.6664527195873
;;; Empirical std: 12.355826670829732
;;; Empirical skewness: 0.06463133604333468
;;; Empirical kurtosis: 3.0240293647779923
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 2
;;; Elapsed time: 1678.043541 ms
;;; Empirical mean: 194.182294832495
;;; Empirical variance: 151.91999553519628
;;; Empirical std: 12.32558296938511
;;; Empirical skewness: 0.11264102129999863
;;; Empirical kurtosis: 2.998728463936113
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 3
;;; Elapsed time: 1686.643602 ms
;;; Empirical mean: 194.20508807606436
;;; Empirical variance: 151.86241267047157
;;; Empirical std: 12.323246839630844
;;; Empirical skewness: 0.10247180238887463
;;; Empirical kurtosis: 3.0762265037938654
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 4
;;; Elapsed time: 1708.586868 ms
;;; Empirical mean: 194.19898195196012
;;; Empirical variance: 150.12321360731752
;;; Empirical std: 12.252477855818288
;;; Empirical skewness: 0.08150849549010775
;;; Empirical kurtosis: 3.0495434324801094
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 5
;;; Elapsed time: 1695.433023 ms
;;; Empirical mean: 194.1038025827781
;;; Empirical variance: 151.17980951790085
;;; Empirical std: 12.295519896202066
;;; Empirical skewness: 0.043122555049256064
;;; Empirical kurtosis: 3.0338652017478562
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 6
;;; Elapsed time: 1716.005996 ms
;;; Empirical mean: 194.31277201700516
;;; Empirical variance: 150.2618000211419
;;; Empirical std: 12.258131995583256
;;; Empirical skewness: 0.09179955044705639
;;; Empirical kurtosis: 3.0815256044727497
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 7
;;; Elapsed time: 1688.750582 ms
;;; Empirical mean: 194.41327061987747
;;; Empirical variance: 152.7922345866254
;;; Empirical std: 12.360915604704427
;;; Empirical skewness: 0.07358403656787703
;;; Empirical kurtosis: 2.9924993721599384
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 8
;;; Elapsed time: 1725.212295 ms
;;; Empirical mean: 194.18865985089232
;;; Empirical variance: 155.8928120287644
;;; Empirical std: 12.485704306476444
;;; Empirical skewness: 0.03943997451150399
;;; Empirical kurtosis: 3.0339990161862453
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 9
;;; Elapsed time: 1701.358121 ms
;;; Empirical mean: 194.2057300001863
;;; Empirical variance: 154.5226518359702
;;; Empirical std: 12.430714051733721
;;; Empirical skewness: 0.07562116271041101
;;; Empirical kurtosis: 3.019597436949041
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 10
;;; Elapsed time: 1710.55503 ms
;;; Empirical mean: 194.34327736038898
;;; Empirical variance: 152.30366730217474
;;; Empirical std: 12.341137196473214
;;; Empirical skewness: 0.07735851889185091
;;; Empirical kurtosis: 3.0178296751663276
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 11
;;; Elapsed time: 1689.765903 ms
;;; Empirical mean: 194.1060642005429
;;; Empirical variance: 152.50114433449184
;;; Empirical std: 12.349135367890817
;;; Empirical skewness: 0.0907258267805161
;;; Empirical kurtosis: 3.010205568870741
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 12
;;; Elapsed time: 1736.831228 ms
;;; Empirical mean: 194.1424467475549
;;; Empirical variance: 153.28843808291916
;;; Empirical std: 12.380970805349602
;;; Empirical skewness: 0.07153550044452807
;;; Empirical kurtosis: 2.9965077773453745
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 13
;;; Elapsed time: 1711.643662 ms
;;; Empirical mean: 194.1704707568765
;;; Empirical variance: 151.2967839624092
;;; Empirical std: 12.300275767738267
;;; Empirical skewness: 0.1096703315059449
;;; Empirical kurtosis: 2.9814403638919633
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 14
;;; Elapsed time: 1710.340522 ms
;;; Empirical mean: 194.6559043980508
;;; Empirical variance: 155.06858503714054
;;; Empirical std: 12.452653734732229
;;; Empirical skewness: 0.04058428118695334
;;; Empirical kurtosis: 3.0495620723416743
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 15
;;; Elapsed time: 1740.80288 ms
;;; Empirical mean: 194.06005905216398
;;; Empirical variance: 154.50627368926416
;;; Empirical std: 12.430055256887
;;; Empirical skewness: 0.057374213460350175
;;; Empirical kurtosis: 3.042624763438716
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 16
;;; Elapsed time: 1741.996764 ms
;;; Empirical mean: 194.13817092076292
;;; Empirical variance: 154.69709885066928
;;; Empirical std: 12.437728846162763
;;; Empirical skewness: 0.05657596576508731
;;; Empirical kurtosis: 2.9209578387230235
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 17
;;; Elapsed time: 1693.078654 ms
;;; Empirical mean: 194.28859777984508
;;; Empirical variance: 154.43751623066984
;;; Empirical std: 12.42728917466194
;;; Empirical skewness: 0.08655361813798938
;;; Empirical kurtosis: 3.0404079604626086
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 18
;;; Elapsed time: 1716.019127 ms
;;; Empirical mean: 194.1622289468083
;;; Empirical variance: 152.77737042658902
;;; Empirical std: 12.360314333648194
;;; Empirical skewness: 0.0917470636156682
;;; Empirical kurtosis: 3.1321476890537383
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 19
;;; Elapsed time: 1717.102982 ms
;;; Empirical mean: 194.35996894680807
;;; Empirical variance: 154.71116234306183
;;; Empirical std: 12.4382941894402
;;; Empirical skewness: 0.06060705781955499
;;; Empirical kurtosis: 2.994769648782452
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 1415.752738 ms
;;; Empirical mean: 194.3103
;;; Empirical variance: 153.10641391000004
;;; Empirical std: 12.373617656530367
;;; Empirical skewness: 0.060385038660293884
;;; Empirical kurtosis: 3.0366938439555513
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 1391.302062 ms
;;; Empirical mean: 194.14450000000014
;;; Empirical variance: 156.58561975000012
;;; Empirical std: 12.513417588732509
;;; Empirical skewness: 0.01953353148280863
;;; Empirical kurtosis: 3.1463738729035824
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 1427.784021 ms
;;; Empirical mean: 194.4243
;;; Empirical variance: 157.4552695100001
;;; Empirical std: 12.548118166083714
;;; Empirical skewness: 0.07096974352146143
;;; Empirical kurtosis: 2.971518814092971
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 1406.368318 ms
;;; Empirical mean: 194.34849999999994
;;; Empirical variance: 158.68644775
;;; Empirical std: 12.597080921785015
;;; Empirical skewness: 0.07973757239653197
;;; Empirical kurtosis: 2.976090566043231
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 1416.493517 ms
;;; Empirical mean: 194.2435
;;; Empirical variance: 158.3396077500001
;;; Empirical std: 12.583306709684864
;;; Empirical skewness: 0.09927897001742733
;;; Empirical kurtosis: 3.008739053097719
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 1418.735863 ms
;;; Empirical mean: 194.3849999999999
;;; Empirical variance: 158.68257500000004
;;; Empirical std: 12.596927204679721
;;; Empirical skewness: 0.13519907073528922
;;; Empirical kurtosis: 2.9068475988111038
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 1417.037955 ms
;;; Empirical mean: 194.36229999999983
;;; Empirical variance: 156.17263871
;;; Empirical std: 12.496905165279923
;;; Empirical skewness: 0.10610728711928204
;;; Empirical kurtosis: 2.929390988280673
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 1400.032419 ms
;;; Empirical mean: 193.83419999999995
;;; Empirical variance: 149.45731036000004
;;; Empirical std: 12.225273426799093
;;; Empirical skewness: 0.058426316981799335
;;; Empirical kurtosis: 3.048925631454951
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 1395.911574 ms
;;; Empirical mean: 194.67839999999984
;;; Empirical variance: 150.08097343999995
;;; Empirical std: 12.25075399475477
;;; Empirical skewness: 0.06536824364476013
;;; Empirical kurtosis: 2.919113980711393
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 1435.741909 ms
;;; Empirical mean: 194.42640000000011
;;; Empirical variance: 160.23038304000008
;;; Empirical std: 12.65821405412312
;;; Empirical skewness: 0.09665596122769328
;;; Empirical kurtosis: 2.9612449132208605
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 1404.14038 ms
;;; Empirical mean: 194.09190000000004
;;; Empirical variance: 153.67425439000013
;;; Empirical std: 12.39654203356727
;;; Empirical skewness: 0.15337418885232204
;;; Empirical kurtosis: 2.8962670643358646
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 1405.102344 ms
;;; Empirical mean: 193.90369999999996
;;; Empirical variance: 159.85322631
;;; Empirical std: 12.643307570015056
;;; Empirical skewness: 0.06796922017400862
;;; Empirical kurtosis: 2.9255897623338853
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 1411.192399 ms
;;; Empirical mean: 194.20009999999994
;;; Empirical variance: 156.02845999000007
;;; Empirical std: 12.491135256252734
;;; Empirical skewness: -3.418193456395627E-4
;;; Empirical kurtosis: 3.000254363680606
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 1446.25723 ms
;;; Empirical mean: 194.60049999999995
;;; Empirical variance: 158.1042997500001
;;; Empirical std: 12.573953226809781
;;; Empirical skewness: 0.09248673391440664
;;; Empirical kurtosis: 3.0494475062301416
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 1453.239758 ms
;;; Empirical mean: 194.51439999999985
;;; Empirical variance: 147.92479264
;;; Empirical std: 12.16243366436175
;;; Empirical skewness: 0.05276160398116855
;;; Empirical kurtosis: 3.0368033850273557
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 1399.071543 ms
;;; Empirical mean: 194.3036
;;; Empirical variance: 152.23822704000006
;;; Empirical std: 12.338485605616276
;;; Empirical skewness: 0.0951380265996414
;;; Empirical kurtosis: 3.089262322894407
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 1423.696908 ms
;;; Empirical mean: 194.1623
;;; Empirical variance: 156.44335871000013
;;; Empirical std: 12.507731957073597
;;; Empirical skewness: 0.08813379976217829
;;; Empirical kurtosis: 2.976238673323622
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 1410.93265 ms
;;; Empirical mean: 194.43699999999993
;;; Empirical variance: 154.266431
;;; Empirical std: 12.42040381791188
;;; Empirical skewness: 0.0836836935253285
;;; Empirical kurtosis: 2.9613441097440285
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 1415.174852 ms
;;; Empirical mean: 194.1657000000001
;;; Empirical variance: 151.2122435100001
;;; Empirical std: 12.296838760836058
;;; Empirical skewness: 0.08074632339956098
;;; Empirical kurtosis: 2.896330648398886
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 1434.772021 ms
;;; Empirical mean: 194.13989999999998
;;; Empirical variance: 157.15732799000006
;;; Empirical std: 12.53624058440169
;;; Empirical skewness: 0.07830575636918094
;;; Empirical kurtosis: 2.8872937266761696
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 0
;;; Elapsed time: 1832.332061 ms
;;; Empirical mean: 194.30659999999992
;;; Empirical variance: 158.16359644000005
;;; Empirical std: 12.576310923319289
;;; Empirical skewness: 0.08702490318981247
;;; Empirical kurtosis: 3.0267084219236873
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 1
;;; Elapsed time: 1829.473344 ms
;;; Empirical mean: 194.23169999999988
;;; Empirical variance: 152.84081511000002
;;; Empirical std: 12.362880534487099
;;; Empirical skewness: 0.08530633063929122
;;; Empirical kurtosis: 3.0238766744259227
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 2
;;; Elapsed time: 1845.990589 ms
;;; Empirical mean: 194.3158
;;; Empirical variance: 149.78467035999998
;;; Empirical std: 12.238654761042978
;;; Empirical skewness: 0.07035474635632522
;;; Empirical kurtosis: 2.9427731823970924
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 3
;;; Elapsed time: 1824.728263 ms
;;; Empirical mean: 194.16770000000002
;;; Empirical variance: 151.0545767100001
;;; Empirical std: 12.29042622165725
;;; Empirical skewness: 0.11573278629620806
;;; Empirical kurtosis: 2.953633125144654
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 4
;;; Elapsed time: 1832.008064 ms
;;; Empirical mean: 194.21829999999986
;;; Empirical variance: 150.99784510999996
;;; Empirical std: 12.288118045901088
;;; Empirical skewness: 0.09506341777041974
;;; Empirical kurtosis: 3.0120351598612163
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 5
;;; Elapsed time: 1841.919777 ms
;;; Empirical mean: 194.26380000000003
;;; Empirical variance: 149.24180956000006
;;; Empirical std: 12.216456505877638
;;; Empirical skewness: 0.08646432023654914
;;; Empirical kurtosis: 3.021905835295155
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 6
;;; Elapsed time: 1916.359277 ms
;;; Empirical mean: 194.24840000000003
;;; Empirical variance: 149.3160974400001
;;; Empirical std: 12.219496611563018
;;; Empirical skewness: 0.04026078256398529
;;; Empirical kurtosis: 2.9758091059309484
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 7
;;; Elapsed time: 1834.287436 ms
;;; Empirical mean: 194.15030000000002
;;; Empirical variance: 153.1999099100001
;;; Empirical std: 12.377395118117548
;;; Empirical skewness: 0.0835269191986198
;;; Empirical kurtosis: 2.9857800562787613
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 8
;;; Elapsed time: 1850.078293 ms
;;; Empirical mean: 194.3426
;;; Empirical variance: 151.21622524
;;; Empirical std: 12.297000660323638
;;; Empirical skewness: 0.09733894324638402
;;; Empirical kurtosis: 2.96069088149593
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 9
;;; Elapsed time: 1827.397854 ms
;;; Empirical mean: 194.2603000000001
;;; Empirical variance: 153.09794391000017
;;; Empirical std: 12.37327539134243
;;; Empirical skewness: 0.10282368764147086
;;; Empirical kurtosis: 2.957232646783272
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 10
;;; Elapsed time: 1862.32755 ms
;;; Empirical mean: 194.34650000000002
;;; Empirical variance: 154.45423775000003
;;; Empirical std: 12.427961930662647
;;; Empirical skewness: 0.09676673391911608
;;; Empirical kurtosis: 3.0118610924571625
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 11
;;; Elapsed time: 1859.946807 ms
;;; Empirical mean: 194.2363
;;; Empirical variance: 151.32586231000002
;;; Empirical std: 12.301457731098376
;;; Empirical skewness: 0.08109732708246802
;;; Empirical kurtosis: 2.928547388272632
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 12
;;; Elapsed time: 1856.239667 ms
;;; Empirical mean: 194.28339999999994
;;; Empirical variance: 147.80248444000003
;;; Empirical std: 12.157404510832073
;;; Empirical skewness: 0.06318042631178127
;;; Empirical kurtosis: 3.0710819354985555
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 13
;;; Elapsed time: 1820.407435 ms
;;; Empirical mean: 194.0668000000001
;;; Empirical variance: 148.6433377600001
;;; Empirical std: 12.191937407975818
;;; Empirical skewness: 0.04697793590691475
;;; Empirical kurtosis: 2.986141065129367
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 14
;;; Elapsed time: 1829.043978 ms
;;; Empirical mean: 194.2402
;;; Empirical variance: 152.99710396000006
;;; Empirical std: 12.369199810820426
;;; Empirical skewness: 0.030076765929132897
;;; Empirical kurtosis: 2.9765045417196796
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 15
;;; Elapsed time: 1841.841907 ms
;;; Empirical mean: 194.2421999999999
;;; Empirical variance: 151.65073916000006
;;; Empirical std: 12.31465546249671
;;; Empirical skewness: 0.12487145556471746
;;; Empirical kurtosis: 3.0679006720432183
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 16
;;; Elapsed time: 1837.323909 ms
;;; Empirical mean: 194.0212000000001
;;; Empirical variance: 149.46975056
;;; Empirical std: 12.225782206468427
;;; Empirical skewness: 0.0635467286243486
;;; Empirical kurtosis: 2.972555304469815
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 17
;;; Elapsed time: 1830.763055 ms
;;; Empirical mean: 194.57430000000002
;;; Empirical variance: 157.20647951000007
;;; Empirical std: 12.538200808329721
;;; Empirical skewness: 0.09345345782586599
;;; Empirical kurtosis: 3.014136925886431
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 18
;;; Elapsed time: 1853.278571 ms
;;; Empirical mean: 194.279
;;; Empirical variance: 152.639959
;;; Empirical std: 12.354754509904275
;;; Empirical skewness: 0.12448425019530568
;;; Empirical kurtosis: 2.9480775244714446
;;; 
;;; Method :pgibbs with 10000 samples and options [:number-of-particles 1000]
;;; Chain no. 19
;;; Elapsed time: 1831.684349 ms
;;; Empirical mean: 194.2499
;;; Empirical variance: 150.31064999000006
;;; Empirical std: 12.260124387215656
;;; Empirical skewness: 0.06474659978840198
;;; Empirical kurtosis: 2.924922360979742
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 705.47291 ms
;;; Empirical mean: 193.3277340730723
;;; Empirical variance: 150.73923748196697
;;; Empirical std: 12.277590866369794
;;; Empirical skewness: 0.04115531553035584
;;; Empirical kurtosis: 3.0681797273751554
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 681.963734 ms
;;; Empirical mean: 193.3997430715424
;;; Empirical variance: 155.5148243721398
;;; Empirical std: 12.470558302343155
;;; Empirical skewness: 0.079737736528275
;;; Empirical kurtosis: 3.076233960625335
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 676.424042 ms
;;; Empirical mean: 192.95559145214773
;;; Empirical variance: 145.8708925727109
;;; Empirical std: 12.077702288627208
;;; Empirical skewness: 0.0835909229999595
;;; Empirical kurtosis: 2.873562897227259
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 671.727471 ms
;;; Empirical mean: 193.9975986682043
;;; Empirical variance: 141.73691552505537
;;; Empirical std: 11.905331390812075
;;; Empirical skewness: -0.02430477148513537
;;; Empirical kurtosis: 2.964524755934575
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 679.880183 ms
;;; Empirical mean: 193.02091096549296
;;; Empirical variance: 145.08204444929936
;;; Empirical std: 12.045000807359846
;;; Empirical skewness: -0.029321329865105356
;;; Empirical kurtosis: 3.303048467692526
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 671.276753 ms
;;; Empirical mean: 193.64482184055362
;;; Empirical variance: 143.69367031187667
;;; Empirical std: 11.987229467724253
;;; Empirical skewness: -0.02444947479704681
;;; Empirical kurtosis: 2.916666014628971
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 682.389032 ms
;;; Empirical mean: 193.5360680360595
;;; Empirical variance: 152.1215536417642
;;; Empirical std: 12.333756671905125
;;; Empirical skewness: 0.02007649128949954
;;; Empirical kurtosis: 3.0335807036757925
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 674.491568 ms
;;; Empirical mean: 193.55683197013798
;;; Empirical variance: 141.28684600342297
;;; Empirical std: 11.886414345942303
;;; Empirical skewness: 0.06447833125091368
;;; Empirical kurtosis: 2.9252636405500603
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 684.66523 ms
;;; Empirical mean: 193.7683221093043
;;; Empirical variance: 146.41885266745294
;;; Empirical std: 12.100365807175127
;;; Empirical skewness: 0.1228481579316287
;;; Empirical kurtosis: 3.0511741466551916
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 673.154309 ms
;;; Empirical mean: 193.81329980839752
;;; Empirical variance: 148.5427855404087
;;; Empirical std: 12.187812992510539
;;; Empirical skewness: 0.06435746706299446
;;; Empirical kurtosis: 2.88705797202366
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 684.521491 ms
;;; Empirical mean: 193.7885128436203
;;; Empirical variance: 149.21358800692917
;;; Empirical std: 12.215301388296941
;;; Empirical skewness: 0.10938828060779865
;;; Empirical kurtosis: 2.937910038539006
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 689.102443 ms
;;; Empirical mean: 193.68956694305683
;;; Empirical variance: 149.44441909629813
;;; Empirical std: 12.224746177172683
;;; Empirical skewness: 0.056361996266105614
;;; Empirical kurtosis: 3.031655910412107
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 690.444877 ms
;;; Empirical mean: 193.47510532712926
;;; Empirical variance: 146.67704793103795
;;; Empirical std: 12.11103001115256
;;; Empirical skewness: 0.10469488004012822
;;; Empirical kurtosis: 2.839406088554621
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 679.755916 ms
;;; Empirical mean: 193.16111605564907
;;; Empirical variance: 143.43881108003313
;;; Empirical std: 11.976594302222695
;;; Empirical skewness: 0.09110539371411253
;;; Empirical kurtosis: 3.0935380506991326
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 683.290015 ms
;;; Empirical mean: 193.9249260588957
;;; Empirical variance: 139.69741867360446
;;; Empirical std: 11.819366255159558
;;; Empirical skewness: 0.1501787350602911
;;; Empirical kurtosis: 3.127361231014301
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 675.64986 ms
;;; Empirical mean: 193.5057868670867
;;; Empirical variance: 149.02333210499847
;;; Empirical std: 12.207511298581643
;;; Empirical skewness: 0.13292014324438203
;;; Empirical kurtosis: 3.1349538539261905
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 717.885421 ms
;;; Empirical mean: 193.74969098101914
;;; Empirical variance: 142.59668237872074
;;; Empirical std: 11.94138527888288
;;; Empirical skewness: 0.0704648576593434
;;; Empirical kurtosis: 3.182126194668189
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 699.390609 ms
;;; Empirical mean: 193.73727529747933
;;; Empirical variance: 150.94040566429263
;;; Empirical std: 12.285780629015505
;;; Empirical skewness: 0.1660984508905992
;;; Empirical kurtosis: 3.030146800154058
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 683.243801 ms
;;; Empirical mean: 194.09738330034068
;;; Empirical variance: 140.77339680568107
;;; Empirical std: 11.864796534525194
;;; Empirical skewness: 0.16337111971079107
;;; Empirical kurtosis: 3.2057194314773
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 696.813985 ms
;;; Empirical mean: 193.28231180226874
;;; Empirical variance: 147.92348734825072
;;; Empirical std: 12.162380003447135
;;; Empirical skewness: 0.03396390596380381
;;; Empirical kurtosis: 3.0952294148748605
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 0
;;; Elapsed time: 1483.347699 ms
;;; Empirical mean: 194.19942590381834
;;; Empirical variance: 153.4811741480861
;;; Empirical std: 12.388751920515888
;;; Empirical skewness: 0.07660823996875116
;;; Empirical kurtosis: 3.0743413804670103
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 1
;;; Elapsed time: 1511.266034 ms
;;; Empirical mean: 194.24968820923047
;;; Empirical variance: 150.64977556520464
;;; Empirical std: 12.273947024702553
;;; Empirical skewness: 0.034678584638323795
;;; Empirical kurtosis: 3.01378140925295
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 2
;;; Elapsed time: 1430.351073 ms
;;; Empirical mean: 194.44951881519324
;;; Empirical variance: 152.3510194047538
;;; Empirical std: 12.343055513314107
;;; Empirical skewness: 0.05565868325602069
;;; Empirical kurtosis: 3.02479320639983
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 3
;;; Elapsed time: 1411.690472 ms
;;; Empirical mean: 194.3970125866484
;;; Empirical variance: 152.8947856865488
;;; Empirical std: 12.36506310887853
;;; Empirical skewness: 0.06550537593681778
;;; Empirical kurtosis: 3.0508098537228268
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 4
;;; Elapsed time: 1372.39851 ms
;;; Empirical mean: 194.5000282040019
;;; Empirical variance: 154.03900099324017
;;; Empirical std: 12.411244941311898
;;; Empirical skewness: 0.09181035353842162
;;; Empirical kurtosis: 2.9692889684463486
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 5
;;; Elapsed time: 1384.278218 ms
;;; Empirical mean: 194.0326783585851
;;; Empirical variance: 149.29282849387712
;;; Empirical std: 12.218544450705949
;;; Empirical skewness: 0.09059531583492297
;;; Empirical kurtosis: 2.966114926408452
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 6
;;; Elapsed time: 1389.582265 ms
;;; Empirical mean: 194.3443146503434
;;; Empirical variance: 151.41975382072508
;;; Empirical std: 12.305273415114558
;;; Empirical skewness: 0.09878804259330279
;;; Empirical kurtosis: 2.970858134046253
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 7
;;; Elapsed time: 1392.677468 ms
;;; Empirical mean: 194.31147016503948
;;; Empirical variance: 149.96745747069005
;;; Empirical std: 12.24612009865533
;;; Empirical skewness: 0.05828938525849661
;;; Empirical kurtosis: 2.9551973960065467
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 8
;;; Elapsed time: 1401.756506 ms
;;; Empirical mean: 194.1983108499639
;;; Empirical variance: 155.25242978446497
;;; Empirical std: 12.460033297887488
;;; Empirical skewness: 0.12896317805065127
;;; Empirical kurtosis: 3.0560338006735304
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 9
;;; Elapsed time: 1406.717693 ms
;;; Empirical mean: 194.25442673711026
;;; Empirical variance: 150.72171847200266
;;; Empirical std: 12.276877390933032
;;; Empirical skewness: 0.06256197899433633
;;; Empirical kurtosis: 3.086255900423677
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 10
;;; Elapsed time: 1392.776319 ms
;;; Empirical mean: 194.22833102390305
;;; Empirical variance: 150.0096656314821
;;; Empirical std: 12.24784330531225
;;; Empirical skewness: 0.06111191192679175
;;; Empirical kurtosis: 3.0117439151503813
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 11
;;; Elapsed time: 1385.114502 ms
;;; Empirical mean: 194.28494247071583
;;; Empirical variance: 150.46600724863555
;;; Empirical std: 12.266458627029872
;;; Empirical skewness: 0.14383014887246387
;;; Empirical kurtosis: 2.9832156124897935
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 12
;;; Elapsed time: 1360.479205 ms
;;; Empirical mean: 194.45387524725317
;;; Empirical variance: 153.7789501656479
;;; Empirical std: 12.400764096040529
;;; Empirical skewness: 0.06704039981447742
;;; Empirical kurtosis: 2.9871486366868663
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 13
;;; Elapsed time: 1360.86285 ms
;;; Empirical mean: 194.19875699173335
;;; Empirical variance: 151.76866675590847
;;; Empirical std: 12.319442631706536
;;; Empirical skewness: 0.07833527526117817
;;; Empirical kurtosis: 3.0164359149366096
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 14
;;; Elapsed time: 1385.663722 ms
;;; Empirical mean: 194.3100354972224
;;; Empirical variance: 152.12106081961488
;;; Empirical std: 12.33373669329838
;;; Empirical skewness: 0.07186678207588262
;;; Empirical kurtosis: 2.9544408543950205
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 15
;;; Elapsed time: 1369.276616 ms
;;; Empirical mean: 194.4756247035798
;;; Empirical variance: 154.0275804226356
;;; Empirical std: 12.41078484313686
;;; Empirical skewness: 0.08450977959055322
;;; Empirical kurtosis: 2.963101085166563
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 16
;;; Elapsed time: 1401.182342 ms
;;; Empirical mean: 194.1463161058163
;;; Empirical variance: 151.96620476343745
;;; Empirical std: 12.327457351921257
;;; Empirical skewness: 0.11515136425753271
;;; Empirical kurtosis: 3.089802772080397
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 17
;;; Elapsed time: 1379.692677 ms
;;; Empirical mean: 194.25180207533157
;;; Empirical variance: 153.11849265289393
;;; Empirical std: 12.374105731441523
;;; Empirical skewness: 0.05110404206231074
;;; Empirical kurtosis: 2.9560849349563747
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 18
;;; Elapsed time: 1562.230475 ms
;;; Empirical mean: 194.13103918715964
;;; Empirical variance: 158.22497884841738
;;; Empirical std: 12.578751084603645
;;; Empirical skewness: 0.10446369933381114
;;; Empirical kurtosis: 3.123490846342697
;;; 
;;; Method :ipmcmc with 10000 samples and options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 19
;;; Elapsed time: 1749.823634 ms
;;; Empirical mean: 194.42768751532756
;;; Empirical variance: 154.6805749511744
;;; Empirical std: 12.437064563279167
;;; Empirical skewness: 0.04019272259766809
;;; Empirical kurtosis: 3.0379033361997516
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-unkown'>#multifn[print-method 0x2c7bcb60]</span>","value":"#multifn[print-method 0x2c7bcb60]"}],"value":"[nil,#multifn[print-method 0x2c7bcb60]]"},{"type":"html","content":"<span class='clj-unkown'>#multifn[print-method 0x2c7bcb60]</span>","value":"#multifn[print-method 0x2c7bcb60]"}],"value":"[[nil,#multifn[print-method 0x2c7bcb60]],#multifn[print-method 0x2c7bcb60]]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/model</span>","value":"#'model/model"}],"value":"[[[nil,#multifn[print-method 0x2c7bcb60]],#multifn[print-method 0x2c7bcb60]],#'model/model]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/model_name</span>","value":"#'model/model_name"}],"value":"[[[[nil,#multifn[print-method 0x2c7bcb60]],#multifn[print-method 0x2c7bcb60]],#'model/model],#'model/model_name]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/outfile</span>","value":"#'model/outfile"}],"value":"[[[[[nil,#multifn[print-method 0x2c7bcb60]],#multifn[print-method 0x2c7bcb60]],#'model/model],#'model/model_name],#'model/outfile]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/configurations</span>","value":"#'model/configurations"}],"value":"[[[[[[nil,#multifn[print-method 0x2c7bcb60]],#multifn[print-method 0x2c7bcb60]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/num_samples_options</span>","value":"#'model/num_samples_options"}],"value":"[[[[[[[nil,#multifn[print-method 0x2c7bcb60]],#multifn[print-method 0x2c7bcb60]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/thinning</span>","value":"#'model/thinning"}],"value":"[[[[[[[[nil,#multifn[print-method 0x2c7bcb60]],#multifn[print-method 0x2c7bcb60]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning]"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"[[[[[[[[[nil,#multifn[print-method 0x2c7bcb60]],#multifn[print-method 0x2c7bcb60]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/num-chains</span>","value":"#'model/num-chains"}],"value":"[[[[[[[[[[nil,#multifn[print-method 0x2c7bcb60]],#multifn[print-method 0x2c7bcb60]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains]"},{"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)"}],"value":"[[[[[[[[[[[nil,#multifn[print-method 0x2c7bcb60]],#multifn[print-method 0x2c7bcb60]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains],(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)]"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"[[[[[[[[[[[[nil,#multifn[print-method 0x2c7bcb60]],#multifn[print-method 0x2c7bcb60]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains],(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)],nil]"}
;; <=

;; @@

;; @@
