;; gorilla-repl.fileformat = 1

;; **
;;; # Gorilla REPL
;;; 
;;; Welcome to gorilla :-)
;;; 
;;; Shift + enter evaluates code. Hit alt+g twice in quick succession or click the menu icon (upper-right corner) for more commands ...
;;; 
;;; It's a good habit to run each worksheet in its own namespace: feel free to use the declaration we've provided below if you'd like.
;; **

;; @@

(ns model
  (:require [gorilla-plot.core :as plot])
  (:use [anglican core emit runtime stat
          [state :only [get-predicts get-log-weight get-result]]]))

(defdist geometric
"Geometric distribution on support {0,1,2....}"
[p] []
(sample* [this]
        (loop [value 0]
            (if (sample* (flip p))
            value
            (recur (inc value)))))
(observe* [this value] (+ (log p) (* value (log (- 1 p))))))

(defdist dirac [x]
    (sample* [this] x)
    (observe* [this value]
              (if (= value x)
                0
                (- (/ 1.0 0.0)))))


(def data [4 5 4 0 1 4 3 4 0 6 3 3 4 0 2 6 3 3 5 4 5 3 1 4 4 1 5 5 3 4 2 5 2 2 3 4 2 1 3 -1 2 1 1 1 1 3 0 0 1 0 1 1 0 0 3 1 0 3 2 2 0 1 1 1 0 1 0 1 0 0 0 2 1 0 0 0 1 1 0 2 3 3 1 -1 2 1 1 1 1 2 4 2 0 0 1 4 0 0 0 1 0 0 0 0 0 1 0 0 1 0 1]
  )

(with-primitive-procedures [dirac geometric]
  (defquery model [method- options- ]
    (let [
      switchpoint (sample (uniform-discrete 0 (count data)))
      rate1 (sample (geometric 0.1))
      rate2 (sample (geometric 0.1))]
      (doall (map
        (fn [i]
          (let [x (nth data i)
                rate (if (< i switchpoint) rate1 rate2)
                dist (if (zero? rate) (dirac 0) (poisson (* 0.1 rate)))]
            (if (<= 0 (nth data i)) (observe dist (nth data i)))))
        (range 0 (count data))))
      switchpoint
    )
  )
)


(def model_name "switchpoint")
(def outfile "output/switchpoint_anglican.json")

; (def configurations [:rmh []])
(def configurations
  [
    [:importance []]
    [:lmh []]
    [:rmh []]
    [:rmh [:alpha 0.8 :sigma 2]]
    [:smc []]
    [:smc [:number-of-particles 1000]]
    [:pgibbs []]
    [:pgibbs [:number-of-particles 1000]]
    [:ipmcmc []]
    [:ipmcmc [:number-of-particles 1000 :number-of-nodes 32]]
  ])

; (def num_samples_options [1000])
(def num_samples_options [1000 10000])
(def thinning 1)

(spit outfile "[\n" :append false)

(def num-chains 20)

(doall
  (for [ num_samples num_samples_options
         [method options] configurations
         chain (range 0 num-chains)]
    (do
      (println (format "\nMethod %s with options %s" method options))
      (println (format "Chain no. %s" chain))
      (let [start (. System (nanoTime))
            warmup (/ num_samples 5)
            samples (take-nth thinning (take (* num_samples thinning) (drop warmup (apply doquery method model [method options] options))))
            results (collect-results samples)
            values (map (fn [s] (get-result s)) samples)
            max-value (apply max values)
            mean (empirical-mean results)
            variance (empirical-variance results)
            std (empirical-std results)
            skewness (if (zero? std) (/ 0.0 0.0) (empirical-skew results))
            kurtosis (if (zero? std) (/ 0.0 0.0) (empirical-kurtosis results))
            distribution (empirical-distribution (collect-results samples))
            masses (for [n (range 0 (inc max-value))] (get distribution n 0.0))
            end (. System (nanoTime))
            elapsed_ms (/ (- end start) 1e6)]
        (println (format "Elapsed time: %s ms" elapsed_ms))
        (println (format "Empirical mean: %s" mean))
        (println (format "Empirical variance: %s" variance))
        (println (format "Empirical std: %s" std))
        (println (format "Empirical skewness: %s" skewness))
        (println (format "Empirical kurtosis: %s" kurtosis))
        (spit outfile (format
                   "{\"model\": \"%s\", \"system\": \"anglican\", \"method\": \"%s\", \"options\": \"%s\", \"num_samples\": %s, \"time_ms\": %s, \"total\": 1.0, \"mean\": %s, \"variance\": %s, \"stddev\": %s, \"skewness\": %s, \"kurtosis\": %s, \"masses\": [%s] },\n"
                   model_name method options num_samples elapsed_ms mean variance std skewness kurtosis
                   (clojure.string/join ", " masses)) :append true)
        (if false (do
          (println "Empirical distribution:")
          (doall (for [n (range 0 (inc max-value))]
            (println (format "p(%s) = %s" n (get distribution n 0.0)))))))
        ; (println "List of samples (format: sample log-weight):")
        ; (doall (map (fn [s] (println (format "%s %s" (get-result s) (get-log-weight s)))) samples))
        ; values need to be adjusted if they are weighted!
        ; (plot/histogram values :normalize :probability)
      )
    )
  )
)

(spit outfile "]\n" :append true)
;; @@
;; ->
;;; 
;;; Method :importance with options []
;;; Chain no. 0
;;; Elapsed time: 1385.35935 ms
;;; Empirical mean: 38.54240304903747
;;; Empirical variance: 5.914114748613068
;;; Empirical std: 2.4318952996815195
;;; Empirical skewness: 0.4657143017352132
;;; Empirical kurtosis: 2.1060863805198795
;;; 
;;; Method :importance with options []
;;; Chain no. 1
;;; Elapsed time: 1375.467954 ms
;;; Empirical mean: 40.86060363291846
;;; Empirical variance: 22.066192954848553
;;; Empirical std: 4.697466652872435
;;; Empirical skewness: -0.8356284651654606
;;; Empirical kurtosis: 1.7306484119610075
;;; 
;;; Method :importance with options []
;;; Chain no. 2
;;; Elapsed time: 1394.463599 ms
;;; Empirical mean: 37.00581335259124
;;; Empirical variance: 0.03690741777559662
;;; Empirical std: 0.1921130338514194
;;; Empirical skewness: 56.585787930431536
;;; Empirical kurtosis: 5223.416399352591
;;; 
;;; Method :importance with options []
;;; Chain no. 3
;;; Elapsed time: 1461.307475 ms
;;; Empirical mean: 40.82526804276694
;;; Empirical variance: 12.980944144707594
;;; Empirical std: 3.6029077346925766
;;; Empirical skewness: 0.5433732121955349
;;; Empirical kurtosis: 1.4002483126037228
;;; 
;;; Method :importance with options []
;;; Chain no. 4
;;; Elapsed time: 1546.02729 ms
;;; Empirical mean: 42.285940278939734
;;; Empirical variance: 18.291306308034187
;;; Empirical std: 4.276833677854937
;;; Empirical skewness: 0.05592483536509971
;;; Empirical kurtosis: 3.655413007328423
;;; 
;;; Method :importance with options []
;;; Chain no. 5
;;; Elapsed time: 1375.780754 ms
;;; Empirical mean: 45.3377249727599
;;; Empirical variance: 9.40284742883167
;;; Empirical std: 3.066406272631151
;;; Empirical skewness: 0.19222852855683173
;;; Empirical kurtosis: 2.806492541238196
;;; 
;;; Method :importance with options []
;;; Chain no. 6
;;; Elapsed time: 1371.449105 ms
;;; Empirical mean: 42.80381745030763
;;; Empirical variance: 6.617639075601492
;;; Empirical std: 2.572477225477709
;;; Empirical skewness: -1.3201784186621972
;;; Empirical kurtosis: 5.717376765361801
;;; 
;;; Method :importance with options []
;;; Chain no. 7
;;; Elapsed time: 1442.930441 ms
;;; Empirical mean: 37.9562174993933
;;; Empirical variance: 2.0732194660582484
;;; Empirical std: 1.4398678640966498
;;; Empirical skewness: 5.715954811847539
;;; Empirical kurtosis: 58.96259222851965
;;; 
;;; Method :importance with options []
;;; Chain no. 8
;;; Elapsed time: 1381.686341 ms
;;; Empirical mean: 44.66272129073442
;;; Empirical variance: 4.977125644826042
;;; Empirical std: 2.2309472528112453
;;; Empirical skewness: -3.0066657134854915
;;; Empirical kurtosis: 16.54292866050352
;;; 
;;; Method :importance with options []
;;; Chain no. 9
;;; Elapsed time: 1368.531669 ms
;;; Empirical mean: 39.53230803721376
;;; Empirical variance: 5.167032297641634
;;; Empirical std: 2.273110709499569
;;; Empirical skewness: 0.8429826046647911
;;; Empirical kurtosis: 1.9035640285919995
;;; 
;;; Method :importance with options []
;;; Chain no. 10
;;; Elapsed time: 1366.743561 ms
;;; Empirical mean: 36.39558539011264
;;; Empirical variance: 3.538071228649498
;;; Empirical std: 1.8809761371823668
;;; Empirical skewness: 4.349113986411794
;;; Empirical kurtosis: 22.129338818331124
;;; 
;;; Method :importance with options []
;;; Chain no. 11
;;; Elapsed time: 1428.210345 ms
;;; Empirical mean: 40.407474609429485
;;; Empirical variance: 2.723322003333804
;;; Empirical std: 1.6502490731201165
;;; Empirical skewness: 3.667237057699407
;;; Empirical kurtosis: 15.048799045774802
;;; 
;;; Method :importance with options []
;;; Chain no. 12
;;; Elapsed time: 1498.056538 ms
;;; Empirical mean: 40.19400345350503
;;; Empirical variance: 0.7465360961189227
;;; Empirical std: 0.8640232034609503
;;; Empirical skewness: 7.31900719279075
;;; Empirical kurtosis: 147.02780375716878
;;; 
;;; Method :importance with options []
;;; Chain no. 13
;;; Elapsed time: 1765.32629 ms
;;; Empirical mean: 44.0253059660953
;;; Empirical variance: 0.19445595415190778
;;; Empirical std: 0.44097160243252376
;;; Empirical skewness: 19.032898116099236
;;; Empirical kurtosis: 486.7697851747322
;;; 
;;; Method :importance with options []
;;; Chain no. 14
;;; Elapsed time: 1364.113329 ms
;;; Empirical mean: 41.0587378343497
;;; Empirical variance: 0.10320987930082777
;;; Empirical std: 0.3212629441763052
;;; Empirical skewness: -2.6146322050743755
;;; Empirical kurtosis: 132.32161485973973
;;; 
;;; Method :importance with options []
;;; Chain no. 15
;;; Elapsed time: 1466.270833 ms
;;; Empirical mean: 40.9119498008597
;;; Empirical variance: 6.343736461318005
;;; Empirical std: 2.518677522295779
;;; Empirical skewness: 0.715466049937375
;;; Empirical kurtosis: 4.40900004336543
;;; 
;;; Method :importance with options []
;;; Chain no. 16
;;; Elapsed time: 1943.341018 ms
;;; Empirical mean: 40.561883222539336
;;; Empirical variance: 0.9105365437062357
;;; Empirical std: 0.9542203852916975
;;; Empirical skewness: -4.11377351621591
;;; Empirical kurtosis: 24.78996906680049
;;; 
;;; Method :importance with options []
;;; Chain no. 17
;;; Elapsed time: 2194.808092 ms
;;; Empirical mean: 41.5617515086655
;;; Empirical variance: 7.001678617406051
;;; Empirical std: 2.6460685209204335
;;; Empirical skewness: 0.32605156183747547
;;; Empirical kurtosis: 2.9564999250491426
;;; 
;;; Method :importance with options []
;;; Chain no. 18
;;; Elapsed time: 1669.188411 ms
;;; Empirical mean: 46.150565495173836
;;; Empirical variance: 5.319936994088804
;;; Empirical std: 2.306498860630285
;;; Empirical skewness: -3.2695934458791553
;;; Empirical kurtosis: 21.396754585383334
;;; 
;;; Method :importance with options []
;;; Chain no. 19
;;; Elapsed time: 2128.106187 ms
;;; Empirical mean: 37.31939840992655
;;; Empirical variance: 8.117862549873072
;;; Empirical std: 2.849186296098076
;;; Empirical skewness: 0.02674578862972793
;;; Empirical kurtosis: 5.0396667648925675
;;; 
;;; Method :lmh with options []
;;; Chain no. 0
;;; Elapsed time: 1771.840312 ms
;;; Empirical mean: 40.099000000000004
;;; Empirical variance: 5.151199000000006
;;; Empirical std: 2.26962529947126
;;; Empirical skewness: 0.6627260597867753
;;; Empirical kurtosis: 4.521967165557841
;;; 
;;; Method :lmh with options []
;;; Chain no. 1
;;; Elapsed time: 1634.919172 ms
;;; Empirical mean: 39.623999999999995
;;; Empirical variance: 4.024624000000005
;;; Empirical std: 2.006146554965515
;;; Empirical skewness: 0.8893913895934298
;;; Empirical kurtosis: 6.1024000529374245
;;; 
;;; Method :lmh with options []
;;; Chain no. 2
;;; Elapsed time: 1611.167636 ms
;;; Empirical mean: 39.84200000000001
;;; Empirical variance: 6.749036000000002
;;; Empirical std: 2.5978906828425252
;;; Empirical skewness: -0.18188363342527608
;;; Empirical kurtosis: 1.8075949645693343
;;; 
;;; Method :lmh with options []
;;; Chain no. 3
;;; Elapsed time: 1208.616669 ms
;;; Empirical mean: 40.227
;;; Empirical variance: 2.591471000000002
;;; Empirical std: 1.6098046465332376
;;; Empirical skewness: -0.9913050594307178
;;; Empirical kurtosis: 5.56636404560498
;;; 
;;; Method :lmh with options []
;;; Chain no. 4
;;; Elapsed time: 1136.444125 ms
;;; Empirical mean: 40.126000000000005
;;; Empirical variance: 6.902124000000004
;;; Empirical std: 2.627189372694706
;;; Empirical skewness: -0.06557056184791941
;;; Empirical kurtosis: 2.0321773186753376
;;; 
;;; Method :lmh with options []
;;; Chain no. 5
;;; Elapsed time: 1138.088066 ms
;;; Empirical mean: 37.597
;;; Empirical variance: 3.0025910000000016
;;; Empirical std: 1.7327986034158736
;;; Empirical skewness: 1.1264102113441425
;;; Empirical kurtosis: 3.3928032848276883
;;; 
;;; Method :lmh with options []
;;; Chain no. 6
;;; Elapsed time: 1300.61439 ms
;;; Empirical mean: 39.721000000000004
;;; Empirical variance: 1.0451590000000006
;;; Empirical std: 1.0223301814971524
;;; Empirical skewness: 0.027623573765360752
;;; Empirical kurtosis: 3.1104684194599725
;;; 
;;; Method :lmh with options []
;;; Chain no. 7
;;; Elapsed time: 1319.584031 ms
;;; Empirical mean: 43.57500000000001
;;; Empirical variance: 217.24437500000025
;;; Empirical std: 14.73921215669278
;;; Empirical skewness: 3.2785473140163397
;;; Empirical kurtosis: 12.238190938014915
;;; 
;;; Method :lmh with options []
;;; Chain no. 8
;;; Elapsed time: 1205.30772 ms
;;; Empirical mean: 39.67400000000001
;;; Empirical variance: 2.3157240000000017
;;; Empirical std: 1.5217503080334833
;;; Empirical skewness: -0.2810978898783385
;;; Empirical kurtosis: 2.7573230727075195
;;; 
;;; Method :lmh with options []
;;; Chain no. 9
;;; Elapsed time: 1279.849521 ms
;;; Empirical mean: 39.858999999999995
;;; Empirical variance: 2.3411190000000017
;;; Empirical std: 1.5300715669536513
;;; Empirical skewness: -0.5275995076044803
;;; Empirical kurtosis: 3.327125507215447
;;; 
;;; Method :lmh with options []
;;; Chain no. 10
;;; Elapsed time: 1356.968309 ms
;;; Empirical mean: 39.37299999999999
;;; Empirical variance: 4.009871
;;; Empirical std: 2.002466229428102
;;; Empirical skewness: -0.6637840010555098
;;; Empirical kurtosis: 2.2970600396905065
;;; 
;;; Method :lmh with options []
;;; Chain no. 11
;;; Elapsed time: 1190.321741 ms
;;; Empirical mean: 40.748
;;; Empirical variance: 1.1744960000000007
;;; Empirical std: 1.0837416666346278
;;; Empirical skewness: 1.1579696677673281
;;; Empirical kurtosis: 5.93586119630544
;;; 
;;; Method :lmh with options []
;;; Chain no. 12
;;; Elapsed time: 1212.67137 ms
;;; Empirical mean: 40.031
;;; Empirical variance: 1.6040390000000015
;;; Empirical std: 1.2665066126949365
;;; Empirical skewness: -0.4096454673908902
;;; Empirical kurtosis: 3.3887070378790565
;;; 
;;; Method :lmh with options []
;;; Chain no. 13
;;; Elapsed time: 1206.000072 ms
;;; Empirical mean: 39.775000000000006
;;; Empirical variance: 4.184375000000001
;;; Empirical std: 2.0455744914326637
;;; Empirical skewness: -0.4373126680464795
;;; Empirical kurtosis: 2.3199843607164
;;; 
;;; Method :lmh with options []
;;; Chain no. 14
;;; Elapsed time: 1129.53375 ms
;;; Empirical mean: 40.964000000000006
;;; Empirical variance: 5.860704000000005
;;; Empirical std: 2.4208890928747655
;;; Empirical skewness: -0.12707800786055104
;;; Empirical kurtosis: 2.497646590320773
;;; 
;;; Method :lmh with options []
;;; Chain no. 15
;;; Elapsed time: 1481.071132 ms
;;; Empirical mean: 40.118
;;; Empirical variance: 5.4160760000000066
;;; Empirical std: 2.327246441612922
;;; Empirical skewness: 0.23936337245545164
;;; Empirical kurtosis: 4.092921679859431
;;; 
;;; Method :lmh with options []
;;; Chain no. 16
;;; Elapsed time: 1168.937521 ms
;;; Empirical mean: 40.696999999999996
;;; Empirical variance: 1.7631910000000015
;;; Empirical std: 1.327852024888316
;;; Empirical skewness: -0.0172733102336435
;;; Empirical kurtosis: 5.700608981771763
;;; 
;;; Method :lmh with options []
;;; Chain no. 17
;;; Elapsed time: 1080.803775 ms
;;; Empirical mean: 39.819
;;; Empirical variance: 9.198239000000006
;;; Empirical std: 3.0328598714744484
;;; Empirical skewness: 0.5740972423148659
;;; Empirical kurtosis: 2.5346321030209205
;;; 
;;; Method :lmh with options []
;;; Chain no. 18
;;; Elapsed time: 1181.776432 ms
;;; Empirical mean: 38.894999999999996
;;; Empirical variance: 0.9599750000000006
;;; Empirical std: 0.9797831392711351
;;; Empirical skewness: 0.49815652795324605
;;; Empirical kurtosis: 13.607660760999075
;;; 
;;; Method :lmh with options []
;;; Chain no. 19
;;; Elapsed time: 1149.553314 ms
;;; Empirical mean: 38.725
;;; Empirical variance: 4.567375
;;; Empirical std: 2.1371417828492336
;;; Empirical skewness: -0.029107163787068457
;;; Empirical kurtosis: 1.265922640476058
;;; 
;;; Method :rmh with options []
;;; Chain no. 0
;;; Elapsed time: 1212.620572 ms
;;; Empirical mean: 39.108000000000004
;;; Empirical variance: 6.5803360000000035
;;; Empirical std: 2.5652165600588197
;;; Empirical skewness: 0.17859678717746963
;;; Empirical kurtosis: 2.85844436728318
;;; 
;;; Method :rmh with options []
;;; Chain no. 1
;;; Elapsed time: 1199.946554 ms
;;; Empirical mean: 41.093
;;; Empirical variance: 12.152351000000007
;;; Empirical std: 3.4860222317133904
;;; Empirical skewness: -0.25829652793992836
;;; Empirical kurtosis: 3.2883127797994214
;;; 
;;; Method :rmh with options []
;;; Chain no. 2
;;; Elapsed time: 1378.159709 ms
;;; Empirical mean: 40.03000000000001
;;; Empirical variance: 2.6431000000000022
;;; Empirical std: 1.625761360101784
;;; Empirical skewness: -0.8540524609562564
;;; Empirical kurtosis: 3.4712557817659704
;;; 
;;; Method :rmh with options []
;;; Chain no. 3
;;; Elapsed time: 1301.975322 ms
;;; Empirical mean: 40.056
;;; Empirical variance: 2.6308640000000025
;;; Empirical std: 1.6219938347601703
;;; Empirical skewness: -0.5643361611186121
;;; Empirical kurtosis: 3.5772753316553616
;;; 
;;; Method :rmh with options []
;;; Chain no. 4
;;; Elapsed time: 1230.35727 ms
;;; Empirical mean: 39.674
;;; Empirical variance: 5.2037240000000065
;;; Empirical std: 2.2811672450743297
;;; Empirical skewness: 1.384760405790802
;;; Empirical kurtosis: 4.977170589686535
;;; 
;;; Method :rmh with options []
;;; Chain no. 5
;;; Elapsed time: 1279.570693 ms
;;; Empirical mean: 38.847
;;; Empirical variance: 6.153591000000002
;;; Empirical std: 2.480643263349247
;;; Empirical skewness: 0.2656479501015478
;;; Empirical kurtosis: 1.8036400459364348
;;; 
;;; Method :rmh with options []
;;; Chain no. 6
;;; Elapsed time: 1269.317258 ms
;;; Empirical mean: 40.483999999999995
;;; Empirical variance: 2.277744000000004
;;; Empirical std: 1.5092196659201087
;;; Empirical skewness: -0.01822896777706563
;;; Empirical kurtosis: 8.916719554767942
;;; 
;;; Method :rmh with options []
;;; Chain no. 7
;;; Elapsed time: 1428.061523 ms
;;; Empirical mean: 39.708000000000006
;;; Empirical variance: 5.394736
;;; Empirical std: 2.32265709910008
;;; Empirical skewness: 0.044108575304750715
;;; Empirical kurtosis: 1.9712764499022475
;;; 
;;; Method :rmh with options []
;;; Chain no. 8
;;; Elapsed time: 1269.085353 ms
;;; Empirical mean: 38.903000000000006
;;; Empirical variance: 4.2375910000000045
;;; Empirical std: 2.05854098817585
;;; Empirical skewness: 0.5609204117071578
;;; Empirical kurtosis: 3.978920775087311
;;; 
;;; Method :rmh with options []
;;; Chain no. 9
;;; Elapsed time: 1230.83984 ms
;;; Empirical mean: 41.03299999999999
;;; Empirical variance: 12.855910999999999
;;; Empirical std: 3.5855140496168745
;;; Empirical skewness: -0.0436873622516357
;;; Empirical kurtosis: 2.108316325094593
;;; 
;;; Method :rmh with options []
;;; Chain no. 10
;;; Elapsed time: 1189.076648 ms
;;; Empirical mean: 39.406
;;; Empirical variance: 9.789164000000001
;;; Empirical std: 3.128763973200919
;;; Empirical skewness: 0.3360284813159428
;;; Empirical kurtosis: 2.1921335354187064
;;; 
;;; Method :rmh with options []
;;; Chain no. 11
;;; Elapsed time: 1203.480867 ms
;;; Empirical mean: 40.915
;;; Empirical variance: 1.779775000000002
;;; Empirical std: 1.3340820814327738
;;; Empirical skewness: -1.039664195005055
;;; Empirical kurtosis: 5.271655149925051
;;; 
;;; Method :rmh with options []
;;; Chain no. 12
;;; Elapsed time: 1342.273344 ms
;;; Empirical mean: 39.178
;;; Empirical variance: 3.8483160000000045
;;; Empirical std: 1.961712517164532
;;; Empirical skewness: 1.4514424518111622
;;; Empirical kurtosis: 7.1938631726536775
;;; 
;;; Method :rmh with options []
;;; Chain no. 13
;;; Elapsed time: 1191.803855 ms
;;; Empirical mean: 40.031
;;; Empirical variance: 11.888039000000008
;;; Empirical std: 3.4479035659368447
;;; Empirical skewness: -0.20143587976871416
;;; Empirical kurtosis: 2.7440791085339886
;;; 
;;; Method :rmh with options []
;;; Chain no. 14
;;; Elapsed time: 1258.8485 ms
;;; Empirical mean: 40.699999999999996
;;; Empirical variance: 5.526000000000003
;;; Empirical std: 2.3507445628991688
;;; Empirical skewness: -0.5152801767363835
;;; Empirical kurtosis: 3.3372178425706784
;;; 
;;; Method :rmh with options []
;;; Chain no. 15
;;; Elapsed time: 1191.214807 ms
;;; Empirical mean: 39.30099999999999
;;; Empirical variance: 2.6703990000000006
;;; Empirical std: 1.634135551293099
;;; Empirical skewness: -0.6644771124655813
;;; Empirical kurtosis: 4.002270536476895
;;; 
;;; Method :rmh with options []
;;; Chain no. 16
;;; Elapsed time: 1176.366066 ms
;;; Empirical mean: 40.009
;;; Empirical variance: 7.9949190000000065
;;; Empirical std: 2.827528779694383
;;; Empirical skewness: 0.29077130539346147
;;; Empirical kurtosis: 2.813861947296026
;;; 
;;; Method :rmh with options []
;;; Chain no. 17
;;; Elapsed time: 1102.239283 ms
;;; Empirical mean: 39.096
;;; Empirical variance: 4.724784
;;; Empirical std: 2.173656826640305
;;; Empirical skewness: -0.1191446444929298
;;; Empirical kurtosis: 1.900166422712908
;;; 
;;; Method :rmh with options []
;;; Chain no. 18
;;; Elapsed time: 1159.972726 ms
;;; Empirical mean: 49.694
;;; Empirical variance: 373.86836399999976
;;; Empirical std: 19.335675938533925
;;; Empirical skewness: 1.771628743496657
;;; Empirical kurtosis: 4.542335345120286
;;; 
;;; Method :rmh with options []
;;; Chain no. 19
;;; Elapsed time: 1339.716405 ms
;;; Empirical mean: 39.91300000000001
;;; Empirical variance: 3.1574310000000034
;;; Empirical std: 1.7769161488376437
;;; Empirical skewness: -0.29734514570569326
;;; Empirical kurtosis: 3.178234711801708
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 0
;;; Elapsed time: 1264.966146 ms
;;; Empirical mean: 39.522000000000006
;;; Empirical variance: 10.835515999999997
;;; Empirical std: 3.291734497191412
;;; Empirical skewness: 0.798408413038493
;;; Empirical kurtosis: 2.434698113864144
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 1
;;; Elapsed time: 1332.961471 ms
;;; Empirical mean: 40.85900000000001
;;; Empirical variance: 4.179119000000003
;;; Empirical std: 2.044289363079504
;;; Empirical skewness: 0.8600360552540903
;;; Empirical kurtosis: 3.2174067884247646
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 2
;;; Elapsed time: 1219.596755 ms
;;; Empirical mean: 39.623999999999995
;;; Empirical variance: 7.476624000000005
;;; Empirical std: 2.734341602653188
;;; Empirical skewness: 0.7366000508877787
;;; Empirical kurtosis: 2.791567117606741
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 3
;;; Elapsed time: 1384.34623 ms
;;; Empirical mean: 42.2
;;; Empirical variance: 17.89
;;; Empirical std: 4.2296571965113205
;;; Empirical skewness: 0.08056185311682881
;;; Empirical kurtosis: 2.0368496254203596
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 4
;;; Elapsed time: 1182.395479 ms
;;; Empirical mean: 40.17200000000001
;;; Empirical variance: 3.670416000000003
;;; Empirical std: 1.915832978106391
;;; Empirical skewness: 0.2236914628503078
;;; Empirical kurtosis: 3.1458988338379554
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 5
;;; Elapsed time: 1155.6973 ms
;;; Empirical mean: 40.469
;;; Empirical variance: 5.501039000000001
;;; Empirical std: 2.3454293849954215
;;; Empirical skewness: -0.059068822574595975
;;; Empirical kurtosis: 2.494603743546278
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 6
;;; Elapsed time: 1426.265239 ms
;;; Empirical mean: 40.575
;;; Empirical variance: 2.324375000000001
;;; Empirical std: 1.5245901088489329
;;; Empirical skewness: -0.41331047820403227
;;; Empirical kurtosis: 3.479621481110389
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 7
;;; Elapsed time: 1179.209656 ms
;;; Empirical mean: 41.122
;;; Empirical variance: 7.211116000000002
;;; Empirical std: 2.685352118438102
;;; Empirical skewness: 0.19649039062276433
;;; Empirical kurtosis: 2.320548999947828
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 8
;;; Elapsed time: 1300.555194 ms
;;; Empirical mean: 39.468
;;; Empirical variance: 3.5909760000000035
;;; Empirical std: 1.8949870711960024
;;; Empirical skewness: 0.0710913898346127
;;; Empirical kurtosis: 2.1969072855040186
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 9
;;; Elapsed time: 1146.41526 ms
;;; Empirical mean: 39.589999999999996
;;; Empirical variance: 7.541900000000006
;;; Empirical std: 2.7462519913511225
;;; Empirical skewness: 0.7245793988726866
;;; Empirical kurtosis: 3.2707168977154346
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 10
;;; Elapsed time: 1274.700861 ms
;;; Empirical mean: 40.641999999999996
;;; Empirical variance: 1.909836000000001
;;; Empirical std: 1.3819681617171942
;;; Empirical skewness: 0.5770574754779533
;;; Empirical kurtosis: 5.519969672280698
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 11
;;; Elapsed time: 1325.910518 ms
;;; Empirical mean: 38.864999999999995
;;; Empirical variance: 3.800775000000001
;;; Empirical std: 1.94955764213321
;;; Empirical skewness: 0.45868451424424483
;;; Empirical kurtosis: 2.5273894291117966
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 12
;;; Elapsed time: 1174.215966 ms
;;; Empirical mean: 39.703
;;; Empirical variance: 3.360791000000003
;;; Empirical std: 1.8332460282242542
;;; Empirical skewness: -0.9291010399518462
;;; Empirical kurtosis: 3.0256085235324783
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 13
;;; Elapsed time: 1322.553968 ms
;;; Empirical mean: 77.298
;;; Empirical variance: 679.0591960000002
;;; Empirical std: 26.058764283825894
;;; Empirical skewness: -0.6847181518136396
;;; Empirical kurtosis: 1.4866980538931287
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 14
;;; Elapsed time: 1650.710261 ms
;;; Empirical mean: 39.206
;;; Empirical variance: 7.521564000000006
;;; Empirical std: 2.742546991393221
;;; Empirical skewness: 0.5157459012404676
;;; Empirical kurtosis: 2.335574365177494
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 15
;;; Elapsed time: 1239.400657 ms
;;; Empirical mean: 40.391
;;; Empirical variance: 3.1321190000000025
;;; Empirical std: 1.7697793647796898
;;; Empirical skewness: -0.2999646746106236
;;; Empirical kurtosis: 3.4239085215234373
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 16
;;; Elapsed time: 1219.218547 ms
;;; Empirical mean: 40.16000000000001
;;; Empirical variance: 7.966400000000009
;;; Empirical std: 2.822481177970902
;;; Empirical skewness: 0.998177361326738
;;; Empirical kurtosis: 3.3738357861402912
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 17
;;; Elapsed time: 1270.842103 ms
;;; Empirical mean: 39.036
;;; Empirical variance: 2.6927040000000018
;;; Empirical std: 1.6409460685836088
;;; Empirical skewness: -1.0095679213223823
;;; Empirical kurtosis: 4.301876758654113
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 18
;;; Elapsed time: 1686.154702 ms
;;; Empirical mean: 38.38799999999999
;;; Empirical variance: 7.109456000000003
;;; Empirical std: 2.6663563152737115
;;; Empirical skewness: -0.9040677277788643
;;; Empirical kurtosis: 3.087456244389897
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 19
;;; Elapsed time: 1416.821587 ms
;;; Empirical mean: 38.79900000000001
;;; Empirical variance: 5.942599000000002
;;; Empirical std: 2.4377446543885606
;;; Empirical skewness: 0.5376375839212915
;;; Empirical kurtosis: 2.1119608813192072
;;; 
;;; Method :smc with options []
;;; Chain no. 0
;;; Elapsed time: 3463.100443 ms
;;; Empirical mean: 40.80331371101854
;;; Empirical variance: 6.140611548562449
;;; Empirical std: 2.4780257360573255
;;; Empirical skewness: 0.8618203607330751
;;; Empirical kurtosis: 3.2056918229455973
;;; 
;;; Method :smc with options []
;;; Chain no. 1
;;; Elapsed time: 2729.75165 ms
;;; Empirical mean: 39.833401031424586
;;; Empirical variance: 7.227123706710448
;;; Empirical std: 2.688331026252245
;;; Empirical skewness: 1.4766168105772073
;;; Empirical kurtosis: 4.681508265581787
;;; 
;;; Method :smc with options []
;;; Chain no. 2
;;; Elapsed time: 2801.785813 ms
;;; Empirical mean: 43.3153599057115
;;; Empirical variance: 0.8705749029704497
;;; Empirical std: 0.9330460347541538
;;; Empirical skewness: -5.278148522591004
;;; Empirical kurtosis: 64.06816236488388
;;; 
;;; Method :smc with options []
;;; Chain no. 3
;;; Elapsed time: 3017.796978 ms
;;; Empirical mean: 36.79592572873915
;;; Empirical variance: 3.853884717082295
;;; Empirical std: 1.9631313550250007
;;; Empirical skewness: 2.9826447350206746
;;; Empirical kurtosis: 13.055328367330167
;;; 
;;; Method :smc with options []
;;; Chain no. 4
;;; Elapsed time: 2892.864374 ms
;;; Empirical mean: 35.992388752621316
;;; Empirical variance: 4.2821873886620185
;;; Empirical std: 2.0693446761383223
;;; Empirical skewness: 2.091199330352166
;;; Empirical kurtosis: 15.18940022983102
;;; 
;;; Method :smc with options []
;;; Chain no. 5
;;; Elapsed time: 2905.641816 ms
;;; Empirical mean: 37.84676705814847
;;; Empirical variance: 5.624018903490459
;;; Empirical std: 2.371501402801706
;;; Empirical skewness: 2.541009855051267
;;; Empirical kurtosis: 9.569298295395303
;;; 
;;; Method :smc with options []
;;; Chain no. 6
;;; Elapsed time: 2820.93093 ms
;;; Empirical mean: 42.31962576384212
;;; Empirical variance: 8.434953483003124
;;; Empirical std: 2.9042991380026826
;;; Empirical skewness: -1.64264442808539
;;; Empirical kurtosis: 5.11922801019239
;;; 
;;; Method :smc with options []
;;; Chain no. 7
;;; Elapsed time: 2727.310131 ms
;;; Empirical mean: 42.94960341716286
;;; Empirical variance: 4.085806985682145
;;; Empirical std: 2.0213379197160837
;;; Empirical skewness: -2.0622537184559517
;;; Empirical kurtosis: 8.651883958766676
;;; 
;;; Method :smc with options []
;;; Chain no. 8
;;; Elapsed time: 2574.722973 ms
;;; Empirical mean: 41.77404613161264
;;; Empirical variance: 38.79398106987873
;;; Empirical std: 6.228481441722271
;;; Empirical skewness: -0.39358141782050043
;;; Empirical kurtosis: 1.2595071284505808
;;; 
;;; Method :smc with options []
;;; Chain no. 9
;;; Elapsed time: 2693.719058 ms
;;; Empirical mean: 40.99584465434359
;;; Empirical variance: 0.06814541100183352
;;; Empirical std: 0.26104676018260314
;;; Empirical skewness: 18.30192719298244
;;; Empirical kurtosis: 2029.8114125007894
;;; 
;;; Method :smc with options []
;;; Chain no. 10
;;; Elapsed time: 2865.160732 ms
;;; Empirical mean: 44.166108583348056
;;; Empirical variance: 3.9990562275972623
;;; Empirical std: 1.999764042980387
;;; Empirical skewness: -0.2692773569379032
;;; Empirical kurtosis: 1.8038910328686502
;;; 
;;; Method :smc with options []
;;; Chain no. 11
;;; Elapsed time: 3010.977033 ms
;;; Empirical mean: 37.427057214125014
;;; Empirical variance: 12.21408379637314
;;; Empirical std: 3.494865347387956
;;; Empirical skewness: 2.0479211155841326
;;; Empirical kurtosis: 5.236968567887647
;;; 
;;; Method :smc with options []
;;; Chain no. 12
;;; Elapsed time: 3184.22075 ms
;;; Empirical mean: 43.2168522659317
;;; Empirical variance: 15.712943940530613
;;; Empirical std: 3.9639555926537082
;;; Empirical skewness: -1.3975580578483207
;;; Empirical kurtosis: 6.709041407011289
;;; 
;;; Method :smc with options []
;;; Chain no. 13
;;; Elapsed time: 2903.946191 ms
;;; Empirical mean: 34.5691926550739
;;; Empirical variance: 8.225745800669452
;;; Empirical std: 2.868056101381117
;;; Empirical skewness: 4.98452741956429
;;; Empirical kurtosis: 26.40777139309776
;;; 
;;; Method :smc with options []
;;; Chain no. 14
;;; Elapsed time: 2696.407539 ms
;;; Empirical mean: 37.08948968247974
;;; Empirical variance: 1.15929626881551
;;; Empirical std: 1.076706212861944
;;; Empirical skewness: 3.9096600983491796
;;; Empirical kurtosis: 27.781834712003814
;;; 
;;; Method :smc with options []
;;; Chain no. 15
;;; Elapsed time: 2865.451665 ms
;;; Empirical mean: 41.35167902066066
;;; Empirical variance: 4.378159126367069
;;; Empirical std: 2.092405105701826
;;; Empirical skewness: -2.606295013096985
;;; Empirical kurtosis: 8.514819841512997
;;; 
;;; Method :smc with options []
;;; Chain no. 16
;;; Elapsed time: 2915.3999 ms
;;; Empirical mean: 43.81737379551762
;;; Empirical variance: 1.9922749115260516
;;; Empirical std: 1.411479688669324
;;; Empirical skewness: -6.092255489571505
;;; Empirical kurtosis: 44.24336051685731
;;; 
;;; Method :smc with options []
;;; Chain no. 17
;;; Elapsed time: 2704.91095 ms
;;; Empirical mean: 40.404176545734806
;;; Empirical variance: 0.33328552852055293
;;; Empirical std: 0.5773088675228824
;;; Empirical skewness: 1.8097275664241916
;;; Empirical kurtosis: 33.01586582382816
;;; 
;;; Method :smc with options []
;;; Chain no. 18
;;; Elapsed time: 2877.183198 ms
;;; Empirical mean: 39.513118954338445
;;; Empirical variance: 2.3104279938281613
;;; Empirical std: 1.5200092084682124
;;; Empirical skewness: -1.9117450233976265
;;; Empirical kurtosis: 21.10711085070049
;;; 
;;; Method :smc with options []
;;; Chain no. 19
;;; Elapsed time: 3262.907998 ms
;;; Empirical mean: 40.25600285031067
;;; Empirical variance: 1.2766238766201676
;;; Empirical std: 1.1298778149075093
;;; Empirical skewness: -1.2928923329941684
;;; Empirical kurtosis: 20.247761303680136
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 0
;;; Elapsed time: 3521.473658 ms
;;; Empirical mean: 40.05991798114587
;;; Empirical variance: 0.20138031965510123
;;; Empirical std: 0.4487541862257122
;;; Empirical skewness: -4.083374688053442
;;; Empirical kurtosis: 48.99770312178109
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 1
;;; Elapsed time: 3442.86721 ms
;;; Empirical mean: 42.32919150093528
;;; Empirical variance: 1.57569973006668
;;; Empirical std: 1.2552687879759776
;;; Empirical skewness: 3.4485189282177595
;;; Empirical kurtosis: 12.995967963838407
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 2
;;; Elapsed time: 2987.000536 ms
;;; Empirical mean: 38.99134841743465
;;; Empirical variance: 0.010061462570650536
;;; Empirical std: 0.10030684209290279
;;; Empirical skewness: 13.998029069855134
;;; Empirical kurtosis: 4484.769323851052
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 3
;;; Elapsed time: 3011.643322 ms
;;; Empirical mean: 40.47734467627108
;;; Empirical variance: 9.108486123590493
;;; Empirical std: 3.018026859322245
;;; Empirical skewness: 0.4220634676996329
;;; Empirical kurtosis: 1.246174878385637
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 4
;;; Elapsed time: 3527.882322 ms
;;; Empirical mean: 39.918460826089614
;;; Empirical variance: 1.3170424222930834
;;; Empirical std: 1.1476246870353928
;;; Empirical skewness: -0.9112190977538613
;;; Empirical kurtosis: 5.139035500956805
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 5
;;; Elapsed time: 3491.621141 ms
;;; Empirical mean: 38.09155937550359
;;; Empirical variance: 4.798101037526859
;;; Empirical std: 2.1904568102400144
;;; Empirical skewness: -0.6771764646263142
;;; Empirical kurtosis: 4.4934186141940335
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 6
;;; Elapsed time: 3590.926076 ms
;;; Empirical mean: 44.94507895966638
;;; Empirical variance: 5.445696993224724
;;; Empirical std: 2.3336017212079536
;;; Empirical skewness: -1.8431518459148122
;;; Empirical kurtosis: 4.954629033536765
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 7
;;; Elapsed time: 3161.69144 ms
;;; Empirical mean: 40.37720264996012
;;; Empirical variance: 2.12407479875834
;;; Empirical std: 1.4574205977542447
;;; Empirical skewness: 3.6089947842961583
;;; Empirical kurtosis: 14.052591165863905
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 8
;;; Elapsed time: 2980.848673 ms
;;; Empirical mean: 39.000053923146496
;;; Empirical variance: 4.313822642438216E-4
;;; Empirical std: 0.020769743961922632
;;; Empirical skewness: 385.1704753016352
;;; Empirical kurtosis: 148357.2950440876
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 9
;;; Elapsed time: 3050.81304 ms
;;; Empirical mean: 40.609896496397766
;;; Empirical variance: 5.761398678596111
;;; Empirical std: 2.400291373686976
;;; Empirical skewness: -0.27426567675502384
;;; Empirical kurtosis: 2.1195454544450585
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 10
;;; Elapsed time: 2904.826936 ms
;;; Empirical mean: 37.17052974269746
;;; Empirical variance: 2.276226305215445
;;; Empirical std: 1.5087167743534386
;;; Empirical skewness: 1.1510024993895218
;;; Empirical kurtosis: 2.714825772644916
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 11
;;; Elapsed time: 3315.008601 ms
;;; Empirical mean: 39.46693355449979
;;; Empirical variance: 1.9187224852676221
;;; Empirical std: 1.3851795859265406
;;; Empirical skewness: 3.1784471281020825
;;; Empirical kurtosis: 14.536133691834836
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 12
;;; Elapsed time: 4120.515597 ms
;;; Empirical mean: 41.002751236266
;;; Empirical variance: 0.9799915655963791
;;; Empirical std: 0.9899452336348608
;;; Empirical skewness: 5.019323463125064
;;; Empirical kurtosis: 31.25740605088045
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 13
;;; Elapsed time: 3996.948785 ms
;;; Empirical mean: 38.00509565504615
;;; Empirical variance: 0.06607242048368657
;;; Empirical std: 0.2570455611048099
;;; Empirical skewness: 50.52654209024238
;;; Empirical kurtosis: 2554.330308091267
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 14
;;; Elapsed time: 3691.099556 ms
;;; Empirical mean: 38.10891052064068
;;; Empirical variance: 1.160206338724944
;;; Empirical std: 1.0771287475157945
;;; Empirical skewness: 6.530482482625755
;;; Empirical kurtosis: 64.25122911818116
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 15
;;; Elapsed time: 3253.069336 ms
;;; Empirical mean: 39.38139498492715
;;; Empirical variance: 2.287990175903774
;;; Empirical std: 1.5126103847004932
;;; Empirical skewness: -2.913223095382017
;;; Empirical kurtosis: 11.631172298525891
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 16
;;; Elapsed time: 2920.306883 ms
;;; Empirical mean: 40.99760157112929
;;; Empirical variance: 0.08063426389771376
;;; Empirical std: 0.2839617296357271
;;; Empirical skewness: -0.8522407907751386
;;; Empirical kurtosis: 517.7878337470046
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 17
;;; Elapsed time: 3050.404797 ms
;;; Empirical mean: 39.87781660333435
;;; Empirical variance: 12.107254614244894
;;; Empirical std: 3.4795480474114586
;;; Empirical skewness: -0.21716418235145352
;;; Empirical kurtosis: 1.0471602820963748
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 18
;;; Elapsed time: 2923.073206 ms
;;; Empirical mean: 38.13464759541736
;;; Empirical variance: 3.183074771008412
;;; Empirical std: 1.784117364695611
;;; Empirical skewness: 0.9043481348247524
;;; Empirical kurtosis: 1.8818246461847912
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 19
;;; Elapsed time: 3008.016989 ms
;;; Empirical mean: 39.36046334193492
;;; Empirical variance: 8.394718011801823
;;; Empirical std: 2.8973639764105963
;;; Empirical skewness: 1.2175711502576363
;;; Empirical kurtosis: 3.569454010137368
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 0
;;; Elapsed time: 1942.182173 ms
;;; Empirical mean: 32.17600000000002
;;; Empirical variance: 1001.4050239999999
;;; Empirical std: 31.644984183911358
;;; Empirical skewness: 0.6876254193601931
;;; Empirical kurtosis: 2.062389487145607
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 1
;;; Elapsed time: 2017.370573 ms
;;; Empirical mean: 54.03600000000002
;;; Empirical variance: 829.2427040000003
;;; Empirical std: 28.79657451850828
;;; Empirical skewness: 0.5526606929127551
;;; Empirical kurtosis: 1.9329207931115984
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 2
;;; Elapsed time: 1957.020042 ms
;;; Empirical mean: 63.01800000000002
;;; Empirical variance: 1065.9536760000003
;;; Empirical std: 32.64894601667871
;;; Empirical skewness: -0.6754234002742768
;;; Empirical kurtosis: 2.0660341438559415
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 3
;;; Elapsed time: 2379.787901 ms
;;; Empirical mean: 48.79
;;; Empirical variance: 813.1499000000002
;;; Empirical std: 28.515783348875413
;;; Empirical skewness: 0.006276788368055896
;;; Empirical kurtosis: 2.0134634690953312
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 4
;;; Elapsed time: 2015.777585 ms
;;; Empirical mean: 53.65200000000003
;;; Empirical variance: 642.5268960000001
;;; Empirical std: 25.348114249387468
;;; Empirical skewness: 0.11911504061045855
;;; Empirical kurtosis: 2.5875137584557186
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 5
;;; Elapsed time: 1903.795644 ms
;;; Empirical mean: 46.941999999999986
;;; Empirical variance: 798.2026360000002
;;; Empirical std: 28.252480174313906
;;; Empirical skewness: 0.3323573058166996
;;; Empirical kurtosis: 2.241985161080142
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 6
;;; Elapsed time: 1941.779024 ms
;;; Empirical mean: 69.608
;;; Empirical variance: 913.7703360000007
;;; Empirical std: 30.228634372065184
;;; Empirical skewness: -0.99146132475993
;;; Empirical kurtosis: 2.6834973952952823
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 7
;;; Elapsed time: 1955.445797 ms
;;; Empirical mean: 65.98999999999997
;;; Empirical variance: 687.2898999999998
;;; Empirical std: 26.21621444831423
;;; Empirical skewness: -0.4154230985289509
;;; Empirical kurtosis: 2.0892134082440763
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 8
;;; Elapsed time: 1873.858219 ms
;;; Empirical mean: 41.534000000000006
;;; Empirical variance: 635.6768440000002
;;; Empirical std: 25.21263262731602
;;; Empirical skewness: 0.4737484451781019
;;; Empirical kurtosis: 2.5106637764503312
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 9
;;; Elapsed time: 1894.408658 ms
;;; Empirical mean: 52.60999999999999
;;; Empirical variance: 1070.2179000000003
;;; Empirical std: 32.71418499672581
;;; Empirical skewness: 0.0936916842684195
;;; Empirical kurtosis: 1.6655416694422733
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 10
;;; Elapsed time: 1923.798937 ms
;;; Empirical mean: 50.47399999999999
;;; Empirical variance: 738.7293239999999
;;; Empirical std: 27.179575493373694
;;; Empirical skewness: 0.5632336683754128
;;; Empirical kurtosis: 2.4639963989451896
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 11
;;; Elapsed time: 1923.678079 ms
;;; Empirical mean: 49.307999999999986
;;; Empirical variance: 1293.3931359999997
;;; Empirical std: 35.963775330184674
;;; Empirical skewness: 0.3627939079694576
;;; Empirical kurtosis: 1.7588142819668673
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 12
;;; Elapsed time: 1898.183873 ms
;;; Empirical mean: 45.32800000000002
;;; Empirical variance: 1300.676416
;;; Empirical std: 36.06489173697878
;;; Empirical skewness: 0.05426823130207357
;;; Empirical kurtosis: 1.5063956364248074
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 13
;;; Elapsed time: 1997.659737 ms
;;; Empirical mean: 65.33200000000001
;;; Empirical variance: 913.8697760000005
;;; Empirical std: 30.230279125406707
;;; Empirical skewness: -0.700551860754656
;;; Empirical kurtosis: 2.436204068651874
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 14
;;; Elapsed time: 2030.747131 ms
;;; Empirical mean: 51.56799999999999
;;; Empirical variance: 721.065376
;;; Empirical std: 26.852660501335805
;;; Empirical skewness: -0.07018231039060159
;;; Empirical kurtosis: 2.423481834920815
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 15
;;; Elapsed time: 2341.498887 ms
;;; Empirical mean: 70.52400000000002
;;; Empirical variance: 729.881424
;;; Empirical std: 27.016317735768506
;;; Empirical skewness: -0.7531891607563711
;;; Empirical kurtosis: 2.791237340119277
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 16
;;; Elapsed time: 2482.344285 ms
;;; Empirical mean: 52.72200000000001
;;; Empirical variance: 967.880716
;;; Empirical std: 31.110781346665018
;;; Empirical skewness: -0.002259068485224448
;;; Empirical kurtosis: 1.7858800434982773
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 17
;;; Elapsed time: 2055.618688 ms
;;; Empirical mean: 46.568000000000005
;;; Empirical variance: 1320.5493760000002
;;; Empirical std: 36.339364001038874
;;; Empirical skewness: 0.21261375082516207
;;; Empirical kurtosis: 1.5777961807328769
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 18
;;; Elapsed time: 2795.486949 ms
;;; Empirical mean: 62.91199999999999
;;; Empirical variance: 1050.4122559999998
;;; Empirical std: 32.410064115950156
;;; Empirical skewness: -0.2130435231370659
;;; Empirical kurtosis: 1.9571639395893776
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 19
;;; Elapsed time: 2206.074568 ms
;;; Empirical mean: 43.14399999999999
;;; Empirical variance: 1051.043264
;;; Empirical std: 32.41979740837379
;;; Empirical skewness: 0.363618848110939
;;; Empirical kurtosis: 1.794440004550639
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 0
;;; Elapsed time: 3331.337845 ms
;;; Empirical mean: 39.309999999999995
;;; Empirical variance: 8.523899999999998
;;; Empirical std: 2.9195718864244458
;;; Empirical skewness: 0.47267204017712233
;;; Empirical kurtosis: 1.223418857565202
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 1
;;; Elapsed time: 3064.427155 ms
;;; Empirical mean: 43.55700000000001
;;; Empirical variance: 15.188750999999991
;;; Empirical std: 3.897274817099763
;;; Empirical skewness: -1.2472428400857303
;;; Empirical kurtosis: 3.0411802915376893
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 2
;;; Elapsed time: 3216.402321 ms
;;; Empirical mean: 44.924
;;; Empirical variance: 21.238223999999988
;;; Empirical std: 4.608494765104978
;;; Empirical skewness: -1.2675726072473732
;;; Empirical kurtosis: 3.0624783600508185
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 3
;;; Elapsed time: 3240.89212 ms
;;; Empirical mean: 39.5
;;; Empirical variance: 1.1960000000000008
;;; Empirical std: 1.0936178491593858
;;; Empirical skewness: 0.01605545158355842
;;; Empirical kurtosis: 5.33795762910929
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 4
;;; Elapsed time: 3201.040399 ms
;;; Empirical mean: 38.199
;;; Empirical variance: 0.15939899999999999
;;; Empirical std: 0.39924804320121593
;;; Empirical skewness: 1.5078345661336297
;;; Empirical kurtosis: 3.273565078827376
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 5
;;; Elapsed time: 3130.437567 ms
;;; Empirical mean: 44.775999999999996
;;; Empirical variance: 5.845823999999999
;;; Empirical std: 2.417813888619221
;;; Empirical skewness: -1.4690957053061204
;;; Empirical kurtosis: 3.1582421913488923
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 6
;;; Elapsed time: 3127.106144 ms
;;; Empirical mean: 39.949000000000005
;;; Empirical variance: 1.3063990000000005
;;; Empirical std: 1.1429781275247575
;;; Empirical skewness: -1.258375683140938
;;; Empirical kurtosis: 9.586628895380318
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 7
;;; Elapsed time: 3281.821241 ms
;;; Empirical mean: 37.071
;;; Empirical variance: 4.0999589999999975
;;; Empirical std: 2.0248355488779817
;;; Empirical skewness: 1.3993869958196388
;;; Empirical kurtosis: 2.995993268462684
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 8
;;; Elapsed time: 3102.40943 ms
;;; Empirical mean: 43.437999999999995
;;; Empirical variance: 42.742156
;;; Empirical std: 6.537748542120598
;;; Empirical skewness: -1.0825506784710306
;;; Empirical kurtosis: 2.586140639539086
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 9
;;; Elapsed time: 3122.142962 ms
;;; Empirical mean: 44.0
;;; Empirical variance: 0.0
;;; Empirical std: 0.0
;;; Empirical skewness: NaN
;;; Empirical kurtosis: NaN
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 10
;;; Elapsed time: 3203.966698 ms
;;; Empirical mean: 42.06
;;; Empirical variance: 0.3564000000000003
;;; Empirical std: 0.5969924622639723
;;; Empirical skewness: 9.84937058954026
;;; Empirical kurtosis: 98.01010101010074
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 11
;;; Elapsed time: 3494.779457 ms
;;; Empirical mean: 39.074
;;; Empirical variance: 7.440524000000007
;;; Empirical std: 2.7277323915663
;;; Empirical skewness: 1.547604082832866
;;; Empirical kurtosis: 5.125741349947828
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 12
;;; Elapsed time: 4352.161407 ms
;;; Empirical mean: 40.0
;;; Empirical variance: 0.0
;;; Empirical std: 0.0
;;; Empirical skewness: NaN
;;; Empirical kurtosis: NaN
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 13
;;; Elapsed time: 4155.472452 ms
;;; Empirical mean: 41.978
;;; Empirical variance: 0.6755160000000002
;;; Empirical std: 0.8218978038661499
;;; Empirical skewness: -2.6718148859029998
;;; Empirical kurtosis: 18.257942512353722
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 14
;;; Elapsed time: 3951.005431 ms
;;; Empirical mean: 42.944
;;; Empirical variance: 3.6048640000000005
;;; Empirical std: 1.898647939982555
;;; Empirical skewness: 1.9598293863215956
;;; Empirical kurtosis: 6.5761106779932055
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 15
;;; Elapsed time: 3307.053709 ms
;;; Empirical mean: 41.626000000000005
;;; Empirical variance: 0.608124
;;; Empirical std: 0.7798230568532839
;;; Empirical skewness: -1.6054924113837343
;;; Empirical kurtosis: 3.57760588301074
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 16
;;; Elapsed time: 2922.210755 ms
;;; Empirical mean: 43.842
;;; Empirical variance: 7.9750359999999985
;;; Empirical std: 2.8240106232094804
;;; Empirical skewness: 0.01607841338608365
;;; Empirical kurtosis: 1.0539690955189738
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 17
;;; Elapsed time: 2992.513531 ms
;;; Empirical mean: 40.400000000000006
;;; Empirical variance: 1.44
;;; Empirical std: 1.2
;;; Empirical skewness: -1.5000000000000142
;;; Empirical kurtosis: 3.2500000000000284
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 18
;;; Elapsed time: 2966.228509 ms
;;; Empirical mean: 42.345000000000006
;;; Empirical variance: 4.025975000000002
;;; Empirical std: 2.0064832418936374
;;; Empirical skewness: -1.246952673225076
;;; Empirical kurtosis: 4.35922270313324
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 19
;;; Elapsed time: 3044.474013 ms
;;; Empirical mean: 40.690000000000005
;;; Empirical variance: 19.593899999999994
;;; Empirical std: 4.426499745848857
;;; Empirical skewness: 0.3659776557130065
;;; Empirical kurtosis: 1.1339396444811893
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 0
;;; Elapsed time: 580.958179 ms
;;; Empirical mean: 54.21211863922403
;;; Empirical variance: 632.1520256284422
;;; Empirical std: 25.142633625546114
;;; Empirical skewness: 0.472044185039604
;;; Empirical kurtosis: 2.65384903782528
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 1
;;; Elapsed time: 587.699591 ms
;;; Empirical mean: 46.47956029187687
;;; Empirical variance: 955.4937814339519
;;; Empirical std: 30.91106244427635
;;; Empirical skewness: 0.21987260830502148
;;; Empirical kurtosis: 1.9002802262185534
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 2
;;; Elapsed time: 567.040855 ms
;;; Empirical mean: 42.77097636323245
;;; Empirical variance: 622.8630504864244
;;; Empirical std: 24.95722441471456
;;; Empirical skewness: 0.03353737621941512
;;; Empirical kurtosis: 2.077972960758438
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 3
;;; Elapsed time: 579.817006 ms
;;; Empirical mean: 64.37878787878789
;;; Empirical variance: 1369.2807621671263
;;; Empirical std: 37.00379388883154
;;; Empirical skewness: -0.2881966781528439
;;; Empirical kurtosis: 1.6106831624172322
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 4
;;; Elapsed time: 573.465952 ms
;;; Empirical mean: 51.445981105491185
;;; Empirical variance: 559.9179761234057
;;; Empirical std: 23.66258599822525
;;; Empirical skewness: 0.24131912532385735
;;; Empirical kurtosis: 2.4329754849855525
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 5
;;; Elapsed time: 596.465236 ms
;;; Empirical mean: 46.421082814281206
;;; Empirical variance: 654.4570406023814
;;; Empirical std: 25.582357995352606
;;; Empirical skewness: 0.4206986616363185
;;; Empirical kurtosis: 2.7659511319261676
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 6
;;; Elapsed time: 601.155298 ms
;;; Empirical mean: 49.554633626030444
;;; Empirical variance: 1082.7524829557503
;;; Empirical std: 32.90520449648886
;;; Empirical skewness: 0.37569161954268077
;;; Empirical kurtosis: 1.7112115759690103
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 7
;;; Elapsed time: 634.16854 ms
;;; Empirical mean: 55.779923829360335
;;; Empirical variance: 644.048072090283
;;; Empirical std: 25.378102216089424
;;; Empirical skewness: 0.09445901139946587
;;; Empirical kurtosis: 2.4602552494320125
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 8
;;; Elapsed time: 690.830762 ms
;;; Empirical mean: 63.1743904414407
;;; Empirical variance: 830.8892817079777
;;; Empirical std: 28.82515015933096
;;; Empirical skewness: -0.267487826502381
;;; Empirical kurtosis: 1.9124417349491611
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 9
;;; Elapsed time: 690.375792 ms
;;; Empirical mean: 51.609195461032904
;;; Empirical variance: 1053.5867319641043
;;; Empirical std: 32.45900078505351
;;; Empirical skewness: -0.09628305287998334
;;; Empirical kurtosis: 1.7093625522919234
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 10
;;; Elapsed time: 703.823286 ms
;;; Empirical mean: 55.87036318279852
;;; Empirical variance: 741.6878522749583
;;; Empirical std: 27.23394668928759
;;; Empirical skewness: 0.23675143927703124
;;; Empirical kurtosis: 2.251809881614904
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 11
;;; Elapsed time: 740.855481 ms
;;; Empirical mean: 51.7148310878002
;;; Empirical variance: 1106.1961499640054
;;; Empirical std: 33.25952720596018
;;; Empirical skewness: -0.04572112842396435
;;; Empirical kurtosis: 1.7043430842967353
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 12
;;; Elapsed time: 687.864734 ms
;;; Empirical mean: 67.17423817853641
;;; Empirical variance: 1160.287835594484
;;; Empirical std: 34.062998041782585
;;; Empirical skewness: -0.6121405834742264
;;; Empirical kurtosis: 2.1135983099575384
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 13
;;; Elapsed time: 825.755545 ms
;;; Empirical mean: 65.95842178545242
;;; Empirical variance: 953.2066766975088
;;; Empirical std: 30.874045356860975
;;; Empirical skewness: -0.28084603705205435
;;; Empirical kurtosis: 2.099573260467649
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 14
;;; Elapsed time: 642.208065 ms
;;; Empirical mean: 52.46183206107278
;;; Empirical variance: 768.7599935898746
;;; Empirical std: 27.726521483768472
;;; Empirical skewness: 0.3734046292737766
;;; Empirical kurtosis: 2.050573443045878
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 15
;;; Elapsed time: 667.684071 ms
;;; Empirical mean: 55.007575767667035
;;; Empirical variance: 483.1211543193476
;;; Empirical std: 21.980017159214132
;;; Empirical skewness: 0.8337409765534848
;;; Empirical kurtosis: 2.4036830524508366
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 16
;;; Elapsed time: 622.9116 ms
;;; Empirical mean: 54.49885315225246
;;; Empirical variance: 812.2091048570389
;;; Empirical std: 28.499282532320684
;;; Empirical skewness: -0.18325533058994656
;;; Empirical kurtosis: 1.6485947977665898
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 17
;;; Elapsed time: 701.337573 ms
;;; Empirical mean: 59.56830580784318
;;; Empirical variance: 747.0773102320292
;;; Empirical std: 27.332715017576085
;;; Empirical skewness: 0.2559315560993898
;;; Empirical kurtosis: 2.1867572259337256
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 18
;;; Elapsed time: 725.873079 ms
;;; Empirical mean: 67.87079419374125
;;; Empirical variance: 1010.6322000223363
;;; Empirical std: 31.790441960160546
;;; Empirical skewness: -0.17298769145071513
;;; Empirical kurtosis: 1.6717001709075061
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 19
;;; Elapsed time: 651.518751 ms
;;; Empirical mean: 46.669201520912544
;;; Empirical variance: 1107.2784050658533
;;; Empirical std: 33.2757930794422
;;; Empirical skewness: 0.2614098828513519
;;; Empirical kurtosis: 1.606891360387346
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 0
;;; Elapsed time: 14392.170291 ms
;;; Empirical mean: 47.026347296248595
;;; Empirical variance: 1.7614850751774758
;;; Empirical std: 1.3272095068893515
;;; Empirical skewness: 1.2368967066890317
;;; Empirical kurtosis: 6.220500050689361
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 1
;;; Elapsed time: 13377.647451 ms
;;; Empirical mean: 44.47433887019773
;;; Empirical variance: 3.92896941503317
;;; Empirical std: 1.9821628124433093
;;; Empirical skewness: -3.5056039275755446
;;; Empirical kurtosis: 13.289258897033092
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 2
;;; Elapsed time: 14589.408064 ms
;;; Empirical mean: 38.85504261485276
;;; Empirical variance: 12.574261771616566
;;; Empirical std: 3.5460205543138867
;;; Empirical skewness: 2.3700260789480785
;;; Empirical kurtosis: 7.099990362505602
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 3
;;; Elapsed time: 13933.031575 ms
;;; Empirical mean: 37.1294905322788
;;; Empirical variance: 2.1873997943143317
;;; Empirical std: 1.4789860696823116
;;; Empirical skewness: 0.659624680247525
;;; Empirical kurtosis: 2.033083931628
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 4
;;; Elapsed time: 13697.08097 ms
;;; Empirical mean: 40.69107395757294
;;; Empirical variance: 22.677323639243944
;;; Empirical std: 4.762071360158722
;;; Empirical skewness: -0.6696949604001988
;;; Empirical kurtosis: 1.5219569124872243
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 5
;;; Elapsed time: 13517.922421 ms
;;; Empirical mean: 40.91861363540811
;;; Empirical variance: 4.155543911589969
;;; Empirical std: 2.038515124199467
;;; Empirical skewness: 1.0010836869711812
;;; Empirical kurtosis: 5.680842849178332
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 6
;;; Elapsed time: 14553.524554 ms
;;; Empirical mean: 38.42001012555381
;;; Empirical variance: 9.403709031301334
;;; Empirical std: 3.06654676000568
;;; Empirical skewness: -0.10755482358144298
;;; Empirical kurtosis: 1.9000871925922707
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 7
;;; Elapsed time: 15336.332767 ms
;;; Empirical mean: 40.152878001927675
;;; Empirical variance: 6.423741273863368
;;; Empirical std: 2.5345100658437656
;;; Empirical skewness: 0.8000825643119159
;;; Empirical kurtosis: 1.7643039789223647
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 8
;;; Elapsed time: 15264.404167 ms
;;; Empirical mean: 41.80794152211094
;;; Empirical variance: 14.410068823069368
;;; Empirical std: 3.796059644298199
;;; Empirical skewness: -0.5967565628992341
;;; Empirical kurtosis: 3.3157121697210576
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 9
;;; Elapsed time: 14385.240832 ms
;;; Empirical mean: 38.22039261209645
;;; Empirical variance: 12.293087670622493
;;; Empirical std: 3.506149978341271
;;; Empirical skewness: 1.0332411048731776
;;; Empirical kurtosis: 2.2905028218890138
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 10
;;; Elapsed time: 13801.48842 ms
;;; Empirical mean: 43.300767846211365
;;; Empirical variance: 0.21030654889674236
;;; Empirical std: 0.4585919197900704
;;; Empirical skewness: 0.8688864552163899
;;; Empirical kurtosis: 1.754963672058495
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 11
;;; Elapsed time: 14423.817849 ms
;;; Empirical mean: 42.83295028321753
;;; Empirical variance: 35.67823867062606
;;; Empirical std: 5.9731263732342095
;;; Empirical skewness: -1.01309333975843
;;; Empirical kurtosis: 2.0708626541879123
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 12
;;; Elapsed time: 14040.026984 ms
;;; Empirical mean: 45.58650322820841
;;; Empirical variance: 1.4122077643831175
;;; Empirical std: 1.1883634815926976
;;; Empirical skewness: -6.809634344900446
;;; Empirical kurtosis: 54.936340059180566
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 13
;;; Elapsed time: 14389.992432 ms
;;; Empirical mean: 39.065765767505276
;;; Empirical variance: 2.6399271130151516
;;; Empirical std: 1.6247852513532832
;;; Empirical skewness: -0.2716954319547411
;;; Empirical kurtosis: 1.3572984025489276
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 14
;;; Elapsed time: 14152.180948 ms
;;; Empirical mean: 41.23644374774987
;;; Empirical variance: 3.8881971670739275
;;; Empirical std: 1.9718512030764206
;;; Empirical skewness: -0.5169634102444364
;;; Empirical kurtosis: 1.4767819945594773
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 15
;;; Elapsed time: 13628.546983 ms
;;; Empirical mean: 40.670982786773045
;;; Empirical variance: 17.0585393136229
;;; Empirical std: 4.130198459350701
;;; Empirical skewness: -0.216273376418348
;;; Empirical kurtosis: 1.6420124542872154
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 16
;;; Elapsed time: 13967.007462 ms
;;; Empirical mean: 39.93093445785894
;;; Empirical variance: 3.725527781903287
;;; Empirical std: 1.930162630946752
;;; Empirical skewness: 0.041758998577810146
;;; Empirical kurtosis: 5.771720175410509
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 17
;;; Elapsed time: 14549.459237 ms
;;; Empirical mean: 41.00000117492837
;;; Empirical variance: 1.4099139145972385E-5
;;; Empirical std: 0.0037548820415523555
;;; Empirical skewness: 3195.8387819773397
;;; Empirical kurtosis: 1.0213386520390395E7
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 18
;;; Elapsed time: 14001.363256 ms
;;; Empirical mean: 43.55352429561609
;;; Empirical variance: 16.033037967314428
;;; Empirical std: 4.004127616262302
;;; Empirical skewness: -1.0257037199693344
;;; Empirical kurtosis: 2.052068121158931
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 19
;;; Elapsed time: 14051.328049 ms
;;; Empirical mean: 37.5825889016227
;;; Empirical variance: 6.857280639911866
;;; Empirical std: 2.618640991031773
;;; Empirical skewness: 1.4605645478366645
;;; Empirical kurtosis: 4.544867704660625
;;; 
;;; Method :importance with options []
;;; Chain no. 0
;;; Elapsed time: 14830.771169 ms
;;; Empirical mean: 41.30452716871063
;;; Empirical variance: 5.278317328649171
;;; Empirical std: 2.2974588850835116
;;; Empirical skewness: 0.020746493149233887
;;; Empirical kurtosis: 4.599361453558087
;;; 
;;; Method :importance with options []
;;; Chain no. 1
;;; Elapsed time: 13720.591235 ms
;;; Empirical mean: 38.481067014832234
;;; Empirical variance: 10.929284461451656
;;; Empirical std: 3.3059468328228836
;;; Empirical skewness: 0.7889654415287691
;;; Empirical kurtosis: 2.4228232049688354
;;; 
;;; Method :importance with options []
;;; Chain no. 2
;;; Elapsed time: 14295.475849 ms
;;; Empirical mean: 39.75574080791897
;;; Empirical variance: 7.214665606670456
;;; Empirical std: 2.6860129572789586
;;; Empirical skewness: 0.22224698562858725
;;; Empirical kurtosis: 2.8616366050439113
;;; 
;;; Method :importance with options []
;;; Chain no. 3
;;; Elapsed time: 13829.26111 ms
;;; Empirical mean: 38.79550197982019
;;; Empirical variance: 4.641618397793844
;;; Empirical std: 2.154441551259594
;;; Empirical skewness: 0.9103565341669164
;;; Empirical kurtosis: 4.908020955795735
;;; 
;;; Method :importance with options []
;;; Chain no. 4
;;; Elapsed time: 12871.640173 ms
;;; Empirical mean: 40.60025131466398
;;; Empirical variance: 4.5036345080518725
;;; Empirical std: 2.1221768324180417
;;; Empirical skewness: -0.235663113781596
;;; Empirical kurtosis: 5.058247640675452
;;; 
;;; Method :importance with options []
;;; Chain no. 5
;;; Elapsed time: 13021.490394 ms
;;; Empirical mean: 38.08743262050709
;;; Empirical variance: 9.05957092015349
;;; Empirical std: 3.009912111699192
;;; Empirical skewness: 1.5356060523881299
;;; Empirical kurtosis: 4.842585638394252
;;; 
;;; Method :importance with options []
;;; Chain no. 6
;;; Elapsed time: 13072.828376 ms
;;; Empirical mean: 39.40664447656936
;;; Empirical variance: 2.5023319947772165
;;; Empirical std: 1.5818760996921397
;;; Empirical skewness: 1.1770586388185917
;;; Empirical kurtosis: 8.824992010362452
;;; 
;;; Method :importance with options []
;;; Chain no. 7
;;; Elapsed time: 13534.777622 ms
;;; Empirical mean: 40.39075500821413
;;; Empirical variance: 6.106554566111398
;;; Empirical std: 2.4711443839062497
;;; Empirical skewness: -0.23948387692901424
;;; Empirical kurtosis: 3.1931496614111885
;;; 
;;; Method :importance with options []
;;; Chain no. 8
;;; Elapsed time: 13068.942062 ms
;;; Empirical mean: 38.9338724339434
;;; Empirical variance: 10.389411917930492
;;; Empirical std: 3.223261068844795
;;; Empirical skewness: 1.305710991748238
;;; Empirical kurtosis: 3.385452571891094
;;; 
;;; Method :importance with options []
;;; Chain no. 9
;;; Elapsed time: 13104.306172 ms
;;; Empirical mean: 37.93879724995242
;;; Empirical variance: 9.890359412110028
;;; Empirical std: 3.1448941813851268
;;; Empirical skewness: 1.6862149402697215
;;; Empirical kurtosis: 4.816054619645115
;;; 
;;; Method :importance with options []
;;; Chain no. 10
;;; Elapsed time: 13356.955684 ms
;;; Empirical mean: 38.73801953088789
;;; Empirical variance: 10.828577642426556
;;; Empirical std: 3.2906804224091033
;;; Empirical skewness: 1.0610291155789107
;;; Empirical kurtosis: 2.862835901714975
;;; 
;;; Method :importance with options []
;;; Chain no. 11
;;; Elapsed time: 13283.41396 ms
;;; Empirical mean: 39.449454498984345
;;; Empirical variance: 6.453097114592044
;;; Empirical std: 2.5402946905018804
;;; Empirical skewness: 1.3920685246155904
;;; Empirical kurtosis: 4.754337371381311
;;; 
;;; Method :importance with options []
;;; Chain no. 12
;;; Elapsed time: 13118.853829 ms
;;; Empirical mean: 40.046361869446024
;;; Empirical variance: 4.411495033681554
;;; Empirical std: 2.100355930236957
;;; Empirical skewness: -0.0918864661955051
;;; Empirical kurtosis: 4.432623814708782
;;; 
;;; Method :importance with options []
;;; Chain no. 13
;;; Elapsed time: 12902.029371 ms
;;; Empirical mean: 38.85599426005884
;;; Empirical variance: 7.210586199253091
;;; Empirical std: 2.6852534702059487
;;; Empirical skewness: 0.361561401291251
;;; Empirical kurtosis: 2.311862312981401
;;; 
;;; Method :importance with options []
;;; Chain no. 14
;;; Elapsed time: 13228.305745 ms
;;; Empirical mean: 39.83144432152039
;;; Empirical variance: 2.0576573954054447
;;; Empirical std: 1.4344536923182445
;;; Empirical skewness: 1.582443439475935
;;; Empirical kurtosis: 10.926190060750445
;;; 
;;; Method :importance with options []
;;; Chain no. 15
;;; Elapsed time: 12957.325215 ms
;;; Empirical mean: 38.54254827541638
;;; Empirical variance: 8.376837225591073
;;; Empirical std: 2.894276632526869
;;; Empirical skewness: 1.517621636238683
;;; Empirical kurtosis: 4.448505660096202
;;; 
;;; Method :importance with options []
;;; Chain no. 16
;;; Elapsed time: 13494.082844 ms
;;; Empirical mean: 40.51352562653266
;;; Empirical variance: 3.4231500179087937
;;; Empirical std: 1.850175672175157
;;; Empirical skewness: 0.9077781813310825
;;; Empirical kurtosis: 5.924902447856892
;;; 
;;; Method :importance with options []
;;; Chain no. 17
;;; Elapsed time: 13205.600277 ms
;;; Empirical mean: 41.13133116264153
;;; Empirical variance: 3.650808653839887
;;; Empirical std: 1.9107089401161776
;;; Empirical skewness: -1.6406087255231727
;;; Empirical kurtosis: 9.445531860809655
;;; 
;;; Method :importance with options []
;;; Chain no. 18
;;; Elapsed time: 12846.531837 ms
;;; Empirical mean: 41.48028493571989
;;; Empirical variance: 8.315981694042787
;;; Empirical std: 2.8837443877782905
;;; Empirical skewness: -0.3494849270714318
;;; Empirical kurtosis: 2.837318344941234
;;; 
;;; Method :importance with options []
;;; Chain no. 19
;;; Elapsed time: 12893.721648 ms
;;; Empirical mean: 39.744215420385174
;;; Empirical variance: 8.42528116208375
;;; Empirical std: 2.902633487384129
;;; Empirical skewness: 0.5506025739294212
;;; Empirical kurtosis: 2.5259976261613537
;;; 
;;; Method :lmh with options []
;;; Chain no. 0
;;; Elapsed time: 10848.235323 ms
;;; Empirical mean: 39.85060000000001
;;; Empirical variance: 6.872279639999999
;;; Empirical std: 2.6215033168012583
;;; Empirical skewness: 0.48554807846719455
;;; Empirical kurtosis: 3.4478713328820474
;;; 
;;; Method :lmh with options []
;;; Chain no. 1
;;; Elapsed time: 10866.553274 ms
;;; Empirical mean: 39.7812
;;; Empirical variance: 6.829926559999999
;;; Empirical std: 2.61341281851911
;;; Empirical skewness: 0.3548260585849954
;;; Empirical kurtosis: 3.0308997980285364
;;; 
;;; Method :lmh with options []
;;; Chain no. 2
;;; Elapsed time: 11215.032896 ms
;;; Empirical mean: 39.895799999999994
;;; Empirical variance: 5.758342359999997
;;; Empirical std: 2.3996546334837427
;;; Empirical skewness: 0.5490788181507094
;;; Empirical kurtosis: 3.703130888749865
;;; 
;;; Method :lmh with options []
;;; Chain no. 3
;;; Elapsed time: 10819.350429 ms
;;; Empirical mean: 39.6661
;;; Empirical variance: 5.687810789999999
;;; Empirical std: 2.3849131619411215
;;; Empirical skewness: 0.1399642785871929
;;; Empirical kurtosis: 4.127769798510593
;;; 
;;; Method :lmh with options []
;;; Chain no. 4
;;; Elapsed time: 11960.24012 ms
;;; Empirical mean: 39.9257
;;; Empirical variance: 5.807579509999999
;;; Empirical std: 2.409892012103447
;;; Empirical skewness: 0.27550149839379423
;;; Empirical kurtosis: 3.6157592558134146
;;; 
;;; Method :lmh with options []
;;; Chain no. 5
;;; Elapsed time: 12999.697435 ms
;;; Empirical mean: 40.029799999999994
;;; Empirical variance: 7.0049119599999985
;;; Empirical std: 2.646679421463808
;;; Empirical skewness: 0.5270907681468466
;;; Empirical kurtosis: 3.312476866581243
;;; 
;;; Method :lmh with options []
;;; Chain no. 6
;;; Elapsed time: 12280.919522 ms
;;; Empirical mean: 39.698699999999995
;;; Empirical variance: 5.349118310000001
;;; Empirical std: 2.3128160994769993
;;; Empirical skewness: 0.40393724927174113
;;; Empirical kurtosis: 3.3739136897842226
;;; 
;;; Method :lmh with options []
;;; Chain no. 7
;;; Elapsed time: 10894.170509 ms
;;; Empirical mean: 40.10450000000001
;;; Empirical variance: 7.0571797499999995
;;; Empirical std: 2.6565352905617496
;;; Empirical skewness: 0.44136381257751905
;;; Empirical kurtosis: 3.0151528818869027
;;; 
;;; Method :lmh with options []
;;; Chain no. 8
;;; Elapsed time: 11359.253643 ms
;;; Empirical mean: 39.491
;;; Empirical variance: 4.994519000000003
;;; Empirical std: 2.234842052584478
;;; Empirical skewness: 0.4930050101267036
;;; Empirical kurtosis: 3.9161521523538765
;;; 
;;; Method :lmh with options []
;;; Chain no. 9
;;; Elapsed time: 13122.557207 ms
;;; Empirical mean: 39.28219999999999
;;; Empirical variance: 5.2261631600000005
;;; Empirical std: 2.2860803048012115
;;; Empirical skewness: 0.0802898078751161
;;; Empirical kurtosis: 4.086302185890683
;;; 
;;; Method :lmh with options []
;;; Chain no. 10
;;; Elapsed time: 12007.434993 ms
;;; Empirical mean: 39.6867
;;; Empirical variance: 6.143743110000001
;;; Empirical std: 2.4786575217242097
;;; Empirical skewness: 0.17758832130524865
;;; Empirical kurtosis: 3.21048087394343
;;; 
;;; Method :lmh with options []
;;; Chain no. 11
;;; Elapsed time: 11472.251828 ms
;;; Empirical mean: 39.870400000000004
;;; Empirical variance: 4.435403839999998
;;; Empirical std: 2.1060398476762017
;;; Empirical skewness: 0.20071022934907218
;;; Empirical kurtosis: 3.954151476352999
;;; 
;;; Method :lmh with options []
;;; Chain no. 12
;;; Elapsed time: 10908.84562 ms
;;; Empirical mean: 39.618900000000004
;;; Empirical variance: 5.621062789999995
;;; Empirical std: 2.3708780630812702
;;; Empirical skewness: 0.05672838934395952
;;; Empirical kurtosis: 3.0899060826630693
;;; 
;;; Method :lmh with options []
;;; Chain no. 13
;;; Elapsed time: 10948.902147 ms
;;; Empirical mean: 39.76349999999999
;;; Empirical variance: 5.429367749999999
;;; Empirical std: 2.330100373374503
;;; Empirical skewness: 0.32440704585141145
;;; Empirical kurtosis: 3.3696465449718978
;;; 
;;; Method :lmh with options []
;;; Chain no. 14
;;; Elapsed time: 12093.273488 ms
;;; Empirical mean: 39.6265
;;; Empirical variance: 5.521197749999998
;;; Empirical std: 2.3497229091958904
;;; Empirical skewness: -0.1853111112964426
;;; Empirical kurtosis: 3.5647272045666196
;;; 
;;; Method :lmh with options []
;;; Chain no. 15
;;; Elapsed time: 11316.8248 ms
;;; Empirical mean: 39.956699999999984
;;; Empirical variance: 6.193625109999998
;;; Empirical std: 2.4886994816570356
;;; Empirical skewness: 0.22183191911091366
;;; Empirical kurtosis: 3.7603941725024943
;;; 
;;; Method :lmh with options []
;;; Chain no. 16
;;; Elapsed time: 11709.036185 ms
;;; Empirical mean: 39.72969999999999
;;; Empirical variance: 4.998437909999999
;;; Empirical std: 2.235718656271401
;;; Empirical skewness: -0.5167844186768191
;;; Empirical kurtosis: 4.100967087981429
;;; 
;;; Method :lmh with options []
;;; Chain no. 17
;;; Elapsed time: 11821.955305 ms
;;; Empirical mean: 39.89359999999999
;;; Empirical variance: 5.355479039999998
;;; Empirical std: 2.314190795937102
;;; Empirical skewness: 0.08214509382969423
;;; Empirical kurtosis: 3.167619523947518
;;; 
;;; Method :lmh with options []
;;; Chain no. 18
;;; Elapsed time: 11553.191122 ms
;;; Empirical mean: 39.850899999999996
;;; Empirical variance: 5.376669189999998
;;; Empirical std: 2.31876458270347
;;; Empirical skewness: -0.32064871093770775
;;; Empirical kurtosis: 3.937332905711537
;;; 
;;; Method :lmh with options []
;;; Chain no. 19
;;; Elapsed time: 11425.290399 ms
;;; Empirical mean: 40.05359999999999
;;; Empirical variance: 6.017527039999998
;;; Empirical std: 2.4530648258861807
;;; Empirical skewness: 0.4246910516665907
;;; Empirical kurtosis: 3.6820728430653578
;;; 
;;; Method :rmh with options []
;;; Chain no. 0
;;; Elapsed time: 12353.584523 ms
;;; Empirical mean: 39.798899999999996
;;; Empirical variance: 3.9632587900000016
;;; Empirical std: 1.9907935076245356
;;; Empirical skewness: -0.3815617044503056
;;; Empirical kurtosis: 4.094008864102433
;;; 
;;; Method :rmh with options []
;;; Chain no. 1
;;; Elapsed time: 11747.441987 ms
;;; Empirical mean: 39.77410000000001
;;; Empirical variance: 6.272069190000002
;;; Empirical std: 2.5044099484708973
;;; Empirical skewness: 0.3291866753503294
;;; Empirical kurtosis: 3.209509688310376
;;; 
;;; Method :rmh with options []
;;; Chain no. 2
;;; Elapsed time: 12189.926921 ms
;;; Empirical mean: 39.81899999999999
;;; Empirical variance: 7.0370389999999965
;;; Empirical std: 2.652741789168331
;;; Empirical skewness: -0.28179311084089215
;;; Empirical kurtosis: 4.723194117338741
;;; 
;;; Method :rmh with options []
;;; Chain no. 3
;;; Elapsed time: 11431.133286 ms
;;; Empirical mean: 39.90410000000001
;;; Empirical variance: 4.3143031899999995
;;; Empirical std: 2.077090077488215
;;; Empirical skewness: 0.2399241037172183
;;; Empirical kurtosis: 3.7952875133217936
;;; 
;;; Method :rmh with options []
;;; Chain no. 4
;;; Elapsed time: 11127.539972 ms
;;; Empirical mean: 39.991299999999995
;;; Empirical variance: 4.330224310000003
;;; Empirical std: 2.0809191022238234
;;; Empirical skewness: 0.46409840857998513
;;; Empirical kurtosis: 5.120168007473683
;;; 
;;; Method :rmh with options []
;;; Chain no. 5
;;; Elapsed time: 12002.095601 ms
;;; Empirical mean: 40.00369999999998
;;; Empirical variance: 5.185886310000001
;;; Empirical std: 2.2772541162549254
;;; Empirical skewness: -0.07233646922856503
;;; Empirical kurtosis: 3.988980568973484
;;; 
;;; Method :rmh with options []
;;; Chain no. 6
;;; Elapsed time: 12015.3185 ms
;;; Empirical mean: 40.036300000000004
;;; Empirical variance: 3.712582310000001
;;; Empirical std: 1.926806246097412
;;; Empirical skewness: -0.17189636279584086
;;; Empirical kurtosis: 4.670989037374793
;;; 
;;; Method :rmh with options []
;;; Chain no. 7
;;; Elapsed time: 13618.575805 ms
;;; Empirical mean: 39.75940000000001
;;; Empirical variance: 5.78671164
;;; Empirical std: 2.4055584881686
;;; Empirical skewness: 0.39751923545235607
;;; Empirical kurtosis: 4.153220215193494
;;; 
;;; Method :rmh with options []
;;; Chain no. 8
;;; Elapsed time: 12737.408473 ms
;;; Empirical mean: 39.7137
;;; Empirical variance: 4.872532309999999
;;; Empirical std: 2.207381324103291
;;; Empirical skewness: 0.07274256633032349
;;; Empirical kurtosis: 4.347202938135719
;;; 
;;; Method :rmh with options []
;;; Chain no. 9
;;; Elapsed time: 14171.214052 ms
;;; Empirical mean: 40.088699999999996
;;; Empirical variance: 6.67163231
;;; Empirical std: 2.5829503111751877
;;; Empirical skewness: -0.015500902547841425
;;; Empirical kurtosis: 4.117404621622193
;;; 
;;; Method :rmh with options []
;;; Chain no. 10
;;; Elapsed time: 13946.514726 ms
;;; Empirical mean: 39.69250000000001
;;; Empirical variance: 6.9535437500000015
;;; Empirical std: 2.6369572901357357
;;; Empirical skewness: 0.37794204666373227
;;; Empirical kurtosis: 2.867553262714025
;;; 
;;; Method :rmh with options []
;;; Chain no. 11
;;; Elapsed time: 14205.65652 ms
;;; Empirical mean: 39.870900000000006
;;; Empirical variance: 4.992033189999998
;;; Empirical std: 2.234285834444644
;;; Empirical skewness: 0.41549401745540937
;;; Empirical kurtosis: 4.080559241928424
;;; 
;;; Method :rmh with options []
;;; Chain no. 12
;;; Elapsed time: 11923.818983 ms
;;; Empirical mean: 39.82300000000001
;;; Empirical variance: 4.862471000000001
;;; Empirical std: 2.2051011314676705
;;; Empirical skewness: -0.04602421247954255
;;; Empirical kurtosis: 3.4722855454938926
;;; 
;;; Method :rmh with options []
;;; Chain no. 13
;;; Elapsed time: 11477.460888 ms
;;; Empirical mean: 39.5492
;;; Empirical variance: 5.077579359999998
;;; Empirical std: 2.253348477266665
;;; Empirical skewness: 0.2709185942094582
;;; Empirical kurtosis: 3.3089730376155457
;;; 
;;; Method :rmh with options []
;;; Chain no. 14
;;; Elapsed time: 11797.19525 ms
;;; Empirical mean: 39.9137
;;; Empirical variance: 5.6186523100000025
;;; Empirical std: 2.3703696568257033
;;; Empirical skewness: 0.2618664757578076
;;; Empirical kurtosis: 2.986174234682819
;;; 
;;; Method :rmh with options []
;;; Chain no. 15
;;; Elapsed time: 10987.991967 ms
;;; Empirical mean: 40.020300000000006
;;; Empirical variance: 7.042687910000001
;;; Empirical std: 2.653806306044207
;;; Empirical skewness: 0.5398410889594668
;;; Empirical kurtosis: 3.1424139222739393
;;; 
;;; Method :rmh with options []
;;; Chain no. 16
;;; Elapsed time: 11333.186958 ms
;;; Empirical mean: 39.316600000000015
;;; Empirical variance: 4.3821644399999995
;;; Empirical std: 2.093361994495935
;;; Empirical skewness: 0.029551089801159492
;;; Empirical kurtosis: 2.8089221466674275
;;; 
;;; Method :rmh with options []
;;; Chain no. 17
;;; Elapsed time: 11264.398647 ms
;;; Empirical mean: 40.0445
;;; Empirical variance: 6.774719749999997
;;; Empirical std: 2.602829181871142
;;; Empirical skewness: 0.35103685960000536
;;; Empirical kurtosis: 3.361957511216026
;;; 
;;; Method :rmh with options []
;;; Chain no. 18
;;; Elapsed time: 11630.557941 ms
;;; Empirical mean: 39.8737
;;; Empirical variance: 6.129548310000001
;;; Empirical std: 2.475792461011222
;;; Empirical skewness: 0.3094463974002133
;;; Empirical kurtosis: 3.063040504406145
;;; 
;;; Method :rmh with options []
;;; Chain no. 19
;;; Elapsed time: 11634.155273 ms
;;; Empirical mean: 39.5492
;;; Empirical variance: 4.757979360000001
;;; Empirical std: 2.1812792943591615
;;; Empirical skewness: 0.29922518462540026
;;; Empirical kurtosis: 4.736411605752221
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 0
;;; Elapsed time: 11630.666545 ms
;;; Empirical mean: 39.63880000000001
;;; Empirical variance: 5.795934560000002
;;; Empirical std: 2.407474726762464
;;; Empirical skewness: -0.060027751729331974
;;; Empirical kurtosis: 3.1957923761652163
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 1
;;; Elapsed time: 11181.471189 ms
;;; Empirical mean: 39.82300000000001
;;; Empirical variance: 5.494871000000003
;;; Empirical std: 2.3441141183824654
;;; Empirical skewness: 0.248755536806404
;;; Empirical kurtosis: 3.8208234818866047
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 2
;;; Elapsed time: 11458.913569 ms
;;; Empirical mean: 39.533600000000014
;;; Empirical variance: 4.928471040000001
;;; Empirical std: 2.2200159999423428
;;; Empirical skewness: 0.5191503934349675
;;; Empirical kurtosis: 4.301319239441579
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 3
;;; Elapsed time: 12731.959686 ms
;;; Empirical mean: 39.8282
;;; Empirical variance: 6.085284760000003
;;; Empirical std: 2.4668369950201416
;;; Empirical skewness: 0.26217574795837567
;;; Empirical kurtosis: 3.3585341639451274
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 4
;;; Elapsed time: 14107.383931 ms
;;; Empirical mean: 40.1495
;;; Empirical variance: 8.01454975
;;; Empirical std: 2.8309980130688897
;;; Empirical skewness: 0.5113995274151575
;;; Empirical kurtosis: 2.926216764380118
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 5
;;; Elapsed time: 12340.964286 ms
;;; Empirical mean: 39.506299999999996
;;; Empirical variance: 6.06036031
;;; Empirical std: 2.461779906896634
;;; Empirical skewness: 0.6096700418277036
;;; Empirical kurtosis: 5.163679256180169
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 6
;;; Elapsed time: 12161.035208 ms
;;; Empirical mean: 39.546499999999995
;;; Empirical variance: 7.020237749999999
;;; Empirical std: 2.649573125995959
;;; Empirical skewness: 0.2918933569289464
;;; Empirical kurtosis: 3.320100220676344
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 7
;;; Elapsed time: 11817.452628 ms
;;; Empirical mean: 39.7768
;;; Empirical variance: 5.577581760000001
;;; Empirical std: 2.3616904454225156
;;; Empirical skewness: -0.02992075744992764
;;; Empirical kurtosis: 3.5866773326732626
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 8
;;; Elapsed time: 11411.692848 ms
;;; Empirical mean: 39.9105
;;; Empirical variance: 3.834489750000001
;;; Empirical std: 1.9581853206476656
;;; Empirical skewness: 0.04354681382523895
;;; Empirical kurtosis: 4.342940146580161
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 9
;;; Elapsed time: 11423.601717 ms
;;; Empirical mean: 39.903999999999996
;;; Empirical variance: 6.133184000000002
;;; Empirical std: 2.476526599897526
;;; Empirical skewness: 0.196697810410288
;;; Empirical kurtosis: 3.1439440341617773
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 10
;;; Elapsed time: 12578.056481 ms
;;; Empirical mean: 40.146600000000014
;;; Empirical variance: 5.843908440000004
;;; Empirical std: 2.417417721454032
;;; Empirical skewness: 0.2850464052779579
;;; Empirical kurtosis: 3.7054789468595977
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 11
;;; Elapsed time: 11683.23344 ms
;;; Empirical mean: 39.8898
;;; Empirical variance: 4.8990559599999965
;;; Empirical std: 2.21338111494609
;;; Empirical skewness: 0.22287740024962627
;;; Empirical kurtosis: 3.482032832412902
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 12
;;; Elapsed time: 11122.584649 ms
;;; Empirical mean: 39.72480000000002
;;; Empirical variance: 6.287664960000005
;;; Empirical std: 2.5075216768753976
;;; Empirical skewness: 0.3432564940897176
;;; Empirical kurtosis: 3.0602446189873325
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 13
;;; Elapsed time: 11280.68718 ms
;;; Empirical mean: 39.404799999999994
;;; Empirical variance: 5.699736959999997
;;; Empirical std: 2.387412188961093
;;; Empirical skewness: 0.1977500387361073
;;; Empirical kurtosis: 3.6321041865231183
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 14
;;; Elapsed time: 11375.061985 ms
;;; Empirical mean: 39.93750000000001
;;; Empirical variance: 5.762993750000001
;;; Empirical std: 2.4006236168962434
;;; Empirical skewness: 0.34682921233055064
;;; Empirical kurtosis: 4.388695814864905
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 15
;;; Elapsed time: 11607.522096 ms
;;; Empirical mean: 39.63999999999999
;;; Empirical variance: 6.7385999999999955
;;; Empirical std: 2.595881353220905
;;; Empirical skewness: 0.39473500779309
;;; Empirical kurtosis: 2.984038266636434
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 16
;;; Elapsed time: 11714.17892 ms
;;; Empirical mean: 39.92240000000001
;;; Empirical variance: 6.72277824
;;; Empirical std: 2.592832088662897
;;; Empirical skewness: 0.13757817839338699
;;; Empirical kurtosis: 2.827067883630434
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 17
;;; Elapsed time: 11553.591391 ms
;;; Empirical mean: 39.84020000000002
;;; Empirical variance: 6.79046396
;;; Empirical std: 2.6058518683916017
;;; Empirical skewness: -0.07880053091033976
;;; Empirical kurtosis: 3.8056588893073453
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 18
;;; Elapsed time: 12357.136708 ms
;;; Empirical mean: 39.8076
;;; Empirical variance: 6.304382240000001
;;; Empirical std: 2.5108528909515986
;;; Empirical skewness: 0.23739480049064002
;;; Empirical kurtosis: 3.980597804293084
;;; 
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 19
;;; Elapsed time: 11075.939204 ms
;;; Empirical mean: 39.946400000000004
;;; Empirical variance: 6.7391270400000005
;;; Empirical std: 2.5959828658910675
;;; Empirical skewness: 0.5566616641899167
;;; Empirical kurtosis: 3.097285048741224
;;; 
;;; Method :smc with options []
;;; Chain no. 0
;;; Elapsed time: 25486.580342 ms
;;; Empirical mean: 38.04624038036452
;;; Empirical variance: 7.78945377592818
;;; Empirical std: 2.790959293133488
;;; Empirical skewness: 0.8058933135638134
;;; Empirical kurtosis: 2.593768600454516
;;; 
;;; Method :smc with options []
;;; Chain no. 1
;;; Elapsed time: 25577.535297 ms
;;; Empirical mean: 37.511641527317295
;;; Empirical variance: 7.8384271883281995
;;; Empirical std: 2.7997191266854253
;;; Empirical skewness: 1.6196466431014318
;;; Empirical kurtosis: 5.217065320419389
;;; 
;;; Method :smc with options []
;;; Chain no. 2
;;; Elapsed time: 25115.74782 ms
;;; Empirical mean: 38.90561431436799
;;; Empirical variance: 4.089174742956667
;;; Empirical std: 2.0221707996498877
;;; Empirical skewness: 0.13708480604637424
;;; Empirical kurtosis: 3.144107210931941
;;; 
;;; Method :smc with options []
;;; Chain no. 3
;;; Elapsed time: 25304.412905 ms
;;; Empirical mean: 39.7258648062457
;;; Empirical variance: 8.565877873111988
;;; Empirical std: 2.9267521031190853
;;; Empirical skewness: 0.3675918648177527
;;; Empirical kurtosis: 3.069593840298994
;;; 
;;; Method :smc with options []
;;; Chain no. 4
;;; Elapsed time: 24944.502222 ms
;;; Empirical mean: 40.726656105575856
;;; Empirical variance: 5.327596482978843
;;; Empirical std: 2.308158678032956
;;; Empirical skewness: -0.6984750465355298
;;; Empirical kurtosis: 4.694621079041324
;;; 
;;; Method :smc with options []
;;; Chain no. 5
;;; Elapsed time: 25435.708275 ms
;;; Empirical mean: 38.920014499943065
;;; Empirical variance: 9.061795013945176
;;; Empirical std: 3.01028155061037
;;; Empirical skewness: 0.7095897139882709
;;; Empirical kurtosis: 2.8623912796152777
;;; 
;;; Method :smc with options []
;;; Chain no. 6
;;; Elapsed time: 25811.982008 ms
;;; Empirical mean: 40.072640563883354
;;; Empirical variance: 5.101265288712766
;;; Empirical std: 2.258598080383663
;;; Empirical skewness: -0.8975014470847819
;;; Empirical kurtosis: 3.351274501833644
;;; 
;;; Method :smc with options []
;;; Chain no. 7
;;; Elapsed time: 28519.285721 ms
;;; Empirical mean: 39.455148455189175
;;; Empirical variance: 4.801124454399453
;;; Empirical std: 2.1911468354264745
;;; Empirical skewness: 0.6961258314300793
;;; Empirical kurtosis: 3.9818273398238557
;;; 
;;; Method :smc with options []
;;; Chain no. 8
;;; Elapsed time: 27756.418079 ms
;;; Empirical mean: 39.47363848269589
;;; Empirical variance: 3.2737666598289064
;;; Empirical std: 1.8093553160805387
;;; Empirical skewness: 0.3493991613216018
;;; Empirical kurtosis: 5.771213410512966
;;; 
;;; Method :smc with options []
;;; Chain no. 9
;;; Elapsed time: 25810.502105 ms
;;; Empirical mean: 40.4674621560932
;;; Empirical variance: 4.8308730506380275
;;; Empirical std: 2.197924714506397
;;; Empirical skewness: 0.6185758498684483
;;; Empirical kurtosis: 5.890770222790424
;;; 
;;; Method :smc with options []
;;; Chain no. 10
;;; Elapsed time: 25185.310771 ms
;;; Empirical mean: 40.81068320257617
;;; Empirical variance: 2.4419272148770084
;;; Empirical std: 1.562666699868212
;;; Empirical skewness: -0.1243494890684471
;;; Empirical kurtosis: 11.578003017894217
;;; 
;;; Method :smc with options []
;;; Chain no. 11
;;; Elapsed time: 23169.994031 ms
;;; Empirical mean: 39.59201738244992
;;; Empirical variance: 6.368269978072347
;;; Empirical std: 2.5235431397288113
;;; Empirical skewness: 0.13566419265304325
;;; Empirical kurtosis: 3.3598943948899733
;;; 
;;; Method :smc with options []
;;; Chain no. 12
;;; Elapsed time: 22930.803203 ms
;;; Empirical mean: 40.513381670231155
;;; Empirical variance: 2.4539200808145134
;;; Empirical std: 1.5664993076329505
;;; Empirical skewness: -1.8858358668962776
;;; Empirical kurtosis: 7.727873173169481
;;; 
;;; Method :smc with options []
;;; Chain no. 13
;;; Elapsed time: 25458.948318 ms
;;; Empirical mean: 40.985756589120044
;;; Empirical variance: 1.9444896976480996
;;; Empirical std: 1.394449603839486
;;; Empirical skewness: -0.4417068504568649
;;; Empirical kurtosis: 14.172268705140878
;;; 
;;; Method :smc with options []
;;; Chain no. 14
;;; Elapsed time: 25149.767574 ms
;;; Empirical mean: 39.040008530894255
;;; Empirical variance: 13.939275262872764
;;; Empirical std: 3.7335338839861576
;;; Empirical skewness: 1.0196207384917477
;;; Empirical kurtosis: 2.6764622382153447
;;; 
;;; Method :smc with options []
;;; Chain no. 15
;;; Elapsed time: 23133.696698 ms
;;; Empirical mean: 38.28677962630841
;;; Empirical variance: 6.558250441676719
;;; Empirical std: 2.5609081283163437
;;; Empirical skewness: 0.5411365994326247
;;; Empirical kurtosis: 2.2973585800571517
;;; 
;;; Method :smc with options []
;;; Chain no. 16
;;; Elapsed time: 23449.852473 ms
;;; Empirical mean: 39.12534666635589
;;; Empirical variance: 4.754684191291491
;;; Empirical std: 2.1805238341489166
;;; Empirical skewness: 1.558958683720412
;;; Empirical kurtosis: 6.5576967923315514
;;; 
;;; Method :smc with options []
;;; Chain no. 17
;;; Elapsed time: 23067.9426 ms
;;; Empirical mean: 39.97794164646582
;;; Empirical variance: 2.5022739530640608
;;; Empirical std: 1.5818577537389575
;;; Empirical skewness: 0.20147874794984488
;;; Empirical kurtosis: 8.444335318977956
;;; 
;;; Method :smc with options []
;;; Chain no. 18
;;; Elapsed time: 22972.36331 ms
;;; Empirical mean: 40.72109880617462
;;; Empirical variance: 4.367131307214838
;;; Empirical std: 2.0897682424649004
;;; Empirical skewness: -0.7699541556846723
;;; Empirical kurtosis: 5.828338166791316
;;; 
;;; Method :smc with options []
;;; Chain no. 19
;;; Elapsed time: 23108.246592 ms
;;; Empirical mean: 38.76684813058828
;;; Empirical variance: 6.478333856310756
;;; Empirical std: 2.545257129704336
;;; Empirical skewness: 0.7248112016412359
;;; Empirical kurtosis: 2.699349408011682
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 0
;;; Elapsed time: 15604.173671 ms
;;; Empirical mean: 39.98432455609531
;;; Empirical variance: 5.061461541041158
;;; Empirical std: 2.2497692195070047
;;; Empirical skewness: 0.9929279698789198
;;; Empirical kurtosis: 6.90047775314572
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 1
;;; Elapsed time: 15754.670751 ms
;;; Empirical mean: 38.92260361877354
;;; Empirical variance: 5.742513789546838
;;; Empirical std: 2.396354270458948
;;; Empirical skewness: 0.5031982478218919
;;; Empirical kurtosis: 2.6927167340907503
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 2
;;; Elapsed time: 16000.541121 ms
;;; Empirical mean: 41.17136515039807
;;; Empirical variance: 5.624251384577243
;;; Empirical std: 2.371550417886418
;;; Empirical skewness: -0.7600256198432787
;;; Empirical kurtosis: 3.937752074086454
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 3
;;; Elapsed time: 15618.265894 ms
;;; Empirical mean: 40.718150870103706
;;; Empirical variance: 6.6721793410498496
;;; Empirical std: 2.5830562016823886
;;; Empirical skewness: 1.0286934033038524
;;; Empirical kurtosis: 2.4035534056079686
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 4
;;; Elapsed time: 15511.504301 ms
;;; Empirical mean: 39.652202724448735
;;; Empirical variance: 4.81965085545438
;;; Empirical std: 2.195370323078633
;;; Empirical skewness: 0.10005418723458336
;;; Empirical kurtosis: 3.27019559692661
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 5
;;; Elapsed time: 15655.954624 ms
;;; Empirical mean: 40.58314696109209
;;; Empirical variance: 3.2342240839240666
;;; Empirical std: 1.7983948631832962
;;; Empirical skewness: -0.11836678854518562
;;; Empirical kurtosis: 5.858182483416957
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 6
;;; Elapsed time: 15820.68909 ms
;;; Empirical mean: 40.848315699582606
;;; Empirical variance: 1.9670077258512282
;;; Empirical std: 1.4025005261500718
;;; Empirical skewness: 1.1256768252584954
;;; Empirical kurtosis: 11.682003054137143
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 7
;;; Elapsed time: 15595.782099 ms
;;; Empirical mean: 40.61036313373986
;;; Empirical variance: 5.058938944896462
;;; Empirical std: 2.2492085152107313
;;; Empirical skewness: 0.12085037359200797
;;; Empirical kurtosis: 4.223932251699915
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 8
;;; Elapsed time: 15531.676651 ms
;;; Empirical mean: 41.412445491712774
;;; Empirical variance: 6.379835947603013
;;; Empirical std: 2.5258337133712927
;;; Empirical skewness: -0.04449213222931086
;;; Empirical kurtosis: 3.4468412449496335
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 9
;;; Elapsed time: 15675.298986 ms
;;; Empirical mean: 40.444963205438555
;;; Empirical variance: 4.744228684359227
;;; Empirical std: 2.178125038733825
;;; Empirical skewness: 0.6519682332206574
;;; Empirical kurtosis: 4.071838803256936
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 10
;;; Elapsed time: 15848.276583 ms
;;; Empirical mean: 39.713118648972284
;;; Empirical variance: 11.490934482731058
;;; Empirical std: 3.3898280904392566
;;; Empirical skewness: 0.6816219495360502
;;; Empirical kurtosis: 2.9705300998627098
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 11
;;; Elapsed time: 15586.565309 ms
;;; Empirical mean: 39.30796108942164
;;; Empirical variance: 8.599535978241882
;;; Empirical std: 2.932496543602717
;;; Empirical skewness: -0.3614984382074134
;;; Empirical kurtosis: 3.7018035391715856
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 12
;;; Elapsed time: 15497.138758 ms
;;; Empirical mean: 40.346601071593845
;;; Empirical variance: 7.50367917410537
;;; Empirical std: 2.739284427383431
;;; Empirical skewness: -1.8176890437828294E-4
;;; Empirical kurtosis: 3.1771635459896372
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 13
;;; Elapsed time: 15648.632036 ms
;;; Empirical mean: 40.607729410879294
;;; Empirical variance: 3.8782187208935603
;;; Empirical std: 1.9693193547247638
;;; Empirical skewness: -0.6224492887696498
;;; Empirical kurtosis: 7.199776913574172
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 14
;;; Elapsed time: 15823.238846 ms
;;; Empirical mean: 39.18048336601203
;;; Empirical variance: 6.880208092126884
;;; Empirical std: 2.623015076610671
;;; Empirical skewness: 0.4644188977341657
;;; Empirical kurtosis: 1.947103309921399
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 15
;;; Elapsed time: 15800.63552 ms
;;; Empirical mean: 40.53165591211152
;;; Empirical variance: 4.891215483798542
;;; Empirical std: 2.21160925206026
;;; Empirical skewness: 0.9184655277585585
;;; Empirical kurtosis: 4.767631769017472
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 16
;;; Elapsed time: 15520.825499 ms
;;; Empirical mean: 39.67137044128034
;;; Empirical variance: 11.31135449689953
;;; Empirical std: 3.3632357183075245
;;; Empirical skewness: 0.6670932030327844
;;; Empirical kurtosis: 2.54768792455868
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 17
;;; Elapsed time: 15594.134528 ms
;;; Empirical mean: 39.338559784215846
;;; Empirical variance: 7.0757219379142615
;;; Empirical std: 2.660022920561825
;;; Empirical skewness: 1.0970682964906313
;;; Empirical kurtosis: 4.342421209893453
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 18
;;; Elapsed time: 15697.702991 ms
;;; Empirical mean: 38.52907207974418
;;; Empirical variance: 7.311544464605029
;;; Empirical std: 2.703986772268871
;;; Empirical skewness: 0.564790079704153
;;; Empirical kurtosis: 3.1933016931837486
;;; 
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 19
;;; Elapsed time: 15651.51171 ms
;;; Empirical mean: 40.09492148833437
;;; Empirical variance: 4.05805653373374
;;; Empirical std: 2.014461847177489
;;; Empirical skewness: 1.4934539241570972
;;; Empirical kurtosis: 6.732480177436695
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 0
;;; Elapsed time: 17336.892385 ms
;;; Empirical mean: 55.843800000000016
;;; Empirical variance: 937.7042015600008
;;; Empirical std: 30.621956200739376
;;; Empirical skewness: 0.10232328891904806
;;; Empirical kurtosis: 1.8645560231735676
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 1
;;; Elapsed time: 17392.623194 ms
;;; Empirical mean: 61.09339999999997
;;; Empirical variance: 1100.53787644
;;; Empirical std: 33.17435570497187
;;; Empirical skewness: -0.15893103645015705
;;; Empirical kurtosis: 1.700722119112094
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 2
;;; Elapsed time: 17545.632793 ms
;;; Empirical mean: 54.39239999999998
;;; Empirical variance: 987.62442224
;;; Empirical std: 31.426492362972994
;;; Empirical skewness: -0.023302464045104058
;;; Empirical kurtosis: 1.8819119554818808
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 3
;;; Elapsed time: 17557.642667 ms
;;; Empirical mean: 61.71700000000003
;;; Empirical variance: 942.310111
;;; Empirical std: 30.69707007191403
;;; Empirical skewness: -0.22496891262164542
;;; Empirical kurtosis: 1.769608117837333
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 4
;;; Elapsed time: 17446.384723 ms
;;; Empirical mean: 60.382200000000005
;;; Empirical variance: 877.7449231600004
;;; Empirical std: 29.626760254202626
;;; Empirical skewness: -0.28600112195942456
;;; Empirical kurtosis: 2.0416778051889684
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 5
;;; Elapsed time: 17565.099696 ms
;;; Empirical mean: 58.3663
;;; Empirical variance: 1000.85672431
;;; Empirical std: 31.63631970236108
;;; Empirical skewness: -0.24114374586038795
;;; Empirical kurtosis: 1.9027636602597282
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 6
;;; Elapsed time: 17755.954141 ms
;;; Empirical mean: 53.79299999999997
;;; Empirical variance: 1080.5541510000005
;;; Empirical std: 32.87178350804837
;;; Empirical skewness: 0.0644318753710761
;;; Empirical kurtosis: 1.7964280813429752
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 7
;;; Elapsed time: 17246.754055 ms
;;; Empirical mean: 58.06319999999995
;;; Empirical variance: 1059.3260057599985
;;; Empirical std: 32.54728876204589
;;; Empirical skewness: -0.10007756564114037
;;; Empirical kurtosis: 1.7408191073486379
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 8
;;; Elapsed time: 17343.861873 ms
;;; Empirical mean: 54.76819999999999
;;; Empirical variance: 999.6860687599999
;;; Empirical std: 31.617812523322986
;;; Empirical skewness: -0.014648257164907596
;;; Empirical kurtosis: 1.905997935009402
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 9
;;; Elapsed time: 17328.626993 ms
;;; Empirical mean: 60.29359999999999
;;; Empirical variance: 1052.5057990399996
;;; Empirical std: 32.4423457696881
;;; Empirical skewness: -0.11896548111697688
;;; Empirical kurtosis: 1.7407877815433972
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 10
;;; Elapsed time: 17575.18604 ms
;;; Empirical mean: 58.02920000000004
;;; Empirical variance: 824.6427473600005
;;; Empirical std: 28.71659358907321
;;; Empirical skewness: -0.11708015924872554
;;; Empirical kurtosis: 2.0955577980702453
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 11
;;; Elapsed time: 17302.210299 ms
;;; Empirical mean: 60.543399999999956
;;; Empirical variance: 959.0701164400001
;;; Empirical std: 30.968857202680244
;;; Empirical skewness: -0.2333269008927174
;;; Empirical kurtosis: 1.8852822503745237
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 12
;;; Elapsed time: 17531.539713 ms
;;; Empirical mean: 56.985199999999985
;;; Empirical variance: 1021.19458096
;;; Empirical std: 31.956135263201023
;;; Empirical skewness: -0.18811995402180753
;;; Empirical kurtosis: 1.9228205894685493
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 13
;;; Elapsed time: 17663.17384 ms
;;; Empirical mean: 57.22459999999999
;;; Empirical variance: 781.0769548400004
;;; Empirical std: 27.947754021387844
;;; Empirical skewness: 0.03874065647147103
;;; Empirical kurtosis: 2.120637646668223
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 14
;;; Elapsed time: 17533.342916 ms
;;; Empirical mean: 60.61320000000001
;;; Empirical variance: 884.5527857599999
;;; Empirical std: 29.74143214036607
;;; Empirical skewness: -0.305217704755566
;;; Empirical kurtosis: 1.9809895107504616
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 15
;;; Elapsed time: 17235.43089 ms
;;; Empirical mean: 58.82199999999999
;;; Empirical variance: 1099.9619160000007
;;; Empirical std: 33.165673760682154
;;; Empirical skewness: -0.20733383533886546
;;; Empirical kurtosis: 1.9067748265758888
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 16
;;; Elapsed time: 17368.224459 ms
;;; Empirical mean: 54.88259999999998
;;; Empirical variance: 954.0212172399999
;;; Empirical std: 30.887233887805493
;;; Empirical skewness: -0.22232974199796895
;;; Empirical kurtosis: 1.8457365851384901
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 17
;;; Elapsed time: 17769.302584 ms
;;; Empirical mean: 57.91040000000001
;;; Empirical variance: 1139.6083718400002
;;; Empirical std: 33.75808602157415
;;; Empirical skewness: -0.23353224887374566
;;; Empirical kurtosis: 1.8540947717299177
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 18
;;; Elapsed time: 17392.671915 ms
;;; Empirical mean: 58.33879999999999
;;; Empirical variance: 989.6984145600003
;;; Empirical std: 31.459472572819784
;;; Empirical skewness: -0.09359989214411771
;;; Empirical kurtosis: 1.906177451717974
;;; 
;;; Method :pgibbs with options []
;;; Chain no. 19
;;; Elapsed time: 17441.233554 ms
;;; Empirical mean: 52.52180000000001
;;; Empirical variance: 992.4819247599994
;;; Empirical std: 31.503681130306017
;;; Empirical skewness: 0.08488250887588333
;;; Empirical kurtosis: 1.810097673303189
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 0
;;; Elapsed time: 16075.90594 ms
;;; Empirical mean: 40.9717
;;; Empirical variance: 0.4606991099999997
;;; Empirical std: 0.6787481933677612
;;; Empirical skewness: -2.4021974116372284
;;; Empirical kurtosis: 50.228370921056666
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 1
;;; Elapsed time: 16315.984497 ms
;;; Empirical mean: 37.36039999999999
;;; Empirical variance: 1.30091184
;;; Empirical std: 1.1405752232974378
;;; Empirical skewness: 3.068931007303446
;;; Empirical kurtosis: 11.947706809205139
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 2
;;; Elapsed time: 16012.028596 ms
;;; Empirical mean: 37.962
;;; Empirical variance: 6.362156000000008
;;; Empirical std: 2.5223314611684184
;;; Empirical skewness: 2.066814795598518
;;; Empirical kurtosis: 7.057017000888199
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 3
;;; Elapsed time: 16198.537679 ms
;;; Empirical mean: 36.180099999999996
;;; Empirical variance: 0.9620639899999996
;;; Empirical std: 0.9808486070745065
;;; Empirical skewness: 7.022234378271102
;;; Empirical kurtosis: 61.84026556018962
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 4
;;; Elapsed time: 16080.358297 ms
;;; Empirical mean: 42.15049999999999
;;; Empirical variance: 9.028049749999994
;;; Empirical std: 3.004671321459303
;;; Empirical skewness: 0.07798059249595767
;;; Empirical kurtosis: 2.056261969704086
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 5
;;; Elapsed time: 16494.952016 ms
;;; Empirical mean: 37.9769
;;; Empirical variance: 12.137766390000017
;;; Empirical std: 3.483929733792003
;;; Empirical skewness: 1.2968547576031721
;;; Empirical kurtosis: 3.1378592124062386
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 6
;;; Elapsed time: 15946.540599 ms
;;; Empirical mean: 38.7138
;;; Empirical variance: 6.624289560000013
;;; Empirical std: 2.5737695234810776
;;; Empirical skewness: 0.9476264804778955
;;; Empirical kurtosis: 3.6674787395349195
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 7
;;; Elapsed time: 16025.742937 ms
;;; Empirical mean: 40.678200000000004
;;; Empirical variance: 5.514444759999998
;;; Empirical std: 2.348285493716639
;;; Empirical skewness: -1.3802883414608067
;;; Empirical kurtosis: 3.427366225943722
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 8
;;; Elapsed time: 16056.969546 ms
;;; Empirical mean: 40.49550000000001
;;; Empirical variance: 0.769979750000004
;;; Empirical std: 0.8774849001549849
;;; Empirical skewness: 0.9381690844287264
;;; Empirical kurtosis: 32.38661843006403
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 9
;;; Elapsed time: 16638.554431 ms
;;; Empirical mean: 39.59450000000001
;;; Empirical variance: 14.381269750000024
;;; Empirical std: 3.792264462033209
;;; Empirical skewness: 0.8419660835640418
;;; Empirical kurtosis: 2.116011513740944
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 10
;;; Elapsed time: 16055.95674 ms
;;; Empirical mean: 40.644099999999995
;;; Empirical variance: 4.363635190000013
;;; Empirical std: 2.088931590550541
;;; Empirical skewness: -1.170265956869183
;;; Empirical kurtosis: 4.3796918165640495
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 11
;;; Elapsed time: 15909.590658 ms
;;; Empirical mean: 39.8542
;;; Empirical variance: 0.9043423600000003
;;; Empirical std: 0.9509691687957083
;;; Empirical skewness: -4.159923495333825
;;; Empirical kurtosis: 28.333561144335984
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 12
;;; Elapsed time: 19908.831557 ms
;;; Empirical mean: 41.951600000000006
;;; Empirical variance: 9.641657440000005
;;; Empirical std: 3.1051018405198896
;;; Empirical skewness: -0.08179207397049726
;;; Empirical kurtosis: 2.0118596266018534
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 13
;;; Elapsed time: 20569.712278 ms
;;; Empirical mean: 39.2179
;;; Empirical variance: 1.1472195900000008
;;; Empirical std: 1.0710833721050854
;;; Empirical skewness: -0.14940448739603165
;;; Empirical kurtosis: 10.560822991063135
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 14
;;; Elapsed time: 20778.050834 ms
;;; Empirical mean: 41.287600000000005
;;; Empirical variance: 3.159086240000006
;;; Empirical std: 1.7773818498004321
;;; Empirical skewness: -0.325178944894886
;;; Empirical kurtosis: 5.636693003223963
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 15
;;; Elapsed time: 21877.359422 ms
;;; Empirical mean: 40.51089999999999
;;; Empirical variance: 2.5532811900000127
;;; Empirical std: 1.5978989924272475
;;; Empirical skewness: 1.4519357212083643
;;; Empirical kurtosis: 8.265513341877543
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 16
;;; Elapsed time: 21011.021037 ms
;;; Empirical mean: 41.07850000000001
;;; Empirical variance: 10.515737750000007
;;; Empirical std: 3.242797827494031
;;; Empirical skewness: 0.018115911085198576
;;; Empirical kurtosis: 1.4545487860626505
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 17
;;; Elapsed time: 19460.22252 ms
;;; Empirical mean: 36.7587
;;; Empirical variance: 2.892474310000009
;;; Empirical std: 1.7007275825363712
;;; Empirical skewness: 3.2402124053276005
;;; Empirical kurtosis: 13.453196441257194
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 18
;;; Elapsed time: 21728.828314 ms
;;; Empirical mean: 38.815200000000004
;;; Empirical variance: 7.241648960000001
;;; Empirical std: 2.691031207548512
;;; Empirical skewness: 0.20975910680553542
;;; Empirical kurtosis: 2.4251182362749986
;;; 
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 19
;;; Elapsed time: 23487.81365 ms
;;; Empirical mean: 38.400499999999994
;;; Empirical variance: 2.624899750000009
;;; Empirical std: 1.6201542364849122
;;; Empirical skewness: -0.6392480573450529
;;; Empirical kurtosis: 5.048417258481066
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 0
;;; Elapsed time: 5824.420356 ms
;;; Empirical mean: 60.281446584992906
;;; Empirical variance: 815.1999314985488
;;; Empirical std: 28.551706279985243
;;; Empirical skewness: -0.1819058829069157
;;; Empirical kurtosis: 2.0289495925251018
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 1
;;; Elapsed time: 6168.677269 ms
;;; Empirical mean: 48.29272594610388
;;; Empirical variance: 719.7712520241164
;;; Empirical std: 26.828552924526445
;;; Empirical skewness: 0.14318493223354609
;;; Empirical kurtosis: 1.842808251083635
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 2
;;; Elapsed time: 6019.729274 ms
;;; Empirical mean: 56.02527348170423
;;; Empirical variance: 604.6923073092281
;;; Empirical std: 24.59049221364282
;;; Empirical skewness: 0.21650944997216615
;;; Empirical kurtosis: 2.10081834927198
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 3
;;; Elapsed time: 6705.848352 ms
;;; Empirical mean: 56.31261125930219
;;; Empirical variance: 1045.3322780285935
;;; Empirical std: 32.33159875460218
;;; Empirical skewness: -0.08268838467863979
;;; Empirical kurtosis: 2.0000701701098365
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 4
;;; Elapsed time: 6441.348047 ms
;;; Empirical mean: 47.27348878660382
;;; Empirical variance: 998.1218399632336
;;; Empirical std: 31.593066327332547
;;; Empirical skewness: 0.5768623728249034
;;; Empirical kurtosis: 2.574809542957921
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 5
;;; Elapsed time: 6615.941656 ms
;;; Empirical mean: 55.54282142178258
;;; Empirical variance: 945.6004151413503
;;; Empirical std: 30.75061650018338
;;; Empirical skewness: 0.09142206062738455
;;; Empirical kurtosis: 2.0349870335487172
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 6
;;; Elapsed time: 7107.84176 ms
;;; Empirical mean: 64.37433959740397
;;; Empirical variance: 610.3398684172981
;;; Empirical std: 24.70505754733832
;;; Empirical skewness: -0.261441518381986
;;; Empirical kurtosis: 1.9230899610957177
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 7
;;; Elapsed time: 6336.480338 ms
;;; Empirical mean: 55.63404774324989
;;; Empirical variance: 875.5619308039416
;;; Empirical std: 29.58989575520572
;;; Empirical skewness: -0.26943526676004675
;;; Empirical kurtosis: 2.2593883657726397
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 8
;;; Elapsed time: 6711.813015 ms
;;; Empirical mean: 65.47308567096243
;;; Empirical variance: 871.1924143603586
;;; Empirical std: 29.515968802672877
;;; Empirical skewness: -0.07686027386337366
;;; Empirical kurtosis: 1.9327232868719146
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 9
;;; Elapsed time: 6403.475851 ms
;;; Empirical mean: 70.62499867058102
;;; Empirical variance: 521.9844114440223
;;; Empirical std: 22.846978168764952
;;; Empirical skewness: -0.12307082983144105
;;; Empirical kurtosis: 1.5417326530045168
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 10
;;; Elapsed time: 7005.630674 ms
;;; Empirical mean: 47.746935881755384
;;; Empirical variance: 869.6624857144577
;;; Empirical std: 29.490040449522237
;;; Empirical skewness: 0.3051404803850723
;;; Empirical kurtosis: 2.282657849057735
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 11
;;; Elapsed time: 6459.98691 ms
;;; Empirical mean: 70.7499999912118
;;; Empirical variance: 768.5625000580794
;;; Empirical std: 27.72295979974143
;;; Empirical skewness: -0.25408229080421585
;;; Empirical kurtosis: 1.9839134106005505
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 12
;;; Elapsed time: 7369.240934 ms
;;; Empirical mean: 49.88804126453903
;;; Empirical variance: 661.7495012452584
;;; Empirical std: 25.724492244653895
;;; Empirical skewness: 0.5586833750590654
;;; Empirical kurtosis: 2.5447866367747505
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 13
;;; Elapsed time: 6563.190119 ms
;;; Empirical mean: 61.67771913320431
;;; Empirical variance: 860.2077982176812
;;; Empirical std: 29.32929931344561
;;; Empirical skewness: -0.4387718548413363
;;; Empirical kurtosis: 2.0486785620427876
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 14
;;; Elapsed time: 7356.953332 ms
;;; Empirical mean: 63.06145288835097
;;; Empirical variance: 493.21748408933297
;;; Empirical std: 22.208500266549585
;;; Empirical skewness: 0.04345080225757841
;;; Empirical kurtosis: 1.9550568362976462
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 15
;;; Elapsed time: 7355.592467 ms
;;; Empirical mean: 60.31045890332771
;;; Empirical variance: 506.10919609204393
;;; Empirical std: 22.496870806670955
;;; Empirical skewness: -0.06229141442856113
;;; Empirical kurtosis: 2.1220197537185648
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 16
;;; Elapsed time: 6883.389901 ms
;;; Empirical mean: 64.1754124668958
;;; Empirical variance: 1216.9027274946352
;;; Empirical std: 34.88413289010686
;;; Empirical skewness: -0.3740224828968575
;;; Empirical kurtosis: 1.555370696757266
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 17
;;; Elapsed time: 7351.934847 ms
;;; Empirical mean: 68.66558688568996
;;; Empirical variance: 720.9482991784034
;;; Empirical std: 26.85048042732948
;;; Empirical skewness: -0.3208177630418429
;;; Empirical kurtosis: 1.7279555672814169
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 18
;;; Elapsed time: 7811.127015 ms
;;; Empirical mean: 55.84931536944343
;;; Empirical variance: 1067.0334000188384
;;; Empirical std: 32.665477189516736
;;; Empirical skewness: 0.2778437366775837
;;; Empirical kurtosis: 1.9225697618101514
;;; 
;;; Method :ipmcmc with options []
;;; Chain no. 19
;;; Elapsed time: 7585.947396 ms
;;; Empirical mean: 54.61757144995832
;;; Empirical variance: 790.8852018270409
;;; Empirical std: 28.122681270231702
;;; Empirical skewness: -0.20001857661269679
;;; Empirical kurtosis: 2.188225495528678
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 0
;;; Elapsed time: 14830.068004 ms
;;; Empirical mean: 41.00489625349648
;;; Empirical variance: 17.473561315259346
;;; Empirical std: 4.180138911000369
;;; Empirical skewness: -0.5488477179537373
;;; Empirical kurtosis: 2.610113028229513
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 1
;;; Elapsed time: 14637.65496 ms
;;; Empirical mean: 38.58072321983462
;;; Empirical variance: 8.437356592940016
;;; Empirical std: 2.9047128245215594
;;; Empirical skewness: 0.6504744901997508
;;; Empirical kurtosis: 2.908557675533565
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 2
;;; Elapsed time: 15230.857157 ms
;;; Empirical mean: 40.76150797365305
;;; Empirical variance: 13.20995780695778
;;; Empirical std: 3.6345505646445173
;;; Empirical skewness: 0.5934370206597968
;;; Empirical kurtosis: 1.7306121935027128
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 3
;;; Elapsed time: 14535.409505 ms
;;; Empirical mean: 39.2179118015477
;;; Empirical variance: 14.820194202696706
;;; Empirical std: 3.849700534158041
;;; Empirical skewness: 0.30545551730021153
;;; Empirical kurtosis: 1.5783158725809594
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 4
;;; Elapsed time: 15257.869006 ms
;;; Empirical mean: 39.62908217015029
;;; Empirical variance: 6.311725830783119
;;; Empirical std: 2.5123148351237985
;;; Empirical skewness: 0.2178840479557149
;;; Empirical kurtosis: 2.1831779583320943
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 5
;;; Elapsed time: 14800.991562 ms
;;; Empirical mean: 41.58674400342772
;;; Empirical variance: 4.63039299845145
;;; Empirical std: 2.151834798131922
;;; Empirical skewness: 0.9659153406534497
;;; Empirical kurtosis: 5.7079150701169254
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 6
;;; Elapsed time: 15627.358505 ms
;;; Empirical mean: 39.35645969532735
;;; Empirical variance: 12.402972909892545
;;; Empirical std: 3.521785471872548
;;; Empirical skewness: 0.6686202511943851
;;; Empirical kurtosis: 2.9765551128635397
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 7
;;; Elapsed time: 14448.981123 ms
;;; Empirical mean: 41.6236753353949
;;; Empirical variance: 3.6481471653774653
;;; Empirical std: 1.9100123469175443
;;; Empirical skewness: 1.1992274883173017
;;; Empirical kurtosis: 4.915365970928274
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 8
;;; Elapsed time: 14746.007328 ms
;;; Empirical mean: 38.851743457585926
;;; Empirical variance: 5.8630387917972895
;;; Empirical std: 2.4213712626933708
;;; Empirical skewness: -0.08743212489292357
;;; Empirical kurtosis: 6.1437115266933535
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 9
;;; Elapsed time: 15802.894484 ms
;;; Empirical mean: 40.055812150635404
;;; Empirical variance: 7.2955938265927855
;;; Empirical std: 2.701035695171907
;;; Empirical skewness: -0.3470573903041154
;;; Empirical kurtosis: 3.0219017418013268
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 10
;;; Elapsed time: 16111.316103 ms
;;; Empirical mean: 38.54125824574104
;;; Empirical variance: 7.5473915470998945
;;; Empirical std: 2.7472516351983294
;;; Empirical skewness: 0.2862460375667739
;;; Empirical kurtosis: 1.9525118162606245
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 11
;;; Elapsed time: 15930.344093 ms
;;; Empirical mean: 39.055765693815005
;;; Empirical variance: 8.30445102776508
;;; Empirical std: 2.8817444417860996
;;; Empirical skewness: 1.4227840721770861
;;; Empirical kurtosis: 4.464605329214598
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 12
;;; Elapsed time: 18118.844885 ms
;;; Empirical mean: 39.312622417201766
;;; Empirical variance: 3.106162353593951
;;; Empirical std: 1.7624308081720403
;;; Empirical skewness: 2.0051001319613513
;;; Empirical kurtosis: 9.49075444120478
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 13
;;; Elapsed time: 14255.20923 ms
;;; Empirical mean: 39.786492012502634
;;; Empirical variance: 10.044704689136159
;;; Empirical std: 3.1693382099637395
;;; Empirical skewness: 0.5602864323805973
;;; Empirical kurtosis: 2.9185570253373636
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 14
;;; Elapsed time: 13694.025591 ms
;;; Empirical mean: 41.4855332680609
;;; Empirical variance: 8.864531873314403
;;; Empirical std: 2.9773363722150044
;;; Empirical skewness: 0.3850733535031379
;;; Empirical kurtosis: 2.1299141543766615
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 15
;;; Elapsed time: 14249.021424 ms
;;; Empirical mean: 39.60828697800138
;;; Empirical variance: 5.739144576230907
;;; Empirical std: 2.3956511799990636
;;; Empirical skewness: 0.5372913706225777
;;; Empirical kurtosis: 4.463442419435472
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 16
;;; Elapsed time: 14587.279408 ms
;;; Empirical mean: 40.65413436640464
;;; Empirical variance: 4.106563213856951
;;; Empirical std: 2.02646569520852
;;; Empirical skewness: 0.36404691975996273
;;; Empirical kurtosis: 4.043362865277311
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 17
;;; Elapsed time: 15901.851776 ms
;;; Empirical mean: 39.84148342572724
;;; Empirical variance: 9.521314611257718
;;; Empirical std: 3.085662750732445
;;; Empirical skewness: 0.7153863135401994
;;; Empirical kurtosis: 3.305321833209744
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 18
;;; Elapsed time: 16021.45286 ms
;;; Empirical mean: 40.80711366395789
;;; Empirical variance: 6.4391598005180875
;;; Empirical std: 2.5375499602013925
;;; Empirical skewness: -0.3691795165360814
;;; Empirical kurtosis: 3.1770025264357864
;;; 
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 19
;;; Elapsed time: 15228.202258 ms
;;; Empirical mean: 40.93111400396046
;;; Empirical variance: 7.526057517229325
;;; Empirical std: 2.743366092454546
;;; Empirical skewness: -0.7966333934762346
;;; Empirical kurtosis: 5.993181435463895
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-unkown'>#multifn[print-method 0x1b74665d]</span>","value":"#multifn[print-method 0x1b74665d]"}],"value":"[nil,#multifn[print-method 0x1b74665d]]"},{"type":"html","content":"<span class='clj-unkown'>#multifn[print-method 0x1b74665d]</span>","value":"#multifn[print-method 0x1b74665d]"}],"value":"[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/data</span>","value":"#'model/data"}],"value":"[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/model</span>","value":"#'model/model"}],"value":"[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/model_name</span>","value":"#'model/model_name"}],"value":"[[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model],#'model/model_name]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/outfile</span>","value":"#'model/outfile"}],"value":"[[[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model],#'model/model_name],#'model/outfile]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/configurations</span>","value":"#'model/configurations"}],"value":"[[[[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/num_samples_options</span>","value":"#'model/num_samples_options"}],"value":"[[[[[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/thinning</span>","value":"#'model/thinning"}],"value":"[[[[[[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning]"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"[[[[[[[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/num-chains</span>","value":"#'model/num-chains"}],"value":"[[[[[[[[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains]"},{"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)"}],"value":"[[[[[[[[[[[[nil,#multifn[print-method 0x1b74665d]],#multifn[print-method 0x1b74665d]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains],(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)]"}
;; <=

;; @@

;; @@
