from pydantic import BaseModel
from enum import Enum
import json
from Base.Types import Vector
from Base.DeliveryManState import DeliveryManState, DeliveryManPhysicalState
from Config import Config
from typing import Optional

class Action(Enum):
    DO_NOTHING = 0
    BID_ORDER = 1
    PICK_UP_ORDER = 2
    DELIVER_ORDER = 3
    BUY_BEVERAGE = 4
    OPEN_SHARED_ORDER = 5
    GO_TO_MEETING_POINT = 6
    CANCEL_ORDER = 7
    CHANGE_WALKING_SPEED = 8
    BUY_BIKE = 9

    def __str__(self):
        return self.name

    def __repr__(self):
        return self.__str__()

class ActionSpace(BaseModel):
    choice: Action = Action.DO_NOTHING
    index: Optional[int] = None
    target_point: Optional[Vector] = None
    meeting_point: Optional[Vector] = None
    next_waypoint: Optional[Vector] = None
    new_speed: Optional[float] = None
    bid_price: Optional[float] = None

    def __str__(self):
        return f"ActionSpace(choice={self.choice}, index={self.index}, target_point={self.target_point}, meeting_point={self.meeting_point}, next_waypoint={self.next_waypoint}, new_speed={self.new_speed}, bid_price={self.bid_price})"

    def __repr__(self):
        return self.__str__()

    @classmethod
    def from_json(cls, json_str):
        if type(json_str) == str:
            try:
                json_str = json.loads(json_str)
            except Exception as e:
                print(f"parse action space from json failed: {e}, using default action: DO_NOTHING")
                return cls()
        try:
            choice = json_str.get('choice', 0)
            index = json_str.get('index', 0) if json_str.get('index', 0) is not None else None
            target_point = Vector(json_str.get('target_point', [0, 0])) if json_str.get('target_point', [0, 0]) is not None else None
            next_waypoint = Vector(json_str.get('next_waypoint', [0, 0])) if json_str.get('next_waypoint', [0, 0]) is not None else None
            meeting_point = Vector(json_str.get('meeting_point', [0, 0])) if json_str.get('meeting_point', [0, 0]) is not None else None
            new_speed = json_str.get('new_speed', 0) if json_str.get('new_speed', 0) is not None else None
            bid_price = json_str.get('bid_price', 0) if json_str.get('bid_price', 0) is not None else None
            return cls(choice=choice, index=index, target_point=target_point, meeting_point=meeting_point, next_waypoint=next_waypoint, new_speed=new_speed, bid_price=bid_price)
        except Exception as e:
            print(f"parse action space from json failed: {e}, using default action: DO_NOTHING")
            return cls()

    @classmethod
    def to_json_schema(cls):
        return {
            'name': 'ActionSpace',
            'strict': True,
            'schema': {
                'type': 'object',
                'properties': {
                    'choice': {'type': 'integer', 'description': 'The choice of the action. The action can be one of the following: 0: DO_NOTHING, 1: BID_ORDER, 2: PICK_UP_ORDER, 3: DELIVER_ORDER, 4: BUY_BEVERAGE, 5: OPEN_SHARED_ORDER, 6: GO_TO_MEETING_POINT, 7: CANCEL_ORDER, 8: CHANGE_WALKING_SPEED, 9: BUY_BIKE.'},
                    'index': {'type': 'integer', 'description': 'The index is the index of the order in the delivery man\'s orders list.'},
                    'target_point': {'type': 'array', 'description': 'The target point of the action. The target point is the point where the delivery man is going to move to. example: [10, 10]'},
                    'next_waypoint': {'type': 'array', 'description': 'The next waypoint of the action. The next waypoint is the point where the delivery man is going to move to. example: [10, 10]'},
                    'meeting_point': {'type': 'array', 'description': 'The meeting point of the action. The meeting point is the point where the delivery man is going to meet the order. example: [10, 10]'},
                    'new_speed': {'type': 'number', 'description': 'The new speed of the action. The new speed is the speed of the delivery man.'},
                    'bid_price': {'type': 'number', 'description': 'The bid price of the action. The bid price is the price of the delivery man is going to bid for the order.'},
                },
                'required': ['choice']
            }
        }

    def execute(self, delivery_manager, delivery_man, communicator):
        result = ''
        if self.choice == Action.BID_ORDER:
            result = self.bid_order(delivery_manager, delivery_man)
        # pick up an order
        elif self.choice == Action.PICK_UP_ORDER:
            result = self.pick_up_order(delivery_manager, delivery_man)
        # deliver an order
        elif self.choice == Action.DELIVER_ORDER:
            result = self.deliver_order(delivery_manager, delivery_man)
        # go buy and use a beverage
        elif self.choice == Action.BUY_BEVERAGE:
            result = self.buy_beverage(delivery_manager, delivery_man)
        # open a shared order
        elif self.choice == Action.OPEN_SHARED_ORDER:
            result = self.open_shared_order(delivery_manager, delivery_man)
        # go to the meeting point
        elif self.choice == Action.GO_TO_MEETING_POINT:
            result = self.go_to_meeting_point(delivery_manager, delivery_man)
        # cancel an order
        elif self.choice == Action.CANCEL_ORDER:
            result = self.cancel_order(delivery_manager, delivery_man)
        # change speed
        elif self.choice == Action.CHANGE_WALKING_SPEED:
            result = self.change_speed(delivery_man)
        # buy a bike
        elif self.choice == Action.BUY_BIKE:
            result = self.buy_bike(delivery_man)
        # do nothing
        elif self.choice == Action.DO_NOTHING:
            result = self.do_nothing(delivery_man)
        return result

    ###########################Action Space Related###########################
    def bid_order(self, delivery_manager, delivery_man):
        all_orders = delivery_manager.get_orders()
        if self.index in range(len(all_orders)) and self.bid_price is not None:
            order = all_orders[self.index]
            if self.bid_price > order.max_sale_price or self.bid_price < order.min_sale_price:
                with delivery_man.lock:
                    delivery_man.invalid_num += 1
                    delivery_man.invalid_bid_order_num += 1
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is trying to bid for an order but the bid price is out of range")
                return f'I tried to bid for an order but the bid price: {self.bid_price} is out of range'
            if delivery_manager.bid_order(order, delivery_man, self.bid_price):
                with delivery_man.lock:
                    delivery_man.bid_order_num += 1
                delivery_man.logger.info(f"DeliveryMan {delivery_man.id} bid {self.bid_price} for order {order}")
                delivery_man.action = Action.BID_ORDER
                delivery_manager.update_notification(f"DeliveryMan {delivery_man.id} bid {self.bid_price} for order {order}!")
                return f"I bid {self.bid_price} for order {order}"
            else:
                with delivery_man.lock:
                    delivery_man.invalid_num += 1
                    delivery_man.invalid_bid_order_num += 1
                delivery_man.logger.info(f"DeliveryMan {delivery_man.id} failed to bid for order {order} because it is already allocated to another delivery man")
                return f'I failed to bid for order {order}, it is already allocated to another delivery man'
        else:
            with delivery_man.lock:
                delivery_man.invalid_num += 1
                delivery_man.invalid_bid_order_num += 1
            if self.index not in range(len(all_orders)):
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is trying to bid for an order but the order has been taken")
                return f'I tried to bid for an order but the order has been taken'
            elif self.bid_price is None:
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is trying to bid for an order but the bid price is not set")
                return f'I tried to bid for an order but the bid price is not set'

    def pick_up_order(self, delivery_manager, delivery_man):
        if (self.index in range(len(delivery_man.orders)) and
            delivery_man.orders[self.index].has_picked_up == False and
            delivery_man.is_valid_next_waypoint(self.next_waypoint, delivery_manager)):
            delivery_man.pick_up_order_num += 1
            delivery_man.logger.info(f"DeliveryMan {delivery_man.id} is picking up order {delivery_man.orders[self.index]}, target point is {self.target_point}, move to {self.next_waypoint} first")
            delivery_man.current_order = delivery_man.orders[self.index]
            delivery_man.action = Action.PICK_UP_ORDER
            with delivery_man.lock:
                delivery_man.next_waypoint = self.next_waypoint
                delivery_man.state = DeliveryManState.MOVING
            return f"I pick up order {delivery_man.orders[self.index]} and move to {self.next_waypoint} first"
        else:
            with delivery_man.lock:
                delivery_man.invalid_num += 1
                delivery_man.invalid_pick_up_order_num += 1
            if self.index not in range(len(delivery_man.orders)):
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is picking up an order but the order {self.index} has not been taken")
                return f'I tried to pick up an order but the order index: {self.index} has not been taken'
            elif not delivery_man.is_valid_next_waypoint(self.next_waypoint, delivery_manager):
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is picking up an order but the next waypoint is invalid")
                return f'I tried to pick up an order but the order: {delivery_man.orders[self.index]} is already picked up or the next waypoint: {self.next_waypoint} is invalid'
            elif delivery_man.orders[self.index].has_picked_up == True:
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is picking up an order but the order is already picked up")
                return f'I tried to pick up an order but the order: {delivery_man.orders[self.index]} is already picked up'

    def deliver_order(self, delivery_manager, delivery_man):
        if (self.index in range(len(delivery_man.orders)) and
            delivery_man.orders[self.index].has_picked_up == True and
            delivery_man.is_valid_next_waypoint(self.next_waypoint, delivery_manager)):
            delivery_man.deliver_order_num += 1
            delivery_man.logger.info(f"DeliveryMan {delivery_man.id} is delivering order {delivery_man.orders[self.index]}, target point is {self.target_point}, move to {self.next_waypoint} first")
            delivery_man.current_order = delivery_man.orders[self.index]
            delivery_man.action = Action.DELIVER_ORDER
            with delivery_man.lock:
                delivery_man.next_waypoint = self.next_waypoint
                delivery_man.state = DeliveryManState.MOVING
            return f"I deliver order {delivery_man.orders[self.index]} and move to {self.next_waypoint} first"
        else:
            with delivery_man.lock:
                delivery_man.invalid_num += 1
                delivery_man.invalid_deliver_order_num += 1
            if self.index not in range(len(delivery_man.orders)):
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is delivering an order but the order index is out of range")
                return f'I tried to deliver an order but the order index: {self.index} is out of range'
            elif delivery_man.orders[self.index].has_picked_up == False:
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is delivering an order but the order is not picked up")
                return f'I tried to deliver an order but the order: {delivery_man.orders[self.index]} is not picked up'
            else:
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is delivering an order but the next waypoint is invalid")
                return f'I tried to deliver an order but the next waypoint: {self.next_waypoint} is invalid'

    def buy_beverage(self, delivery_manager, delivery_man):
        if delivery_man.money >= Config.COST_OF_BEVERAGE:
            delivery_man.logger.info(f"DeliveryMan {delivery_man.id} is going to buy and use a beverage")
            delivery_man.action = Action.BUY_BEVERAGE
            delivery_man.money -= Config.COST_OF_BEVERAGE
            delivery_man.energy += Config.DELIVERY_MAN_RECOVER_ENERGY_AMOUNT
            Config.COST_OF_BEVERAGE *= 1.05
            with delivery_man.lock:
                delivery_man.buy_beverage_num += 1
            return f"I buy beverage at {self.target_point} and move to {self.next_waypoint} first"
        else:
            with delivery_man.lock:
                delivery_man.invalid_num += 1
                delivery_man.invalid_buy_beverage_num += 1
            delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is going to buy and use a beverage but the money is not enough")
            return f'I tried to buy and use a beverage but I do not have enough money: {delivery_man.money}'

    def open_shared_order(self, delivery_manager, delivery_man):
        if (self.index in range(len(delivery_man.orders)) and
            delivery_man.is_valid_meeting_point(self.meeting_point, delivery_manager)):
            delivery_man.logger.info(f"DeliveryMan {delivery_man.id} is opening a shared order {delivery_man.orders[self.index]} at {self.meeting_point}")
            order = delivery_man.orders[self.index]
            delivery_man.open_shared_order_num += 1
            if not order.is_shared:
                order.is_shared = True
                order.meeting_point = self.meeting_point
                delivery_manager.add_shared_order(order)
                delivery_man.action = Action.OPEN_SHARED_ORDER
                with delivery_man.lock:
                    delivery_man.open_shared_order_num += 1
                delivery_manager.update_notification(f"DeliveryMan {delivery_man.id} open shared order {order} at {self.meeting_point}! Good for him and Accept that order!")
                return f"I open shared order {order}"
            else:
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is opening a shared order {delivery_man.orders[self.index]} at {self.meeting_point} but the order is already shared")
                with delivery_man.lock:
                    delivery_man.invalid_num += 1
                    delivery_man.invalid_open_shared_order_num += 1
                return f'I tried to open a shared order but the order {order} is already shared'
        else:
            with delivery_man.lock:
                delivery_man.invalid_num += 1
                delivery_man.invalid_open_shared_order_num += 1
            if self.index not in range(len(delivery_man.orders)):
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is opening a shared order but the order index is out of range")
                return f'I tried to open a shared order but the order index: {self.index} is out of range'
            elif not delivery_man.is_valid_meeting_point(self.meeting_point, delivery_manager):
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is opening a shared order but the meeting point is invalid")
                return f'I tried to open a shared order but the meeting point: {self.meeting_point} is invalid'
            elif delivery_man.orders[self.index].is_shared:
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is opening a shared order but the order is already shared")
                return f'I tried to open a shared order but the order {delivery_man.orders[self.index]} is already shared'

    def go_to_meeting_point(self, delivery_manager, delivery_man):
        if (self.index in range(len(delivery_man.orders)) and
            delivery_man.orders[self.index].is_shared and
            delivery_man.is_valid_next_waypoint(self.next_waypoint, delivery_manager)):
            delivery_man.go_to_meeting_point_num += 1
            delivery_man.logger.info(f"DeliveryMan {delivery_man.id} is going to the meeting point of order {delivery_man.orders[self.index]}, move to {self.next_waypoint} first")
            order = delivery_man.orders[self.index]
            delivery_man.current_order = order
            delivery_man.action = Action.GO_TO_MEETING_POINT
            with delivery_man.lock:
                delivery_man.next_waypoint = self.next_waypoint
                delivery_man.state = DeliveryManState.MOVING
            return f"I go to the meeting point of order {order} and move to {self.next_waypoint} first"
        else:
            with delivery_man.lock:
                delivery_man.invalid_num += 1
                delivery_man.invalid_go_to_meeting_point_num += 1
            if self.index not in range(len(delivery_man.orders)):
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is going to the meeting point of order but the order index is out of range")
                return f'I tried to go to the meeting point of order but the order index: {self.index} is out of range'
            elif not delivery_man.orders[self.index].is_shared:
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is going to the meeting point of order but the order is not shared")
                return f'I tried to go to the meeting point of order but the order: {delivery_man.orders[self.index]} is not shared'
            else:
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is going to the meeting point of order but the next waypoint is invalid")
                return f'I tried to go to the meeting point of order but the next waypoint: {self.next_waypoint} is invalid'

    def cancel_order(self, delivery_manager, delivery_man):
        if self.index in range(len(delivery_man.orders)) and delivery_man.orders[self.index].is_shared:
            order = delivery_man.orders[self.index]
            if delivery_manager.cancel_shared_order(order):
                delivery_man.logger.info(f"DeliveryMan {delivery_man.id} is canceling the shared order {order}")
                delivery_man.cancel_order_num += 1
                order.is_shared = False
                order.meeting_point = None
                delivery_man.action = Action.CANCEL_ORDER
                return f"I cancel the shared order {order}"
        else:
            with delivery_man.lock:
                delivery_man.invalid_num += 1
                delivery_man.invalid_cancel_order_num += 1
            if self.index not in range(len(delivery_man.orders)):
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is canceling the shared order but the order index is out of range")
                return f'I tried to cancel the shared order but the order {self.index} has not been taken'
            elif not delivery_man.orders[self.index].is_shared:
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is canceling the shared order but the order is not shared")
                return f'I tried to cancel the shared order but the order: {delivery_man.orders[self.index]} is not shared'

    def change_speed(self, delivery_man):
        if self.new_speed is not None and Config.DELIVERY_MAN_MIN_SPEED <= self.new_speed <= Config.DELIVERY_MAN_MAX_SPEED and delivery_man.physical_state == DeliveryManPhysicalState.WALKING:
            delivery_man.logger.info(f"DeliveryMan {delivery_man.id} is changing speed to {self.new_speed}")
            delivery_man.change_speed_num += 1
            # delivery_man.set_speed(self.new_speed, communicator)
            delivery_man.set_speed(self.new_speed)
            delivery_man.action = Action.CHANGE_WALKING_SPEED
            return f"I change speed to {self.new_speed}"
        else:
            with delivery_man.lock:
                delivery_man.invalid_num += 1
                delivery_man.invalid_change_speed_num += 1
            if not delivery_man.physical_state == DeliveryManPhysicalState.WALKING:
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is changing speed but the physical state is not walking")
                return f'I tried to change speed but I am not walking'
            elif self.new_speed is None:
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is changing speed but the speed is not valid")
                return f'I tried to change speed but I did not offer a valid speed'
            elif not (Config.DELIVERY_MAN_MIN_SPEED <= self.new_speed <= Config.DELIVERY_MAN_MAX_SPEED):
                delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is changing speed but the speed is out of range")
                return f'I tried to change speed but the speed: {self.new_speed} is out of range'

    def buy_bike(self, delivery_man):
        if delivery_man.money >= Config.PRICE_OF_BIKE and delivery_man.physical_state == DeliveryManPhysicalState.WALKING:
            delivery_man.logger.info(f"DeliveryMan {delivery_man.id} is buying a bike")
            delivery_man.action = Action.BUY_BIKE
            delivery_man.physical_state = DeliveryManPhysicalState.DRIVING
            with delivery_man.lock:
                delivery_man.money -= Config.PRICE_OF_BIKE
            delivery_man.set_speed(1100)
            return f"I bought a bike"
        else:
            delivery_man.logger.warning(f"DeliveryMan {delivery_man.id} is trying to buy a bike but the money is not enough")
            with delivery_man.lock:
                delivery_man.invalid_num += 1
                delivery_man.invalid_buy_bike_num += 1
            return 'I tried to buy a bike but I do not have enough money'

    def do_nothing(self, delivery_man):
        delivery_man.action = Action.DO_NOTHING
        delivery_man.do_nothing_num += 1
        return f"I did nothing"
