import time
import math
import traceback
import logging
import numpy as np
from threading import Lock
import sys
from typing import Tuple, List, Dict, Optional
import os
import random

from Config import Config
from Base.Order import Order
from Base.DeliveryManState import DeliveryManState, DeliveryManPhysicalState, DeliveryManDrivingState, DeliveryManWalkingState
from Base.ActionSpace import Action, ActionSpace
from Navigation import Navigator
from Base.Types import Vector
from llm.openai_model import UEOpenAIModel
from llm.base_model import BaseModel
from persona_hub.code.persona_utils import DeliveryPersona
from Prompt.delivery_man_prompt import \
    delivery_man_context_prompt, \
    delivery_man_user_prompt, \
    delivery_man_reasoning_user_prompt, \
    delivery_man_system_prompt

class DeliveryMan:
    _id_counter = 0
    _camera_id_counter = 1
    def __init__(self, position: Vector, direction: Vector, use_planner: bool = False, persona: Optional[DeliveryPersona] = None, communicator = None, delivery_manager = None):
        super().__init__()
        self.id = DeliveryMan._id_counter
        DeliveryMan._id_counter += 1
        self.communicator = communicator
        self.delivery_manager = delivery_manager
        # Add Camera ID
        # TODO: Maybe there is a better way to assign camera ID using constructor params
        self.camera_id = DeliveryMan._camera_id_counter
        DeliveryMan._camera_id_counter += 1
        # current_view = self.communicator.get_camera_observation(camera_id = self.camera_id)

        # Mapping from low-level action strings to Action Enum
        # TODO: Need to fill this based on config
        self.action_str_to_enum = {}

        self.use_planner: bool = use_planner

        ## Unreal related
        self.position: Vector = position    # coordinate that the delivery man is located in UE
        self.direction: Vector = direction  # direction that the delivery man is facing in UE
        self.yaw: float = 0

        ## Task related
        ### State
        self.state: DeliveryManState = DeliveryManState.IDLE
        self.physical_state: DeliveryManPhysicalState = DeliveryManPhysicalState.WALKING
        self.driving_state: DeliveryManDrivingState = DeliveryManDrivingState.WAITING
        self.walking_state: DeliveryManWalkingState = DeliveryManWalkingState.STOP
        self.speed: float = Config.DELIVERY_MAN_DEFAULT_SPEED
        self.max_energy: float = Config.DELIVERY_MAN_DEFAULT_MAX_ENERGY
        self.action: Action = Action.DO_NOTHING

        self.state_history = []
        self.history_actions_num = 3

        self.next_waypoint: Vector = None
        self.orders: List[Order] = []
        self.current_order: Order = None
        self.last_state: Tuple[Vector, str] = (self.position, 'do nothing')

        self.persona: DeliveryPersona = persona

        ### Navigation
        self.throttle: float = 0
        self.brake: float = 0
        self.steering: float = 0

        self.step: int = 0

        # TODO: We may need to find a way to avoid initializing this in case we give full control to user
        ### LLM
        # model_types = Config.MODEL_TYPES
        # model_type = random.choice(model_types)
        model_name = Config.MODEL_NAMES[self.id]
        if model_name.startswith("gpt"):
            self.llm = UEOpenAIModel(model=model_name)
        else:
            self.llm = BaseModel(url="https://openrouter.ai/api/v1",
                            api_key=os.getenv("OPENROUTER_API_KEY"),
                            model=model_name
                            )

        ### Multithreading
        self.lock = Lock()

        ### Debug
        self.logger = logging.getLogger(__name__)

        ### Attributes
        self.money: float = Config.DELIVERY_MAN_DEFAULT_INITIAL_MONEY
        self.revenue: float = 0
        self.energy: float = Config.DELIVERY_MAN_DEFAULT_INITIAL_ENERGY
        self.move_speed: float = Config.DELIVERY_MAN_DEFAULT_SPEED     #cm/s no more than 250

        ### Evaluation
        self.total_consumed_energy: float = 0
        self.invalid_num: int = 0
        self.invalid_bid_order_num: int = 0
        self.invalid_pick_up_order_num: int = 0
        self.invalid_deliver_order_num: int = 0
        self.invalid_buy_beverage_num: int = 0
        self.invalid_open_shared_order_num: int = 0
        self.invalid_go_to_meeting_point_num: int = 0
        self.invalid_cancel_order_num: int = 0
        self.invalid_change_speed_num: int = 0
        self.invalid_buy_bike_num: int = 0

        self.bid_order_num: int = 0
        self.pick_up_order_num: int = 0
        self.deliver_order_num: int = 0
        self.buy_beverage_num: int = 0
        self.open_shared_order_num: int = 0
        self.go_to_meeting_point_num: int = 0
        self.cancel_order_num: int = 0
        self.change_speed_num: int = 0
        self.buy_bike_num: int = 0
        self.do_nothing_num: int = 0

        self.decision_num: int = 0

        self.open_shared_order_num: int = 0
        self.success_order_num: int = 0
        self.success_shared_order_num: int = 0
        self.delay_order_num: int = 0
        self.step_delay_order_num: int = 0
        self.total_delay_time: float = 0
        self.total_spent_time: float = 0
        self.total_spent_step: int = 0
        self.total_delay_step: int = 0
        # self.set_speed(self.move_speed)

    def __repr__(self):
        return f"DeliveryMan(id={self.id}, position={self.position})"

    def __str__(self):
        return f"DeliveryMan(id={self.id}, position={self.position})"

    def load_persona(self, persona: DeliveryPersona):
        self.persona = persona
        self.money = persona.initial_money
        self.energy = persona.initial_energy
        self.max_energy = Config.DELIVERY_MAN_DEFAULT_MAX_ENERGY
        self.move_speed = persona.move_speed
        self.energy_consumption = persona.energy_consumption
        self.history_actions_num = persona.history_actions_num
        self.decision_strategy = persona.decision_strategy
        self.bigfive = persona.bigfive
        self.generated_profile = persona.generated_profile
        self.system_prompt = delivery_man_system_prompt
        # self.system_prompt = persona.system_prompt
        self.system_prompt = self.system_prompt + f'\n\nyou have a map of the city structured as a graph with nodes and edges:\n{self.delivery_manager.map}'

    def decide_llm(self, delivery_manager, communicator):
        '''
        This function will decide what to do based on the current state of the delivery man and the delivery manager
        using the Large Language Model (LLM).
        The actions are:
        0. Do nothing
        1. Accept an order
        2. Pick up an order
        3. Deliver an order
        4. Go to a supply point to buy and use a beverage
        5. Open a shared order
        6. Go to the meeting point
        7. Cancel a shared order
        8. Change speed
        9. Buy a bike
        '''
        self.decision_num += 1

        # Get all necessary data before acquiring any locks
        all_orders = delivery_manager.get_orders()
        orders_to_bid = [
            order for order in all_orders
        ]
        orders_to_bid = orders_to_bid.copy()

        possible_next_waypoints = self.get_possible_next_waypoints(delivery_manager)
        notification = delivery_manager.get_notification()

        # Now we can safely format the context without holding any locks
        context_prompt = delivery_man_context_prompt.format(
            position=self.get_self_position_on_map(delivery_manager),
            possible_next_waypoints=possible_next_waypoints,
            orders=self.orders[:10],
            orders_to_bid=orders_to_bid[:10],
            speed=self.move_speed,
            physical_state=self.physical_state,
            money=self.money,
            energy=self.energy,
            history=self.state_history[-self.history_actions_num:],
            cost_of_beverage=Config.COST_OF_BEVERAGE,
            recover_energy_amount=Config.DELIVERY_MAN_RECOVER_ENERGY_AMOUNT,
            price_of_bike=Config.PRICE_OF_BIKE,
            notification=notification,
            picked_up_order=self.current_order
        )
        # self.logger.warning(context_prompt)
        response = self.llm.react(system_prompt=self.system_prompt,
                                context_prompt=context_prompt,
                                user_prompt=delivery_man_user_prompt,
                                reasoning_prompt=delivery_man_reasoning_user_prompt)
        # print(response,flush=True)
        action = ActionSpace.from_json(response)

        self.logger.info(f"DeliveryMan {self.id}({self.position}) decided to {action}")

        log = action.execute(delivery_manager, self, communicator)
        if log != '':
            current_state = (self.get_self_position_on_map(delivery_manager), log)
            self.state_history.append(current_state)
            if len(self.state_history) > self.history_actions_num:
                self.state_history.pop(0)
            self.last_state = current_state
        # print(f"DeliveryMan {self.id} decided to {action}")
        # self.logger.info(f"DeliveryMan {self.id}({self.position}) has done {action}")

    def update_delivery_man(self, delivery_manager, dt, communicator, exit_event, navigator):
        """
        Update the delivery man's state
        """
        self.logger.info(f"Thread for DeliveryMan {self.id} started")
        self.set_speed(self.move_speed)
        try:
            while not exit_event.is_set():
                if self.state == DeliveryManState.IDLE:
                    # Step where we need to decide the next high-level move for agent
                    self.decide_llm(delivery_manager, communicator)
                    self.step += 1
                elif self.state == DeliveryManState.MOVING:
                    navigator.move_to_next_waypoint_silent(self)
                    # navigator.move_to_next_waypoint_silent(self)
                    if self.current_order is not None:
                        if self.next_waypoint == self.current_order.store_position:
                            self.logger.info(f"DeliveryMan {self.id} has picked up order {self.current_order.id}")
                            self.current_order.has_picked_up = True
                            delivery_manager.update_notification(f"DeliveryMan {self.id} has picked up order {self.current_order.id}!")
                            self.current_order = None
                            with self.lock:
                                self.state = DeliveryManState.IDLE
                        elif self.next_waypoint == self.current_order.customer_position and self.current_order.has_picked_up:
                            self.logger.info(f"DeliveryMan {self.id} has delivered order {self.current_order.id}")
                            self.current_order.has_delivered = True
                            delivery_manager.finish_order(self.current_order, self.id)
                            with self.lock:
                                self.orders.remove(self.current_order)
                                self.current_order = None
                                self.state = DeliveryManState.IDLE
                        elif self.current_order.is_shared and self.next_waypoint == self.current_order.meeting_point:
                            self.state = DeliveryManState.WAITING
                        else:
                            self.current_order = None
                            with self.lock:
                                self.state = DeliveryManState.IDLE
                    with self.lock:
                        self.next_waypoint = None
                elif self.state == DeliveryManState.WAITING:
                    if self.check_if_meet_with_other_delivery_men():
                        with self.lock:
                            self.state = DeliveryManState.IDLE
                        self.orders.remove(self.current_order)
                        self.current_order = None
                if self.state != DeliveryManState.MOVING:
                    self.auto_recover_energy(dt)
                time.sleep(dt)
        except Exception as e:
            self.logger.error(f"Error in DeliveryMan {self.id}: {e}")
            self.logger.error(traceback.format_exc())
            self.logger.error(f"Thread for DeliveryMan {self.id} is dead")

    def auto_recover_energy(self, dt, recover_rate=0.5):
        """
        Automatically recover energy, recover recover_rate unit energy per second
        """
        with self.lock:
            if self.energy < self.max_energy // 3:
                self.energy += recover_rate * dt
                if self.energy > self.max_energy:
                    self.energy = self.max_energy

    ###########################Getters and Setters##########################
    def lose_energy(self, dt):
        with self.lock:
            if self.physical_state == DeliveryManPhysicalState.WALKING:
                base_consumption = self.move_speed / 100 * dt / 10 * self.energy_consumption
                energy_ratio = min(self.total_consumed_energy / self.max_energy, 100)
                consumed_energy = base_consumption * (1 + self.energy_consumption * (energy_ratio) ** 2)
                self.energy -= consumed_energy if self.energy - consumed_energy > 0 else 0
                self.total_consumed_energy += consumed_energy
                if self.energy <= 0:
                    sys.exit(f"DeliveryMan {self.id} has no energy left")
            elif self.physical_state == DeliveryManPhysicalState.DRIVING:
                base_consumption = self.move_speed / 100 * dt / 100
                energy_ratio = min(self.total_consumed_energy / self.max_energy, 100)
                consumed_energy = base_consumption * (1 + self.energy_consumption * (energy_ratio) ** 2)
                self.energy -= consumed_energy if self.energy - consumed_energy > 0 else 0
                self.total_consumed_energy += consumed_energy
                if self.energy <= 0:
                    sys.exit(f"DeliveryMan {self.id} has no energy left")

    def set_speed(self, speed):
        with self.lock:
            self.move_speed = speed
            # self.delivery_manager.update_notification(f"DeliveryMan {self.id} has set speed to {speed} cm/s")
            # self.communicator.delivery_man_set_speed(self.id, speed)

    def set_driving_state(self, throttle, brake, steering):
        self.throttle = throttle
        self.brake = brake
        self.steering = steering

    def earn_money(self, amount, is_shared, delay_time, spent_time, delay_step, spent_step):
        '''
        Earn money from an successful order
        '''
        self.logger.info(f"DeliveryMan {self.id} earns {amount} dollars, spent {spent_time} seconds, delay {delay_time} seconds, spent step {delay_step}")
        with self.lock:
            self.money += amount
            self.revenue += amount
            self.success_order_num += 1
            self.total_spent_time += spent_time
            self.total_spent_step += spent_step
            if is_shared:
                self.success_shared_order_num += 1
            if delay_time > 0:
                self.delay_order_num += 1
                self.total_delay_time += delay_time
            if delay_step > 0:
                self.step_delay_order_num += 1
                self.total_delay_step += delay_step
            self.delivery_manager.update_notification(f"DeliveryMan {self.id} has earned {amount} dollars!")

    def get_energy(self):
        return self.energy

    def get_money(self):
        return self.money

    def get_revenue(self):
        return self.revenue

    def get_total_delay_time(self):
        return self.total_delay_time

    def get_delay_order_num(self):
        return self.delay_order_num

    def get_total_consumed_energy(self):
        return self.total_consumed_energy

    def get_success_order_num(self):
        return self.success_order_num

    def get_success_shared_order_num(self):
        return self.success_shared_order_num

    def get_speed(self):
        return self.move_speed

    def get_step(self):
        return self.step

    def get_state(self):
        return self.state

    def get_next_waypoint(self):
        with self.lock:
            return self.next_waypoint

    def get_current_view(self, communicator):
        return communicator.get_camera_observation(self.camera_id)

    def get_position(self):
        return self.position

    def get_direction(self):
        return self.direction

    def get_yaw(self):
        return self.yaw

    def add_order(self, order: Order):
        with self.lock:
            self.orders.append(order)

    ########################Utility Functions########################
    def get_on_a_bike(self, communicator):
        # spawn a bike near the delivery man
        communicator.spawn_scooter(self)

    def check_if_meet_with_other_delivery_men(self):
        """
        Check if the current delivery man meets with other delivery men
        Returns: True if the current delivery man meets with other delivery men, False otherwise
        """
        other_delivery_men = self.current_order.get_delivery_men()
        for other_delivery_man in other_delivery_men:
            if other_delivery_man.id == self.id:
                continue
            if self.position.distance(other_delivery_man.position) < Config.DELIVERY_MAN_MEET_DISTANCE:
                return True
        return False

    def is_valid_next_waypoint(self, next_waypoint, delivery_manager):
        """
        Check if the next waypoint is valid
        Returns: True if the next waypoint is valid, False otherwise
        """
        if next_waypoint is None:
            return False

        all_points = delivery_manager.map.get_points()
        if not next_waypoint in all_points:
            return False

        if next_waypoint in self.get_possible_next_waypoints(delivery_manager):
            return True
        return False

    def is_valid_meeting_point(self, meeting_point, delivery_manager):
        """
        Check if the meeting point is valid
        Returns: True if the meeting point is valid, False otherwise
        """
        if meeting_point is None:
            return False

        all_points = delivery_manager.map.get_points()
        if not meeting_point in all_points:
            return False
        return True

    def get_possible_next_waypoints(self, delivery_manager):
        """
        Get the possible next waypoints from the current position
        Returns: A list of possible next waypoints (List[Vector])
        """
        current_node = None
        min_distance = float('inf')
        for node in delivery_manager.map.nodes:
            distance = self.position.distance(node.position)
            if distance < min_distance:
                min_distance = distance
                current_node = node
        return delivery_manager.map.get_adjacent_points(current_node)

    def get_self_position_on_map(self, delivery_manager):
        """
        Find the closest node on the map to the current position
        Returns: The closest node's position (Vector)
        """
        current_node = None
        min_distance = float('inf')
        for node in delivery_manager.map.nodes:
            distance = self.position.distance(node.position)
            if distance < min_distance:
                min_distance = distance
                current_node = node
        return current_node.position

    ###########################Unreal Engine Related###########################
    def set_position(self, position):
        with self.lock:
            self.position = position

    def set_direction(self, yaw):
        with self.lock:
            self.yaw = yaw
            self.direction = Vector(math.cos(math.radians(yaw)), math.sin(math.radians(yaw))).normalize()

    def get_dict(self):
        return {
            "id": self.id,
            "position": {"x": self.position.x, "y": self.position.y} if self.position else None,
            "direction": {"x": self.direction.x, "y": self.direction.y} if self.direction else None,
            "yaw": self.yaw,
            "speed": self.move_speed,
            "energy": self.energy,
            "money": self.money,
            "revenue": self.revenue,
            "state": self.state.value if self.state else None,
            "physical_state": self.physical_state.value if self.physical_state else None,
            "driving_state": self.driving_state.value if self.driving_state else None,
            "walking_state": self.walking_state.value if self.walking_state else None,
            "orders": [order.get_dict() for order in self.orders] if self.orders else [],
            "current_order": self.current_order.get_dict() if self.current_order else None,
            "next_waypoint": {"x": self.next_waypoint.x, "y": self.next_waypoint.y} if self.next_waypoint else None,
            "action": self.action.value if self.action else None,
            "state_history": [({"x": pos.x, "y": pos.y}, state) for pos, state in self.state_history] if self.state_history else [],
            "history_actions_num": self.history_actions_num,
            "last_state": ({"x": self.last_state[0].x, "y": self.last_state[0].y}, self.last_state[1]) if self.last_state else None,
            "throttle": self.throttle,
            "brake": self.brake,
            "steering": self.steering,
            "step": self.step,
            "total_consumed_energy": self.total_consumed_energy,
            "invalid_num": self.invalid_num,
            "invalid_bid_order_num": self.invalid_bid_order_num,
            "invalid_pick_up_order_num": self.invalid_pick_up_order_num,
            "invalid_deliver_order_num": self.invalid_deliver_order_num,
            "invalid_buy_beverage_num": self.invalid_buy_beverage_num,
            "invalid_open_shared_order_num": self.invalid_open_shared_order_num,
            "invalid_go_to_meeting_point_num": self.invalid_go_to_meeting_point_num,
            "invalid_cancel_order_num": self.invalid_cancel_order_num,
            "invalid_change_speed_num": self.invalid_change_speed_num,
            "invalid_buy_bike_num": self.invalid_buy_bike_num,
            "bid_order_num": self.bid_order_num,
            "pick_up_order_num": self.pick_up_order_num,
            "deliver_order_num": self.deliver_order_num,
            "buy_beverage_num": self.buy_beverage_num,
            "open_shared_order_num": self.open_shared_order_num,
            "go_to_meeting_point_num": self.go_to_meeting_point_num,
            "cancel_order_num": self.cancel_order_num,
            "change_speed_num": self.change_speed_num,
            "buy_bike_num": self.buy_bike_num,
            "do_nothing_num": self.do_nothing_num,
            "decision_num": self.decision_num,
            "success_order_num": self.success_order_num,
            "success_shared_order_num": self.success_shared_order_num,
            "delay_order_num": self.delay_order_num,
            "step_delay_order_num": self.step_delay_order_num,
            "total_delay_time": self.total_delay_time,
            "total_spent_time": self.total_spent_time,
            "total_spent_step": self.total_spent_step,
            "total_delay_step": self.total_delay_step,
        }