import math
import json
import re
from simworld.communicator.communicator import Communicator
from Communicator.unrealcv_delivery import DeliveryUnrealCV
from Base import DeliveryMan
from typing import List
from Base.Types import Vector
from Config import Config
import time

class DeliveryCommunicator(Communicator):
    def __init__(self, unrealcv: DeliveryUnrealCV):
        super().__init__(unrealcv)
        self.unrealcv = unrealcv
        self.delivery_manager_name = None
        self.delivery_man_id_to_name = {}

    def delivery_man_turn_around(self, delivery_man_id, angle, clockwise):
        self.unrealcv.d_turn_around(self.get_delivery_man_name(delivery_man_id), angle, clockwise)

    def delivery_man_step_forward(self, delivery_man_id):
        self.unrealcv.d_step_forward(self.get_delivery_man_name(delivery_man_id))

    def delivery_man_set_speed(self, delivery_man_id, speed):
        self.unrealcv.d_set_speed(self.get_delivery_man_name(delivery_man_id), speed)

    def delivery_man_move_forward(self, delivery_man_id):
        print(f"[delivery_man_move_forward]: DeliveryMan {delivery_man_id} move forward")
        self.unrealcv.d_move_forward(self.get_delivery_man_name(delivery_man_id))

    def delivery_man_stop(self, delivery_man_id):
        self.unrealcv.d_stop(self.get_delivery_man_name(delivery_man_id))

    def delivery_man_rotate(self, delivery_man_id, angle, turn_direction):
        self.unrealcv.d_rotate(self.get_delivery_man_name(delivery_man_id), angle, turn_direction)

    def scooter_set_state(self, delivery_man_id, throttle, brake, steering):
        self.unrealcv.s_set_state(self.get_delivery_man_name(delivery_man_id), throttle, brake, steering)

    def scooter_making_u_turn(self, delivery_man_id):
        self.unrealcv.making_u_turn(self.get_delivery_man_name(delivery_man_id))

    def get_position_and_direction(self, delivery_man_ids):
        try:
            if self.delivery_manager_name is None:
                print("Warning: delivery_manager_name is not set")
                return {}

            info_str = self.unrealcv.get_informations(self.delivery_manager_name)
            if not info_str:
                print("Warning: No information received from Unreal Engine")
                return {}
            # print(f"Received info from UE: {info_str}")  # Debug print
            info = json.loads(info_str)
            result = {}
            d_locations = info["DLocations"]
            d_rotations = info["DRotations"]
            s_locations = info["SLocations"]
            s_rotations = info["SRotations"]
            for delivery_man_id in delivery_man_ids:
                name = self.get_delivery_man_name(delivery_man_id)
                # Parse location
                location_pattern = f"{name}X=(.*?) Y=(.*?) Z="
                match = re.search(location_pattern, d_locations)
                if match:
                    x, y = float(match.group(1)), float(match.group(2))
                    position = Vector(x, y)
                    # Parse rotation
                    rotation_pattern = f"{name}P=.*? Y=(.*?) R="
                    match = re.search(rotation_pattern, d_rotations)
                    if match:
                        direction = float(match.group(1))
                        result[delivery_man_id] = (position, direction)
                    else:
                        print(f"Warning: Could not parse rotation for {name}")
                    continue

                match = re.search(location_pattern, s_locations)
                if match:
                    x, y = float(match.group(1)), float(match.group(2))
                    position = Vector(x, y)
                    # Parse rotation
                    rotation_pattern = f"{name}P=.*? Y=(.*?) R="
                    match = re.search(rotation_pattern, s_rotations)
                    if match:
                        direction = float(match.group(1))
                        result[delivery_man_id] = (position, direction)
                    else:
                        print(f"Warning: Could not parse rotation for {name}")
                else:
                    print(f"Warning: Could not parse location for {name}")

            return result
        except json.JSONDecodeError as e:
            print(f"Error decoding JSON: {e}")
            print(f"Raw data: {info_str}")
            return {}
        except Exception as e:
            print(f"Error in get_position_and_direction: {e}")
            return {}

    def spawn_delivery_men(self, delivery_men: List[DeliveryMan]):
        for delivery_man in delivery_men:
            name = f'GEN_DELIVERY_MAN_{delivery_man.id}'
            self.delivery_man_id_to_name[delivery_man.id] = name
            model_name = Config.DELIVERY_MAN_MODEL_PATH
            self.unrealcv.spawn_bp_asset(model_name, name)
            # Convert 2D position to 3D (x,y -> x,y,z)
            location_3d = (
                delivery_man.position.x,
                delivery_man.position.y,
                110 # Z coordinate (ground level)
            )
            # Convert 2D direction to 3D orientation (assuming rotation around Z axis)
            orientation_3d = (
                0,  # Pitch
                math.degrees(math.atan2(delivery_man.direction.y, delivery_man.direction.x)),  # Yaw
                0  # Roll
            )
            self.unrealcv.set_location(location_3d, name)
            self.unrealcv.set_orientation(orientation_3d, name)
            self.unrealcv.set_scale((1, 1, 1), name)  # Default scale
            self.unrealcv.set_collision(name, True)
            self.unrealcv.set_movable(name, True)

    def spawn_scooter(self, delivery_man):
        old_name = self.get_delivery_man_name(delivery_man.id)
        name = f'GEN_SCOOTER_{delivery_man.id}'
        self.delivery_man_id_to_name[delivery_man.id] = name
        model_name = Config.SCOOTER_MODEL_PATH
        self.unrealcv.spawn_bp_asset(model_name, name)
        location_3d = (
            delivery_man.position.x + 100,
            delivery_man.position.y + 100,
            0 # Z coordinate (ground level)
        )
        orientation_3d = (
            0,  # Pitch
            math.degrees(math.atan2(delivery_man.direction.y, delivery_man.direction.x)),  # Yaw
            0  # Roll
        )
        self.unrealcv.set_location(location_3d, name)
        self.unrealcv.set_orientation(orientation_3d, name)
        self.unrealcv.set_scale((1, 1, 1), name)  # Default scale
        self.unrealcv.set_collision(name, True)
        self.unrealcv.set_movable(name, True)

        self.unrealcv.d_get_on_a_bike(old_name)
        time.sleep(5)
        self.unrealcv.update_agents(self.delivery_manager_name)

    def spawn_delivery_manager(self):
        self.delivery_manager_name = 'GEN_DeliveryManager'
        self.unrealcv.spawn_bp_asset(Config.DELIVERY_MANAGER_MODEL_PATH, self.delivery_manager_name)

    def get_delivery_man_name(self, id):
        if id not in self.delivery_man_id_to_name:
            raise ValueError(f"Delivery man with id {id} not found")
        return self.delivery_man_id_to_name[id]

    def delivery_man_pick_up(self, delivery_man_id, object_name):
        self.unrealcv.d_pick_up(self.get_delivery_man_name(delivery_man_id), object_name)



