import time
import math
import numpy as np
from Base.DeliveryManState import DeliveryManPhysicalState, DeliveryManDrivingState, DeliveryManWalkingState
from Base.PIDController import PIDController
from Config import Config

class SilentNavigator:
    def __init__(self):
        pass

    def move_to_next_waypoint_silent(self, delivery_man):
        if delivery_man.next_waypoint is None:
            raise ValueError(f"DeliveryMan {delivery_man.id} has no next waypoint")

        delivery_man.logger.info(f"DeliveryMan {delivery_man.id} is moving to {delivery_man.next_waypoint} (silent mode)")
        if delivery_man.physical_state == DeliveryManPhysicalState.WALKING:
            distance = delivery_man.position.distance(delivery_man.next_waypoint)
            dt = distance / delivery_man.move_speed
            time.sleep(dt)
            delivery_man.lose_energy(dt)
            delivery_man.position = delivery_man.next_waypoint
            delivery_man.walking_state = DeliveryManWalkingState.STOP
        elif delivery_man.physical_state == DeliveryManPhysicalState.DRIVING:
            distance = delivery_man.position.distance(delivery_man.next_waypoint)
            dt = distance / delivery_man.move_speed
            time.sleep(dt)
            delivery_man.lose_energy(dt)
            delivery_man.position = delivery_man.next_waypoint
            delivery_man.driving_state = DeliveryManDrivingState.WAITING

class Navigator:
    def __init__(self, communicator, exit_event, dt):
        self.communicator = communicator
        self.exit_event = exit_event
        self.dt = dt

        self.steering_pid = PIDController(k_p=Config.PID_KP, k_i=Config.PID_KI, k_d=Config.PID_KD)

    def walk_to_next_waypoint(self, delivery_man):
        delivery_man.logger.info(f"DeliveryMan {delivery_man.id} is walking from {delivery_man.position} to {delivery_man.next_waypoint}")
        while not self.exit_event.is_set() and not self.walk_arrive_at_waypoint(delivery_man):
            delivery_man.lose_energy(self.dt)

            if delivery_man.walking_state == DeliveryManWalkingState.TURN_AROUND:
                if self.complete_turn(delivery_man):
                    delivery_man.walking_state = DeliveryManWalkingState.MOVE_FORWARD
            else:
                if delivery_man.walking_state == DeliveryManWalkingState.STOP:
                    delivery_man.walking_state = DeliveryManWalkingState.MOVE_FORWARD
                    self.communicator.delivery_man_move_forward(delivery_man.id)

                angle, turn_direction = self.compute_walking_control(delivery_man)
                if angle != 0:
                    delivery_man.walking_state = DeliveryManWalkingState.TURN_AROUND
                    self.communicator.delivery_man_rotate(delivery_man.id, angle, turn_direction)
            time.sleep(self.dt)
        delivery_man.walking_state = DeliveryManWalkingState.STOP
        self.communicator.delivery_man_stop(delivery_man.id)
        delivery_man.logger.info(f"DeliveryMan {delivery_man.id} has arrived at {delivery_man.next_waypoint}")

    def drive_to_next_waypoint(self, delivery_man):
        delivery_man.logger.info(f"DeliveryMan {delivery_man.id} is driving from {delivery_man.position} to {delivery_man.next_waypoint}")

        target_x, target_y = delivery_man.next_waypoint.x, delivery_man.next_waypoint.y

        # Compute target yaw
        delta_x = target_x - delivery_man.position.x
        delta_y = target_y - delivery_man.position.y
        target_yaw = np.degrees(np.arctan2(delta_y, delta_x))

        # Compute yaw error
        yaw_error = target_yaw - delivery_man.yaw
        if yaw_error > 180:
            yaw_error -= 360
        elif yaw_error < -180:
            yaw_error += 360

        if abs(yaw_error) > 160:
            delivery_man.driving_state = DeliveryManDrivingState.MAKING_U_TURN
            self.communicator.scooter_making_u_turn(delivery_man.id)

        while not self.exit_event.is_set() and not self.drive_arrive_at_waypoint(delivery_man):
            delivery_man.lose_energy(self.dt)
            if delivery_man.driving_state == DeliveryManDrivingState.MAKING_U_TURN:
                if self.completed_u_turn(delivery_man):
                    delivery_man.driving_state = DeliveryManDrivingState.MOVING
                    self.steering_pid.reset()
                else:
                    continue

            throttle, brake, steering, is_updated = self.compute_driving_control(delivery_man)
            if not delivery_man.driving_state == DeliveryManDrivingState.MOVING:
                delivery_man.driving_state = DeliveryManDrivingState.MOVING
            if is_updated:
                delivery_man.set_driving_state(throttle, brake, steering)
                self.communicator.scooter_set_state(delivery_man.id, throttle, brake, steering)

            time.sleep(self.dt)
        delivery_man.driving_state = DeliveryManDrivingState.WAITING
        self.communicator.scooter_set_state(delivery_man.id, 0, 1, 0)
        delivery_man.set_driving_state(0, 0, 0)
        delivery_man.logger.info(f"DeliveryMan {delivery_man.id} has arrived at {delivery_man.next_waypoint}")

    def move_to_next_waypoint(self, delivery_man):
        if delivery_man.next_waypoint is None:
            raise ValueError(f"DeliveryMan {delivery_man.id} has no next waypoint")

        if delivery_man.physical_state == DeliveryManPhysicalState.WALKING:
            self.walk_to_next_waypoint(delivery_man)
        elif delivery_man.physical_state == DeliveryManPhysicalState.DRIVING:
            self.drive_to_next_waypoint(delivery_man)

    def drive_arrive_at_waypoint(self, delivery_man):
        if delivery_man.position.distance(delivery_man.next_waypoint) < Config.DELIVERY_MAN_DRIVE_ARRIVE_WAYPOINT_DISTANCE:
            return True
        return False

    def walk_arrive_at_waypoint(self, delivery_man):
        if delivery_man.position.distance(delivery_man.next_waypoint) < Config.DELIVERY_MAN_WALK_ARRIVE_WAYPOINT_DISTANCE:
            return True
        return False

    def compute_driving_control(self, delivery_man):
        """Compute throttle, brake, steering"""
        target_x, target_y = delivery_man.next_waypoint.x, delivery_man.next_waypoint.y

        # Compute target yaw
        delta_x = target_x - delivery_man.position.x
        delta_y = target_y - delivery_man.position.y
        target_yaw = np.degrees(np.arctan2(delta_y, delta_x))

        # Compute yaw error
        yaw_error = target_yaw - delivery_man.yaw
        if yaw_error > 180:
            yaw_error -= 360
        elif yaw_error < -180:
            yaw_error += 360

        if abs(yaw_error) < 2:
            yaw_error = 0
        steering = self.steering_pid.update(yaw_error, self.dt)
        steering = np.clip(steering, -0.5, 0.5)

        throttle = 0.5
        brake = 0

        if throttle == delivery_man.throttle and brake == delivery_man.brake and steering == delivery_man.steering:
            return throttle, brake, steering, False

        delivery_man.throttle = throttle
        delivery_man.brake = brake
        delivery_man.steering = steering

        return throttle, brake, steering, True

    def compute_walking_control(self, delivery_man):
        to_waypoint = delivery_man.next_waypoint - delivery_man.position

        angle = math.degrees(math.acos(np.clip(delivery_man.direction.dot(to_waypoint.normalize()), -1, 1)))

        cross_product = delivery_man.direction.cross(to_waypoint)
        turn_direction = 'left' if cross_product < 0 else 'right'

        if angle < 2:
            return 0, None
        else:
            return angle, turn_direction

    def complete_turn(self, delivery_man):
        if delivery_man.walking_state == DeliveryManWalkingState.TURN_AROUND:
            to_waypoint = delivery_man.next_waypoint - delivery_man.position
            angle = math.degrees(math.acos(np.clip(delivery_man.direction.dot(to_waypoint.normalize()), -1, 1)))
            # complete turn if agent is facing the waypoint or the angle has passed the point
            return angle < 5
        return False

    def completed_u_turn(self, delivery_man):
        # Calculate angle between vehicle direction and lane direction
        to_target = delivery_man.next_waypoint - delivery_man.position
        angle_diff = abs(np.degrees(np.arccos(delivery_man.direction.dot(to_target.normalize()))))
        return angle_diff < 10