import base64
import io
import cv2
import re
import json
from PIL import Image
import numpy as np
import time

from simworld.llm.base_llm import BaseLLM
from agent.action_space import ActionSpace


class NavLLM(BaseLLM):
    def __init__(self, model_name, url, api_key):
        super().__init__(model_name, url, api_key)

    def generate_nav_instructions_openai(self, images, system_prompt, user_prompt, max_tokens=None, 
                                         temperature=0.7, top_p=1.0, response_format=ActionSpace):
        start_time = time.time()
        user_content = []
        user_content.append({"type": "text", "text": user_prompt})

        for image in images:
            img_data = self._process_image_to_base64(image)
            user_content.append({
                "type": "image_url",
                "image_url": {"url": f"data:image/jpeg;base64,{img_data}"}
            })
        # print('user_content', user_content)

        try:
            response = self.client.beta.chat.completions.parse(
                model=self.model_name,
                messages=[{"role": "system", "content": system_prompt}, {"role": "user", "content": user_content}],
                max_tokens=max_tokens,
                temperature=temperature,
                top_p=top_p,
                response_format=response_format,
            )
            action_json = response.choices[0].message.content
        except Exception as e:
            print(f"Error in generate_nav_instructions_openai: {e}")
            action_json = None

        return action_json, time.time() - start_time
        
    


    def generate_nav_instructions_openrouter(self, images, system_prompt, user_prompt, max_tokens=None, 
                                         temperature=0.7, top_p=1.0, response_format=ActionSpace):
        start_time = time.time()
        user_content = []
        user_prompt += "\nPlease respond in valid JSON format following this schema: " + str(response_format.to_json_schema())
        user_content.append({"type": "text", "text": user_prompt})

        for image in images:
            img_data = self._process_image_to_base64(image)
            user_content.append({
                "type": "image_url",
                "image_url": {"url": f"data:image/jpeg;base64,{img_data}"}
            })
        # print('user_content', user_content)

        action_response = None
        try:
            response = self.client.beta.chat.completions.parse(
                model=self.model_name,
                messages=[{"role": "system", "content": system_prompt}, {"role": "user", "content": user_content}],
                max_tokens=max_tokens,
                temperature=temperature,
                top_p=top_p,
            )
            action_response = response.choices[0].message.content
        except Exception as e:
            print(f"Error in generate_nav_instructions_openrouter: {e}")
            action_response = None
        
        if action_response is None:
            print("Warning: Failed to get action response, using default")
            action_json = None
        else:
            action_json = self._extract_json_and_fix_escapes(action_response)

        return action_json, time.time() - start_time
        

    def _process_image_to_base64(self, image: np.ndarray) -> str:
        """Convert numpy array image to base64 string.

        Args:
            image (np.ndarray): Image array (1 or 3 channels)

        Returns:
            str: Base64 encoded image string
        """
        # Convert single channel to 3 channels if needed
        if len(image.shape) == 2 or (len(image.shape) == 3 and image.shape[2] == 1):
            image = cv2.cvtColor(image, cv2.COLOR_GRAY2RGB)

        # Ensure uint8 type
        if image.dtype != np.uint8:
            image = (image * 255).astype(np.uint8)

        # Convert to PIL Image
        pil_image = Image.fromarray(image)

        # Convert to base64
        buffered = io.BytesIO()
        pil_image.save(buffered, format="JPEG")
        img_str = base64.b64encode(buffered.getvalue()).decode()

        return img_str

    def _extract_json_and_fix_escapes(self,text):
        # Extract content from first { to last }
        pattern = r'(\{.*\})'
        match = re.search(pattern, text, re.DOTALL)

        if match:
            json_str = match.group(1)
            # Fix invalid escape sequences in JSON
            fixed_json = re.sub(r'\\(?!["\\/bfnrt]|u[0-9a-fA-F]{4})', r'\\\\', json_str)
            try:
                # Try to parse the fixed JSON
                json_obj = json.loads(fixed_json)
                return json_obj
            except json.JSONDecodeError as e:
                print(f'JSON parsing error: {e}')
                print(f'Fixed JSON: {fixed_json}')
                # Return the fixed string if parsing fails
                return fixed_json
        else:
            print(f'No JSON found in the text: {text}')
            return None
