import time
import random
from threading import Lock
from utils.util import estimated_delivery_time, estimated_delivery_step

# TODO: add order state
class Order:
    _id_counter = 0
    def __init__(self, customer_position, store_position, map_obj):
        self.id = Order._id_counter
        Order._id_counter += 1

        self.max_sale_price = random.uniform(140, 150)
        self.min_sale_price = random.uniform(100, 120)
        self.sale_price = None
        self.customer_position = customer_position
        self.store_position = store_position
        self.estimated_time = estimated_delivery_time(customer_position, store_position)
        self.estimated_step = estimated_delivery_step(map_obj, customer_position, store_position)
        self.start_time = time.time()
        self.spent_step = 0
        self.delivery_time = 0
        self.delivery_men = []
        self.bids = []
        self.has_picked_up = False
        self.has_delivered = False

        # shared order related
        self.is_shared = False
        self.meeting_point = None


        self.lock = Lock()

    def get_dict(self):
        return {
            "id": self.id,
            "max_sale_price": self.max_sale_price,
            "min_sale_price": self.min_sale_price,
            "customer_position": {"x": self.customer_position.x, "y": self.customer_position.y} if self.customer_position else None,
            "store_position": {"x": self.store_position.x, "y": self.store_position.y} if self.store_position else None,
            "has_picked_up": self.has_picked_up,
            "has_delivered": self.has_delivered,
            "estimated_time": self.estimated_time,
            "is_shared": self.is_shared,
            "meeting_point": {"x": self.meeting_point.x, "y": self.meeting_point.y} if self.meeting_point else None,
            "spent_time": self.spent_time,
            "delivery_time": self.delivery_time,
            "delivery_men": [dm.id for dm in self.delivery_men],
            "bids": [(dm.id, price) for dm, price in self.bids],
        }

    def __str__(self):
        return f"Order(max_sale_price={self.max_sale_price}, min_sale_price={self.min_sale_price}, customer_position={self.customer_position}, store_position={self.store_position}, has_picked_up={self.has_picked_up}, estimated_time={self.estimated_time}, is_shared={self.is_shared}, meeting_point={self.meeting_point}, spent_time={self.spent_time})"

    def __repr__(self):
        return f"Order(max_sale_price={self.max_sale_price}, min_sale_price={self.min_sale_price}, customer_position={self.customer_position}, store_position={self.store_position}, has_picked_up={self.has_picked_up}, estimated_time={self.estimated_time}, is_shared={self.is_shared}, meeting_point={self.meeting_point}, spent_time={self.spent_time})"

    def copy(self, map_obj):
        return Order(self.customer_position, self.store_position, map_obj)

    def add_delivery_man(self, delivery_man):
        with self.lock:
            self.delivery_men.append(delivery_man)

    def get_delivery_men(self):
        with self.lock:
            return self.delivery_men.copy()

    def get_delivery_men_num(self):
        with self.lock:
            return len(self.delivery_men)

    def distribute_payment(self):
        if not self.has_delivered:
            raise ValueError("Order has not been delivered")
        with self.lock:
            self.delivery_time = self.spent_time

        amount = self.sale_price
        if self.delivery_time > self.estimated_time:
            deduction = min((self.delivery_time - self.estimated_time) * 0.3, self.sale_price * 0.5)
            amount = self.sale_price - deduction

        for delivery_man in self.delivery_men:
            delivery_man.earn_money(amount / len(self.delivery_men), self.is_shared, 0 if self.delivery_time <= self.estimated_time else self.delivery_time - self.estimated_time, self.delivery_time, self.spent_step - self.estimated_step, self.spent_step)

    @property
    def spent_time(self):
        return time.time() - self.start_time

    def bid_order(self, delivery_man, bid_price: float):
        with self.lock:
            # remove all previous bids
            self.bids = [(dm, price) for dm, price in self.bids if dm != delivery_man]
            # add new bid
            self.bids.append((delivery_man, bid_price))
            # keep the lowest bid
            temp = {}
            for dm, price in self.bids:
                if dm not in temp or price < temp[dm]:
                    temp[dm] = price
            self.bids = [(dm, price) for dm, price in temp.items()]

