import json
import statistics
import os
import csv
import datetime
from typing import List, Dict, Any

from Base import DeliveryMan
from Config import Config


class Evaluator:
    """
    Evaluate detailed performance metrics for a fleet of delivery men.
    """

    def __init__(self, delivery_manager):
        """
        Initialize with a delivery manager that provides DeliveryMan instances.
        :param delivery_manager: Manager object with get_delivery_men() method.
        """
        self.delivery_manager = delivery_manager
        self.delivery_men: List[DeliveryMan] = delivery_manager.get_delivery_men()
        self.num_men: int = len(self.delivery_men)
        # Track initial money for each delivery man
        self.initial_money = {dm.id: dm.money for dm in self.delivery_men}

    def evaluate(self, state_path: str, output_path: str, step) -> None:
        """
        Compute metrics and write results to a CSV file.
        :param output_path: File path for CSV output.
        :param step: Current simulation step.
        """
        metrics = self._collect_metrics()

        # Check if file exists to determine if headers need to be written
        file_exists = os.path.isfile(output_path)

        # Get formatted timestamp
        current_time = datetime.datetime.now().strftime("%y-%m-%d:%H-%M-%S")

        # Write to CSV file
        with open(output_path, 'a', newline='') as csv_file:
            csv_writer = csv.writer(csv_file)

            # Process each model group
            for model_name, model_metrics in metrics.items():
                # Build CSV row data
                # Start with step, timestamp, and model name
                row_data = [step, current_time, model_name]

                # Initialize headers with step, timestamp, and model_name
                headers = ['step', 'timestamp', 'model_name']

                # Add all metrics with their three statistics (avg, median, stdev)
                for metric_name, metric_values in model_metrics.items():
                    # Add average value
                    row_data.append(metric_values['avg'])
                    headers.append(f"{metric_name}_avg")
                    # Add median value
                    row_data.append(metric_values['median'])
                    headers.append(f"{metric_name}_median")
                    # Add standard deviation
                    row_data.append(metric_values['stdev'])
                    headers.append(f"{metric_name}_stdev")

                # If file doesn't exist, write headers first
                if not file_exists:
                    csv_writer.writerow(headers)
                    file_exists = True

                # Write current step's data
                csv_writer.writerow(row_data)

        # state = self.save_state(step)
        # with open(state_path, 'a', newline='') as json_file:
        #     json.dump(state, json_file, indent=4)

    @staticmethod
    def _aggregate(values: List[float]) -> Dict[str, float]:
        """
        Aggregate a list of floats into average, median, and standard deviation.
        Returns zeros if list is empty or single element.
        """
        if not values:
            return {"avg": 0.0, "median": 0.0, "stdev": 0.0}
        return {
            "avg": statistics.mean(values),
            "median": statistics.median(values),
            "stdev": statistics.stdev(values) if len(values) > 1 else 0.0
        }

    # ----- Individual metric getters -----
    def _profit(self, dm: DeliveryMan) -> float:
        return dm.money - self.initial_money[dm.id]

    @staticmethod
    def _revenue(dm: DeliveryMan) -> float:
        return dm.revenue

    @staticmethod
    def _consumed_energy(dm: DeliveryMan) -> float:
        return dm.total_consumed_energy

    @staticmethod
    def _success_orders(dm: DeliveryMan) -> int:
        return dm.success_order_num

    @staticmethod
    def _shared_success_orders(dm: DeliveryMan) -> int:
        return dm.success_shared_order_num

    @staticmethod
    def _delay_rate(dm: DeliveryMan) -> float:
        return (dm.delay_order_num / dm.success_order_num
                if dm.success_order_num > 0 else 0.0)

    @staticmethod
    def _spent_time(dm: DeliveryMan) -> float:
        return dm.total_spent_time

    @staticmethod
    def _spent_step(dm: DeliveryMan) -> float:
        return dm.total_spent_step

    @staticmethod
    def _delay_step(dm: DeliveryMan) -> float:
        return dm.total_delay_step

    @staticmethod
    def _step_delay_rate(dm: DeliveryMan) -> float:
        return (dm.step_delay_order_num / dm.total_spent_step
                if dm.total_spent_step > 0 else 0.0)

    @staticmethod
    def _delay_time(dm: DeliveryMan) -> float:
        return dm.total_delay_time

    @staticmethod
    def _open_shared(dm: DeliveryMan) -> int:
        return dm.open_shared_order_num

    # Invalid decision rates
    @staticmethod
    def _invalid_rate(dm: DeliveryMan) -> float:
        decision_num = sum([dm.bid_order_num, dm.pick_up_order_num, dm.deliver_order_num, dm.buy_beverage_num, dm.open_shared_order_num, dm.go_to_meeting_point_num, dm.cancel_order_num, dm.change_speed_num, dm.buy_bike_num, dm.do_nothing_num])
        return dm.invalid_num / decision_num if decision_num > 0 else 0.0

    @staticmethod
    def _invalid_bid(dm: DeliveryMan) -> float:
        return dm.invalid_bid_order_num / dm.bid_order_num if dm.bid_order_num > 0 else 0.0

    @staticmethod
    def _invalid_pickup(dm: DeliveryMan) -> float:
        return dm.invalid_pick_up_order_num / dm.pick_up_order_num if dm.pick_up_order_num > 0 else 0.0

    @staticmethod
    def _invalid_deliver(dm: DeliveryMan) -> float:
        return dm.invalid_deliver_order_num / dm.deliver_order_num if dm.deliver_order_num > 0 else 0.0

    @staticmethod
    def _invalid_beverage(dm: DeliveryMan) -> float:
        return dm.invalid_buy_beverage_num / dm.buy_beverage_num if dm.buy_beverage_num > 0 else 0.0

    @staticmethod
    def _invalid_open_shared(dm: DeliveryMan) -> float:
        return dm.invalid_open_shared_order_num / dm.open_shared_order_num if dm.open_shared_order_num > 0 else 0.0

    @staticmethod
    def _invalid_meet(dm: DeliveryMan) -> float:
        return dm.invalid_go_to_meeting_point_num / dm.go_to_meeting_point_num if dm.go_to_meeting_point_num > 0 else 0.0

    @staticmethod
    def _invalid_cancel(dm: DeliveryMan) -> float:
        return dm.invalid_cancel_order_num / dm.cancel_order_num if dm.cancel_order_num > 0 else 0.0

    @staticmethod
    def _invalid_speed(dm: DeliveryMan) -> float:
        return dm.invalid_change_speed_num / dm.change_speed_num if dm.change_speed_num > 0 else 0.0

    @staticmethod
    def _invalid_bike(dm: DeliveryMan) -> float:
        return dm.invalid_buy_bike_num / dm.buy_bike_num if dm.buy_bike_num > 0 else 0.0

    @staticmethod
    def _energy_efficiency(dm: DeliveryMan) -> float:
        return (dm.money / dm.total_consumed_energy
                if dm.total_consumed_energy > 0 else 0.0)

    def _collect_metrics(self) -> Dict[str, Any]:
        """
        Collect metrics for all delivery men and return aggregated results.
        """
        # Group delivery men by model_name
        model_groups = {}
        for dm in self.delivery_men:
            model_name = dm.llm.model_name
            if model_name not in model_groups:
                model_groups[model_name] = []
            model_groups[model_name].append(dm)

        # Initialize results dictionary
        results = {}

        # Process each model group
        for model_name, dms in model_groups.items():
            # Gather raw values for this model group
            profits = [self._profit(dm) for dm in dms]
            revenues = [self._revenue(dm) for dm in dms]
            energies = [self._consumed_energy(dm) for dm in dms]
            spent_times = [self._spent_time(dm) for dm in dms]
            delay_times = [self._delay_time(dm) for dm in dms]
            delay_rates = [self._delay_rate(dm) for dm in dms]
            shared_counts = [self._open_shared(dm) for dm in dms]
            invalids = [self._invalid_rate(dm) for dm in dms]
            success_orders = [self._success_orders(dm) for dm in dms]
            spent_steps = [self._spent_step(dm) for dm in dms]
            delay_steps = [self._delay_step(dm) for dm in dms]
            step_delay_rates = [self._step_delay_rate(dm) for dm in dms]
            # Detailed invalid rates
            invalid_bids = [self._invalid_bid(dm) for dm in dms]
            invalid_pickups = [self._invalid_pickup(dm) for dm in dms]
            invalid_delivers = [self._invalid_deliver(dm) for dm in dms]
            invalid_bevs = [self._invalid_beverage(dm) for dm in dms]
            invalid_opens = [self._invalid_open_shared(dm) for dm in dms]
            invalid_meets = [self._invalid_meet(dm) for dm in dms]
            invalid_cancels = [self._invalid_cancel(dm) for dm in dms]
            invalid_speeds = [self._invalid_speed(dm) for dm in dms]
            invalid_bikes = [self._invalid_bike(dm) for dm in dms]
            # Energy efficiency
            efficiencies = [self._energy_efficiency(dm) for dm in dms]

            # Add metrics for this model group
            results[model_name] = {
                "profit": self._aggregate(profits),
                "success_orders": self._aggregate(success_orders),
                "revenue": self._aggregate(revenues),
                "consumed_energy": self._aggregate(energies),
                "waiting_time": self._aggregate(spent_times),
                "delay_time": self._aggregate(delay_times),
                "delay_rate": self._aggregate(delay_rates),
                "shared_order_count": self._aggregate(shared_counts),
                "energy_efficiency": self._aggregate(efficiencies),
                "invalid_decision_rate": self._aggregate(invalids),
                "spent_step": self._aggregate(spent_steps),
                "delay_step": self._aggregate(delay_steps),
                "step_delay_rate": self._aggregate(step_delay_rates),
                # Detailed invalid rates
                "invalid_bid_order_rate": self._aggregate(invalid_bids),
                "invalid_pick_up_order_rate": self._aggregate(invalid_pickups),
                "invalid_deliver_order_rate": self._aggregate(invalid_delivers),
                "invalid_buy_beverage_rate": self._aggregate(invalid_bevs),
                "invalid_open_shared_order_rate": self._aggregate(invalid_opens),
                "invalid_go_to_meeting_point_rate": self._aggregate(invalid_meets),
                "invalid_cancel_order_rate": self._aggregate(invalid_cancels),
                "invalid_change_speed_rate": self._aggregate(invalid_speeds),
                "invalid_buy_bike_rate": self._aggregate(invalid_bikes),
            }

        return results

    def save_state(self, step: int) -> None:
        """
        Save the current state of the delivery manager and all delivery men to a JSON file.
        :param step: Current simulation step.
        """
        # 只保存关键状态信息
        state = {
            "step": step,
            "timestamp": datetime.datetime.now().strftime("%y-%m-%d:%H-%M-%S"),
            "metrics": {
                "total_orders": len(self.delivery_manager.orders),
                "active_delivery_men": len(self.delivery_men),
                "total_revenue": sum(dm.revenue for dm in self.delivery_men),
                "total_energy_consumed": sum(dm.total_consumed_energy for dm in self.delivery_men)
            },
            "delivery_men": [
                {
                    "id": dm.id,
                    "model_name": dm.llm.model_name,
                    "position": {"x": dm.position.x, "y": dm.position.y},
                    "state": dm.state.value,
                    "energy": dm.energy,
                    "money": dm.money,
                    "current_order": dm.current_order.id if dm.current_order else None,
                    "orders": [order.id for order in dm.orders]
                }
                for dm in self.delivery_men
            ],
            "orders": [
                {
                    "id": order.id,
                    "status": {
                        "picked_up": order.has_picked_up,
                        "delivered": order.has_delivered,
                        "shared": order.is_shared,
                        "meeting_point": {"x": order.meeting_point.x, "y": order.meeting_point.y} if order.meeting_point else None,
                        "bids": [(dm.id, price) for dm, price in order.bids]
                    },
                    "positions": {
                        "store": {"x": order.store_position.x, "y": order.store_position.y},
                        "customer": {"x": order.customer_position.x, "y": order.customer_position.y}
                    }
                }
                for order in self.delivery_manager.orders
            ]
        }
        return state

    def evaluate_individual(self, output_path: str, step) -> None:
        """
        Compute metrics for each individual delivery man and write results to a CSV file.
        :param output_path: File path for CSV output.
        :param step: Current simulation step.
        """
        # Check if file exists to determine if headers need to be written
        file_exists = os.path.isfile(output_path)

        # Get formatted timestamp
        current_time = datetime.datetime.now().strftime("%y-%m-%d:%H-%M-%S")

        # Write to CSV file
        with open(output_path, 'a', newline='') as csv_file:
            csv_writer = csv.writer(csv_file)

            # Define headers
            headers = ['step', 'timestamp', 'id', 'model_name', 
                      'profit', 'revenue', 'consumed_energy', 'success_orders',
                      'spent_time', 'delay_time', 'delay_rate', 'shared_order_count',
                      'energy_efficiency', 'invalid_decision_rate', 'spent_step',
                      'delay_step', 'step_delay_rate', 'invalid_bid_order_rate',
                      'invalid_pick_up_order_rate', 'invalid_deliver_order_rate',
                      'invalid_buy_beverage_rate', 'invalid_open_shared_order_rate',
                      'invalid_go_to_meeting_point_rate', 'invalid_cancel_order_rate',
                      'invalid_change_speed_rate', 'invalid_buy_bike_rate']

            # If file doesn't exist, write headers first
            if not file_exists:
                csv_writer.writerow(headers)

            # Process each delivery man
            for dm in self.delivery_men:
                # Calculate all metrics for this delivery man
                row_data = [
                    step,
                    current_time,
                    dm.id,
                    dm.llm.model_name,
                    self._profit(dm),
                    self._revenue(dm),
                    self._consumed_energy(dm),
                    self._success_orders(dm),
                    self._spent_time(dm),
                    self._delay_time(dm),
                    self._delay_rate(dm),
                    self._open_shared(dm),
                    self._energy_efficiency(dm),
                    self._invalid_rate(dm),
                    self._spent_step(dm),
                    self._delay_step(dm),
                    self._step_delay_rate(dm),
                    self._invalid_bid(dm),
                    self._invalid_pickup(dm),
                    self._invalid_deliver(dm),
                    self._invalid_beverage(dm),
                    self._invalid_open_shared(dm),
                    self._invalid_meet(dm),
                    self._invalid_cancel(dm),
                    self._invalid_speed(dm),
                    self._invalid_bike(dm)
                ]

                # Write current delivery man's data
                csv_writer.writerow(row_data)