from typing import List, Tuple

class Memory:
    """
    Memory buffer to be used by LLM agent
    """
    msgs: List[Tuple[str, str]] = []

    def __init__(self):
        # Do nothing
        self.msgs = []

    def add_memory(self, user_msg: str, agent_response: str):
        """
        Method to add information into memory

        Parameters
        ----------
        user_msg : str
            The user message to store
        agent_response : str
            The agent response to user message
        """
        self.msgs.append((user_msg, agent_response))

    def get_memory(self) -> List[Tuple[str, str]]:
        """
        Method to get information from memory

        Returns
        -------
        List[Tuple[str, str]]
            List of (user_msg, agent_response) tuples saved in memory
        """
        return self.msgs

    def clear_memory(self):
        self.msgs.clear()


if __name__ == "__main__":
    import os
    import openai

    client = openai.OpenAI(api_key=os.getenv("OPENAI_API_KEY", None))

    system_msg = "You are an encyclopedia containing all the world's knowledge"
    user_msgs = [
        "What was the famous equation formulated by Albert Einstein?",
        "When was he born?",
        "Where did he study?",
    ]

    agent_memory = Memory()

    for msg in user_msgs:
        chat_msgs = [{"role": "system", "content": system_msg}]

        for user_msg, agent_response in agent_memory.get_memory():
            chat_msgs.append({"role": "user", "content": user_msg})
            chat_msgs.append({"role": "assistant", "content": agent_response})

        chat_msgs.append({"role": "user", "content": msg})

        response = client.chat.completions.create(
            model="gpt-3.5-turbo",
            messages=chat_msgs,
        )

        print(msg)
        print(response.choices[0].message.content)
        print()
        agent_memory.add_memory(msg, response.choices[0].message.content)
