import json
from pydantic import BaseModel
from enum import Enum
from typing import Optional
from simworld.utils.vector import Vector

class Action(Enum):
    DO_NOTHING = 0
    STEP_FORWARD = 1
    TURN_AROUND = 2
    CHOOSE_WAYPOINT = 3

    def __str__(self):
        return self.name

    def __repr__(self):
        return self.__str__()

class ActionSpace(BaseModel):
    choice: Action = Action.DO_NOTHING
    duration: Optional[float] = None
    direction: Optional[int] = None
    angle: Optional[float] = None
    clockwise: Optional[bool] = None
    new_waypoint: Optional[Vector] = None

    def __str__(self):
        return f"ActionSpace(choice={self.choice}, duration={self.duration}, direction={self.direction}, angle={self.angle}, clockwise={self.clockwise}, new_waypoint={self.new_waypoint})"

    def __repr__(self):
        return self.__str__()
    
    @classmethod
    def from_json(cls, json_str):
        if type(json_str) == str:
            try:
                json_str = json.loads(json_str)
            except Exception as e:
                print(f"parse action space from json failed: {e}, using default action: DO_NOTHING")
                return cls()
        try:
            choice = json_str.get('choice', 0)
            duration = json_str.get('duration', 0) if json_str.get('duration', 0) is not None else None
            direction = json_str.get('direction', 0) if json_str.get('direction', 0) is not None else None
            angle = json_str.get('angle', 0) if json_str.get('angle', 0) is not None else None
            clockwise = json_str.get('clockwise', True) if json_str.get('clockwise', True) is not None else None
            new_waypoint = Vector(json_str.get('new_waypoint', [0, 0])) if json_str.get('new_waypoint', [0, 0]) is not None else None
            return cls(choice=choice, duration=duration, direction=direction, angle=angle, clockwise=clockwise, new_waypoint=new_waypoint)
        except Exception as e:
            print(f"parse action space from json failed: {e}, using default action: DO_NOTHING")
            return cls()

    @classmethod
    def to_json_schema(cls):
        return {
            'name': 'ActionSpace',
            'strict': True,
            'schema': {
                'type': 'object',
                'properties': {
                    'choice': {'type': 'integer', 'description': 'The choice of the action. The action can be one of the following: 0: DO_NOTHING, 1: STEP_FORWARD, 2: TURN_AROUND, 3: CHOOSE_WAYPOINT.'},
                    'duration': {'type': 'number', 'description': 'The duration of step forward. The duration is the time of the action.'},
                    'direction': {'type': 'number', 'description': 'The direction of step forward. The direction is the direction of the action. 0 means forward, 1 means backward.'},
                    'angle': {'type': 'number', 'description': 'The angle of turn around. The angle is the angle of the action.'},
                    'clockwise': {'type': 'boolean', 'description': 'The clockwise of turn around. The clockwise is the direction of the action.'},
                    'new_waypoint': {'type': 'array', 'description': 'The new waypoint of the action. The new waypoint is the new waypoint of the action. Example: [10, 10]'},
                },
                'required': ['choice']
            }
        }


