import torch.nn as nn
import torch.nn.functional as F


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_channels, out_channels, stride=1):
        super(BasicBlock, self).__init__()
        self.conv1 = nn.Conv2d(
            in_channels, out_channels, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(out_channels)
        self.conv2 = nn.Conv2d(
            out_channels, out_channels, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(out_channels)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_channels != self.expansion * out_channels:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_channels, self.expansion * out_channels,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion * out_channels)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, in_channels, out_channels, stride=1):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv2d(in_channels, out_channels, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(out_channels)
        self.conv2 = nn.Conv2d(out_channels, out_channels, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(out_channels)
        self.conv3 = nn.Conv2d(out_channels, out_channels * self.expansion, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(out_channels * self.expansion)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_channels != out_channels * self.expansion:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_channels, out_channels * self.expansion,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(out_channels * self.expansion)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = F.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class ResNet(nn.Module):
    def __init__(self, block, layers, in_channels=3, n_kernels=64, out_dim=200):
        super(ResNet, self).__init__()
        self.in_channels = n_kernels

        self.conv1 = nn.Conv2d(in_channels, n_kernels, kernel_size=7, stride=2, padding=3, bias=False)
        self.bn1 = nn.BatchNorm2d(n_kernels)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)

        self.layer1 = self._make_layer(block, n_kernels, layers[0], stride=1)
        self.layer2 = self._make_layer(block, n_kernels * 2, layers[1], stride=2)
        self.layer3 = self._make_layer(block, n_kernels * 4, layers[2], stride=2)
        self.layer4 = self._make_layer(block, n_kernels * 8, layers[3], stride=2)

        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.fc1 = nn.Linear(n_kernels * 8 * block.expansion, 500)
        self.fc2 = nn.Linear(500, out_dim)

    def _make_layer(self, block, out_channels, blocks, stride=1):
        layers = []
        layers.append(block(self.in_channels, out_channels, stride))
        self.in_channels = out_channels * block.expansion
        for _ in range(1, blocks):
            layers.append(block(self.in_channels, out_channels))

        return nn.Sequential(*layers)

    def forward(self, x):
        x = F.relu(self.bn1(self.conv1(x)))
        x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)

        x = self.avgpool(x)
        x = x.view(x.size(0), -1)

        o = F.relu(self.fc1(x))
        x = self.fc2(o)
        return x, o


def ResNet18(in_channels=3, n_kernels=64, out_dim=200):
    return ResNet(BasicBlock, [2, 2, 2, 2], in_channels, n_kernels, out_dim)


def ResNet34(in_channels=3, n_kernels=64, out_dim=200):
    return ResNet(BasicBlock, [3, 4, 6, 3], in_channels, n_kernels, out_dim)


def ResNet50(in_channels=3, n_kernels=64, out_dim=200):
    return ResNet(Bottleneck, [3, 4, 6, 3], in_channels, n_kernels, out_dim)


def ResNet101(in_channels=3, n_kernels=64, out_dim=200):
    return ResNet(Bottleneck, [3, 4, 23, 3], in_channels, n_kernels, out_dim)


def ResNet152(in_channels=3, n_kernels=64, out_dim=200):
    return ResNet(Bottleneck, [3, 8, 36, 3], in_channels, n_kernels, out_dim)

