"""mobilenetv2 in pytorch



[1] Mark Sandler, Andrew Howard, Menglong Zhu, Andrey Zhmoginov, Liang-Chieh Chen

    MobileNetV2: Inverted Residuals and Linear Bottlenecks
    https://arxiv.org/abs/1801.04381
"""

import torch
import torch.nn as nn
import torch.nn.functional as F
from collections import OrderedDict
import copy

class LinearBottleNeck(nn.Module):

    def __init__(self, in_channels, out_channels, stride, t=6, class_num=100):
        super().__init__()

        self.residual = nn.Sequential(
            nn.Conv2d(in_channels, in_channels * t, 1),
            nn.BatchNorm2d(in_channels * t),
            nn.ReLU6(inplace=True),

            nn.Conv2d(in_channels * t, in_channels * t, 3, stride=stride, padding=1, groups=in_channels * t),
            nn.BatchNorm2d(in_channels * t),
            nn.ReLU6(inplace=True),

            nn.Conv2d(in_channels * t, out_channels, 1),
            nn.BatchNorm2d(out_channels)
        )

        self.stride = stride
        self.in_channels = in_channels
        self.out_channels = out_channels

    def __iter__(self):
        return iter(self.residual)


    def __len__(self) -> int:
        return self.residual.__len__()

    def forward(self, x):

        residual = self.residual(x)

        if self.stride == 1 and self.in_channels == self.out_channels:
            residual += x

        return residual

class MobileNetV2(nn.Module):

    def __init__(self, cutting_layer,num_agent,num_class):
        super().__init__()

        self.features = nn.Sequential(OrderedDict([
            ('conv0',nn.Conv2d(3, 32, kernel_size=1, padding=1)),
            ('bn1',nn.BatchNorm2d(32)),
            ('relu1',nn.ReLU6(inplace=True)),
            ('stage1',LinearBottleNeck(32, 16, 1, 1)),
            ('stage2',self._make_stage(2, 16, 24, 2, 6)),
            ('stage3',self._make_stage(3, 24, 32, 2, 6)),
            ('stage4',self._make_stage(4, 32, 64, 2, 6)),
            ('stage5',self._make_stage(3, 64, 96, 1, 6)),
            ('stage6',self._make_stage(3, 96, 160, 1, 6)),
            ('stage7',LinearBottleNeck(160, 320, 1, 6)),
            ('conv1',nn.Conv2d(320, 1280, kernel_size=1)),
            ('bn2', nn.BatchNorm2d(1280)),
            ('relu2', nn.ReLU6(inplace=True)),
            ('avgpool',nn.AdaptiveAvgPool2d((1, 1)))

        ])
        )


        self.classifier = nn.Conv2d(1280, num_class, kernel_size=1)

        # construct local and cloud
        for i in range(num_agent):
            if i == 0:
                self.local_list = [self.features[:cutting_layer]]
            else:
                self.local_list.append(copy.deepcopy(self.local_list[0]))
        self.cloud = self.features[cutting_layer::]

    def forward(self, x):
        x = self.features(x)
        x = self.conv2(x)
        x = x.view(x.size(0), -1)

        return x

    def _make_stage(self, repeat, in_channels, out_channels, stride, t):

        layers = []
        layers.append(LinearBottleNeck(in_channels, out_channels, stride, t))

        while repeat - 1:
            layers.append(LinearBottleNeck(out_channels, out_channels, 1, t))
            repeat -= 1

        return nn.Sequential(*layers)

def mobilenetv2(cutting_layer, num_agent=1, num_class=10):
    return MobileNetV2(cutting_layer,num_agent,num_class)