import torch
import torch.nn as nn
import torch.nn.functional as F
from torch.nn import init
import math
import numpy as np
import copy
def init_weights(m):
    if isinstance(m, nn.Conv2d):
        n = m.kernel_size[0] * m.kernel_size[1] * m.out_channels
        m.weight.data.normal_(0, math.sqrt(2. / n))
        #m.bias.data.zero_()
    elif isinstance(m, nn.BatchNorm2d):
      m.weight.data.fill_(1)
      m.bias.data.zero_()
    elif isinstance(m, nn.Linear):
      init.kaiming_normal(m.weight)
      m.bias.data.zero_()

class DownsampleA(nn.Module):

  def __init__(self, nIn, nOut, stride):
    super(DownsampleA, self).__init__()
    assert stride == 2
    self.avg = nn.AvgPool2d(kernel_size=1, stride=stride)

  def forward(self, x):
    x = self.avg(x)
    return torch.cat((x, x.mul(0)), 1)

class ResNetBasicblock(nn.Module):
  expansion = 1
  """
  RexNet basicblock (https://github.com/facebook/fb.resnet.torch/blob/master/models/resnet.lua)
  """
  def __init__(self, inplanes, planes, stride=1, downsample=None):
    super(ResNetBasicblock, self).__init__()

    self.conv_a = nn.Conv2d(inplanes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
    self.bn_a = nn.BatchNorm2d(planes)

    self.conv_b = nn.Conv2d(planes, planes, kernel_size=3, stride=1, padding=1, bias=False)
    self.bn_b = nn.BatchNorm2d(planes)

    self.downsample = downsample


  def forward(self, x):
    residual = x

    basicblock = self.conv_a(x)
    basicblock = self.bn_a(basicblock)
    basicblock = F.relu(basicblock, inplace=True)

    basicblock = self.conv_b(basicblock)
    basicblock = self.bn_b(basicblock)

    if self.downsample is not None:
      residual = self.downsample(x)
    
    return F.relu(residual + basicblock, inplace=True)

class CifarResNet(nn.Module):
  """
  ResNet optimized for the Cifar dataset, as specified in
  https://arxiv.org/abs/1512.03385.pdf
  """
  def __init__(self, block, depth, cutting_layer, num_agent = 1, num_class = 10):
    """ Constructor
    Args:
      depth: number of layers.
      num_classes: number of classes
      base_width: base width
    """
    super(CifarResNet, self).__init__()

    #Model type specifies number of layers for CIFAR-10 and CIFAR-100 model
    assert (depth - 2) % 6 == 0, 'depth should be one of 20, 32, 44, 56, 110'
    layer_blocks = (depth - 2) // 6
    print ('CifarResNet : Depth : {} , Layers for each block : {}'.format(depth, layer_blocks))
    self.current_agent = 0
    self.initialize = True

    layers = []
    layers.append(conv3x3(3, 16))
    self.inplanes = 16
    self.stage_1 = self._make_layer(block, 16, layer_blocks, 1)
    layers.extend(self.stage_1)
    self.stage_2 = self._make_layer(block, 32, layer_blocks, 2)
    layers.extend(self.stage_2)
    self.stage_3 = self._make_layer(block, 64, layer_blocks, 2)
    layers.extend(self.stage_3)
    local_layer_list = layers[:cutting_layer]
    cloud_layer_list = layers[cutting_layer:]

    if cutting_layer <=4:
        in_channels = 16
    elif cutting_layer <=7:
        in_channels = 32
    else:
      adds_bottleneck = False
      # in_channels = local[-1].conv_b.out_channels
      raise ValueError("Not sure about input channel size")

    self.local = nn.Sequential(*local_layer_list)
    self.cloud = nn.Sequential(*cloud_layer_list)
    self.local_list = []

    for i in range(num_agent):
        if i == 0:
            self.local_list.append(self.local)
            self.local_list[0].apply(init_weights)
        else:
            new_copy = copy.deepcopy(self.local_list[0])
            self.local_list.append(new_copy.cuda())

    self.classifier = nn.Linear(64*block.expansion, num_class)
    for m in self.cloud:
        if isinstance(m, nn.Conv2d):
            n = m.kernel_size[0] * m.kernel_size[1] * m.out_channels
            m.weight.data.normal_(0, math.sqrt(2. / n))
            if m.bias is not None:
                m.bias.data.zero_()


  def switch_model(self, agent_id):
        self.current_agent = agent_id
        self.local = self.local_list[agent_id]

  def get_smashed_data_size(self,bottleneck=False):
        with torch.no_grad():
            noise_input = torch.randn([1, 3, 32, 32])
            device = next(self.local.parameters()).device
            noise_input = noise_input.to(device)
            smashed_data = self.local(noise_input)
            if bottleneck:
                smashed_data = self.local_b_list[-1](smashed_data)
        return smashed_data.size()



  def get_MAC_param(self):
        with torch.no_grad():
            noise_input = torch.randn([1, 3, 32, 32])
            device = next(self.local.parameters()).device
            noise_input = noise_input.to(device)
            client_macs, client_params = profile(self.local, inputs=(noise_input, ))
            noise_smash = torch.randn(self.get_smashed_data_size())
            device = next(self.cloud.parameters()).device
            noise_smash = noise_smash.to(device)
            server_macs, server_params = profile(self.cloud, inputs=(noise_smash, ))
        return client_macs, client_params, server_macs, server_params

  def get_inference_time(self, federated = False):
      import time
      with torch.no_grad():
          noise_input = torch.randn([128, 3, 32, 32])
          
          if not federated:
            #CPU warm up
            self.local.cpu()
            self.local.eval()
            smashed_data = self.local(noise_input) #CPU warm up
            
            start_time = time.time()
            for _ in range(500):
                smashed_data = self.local(noise_input)
            lapse_cpu = (time.time() - start_time)/500
          else:
            self.local.cpu()
            self.cloud.cpu()
            self.classifier.cpu()
            self.local.eval()
            self.cloud.eval()
            self.classifier.eval()

            smashed_data = self.local(noise_input) #CPU warm up
            output = self.cloud(smashed_data)
            output = F.avg_pool2d(output, 8)
            output = output.view(output.size(0), -1)
            output = self.classifier(output)
            start_time = time.time()
            for _ in range(500):
                smashed_data = self.local(noise_input)
                output = self.cloud(smashed_data)
                output = F.avg_pool2d(output, 8)
                output = output.view(output.size(0), -1)
                output = self.classifier(output)
            lapse_cpu = (time.time() - start_time)/500
          
          if not federated:
            self.local.cuda()
            smashed_data = smashed_data.cuda()
            self.cloud.eval()
            #GPU-WARM-UP
            for _ in range(100):  #GPU warm up
                output = self.cloud(smashed_data)
            start_time = time.time()
            for _ in range(500):
                output = self.cloud(smashed_data)
            lapse_gpu = (time.time() - start_time)/500
          else:
            self.local.cuda()
            self.cloud.cuda()
            self.classifier.cuda()
            lapse_gpu = 0.0
          del noise_input, output, smashed_data
      return lapse_cpu, lapse_gpu

  def get_train_time(self, federated = False):
        import time
        noise_input = torch.randn([128, 3, 32, 32])
        noise_label = torch.randint(0, 10, [128, ])
        self.local.cpu()
        self.cloud.cpu()
        self.classifier.cpu()
        self.local.train()
        self.cloud.train()
        self.classifier.train()
        
        criterion = torch.nn.CrossEntropyLoss()
        
        '''Calculate client backward on CPU'''
        smashed_data = self.local(noise_input) #CPU warm up
        output = self.cloud(smashed_data)
        output = F.avg_pool2d(output, 8)
        output = output.view(output.size(0), -1)
        output = self.classifier(output)
        f_loss = criterion(output, noise_label)

        f_loss.backward()

        lapse_cpu_all = 0
        for _ in range(500):
            smashed_data = self.local(noise_input)
            output = self.cloud(smashed_data)
            output = F.avg_pool2d(output, 8)
            output = output.view(output.size(0), -1)
            output = self.classifier(output)
            f_loss = criterion(output, noise_label)
            start_time = time.time()
            f_loss.backward()
            #First time we calculate CPU overall train time.
            lapse_cpu_all += (time.time() - start_time)
        lapse_cpu_all = lapse_cpu_all / 500.

        if not federated:
          lapse_cpu_server = 0
          for _ in range(500):
              smashed_data = self.local(noise_input)
              output = self.cloud(smashed_data.detach())
              output = F.avg_pool2d(output, 8)
              output = output.view(output.size(0), -1)
              output = self.classifier(output)
              f_loss = criterion(output, noise_label)

              start_time = time.time()
              f_loss.backward()
              #First time we calculate CPU server train time by detaching smashed-data.
              lapse_cpu_server += (time.time() - start_time)
          lapse_cpu_server = lapse_cpu_server / 500.

          lapse_cpu_client = lapse_cpu_all - lapse_cpu_server
        else:
          lapse_cpu_client = lapse_cpu_all
        
        '''Calculate Server backward on GPU'''
        
        self.local.cuda()
        self.cloud.cuda()
        self.classifier.cuda()
        if not federated:
          criterion.cuda()
          noise_input = noise_input.cuda()
          noise_label = noise_label.cuda()
          
          #GPU warmup
          for _ in range(100):
              smashed_data = self.local(noise_input)
              output = self.cloud(smashed_data.detach())
              output = F.avg_pool2d(output, 8)
              output = output.view(output.size(0), -1)
              output = self.classifier(output)
              f_loss = criterion(output, noise_label)
              f_loss.backward()

          lapse_gpu_server = 0
          for _ in range(500):
              smashed_data = self.local(noise_input)
              output = self.cloud(smashed_data.detach())
              output = F.avg_pool2d(output, 8)
              output = output.view(output.size(0), -1)
              output = self.classifier(output)
              f_loss = criterion(output, noise_label)

              start_time = time.time()
              f_loss.backward()
              #First time we calculate CPU server train time by detaching smashed-data.
              lapse_gpu_server += (time.time() - start_time)
          lapse_gpu_server = lapse_gpu_server / 500.
        else:
          lapse_gpu_server = 0.0
        return lapse_cpu_client, lapse_gpu_server
  
  def _make_layer(self, block, planes, blocks, stride=1):
      downsample = None
      if stride != 1 or self.inplanes != planes * block.expansion:
          downsample = DownsampleA(self.inplanes, planes * block.expansion, stride)

      layers = []
      layers.append(block(self.inplanes, planes, stride, downsample))
      self.inplanes = planes * block.expansion
      for i in range(1, blocks):
          layers.append(block(self.inplanes, planes))
      return layers

  def forward(self, x):
      self.local_output = self.local(x)
      if self.initialize:
          self.logger.debug(f"Parameter Volume: {4 * np.prod(list(self.local_output.size()))} Byte per batch!")
          self.logger.debug(f"Size of output: {(list(self.local_output.size()))} ")
          self.initialize = False
      x = self.cloud(self.local_output)
      x = F.avg_pool2d(x, 8)
      x = x.view(x.size(0), -1)
      x = self.classifier(x)
      return x

class conv3x3(nn.Module):
    def __init__(self, in_planes, planes, stride=1):
        super(conv3x3, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=3, stride = stride, padding = 1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.relu = nn.ReLU(inplace=True)

    def forward(self, x):
        out = self.relu(self.bn1(self.conv1(x)))
        return out

def ResNet20(cutting_layer, num_agent = 1, num_class = 10):
  """Constructs a ResNet-20 model for CIFAR-10 (by default)
  Args:
    num_classes (uint): number of classes
  """
  model = CifarResNet(ResNetBasicblock, 20, cutting_layer, num_agent = num_agent, num_class = num_class)
  return model

def ResNet32(cutting_layer, num_agent = 1, num_class = 10):
  """Constructs a ResNet-32 model for CIFAR-10 (by default)
  Args:
    num_classes (uint): number of classes
  """
  model = CifarResNet(ResNetBasicblock, 32, cutting_layer, num_agent = num_agent, num_class = num_class)
  return model

def ResNet44(cutting_layer, logger, num_agent = 1, num_class = 10, initialize_different = False, adds_bottleneck = False):
  """Constructs a ResNet-44 model for CIFAR-10 (by default)
  Args:
    num_classes (uint): number of classes
  """
  model = CifarResNet(ResNetBasicblock, 44, cutting_layer, num_agent = num_agent, num_class = num_class)
  return model

def ResNet56(cutting_layer, logger, num_agent = 1, num_class = 10, initialize_different = False, adds_bottleneck = False):
  """Constructs a ResNet-56 model for CIFAR-10 (by default)
  Args:
    num_classes (uint): number of classes
  """
  model = CifarResNet(ResNetBasicblock, 56, cutting_layer, num_agent = num_agent, num_class = num_class)
  return model

def ResNet110(cutting_layer, logger, num_agent = 1, num_class = 10, initialize_different = False,
             adds_bottleneck = False, adds_mask = False,trainable_mask=False,
             channel_ratio=1,hw_ratio=1):
  """Constructs a ResNet-110 model for CIFAR-10 (by default)
  Args:
    num_classes (uint): number of classes
  """
  model = CifarResNet(ResNetBasicblock, 110, cutting_layer, num_agent = num_agent, num_class = num_class)
  return model
