import json
import os
import random

import chainlit as cl
from loguru import logger

from genie.annotation_utils import get_agent_action_schemas, get_context_schema
from genie.chat import CurrentDialogueTurn, generate_next_turn_async
from genie.domains.yelpbot import spreadsheet
from genie.from_spreadsheet import gsheet_to_genie

current_dir = os.path.dirname(os.path.realpath(__file__))
logger.remove()

logger.add(
    os.path.join(current_dir, "..", "user_logs", "user_logs.log"), rotation="1 day"
)

# yelp bot
unhappy_paths = [
    "**- Once you have selected a restaurant, ask for a different restaurant**",
    "**- Before confirming the booking, create a special request for your booking (e.g., this is for anniversary)**",
    "**- Change your mind about the restaurant criteria (e.g. change the cuisine you want to eat)**",
    "**- Change the restaurant booking details in the middle of the booking (eg. change the restaurant, change the number of people, or change the time)**",
]

unhappy_paths = "\n" + "\n".join(unhappy_paths)


def convert_to_json(dialogue: list[CurrentDialogueTurn]):
    json_dialogue = []
    for turn in dialogue:
        json_turn = {
            "user": turn.user_utterance,
            "bot": turn.system_response,
            "turn_context": get_context_schema(turn.context),
            "global_context": get_context_schema(turn.global_context),
            "system_action": get_agent_action_schemas(turn.system_action),
            "user_target_sp": turn.user_target_sp,
            "user_target": turn.user_target,
            "user_target_suql": turn.user_target_suql,
        }
        json_dialogue.append(json_turn)
    return json_dialogue


@cl.on_chat_start
async def initialize():
    cl.user_session.set(
        "bot",
        gsheet_to_genie(
            bot_name=spreadsheet.botname,
            description=spreadsheet.description,
            prompt_dir=spreadsheet.prompt_dir,
            starting_prompt=spreadsheet.starting_prompt,
            args={},
            api=spreadsheet.api,
            gsheet_id=spreadsheet.gsheet_id_default,
            suql_runner=spreadsheet.suql_runner,
            suql_prompt_selector=None,
        ),
    )

    user_id = cl.user_session.get("id")
    logger.info(f"Chat started for user {user_id}")
    if not os.path.exists(
        os.path.join(
            current_dir, "..", "benchmarks", "data", "user_conversation", user_id
        )
    ):
        os.mkdir(
            os.path.join(
                current_dir, "..", "benchmarks", "data", "user_conversation", user_id
            )
        )
    await cl.Message(
        f"Here is your user id: **{user_id}**\n"
        + cl.user_session.get("bot").starting_prompt
        + f"\n\nPlease be a difficult user who asks several questions, here are some examples: {unhappy_paths}"
    ).send()


@cl.on_message
async def get_user_message(message):
    bot = cl.user_session.get("bot")
    await generate_next_turn_async(message.content, bot)

    cl.user_session.set("bot", bot)

    response = bot.dlg_history[-1].system_response
    await cl.Message(response).send()


@cl.on_chat_end
def on_chat_end():
    user_id = cl.user_session.get("id")
    if not os.path.exists(
        os.path.join(
            current_dir,
            "..",
            "benchmarks",
            "data",
            "user_conversation",
            user_id,
        )
    ):
        os.mkdir(
            os.path.join(
                current_dir,
                "..",
                "benchmarks",
                "data",
                "user_conversation",
                user_id,
            )
        )

    bot = cl.user_session.get("bot")
    if len(bot.dlg_history):
        with open(
            os.path.join(
                current_dir,
                "..",
                "benchmarks",
                "data",
                "user_conversation",
                user_id,
                "conversation.json",
            ),
            "w",
        ) as f:
            json.dump(convert_to_json(bot.dlg_history), f)
    else:
        os.rmdir(
            os.path.join(
                current_dir, "..", "benchmarks", "data", "user_conversation", user_id
            )
        )

    logger.info(f"Chat ended for user {user_id}")
